#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import sortIntf
from CliDynamicSymbol import CliDynamicPlugin
import Tac
import ConfigMount
import LazyMount
import SmashLazyMount

LoopProtectModel = CliDynamicPlugin( "LoopProtectModels" )
LoopProtectCli = CliDynamicPlugin( "LoopProtectCli" )

config = None
status = None
bridgingConfig = None
allIntfConfigDir = None # For intf status
packetCounter = None

def tacTimeToUtc( time ):
   return time + Tac.utcNow() - Tac.now()

#--------------------------------------------------------------------------------
# show loop-protection [ detail ]
#--------------------------------------------------------------------------------
def generateLoopProtectShowModel( mode, args ): 
   assert config is not None

   disableTime = getattr( config.globalIntfConfig, 'disabledTime',
                          LoopProtectCli.DISABLED_TIME_DEFAULT )
   transmitInterval = getattr( config.globalIntfConfig, 'transmitInterval',
                               LoopProtectCli.TRANSMIT_INTERVAL_DEFAULT )
   rateLimit = getattr( config.transmitConfig, 'rateLimit',
                        LoopProtectCli.RATE_LIMIT_DEFAULT )
   vlansEnabled = config.vlanEnabled
   model = LoopProtectModel.LoopProtectShow( enabled=config.globalEnabled,
                            transmitInterval=transmitInterval,
                            disableTime=disableTime,
                            rateLimit=rateLimit,
                            totalVlansEnabled=len( vlansEnabled ) )

   if 'detail' in args and config.globalEnabled:
      vlanList = []
      for vlanId in vlansEnabled:
         # Need to generate the rows for this table:
         # Vlan    Loop     Disabled Intfs Total Latest
         #         Detected                Intfs Disabled Time
         # ------ --------- -------------- ----- -------------
         #      1 Yes       Et1-2          20    1/1/01 18:01

         vlanConfig = bridgingConfig.vlanConfig.get( vlanId )
         if not vlanConfig:
            vlanList.append( [ vlanId, None, [], 0, None ] )
            continue
         
         intfsExcluded = { intf for ( intf, value ) in
                           config.intfExclude.items() if value }
         intfsInVlan = set( vlanConfig.intf )
         intfs = [ status.intfStatus[ intf ] for intf in
                      intfsInVlan - intfsExcluded if intf in status.intfStatus ]

         loopDetected = any( intf for intf in intfs if intf.disabled )
         if not loopDetected:
            vlanList.append( [ vlanId, False, [], len( intfs ), None ] )
            continue

         disabledIntfs = [ intf for intf in intfs if intf.disabled ]
         latestDisabledTime = max( intf.disabledAt for intf in disabledIntfs )

         vlanList.append( [ vlanId, True,
                            sortIntf( [ intf.intfId for intf in disabledIntfs ] ),
                            len( intfs ),
                            tacTimeToUtc( latestDisabledTime ) ] )

      model.detail = LoopProtectModel.loopProtectDetail(
         'ff:ff:ff:ff:ff:ff',
         '0x88b7',
         'Interface Disable',
         vlanList )
      
   return model

#--------------------------------------------------------------------------------
# show loop-protection counters
#--------------------------------------------------------------------------------

def generateLoopProtectCountersModel( mode, args ):
   def _generateCounters( counters ): 
      output = {} 
      for instance in sorted( counters ): 
         ci = counters[ instance ]
         if not isinstance( instance, ( int, str ) ):
            instance = str( instance )
         output[ instance ] = [ ci.tx, ci.rx, ci.rxOther ]
      return output
   return LoopProtectModel.loopProtectCounters( config.globalEnabled,
                               _generateCounters( packetCounter.counterVlan ),
                               _generateCounters( packetCounter.counterIntf ) )

#--------------------------------------------------------------------------------
# show loop-protection vlan VLANSET
#--------------------------------------------------------------------------------

def generateLoopProtectVlanModel( mode, args ): # vlanSet=None
   vlans = {}
   vlanSet = args[ 'VLANSET' ]
   for vlanId in vlanSet:
      # Need to generate rows for this table:
      # Vlan Intf LP Enabled State LP Disabled  Disabled at  Bring up at
      # ---- ---- ---------- ----- ----------- ------------ -------------
      # 3    Et1  Yes        Down  Yes         1/1/01 17:21  1/1/01 18:21
      vlans[ vlanId ] = []
      if vlanId not in config.vlanEnabled:
         vlans[ vlanId ].append( [ None, False, None, None, None, None ] )
         continue
      if vlanId not in bridgingConfig.vlanConfig or \
             not bridgingConfig.vlanConfig[ vlanId ].intf:
         vlans[ vlanId ].append( [ None, True, None, None, None, None ] )
         continue

      for intfId in bridgingConfig.vlanConfig[ vlanId ].intf:
         if config.intfExclude.get( intfId ):
            vlans[ vlanId ].append( [ intfId, False, None, None, None, None ] )
            continue

         intfStatus = status.intfStatus.get( intfId )
         if not intfStatus:
            # If in bridgingConfig but not in status.intfStatus( not in topoLib )
            # then there is some churning going on. Pass for now.
            continue

         state = allIntfConfigDir.intfConfig[ intfId ].enabledStateLocal
         if intfStatus.disabled:
            if intfStatus.disabledUntil == 0:
               disabledUntil = intfStatus.disabledUntil
            else:
               disabledUntil = tacTimeToUtc( intfStatus.disabledUntil )
            vlans[ vlanId ].append( [
                  intfId, True, state, True,
                  tacTimeToUtc( intfStatus.disabledAt ),
                  disabledUntil ] )
         else:
            vlans[ vlanId ].append( [ intfId, True, state, False, None, None ] )

   return LoopProtectModel.LoopProtectVlan( enabled=config.globalEnabled,
                           vlanIntfs=LoopProtectModel.loopProtectVlanIntfs( vlans ) )

def Plugin( entityManager ):
   global config, bridgingConfig
   global status
   global allIntfConfigDir
   global packetCounter

   config = ConfigMount.mount( entityManager, 'loopprotect/config',
                               'LoopProtect::Config::Config', 'w' )
   status = LazyMount.mount( entityManager, 'loopprotect/status',
                             'LoopProtect::Status::Status', 'r' )
   bridgingConfig = LazyMount.mount( entityManager, 'bridging/config',
                                     'Bridging::Config', 'r' )
   allIntfConfigDir = LazyMount.mount( entityManager, 'interface/config/all',
                                       'Interface::AllIntfConfigDir', 'r' )
   packetCounter = SmashLazyMount.mount( entityManager, 'loopprotect/counter',
                                         'LoopProtect::Status::PacketCounter',
                                         SmashLazyMount.mountInfo( 'reader' ) )