# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.MacsecCli import config
from CliPlugin.MacAddr import macAddrMatcher
from CliMode.Macsec import MacsecBaseMode
import BasicCli
import CliCommand
import CliMatcher
import Toggles.MacsecCommonToggleLib as macsecToggle

def getProfileNames( mode=None ):
   return config.profile

# Macsec mode
class MacsecMode( MacsecBaseMode, BasicCli.ConfigModeBase ):
   name = "MAC security configuration"

   def __init__( self, parent, session ):
      MacsecBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# "[ no | default ] ptp bypass", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class GlobalPtpBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp bypass'
   noOrDefaultSyntax = syntax
   data = {
         'ptp' : 'Set PTP processing',
         'bypass' : 'Transmit and Receive without protection',
   }
   handler = "MacsecCliHandler.handlePtpBypass"
   noHandler = "MacsecCliHandler.handleNoPtpBypass"
   defaultHandler = "MacsecCliHandler.handleDefaultPtpBypass"

MacsecMode.addCommandClass( GlobalPtpBypassCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] profile PROFILENAME", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class ProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'MAC security profile commands',
      'PROFILENAME': CliMatcher.DynamicNameMatcher( getProfileNames,
                        'Profile name' ),
   }

   handler = "MacsecProfileCliHandler.gotoMacsecProfileMode"
   noOrDefaultHandler = "MacsecCliHandler.noMacsecProfileMode"

MacsecMode.addCommandClass( ProfileCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] fips restrictions", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class FipsRestrictionsCmd( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = syntax
   data = {
      'fips': 'FIPS settings',
      'restrictions': 'Use FIPS compliant algorithms',
   }

   handler = "MacsecCliHandler.handleFipsRestrictions"
   noOrDefaultHandler = "MacsecCliHandler.handleNoFipsRestrictions"

MacsecMode.addCommandClass( FipsRestrictionsCmd )

#-------------------------------------------------------------------------
# The "[no | default] delay protection" command, in "config-mac-sec" mode.
#-------------------------------------------------------------------------
class DelayProtectionCmd( CliCommand.CliCommandClass ):
   syntax = 'delay protection'
   noOrDefaultSyntax = syntax
   data = {
      "delay" : "MAC security data delay configuration",
      "protection" : "Data delay protection",
   }
   hidden = not macsecToggle.toggleMacsecDelayProtectionEnabled()

   handler = "MacsecCliHandler.handleDelayProtection"
   noOrDefaultHandler = "MacsecCliHandler.handleNoDelayProtection"

MacsecMode.addCommandClass( DelayProtectionCmd )

#--------------------------------------------------------------------------------
# "( no | default ) license ..."
# "license LICENSEENAME AUTHKEYSTR", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class LicenseCmd( CliCommand.CliCommandClass ):
   syntax = 'license LICENSEENAME AUTHKEYSTR'
   noOrDefaultSyntax = 'license ...'
   data = {
      'license': 'MAC security legacy license configuration',
      'LICENSEENAME': CliMatcher.PatternMatcher( pattern='[\\w-]+',
                        helpdesc='Licensee name', helpname='WORD' ),
      'AUTHKEYSTR': CliMatcher.PatternMatcher( pattern='[a-fA-F0-9]{1,8}',
                        helpdesc='Key to authorize Mac security',
                        helpname='8-digit hex number' ),
   }

   handler = "MacsecCliHandler.setMacsecLicense"
   noOrDefaultHandler = "MacsecCliHandler.noMacsecLicense"

MacsecMode.addCommandClass( LicenseCmd )

class L2ProtocolEthFlowControlCmd( CliCommand.CliCommandClass ) :
   syntax = 'l2-protocol ethernet-flow-control ( encrypt | bypass )'
   noOrDefaultSyntax = 'l2-protocol ethernet-flow-control ...'
   data = {
      'l2-protocol': 'Set L2 protocol processing',
      'ethernet-flow-control': 'Ethernet Flow Control frame processing',
      'encrypt': 'Transmit/Receive with protection',
      'bypass': 'Transmit without protection',
   }

#--------------------------------------------------------------------------------
# "[ no | default ] l2-protocol ethernet-flow-control ( encrypt | bypass )",
# in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class GlobalL2ProtocolEthFlowControlCmd( CliCommand.CliCommandClass ):
   syntax = L2ProtocolEthFlowControlCmd.syntax
   noOrDefaultSyntax = L2ProtocolEthFlowControlCmd.noOrDefaultSyntax
   data = dict( L2ProtocolEthFlowControlCmd.data )
   handler = "MacsecCliHandler.handleGlobalEthFlowControlEncrypt"
   noOrDefaultHandler = "MacsecCliHandler.handleGlobalDefaultEthFlowControlEncrypt"

MacsecMode.addCommandClass( GlobalL2ProtocolEthFlowControlCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] shutdown", in "config-mac-sec" mode
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable MAC security',
   }

   handler = "MacsecCliHandler.shutDownMacsec"
   noOrDefaultHandler = "MacsecCliHandler.noShutDownMacsec"

MacsecMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] eapol mac destination MAC
#--------------------------------------------------------------------------------
matcherEapol = CliMatcher.KeywordMatcher( 'eapol',
               helpdesc='EAPoL attributes' )

class EapolDestinationMacCmd( CliCommand.CliCommandClass ):
   syntax = 'eapol mac destination MAC'
   noOrDefaultSyntax = 'eapol mac destination ...'
   data = {
       'eapol': matcherEapol,
       'mac': 'MAC addresses',
       'destination': 'Configure destination MAC',
       'MAC': macAddrMatcher,
   }
   handler = "MacsecCliHandler.setMacsecEapolDestinationMac"
   noOrDefaultHandler = handler

MacsecMode.addCommandClass( EapolDestinationMacCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] eapol ethertype TYPE
#--------------------------------------------------------------------------------
class EapolEtherTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'eapol ethertype TYPE'
   noOrDefaultSyntax = 'eapol ethertype ...'
   data = {
       'eapol': matcherEapol,
       'ethertype': 'Configure Ethernet type',
       'TYPE': CliMatcher.IntegerMatcher( 0x600, 0xFFFF,
                                          helpdesc='Ethernet type for EAPoL' )
   }
   handler = "MacsecCliHandler.setMacsecEapolEtherType"
   noOrDefaultHandler = handler

if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
   MacsecMode.addCommandClass( EapolEtherTypeCmd )
