# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.MacsecCli import (
   config,
   sharedSecretConfig
)
from CliMode.Macsec import MacsecProfileBaseMode
from CliMode.Macsec import MacsecProfileSecureChannelMode
from CliMode.Macsec import MacsecProfileStaticSakMode
import BasicCli
import CliCommand
import CliMatcher
import ReversibleSecretCli
from MacsecCommon import cliToTacCipherSuite
import Tac
import Toggles.MacsecCommonToggleLib as macsecToggle
from TypeFuture import TacLazyType
import string
MacsecSak = TacLazyType( "Macsec::CliSak" )
MacsecSci = TacLazyType( "Macsec::Sci" )
PtpBypass = TacLazyType( "Macsec::PtpBypass" )
Cak = TacLazyType( "Macsec::Cak" )
KeyDerivationPadding = TacLazyType( "Macsec::KeyDerivationPadding" )

# Macsec Profile Mode
class MacsecProfileMode( MacsecProfileBaseMode, BasicCli.ConfigModeBase ):
   name = 'MAC security profile configuration'

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      MacsecProfileBaseMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.updateInterfaceList()

   def profile( self ):
      return config.profile[ self.profileName ]

   def updateInterfaceList( self ):
      # Walk through all interfaces which have this profile enabled and update the
      # intf collection.
      profile = self.profile()
      for intf in config.intfConfig.values():
         if intf.profileName == self.profileName:
            profile.intf[ intf.intfId ] = True

   def validateKeySize( self, cakSecret, cipher, fallback ):
      # Check if the configured key size is consistent with the cipher in use.
      # We only care if the cipher in use is 128bits and the key configured is
      # longer in size.
      keyType = 'fallback' if fallback else 'primary'
      decodedCakLength = cakSecret.clearTextSize()

      if cipher in [ 'gcmAesXpn128', 'gcmAes128' ] and decodedCakLength > 32:
         self.addError( 'Configured %s key is too big for the cipher in use' % \
               keyType )
         return False

      if cipher in [ 'gcmAesXpn256', 'gcmAes256' ] and decodedCakLength > 64:
         self.addError( 'Configured %s key is too big for the cipher in use' % \
               keyType )
         return False

      return True

   def parseCak( self, ckn, cakSecret ):
      # Verify that CKN is non zero.
      if int( ckn, 16 ) == 0:
         self.addError( 'Key name cannot be zero' )
         return None

      cakClearTextLower = cakSecret.getClearText().lower()
      try:
         intCak = int( cakClearTextLower, 16 )
      except ValueError:
         self.addError( 'Invalid Encrypted Key' )
         return None

      if intCak == 0:
         self.addError( 'Key value cannot be zero' )
         return None

      # Ensure ClearText value is strictly hex char. Should exclude any
      # non-hex char such as "/n" or any other non-hex char.
      # Handling Bug-640182
      try:
         validCak = all( c in string.hexdigits for c in cakClearTextLower )
      except ValueError:
         self.addError( 'Invalid Encrypted Key' )
         return None

      if not validCak:
         self.addError( 'Invalid Encrypted Key' )
         return None

      # convert the clearText to be lower case of input
      cakSecret.setClearText( cakClearTextLower )

      if len( ckn ) % 2 != 0:
         # If the CKN is not even length in size, make it even. It's important
         # for our crypto libraries that the size of ckn be an even number of hex
         # octets.
         ckn = "0" + ckn
      cakVal = Tac.Value( "Macsec::Cak", ckn=ckn.lower(), cakSecret=cakSecret )
      return cakVal

   def addKey( self, args ):
      cknString = args[ 'CKN' ]
      cakSecret = args[ 'SECRET' ]
      fallback = 'fallback' in args

      cakVal = self.parseCak( cknString, cakSecret )

      profile = self.profile()

      if not self.validateKeySize( cakSecret, profile.cipherSuite, fallback ):
         return

      if not cakVal:
         return

      # Throw an error if the key matches the default key. The two keys ought to be
      # different.
      conflict = False
      if fallback and profile.key.ckn == cakVal.ckn:
         conflict = True
      if not fallback and profile.defaultKey.ckn == cakVal.ckn:
         conflict = True
      if conflict:
         self.addError( 'Primary key should be different from the fallback key' )
         return

      if not fallback:
         # If source dot1x was enabled, remove it now.
         if profile.dot1xEnabled:
            self.dot1xDisabled()
         profile.secretProfileName = ''
         self.noStaticSak()
         # Add primary key
         profile.key = cakVal
      else:
         # Add the default key
         profile.defaultSecretProfileName = ''
         profile.defaultKey = cakVal

   def noKey( self, args ):
      cknString = args[ 'CKN' ]
      cakSecret = args[ 'SECRET' ]
      fallback = 'fallback' in args

      cakVal = self.parseCak( cknString, cakSecret )

      profile = self.profile()
      # Delete the key from the profile.
      if not fallback:
         if profile.key.ckn == cakVal.ckn:
            profile.key = Cak()
      else:
         if profile.defaultKey.ckn == cakVal.ckn:
            profile.defaultKey = Cak()

   def addKeyServerPriority( self, args ):
      # Update the key server priority for this profile.
      self.profile().keyServerPriority = args[ 'PRIORITY' ]

   def noKeyServerPriority( self, args ):
      # Restore the key server priority to its default.
      profile = self.profile()
      profile.keyServerPriority = profile.keyServerPriorityDefault

   def addReKeyPeriod( self, args ):
      # Update the session re-key period in this profile.
      self.profile().sessionReKeyPeriod = args[ 'REKEYPERIOD' ]

   def noReKeyPeriod( self, args ):
      # Restore the session re-key period to its default.
      profile = self.profile()
      profile.sessionReKeyPeriod = profile.sessionReKeyPeriodDefault

   def setRetireDelay( self, args ):
      profile = self.profile()
      period = args.get( 'PERIOD', profile.oldKeyRetirementPeriodDefault )
      profile.oldKeyRetirementPeriod = period

   def setTransmitDelay( self, args ):
      profile = self.profile()
      period = args.get( 'PERIOD', profile.latestKeyEnablePeriodDefault )
      profile.latestKeyEnablePeriod = period

   def addMkaLifeTime( self, args ):
      # Update MKA session lifetime for this profile
      self.profile().mkaLifeTime = args[ 'LIFETIME' ]

   def noMkaLifeTime( self, args ):
      # Restore the session mka lifetime for this profile
      self.profile().mkaLifeTime = self.profile().mkaLifeTimeDefault

   def setLpnValueZero( self, args=None ):
      # Enable/Disable publishing of LPN values as 0
      disabled = 'disabled' in args
      self.profile().lpnValueZeroNonXpn = disabled
      self.profile().lpnValueZeroXpn = disabled

   def unsetLpnValueZero( self, args=None ):
      # Enable publishing of real LPN values for non-Xpn cipher suites and zero for
      # Xpn cipher suites
      self.profile().lpnValueZeroNonXpn = self.profile().lpnValueZeroNonXpnDefault
      self.profile().lpnValueZeroXpn = self.profile().lpnValueZeroXpnDefault

   def setAnMode( self, args ):
      profile = self.profile()
      profile.only1BitAnMode = args.get( 'AN' ) == '1'

   def dot1xEnabled( self, args ):
      # Delete any configured primary key and shared-secret profile.
      self.profile().key = Cak()
      self.profile().secretProfileName = ''
      self.noStaticSak()

      # Enable dot1x setting on the profile.
      self.profile().dot1xEnabled = True

   def dot1xDisabled( self, args=None ):
      # Disable dot1x setting on the profile.
      self.profile().dot1xEnabled = False

   def setSecretProfileName( self, args ):
      profile = self.profile()
      fallback = 'fallback' in args
      if not fallback:
         profile.key = Cak()
         profile.dot1xEnabled = False
         self.noStaticSak()
         profile.secretProfileName = args[ 'PROFILE_NAME' ]
      else:
         profile.defaultKey = Cak()
         profile.defaultSecretProfileName = args[ 'PROFILE_NAME' ]

   def noSecretProfileName( self, args=None ):
      fallback = 'fallback' in args
      if not fallback:
         self.profile().secretProfileName = ''
      else:
         self.profile().defaultSecretProfileName = ''

   def configureGroupCak( self, args ):
      groupCakLifetime = args.get( 'GROUPCAKLIFETIME' )
      lifetime = groupCakLifetime if groupCakLifetime else \
                 self.profile().groupCak.lifetimeDefault
      self.profile().groupCak = Tac.Value( "Macsec::GroupCak",
                                           enabled=True,
                                           lifetime = lifetime )

   def noGroupCak( self, args ):
      self.profile().groupCak = Tac.Value( "Macsec::GroupCak" )

   def configureCipherSuite( self , args ):
      cipher = args.get( 'CIPHER' )
      # Configure the cipher in the profile
      cipherSuite = cliToTacCipherSuite[ cipher ]

      primaryKey = self.profile().key.cakSecret
      defaultKey = self.profile().defaultKey.cakSecret
      if not self.validateKeySize( primaryKey, cipherSuite, False ) or \
            not self.validateKeySize( defaultKey, cipherSuite, True ):
         return

      self.profile().cipherSuite = cipherSuite

   def noCipherSuite( self, args ):
      # Configure default ciphersuite
      primaryKey = self.profile().key.cakSecret
      defaultKey = self.profile().defaultKey.cakSecret
      cipherSuite = self.profile().cipherSuiteDefault # Default

      if not self.validateKeySize( primaryKey, cipherSuite, False ) or \
            not self.validateKeySize( defaultKey, cipherSuite, True ):
         return

      self.profile().cipherSuite = cipherSuite

   def includeSci ( self, args ):
      # Enable sci inclusion
      self.profile().includeSci = True

   def noIncludeSci ( self, args ):
      # Disable sci inclusion
      self.profile().includeSci = self.profile().includeSciDefault

   def bypassLldp ( self, args ):
      # Transmit/Receive LLDP frames without protection
      self.profile().bypassLldp = True
      # Transmit/Receive LLDP frames when port is unauthorized
      self.profile().bypassLldpUnauth = 'unauthorized' in args

   def noBypassLldp ( self, args ):
      # Transmit/Receive encrypted LLDP frames
      self.profile().bypassLldp = self.profile().bypassLldpDefault
      # Transmit/Receive LLDP frames when port is unauthorized
      self.profile().bypassLldpUnauth = False

   def ethFlowControlEncrypt ( self, args ):
      if 'bypass' in args:
         # Transmit Ethernet Flow Control frames without protection
         mode = Tac.Type( "Macsec::EthFlowControl" ).bypass
      else:
         # Transmit/Receive Ethernet Flow Control frames with protection
         assert 'encrypt' in args
         mode = Tac.Type( "Macsec::EthFlowControl" ).encrypt

      self.profile().ethFlowControl = mode

   def defaultEthFlowControlEncrypt ( self, args ):
      # Transmit/Receive Ethernet Flow Control frames with platform default encrypt
      self.profile().ethFlowControl =\
            Tac.Type( "Macsec::EthFlowControl" ).platformDefault

   def ptpBypass( self, args ):
      # Bypass PTP traffic
      self.profile().bypassPtp = PtpBypass.ptpBypass

   def noPtpBypass( self, args ):
      # Encrypt PTP traffic
      self.profile().bypassPtp = PtpBypass.ptpEncrypt

   def defaultPtpBypass( self, args ):
      # Revert to the platform default PTP bypass behavior
      self.profile().bypassPtp = PtpBypass.ptpPlatformDefault

   def fipsPostFailureForced( self, args ):
      self.profile().fipsPostFailureForced = True

   def noFipsPostFailureForced( self, args ):
      self.profile().fipsPostFailureForced = False

   def trafficPolicy( self, args ):
      if 'drop' in args:
         self.profile().trafficPolicyOnNoMka = 'blocked'
      elif 'active-sak' in args:
         self.profile().trafficPolicyOnNoMka = 'useActiveSak'
      else:
         self.profile().trafficPolicyOnNoMka = 'unprotected'

   def noOrDefaultTrafficPolicy( self, args ):
      self.profile().trafficPolicyOnNoMka = \
                                       self.profile().trafficPolicyOnNoMkaDefault

   def replayProtectionDisabled ( self, args ):
      # Disable Replay Protection
      self.profile().replayProtection = False

   def noReplayProtectionDisabled ( self, args ):
      # Enable Replay Protection
      self.profile().replayProtection = self.profile().replayProtectionDefault

   def replayProtectionWindow ( self, args ):
      replayProtectionWindow = args[ 'REPLAYPROTECTIONWINDOW' ]
      if replayProtectionWindow is None:
         replayProtectionWindow = self.profile().replayProtectionWindowDefault
      self.profile().replayProtectionWindow = replayProtectionWindow

   def noReplayProtectionWindow ( self, args ):
      self.profile().replayProtectionWindow = \
            self.profile().replayProtectionWindowDefault

   def keyRetire( self, args ):
      self.profile().keyRetire = True

   def noKeyRetire ( self, args ):
      self.profile().keyRetire = self.profile().keyRetireDefault

   def gotoStaticSakMode( self, args ):
      # reset key, dot1xEnabled, secretProfileName
      self.profile().key = Cak()
      self.profile().dot1xEnabled = False
      self.profile().secretProfileName = ''
      childMode = self.childMode( MacsecStaticSakMode )
      self.session_.gotoChildMode( childMode )

   def noStaticSak( self, args=None ):
      self.profile().txStaticSak = MacsecSak()
      self.profile().txStaticSci = MacsecSci()
      self.profile().rxStaticSak.clear()
      self.profile().rxStaticSci = MacsecSci()

class MacsecStaticSakMode( MacsecProfileStaticSakMode,
                           BasicCli.ConfigModeBase ):
   name = "MAC security static SAK configuration"

   def __init__( self, parent, session ):
      self.profile = parent.profile()
      MacsecProfileStaticSakMode.__init__( self, ( self.profile.name ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoSecureChannelMode( self, args ):
      childMode = self.childMode ( MacsecSecureChannelMode, txOrRx= args[ 'TX_RX' ] )
      self.session_.gotoChildMode( childMode )

   def noSecureChannel( self, args ):
      txOrRx = args[ 'TX_RX' ]
      if txOrRx == 'tx' and self.profile.txStaticSak:
         self.profile.txStaticSak = MacsecSak()
         self.profile.txStaticSci = MacsecSci()
      elif txOrRx == 'rx' and self.profile.rxStaticSak:
         self.profile.rxStaticSak.clear()
         self.profile.rxStaticSci = MacsecSci()

class MacsecSecureChannelMode( MacsecProfileSecureChannelMode,
                               BasicCli.ConfigModeBase ):
   name = "MAC security static SAK secure channel configuration"

   def __init__( self, parent, session, txOrRx ):
      self.profile = parent.parent_.profile()
      self.direction = txOrRx
      MacsecProfileSecureChannelMode.__init__( self, ( self.profile.name, txOrRx ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setTxStaticSakParam( self, sak=None, sci=None ):
      if sak:
         self.profile.txStaticSak = sak
      if sci:
         self.profile.txStaticSci = sci

   def setRxStaticSakParam( self, sak=None, sci=None ):
      if sak:
         self.profile.addRxStaticSak( sak )
      if sci:
         self.profile.rxStaticSci = sci

   def addIdentifier( self, args ):
      addrString = args[ 'IDENTIFIER' ]
      portNum = int( addrString.split( ':' )[-1] )
      addr = addrString.split( '::' )[0]
      newSci = Tac.Value( "Macsec::Sci", addr=addr, portNum=portNum  )
      if self.direction == 'tx':
         self.setTxStaticSakParam( sci=newSci )
      elif self.direction == 'rx':
         self.setRxStaticSakParam( sci=newSci )

   def noIdentifier( self, args ):
      if self.direction == 'tx':
         self.profile.txStaticSci = MacsecSci()
      elif self.direction == 'rx':
         self.profile.rxStaticSci = MacsecSci()

   def addKey( self, args ):
      sakSecret = args[ 'SECRET' ]
      an = args[ 'AN' ]
      saltString = args.get( 'SALT', '' )

      sakClearTextLower = sakSecret.getClearText().lower()
      try:
         intSak = int( sakClearTextLower, 16 )
      except ValueError:
         self.addError( 'Invalid Encrypted Key' )
         return

      if intSak == 0:
         self.addError( 'Key value cannot be zero' )
         return

      # convert the clearText to be lower case of input
      sakSecret.setClearText( sakClearTextLower )

      sakLength = len( sakClearTextLower )
      if not sakLength == 32 and not sakLength == 64:
         self.addError( 'Invalid Key length' )
         return

      if saltString:
         sakVal = Tac.Value( 'Macsec::CliSak', an=an, keySecret=sakSecret,
                             salt=saltString )
      else:
         sakVal = Tac.Value( 'Macsec::CliSak', an=an, keySecret=sakSecret )

      if self.direction == 'tx':
         self.setTxStaticSakParam( sak=sakVal )
      elif self.direction == 'rx':
         self.setRxStaticSakParam( sak=sakVal )

   def noKey( self, args ):
      if ( self.direction == 'tx' and
            self.profile.txStaticSak.an == args[ 'AN' ] ):
         self.profile.txStaticSak = MacsecSak()
      elif self.direction == 'rx':
         del self.profile.rxStaticSak[ args[ 'AN' ] ]

# Macsec profile specific commands

keyKwMatcher = CliMatcher.KeywordMatcher( 'key',
   helpdesc='Connectivity association key' )

keyPattern_ = r'(^[a-fA-F0-9][a-fA-F0-9]{1,63}$)'

cknNameMatcher = CliMatcher.PatternMatcher(
   keyPattern_, helpname='CKN',
   helpdesc='Connectivity association key name in hex octets' )

cleartextKeyMatcher = CliMatcher.PatternMatcher(
   keyPattern_, helpname='CAK',
   helpdesc='Connectivity association key in hex octets' )
cakSecretCliExpr = ReversibleSecretCli.ReversiblePasswordCliExpression(
   cleartextMatcher=cleartextKeyMatcher )

secondsMatcher = CliMatcher.KeywordMatcher( 'seconds', 'Number of seconds' )

class KeyCommand( CliCommand.CliCommandClass ):
   syntax = "key CKN SECRET [ fallback ]"
   noOrDefaultSyntax = "key CKN SECRET [ fallback ]"
   data = {
      'key': keyKwMatcher,
      'CKN': cknNameMatcher,
      'SECRET': cakSecretCliExpr,
      'fallback': 'Configure the key as a fallback'
      }

   handler = MacsecProfileMode.addKey
   noOrDefaultHandler = MacsecProfileMode.noKey

MacsecProfileMode.addCommandClass( KeyCommand )

#--------------------------------------------------------------------------------
# "[ no | default ] key retirement immediate", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeyRetirementImmediateCmd( CliCommand.CliCommandClass ):
   syntax = 'key retirement immediate'
   noOrDefaultSyntax = syntax
   data = {
      'key': keyKwMatcher,
      'retirement': 'Retire the key',
      'immediate': 'Retire the key immediately',
   }

   handler = MacsecProfileMode.keyRetire
   noOrDefaultHandler = MacsecProfileMode.noKeyRetire

MacsecProfileMode.addCommandClass( KeyRetirementImmediateCmd )

matcherSource = CliMatcher.KeywordMatcher( 'source',
               helpdesc='List of sources to derive MAC security keys' )

#--------------------------------------------------------------------------------
# "[ no | default ] key source dot1x", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeySourceDot1XCmd( CliCommand.CliCommandClass ):
   syntax = 'key source dot1x'
   noOrDefaultSyntax = syntax
   data = {
      'key': keyKwMatcher,
      'source': matcherSource,
      'dot1x': 'Derive MAC security keys from IEEE 802.1X based port authentication',
   }

   handler = MacsecProfileMode.dot1xEnabled
   noOrDefaultHandler = MacsecProfileMode.dot1xDisabled

MacsecProfileMode.addCommandClass( KeySourceDot1XCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] key source shared-secret profile PROFILE_NAME"
# in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeySourceSharedSecretProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'key source shared-secret profile PROFILE_NAME [ fallback ]'
   noOrDefaultSyntax = 'key source shared-secret profile [ fallback ]'
   data = {
      'key': keyKwMatcher,
      'source': matcherSource,
      'shared-secret': 'Use a shared-secret profile for key',
      'profile': 'Use a shared-secret profile for key',
      'PROFILE_NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: sharedSecretConfig.profile,
         'Shared-secret profile name' ),
      'fallback': 'Configure the profile as a fallback',
   }
   handler = MacsecProfileMode.setSecretProfileName
   noOrDefaultHandler = MacsecProfileMode.noSecretProfileName
MacsecProfileMode.addCommandClass( KeySourceSharedSecretProfileCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] key source sak static", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeySourceStaticSakCmd( CliCommand.CliCommandClass ):
   syntax = "key source sak static"
   noOrDefaultSyntax = "key source sak static"
   data = {
      'key': keyKwMatcher,
      'source': matcherSource,
      'sak': 'Secure association key',
      'static': 'Use static SAK',
   }
   handler = MacsecProfileMode.gotoStaticSakMode
   noOrDefaultHandler = MacsecProfileMode.noStaticSak

MacsecProfileMode.addCommandClass( KeySourceStaticSakCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] key source group-cak", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeySourceGroupCakCmd( CliCommand.CliCommandClass ):
   syntax = 'key source group-cak [ GROUPCAKLIFETIME ]'
   noOrDefaultSyntax = 'key source group-cak ...'
   data = {
      'key': keyKwMatcher,
      'source': matcherSource,
      'group-cak': 'Derive MAC security keys from Group CAK Distribution',
      'GROUPCAKLIFETIME': CliMatcher.IntegerMatcher( 30, 1000,
         helpdesc='Group CAK rekey period in minutes.' ),
   }
   hidden = True

   handler = MacsecProfileMode.configureGroupCak
   noOrDefaultHandler = MacsecProfileMode.noGroupCak

MacsecProfileMode.addCommandClass( KeySourceGroupCakCmd )

matcherMka = CliMatcher.KeywordMatcher( 'mka',
               helpdesc='MAC security key agreement protocol options' )

#--------------------------------------------------------------------------------
# "( no | default ) mka key-server priority ...", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class MkaKeyServerPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ' mka key-server priority PRIORITY '
   noOrDefaultSyntax = 'mka key-server priority ...'
   data = {
      'mka': matcherMka,
      'key-server': 'MKA key server options',
      'priority': 'MKA key server priority',
      'PRIORITY': CliMatcher.IntegerMatcher( 0, 255,
                     helpdesc='Key server priority value' ),
   }

   handler = MacsecProfileMode.addKeyServerPriority
   noOrDefaultHandler = MacsecProfileMode.noKeyServerPriority

MacsecProfileMode.addCommandClass( MkaKeyServerPriorityCmd )

#--------------------------------------------------------------------------------
# "( no | default ) key derivation padding ( prepend | append )",
# in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class KeyDerivationPaddingCmd( CliCommand.CliCommandClass ):
   syntax = ' key derivation padding ( prepend | append ) '
   noOrDefaultSyntax = 'key derivation padding ...'
   data = {
      'key': keyKwMatcher,
      'derivation': 'Deriving final key from CAK/CKN',
      'padding': 'Adding extra zeros to make data of standard length',
      'prepend': 'Pad zeros before the key',
      'append': 'Pad zeros after the key',
   }

   handler = "MacsecCliHandler.addKeyDerivationPadding"
   noOrDefaultHandler = handler

MacsecProfileMode.addCommandClass( KeyDerivationPaddingCmd )

#--------------------------------------------------------------------------------
# "( no | default ) mka session rekey-period ...", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class MkaSessionRekeyPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'mka session rekey-period REKEYPERIOD'
   noOrDefaultSyntax = 'mka session rekey-period ...'
   data = {
      'mka': matcherMka,
      'session': 'Set MKA session options',
      'rekey-period': 'Set MKA session re-key period',
      'REKEYPERIOD': CliMatcher.IntegerMatcher( 30, 100000,
               helpdesc='Session re-key period in seconds.' ),
   }

   handler = MacsecProfileMode.addReKeyPeriod
   noOrDefaultHandler = MacsecProfileMode.noReKeyPeriod

MacsecProfileMode.addCommandClass( MkaSessionRekeyPeriodCmd )

#--------------------------------------------------------------------------------
# "( no | default ) sak rx retire-delay ...",
# in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class SakRxRetireDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'sak rx retire-delay PERIOD seconds'
   noOrDefaultSyntax = 'sak rx retire-delay ...'
   data = {
      'sak': 'Secure association key',
      'rx': 'Receive',
      'retire-delay': 'Set delay value for old SAK retirement',
      'PERIOD': CliMatcher.IntegerMatcher( 3, 180,
                     helpdesc='Retirement delay period in seconds.' ),
      'seconds': secondsMatcher,
   }

   handler = MacsecProfileMode.setRetireDelay
   noOrDefaultHandler = handler

MacsecProfileMode.addCommandClass( SakRxRetireDelayCmd )

#--------------------------------------------------------------------------------
# "( no | default ) sak tx transmit-delay ...",
# in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class SakTxTransmitDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'sak tx transmit-delay PERIOD seconds'
   noOrDefaultSyntax = 'sak tx transmit-delay ...'
   data = {
      'sak': 'Secure association key',
      'tx': 'Transmit',
      'transmit-delay': 'Set delay value for latest SAK transmit enable',
      'PERIOD': CliMatcher.IntegerMatcher( 6, 180,
                     helpdesc='Retirement delay period in seconds.' ),
      'seconds': secondsMatcher,
   }

   handler = MacsecProfileMode.setTransmitDelay
   noOrDefaultHandler = handler

MacsecProfileMode.addCommandClass( SakTxTransmitDelayCmd )

#--------------------------------------------------------------------------------
# "( no | default ) mka session lifetime ...", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class MkaSessionLifeTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'mka session lifetime LIFETIME seconds'
   noOrDefaultSyntax = 'mka session lifetime ...'
   data = {
      'mka': matcherMka,
      'session': 'Set MKA session options',
      'lifetime': 'Set MKA session lifetime period',
      'LIFETIME': CliMatcher.IntegerMatcher( 6, 2**32-1,
               helpdesc='Session lifetime in seconds.' ),
      'seconds': secondsMatcher,
   }

   handler = MacsecProfileMode.addMkaLifeTime
   noOrDefaultHandler = MacsecProfileMode.noMkaLifeTime

MacsecProfileMode.addCommandClass( MkaSessionLifeTimeCmd )

#-----------------------------------------------------------------
# "( no | default ) mka session lpn advertisement [ disabled ]"
# in "config-mac-sec-profile" mode
#-----------------------------------------------------------------
class MkaSessionLpnAdvertisementDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mka session lpn advertisement [ disabled ]'
   noOrDefaultSyntax = syntax
   data = {
      'mka': matcherMka,
      'session': 'Set MKA session options',
      'lpn': 'Local Lowest PN value',
      'advertisement': 'Advertise real LPN values in SakUseInfo parameter of MKPDU',
      'disabled': 'Advertise LPN values as 0 in SakUseInfo parameter of MKPDU',
   }

   handler = MacsecProfileMode.setLpnValueZero
   noOrDefaultHandler = MacsecProfileMode.unsetLpnValueZero

MacsecProfileMode.addCommandClass( MkaSessionLpnAdvertisementDisabledCmd )

#-----------------------------------------------------------------
# "sak an maximum 1|3" and "{no|default} sak an maximum"
# in "config-mac-sec-profile" mode
#-----------------------------------------------------------------
class SakAnMaximumCmd( CliCommand.CliCommandClass ):
   syntax = 'sak an maximum AN'
   noOrDefaultSyntax = 'sak an maximum ...'
   data = {
      'sak': 'Secure association key',
      'an': 'Association number',
      'maximum': 'Maximum value of AN for any SAK',
      'AN': CliMatcher.EnumMatcher( {
         '1' : 'Maximum AN value of 1',
         '3' : 'Maximum AN value of 3',
         } )
   }

   handler = MacsecProfileMode.setAnMode
   noOrDefaultHandler = handler

MacsecProfileMode.addCommandClass( SakAnMaximumCmd )

#---------------------------------------------------------------------
# The "[no | default] cipher" command, in "config-mac-sec-profile" mode.
#---------------------------------------------------------------------
class CipherCmd( CliCommand.CliCommandClass ):
   _cipherMatcher = CliMatcher.EnumMatcher( {
      "aes128-gcm-xpn": "Advanced Encryption Standard (128 bit, "
                        "Galois/Counter mode, Extended Packet Numbering)",
      "aes256-gcm-xpn": "Advanced Encryption Standard (256 bit, "
                        "Galois/Counter mode, Extended Packet Numbering)",
      "aes128-gcm"    : "Advanced Encryption Standard (128 bit, "
                        "Galois/Counter mode)",
      "aes256-gcm"    : "Advanced Encryption Standard (256 bit, "
                        "Galois/Counter mode)"
   } )
   syntax = 'cipher CIPHER'
   noOrDefaultSyntax = 'cipher ...'
   data = {
      'cipher': 'Data encryption algorithm and mode ',
      'CIPHER': _cipherMatcher
   }

   handler = MacsecProfileMode.configureCipherSuite
   noOrDefaultHandler = MacsecProfileMode.noCipherSuite

MacsecProfileMode.addCommandClass( CipherCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] l2-protocol lldp bypass [ unauthorized ]",
#  in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class L2ProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol lldp bypass [ unauthorized ]'
   noOrDefaultSyntax = syntax
   data = {
      'l2-protocol': 'Set L2 protocol processing',
      'lldp': 'LLDP frame processing',
      'bypass': 'Transmit/Receive without protection',
      'unauthorized': 'Bypass when port is unauthorized',
   }

   handler = MacsecProfileMode.bypassLldp
   noOrDefaultHandler = MacsecProfileMode.noBypassLldp

MacsecProfileMode.addCommandClass( L2ProtocolCmd )

class L2ProtocolEthFlowControlCmd( CliCommand.CliCommandClass ) :
   syntax = 'l2-protocol ethernet-flow-control ( encrypt | bypass )'
   noOrDefaultSyntax = 'l2-protocol ethernet-flow-control ...'
   data = {
      'l2-protocol': 'Set L2 protocol processing',
      'ethernet-flow-control': 'Ethernet Flow Control frame processing',
      'encrypt': 'Transmit/Receive with protection',
      'bypass': 'Transmit without protection',
   }

#--------------------------------------------------------------------------------
# "[ no | default ] l2-protocol ethernet-flow-control ( encrypt | bypass )",
# in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class ProfileL2ProtocolEthFlowControlCmd( CliCommand.CliCommandClass ):
   syntax = L2ProtocolEthFlowControlCmd.syntax
   noOrDefaultSyntax = L2ProtocolEthFlowControlCmd.noOrDefaultSyntax
   data = dict( L2ProtocolEthFlowControlCmd.data )
   handler = MacsecProfileMode.ethFlowControlEncrypt
   noOrDefaultHandler = MacsecProfileMode.defaultEthFlowControlEncrypt

MacsecProfileMode.addCommandClass( ProfileL2ProtocolEthFlowControlCmd )

#--------------------------------------------------------------------------------
# "( no | default ) replay protection disabled", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class ReplayProtectionDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'replay protection disabled'
   noOrDefaultSyntax = syntax
   data = {
      'replay': 'Replay protection',
      'protection': 'Replay protection',
      'disabled': 'Disable replay protection',
   }

   handler = MacsecProfileMode.replayProtectionDisabled
   noOrDefaultHandler = MacsecProfileMode.noReplayProtectionDisabled

MacsecProfileMode.addCommandClass( ReplayProtectionDisabledCmd )

#--------------------------------------------------------------------------------
# "( no | default ) replay protection window ...", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class ReplayProtectionWindowCmd( CliCommand.CliCommandClass ):
   syntax = 'replay protection window REPLAYPROTECTIONWINDOW'
   noOrDefaultSyntax = 'replay protection window ...'
   data = {
      'replay': 'Replay protection',
      'protection': 'Replay protection',
      'window': 'Replay protection window',
      'REPLAYPROTECTIONWINDOW': CliMatcher.IntegerMatcher( 0, 2**32-1,
                     helpdesc='Replay protection window size' ),
   }

   handler = MacsecProfileMode.replayProtectionWindow
   noOrDefaultHandler = MacsecProfileMode.noReplayProtectionWindow

MacsecProfileMode.addCommandClass( ReplayProtectionWindowCmd )

#--------------------------------------------------------------------------------
# "( no | default ) ptp bypass", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class PtpBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp bypass'
   noOrDefaultSyntax = syntax
   data = {
      'ptp' : 'Set PTP processing',
      'bypass' : 'Transmit and Receive without protection',
   }

   handler = MacsecProfileMode.ptpBypass
   noHandler = MacsecProfileMode.noPtpBypass
   defaultHandler = MacsecProfileMode.defaultPtpBypass

MacsecProfileMode.addCommandClass( PtpBypassCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] sci", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class SciCmd( CliCommand.CliCommandClass ):
   syntax = 'sci'
   noOrDefaultSyntax = syntax
   data = {
      'sci': 'Include secure channel identifier in data packets',
   }

   handler = MacsecProfileMode.includeSci
   noOrDefaultHandler = MacsecProfileMode.noIncludeSci

MacsecProfileMode.addCommandClass( SciCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] fips post failure forced", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class FipsPostFailureForced( CliCommand.CliCommandClass ):
   syntax = 'fips post failure forced'
   noOrDefaultSyntax = syntax
   data = {
      'fips': 'Federal Information Processing Standard',
      'post': 'Power on Self Test',
      'failure': 'Fail FIPS POST',
      'forced': 'Force a result',
   }

   handler = MacsecProfileMode.fipsPostFailureForced
   noOrDefaultHandler = MacsecProfileMode.noFipsPostFailureForced

if macsecToggle.toggleMacsecFipsPostFailureForcedEnabled():
   MacsecProfileMode.addCommandClass( FipsPostFailureForced )

#--------------------------------------------------------------------------------
# "[ no | default ] traffic unprotected allow", in "config-mac-sec-profile" mode
#--------------------------------------------------------------------------------
class TrafficPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic unprotected ( ( allow [ active-sak ] ) | drop )'
   noOrDefaultSyntax = 'traffic unprotected ...'
   data = {
      'traffic': 'Traffic Policy',
      'unprotected': 'Traffic without MAC security protection',
      'allow' : 'Allow transmit/receive of unprotected traffic',
      'active-sak' : 'Allow transmit/receive of encrypted traffic '
                     'using operational SAK and block otherwise',
      'drop' : 'Block transmit/receive of unprotected traffic',
   }

   handler = MacsecProfileMode.trafficPolicy
   noOrDefaultHandler = MacsecProfileMode.noOrDefaultTrafficPolicy

MacsecProfileMode.addCommandClass( TrafficPolicyCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] secure channel [ tx | rx ]",
# in "config-mac-sec-profile-<profile-name>-static-sak" mode
#--------------------------------------------------------------------------------
class StaticSakCmd( CliCommand.CliCommandClass ):
   syntax = 'secure channel TX_RX'
   noOrDefaultSyntax = syntax
   data = {
         'secure': 'Secure relationship',
         'channel': 'Secure channel',
         'TX_RX': CliMatcher.EnumMatcher( {
            'tx': 'Transmit',
            'rx': 'Receive',
         } ),
   }

   handler = MacsecStaticSakMode.gotoSecureChannelMode
   noOrDefaultHandler = MacsecStaticSakMode.noSecureChannel
MacsecStaticSakMode.addCommandClass( StaticSakCmd )

port = '([0-9]{1,5})'
pair = '([0-9a-fA-F]{1,2})'
idPattern = f'{pair}:{pair}:{pair}:{pair}:{pair}:{pair}::{port}$'

#--------------------------------------------------------------------------------
# "[ no | default ] identifier", in "config-mac-sec-profile-sc" mode
#--------------------------------------------------------------------------------
class StaticSakIdentifierCmd( CliCommand.CliCommandClass ):
   syntax = 'identifier IDENTIFIER'
   noOrDefaultSyntax = 'identifier'
   data = {
         'identifier': 'Secure Channel Identifier',
         'IDENTIFIER': CliMatcher.PatternMatcher( pattern=idPattern,
            helpname='H:H:H:H:H:H::P',
            helpdesc='MAC Address (hexadecimal) with port (decimal)' ),
   }

   handler = MacsecSecureChannelMode.addIdentifier
   noOrDefaultHandler = MacsecSecureChannelMode.noIdentifier

MacsecSecureChannelMode.addCommandClass( StaticSakIdentifierCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] an AN"
# "an AN key SECRET" commands, in "config-mac-sec-profile-sc" mode
# "an AN key SECRET [ salt SALT ]" for debugging
#--------------------------------------------------------------------------------
sakMatcher = CliMatcher.PatternMatcher(
      keyPattern_, helpname='SAK',
      helpdesc='Secure association key in hex octets' )
sakSecretCliExpr = ReversibleSecretCli.ReversiblePasswordCliExpression(
      cleartextMatcher=sakMatcher )
saltPattern_ = r'(^[a-fA-F0-9]{24}$)' # 24 hex chars == 96 bits
saltMatcher = CliMatcher.PatternMatcher(
      saltPattern_, helpname='SALT',
      helpdesc='Secure association salt in hex octets' )

class StaticSakKeyCmd( CliCommand.CliCommandClass ):
   if macsecToggle.toggleMacsecStaticSakNonXpnEnabled():
      syntax = 'an AN key SECRET [ salt SALT ]'
   else:
      syntax = 'an AN key SECRET'
   noOrDefaultSyntax = 'an AN ...'
   data = {
         'an': 'Association Number',
         'AN': CliMatcher.IntegerMatcher( 0, 3, helpdesc='Association number' ),
         'key': 'Static secure association key',
         'SECRET': sakSecretCliExpr,
   }
   if macsecToggle.toggleMacsecStaticSakNonXpnEnabled():
      data.update( {
         'salt' : 'Secure association salt in hex octets',
         'SALT' : saltMatcher,
   } )

   handler = MacsecSecureChannelMode.addKey
   noOrDefaultHandler = MacsecSecureChannelMode.noKey

MacsecSecureChannelMode.addCommandClass( StaticSakKeyCmd )
