# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
   IpGenericAddress,
   IpGenericAddrAndPort,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from IntfModels import Interface
from CliPlugin.MirrorOnDropCliLib import (
   IpProtoType,
   getProtocolStr,
   DropReasonCliMap,
)
import TableOutput

notRunningStr = "Mirror-on-Drop flow tracking is not active"
formatLeft = TableOutput.Format( justify="left" )
formatLeft.noPadLeftIs( True )
formatRight = TableOutput.Format( justify="right" )

class FlowModel( Model ):
   age = Int( help="Time in seconds since flow detection" )
   ingressIntf = Str( help="Flow ingress interface" )
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   ipProtocol = Enum( help="IP protocol", values=IpProtoType.attributes,
                      optional=True )
   ipProtocolNumber = Int( help="IP protocol number" )
   srcPort = Int( help="Source port" )
   dstPort = Int( help="Destination port" )
   emtStatus = Str( help="Exact match entry programming status" )
   dropReason = Str( help="Reason for drop" )

class CollectorModel( Model ):
   addrAndPort = Submodel( help="Collector IP address and port number",
                           valueType=IpGenericAddrAndPort )
   packetCount = Int( help="Total packets sent" )
   byteCount = Int( help="Total bytes sent" )
   stopCount = Int( help="Total stop reports sent" )
   startCount = Int( help="Total start reports sent" )

class ExporterModel( Model ):
   collectors = Dict( keyType=str, valueType=CollectorModel,
                      help="A mapping between collector and its statistics" )

class GroupModel( Model ):
   flows = List( valueType=FlowModel, help="List of flows" )

class DropReasonModel( Model ):
   unknownReason = Int( help="Unkown reason", optional=True )
   ipForwardingLookupMiss = Int( help="IP forwarding lookup miss", optional=True )
   ipVersionError = Int( help="IP version error", optional=True )
   ipTtl01 = Int( help="IP TTL 0/1", optional=True )
   ipMissingArp = Int( help="ARP unresolved", optional=True )
   ipMissingRoute = Int( help="IP missing route", optional=True )
   ipv4ChecksumError = Int( help="IPv4 checksum error", optional=True )
   ipv6UnspecifiedDestination = Int( help="IPv6 unspecified destination",
                                     optional=True )
   ipv6MulticastSource = Int( help="IPv6 multicast source",
                              optional=True )
   portNotVlanMember = Int( help="Port not VLAN member", optional=True )
   saEqualsDa = Int( help="Source address equals destination address",
                     optional=True )
   saMulticast = Int( help="Source address is multicast", optional=True )
   rpf = Int( help="reverse path forwarding", optional=True )
   mtuExceeded = Int( help="MTU exceeded", optional=True )
   hardwareError = Int( help="Hardware error", optional=True )
   saNotFound = Int( help="Source Address not found", optional=True )
   noForwardingAction = Int( help="No forwarding action", optional=True )
   mplsDisabled = Int( help="MPLS disabled", optional=True )
   ingressSourcePortFilter = Int( help="Ingress source port filter", optional=True )
   mplsTtl01 = Int( help="MPLS TTL 0/1", optional=True )
   mplsLabelLookupMiss = Int( help="MPLS label lookup miss", optional=True )
   ipVersionErrorPostMplsDecap = Int( help="IP version error after MPLS decap",
                                      optional=True )
   ipv4ChecksumErrorPostMplsDecap = Int( help="IPv4 checksum error after MPLS decap",
                                         optional=True )
   ipTtl01PostMplsDecap = Int( help="IP TTL 0/1 after MPLS decap", optional=True )
   ipForwardingLookupMissPostMplsDecap = Int(
               help="IP forwarding lookup miss after MPLS decap", optional=True )
   ipMissingArpPostMplsDecap = Int( help="ARP unresolved after MPLS decap",
                                    optional=True )
   ipMissingRoutePostMplsDecap = Int( help="IP missing route after MPLS decap",
                                      optional=True )
   ipv6UnspecifiedDestinationPostMplsDecap = Int(
               help="IPv6 unspecified destination after MPLS decap", optional=True )
   ipv6MulticastSourcePostMplsDecap = Int(
               help="IPv6 multicast source after MPLS decap", optional=True )
   ingressSpanningTreeFilter = Int(
      help="Ingress spanning tree filter", optional=True )
   packetBufferIngressLimit = Int( help="Packet buffer ingress limit",
                                   optional=True )
   packetBufferSharedPoolLimit = Int( help="Packet buffer shared pool limit",
                                      optional=True )
   packetBufferEgrPortSpSharedLimit = Int(
         help="Packet buffer egress port shared limit", optional=True )
   packetBufferEgrQueueSharedLimit = Int(
         help="Packet buffer egress queue shared limit", optional=True )
   packetBufferEgrWred = Int( help="Packet buffer egress WRED", optional=True )
   packetBufferGlobalRejects = Int( help="Packet buffer global reject",
                                    optional=True )
   packetBufferUnknown = Int( help="Packet buffer unknown", optional=True )
   packetBufferVoqSharedReject = Int( help="Packet buffer VOQ shared reject",
                                      optional=True )
   packetBufferVoqWredReject = Int( help="Packet buffer VOQ WRED reject",
                                    optional=True )
   packetBufferVoqLatency = Int( help="Packet buffer VOQ latency", optional=True )
   packetBufferVsqSharedReject = Int( help="Packet buffer VSQ shared reject",
                                      optional=True )
   packetBufferVsqWredReject = Int( help="Packet buffer VSQ WRED reject",
                                    optional=True )
   packetBufferQueueError = Int( help="Packet buffer queue error", optional=True )

class TrackerModel( Model ):
   groups = Dict( keyType=str, valueType=GroupModel,
                  help="A mapping between group name and group" )
   intfs = Dict( keyType=Interface, valueType=int,
                  help="A mapping between ingress interface and flow count" )
   exporters = Dict( keyType=str, valueType=ExporterModel,
                  help="A dictionary of exporters" )
   dropReasonCounters = Dict( keyType=Interface, valueType=DropReasonModel,
                  help='A mapping between interface and drop reason counters' )
   numFlows = Int( help="Total number of flows" )

class ModShowModel( Model ):
   trackers = Dict( keyType=str, valueType=TrackerModel,
                    help="A mapping between tracker name and tracker information" )
   _modelType = Enum( help="Model type to determine rendering options",
                     values=( 'interface', 'flowTable', 'collector', 'dropReason' ) )
   running = Bool( help="Mirror-on-Drop flow tracking is active" )

   def modelTypeIs( self, modelType ):
      self._modelType = modelType

   def renderFlowTable( self ):
      headings = ( "Age (s)", "Interface", "Source", "Destination", "Protocol",
                   "HW status", "Drop Reason" )
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( "Tracker: %s, Unique flows: %s\n" %
                ( trackerName, tracker.numFlows ) )

         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            table = TableOutput.createTable( headings )
            table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                                 formatLeft, formatLeft, formatLeft )
            print( "Group: %s, Flows: %s" % ( groupName, len( group.flows ) ) )

            template = '[%s]:%s' if '6' in groupName else '%s:%s'
            for flow in group.flows:
               table.newRow( flow.age, flow.ingressIntf,
                             template % ( flow.srcAddr, flow.srcPort ),
                             template % ( flow.dstAddr, flow.dstPort ),
                             getProtocolStr( flow.ipProtocol,
                                             flow.ipProtocolNumber ),
                             flow.emtStatus,
                             flow.dropReason )

            print( table.output() )

   def renderInterface( self ):
      headings = ( "Interface", "Flows" )
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( "Tracker: %s, Unique flows: %s\n" %
                ( trackerName, tracker.numFlows ) )
         table = TableOutput.createTable( headings )
         table.formatColumns( formatLeft, formatRight )

         for interface, count in sorted( tracker.intfs.items() ):
            table.newRow( interface, count )

         print( table.output() )

   def renderCollector( self ):
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( "Tracker: %s,  Unique flows: %s" %
               ( trackerName, tracker.numFlows ) )
         # xxx_haseeb % Flows created: 0, expired: 0 ???
         for exporterName, exporter in sorted( tracker.exporters.items() ):
            heading = 'Exporter: %s' % exporterName
            print( "%s\n%s" % ( heading, '-' * len( heading ) ) )

            for collector in exporter.collectors.values():
               print( "Collector: %s port %d" %
                      ( collector.addrAndPort.ip, collector.addrAndPort.port ) )
               print( "%d packets, %d bytes sent" %
                      ( collector.packetCount, collector.byteCount ) )
               print( "%d start, %d stop reports sent" %
                      ( collector.startCount, collector.stopCount ) )

   def renderDropReasonCounters( self ):
      for trackerName, tracker in sorted( self.trackers.items() ):
         if not tracker.dropReasonCounters:
            continue
         print( "" )
         print( "Tracker: ", trackerName )
         for interface in sorted( tracker.dropReasonCounters.keys() ):
            counters = tracker.dropReasonCounters[ interface ]
            if not counters:
               continue
            print( "" )
            print( "Interface: %s" % interface )
            for cliInfo in DropReasonCliMap.values():
               dropReasonKey = cliInfo.dropReasonCliKey
               if counters[ dropReasonKey ]:
                  dropReasonRenderStr = cliInfo.dropReasonCliStr
                  print( "%s : %d" %
                         ( dropReasonRenderStr, counters[ dropReasonKey ] ) )

   def render( self ):
      if not self.running:
         print( notRunningStr )
         return

      if self._modelType == 'interface':
         self.renderInterface()
      elif self._modelType == 'flowTable':
         self.renderFlowTable()
      elif self._modelType == 'collector':
         self.renderCollector()
      elif self._modelType == 'dropReason':
         self.renderDropReasonCounters()
      else:
         # Unsupported model
         assert False
