# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements MPLS configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for MPLS'''

import sys
import AgentCommandRequest
import Arnet.MplsLib
from Arnet.MplsLib import labelStackToMplsLabelOperation as LabelOp
import CliCommand
from CliModel import cliPrinted
from CliPlugin import MplsCli as Globals
from CliPlugin import MplsModel
from CliPlugin import TunnelModels
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.MplsModel import (
   LABEL_RANGE_STATIC,
   LABEL_RANGE_DYNAMIC,
   LABEL_RANGE_UNASSIGNED,
)
from CliPlugin.TunnelFibCli import getMplsViaModelFromTunnelVia
import CliPlugin.TunnelCli
from CliPlugin.TunnelCli import (
   getEndpointFromTunnelId,
   getTunnelIdFromIndex,
   getTunnelIndexFromId,
   getTunnelTypeFromTunnelId,
   getViaModels,
)
from IpLibConsts import DEFAULT_VRF
import LazyMount
import MplsLib
from MplsTypeLib import showTunnelFibIgnoredTunnelTypes
import Tac
import TacSigint
from Toggles.MplsToggleLib import (
   toggleLabelRangeStatusEnabled,
   toggleStaticMplsPopOCEnabled,
)
import Tracing
from TunnelLib import getTunnelViaStatusFromTunnelId
from TypeFuture import TacLazyType

th = Tracing.defaultTraceHandle()
t8 = th.trace8

# pkgdeps library RoutingLib

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
BoundedMplsLabelStack = TacLazyType( 'Arnet::BoundedMplsLabelStack' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
LabelRangeInfo = Tac.Type( 'Mpls::LabelRangeInfo' )
LfibHwProgrammingReturnCode = Tac.Type( 'Mpls::LfibHwProgrammingReturnCode' )
LfibSource = Tac.Type( "Mpls::LFib::Source" )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
MplsSpeculativeParsingConfig = Tac.Type( 'Mpls::SpeculativeParsingConfig' )
RouteKey = Tac.Type( 'Mpls::RouteKey' )
RouteKeyAndMetric = Tac.Type( 'Mpls::RouteKeyAndMetric' )
QosMap = TacLazyType( "QosLib::QosMap" )
StaticTunnelConfigEntry = TacLazyType( 'Tunnel::Static::StaticTunnelConfigEntry' )
TacDyTunIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
TtlMode = Tac.Type( 'Mpls::TtlMode' )
ViaIndex = Tac.Type( 'Mpls::ViaIndex' )
LfibStitchingProtocol = Tac.Type( 'Mpls::LfibStitchingProtocol' )
LfibSourceProtoPref = Tac.Type( 'Mpls::SourceProtoLfibStitchingPreference' )
LfibStitchingPreference = Tac.Type( 'Mpls::LfibStitchingPreference' )
eBgpLuProtocol = LfibStitchingProtocol.lfibStitchingProtocolEbgpLu
ldpProtocol = LfibStitchingProtocol.lfibStitchingProtocolLdp
ospfSrProtocol = LfibStitchingProtocol.lfibStitchingProtocolOspfSr
isisSrProtocol = LfibStitchingProtocol.lfibStitchingProtocolIsisSr
iBgpLuProtocol = LfibStitchingProtocol.lfibStitchingProtocolIbgpLu
lfibStitchingProtocol = frozenset( [ eBgpLuProtocol, ldpProtocol, ospfSrProtocol,
                                     isisSrProtocol, iBgpLuProtocol ] )

implicitNullLabel = MplsLabel.implicitNull
nullLabel = MplsLabel.null

def getForwardingStatus():
   unifiedMode = Tac.Type( 'Smash::Fib::FecMode' ).fecModeUnified
   if Globals.fecModeStatus.fecMode == unifiedMode:
      return( Globals.unifiedForwardingStatus, Globals.unifiedForwarding6Status,
              Globals.unifiedForwardingGenStatus )
   else:
      return( Globals.forwardingStatus, Globals.forwarding6Status,
              Globals.forwardingGenStatus )

def MplsIpCmd_handler( mode, args ):
   Globals.mplsRoutingConfig.mplsRouting = True

def MplsIpCmd_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.mplsRouting = False

nexthopResolutionConfigChangeWarning = \
   "Change will take effect only after switch reboot or forwarding agent restart."

def MplsNextHopResolutionAllowDefaultRouteCmd_handler( mode, args ):
   allow = not CliCommand.isNoOrDefaultCmd( args )
   Globals.mplsRoutingConfig.nexthopResolutionAllowDefaultRoute = allow
   mode.addWarning( nexthopResolutionConfigChangeWarning )

def MplsEntropyLabelPopCmd_handler( mode, args ):
   Globals.mplsRoutingConfig.entropyLabelPop = True

def MplsEntropyLabelPopCmd_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.entropyLabelPop = False

def MplsTunnelEntropyLabelPushCmd_handler( mode, args ):
   Globals.mplsTunnelConfig.entropyLabelPush = True

def MplsTunnelEntropyLabelPushCmd_noOrDefaultHandler( mode, args ):
   Globals.mplsTunnelConfig.entropyLabelPush = False

def validatePopViaPayload( l, newVia ):
   '''
   This function checks all the vias for a the label to see if there are any pop vias
   configured. If there are any then validates that the payload type is the same for
   all the pop vias. If not then it returns and error.
   In addition if there is a pop via configured with the same 
   nexthop, payload-type as the new incoming via, but a different option for the 
   access-list bypass parameter then that via is deleted. The new via is going to 
   replace the deleted via. eg.
   'mpls static top-label x 10.1.1.1 pop payload-type ipv4' then if the user 
   configures 'mpls static top-label x 10.1.1.1 pop payload-type ipv4 access-list 
   bypass' then the old via gets replaced with the new via configured.
   '''
   vias = l.via
   for via in vias:
      if via.labelAction == Globals.MplsLabelAction.pop:
         if newVia.payloadType != via.payloadType:
            # return error string indicating payload mismatch
            return ( "Cannot configure pop operations for the same label with"
                     " different payload type" )
         else:
            break

   viaToCheck = Tac.Value( "Mpls::Via",
                            nextHop=newVia.nextHop,
                            labelAction=newVia.labelAction,
                            outLabelStack=newVia.outLabelStack,
                            skipEgressAcl=not newVia.skipEgressAcl)
   viaToCheck.payloadType = newVia.payloadType
   viaToCheck.intf = newVia.intf
   indexToViaMap = (
      l.indexToPopVia if newVia.labelAction == 'pop' else l.indexToSwapVia )

   if viaToCheck in vias:
      # delete the via and related maps
      oldViaIndex = l.viaToIndex.get( viaToCheck )
      if oldViaIndex is not None:
         # we have some index association with existing via, delete them
         del l.viaToIndex[ viaToCheck ]
         del indexToViaMap[ oldViaIndex.index ]
      del vias[ viaToCheck ]

   return None

topLabelCountWarning = 'Maximum number of top-labels supported is {}'
labelStackSizeWarning = 'Maximum label stack size is {} labels'.format(
      Globals.MplsStackEntryIndex.max + 1 )

def getLSPNameFromRouteKeyMetric( routeKeyMetric ):
   # This is auto generated LSP name based on topLabel and metric
   routeKey = routeKeyMetric.routeKey
   metric = routeKeyMetric.metric
   lspName = ""
   # LSP name is not supported for multi-label routes yet.
   if routeKey.label1 != nullLabel:
      return lspName
   lspName = f"{routeKey.label0}_{metric}"
   return lspName

def handleMplsLabelConfig( mode, inLabels, via, bgpSession=False, bgpPeer=None,
                           lspName=None ):
   cliHelper = Globals.initCliHelper()
   intfNexthop = via[ 'intfNexthop' ]
   nexthop = intfNexthop.get( 'nexthop' )
   if ( nexthop is not None and
        not isinstance( nexthop, Tac.Type( 'Arnet::IpGenAddr' ) ) ):
      nexthop = Tac.Value( 'Arnet::IpGenAddr', nexthop )
   intf = intfNexthop.get( 'intf' )
   nexthopGroup = intfNexthop.get( 'nexthopGroup' )
   isBgpPeerMonitored = via.get( 'isBgpPeerMonitored', False )
   bgpPeer = via.get( 'bgpPeer', Tac.Value( 'Arnet::IpGenAddr', '' ) )
   # if viaIndex is not None then user have provided the value otherwise it
   # indicates no viaIndex given via CLI, therefore auto generated later.
   viaIndex = via.get( 'index' )
   isDynamicTunnel = False
   if isinstance( intf, str ):
      isDynamicTunnel = TacDyTunIntfId.isDynamicTunnelIntfId( intf )
   # You can set at most one of the following
   assert sum( [ int( nexthop is not None),
                 int( nexthopGroup is not None),
                 int( isDynamicTunnel ) ] ) == 1
   labelOp = via[ 'labelOp' ]
   metric = via[ 'metric' ]
   weight = via.get( 'weight', 1 )
   backup = via.get( 'backup' )
   error = None

   assert inLabels
   isMultiLabel = len( inLabels ) > 1

   staticLabelRange = Globals.labelRangeValue( LabelRangeInfo.rangeTypeStatic )
   base = staticLabelRange.base
   size = staticLabelRange.size
   minLabel = base
   maxLabel = base + size - 1
   for inLabel in inLabels:
      if inLabel < minLabel or inLabel > maxLabel:
         # Just display a warning and continue. In interactive configuration,
         # the plugin will not get here since the topLabelValMatcher will not
         # match out of range labels. This warning will just be displayed
         # in the startup config output, or during config-replace, etc.
         mode.addWarning( 'mpls static top-label ' + str( inLabel ) +
                          ' is outside of the mpls label range, base: ' +
                          str( base ) + ', size: ' + str( size ) )

   # Check for multi-label pop routes and display warnings if necessary
   if len( inLabels ) > Globals.MplsStackEntryIndex.max:
      mode.addWarning( labelStackSizeWarning )
      return

   labelStack = BoundedMplsLabelStack()
   for labelVal in inLabels:
      labelVal = int( labelVal )
      labelStack.prepend( labelVal )

   # During startup config, mplsLookupLabelMaxCount is not initialized, so we
   # check some arbitrary "large" number to ensure we aren't trying to configure
   # a gargantuan, non-sensical label stack. This is needed because we aren't
   # enforcing the number of labels we can inhale from 'mpls static top-label'
   if mode.session_.startupConfig():
      if labelStack.stackSize > Globals.maxIngressMplsTopLabels:
         return
   else:
      if labelStack.stackSize > Globals.mplsHwCapability.mplsLookupLabelMaxCount:
         mode.addWarning( topLabelCountWarning.format(
            Globals.mplsHwCapability.mplsLookupLabelMaxCount ) )
         return

   outLabel = None
   skipEgressAcl = False
   if metric is None:
      metric = Tac.Type( 'Mpls::Constants' ).metricDefault

   if labelOp is None:
      assert nexthopGroup or isMultiLabel, ( "Label action none is only allowed "
                                             "with forwarding into nexthop-group "
                                             "or static multi-label pop routes" )
      outLabel = implicitNullLabel
      action = 'forward'
      payloadType = 'undefinedPayload'
   elif 'swap' in labelOp:
      outLabel = labelOp[ 'swap' ][ 'outLabel' ]
      payloadType = 'mpls'
      action = 'swap'
      skipEgressAcl = True
      assert nexthopGroup is None
   elif 'pop' in labelOp:
      outLabel = implicitNullLabel
      action = 'pop'
      if labelOp[ action ] == 'mpls':
         payloadType = labelOp[ action ]
         skipEgressAcl = False
      else:
         payloadType, skipEgressAcl = labelOp[ 'pop' ]
   else:
      assert False, "Label op has to be swap, pop, or None"

   lspNameCli = ""
   if toggleStaticMplsPopOCEnabled():
      # Before adding the new route into the RouteConfigInput, first check for the
      # name association
      routeKey = RouteKeyAndMetric.fromLabelStack( labelStack, metric )
      if lspName is None:
         lspNameCli = getLSPNameFromRouteKeyMetric( routeKey )
      else:
         lspNameCli = lspName

      # If route has different LSP name for the same routeKeyAndMetric key
      # This case should be rejected therefore create a warning and return
      routeConfigInput = Globals.cliMplsRouteConfigReadOnly
      if routeKey in routeConfigInput.route:
         lspNamePresent = routeConfigInput.route[ routeKey ].lspName
         if lspNamePresent != lspNameCli:
            mode.addWarning( "can't configure same route with different name," +
                             " delete current route then configure with name" )
            return
      else:
         # This is new route with RouteKeyAndMetric key
         # First check if any other route has the same name as new route
         oldRouteKey = routeConfigInput.lspNameToRouteKeyAndMetric.get( lspNameCli )
         if oldRouteKey is not None:
            # first delete oldRouteKey and related mapping then add new route
            cliHelper.delMplsRouteByRouteKey( oldRouteKey )

   # otherwise add the the route
   l = cliHelper.addMplsRoute( labelStack, metric, lspNameCli )
   if lspName is not None:
      l.lspNameConfigured = True

   nexthop = nexthop or Tac.Value( 'Arnet::IpGenAddr' )
   via = Tac.Value( "Mpls::Via",
                    nextHop=nexthop,
                    labelAction=action,
                    outLabelStack=LabelOp( [ outLabel ], operation='unknown',
                                           unboundedMplsLabelOperation=True ),
                    skipEgressAcl=skipEgressAcl,
                    payloadType=payloadType )
   via.weight = weight
   if isDynamicTunnel:
      via.intf = intf
   elif intf:
      via.intf = intf.name
   if nexthopGroup:
      via.nexthopGroup = nexthopGroup

   via.isBgpPeerMonitored = isBgpPeerMonitored
   if isBgpPeerMonitored and bgpPeer:
      via.bgpPeer = bgpPeer
   
   maxEcmpSize = Tac.Type( "Mpls::Constants" ).maxEcmpSize
   if via not in l.via and len(l.via) >= maxEcmpSize:
      #return error back to the user
      mode.addError( "Maximum number of next hops supported is %d" % maxEcmpSize )
   else:
      #Globals.mplsHwCapability.mplsSkipEgressAclSupported = True
      if ( Globals.mplsHwCapability.mplsSkipEgressAclSupported and
           l.via and
           action == 'pop' ):
         error = validatePopViaPayload( l, via )
      if error is not None:
         # return the validation error
         mode.addError( error )
      else:
         # check if it is the same via and if doesn't conflict with the existing via
         # index
         if not backup:
            if toggleStaticMplsPopOCEnabled():
               handleViaIndexConflict( mode, l, via, viaIndex )
            else:
               l.via[ via ] = True
         else:
            if not l.via:
               mode.addError( "No primary vias for route with backup via" )
            else:
               l.backupVia[ via ] = True

def handleViaIndexConflict( mode, l, newVia, viaIndex ):
   """
   There are following cases with conflict where we need to reject or overwrite via
   1). If newVia is present in vias but with different index -> reject
   2). If newVia being added not present in the vias.
      i). But the index is present in the map of indexToVia for corresponding action.
         then ovewrite the via.
         a). get the old via ( say via1 from the map ) and delete it from the map
         b). remove it from the vias and add newVia
      ii). If no index is present in the map of indexToVia for corresponding action
         then make ECMP of vias.
   3). If no index is provided in the via then
      i). if via is present in vias -> NO-OP
      ii). if via is not present in vias then add it to the vias -> make ECMP of vias
   """
   indexToViaMap = (
      l.indexToPopVia if newVia.labelAction == 'pop' else l.indexToSwapVia )
   if viaIndex is not None:
      # Case 1) and Case 2)
      # If the via index is user provided then add it to the viaToIndex collection
      newViaIndex = ViaIndex( viaIndex, True )
      oldViaIndex = l.viaToIndex.get( newVia )
      if oldViaIndex is not None and viaIndex != oldViaIndex.index:
          # same via but have different index, then generate error
         mode.addError( "Same via with different index not allowed" )
         return
      if viaIndex in indexToViaMap:
         # If there is already one via with the same index then need to overwrite it
         # i.e. Update the map and delete old via and add new via.
         oldVia = indexToViaMap[ viaIndex ]
         if oldVia == newVia:
            # nothing to do, this via is exactly same as earlier along with index
            if oldViaIndex != newViaIndex:
               # updating the map because it might happen that earlier the via
               # got auto-generated index but now user has configured the same
               # index, so update the flag that it is user-configured.
               del l.viaToIndex[ oldVia ]
               l.viaToIndex[ newVia ] = newViaIndex
            return
         del l.viaToIndex[ oldVia ]
         del l.via[ oldVia ]
      # if we reached here then it means that we have done the conflict part and
      # now only two case needs to handle:
      # 1. same via with index came and earlier it doesn't have index, then give
      #    this index to it.
      # 2. or different via with different index came then add it via list.
      # 3. or different via with same index came then also add it to via list.
      indexToViaMap[ viaIndex ] = newVia
      l.viaToIndex[ newVia ] = newViaIndex
      l.via[ newVia ] = True
   else:
      # case 3)
      if newVia not in l.via:
         # if via is not present in l.via then add it to the l.via list i.e. ECMP
         l.via[ newVia ] = True
      # otherwise do nothing

def handleNoMplsLabelConfig( mode, inLabels, via ):
   cliHelper = Globals.initCliHelper()
   
   # if we get no labels , we will delete all entries
   if None in inLabels:
      cliHelper.delAllMplsRoute()
      return
   assert inLabels
   nullMetric = Tac.Type( "Mpls::RouteMetric" ).null
   skipEgressAcl = False

   if len( inLabels ) > Globals.MplsStackEntryIndex.max:
      mode.addWarning( labelStackSizeWarning )
      return

   labelStack = BoundedMplsLabelStack()
   for labelVal in inLabels:
      labelVal = int( labelVal )
      labelStack.prepend( labelVal )

   if via is not None:
      intfNexthop = via[ 'intfNexthop' ]
      nexthop = intfNexthop.get( 'nexthop' ) or ''
      if not isinstance( nexthop, Tac.Type( 'Arnet::IpGenAddr' ) ):
         nexthop = Tac.Value( 'Arnet::IpGenAddr', nexthop )
      intf = intfNexthop.get( 'intf' )
      intfName = ''
      if isinstance( intf, str ) and TacDyTunIntfId.isDynamicTunnelIntfId( intf ):
         intfName = intf
      elif intf:
         intfName = intf.name
      nexthopGroup = intfNexthop.get( 'nexthopGroup', '' )
      labelOp = via[ 'labelOp' ]
      metric = via[ 'metric' ] if via[ 'metric' ] is not None else nullMetric

      if labelOp is None:
         outLabel = implicitNullLabel
         action = 'forward'
         payloadType = 'undefinedPayload'
      elif 'swap' in labelOp:
         outLabel = labelOp[ 'swap' ][ 'outLabel' ]
         payloadType = 'mpls'
         action = 'swap'
         skipEgressAcl = True
      elif 'pop' in labelOp:
         outLabel = implicitNullLabel
         action = 'pop'
         if labelOp[ action ] == 'mpls':
            payloadType = labelOp[ action ]
            skipEgressAcl = False
         else:
            payloadType, skipEgressAcl = labelOp[ 'pop' ]
      else:
         assert False, "Label op has to be swap, pop, or None"

      weight = via.get( 'weight', 1 )
      isBgpPeerMonitored = via.get( 'isBgpPeerMonitored', False )
      bgpPeer = via.get( 'bgpPeer', Tac.Value( 'Arnet::IpGenAddr', '' ) )
      backup = bool( via.get( 'backup' ) )
      # get the index of via if any associated before deleting it so that we can
      # delete it from the indexToPopVia or indexToSwapVia collection
      # So there are following cases that we have to handle
      # case 1: if via has skipEgressAcl set and while deleting, skipEgressAcl is not
      #         provided by the user then also we need to delete the via and
      #         it's associated indexToViaMap.
      # case 2: if via has exactly same contents except index or indexConfigured flag
      #         then also we need to delete the via and it's indexToViaMap values
      #         if index is matching or no index is provided
      # case 3: if via has exactly same contents except index or indexConfigured flag
      #         but index is not exactly same then don't delete the via. NO-OP
      # get the index of via if any associated before deleting it so that we can
      # delete it from the indexToPopVia or indexToSwapVia collection
      # So there are following cases that we have to handle
      # case 1: if via has skipEgressAcl set and while deleting, skipEgressAcl is not
      #         provided by the user then also we need to delete the via and
      #         it's associated indexToViaMap.
      # case 2: if via has exactly same contents except index or indexConfigured flag
      #         then also we need to delete the via and it's indexToViaMap values
      #         if index is matching or no index is provided
      # case 3: if via has exactly same contents except index or indexConfigured flag
      #         but index is not exactly same then don't delete the via. NO-OP
      index = via.get( 'index' )
      if index is None:
         index = 0
         indexValid = False
      else:
         indexValid = True
      cliHelper.delMplsRouteVia( labelStack, metric, nexthop, nexthopGroup, action,
                                 outLabel, skipEgressAcl, payloadType,
                                 intfName, weight, isBgpPeerMonitored, bgpPeer,
                                 backup, index, indexValid )
   else:
      cliHelper.delMplsRoute( labelStack, nullMetric )

def MplsStaticRoutePopOrSwapCmd_getVia( args ):
   via = None
   addr = args.get( 'ADDR_ON_INTF' ) or args.get( 'ADDR' )
   if addr is not None:
      via = { 'intfNexthop': { 'intf': args.get( 'INTF' ),
                                 'nexthop': addr },
               'metric': args.get( 'METRIC' ) }
      via[ 'index' ] = args.get( 'NH_INDEX' )
      if 'pop' in args:
         via[ 'labelOp' ] = { 'pop': args[ 'PAYLOAD_TYPE_AND_BYPASS' ] }
      else:
         assert 'swap-label' in args
         via[ 'labelOp' ] = { 'swap': { 'outLabel': args[ 'OUT_LABEL' ] } }
      via[ 'weight' ] = args.get( 'WEIGHT', 1 )
      via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
      via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   return via

def MplsStaticRoutePopOrSwapCmd_handler( mode, args ):
   lspName = args.get( 'LSP_NAME' ) if toggleStaticMplsPopOCEnabled() else None
   handleMplsLabelConfig( mode, [ args[ 'IN_LABEL' ] ],
                          MplsStaticRoutePopOrSwapCmd_getVia( args ),
                          lspName=lspName )

def MplsStaticRoutePopOrSwapCmd_noOrDefaultHandler( mode, args ):
   handleNoMplsLabelConfig( mode, [ args.get( 'NO_IN_LABEL' ) ],
                            MplsStaticRoutePopOrSwapCmd_getVia( args ) )

def MplsLookupLabelCountCmd_handler( mode, args ):
   Globals.mplsRoutingConfig.mplsLookupLabelCount = args[ 'LABEL_COUNT' ]

def MplsLookupLabelCountCmd_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.mplsLookupLabelCount = 1

def MplsLookupFallbackIpCmd_handler( mode, args ):
   Globals.mplsRoutingConfig.mplsLookupFallbackIp = True

def MplsLookupFallbackIpCmd_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.mplsLookupFallbackIp = False

def MplsStaticMultiLabelPop_handler( mode, args ):
   # Relevant arguments.
   via = { 'intfNexthop': { 'intf': None,
                              'nexthop': args[ 'ADDR' ] },
            'metric': args.get( 'METRIC' ) }
   if 'pop' in args:
      payloadType = args.get( 'PTYPE', 'autoDecide' )
      skipEgressAcl = False
      via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
   else:
      via[ 'labelOp' ] = None
   via[ 'weight' ] = args.get( 'WEIGHT', 1 )
   via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
   via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   handleMplsLabelConfig( mode, args[ 'LABELS' ], via,
                           bgpSession=( 'bgp' in args ),
                           bgpPeer=args.get( 'BGP_PEER_ADDR' ) )

def MplsStaticMultiLabelPop_noOrDefaultHandler( mode, args ):
   # Relevant arguments.
   if 'ADDR' in args:
      via = { 'intfNexthop': { 'intf': None,
                                 'nexthop': args.get( 'ADDR' ) },
               'metric': args.get( 'METRIC' ) }
      if 'pop' in args:
         payloadType = args.get( 'PTYPE', 'autoDecide' )
         skipEgressAcl = False
         via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
      else:
         via[ 'labelOp' ] = None
      via[ 'weight' ] = args.get( 'WEIGHT', 1 )
      via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
      via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   else:
      via = None
   handleNoMplsLabelConfig( mode, args[ 'LABELS' ], via )

def MplsStaticTopLabelNexthopGroup_handler( mode, args ):
   # Relevant arguments.
   label = args[ 'LABEL' ]
   via = { 'intfNexthop': { 'intf': None,
                            'nexthop': None,
                            'nexthopGroup': args[ 'GROUP' ] },
           'metric': args.get( 'METRIC', None ) }
   via[ 'index' ] = args.get( 'NH_INDEX' )
   if 'pop' in args:
      payloadType = args.get( 'PTYPE', 'autoDecide' )
      via[ 'labelOp' ] = { 'pop': ( payloadType, False ) }
   else:
      via[ 'labelOp' ] = None
   via[ 'weight' ] = args.get( 'WEIGHT', 1 )
   via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
   via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   lspName = args.get( 'LSP_NAME' ) if toggleStaticMplsPopOCEnabled() else None
   handleMplsLabelConfig( mode, [ label ], via,
                          bgpSession=( 'bgp' in args ),
                          bgpPeer=args.get( 'BGP_PEER_ADDR' ),
                          lspName=lspName )

def MplsStaticTopLabelNexthopGroup_noOrDefaultHandler( mode, args ):
   # Relevant arguments.
   label = args[ 'LABEL' ]
   nexthopGroup = args.get( 'GROUP' )
   if nexthopGroup:
      via = { 'intfNexthop': { 'intf': None,
                               'nexthop': None,
                               'nexthopGroup': nexthopGroup },
               'metric': args.get( 'METRIC', None ) }
      via[ 'index' ] = args.get( 'NH_INDEX' )
      if 'pop' in args:
         payloadType = args.get( 'PTYPE', 'autoDecide' )
         via[ 'labelOp' ] = { 'pop' : ( payloadType, False ) }
      else:
         via[ 'labelOp' ] = None
      via[ 'weight' ] = args.get( 'WEIGHT', 1 )
      via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
      via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   else:
      via = None
   handleNoMplsLabelConfig( mode, [ label ], via )

def MplsStaticMultiLabelNexthopGroup_handler( mode, args ):
   # Relevant arguments.
   via = { 'intfNexthop': { 'intf': None,
                              'nexthop': None,
                              'nexthopGroup': args[ 'GROUP' ] },
            'metric': args.get( 'METRIC', None ) }
   if 'pop' in args:
      payloadType = args.get( 'PTYPE', 'autoDecide' )
      skipEgressAcl = False
      via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
   else:
      via[ 'labelOp' ] = None
   via[ 'weight' ] = args.get( 'WEIGHT', 1 )
   via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
   via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   handleMplsLabelConfig( mode, args[ 'LABELS' ], via,
                          bgpSession=( 'bgp' in args ),
                          bgpPeer=args.get( 'BGP_PEER_ADDR' ) )

def MplsStaticMultiLabelNexthopGroup_noOrDefaultHandler( mode, args ):
   # Relevant arguments.
   nexthopGroup = args.get( 'GROUP' )
   if nexthopGroup:
      via = { 'intfNexthop': { 'intf': None,
                                 'nexthop': None,
                                 'nexthopGroup': nexthopGroup },
               'metric': args.get( 'METRIC', None ) }
      if 'pop' in args:
         payloadType = args.get( 'PTYPE', 'autoDecide' )
         skipEgressAcl = False
         via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
      else:
         via[ 'labelOp' ] = None
      via[ 'weight' ] = args.get( 'WEIGHT', 1 )
      via[ 'isBgpPeerMonitored' ] = args[ 'isBgpPeerMonitored' ]
      via[ 'bgpPeer' ] = args[ 'bgpPeer' ]
   else:
      via = None
   handleNoMplsLabelConfig( mode, args[ 'LABELS' ], via )

def MplsStaticMulticastConfigRoute_handler( mode, args ):
   inLabel = args[ 'LABEL' ]
   childMode = mode.childMode( Globals.StaticMulticastMode, inLabel=inLabel )
   mode.session_.gotoChildMode( childMode )

def MplsStaticMulticastConfigRoute_noOrDefaultHandler( mode, args ):
   del Globals.staticMcastLfib.mplsViaSet[ args[ 'LABEL' ] ]

def MplsDisableStaticFecSharing_handler( mode, args ):
   Globals.mplsRoutingConfig.optimizeStaticRoutes = False

def MplsDisableStaticFecSharing_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.optimizeStaticRoutes = True

def MplsDisableIsisSrFecSharing_handler( mode, args ):
   Globals.mplsRoutingConfig.optimizeIsisSrLfibRoutes = False

def MplsDisableIsisSrFecSharing_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.optimizeIsisSrLfibRoutes = True

def MplsStaticMulticastRouteConfigVia_handler( mode, args ):
   nhAddr = args[ 'ADDR' ]
   outLabel = args[ 'LABEL' ]
   if ( nhAddr, outLabel ) not in mode.vias:
      mode.viasUpdated = True
      mode.vias.add( ( nhAddr, outLabel ) )

def MplsStaticMulticastRouteConfigVia_noOrDefaultHandler( mode, args ):
   nhAddr = args[ 'ADDR' ]
   outLabel = args[ 'LABEL' ]
   if ( nhAddr, outLabel ) in mode.vias:
      mode.viasUpdated = True
      mode.vias.remove( ( nhAddr, outLabel ) )

VrfLabel = TacLazyType( 'Mpls::VrfLabel' )

def MplsStaticVrfLabelCmd_handler( mode, args ):
   # Instantiate a via which contains info for vrf-name
   vrfLabel = VrfLabel( args[ 'IN_LABEL' ], args[ 'VRF_NAME' ] )

   # add it to the sysdb
   Globals.mplsVrfLabelConfig.vrfLabel.addMember( vrfLabel )

def MplsStaticVrfLabelCmd_noOrDefaultHandler( mode, args ):
   del Globals.mplsVrfLabelConfig.vrfLabel[ args[ 'NO_IN_LABEL' ] ]

def getNexthopGroupIdToNameMap( cEm ):
   nexthopGroupMap = Tac.newInstance( 'NexthopGroup::NexthopGroupMap', 'NhgMap' )
   _ = Tac.newInstance(
      'NexthopGroup::NexthopGroupNameIdMappingHelper', cEm, nexthopGroupMap,
      Tac.Type( 'TacSmash::TacSmashType' ).reader )
   return nexthopGroupMap

def showMplsLfibRoute( mode, args, detail=False ):
   Globals.showMplsConfigWarnings( mode )
   typeFilter = args.get( 'TYPE' )
   segmentFilter = args.get( 'segmentType' )
   labels = args.get( 'LABELS' )
   prefix = args.get( 'PREFIX' )
   flexAlgoFilter = args.get( 'ALGO-NAME' )
   if prefix is not None:
      prefixFilter = Arnet.IpGenPrefix( str( prefix ) )

   LazyMount.force( Globals.gribiAfts )
   LazyMount.force( Globals.lfibInfo )
   LazyMount.force( Globals.mplsLfibSourceInfoDir )
   LazyMount.force( Globals.mldpOpaqueValueTable )
   LazyMount.force( Globals.cliChurnTestHelper )
   LazyMount.force( Globals.flexAlgoConfig )
   LazyMount.force( Globals.protocolTunnelNameStatus )
   if not Globals.transitLfib or not Globals.lfibInfo:
      return cliPrinted( MplsModel.MplsRoutes() )

   capiRevision = mode.session_.requestedModelRevision()

   labelStack = Arnet.MplsLib.labelListToMplsLabelStack( labels or [] )

   DisplayFilter = Tac.Type( "Mpls::DisplayFilter" )
   SourceProto = Tac.Type( "Mpls::SourceProto" )
   SegmentType = Tac.Type( "Mpls::SegmentType" )
   segmentType = SegmentType.allSegment
   sourceProto = SourceProto.allSource

   if 'gribi' in args:
      sourceProto = SourceProto.gribiSource
   elif 'bgp' in args:
      sourceProto = SourceProto.bgpSource
   elif 'static' in args:
      sourceProto = SourceProto.staticSource
   elif 'rsvp' in args:
      sourceProto = SourceProto.rsvpSource
   elif 'ldp' in args:
      sourceProto = SourceProto.ldpSource
   elif 'mldp' in args:
      sourceProto = SourceProto.mldpSource
   elif 'isis' in args:
      if segmentFilter:
         if segmentFilter == 'prefix-segment':
            segmentType = SegmentType.prefixSegment
         elif segmentFilter == 'adjacency-segment':
            segmentType = SegmentType.adjacencySegment
         # elif algoFilter will be present
      sourceProto = SourceProto.isisSource
   elif 'traffic-engineering' in args:
      sourceProto = SourceProto.teSource
   if typeFilter:
      if typeFilter == 'mpls':
         dispFilter = DisplayFilter.mplsFilter
      if typeFilter == 'evpn':
         dispFilter = DisplayFilter.evpnFilter
      if typeFilter == 'pseudowire':
         dispFilter = DisplayFilter.pseudowireFilter
   else:
      dispFilter = DisplayFilter.all

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   helper = Tac.newInstance( "Mpls::MplsRouteHelper", labelStack, capiRevision,
                             Globals.mplsHwCapability.mplsSupported )
   helper.lfibInfo = Globals.lfibInfo
   helper.mplsLfibSourceInfoDir = Globals.mplsLfibSourceInfoDir
   helper.forwardingStatus, helper.forwarding6Status, helper.forwardingGenStatus = \
      getForwardingStatus()
   helper.churnTestHelper = Globals.cliChurnTestHelper
   helper.srteForwardingStatus = Globals.srteForwardingStatus
   helper.srTePolicyStatus = Globals.srTePolicyStatus
   helper.mplsAft = Globals.gribiAfts.mplsAft
   helper.mldpOpaqueValueTable = Globals.mldpOpaqueValueTable
   helper.pwRouteHelper = Globals.pwRouteHelper
   helper.tunnelFib = Globals.tunnelFib
   helper.rsvpLerTunnelTable = Globals.rsvpLerTunnelTable
   nexthopGroupMap = getNexthopGroupIdToNameMap( Globals.em.cEntityManager() )
   helper.nexthopGroupMap = nexthopGroupMap
   helper.flexAlgoConfig = Globals.flexAlgoConfig
   helper.protocolTunnelNameStatus = Globals.protocolTunnelNameStatus
   helper.sourceProtoFilter = sourceProto
   helper.segmentTypeFilter = segmentType
   helper.lfibVskOverrideConfig = Globals.lfibVskOverrideConfig
   helper.lfibViaSetStatus = Globals.lfibViaSetStatus
   if prefix is not None:
      helper.prefixFilter = prefixFilter
   if flexAlgoFilter is not None:
      helper.flexAlgoFilter = flexAlgoFilter

   helper.render( fd, fmt, Globals.pseudowireLfib, Globals.transitLfib,
                  Globals.decapLfib, Globals.decapLfibHw,
                  Globals.evpnEthernetSegmentLfib, Globals.evpnEthernetSegmentLfibHw,
                  None, dispFilter, True, detail )
   return cliPrinted( MplsModel.MplsRoutes )

def getIpIntfStatus( intf ):
   ipIntfStatus = Globals.ipStatus.ipIntfStatus.get( intf )
   if not ipIntfStatus:
      ipIntfStatus = Globals.ip6Status.intf.get( intf )

   return ipIntfStatus

def intfContainsNextHop( ipIntfStatus, mplsViaNextHop ) -> bool:
   if 'addr' in ipIntfStatus.attributes: # intf with ipv6 addr
      for addr in ipIntfStatus.addr:
         if not addr.address.isLinkLocal:
            return addr.contains( mplsViaNextHop.v6Addr )
      return False
   else:
      return ipIntfStatus.activeAddrWithMask.contains( mplsViaNextHop.v4Addr )

def mplsConfigRouteIntfUp( ipIntfStatus, mplsVia, mplsRouteKey ) -> bool:
   """`show mpls config route` should display status `up` for an mpls route that is 
   configured using an interface if:
   1) the route is correctly configured in hardware (the l2 via mac addr for the 
   route is not equal to ethernet address zero)
   2) the interface is active and its IP address is in the same subnet as the mpls
   next hop
   """
   mplsHwStatus = Globals.mplsHwStatus
   intf = mplsVia.intf
   arpResolved = False
   mplsHwRoute = mplsHwStatus.route.get( mplsRouteKey )
   if mplsHwRoute:
      adj = mplsHwStatus.adjacency.get( mplsHwRoute.adjBaseKey.index )
   else:
      return False
   # find the mpls hardware via that uses the interface
   for mplsHwVia in adj.via:
      if mplsHwVia.l2Via.l2Intf == intf:
         if not Arnet.EthAddr( mplsHwVia.l2Via.macAddr ).isZero:
            arpResolved = True
            break

   return arpResolved and intfContainsNextHop( ipIntfStatus, mplsVia.nextHop )

def showMplsConfigRoute( mode, labels ):
   Globals.showMplsConfigWarnings( mode )
   LazyMount.force( Globals.l3NHResolver )
   LazyMount.force( Globals.bgpPeerInfoStatusDir )
   LazyMount.force( Globals.ipStatus )
   LazyMount.force( Globals.mplsHwStatus )

   labelStack = Arnet.MplsLib.labelListToMplsLabelStack( labels or [] )

   if labels:
      routeKeys = [ RouteKeyAndMetric.fromKey( RouteKey( labelStack ), i )
                    for i in range(
                     Globals.RouteMetric.min, Globals.RouteMetric.max + 1 ) ]
   else:
      routeKeys = list( Globals.mplsRouteConfig.route )

   routeDict = {}
   multiLabelRouteDict = None
   for rk in sorted( routeKeys ):
      route = Globals.mplsRouteConfig.route.get( rk )
      if route is None:
         continue

      # lookup LFIB and Ira RouteResolver to identify if route or its vias are down
      lfibRoute = Globals.transitLfib.lfibRoute
      lRoute = lfibRoute.get( rk.routeKey )
      if lRoute is None or lRoute.source != LfibSource.lfibSourceStaticMpls:
         rkStatus = "down"
      else:
         rkStatus = "up"
      nhvs = Globals.l3NHResolver.vrf.get( DEFAULT_VRF )
      peerStatusTable = ( Globals.bgpPeerInfoStatusDir.get( DEFAULT_VRF )
                          if Globals.bgpPeerInfoStatusDir else None )
      peerStatusEntry = ( peerStatusTable.bgpPeerInfoStatusEntry
                          if peerStatusTable else None )

      # Update route/via with lsp-name if its not auto-generated
      lspName = "-"
      if route.lspNameConfigured:
         lspName = route.lspName

      vias = [ ]
      viaSet = route.via
      for via in viaSet:
         viaModel = MplsModel.MplsVia()
         viaModel.nexthop = via.nextHop
         viaModel.intf = via.intf
         viaModel.labelAction = via.labelAction
         viaModel.outLabel = via.outLabelStack.topLabel()
         outLabels = []
         if via.labelAction == 'swap':
            for i in range( via.outLabelStack.stackSize ):
               outLabels.append( via.outLabelStack.labelStack( i ) )
         viaModel.outLabels = outLabels
         viaModel.payloadType = via.payloadType \
                                 if via.labelAction == 'pop' else 'mpls'
         viaModel.skipEgressAcl = via.skipEgressAcl
         viaModel.ttlMode = via.ttlMode
         viaModel.dscpMode = via.dscpMode
         viaModel.weight = via.weight
         bgpPeerStatus = 'up'
         if via.isBgpPeerMonitored:
            viaModel.bgpPeer = via.bgpPeer if via.bgpPeer else via.nextHop
            peerStatus = ( peerStatusEntry.get( viaModel.bgpPeer )
                           if peerStatusEntry else None )
            bgpPeerStatus = 'down' if ( not peerStatus or
               peerStatus.bgpPeerSessionStatus.bgpState != 'Established' ) else 'up'

         # viaStatus represents whether the next-hop is L3 resolvable.
         # For regular nexthop, lookup it in Nexthop Resolver if the it has been
         # resolved.
         viaStatus = rkStatus
         if via.intf:
            ipIntfStatus = getIpIntfStatus( via.intf )
            # if the configured intf is active and in the same subnet as the next hop
            # display status as up
            if ipIntfStatus and mplsConfigRouteIntfUp( ipIntfStatus, via,
                                                       rk.routeKey ):
               viaStatus = 'up'
            else:
               viaStatus = 'down'
         else:
            # interface is not specified for the nexthop, check that the nexthop is
            # resolvable in the Nexthop Resolver
            if nhvs is None or via.nextHop not in nhvs.nexthop:
               viaStatus = 'down'

         # For linklocal V6 address, the L3 viaStatus is up whenever the linklocal
         # interface is up.
         if viaStatus == "down" and via.nextHop.isV6LinkLocal:
            ipIntf = Globals.ip6Status.intf.get( via.intf )
            if ipIntf is not None and \
                  ipIntf.linkLocalAddrWithMask().address == via.nextHop.v6Addr:
               viaStatus = "up"
         viaModel.status = "up" if ( viaStatus == "up" and
                                     bgpPeerStatus == "up" ) else "down"

         # Update via index if its configured
         viaIndex = route.viaToIndex.get( via )
         if viaIndex is not None and viaIndex.indexConfigured:
            viaModel.index = viaIndex.index

         vias.append( viaModel )

      adjacencyModel = MplsModel.MplsAdjacency(
            viaType=MplsModel.LfibViaType.viaTypeMplsUnknown,
            mplsVias=vias,
            lspName=lspName )
      if rk.routeKey.labelStack.stackSize < 2:
         curRouteDict = routeDict
         dictKey = rk.topLabel
         topLabel = dictKey
         topLabels = None
      else:
         if multiLabelRouteDict is None:
            multiLabelRouteDict = {}
         curRouteDict = multiLabelRouteDict
         dictKey = rk.routeKey.labelStack.cliShowString()
         topLabel = MplsLabel.null
         topLabels = dictKey
      routeModel = curRouteDict.get( dictKey, None )
      if routeModel:
         routeModel.adjacencies[ rk.metric ] = adjacencyModel
      else:
         routeModel = MplsModel.MplsRoute( topLabel=topLabel, topLabels=topLabels,
                                         adjacencies={ rk.metric : adjacencyModel } )
         curRouteDict[ dictKey ] = routeModel
   return MplsModel.MplsRoutes( routes=routeDict,
                                multiLabelRoutes=multiLabelRouteDict )

def defaultLabelRange( rangeType ):
   return MplsLib.labelRangeDefault( Globals.mplsRoutingConfig, rangeType )

def checkConflictingLabelRangeForMplsStatic( rangeType, base, size,
                                             conflictingConfigList ):
   if rangeType != LabelRangeInfo.rangeTypeStatic or \
      not Globals.cliMplsRouteConfig.route:
      return
   confMin = MplsLabel.max
   confMax = MplsLabel.min
   for routeKey in Globals.cliMplsRouteConfig.route:
      if routeKey.topLabel < confMin:
         confMin = routeKey.topLabel
      if routeKey.topLabel > confMax:
         confMax = routeKey.topLabel

   if confMin != MplsLabel.max and confMax != MplsLabel.min:
      staticBase = confMin
      staticSize = confMax - confMin + 1
      if confMin < base or confMax >= base + size:
         conflictingConfigList.append(
                'static MPLS routes (existing range is %d, size %d).' %
                  ( staticBase, staticSize ) )

def CfgMplsLabelRange_rangeOverlap( rangeX, rangeY ):
   if rangeX[ 1 ] == 0 or rangeY[ 1 ] == 0:
      return False
   # Convert ( base, size ) to inclusive ranges [ label1, label2 ] for x,y
   x = ( rangeX[ 0 ], rangeX[ 0 ] + rangeX[ 1 ] - 1 )
   y = ( rangeY[ 0 ], rangeY[ 0 ] + rangeY[ 1 ] - 1 )
   return x[ 0 ] <= y[ 1 ] and y[ 0 ] <= x[ 1 ]

def CfgMplsLabelRange_allowOverlap( *args ):
   allowedOverlapList = ( LabelRangeInfo.rangeTypeSrgb,
                           LabelRangeInfo.rangeTypeBgpSrgb,
                           LabelRangeInfo.rangeTypeOspfSrgb )
   return all( arg in allowedOverlapList for arg in args )

def CfgMplsLabelRange_checkForRangeError( mode, rangeType, base, size ):
   if not mode.session_.guardsEnabled():
      return False

   others = ( set( Globals.mplsRoutingConfig.labelRangeDefault.keys() ) -
              { rangeType } )
   for otherName in others:
      other = Globals.labelRangeValue( otherName )
      if ( CfgMplsLabelRange_rangeOverlap( ( base, size ),
                                             ( other.base, other.size ) ) and
            not CfgMplsLabelRange_allowOverlap( otherName, rangeType ) ):
         mode.addError( 'Label range %s overlaps with mpls label range %s' % (
                                                ( rangeType, otherName ) ) )
         return True

   conflictingConfigList = []
   Globals.checkConflictingLabelRangeHook.notifyExtensions( rangeType, base, size,
                                                      conflictingConfigList )
   if conflictingConfigList:
      for config in conflictingConfigList:
         mode.addError( 'Label range needs to encompass existing %s' % config )
      return True
   return False

def CfgMplsLabelRange_saveRange( rangeType, base, size ):
   value = Tac.Value( 'Mpls::LabelRange', base, size )
   Globals.mplsRoutingConfig.labelRange[ rangeType ] = value
   Globals.handleUpdatedLabelRangeHook.notifyExtensions( rangeType, value )

def CfgMplsLabelRange_handler( mode, args ):
   base = args[ 'BASE' ]
   size = args[ 'SIZE' ]
   rangeType = args[ 'RTYPE' ]
   force = 'force' in args
   if size > MplsLabel.max - base + 1:
      mode.addError( 'Label range base + size must be less than %d' %
                     ( MplsLabel.max + 1 ) )
      return

   if ( rangeType == LabelRangeInfo.rangeTypeDynamic and
         size != 0 and size < LabelRangeInfo.dynamicMinSize and
         mode.session_.guardsEnabled() and not force ):
      mode.addError( 'Dynamic label range size must be at least %d' %
                     LabelRangeInfo.dynamicMinSize )
      return

   # check for overlap
   if CfgMplsLabelRange_checkForRangeError( mode, rangeType, base, size ):
      return
   CfgMplsLabelRange_saveRange( rangeType, base, size )

def CfgMplsLabelRange_noOrDefaultHandler( mode, args ):
   rangeType = args[ 'RTYPE' ]
   rangeDefault = defaultLabelRange( rangeType )
   base = rangeDefault.base
   size = rangeDefault.size
   # check for overlap
   if CfgMplsLabelRange_checkForRangeError( mode, rangeType, base, size ):
      return
   CfgMplsLabelRange_saveRange( rangeType, base, size )

def IntfMplsIpCmd_handler( mode, args ):
   disable = CliCommand.isNoCmd( args )
   intf = mode.intf.name
   if disable:
      Globals.mplsRoutingConfig.mplsRoutingDisabledIntf[ intf ] = True
   else:
      del Globals.mplsRoutingConfig.mplsRoutingDisabledIntf[ intf ]

def ShowMplsLfibRouteCmd_handler( mode, args ):
   return showMplsLfibRoute( mode, args, detail='detail' in args )

def ShowMplsConfigRouteCmd_handler( mode, args ):
   return showMplsConfigRoute( mode, args.get( 'LABELS' ) )

def showMplsHwRoute( mode, labels, typeFilter ):
   Globals.showMplsConfigWarnings( mode )

   LazyMount.force( Globals.mplsHwStatus )
   LazyMount.force( Globals.cliChurnTestHelper )
   LazyMount.force( Globals.protocolTunnelNameStatus )
   if not Globals.mplsHwStatus:
      return MplsModel.MplsHwStatus( mplsHwStatus=[] )

   labelStack = Arnet.MplsLib.labelListToMplsLabelStack( labels or [] )

   DisplayFilter = Tac.Type( "Mpls::DisplayFilter" )

   if typeFilter:
      if typeFilter == 'mpls':
         dispFilter = DisplayFilter.mplsFilter
      if typeFilter == 'evpn':
         dispFilter = DisplayFilter.evpnFilter
      if typeFilter == 'pseudowire':
         dispFilter = DisplayFilter.pseudowireFilter
   else:
      dispFilter = DisplayFilter.all

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   capiRevision = mode.session_.requestedModelRevision()
   helper = Tac.newInstance( "Mpls::MplsRouteHelper", labelStack, capiRevision,
                             Globals.mplsHwCapability.mplsSupported )
   helper.mplsHwStatus = Globals.mplsHwStatus
   helper.forwardingStatus, helper.forwarding6Status, helper.forwardingGenStatus = \
      getForwardingStatus()
   helper.srteForwardingStatus = Globals.srteForwardingStatus
   helper.srTePolicyStatus = Globals.srTePolicyStatus
   helper.tunnelFib = Globals.tunnelFib
   helper.churnTestHelper = Globals.cliChurnTestHelper
   nexthopGroupMap = getNexthopGroupIdToNameMap( Globals.em.cEntityManager() )
   helper.nexthopGroupMap = nexthopGroupMap
   helper.protocolTunnelNameStatus = Globals.protocolTunnelNameStatus
   helper.lfibVskOverrideConfig = Globals.lfibVskOverrideConfig
   helper.lfibViaSetStatus = Globals.lfibViaSetStatus

   with TacSigint.immediateMode():
      helper.render( fd, fmt, Globals.pseudowireLfib, Globals.transitLfib,
                     Globals.decapLfib, Globals.decapLfibHw,
                     Globals.evpnEthernetSegmentLfib,
                     Globals.evpnEthernetSegmentLfibHw,
                     None, dispFilter, False, False )
   return MplsModel.MplsHwStatus

def ShowMplsRouteCmd_handler( mode, args ):
   return showMplsHwRoute( mode, args.get( 'LABELS' ), args.get( 'TYPE' ) )

def showMplsHwRouteSummary( mode, args ):
   Globals.showMplsConfigWarnings( mode )

   numUnprogrammedLabels = numLabels = numHwAdjs = numBackupAdjs = 0
   if Globals.mplsHwStatus:
      numLabels = 0
      l2Adjs = set()
      backupAdjs = set()
      for r in Globals.mplsHwStatus.route.values():
         adj = Globals.mplsHwStatus.adjacencyBase( r.adjBaseKey )
         if not adj:
            # If an adj does not exist for the route, then we ignore the route,
            # as is done in the other show commands.
            continue
         numLabels += 1
         if r.unprogrammed or adj.unprogrammed:
            numUnprogrammedLabels += 1
         if adj.l2Adjacency:
            l2Adjs.add( adj.l2Adjacency )
      numHwAdjs = len( l2Adjs )
      # there will not be any backup adjs for now, preserve the show commands
      # for backward compatibility
      numBackupAdjs = len( backupAdjs )

   numLabelsSet = set()
   numUnprogrammedLabelsSet = set()
   numHwAdjsSet = set()
   if Globals.decapLfib and Globals.decapLfibHw:
      numLabelsSet.update( r.topLabel for r in Globals.decapLfib.lfibRoute )
      for r in Globals.decapLfibHw.unprogrammedRoute.values():
         if r.unprogrammed:
            numUnprogrammedLabelsSet.add( r.key.topLabel )
         else:
            numHwAdjsSet.add( Globals.decapLfib.lfibRoute[ r.key ].viaSetKey )

   if Globals.evpnEthernetSegmentLfib and Globals.evpnEthernetSegmentLfibHw:
      numLabelsSet.update(
         r.topLabel for r in Globals.evpnEthernetSegmentLfib.lfibRoute )
      for r in Globals.evpnEthernetSegmentLfibHw.unprogrammedRoute.values():
         if r.unprogrammed:
            numUnprogrammedLabelsSet.add( r.key.topLabel )
         else:
            numHwAdjsSet.add(
               Globals.evpnEthernetSegmentLfib.lfibRoute[ r.key ].viaSetKey )

   numLabels += len( numLabelsSet )
   numUnprogrammedLabels += len( numUnprogrammedLabelsSet )
   numHwAdjs += len( numHwAdjsSet )
   
   return MplsModel.MplsHwStatusSummary( numLabels=numLabels,
                                         numUnprogrammedLabels=numUnprogrammedLabels,
                                         numHwAdjs=numHwAdjs,
                                         numBackupAdjs=numBackupAdjs )

invalidEndpointErrMsg = 'Invalid endpoint address: neither IPv4 nor IPv6.'
endpointNextHopAfErrMsg = ( 'Endpoint and nexthop addresses must belong to same'
                            ' address family.' )
def cliDisplayOrderLabelList( mplsLabelStack ):
   """Converts a BoundedMplsLabelStack to a list of labels (int) in the order that
   they should be shown in show commands.

   BoundedMplsLabelStack reserves label index 0 for the bottom of the stack, however
   the CLI generally displays labels with the outermost label on the wire
   (top of stack) on the left, so the list representation for that purpose is in
   reverse order when compared to BoundedMplsLabelStack.
   """
   return [ mplsLabelStack.label( i )
            for i in range( mplsLabelStack.stackSize - 1, -1, -1 ) ]

def copyStaticTunnelConfigEntry( staticTunnelConfigEntry ):
   if staticTunnelConfigEntry is None:
      return None

   newConfigEntry = StaticTunnelConfigEntry( staticTunnelConfigEntry.name )
   newConfigEntry.tep = staticTunnelConfigEntry.tep
   # copy existing primary vias, if any
   for via in staticTunnelConfigEntry.via:
      newConfigEntry.via[ via ] = True
   # copy existing backup via
   newConfigEntry.backupVia = staticTunnelConfigEntry.backupVia
   newConfigEntry.inStaticTunnelMode = staticTunnelConfigEntry.inStaticTunnelMode

   return newConfigEntry

def getMplsTunnelNhTepAndIntfId( mode, nexthop, endpoint, intf ):
   nexthopAddr = Arnet.IpGenAddr( str( nexthop ) )
   endpointAddr = Arnet.IpGenPrefix( str( endpoint ) )
   if nexthopAddr.af != endpointAddr.af:
      mode.addError( endpointNextHopAfErrMsg )
      return None, None, None
   if endpointAddr.af == AddressFamily.ipv4:
      ipIntf = Globals.ipConfig.ipIntfConfig.get( intf.name )
   elif endpointAddr.af == AddressFamily.ipv6:
      ipIntf = Globals.ip6Config.intf.get( intf.name )
   else:
      mode.addError( invalidEndpointErrMsg )
      return None, None, None
   intfId = ipIntf.intfId if ipIntf else Tac.Value( "Arnet::IntfId",
                                                    intf.name )
   return nexthopAddr, endpointAddr, intfId

def MplsStaticTunnel_handler( mode, args ):
   tunnelName = args[ 'NAME' ]
   endpoint = ( args.get( 'TEP_V4_ADDR' ) or args.get( 'TEP_V4_PREFIX' ) or
                  args.get( 'TEP_V6_ADDR' ) or args.get( 'TEP_V6_PREFIX' ) )
   nexthop = args.get( 'NEXTHOP_V4' ) or args.get( 'NEXTHOP_V6' )
   intf = args.get( 'INTF' )

   # if using the Null0 interface, give an empty label stack and zero nexthop
   # address for corresponding address family as placeholders
   if 'NULL0_EXPR' in args:
      endpointAddr = Arnet.IpGenPrefix( str( endpoint ) )
      intf = EthPhyIntf( 'Null0', mode )
      if endpointAddr.af == AddressFamily.ipv4:
         nexthop = '0.0.0.0'
      elif endpointAddr.af == AddressFamily.ipv6:
         nexthop = '0::0'
      labels = []
   elif 'imp-null-tunnel' in args:
      labels = [ 3 ]
   else:
      labels = args.get( 'LABELS' )
      if not Globals.validateLabelStackSize( mode, labels ):
         return

   nexthopAddr, tunnelEndpoint, intfId = getMplsTunnelNhTepAndIntfId( mode,
                                                                        nexthop,
                                                                        endpoint,
                                                                        intf )
   if ( nexthopAddr, tunnelEndpoint, intfId ) == ( None, None, None ):
      return

   staticTunnelConfigEntry = Globals.staticTunnelConfig.entry.get( tunnelName )
   if staticTunnelConfigEntry and staticTunnelConfigEntry.tep == tunnelEndpoint:
      newStaticTunnelConfigEntry = \
         copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
   else:
      newStaticTunnelConfigEntry = StaticTunnelConfigEntry( tunnelName )
      newStaticTunnelConfigEntry.tep = tunnelEndpoint
   # add newly configured via
   newMplsVia = Globals.MplsVia(
      nexthopAddr, intfId, Globals.labelOperation( labels ) )
   # Clear existing vias if any, and add newly created via
   newStaticTunnelConfigEntry.via.clear()
   newStaticTunnelConfigEntry.via[ newMplsVia ] = True
   newStaticTunnelConfigEntry.inStaticTunnelMode = False

   if staticTunnelConfigEntry:
      # Check for duplicate and returns if it is the case
      if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
         return
   Globals.staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

def MplsStaticTunnel_noOrDefaultHandler( mode, args ):
   tunnelName = args[ 'NAME' ]

   staticTunnelConfigEntry = Globals.staticTunnelConfig.entry.get( tunnelName )
   if staticTunnelConfigEntry:
      newStaticTunnelConfigEntry = \
         copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
      newStaticTunnelConfigEntry.via.clear()
      # if neither primary via or backup via is still valid, then delete the
      # entry, else update it in the static tunnel config table
      if ( not newStaticTunnelConfigEntry.via and
            newStaticTunnelConfigEntry.backupVia == Globals.MplsVia() ):
         del Globals.staticTunnelConfig.entry[ tunnelName ]
      else:
         Globals.staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

def MplsTunnelStaticConfig_handler( mode, args ):
   tunnelName = args[ 'NAME' ]
   endpoint = ( args.get( 'ENDPOINTV4ADDR' ) or args.get( 'ENDPOINTV4PREFIX' ) or
                  args.get( 'ENDPOINTV6ADDR' ) or args.get( 'ENDPOINTV6PREFIX' ) )

   tunnelEndpoint = Arnet.IpGenPrefix( str( endpoint ) )
   # pylint: disable-next=consider-using-in
   if ( tunnelEndpoint.af != AddressFamily.ipv4 and
         tunnelEndpoint.af != AddressFamily.ipv6 ):
      mode.addError( invalidEndpointErrMsg )
      return
   staticTunnelConfigEntry = Globals.staticTunnelConfig.entry.get( tunnelName )
   if staticTunnelConfigEntry:
      if staticTunnelConfigEntry.tep != tunnelEndpoint:
         errMsg = ( "Existing config for tunnel '{}' endpoint {}"
                  ).format( tunnelName, str( tunnelEndpoint ) )
         mode.addError( errMsg )
         return
      for via in staticTunnelConfigEntry.via:
         # If via.intfId is a FecIdIntfId, this is a resolving tunnel (hidden
         # argument), and not supported for ECMP
         if FecIdIntfId.isFecIdIntfId( via.intfId ):
            errMsg = ( "Existing resolving via config for tunnel '{}' endpoint {}"
                     ).format( tunnelName, str( tunnelEndpoint ) )
            mode.addError( errMsg )
            return
      newStaticTunnelConfigEntry = \
         copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
   else:
      newStaticTunnelConfigEntry = \
         Tac.Value( 'Tunnel::Static::StaticTunnelConfigEntry', tunnelName )
      newStaticTunnelConfigEntry.tep = tunnelEndpoint
   newStaticTunnelConfigEntry.inStaticTunnelMode = True
   Globals.staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )
   childMode = mode.childMode( Globals.TunnelStaticMode, tunName=tunnelName,
                                 tep=tunnelEndpoint )
   mode.session_.gotoChildMode( childMode )

def MplsTunnelStaticConfigVia_checkVia(
      tunName, staticTunnelConfigEntry, nexthopAddr=None, no=False ):
   if not no and not staticTunnelConfigEntry:
      errMsg = ( "Config for tunnel '{}' not found" ).format( tunName )
      return errMsg
   if nexthopAddr and nexthopAddr.af != staticTunnelConfigEntry.tep.af:
      errMsg = endpointNextHopAfErrMsg
      return errMsg
   # pylint: disable-next=consider-using-in
   if ( staticTunnelConfigEntry.tep.af != AddressFamily.ipv4 and
         staticTunnelConfigEntry.tep.af != AddressFamily.ipv6 ):
      errMsg = invalidEndpointErrMsg
      return errMsg
   return None

def MplsTunnelStaticConfigVia_makeVia(
      staticTunnelConfigEntry, nexthopAddr, intf, labels ):
   if staticTunnelConfigEntry.tep.af == AddressFamily.ipv4:
      ipIntf = Globals.ipConfig.ipIntfConfig.get( intf.name )
   else: # staticTunnelConfigEntry.tep.af == AddressFamily.ipv6
      ipIntf = Globals.ip6Config.intf.get( intf.name )
   intfId = ipIntf.intfId if ipIntf else Tac.Value( "Arnet::IntfId", intf.name )
   via = Globals.MplsVia( nexthopAddr, intfId, Globals.labelOperation( labels ) )
   return via

def MplsTunnelStaticConfigVia_handler( mode, args ):
   nexthop = args.get( 'NEXTHOPV4' ) or args.get( 'NEXTHOPV6' )
   intf = args.get( 'INTF' )
   if 'imp-null-tunnel' in args:
      labels = [ 3 ]
   else:
      labels = args.get( 'LABELS' )
      if not Globals.validateLabelStackSize( mode, labels ):
         return
   nexthopAddr = Arnet.IpGenAddr( str( nexthop ) )
   staticTunnelConfigEntry = Globals.staticTunnelConfig.entry.get( mode.tunName )

   error = MplsTunnelStaticConfigVia_checkVia( mode.tunName,
                                                staticTunnelConfigEntry,
                                                nexthopAddr=nexthopAddr )
   if error:
      mode.addError( error )
      return

   newStaticTunnelConfigEntry = \
      copyStaticTunnelConfigEntry( staticTunnelConfigEntry )

   newMplsVia = MplsTunnelStaticConfigVia_makeVia( staticTunnelConfigEntry,
                                                      nexthopAddr, intf, labels )
   newStaticTunnelConfigEntry.via[ newMplsVia ] = True

   # Check for duplicate and returns if it is the case
   if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
      return
   Globals.staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

def MplsTunnelStaticConfigVia_noOrDefaultHandler( mode, args ):
   nexthop = args.get( 'NEXTHOPV4' ) or args.get( 'NEXTHOPV6' )
   intf = args.get( 'INTF' )
   if 'imp-null-tunnel' in args:
      labels = [ 3 ]
   else:
      labels = args.get( 'LABELS' )
      if not Globals.validateLabelStackSize( mode, labels ):
         return
   nexthopAddr = Arnet.IpGenAddr( str( nexthop ) )
   staticTunnelConfigEntry = Globals.staticTunnelConfig.entry.get( mode.tunName )

   error = MplsTunnelStaticConfigVia_checkVia( mode.tunName,
                                                staticTunnelConfigEntry,
                                                no=True )
   if error:
      mode.addError( error )
      return

   if staticTunnelConfigEntry:
      newStaticTunnelConfigEntry = \
         copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
   else:
      return # entry already doesn't exist

   newMplsVia = MplsTunnelStaticConfigVia_makeVia( staticTunnelConfigEntry,
                                                      nexthopAddr, intf, labels )
   del newStaticTunnelConfigEntry.via[ newMplsVia ]
   # Check for duplicate and returns if it is the case
   if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
      return
   Globals.staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

def MplsTunnelTerminationModelCmd_handler( mode, args ):
   ttlModel = args[ 'TTL_MODEL' ]
   dscpModel = args[ 'DSCP_MODEL' ]

   Globals.mplsRoutingConfig.labelTerminationTtlMode = ttlModel
   Globals.mplsRoutingConfig.labelTerminationDscpMode = dscpModel
   Globals.mplsRoutingConfig.labelTerminationModeVersion += 1

def MplsTunnelTerminationModelCmd_noOrDefaultHandler( mode, args ):
   Globals.mplsRoutingConfig.labelTerminationTtlMode = TtlMode.platformDefault
   Globals.mplsRoutingConfig.labelTerminationDscpMode = TtlMode.platformDefault
   Globals.mplsRoutingConfig.labelTerminationModeVersion += 1

tunnelEncapsulationConfigChangeWarning = \
   "Change will take effect only after switch reboot or L3 forwarding agent restart."

def MplsTunnelEncapsulationModelCmd_handler( mode, args ):
   ttlModel = args.get( 'TTL_MODEL_PIPE', 'uniform' )
   expModel = args.get( 'EXP_MODEL_PIPE', 'uniform' )
   ttlValue = args.get( 'TTL_VALUE', 255 )
   expValue = args.get( 'EXP_VALUE', 0 )

   Globals.mplsRoutingConfig.labelEncapsulationTtlMode = ttlModel
   Globals.mplsRoutingConfig.labelEncapsulationExpMode = expModel
   Globals.mplsRoutingConfig.labelEncapsulationTtlValue = ttlValue
   Globals.mplsRoutingConfig.labelEncapsulationExpValue = expValue
   Globals.mplsRoutingConfig.labelEncapsulationModeVersion += 1
   mode.addWarning( tunnelEncapsulationConfigChangeWarning )

def MplsTunnelTerminationPhpModelCmd_handler( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      Globals.mplsRoutingConfig.labelPhpTtlMode = TtlMode.undefinedTtlMode
      Globals.mplsRoutingConfig.labelPhpDscpMode = TtlMode.undefinedTtlMode
   else:
      Globals.mplsRoutingConfig.labelPhpTtlMode = args[ 'TTL_MODEL' ]
      Globals.mplsRoutingConfig.labelPhpDscpMode = args[ 'DSCP_MODEL' ]
   Globals.mplsRoutingConfig.labelPhpModeVersion += 1

def gotoTunnelTerminationMode( mode, args ):
   childMode = mode.childMode( Globals.TunnelTerminationMode )
   mode.session_.gotoChildMode( childMode )

def clearTunnelTerminationMode( mode, args ):
   Globals.mplsTunnelTermVrfQosConfig.vrfQosMap.clear()

def gotoTunnelTerminationVrfMode( mode, vrfName ):
   childMode = mode.childMode( Globals.TunnelTerminationVrfMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )
   # Associate a default map with the vrf if it not already in  mpls tunnel term cfg
   if not vrfName in Globals.mplsTunnelTermVrfQosConfig.vrfQosMap:
      qosMap = QosMap()
      vrfQosMap = Tac.Value( "Mpls::VrfQosMap", vrfName, qosMap )
      Globals.mplsTunnelTermVrfQosConfig.vrfQosMap.addMember( vrfQosMap )

def TunnelTerminationVrfMode_addQosMapToVrf( self, args ):
   mapName = args.get( 'MAP_NAME' )
   qosMap = QosMap( mapName, Globals.QosMapType.mplsDecapDscpToTc )
   vrfName = self.vrfName
   vrfQosMap = Tac.Value( "Mpls::VrfQosMap", vrfName, qosMap )
   Globals.mplsTunnelTermVrfQosConfig.vrfQosMap.addMember( vrfQosMap )

def TunnelTerminationVrfMode_delQosMapToVrf( self, args ):
   del Globals.mplsTunnelTermVrfQosConfig.vrfQosMap[ self.vrfName ]
   # Associate the vrf with a default qos map
   qosMap = QosMap()
   vrfQosMap = Tac.Value( "Mpls::VrfQosMap", self.vrfName, qosMap )
   Globals.mplsTunnelTermVrfQosConfig.vrfQosMap.addMember( vrfQosMap )

def TunnelTerminationVrfModeCmd_handler( mode, args ):
   gotoTunnelTerminationVrfMode( mode, args[ 'VRF_NAME' ] )

def TunnelTerminationVrfModeCmd_noOrDefaultHandler( mode, args ):
   del Globals.mplsTunnelTermVrfQosConfig.vrfQosMap[ args[ 'VRF_NAME' ] ]

def getDestSourceProtoDefaultLfibStitchingPref( destProto, sourceProto ):
   if destProto == eBgpLuProtocol:
      if sourceProto == eBgpLuProtocol:
         return LfibStitchingPreference.defaultEbgpLuToEbgpLuStitchingPref
      elif sourceProto == ldpProtocol:
         return LfibStitchingPreference.defaultEbgpLuToLdpStitchingPref
      elif sourceProto == ospfSrProtocol:
         return LfibStitchingPreference.defaultEbgpLuToOspfSrStitchingPref
      elif sourceProto == isisSrProtocol:
         return LfibStitchingPreference.defaultEbgpLuToIsisSrStitchingPref
      elif sourceProto == iBgpLuProtocol:
         return LfibStitchingPreference.defaultEbgpLuToIbgpLuStitchingPref
   elif destProto == ldpProtocol:
      if sourceProto == eBgpLuProtocol:
         return LfibStitchingPreference.defaultLdpToEbgpLuStitchingPref
      elif sourceProto == ldpProtocol:
         return LfibStitchingPreference.defaultLdpToLdpStitchingPref
      elif sourceProto == ospfSrProtocol:
         return LfibStitchingPreference.defaultLdpToOspfSrStitchingPref
      elif sourceProto == isisSrProtocol:
         return LfibStitchingPreference.defaultLdpToIsisSrStitchingPref
      elif sourceProto == iBgpLuProtocol:
         return LfibStitchingPreference.defaultLdpToIbgpLuStitchingPref
   elif destProto == ospfSrProtocol:
      if sourceProto == eBgpLuProtocol:
         return LfibStitchingPreference.defaultOspfSrToEbgpLuStitchingPref
      elif sourceProto == ldpProtocol:
         return LfibStitchingPreference.defaultOspfSrToLdpStitchingPref
      elif sourceProto == ospfSrProtocol:
         return LfibStitchingPreference.defaultOspfSrToOspfSrStitchingPref
      elif sourceProto == isisSrProtocol:
         return LfibStitchingPreference.defaultOspfSrToIsisSrStitchingPref
      elif sourceProto == iBgpLuProtocol:
         return LfibStitchingPreference.defaultOspfSrToIbgpLuStitchingPref
   elif destProto == isisSrProtocol:
      if sourceProto == eBgpLuProtocol:
         return LfibStitchingPreference.defaultIsisSrToEbgpLuStitchingPref
      elif sourceProto == ldpProtocol:
         return LfibStitchingPreference.defaultIsisSrToLdpStitchingPref
      elif sourceProto == ospfSrProtocol:
         return LfibStitchingPreference.defaultIsisSrToOspfSrStitchingPref
      elif sourceProto == isisSrProtocol:
         return LfibStitchingPreference.defaultIsisSrToIsisSrStitchingPref
      elif sourceProto == iBgpLuProtocol:
         return LfibStitchingPreference.defaultIsisSrToIbgpLuStitchingPref
   elif destProto == iBgpLuProtocol:
      if sourceProto == eBgpLuProtocol:
         return LfibStitchingPreference.defaultIbgpLuToEbgpLuStitchingPref
      elif sourceProto == ldpProtocol:
         return LfibStitchingPreference.defaultIbgpLuToLdpStitchingPref
      elif sourceProto == ospfSrProtocol:
         return LfibStitchingPreference.defaultIbgpLuToOspfSrStitchingPref
      elif sourceProto == isisSrProtocol:
         return LfibStitchingPreference.defaultIbgpLuToIsisSrStitchingPref
      elif sourceProto == iBgpLuProtocol:
         return LfibStitchingPreference.defaultIbgpLuToIbgpLuStitchingPref
   assert False, "Invalid Destination-Source protocols"

def setDefaultLfibStitchingPreferences():
   for destProto in lfibStitchingProtocol:
      destProtoPrefs = Globals.mplsRoutingConfig.destProtoLfibStitchingPreference.\
         newMember( destProto )
      for sourceProto in lfibStitchingProtocol:
         pref = getDestSourceProtoDefaultLfibStitchingPref( destProto, sourceProto )
         if pref != LfibStitchingPreference.invalidPref():
            sourceProtoPref = LfibSourceProtoPref( sourceProto, pref )
            destProtoPrefs.sourceProtocolPreferences.addMember( sourceProtoPref )

def gotoLfibStitchingPreferencesMode( mode, args ):
   setDefaultLfibStitchingPreferences()
   Globals.mplsRoutingConfig.lfibStitchingPrefConfigEnabled = True
   childMode = mode.childMode( Globals.LfibStitchingPreferencesMode )
   mode.session_.gotoChildMode( childMode )

def delLfibStitchingPreferencesMode( mode, args ):
   Globals.mplsRoutingConfig.lfibStitchingPrefConfigEnabled = \
         Globals.mplsRoutingConfig.lfibStitchingPrefConfigEnabledDefault
   Globals.mplsRoutingConfig.destProtoLfibStitchingPreference.clear()

def MplsParsingCmd_handler( mode, args ):
   mplsParsingType = args[ 'PARSING_TYPE' ]
   curCfg = Globals.mplsRoutingConfig.speculativeParsingConfig
   if curCfg is None:
      curCfg = Globals.mplsHwCapability.mplsParsingDefault
   ipv4 = curCfg.parseIpv4 or mplsParsingType == 'ipv4'
   ipv6 = curCfg.parseIpv6 or mplsParsingType == 'ipv6'
   ethernet = curCfg.parseEth or mplsParsingType == 'ethernet'
   controlWord = curCfg.parsePwCw or mplsParsingType == 'control-word'
   newCfg = MplsSpeculativeParsingConfig( ipv4, ipv6, controlWord, ethernet )
   if newCfg == Globals.mplsHwCapability.mplsParsingDefault:
      newCfg = None
   Globals.mplsRoutingConfig.speculativeParsingConfig = newCfg

def MplsParsingCmd_noHandler( mode, args ):
   mplsParsingType = args[ 'PARSING_TYPE' ]
   curCfg = Globals.mplsRoutingConfig.speculativeParsingConfig
   if curCfg is None:
      curCfg = Globals.mplsHwCapability.mplsParsingDefault
   ipv4 = curCfg.parseIpv4 and mplsParsingType != 'ipv4'
   ipv6 = curCfg.parseIpv6 and mplsParsingType != 'ipv6'
   ethernet = curCfg.parseEth and mplsParsingType != 'ethernet'
   controlWord = curCfg.parsePwCw and mplsParsingType != 'control-word'
   newCfg = MplsSpeculativeParsingConfig( ipv4, ipv6, controlWord, ethernet )
   if newCfg == Globals.mplsHwCapability.mplsParsingDefault:
      newCfg = None
   Globals.mplsRoutingConfig.speculativeParsingConfig = newCfg

def MplsParsingCmd_defaultHandler( mode, args ):
   # Should access platform default config here
   mplsParsingType = args[ 'PARSING_TYPE' ]
   curCfg = Globals.mplsRoutingConfig.speculativeParsingConfig
   if curCfg is None:
      curCfg = Globals.mplsHwCapability.mplsParsingDefault
   # If we are default already, no need to change anything
   defaultCfg = Globals.mplsHwCapability.mplsParsingDefault
   ipv4 = ( defaultCfg.parseIpv4 if mplsParsingType == 'ipv4'
            else curCfg.parseIpv4 )
   ipv6 = ( defaultCfg.parseIpv6 if mplsParsingType == 'ipv6'
            else curCfg.parseIpv6 )
   ethernet = ( defaultCfg.parseEth if mplsParsingType == 'ethernet'
               else curCfg.parseEth )
   controlWord = ( defaultCfg.parsePwCw if mplsParsingType == 'control-word'
                  else curCfg.parsePwCw )
   newCfg = MplsSpeculativeParsingConfig( ipv4, ipv6, controlWord, ethernet )
   if newCfg == defaultCfg:
      newCfg = None
   Globals.mplsRoutingConfig.speculativeParsingConfig = newCfg

def ShowMplsParsingSpeculativeCmd_handler( mode, args ):
   cfg = Globals.mplsRoutingConfig.speculativeParsingConfig
   if cfg is None:
      cfg = Globals.mplsHwCapability.mplsParsingDefault
   entries = { "IPv4": cfg.parseIpv4,
               "IPv6": cfg.parseIpv6,
               "Ethernet": cfg.parseEth,
               "Control Word": cfg.parsePwCw }
   return MplsModel.MplsParsingSpeculativeTableModel( entries=entries )

def showTunnelTermVrfQosMaps( mode, args ):
   mplsVrfDscpToTcMapAllModel = MplsModel.MplsVrfDscpToTcMapAllModel()
   vrfName = args.get( 'VRF_NAME' )
   vrfNames = []
   if vrfName:
      if vrfName in Globals.mplsTunnelTermVrfQosConfig.vrfQosMap:
         vrfNames = [ vrfName ]
   else:
      vrfNames = Globals.mplsTunnelTermVrfQosConfig.vrfQosMap

   # Verify if vrfNames are part of HwStatus. Print only for active vrfs.
   for vrfName in vrfNames:
      qosMap = Globals.mplsTunnelTermVrfQosConfig.vrfQosMap[ vrfName ].qosMap
      # If qosMap attached to the vrf is a default qosMap, we do not display that in
      # show output. Likewise absence of qosMap in hwStatus means that the qosMap is
      # not defined in qosStatus.
      if qosMap == QosMap() or \
         qosMap not in Globals.mplsTunnelTermVrfHwStatus.hwStatus:
         continue
      if vrfName in Globals.mplsTunnelTermVrfHwStatus.hwStatus[ qosMap ].vrf:
         qosMapName = qosMap.mapName
         mplsVrfDscpToTcMapAllModel.vrfsToDscpToTcNamedMaps[ vrfName ] = qosMapName
   return mplsVrfDscpToTcMapAllModel

def getStaticTunnelTableEntryModel( tunnelId, endpoint=None, name=None ):
   vias = []

   # fix once the capi model for StaticTunnelTableEntry is fixed
   # to support backup vias
   staticTunnelTableEntry = Globals.staticTunnelTable.entry.get( tunnelId )
   if staticTunnelTableEntry:
      if endpoint and staticTunnelTableEntry.tep != endpoint:
         return None
      if name and staticTunnelTableEntry.getName() != name:
         return None
      for via in staticTunnelTableEntry.via.values():
         viaModels = getViaModels( via, useMplsVias=True )
         # The 'resolving-tunnel' hidden command uses MplsTunnelVia model, which
         # is not supported by MplsModel.StaticTunnelTableEntry, so skip those
         if viaModels:
            vias.extend( via for via in viaModels
                         if isinstance( via, TunnelModels.MplsVia ) )

      return MplsModel.StaticTunnelTableEntry(
         endpoint=staticTunnelTableEntry.tep, name=staticTunnelTableEntry.getName(),
         vias=vias )
   return None

def getStaticTunnelTable( args ):
   staticEntries = {}

   tunnelIds = Globals.staticTunnelTable.entry
   endpoint = None
   name = None
   if 'index' in args:
      # For each index query, generate a V4 index (44th bit set to zero) and a
      # V6 index (44th bit set to one) with getTunnelIdFromIndex. We'll combine
      # both indices into tunnelIds, and pass it along to the for loop below.
      # Note that indices are generated with a monotonically increasing
      # counter. So we should never have a case where we see both a V4 tunnel
      # and a V6 tunnel within the same  "show mpls tunnel static index
      # <tunnel-index>" command.
      tunnelV4Ids = [ getTunnelIdFromIndex( 'staticV4Tunnel',
                                            args[ 'INDEX' ] ) ]
      tunnelV6Ids = [ getTunnelIdFromIndex( 'staticV6Tunnel',
                                            args[ 'INDEX' ] ) ]
      tunnelIds = tunnelV4Ids + tunnelV6Ids
   elif 'name' in args:
      name = args[ 'NAME' ]
   elif 'endpoint' in args:
      endpoint = ( args.get( 'ADDR4' ) or
                   args.get( 'PREFIX4' ) or
                   args.get( 'ADDR6' ) or
                   args.get( 'PREFIX6' ) )
      endpoint = Arnet.IpGenPrefix( getattr( endpoint, 'stringValue', endpoint ) )

   for tunnelId in tunnelIds:
      staticEntryModel = getStaticTunnelTableEntryModel( tunnelId, endpoint, name )
      if staticEntryModel:
         staticEntries[ getTunnelIndexFromId( tunnelId ) ] = staticEntryModel

   return MplsModel.StaticTunnelTable( entries=staticEntries )

def ShowMplsTunnelStaticCmd_handler( mode, args ):
   return getStaticTunnelTable( args=args )

showMplsTunnelFibDeprecatedWarning = (
    "'show mpls tunnel fib' has been deprecated. Please use "
    "'show tunnel fib [options]' moving forward." )

def getMplsTunnelFibEntryModel( tunnelId ):
   tunnelType = getTunnelTypeFromTunnelId( tunnelId )
   if tunnelType in showTunnelFibIgnoredTunnelTypes:
      # Do not show ignored entry types in 'show mpls tunnel fib'
      return None

   # NOTE: The tunnel fib entry is now obtained from the new refactored
   # TunnelFib and not MplsTunnelFib. To defer changing the Mpls CAPI
   # models, we convert the new TunnelFib entry data into the current
   # CAPI model that represents the old MplsTunnelFib. At some point in
   # the future, the CAPI model could be upgraded (again).
   vias = []
   tunnelFibEntry = Globals.tunnelFib.entry.get( tunnelId )
   if tunnelFibEntry:
      endpoint = getEndpointFromTunnelId( tunnelId )
      tunnelViaStatus = getTunnelViaStatusFromTunnelId(
            tunnelId, CliPlugin.TunnelCli.programmingStatus )
      for tunVia in tunnelFibEntry.tunnelVia.values():
         mplsViaModels = getMplsViaModelFromTunnelVia( tunVia )
         for mplsViaModel in mplsViaModels:
            vias.append( mplsViaModel )
      return MplsModel.MplsTunnelFibEntry(
            tunnelIndex=getTunnelIndexFromId( tunnelId ), 
            tunnelType=TunnelModels.tunnelTypeStrDict[ tunnelType ],
            endpoint=endpoint, vias=vias, tunnelViaStatus=tunnelViaStatus )
   return None

def ShowMplsTunnelFib_handler( mode, args ):
   mode.addWarning( showMplsTunnelFibDeprecatedWarning )
   mplsTunnelFibEntries = []

   for tunnelId in Globals.tunnelFib.entry:
      mplsTunnelFibEntryModel = getMplsTunnelFibEntryModel( tunnelId=tunnelId )
      if mplsTunnelFibEntryModel:
         mplsTunnelFibEntries.append( mplsTunnelFibEntryModel )

   # Sort entries by address family, endpoint, index
   mplsTunnelFibEntries.sort( key=lambda entry:
      ( entry.endpoint.af if entry.endpoint else "0",
         entry.endpoint.address if entry.endpoint else "0",
         entry.tunnelIndex ) )
   return MplsModel.MplsTunnelFib( entries=mplsTunnelFibEntries )

def ShowMplsLabelRangeCommand_handler( mode, args ):
   static, dynamicBlock = ShowMplsLabelRangeCommand_getStaticRanges()
   dynamic = ShowMplsLabelRangeCommand_getDynamicRanges()
   static = [ item for item in static if item.blockSize > 0 ]
   # 1. Unassigned ranges: Ranges left after combining operational and
   # conflicting label-ranges ( both static and dynamic blocks ).
   # 2. Unused dynamic ranges: Free ranges within the dynamic block.
   # Unassigned ranges are by nature conflict-free and appear in
   # operational ranges available for assignments.
   conflictingStatic, conflictingDynamicBlock = \
         ShowMplsLabelRangeCommand_getStaticRangeListAndDynBlock(
            Globals.mplsRoutingStatus, 'conflictingLabelRange' )
   # dynamicBlock either belongs in labelRange or
   # conflictingLabelRange collection.
   t8( "dynamic block: operational", dynamicBlock, ", conflicting",
         conflictingDynamicBlock )
   t8( "static ranges: operational", static, ", conflicting", conflictingStatic )
   t8( "dynamic ranges from label block allocation", dynamic )
   dynamicBlock = dynamicBlock or conflictingDynamicBlock
   unusedStatic = ShowMplsLabelRangeCommand_getUnusedRanges(
         static + conflictingStatic + [ dynamicBlock ],
         0,
         MplsLabel.max,
         LABEL_RANGE_STATIC )
   unusedDynamic = ShowMplsLabelRangeCommand_getUnusedRanges(
         dynamic, dynamicBlock.blockStart,
         dynamicBlock.blockEnd(),
         LABEL_RANGE_DYNAMIC )

   operationalRanges = static + unusedStatic
   conflictingRanges = conflictingStatic
   # Add the dynamic/unused dynamic ranges from
   # Mpls::Status::labelBlockAlloc depending on whether the dynamic
   # block belongs to conflicting labelRange.
   if not conflictingDynamicBlock:
      operationalRanges += dynamic + unusedDynamic
   else:
      conflictingRanges += dynamic + unusedDynamic
   operationalRanges = ShowMplsLabelRangeCommand_sortedRanges(
      operationalRanges )
   conflictingRanges = ShowMplsLabelRangeCommand_sortedRanges(
      conflictingRanges )
   return MplsModel.MplsLabelRanges( ranges=operationalRanges,
                                       conflictingRanges=conflictingRanges )

def ShowMplsLabelRangeCommand_getDynamicRanges():
   rangeList = []
   for name, lba in Globals.labelManagerStatus.labelBlockAllocation.items():
      for rangeVal in lba.block.values():
         rangeList.append( MplsModel.MplsLabelRange(
               protocol=name,
               rangeType=LABEL_RANGE_DYNAMIC,
               blockSize=rangeVal.blockSize,
               blockStart=rangeVal.blockStart ) )
   return rangeList

def ShowMplsLabelRangeCommand_getCliProtocolName( rangeType ):
   if rangeType == LabelRangeInfo.rangeTypeL2evpnSharedEs:
      return "l2evpn shared ethernet-segment"
   else:
      return rangeType

def ShowMplsLabelRangeCommand_getStaticRangeListAndDynBlock(
      labelRangeEntity, collectionName ):
   rangeList = []
   dynamicBlock = None
   labelRangeColl = getattr( labelRangeEntity, collectionName )
   for name, lr in labelRangeColl.items():
      protocol = ShowMplsLabelRangeCommand_getCliProtocolName( name )
      rangeVal = MplsModel.MplsLabelRange(
         protocol=protocol,
         rangeType=LABEL_RANGE_STATIC,
         blockSize=lr.size,
         blockStart=lr.base )

      if rangeVal.protocol == LabelRangeInfo.rangeTypeDynamic:
         dynamicBlock = rangeVal
      else:
         rangeList.append( rangeVal )
   return ( rangeList, dynamicBlock )

def ShowMplsLabelRangeCommand_getStaticRanges():
   # MPLS Standard
   mplsReservedRange = MplsModel.MplsLabelRange(
      protocol='reserved',
      rangeType=LABEL_RANGE_STATIC,
      blockSize=MplsLabel.unassignedMin - MplsLabel.min,
      blockStart=MplsLabel.min )

   dynamicBlock = None
   if toggleLabelRangeStatusEnabled():
      labelRangeEntity = Globals.mplsRoutingStatus
   else:
      labelRangeEntity = Globals.mplsRoutingConfig
   ( rangeList, dynamicBlock ) = \
      ShowMplsLabelRangeCommand_getStaticRangeListAndDynBlock( labelRangeEntity,
                                                                  'labelRange' )
   rangeList.append( mplsReservedRange )
   return ( rangeList, dynamicBlock )

def ShowMplsLabelRangeCommand_getUnusedRanges( labelRanges, start, end, typ ):
   labelRanges = ShowMplsLabelRangeCommand_sortedRanges( labelRanges )
   mergedRanges = []
   for labelRange in labelRanges:
      rangeBounds = ( labelRange.blockStart, labelRange.blockEnd() )
      l = len( mergedRanges )
      if l == 0 or mergedRanges[ l - 1 ][ 1 ] < rangeBounds[ 0 ]:
         # The range is distinct from the known ranges
         mergedRanges.append( rangeBounds )
      else:
         # The range overlaps the last range, and only the last range
         mergedRanges[ l - 1 ] = ( mergedRanges[ l - 1 ][ 0 ],
               max( mergedRanges[ l - 1 ][ 1 ], rangeBounds[ 1 ] ) )

   unusedRanges = []
   lastEnd = start - 1
   for r in mergedRanges:
      # Will be true except when first range is at start
      if r[ 0 ] > lastEnd + 1:
         unusedRanges.append(
               ShowMplsLabelRangeCommand_createUnusedRange(
                  lastEnd + 1, r[ 0 ] - 1, typ ) )
      lastEnd = r[ 1 ]

   if lastEnd < end:
      unusedRanges.append( ShowMplsLabelRangeCommand_createUnusedRange(
         lastEnd + 1, end, typ ) )

   return unusedRanges

def ShowMplsLabelRangeCommand_sortedRanges( labelRanges ):
   return sorted( labelRanges, key=lambda r: r.blockStart )

def ShowMplsLabelRangeCommand_createUnusedRange( start, end, typ ):
   return MplsModel.MplsLabelRange(
         protocol=None,
         rangeType=typ if typ != LABEL_RANGE_STATIC else LABEL_RANGE_UNASSIGNED,
         blockSize=end - start + 1,
         blockStart=start )

def ShowMplsFecRequestCmd_handler( mode, args ):
   labels = args.get( 'LABELS' )
   pending = 'pending' in args
   Globals.showMplsConfigWarnings( mode )

   LazyMount.force( Globals.mplsHwStatus )
   if not Globals.mplsHwStatus:
      return MplsModel.MplsFecRequestTable( entries=[] )

   routes = {}
   if labels:
      labelStack = Arnet.MplsLib.labelListToMplsLabelStack( labels )
      routeKeys = [ RouteKey( labelStack ) ]
   else:
      routeKeys = list( Globals.mplsHwStatus.route )

   # for each route, create the table entry
   for rk in routeKeys:
      route = Globals.mplsHwStatus.route.get( rk )
      if route is None:
         continue
      labels = None
      if rk.labelStack.stackSize < 2:
         labels = str( rk.topLabel )
      else:
         labels = rk.labelStack.cliShowString()

      entry = Globals.transitLfib.lfibRoute.get( rk )
      if entry:
         key = entry.viaSetKey
         mplsVskFecIds = \
            Globals.mplsVskFecIdCliCollection.mplsVskFecIdEntry.get( key )
         if mplsVskFecIds:
            requestedFecId = mplsVskFecIds.requestedFecId
            installedFecId = mplsVskFecIds.installedFecId
            programStatus = ( 'synced' if requestedFecId == installedFecId
                              else 'pending' )
            if pending and programStatus == 'synced':
               continue

            entry = MplsModel.MplsFecRequestTableEntry(
               requestedFecId=requestedFecId,
               installedFecId=installedFecId,
               programStatus=programStatus,
               requestedAlternateFecIds=list(
                  mplsVskFecIds.alternateFecId.values() ) )
            routes[ labels ] = entry
   return MplsModel.MplsFecRequestTable( routes=routes )

def ShowTechMplsGr_handler( mode, args ):
   # hardcoding 'Mpls' to avoid importing MplsAgent.name
   AgentCommandRequest.runSocketCommand(
      Globals.em, 'Mpls', 'debug', 'DUMP_GR_STATE' )
