#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import (
   Model,
   Dict,
   Enum,
   Bool,
   Int,
)
from CliPlugin import IntfCli
from CliPlugin.IntfModel import (
   prettyIntfStatusMap,
)
from IntfModels import Interface
from TableOutput import (
   createTable,
   Format,
)
from TypeFuture import TacLazyType

FecHwProgrammingState = TacLazyType( "Routing::Hardware::FecHwProgrammingState" )

def renderBool( value ):
   if value is None:
      return '-'
   elif value:
      return 'Yes'
   return 'No'

class MplsInterfaceInfo( Model ):
   status = Enum( values=list( prettyIntfStatusMap ),
                  help='Operational status of the interface' )
   mplsConfigured = Bool( help='Configuration of MPLS on interface' )
   ldpConfigured = Bool( help='Configuration of LDP on interface' )
   rsvpConfigured = Bool( help='Configuration of RSVP on interface' )

   def render( self ):
      return [ prettyIntfStatusMap[ self.status ],
               renderBool( self.mplsConfigured ),
               renderBool( self.ldpConfigured ),
               renderBool( self.rsvpConfigured ) ]

class MplsInterfaceModel( Model ):
   intfs = Dict( keyType=Interface, valueType=MplsInterfaceInfo,
                 help="A mapping between an Interface and MPLS interface info" )

   def render( self ):
      headers = [ 'Interface', 'Status', 'MPLS', 'LDP', 'RSVP' ]
      f1 = Format( justify='left' )
      f1.padLimitIs( True )
      outputTable = createTable( headers )
      for interface in Arnet.sortIntf( self.intfs ):
         interfaceInfo = self.intfs[ interface ]
         outputTable.newRow( interface, *interfaceInfo.render() )
      outputTable.formatColumns( *[ f1 for _ in headers ] )
      print( outputTable.output() )

# show mpls interfaces <i> counters
class MplsInterfaceIngressCounters( MplsInterfaceModel ):
   # Mpls per interface ingress counters
   class MplsInterfaceIngressCounters( Model ):
      mplsInOctets = Int( help="Number of input MPLS octets", default=0 )
      mplsInPkts = Int( help="Number of input MPLS packets", default=0 )

   interfaces = Dict( keyType=Interface, valueType=MplsInterfaceIngressCounters,
                      help="MPLS interface ingress counters" )

   def printRow( self, intf, inOctets, inPkts ):
      print( f"{intf:10} {inOctets:>18} {inPkts:>15}" )

   def render( self ):
      if not self.interfaces:
         return
      self.printRow( "Interface", "MPLSInOctets", "MPLSInPkts" )

      for intf in Arnet.sortIntf( self.interfaces ):
         x = self.interfaces[ intf ]
         self.printRow( IntfCli.Intf.getShortname( intf ), x.mplsInOctets,
                        x.mplsInPkts )

class MplsRouteAckTableEntry( Model ):
   requestedId = Int( optional=True,
                      help="MPLS route version ID to be acknowledged" )
   acknowledgedId = Int( optional=True, help="Acknowledged MPLS route version ID" )
   installedFecId = Int( optional=True, help="Installed FEC ID for MPLS route" )
   installedFecVersion = Int( optional=True, help="Installed FEC version ID" )
   fecProgrammingStatus = Enum(
      optional=True, values=FecHwProgrammingState.attributes,
      help="Programming status of the installed FEC ID" )

class MplsRouteAckTable( Model ):
   routes = Dict( keyType=str, valueType=MplsRouteAckTableEntry,
                  help="MPLS route acknowledgements by labels" )

   def render( self ):
      headings = (
         "Labels",
         "Requested ID",
         "Acknowledged ID",
         "Installed FEC ID",
         "Installed FEC Version",
         "FEC Programming Status"
      )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      infinity = int( 2**31 - 1 )
      table = createTable( headings, tableWidth=infinity )
      table.formatColumns( fl, fr, fr, fr, fr, fr )
      for labels, entry in sorted( self.routes.items() ):
         table.newRow( labels,
                       entry.requestedId,
                       entry.acknowledgedId,
                       entry.installedFecId,
                       entry.installedFecVersion,
                       entry.fecProgrammingStatus )
      print( table.output() )
