#!/usr/bin/env python3
# Copyright (c) 2014, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import AclCliLib
import BasicCliModes
from BasicCliSession import (
   Session,
   setCommandHandlerCleanupCallback
)
import CliApi
import CliCommand
import CliCommon
from CliDynamicSymbol import CliDynamicPlugin
from ClientCommonLib import (
   createSocket,
   LspPingTypeBgpLu,
   LspPingTypeVpn,
   LspPingTypeLdp,
   LspPingTypeSr,
   pingUseCapi,
   setThreadLocalData,
   tracerouteUseCapi,
)
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
from CliPlugin import MplsUtilCli as Globals
import CliPlugin.MplsCli as MplsCli # pylint: disable=consider-using-from-import
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import ConfigMount
import CmdExtension
import LazyMount
from MplsPingClientLib import renderAndCleanPingSocket
from MplsTracerouteClientLib import cleanTracerouteSocket
import os
import sys
import Tracing
from SysConstants.in_h import IPPROTO_UDP
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "MplsUtilCliHandler" )
t0 = Tracing.trace0
MplsUtilModel = CliDynamicPlugin( "MplsUtilModel" )

genPing = Globals.genPing
genTraceroute = Globals.genTraceroute
aclStatus = None
aclCheckpoint = None
mplsUtilsConfig = None

ArnetIpAddr = TacLazyType( 'Arnet::IpAddr' )
ArnetIp6Addr = TacLazyType( 'Arnet::Ip6Addr' )
ArnetIpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
Ip6Prefix = TacLazyType( 'Arnet::Ip6Prefix' )
MplsOamStandard = TacLazyType( 'MplsUtils::MplsOamStandard' )
Prefix = TacLazyType( 'Arnet::Prefix' )
LspPingConstant = TacLazyType( 'LspPing::LspPingConstant' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
igpProtocolType = TacLazyType( 'LspPing::LspPingSrIgpProtocol' )
LspPingUtil = 'LspPing'
LspPingService = LspPingConstant.lspPingServiceName
LspPingPort = LspPingConstant.lspPingUdpPort
LspTracerouteUtil = 'LspTraceroute'
NtpWarning = 'Warning: NTP synchronization is required for ' \
             '1-Way time measurement accuracy.'

def buildLspPingArgs( options ):
   defaultArgs = {}
   for o in options:
      defaultArgs[ o[ 0 ] ] = o[ 1 ]
   optionToArg = { 'repeat' : '--count',
                   'pad-reply' : '--padReply',
                   'tos' : '--tos',
                   'size' : '--size',
                   'source' : '--src',
                   'standard' : '--standard',
                   'traffic-class' : '--tc',
                   'multi-label' : '--multiLabel',
                   'egress-validate' : '--egressValidateAddress',
                   'jitter' : '--jitter',
                   'node-responder-id' : '--responderAddr',
                   'interval' : '--interval' }
   args = []
   for arg, val in defaultArgs.items():
      args.append( optionToArg[ arg ] )
      args.append( str( val ) )
   return args

def validateLspTracerouteArgs( options, mode ):
   defaultArgs = {}
   for o in options:
      defaultArgs[ o[ 0 ] ] = o[ 1 ]

   if 'base' in defaultArgs:
      testGenAddr = defaultArgs[ 'base' ]

      if testGenAddr.af == AddressFamily.ipv4:
         loopbackPrefix = Prefix( ArnetIpAddr.ipAddrLoopbackBase, 8 )
         testAddr = testGenAddr.v4Addr
      else:
         loopbackPrefix = Ip6Prefix( ArnetIp6Addr( 0, 0, 0xffff, 0x7f000000 ), 108 )
         testAddr = testGenAddr.v6Addr

      if not loopbackPrefix.contains( testAddr ):
         mode.addError( 'base must be within %s' % loopbackPrefix )
         return False

   return True

def buildLspTracerouteArgs( options ):
   defaultArgs = {}
   for o in options:
      defaultArgs[ o[ 0 ] ] = o[ 1 ]
   optionToArg = { 'source' : '--src',
                   'traffic-class' : '--tc',
                   'pad-reply' : '--padReply',
                   'tos' : '--tos',
                   'size' : '--size',
                   'standard' : '--standard',
                   'base' : '--multipathbase',
                   'count' : '--multipathcount',
                   'timeout' : '--interval',
                   'max-ttl' : '--maxTtl',
                   'dstype' : '--dstype',
                   'multi-label' : '--multiLabel',
                   'egress-validate' : '--egressValidateAddress',
                   'jitter' : '--jitter',
                   'node-responder-id' : '--responderAddr',
                 }
   optionToArglessArg = { 'multipath' : '--multipath', }
   args = []
   for arg, val in defaultArgs.items():
      if arg in optionToArg:
         args.append( optionToArg[ arg ] )
         args.append( str( val ) )
      if arg in optionToArglessArg:
         args.append( optionToArglessArg[ arg ] )
   return args

def getRdFromPrefix( mode, prefix, vrf, af ):
   session = Session( BasicCliModes.EnableMode,
                      mode.session.entityManager,
                      privLevel=CliCommon.MAX_PRIV_LVL,
                      disableAaa=True,
                      disableAutoMore=True,
                      isEapiClient=True,
                      shouldPrint=False,
                      disableGuards=not mode.session.guardsEnabled(),
                      interactive=False,
                      cli=mode.session.cli,
                      aaaUser=mode.session.aaaUser() )
   capiExecutor = CliApi.CapiExecutor( mode.session.cli, session,
                                       stateless=False )
   cmd = f'show {af} bgp {prefix} vrf {vrf}'
   result = capiExecutor.executeCommands( [ CliApi.CliCommand( cmd ) ],
                                          textOutput=False,
                                          autoComplete=True,
                                          streamFd=None )
   model = result[ 0 ].model
   rd = None
   if ( model and isinstance( model, dict ) and
        'vrfs' in model and vrf in model[ 'vrfs' ] ):
      routeEntries = model[ 'vrfs' ][ vrf ].get( 'bgpRouteEntries' )
      if ( routeEntries and prefix in routeEntries ):
         bgpRoutePaths = routeEntries[ prefix ].get( 'bgpRoutePaths' )
         for route in bgpRoutePaths:
            rd = route.get( 'importedEvpnPathRd' )
            if rd:
               return rd
   return rd

def doCallLspUtil( mode, lspUtil, args, vrfName=VrfCli.DEFAULT_VRF,
                   servSockPort=None ):
   cliCmdExt = CmdExtension.getCmdExtender()
   args = [ 'sudo', lspUtil ] + args
   args.append( '--cli' )

   def useCapi( lspUtil, args ):
      if lspUtil == LspPingUtil:
         return any( pingUseCapi( kw ) for kw in args )
      elif lspUtil == LspTracerouteUtil:
         return any( tracerouteUseCapi( kw ) for kw in args )
      return False

   useSocket = useCapi( lspUtil, args )
   if useSocket:
      args += [ '--servSockPort', str( servSockPort ) ]
   try:
      # Flush stdout before spawning subprocess as the output to stdout is
      # buffered: that is, print statements actually write to a buffer, and
      # this buffer is only occassionally flushed to the terminal. Each
      # process has a separate buffer, which is why writes from different
      # processes can appear out of order
      sys.stdout.flush()
      # breadth test env
      # If testing arguments passed to the script, calling it with print-arg option.
      if os.environ.get( 'MOCK_LSP_UTIL' ) == 'printArg':
         args.append( '--print-arg' )
      p = cliCmdExt.subprocessPopen( args, mode.session, vrfName=vrfName,
                                     stdout=sys.stdout, stderr=sys.stderr )
      setThreadLocalData( 'p', p )
      if not useSocket:
         p.communicate()
   except OSError as e:
      mode.addError( e.strerror )

def handlerPingRawCmd( mode, args ):
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )
   # explicit-null is returned as '0' or '2'( i.e string )
   # while any other label is returned as int. Hence make all labels alike.
   if ( str( MplsLabel.explicitNullIpv4 ) in args[ 'LABELS' ] and
         str( MplsLabel.explicitNullIpv6 ) in args[ 'LABELS' ] ):
      mode.addError( 'Invalid labelstack with both %d, %d present'
                        % ( MplsLabel.explicitNullIpv4,
                           MplsLabel.explicitNullIpv6 ) )
      return
   labelStack = ','.join( str( lbl ) for lbl in args[ 'LABELS' ] )
   cmdArgs = [ '--type', 'raw', '--label', labelStack ]
   cmdArgs += buildLspPingArgs( options )
   if 'MACADDR' in args:
      cmdArgs += [ '--dmac', str( args[ 'MACADDR' ] ),
                  '--interface', str( args[ 'INTF' ] ), '127.0.0.1' ]
      print( 'LSP raw ping to {}'.format( args[ 'MACADDR' ] ) )
   else:
      destAddr = str( args[ 'NEXTHOP' ] )
      cmdArgs += [ '--nexthop', destAddr, destAddr ]
      print( f'LSP raw ping to {destAddr}' )
   doCallLspUtil( mode, LspPingUtil, cmdArgs, vrfName=vrfName )

def doSetLspPingAclNameMplsUtilsConfigMode( mode, aclType, aclName, vrfName ):
   if vrfName is None:
      vrfName = VrfCli.DEFAULT_VRF
   AclCliLib.setServiceAcl( mode, LspPingService, IPPROTO_UDP,
                            Globals.aclConfig, Globals.aclCpConfig, aclName,
                            aclType=aclType, vrfName=vrfName,
                            port=[ LspPingPort ], sport=[ LspPingPort ] )

def doDelLspPingAclNameMplsUtilsConfigMode( mode, aclType, vrfName ):
   # `default mpls ping` specifies no VRF, but should clear all instead.
   vrfs = Globals.aclCpConfig.cpConfig[ aclType ].serviceAcl
   if vrfName:
      vrfs = [ vrfName ]

   for vrf in vrfs:
      AclCliLib.noServiceAcl( mode, LspPingService,
                              Globals.aclConfig, Globals.aclCpConfig,
                              None, aclType=aclType, vrfName=vrf )

def noLspPingServiceAcl( mode, vrfName ):
   doDelLspPingAclNameMplsUtilsConfigMode( mode, 'ip', vrfName )

def noLspPingServiceAclV6( mode, vrfName ):
   doDelLspPingAclNameMplsUtilsConfigMode( mode, 'ipv6', vrfName )

def handlerPingLdpCmd( mode, args ):
   prefix = args[ 'IP_ADDRESS' ]
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   model = MplsUtilModel.MplsPing( ping=genPing )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )
   args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
            '--type', LspPingTypeLdp ] + buildLspPingArgs( options )
   if mode.session_.outputFormat_ == 'json':
      args.append( '--json' )
   else:
      print( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerPingMplsP2mpMldpCmd( mode, args ):
   rootAddr = args.get( 'ROOTADDR' )
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   model = MplsUtilModel.MplsPing( ping=genPing )
   if rootAddr:
      opqVal = args.get( 'OPQVAL' )
      sourceAddr = args.get( 'SOURCEADDR' )
      groupAddr = args.get( 'GROUPADDR' )
      lspArgs = [ rootAddr.stringValue ]
      lspArgs += [ '--type', 'mldp' ]
      if opqVal:
         lspArgs += [ '--genOpqVal', str( opqVal ) ]
      elif sourceAddr and groupAddr:
         lspArgs += [ '--sourceAddrOpqVal', sourceAddr.stringValue ]
         lspArgs += [ '--groupAddrOpqVal', groupAddr.stringValue ]
      else:
         mode.addError( 'Invalid Opaque Value' )
         return model
      lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
      if mode.session_.outputFormat_ == 'json':
         lspArgs.append( '--json' )
      else:
         print( NtpWarning )
      doCallLspUtil( mode, LspPingUtil, lspArgs, servSockPort=servSockPort )
   else:
      mode.addError( 'Invalid Root Addr' )
   return model

def handlerTracerouteMplsP2mpMldpCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( cleanTracerouteSocket )
   model = MplsUtilModel.MplsTraceroute( traceroute=genTraceroute )
   rootAddr = args[ 'ROOTADDR' ]
   if rootAddr:
      opqVal = args.get( 'OPQVAL' )
      sourceAddr = args.get( 'SOURCEADDR' )
      groupAddr = args.get( 'GROUPADDR' )
      lspArgs = [ rootAddr.stringValue ]
      lspArgs += [ '--type', 'mldp' ]
      if opqVal:
         lspArgs += [ '--genOpqVal', str( args[ 'OPQVAL' ] ) ]
      elif sourceAddr and groupAddr:
         lspArgs += [ '--sourceAddrOpqVal', sourceAddr.stringValue ]
         lspArgs += [ '--groupAddrOpqVal', groupAddr.stringValue ]
      else:
         mode.addError( 'Invalid Opaque Value' )
         return model
      lspArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
      mode.addWarning( NtpWarning )
      doCallLspUtil( mode, LspTracerouteUtil, lspArgs, servSockPort=servSockPort )
   else:
      mode.addError( 'Invalid Root Addr' )
   return model

def handlerPingGenericCmd( mode, args ):
   prefix = args[ 'IP_ADDRESS' ]
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   model = MplsUtilModel.MplsPing( ping=genPing )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )
   args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
            '--type', 'generic' ] + buildLspPingArgs( options )
   if mode.session_.outputFormat_ == 'json':
      args.append( '--json' )
   else:
      print( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def flexAlgoNameToId( algoName ):
   if not algoName:
      return 0
   for algoId, fad in Globals.flexAlgoConfig.definition.items():
      if fad.name == algoName:
         return algoId
   return None

def handlerPingSrCmd( mode, args ):
   prefix = args.get( 'IP_ADDRESS', args.get( 'IPV6_ADDRESS' ) )
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   model = MplsUtilModel.MplsPing( ping=genPing )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   algoName = args.get( 'ALGO_NAME' )
   algoId = flexAlgoNameToId( algoName )
   if algoName and not algoId:
      mode.addError( 'Flexible algorithm %s is not defined' % algoName )
      return model
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )
   srTunnel = args.get( igpProtocolType.isis,
                        args.get( igpProtocolType.ospf, '' ) )
   args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
               '--type', LspPingTypeSr,
               '--srTunnel', srTunnel ] + buildLspPingArgs( options )
   if algoName:
      args += [ '--algorithm', str( algoId ) ]
   if mode.session_.outputFormat_ == 'json':
      args.append( '--json' )
   else:
      print( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerPingMplsStaticCmd( mode, args ):
   fec = args[ 'IP_ADDRESS' ] if 'ip' in args else args[ 'IPV6_ADDRESS' ]
   if not fec:
      mode.addError( 'Invalid prefix' )
      return
   fec = str( fec )
   pingArgs = [ fec, '--type', 'static', ] + buildLspPingArgs( args[ 'OPTIONS' ] )
   print( NtpWarning )
   print( f'LSP ping to static MPLS push label route {fec}' )
   doCallLspUtil( mode, LspPingUtil, pingArgs )

def handlerPingMplsNhgCmd( mode, args ):
   nhgName = args[ 'NHG' ]
   lspArgs = [ nhgName, '--type', 'nexthop-group' ]
   entry = args.get( 'ENTRY' )
   if 'backup' in args:
      lspArgs += [ '--backup' ]
   if entry is not None:
      lspArgs += [ '--entry', str( entry ) ]
   lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
   print( NtpWarning )
   entryInfo = f', {"backup " if ( "backup" in args ) else ""}entry {entry}'
   print( f'LSP ping to nexthop-group {nhgName}'
          f'{entryInfo if ( entry is not None ) else ""}' )
   doCallLspUtil( mode, LspPingUtil, lspArgs )

def handlerPingMplsNhgTunnelCmd( mode, args ):
   lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'nexthop-group-tunnel' ]
   if 'entry' in args:
      lspArgs += [ '--entry', str( args[ 'ENTRY' ] ) ]
   lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
   print( NtpWarning )
   print( 'LSP ping to nexthop-group tunnel {}'.format(
            str( args[ 'ENDPOINT' ] ) ) )
   doCallLspUtil( mode, LspPingUtil, lspArgs )

def handlerPingMplsSrTeCmd( mode, args ):
   lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'SrTePolicy',
               '--color', str( args[ 'COLOR' ] ) ]
   if 'traffic-af' in args:
      trafficAf = 'v4' if 'v4' in args else 'v6'
      lspArgs += [ '--trafficAf', trafficAf ]
   lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
   displayStr = 'LSP ping to SR-TE Policy endpoint {} color {}'.format(
                                                      args[ 'ENDPOINT' ],
                                                      args[ 'COLOR' ] )
   if '--egressValidateAddress' in lspArgs:
      egressValidateAddress = \
                  lspArgs[ lspArgs.index( '--egressValidateAddress' ) + 1 ]
      displayStr += ' egress-validate: '
      displayStr += egressValidateAddress

   print( displayStr )

   doCallLspUtil( mode, LspPingUtil, lspArgs )

def handlerPingRsvpTunnelCmd( mode, args ):
   vrfName = None
   tunnelName = args.get( 'TUNNEL_NAME' )
   subTunnelId = args.get( 'SUB_TUNNEL_ID' )
   options = args.get( 'OPTIONS' )
   model = MplsUtilModel.MplsPing( ping=genPing )
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   args = [ '--type', 'rsvp', '--tunnel', str( tunnelName ) ]
   if subTunnelId:
      args += [ '--sub-tunnel-id', str( subTunnelId ) ]
   if options:
      args += buildLspPingArgs( options )
   if mode.session_.outputFormat_ == 'json':
      args.append( '--json' )
   mode.addWarning( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerPingRsvpCmd( mode, args ):
   vrfName = None
   rsvpId = args.get( 'SESSION_ID' )
   name = args.get( 'SESSION_NAME' )
   lspOption = args.get( 'LSP' )
   options = args.get( 'OPTIONS' )
   model = MplsUtilModel.MplsPing( ping=genPing )
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   # rsvpId could be 0, so explicitly check is not None
   if rsvpId is not None:
      args = [ '--type', 'rsvp', '--session-id', str( rsvpId ) ]
      if lspOption:
         args += [ '--lsp', str( lspOption ) ]
      if options:
         args += buildLspPingArgs( options )
   elif name:
      args = [ '--type', 'rsvp', '--session-name', str( name ) ]
      if options:
         args += buildLspPingArgs( options )
   else:
      mode.addError( 'No session' )
      return model
   if mode.session_.outputFormat_ == 'json':
      args.append( '--json' )
   mode.addWarning( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerPingPseudowireLdpCmd( mode, args ):
   pwName = args.get( 'PW_NAME' )
   options = args.get( 'OPTIONS' )
   model = MplsUtilModel.MplsPing( ping=genPing )
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   if not pwName:
      instanceName = args[ 'INSTANCE_NAME' ]
      groupName = args[ 'GROUP_NAME' ]
      neighborAddress = args[ 'NEIGHBOR_ADDRESS' ]
      pwName = f'profile {instanceName} {groupName} {neighborAddress}'
   args = [ pwName, '--type', 'pwLdp' ]
   args += buildLspPingArgs( options )
   if mode.session_.outputFormat_ == 'json':
      args.append( '--json' )
   mode.addWarning( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, args,
                  servSockPort=servSockPort )
   return model

def handlerPingVpnCmd( mode, args ):
   prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )
   model = MplsUtilModel.MplsPing( ping=genPing )
   if not prefix:
      servSockPort = createSocket()
      setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
      mode.addError( 'Invalid prefix' )
      return model
   af = 'ip' if 'ip' in args else 'ipv6'
   vrf = args[ 'VRF' ]
   prefix = ArnetIpGenPrefix( str( prefix ) ).stringValue
   rd = getRdFromPrefix( mode, prefix, vrf, af )
   options = args.get( 'OPTIONS' )
   lspArgs = [ prefix, '--type', LspPingTypeVpn ]
   lspArgs += [ '--vrf', vrf ]
   if rd:
      lspArgs += [ '--rd', rd ]
   lspArgs += buildLspPingArgs( options )
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   if mode.session_.outputFormat_ == 'json':
      lspArgs.append( '--json' )
   else:
      # for capi json output, no need to store this warning in model.
      print( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, lspArgs, servSockPort=servSockPort )
   return model

def handlerPingBgpLuCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( renderAndCleanPingSocket )
   prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )
   model = MplsUtilModel.MplsPing( ping=genPing )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   prefix = ArnetIpGenPrefix( str( prefix ) ).stringValue
   nexthop = args.get( 'NEXTHOP' ) or args.get( 'NEXTHOP6' )
   labels = args.get( 'LABELS' )
   if labels and not MplsCli.validateLabelStackSize( mode, labels ):
      mode.addError( 'Invalid labelstack size' )
      return model
   options = args.get( 'OPTIONS' )
   lspArgs = [ prefix, '--type', LspPingTypeBgpLu ]
   if nexthop:
      if not isinstance( nexthop, str ):
         nexthop = nexthop.stringValue
      lspArgs += [ '--nexthop', nexthop ]
      if labels:
         lspArgs += [ '--label', ','.join( str( l ) for l in labels ) ]
   lspArgs += buildLspPingArgs( options )
   if mode.session_.outputFormat_ == 'json':
      lspArgs.append( '--json' )
   else:
      # for capi json output, no need to store this warning in model.
      print( NtpWarning )
   doCallLspUtil( mode, LspPingUtil, lspArgs, servSockPort=servSockPort )
   return model

def handlerTracerouteRawCmd( mode, args ):
   # explicit-null is returned as '0' or '2'( i.e string )
   # while any other label is returned as int. Hence make labels all alike.
   if ( str( MplsLabel.explicitNullIpv4 ) in args[ 'LABELS' ] and
         str( MplsLabel.explicitNullIpv6 ) in args[ 'LABELS' ] ):
      mode.addError( 'Invalid labelstack with both %d, %d present'
                        % ( MplsLabel.explicitNullIpv4,
                           MplsLabel.explicitNullIpv6 ) )
   labelStack = ','.join( str( lbl ) for lbl in args[ 'LABELS' ] )
   cmdArgs = [ '--type', 'raw', '--label', labelStack ]
   cmdArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
   if 'MACADDR' in args:
      cmdArgs += [ '--dmac', args[ 'MACADDR' ],
                  '--interface', str( args[ 'INTF' ] ), '127.0.0.1' ]
      print( NtpWarning )
      print( 'LSP traceroute to {}'.format( args[ 'MACADDR' ] ) )
   else:
      destAddr = args[ 'NEXTHOP' ]
      cmdArgs += [ '--nexthop', destAddr, destAddr ]
      print( NtpWarning )
      print( f'LSP traceroute to {destAddr}' )
   doCallLspUtil( mode, LspTracerouteUtil, cmdArgs, vrfName=args.get( 'VRF' ) )

def handlerTracerouteLdpCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( cleanTracerouteSocket )
   model = MplsUtilModel.MplsTraceroute( traceroute=genTraceroute )
   prefix = args[ 'IP_ADDRESS' ]
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )

   args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
            '--type', 'ldp' ]

   if validateLspTracerouteArgs( options, mode ):
      args += buildLspTracerouteArgs( options )
      mode.addWarning( NtpWarning )
      doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName,
                     servSockPort=servSockPort )
   return model

def handlerTracerouteGenericCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( cleanTracerouteSocket )
   model = MplsUtilModel.MplsTraceroute( traceroute=genTraceroute )
   prefix = args[ 'IP_ADDRESS' ]
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )

   args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
            '--type', 'generic' ]

   if validateLspTracerouteArgs( options, mode ):
      args += buildLspTracerouteArgs( options )
      mode.addWarning( NtpWarning )
      doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName,
                     servSockPort=servSockPort )
   return model

def handlerTracerouteSrCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( cleanTracerouteSocket )
   model = MplsUtilModel.MplsTraceroute( traceroute=genTraceroute )
   prefix = args.get( 'IP_ADDRESS', args.get( 'IPV6_ADDRESS' ) )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return model
   algoName = args.get( 'ALGO_NAME' )
   algoId = flexAlgoNameToId( algoName )
   if algoName and not algoId:
      mode.addError( 'Flexible algorithm %s is not defined' % algoName )
      return model
   vrfName = args.get( 'VRF' )
   options = args.get( 'OPTIONS' )
   srTunnel = args.get( igpProtocolType.isis,
                        args.get( igpProtocolType.ospf, '' ) )
   args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
               '--type', 'segment-routing', '--srTunnel', srTunnel ]
   if not validateLspTracerouteArgs( options, mode ):
      return model

   args += buildLspTracerouteArgs( options )
   if algoName:
      args += [ '--algorithm', str( algoId ) ]
      args += [ '--algorithmName', str( algoName ) ]
   mode.addWarning( NtpWarning )
   doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerTracerouteMplsStaticCmd( mode, args ):
   prefix = args.get( 'IP_ADDRESS', args.get( 'IPV6_ADDRESS' ) )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return
   prefix = str( prefix )
   tracerouteArgs = [ prefix, '--type', 'static' ]
   if validateLspTracerouteArgs( args[ 'OPTIONS' ], mode ):
      tracerouteArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
      if 'nexthop' in args:
         nexthopIp = args[ 'NEXTHOP' ] if 'NEXTHOP' in args else \
                     args[ 'NEXTHOP6' ]
         nexthopIp = str( nexthopIp )
         label = str( args[ 'LABEL' ] )
         tracerouteArgs += [ '--label', label, '--nexthop', nexthopIp ]
      print( NtpWarning )
      print( f'LSP traceroute to {prefix}' )
      doCallLspUtil( mode, LspTracerouteUtil, tracerouteArgs )

def handlerTracerouteMplsNhgCmd( mode, args ):
   nhgName = args[ 'NHG' ]
   entry = args[ 'ENTRY' ]
   lspArgs = [ nhgName, '--type', 'nexthop-group', '--entry', str( entry ) ]
   if 'backup' in args:
      lspArgs += [ '--backup' ]
   lspArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
   print( NtpWarning )
   print( f'LSP traceroute to nexthop-group {nhgName}, '
          f'{"backup " if ( "backup" in args ) else ""}entry {entry}' )
   doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

def handlerTracerouteMplsNhgTunnelCmd( mode, args ):
   lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'nexthop-group-tunnel' ]
   if 'entry' in args:
      lspArgs += [ '--entry', str( args[ 'ENTRY' ] ) ]
   lspArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
   print( NtpWarning )
   print( 'LSP traceroute to nexthop-group tunnel', args[ 'ENDPOINT' ] )
   doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

def handlerTracerouteMplsSrTeCmd( mode, args ):
   lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'SrTePolicy',
               '--color', str( args[ 'COLOR' ] ) ]
   if 'traffic-af' in args:
      trafficAf = 'v4' if 'v4' in args else 'v6'
      lspArgs += [ '--trafficAf', trafficAf ]
   lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
   print( NtpWarning )
   print( 'LSP traceroute to SR-TE Policy endpoint {} color {}'.format(
                                                      str( args[ 'ENDPOINT' ] ),
                                                      str( args[ 'COLOR' ] ) ) )
   doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

def handlerTracerouteRsvpTunnelCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( cleanTracerouteSocket )
   model = MplsUtilModel.MplsTraceroute( traceroute=genTraceroute )
   vrfName = None
   tunnelName = args.get( 'TUNNEL_NAME' )
   subTunnelId = args.get( 'SUB_TUNNEL_ID' )
   options = args.get( 'OPTIONS' )
   args = [ '--type', 'rsvp', '--tunnel', str( tunnelName ) ]
   if subTunnelId:
      args += [ '--sub-tunnel-id', str( subTunnelId ) ]
   if options:
      args += buildLspPingArgs( options )
   mode.addWarning( NtpWarning )
   doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerTracerouteRsvpCmd( mode, args ):
   servSockPort = createSocket()
   setCommandHandlerCleanupCallback( cleanTracerouteSocket )
   model = MplsUtilModel.MplsTraceroute( traceroute=genTraceroute )
   rsvpId = args.get( 'SESSION_ID' )
   lsp = args.get( 'LSP' )
   name = args.get( 'SESSION_NAME' )
   options = args.get( 'OPTIONS' )
   vrfName = None
   # rsvpId and lsp could be 0, so explicitly check is not None
   if rsvpId is not None and lsp is not None:
      args = [ '--type', 'rsvp', '--session-id', str( rsvpId ),
               '--lsp', str( lsp ) ]
   elif name:
      args = [ '--type', 'rsvp', '--session-name', str( name ) ]
   else:
      mode.addError( 'No session' )
      return model
   if options:
      args += buildLspTracerouteArgs( options )
   mode.addWarning( NtpWarning )
   doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName,
                  servSockPort=servSockPort )
   return model

def handlerTracerouteBgpLuCmd( mode, args ):
   prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )
   if not prefix:
      mode.addError( 'Invalid prefix' )
      return
   prefix = ArnetIpGenPrefix( str( prefix ) ).stringValue
   nexthop = args.get( 'NEXTHOP' ) or args.get( 'NEXTHOP6' )
   labels = args.get( 'LABELS' )
   if labels and not MplsCli.validateLabelStackSize( mode, labels ):
      return
   options = args.get( 'OPTIONS' )
   lspArgs = [ prefix, '--type', LspPingTypeBgpLu ]
   if nexthop:
      if not isinstance( nexthop, str ):
         nexthop = nexthop.stringValue
      lspArgs += [ '--nexthop', nexthop ]
      if labels:
         lspArgs += [ '--label', ','.join( str( l ) for l in labels ) ]
   lspArgs += buildLspTracerouteArgs( options )
   print( NtpWarning )
   print( 'LSP traceroute to BGP labeled unicast tunnel', prefix )
   doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

def handlerIpAccessGroupAclnameCmd( mode, args ):
   doSetLspPingAclNameMplsUtilsConfigMode(
      mode, 'ip', args[ 'ACL_NAME' ], args.get( 'VRF' ) )

def noOrDefaultHandlerIpAccessGroupAclnameCmd( mode, args ):
   noLspPingServiceAcl( mode, args.get( 'VRF' ) )

def handlerIpv6AccessGroupAclNameCmd( mode, args ):
   doSetLspPingAclNameMplsUtilsConfigMode(
      mode, 'ipv6', args[ 'ACL_NAME' ], args.get( 'VRF' ) )

def noOrDefaultHandlerIpv6AccessGroupAclNameCmd( mode, args ):
   noLspPingServiceAclV6( mode, args.get( 'VRF' ) )

def handlerMplsOamStandardCmd( mode, args ):
   if ( 'arista' in args or 'ietf' in args ):
      if 'arista' in args:
         mplsUtilsConfig.oamStandard = MplsOamStandard.arista
      else:
         mplsUtilsConfig.oamStandard = MplsOamStandard.ietf
   elif ( 'downstream' in args and 'validation' in args and
            'enabled' in args ):
      mplsUtilsConfig.dsMapValidation = True

def noOrDefaultHandlerMplsOamStandardCmd( mode, args ):
   if ( 'arista' in args or 'ietf' in args ):
      if ( 'arista' in args and
            mplsUtilsConfig.oamStandard != MplsOamStandard.arista ):
         mode.addWarning( 'Given standard is not configured' )
      elif ( 'ietf' in args and
               mplsUtilsConfig.oamStandard != MplsOamStandard.ietf ):
         mode.addWarning( 'Given standard is not configured' )
      else:
         mplsUtilsConfig.oamStandard = MplsOamStandard.arista
   elif ( 'downstream' in args and 'validation' in args and
            'enabled' in args ):
      mplsUtilsConfig.dsMapValidation = False

def handlerShowMplsPingAcl( mode, args ):
   aclType = args.get( 'ip', args.get( 'ipv6' ) )
   return AclCli.showServiceAcl( mode,
                                 Globals.aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 args[ '<aclNameExpr>' ],
                                 serviceName=LspPingService )

def handlerClearIpAclCounters( mode, args ):
   aclType = args.get( 'ip', args.get( 'ipv6' ) )
   return AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

def getIntfId( mode, intf ):
   intfId = None
   if intf.config():
      intfId = intf.config().intfId
   else:
      mode.addError( f"Cannot find interface {intf}" )
   return intfId

def handlerMplsIcmpSrcInterface( mode, args ):
   intf = args[ 'SRC_INTF' ]
   intfId = getIntfId( mode, intf )
   if not intfId:
      t0( 'could not fetch intfId from intf ', intf )
      return
   if 'ipv6' in args:
      mplsUtilsConfig.mplsIcmpIpv6SrcIntfId = intfId
   else:
      mplsUtilsConfig.mplsIcmpIpv4SrcIntfId = intfId

def noOrDefaultHandlerMplsIcmpSrcInterface( mode, args ):
   if 'ipv6' in args:
      mplsUtilsConfig.mplsIcmpIpv6SrcIntfId = ""
   else:
      mplsUtilsConfig.mplsIcmpIpv4SrcIntfId = ""

def handlerMplsIcmpTunnelingCmd( mode, args ):
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   if 'fragmentation-needed' in args:
      mplsUtilsConfig.fragNeededTunneling = not noOrDefault
   elif 'ttl-exceeded' in args:
      mplsUtilsConfig.icmpTtlTunneling = not noOrDefault
   else:
      assert False, 'Uknown ICMP type'

def handlerMplsPingCmd( mode, args ):
   childMode = mode.childMode( Globals.MplsUtilsConfigMode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerMplsPingCmd( mode, args ):
   childMode = mode.childMode( Globals.MplsUtilsConfigMode )
   vrfName = None # Clear all.
   noLspPingServiceAcl( childMode, vrfName )
   noLspPingServiceAclV6( childMode, vrfName )

def Plugin( entityManager ):
   global mplsUtilsConfig
   global aclStatus
   global aclCheckpoint

   mplsUtilsConfig = ConfigMount.mount( entityManager, 'mplsutils/config',
                                        'MplsUtils::Config', 'w' )
   aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                'Acl::Status', 'r' )
   aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                   'Acl::CheckpointStatus', 'w' )
