#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.MvpnCli import (
   MvpnIpv4VrfConfigMode,
   MvpnIpv4VrfSpmsiConfigMode,
   MvpnIpv4VrfIntraAsIPmsiConfigMode
)
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   configForVrf,
)
from CliPlugin.RoutingBgpNeighborCli import (
   neighborAfActivateHelper,
   PeerAfStateEnum,
)
from CliPlugin.RoutingBgpInstanceCli import resetBgpAfModeConfig
from IpLibTypes import ProtocolAgentModelType

def MvpnIpv4ModeCmd_handler( mode, args ):
   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.ribd:
      mode.addWarning( 'Routing protocols model multi-agent must be '
                        'configured for MVPN address-family' )
   if 'mvpn-ipv4' not in afModeExtensionHook.afModeExtensions():
      mode.addError( 'Invalid mvpn-ipv4 extension' )
      return
   childMode = mode.childMode(
      afModeExtensionHook.afModeExtension[ 'mvpn-ipv4' ] )
   mode.session_.gotoChildMode( childMode )

def MvpnIpv4ModeCmd_noOrDefaultHandler( mode, args ):
   bgpConfig = configForVrf( mode.vrfName )
   resetBgpAfModeConfig( bgpConfig, 'mvpn-ipv4', mode.vrfName )

def NeighborDefaultEncapProtocolCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4Encap = 'encapMpls'
   # Only P2MP is supported for the mLDP and RSVP-TE protocols
   if args[ 'PROTOCOL' ] == 'mldp':
      config.afMvpnV4PmsiTunnelType = 'mLdpP2mpLsp'
   else:
      assert args[ 'PROTOCOL' ] == 'rsvp'
      config.afMvpnV4PmsiTunnelType = 'rsvpTeP2mpLsp'

def NeighborDefaultEncapProtocolCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4PmsiTunnelType = config.afMvpnV4PmsiTunnelTypeDefault
   if config.nexthopSelfSrcIntfMvpnV4 == config.nexthopSelfSrcIntfMvpnV4Default:
      config.afMvpnV4Encap = config.afMvpnV4EncapDefault

def NeighborSourceInterfaceCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4Encap = 'encapMpls'
   config.nexthopSelfSrcIntfMvpnV4 = args[ 'INTF' ].name

def NeighborSourceInterfaceCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.nexthopSelfSrcIntfMvpnV4 = config.nexthopSelfSrcIntfMvpnV4Default
   if config.afMvpnV4PmsiTunnelType == config.afMvpnV4PmsiTunnelTypeDefault:
      config.afMvpnV4Encap = config.afMvpnV4EncapDefault

def NeighborActivateCmd_handler( mode, args ):
   if 'activate' in args:
      afState = PeerAfStateEnum.afActive
   else:
      afState = PeerAfStateEnum.afInactive
   neighborAfActivateHelper( mode, args[ 'PEER' ], afState )

def NeighborActivateCmd_noOrDefaultHandler( mode, args ):
   neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afDefault )

def MvpnIpv4VrfCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4Vrf = True

def MvpnIpv4VrfCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4Vrf = False

def deleteSpmsiSourceGroup( vrfName ):
   config = configForVrf( vrfName )
   config.afMvpnV4VrfSpmsiTunnelSourceGroup = False

def deleteSpmsiStarStar( vrfName ):
   config = configForVrf( vrfName )
   config.afMvpnV4VrfSpmsiTunnelStarStar = False

def deleteIpmsiTunnelDisable( vrfName ):
   config = configForVrf( vrfName )
   config.afMvpnV4VrfIpmsiTunnelDisable = False

def handlerMvpnIpv4VrfModeCmd( mode, args ):
   childMode = mode.childMode( MvpnIpv4VrfConfigMode, vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerMvpnIpv4VrfModeCmd( mode, args ):
   deleteSpmsiSourceGroup( mode.vrfName )
   deleteSpmsiStarStar( mode.vrfName )
   deleteIpmsiTunnelDisable( mode.vrfName )
   config = configForVrf( mode.vrfName )
   config.afMvpnV4Vrf = False

def handlerMvpnIpv4VrfSpmsiModeCmd( mode, args ):
   childMode = mode.childMode( MvpnIpv4VrfSpmsiConfigMode, vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerMvpnIpv4VrfSpmsiModeCmd( mode, args ):
   deleteSpmsiSourceGroup( mode.vrfName )
   deleteSpmsiStarStar( mode.vrfName )

def handlerMvpnIpv4VrfIntraAsIPmsiModeCmd( mode, args ):
   childMode = mode.childMode( MvpnIpv4VrfIntraAsIPmsiConfigMode,
                               vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerMvpnIpv4VrfIntraAsIPmsiModeCmd( mode, args ):
   deleteIpmsiTunnelDisable( mode.vrfName )

def handlerMvpnIpv4VrfSpmsiSourceGroupCmd( mode, args ):
   config = configForVrf( mode.vrfName_ )
   config.afMvpnV4VrfSpmsiTunnelSourceGroup = True

def noOrDefaultHandlerMvpnIpv4VrfSpmsiSourceGroupCmd( mode, args ):
   deleteSpmsiSourceGroup( mode.vrfName_ )

def handlerMvpnIpv4VrfSpmsiStarStarCmd( mode, args ):
   config = configForVrf( mode.vrfName_ )
   config.afMvpnV4VrfSpmsiTunnelStarStar = True

def noOrDefaultHandlerMvpnIpv4VrfSpmsiStarStarCmd( mode, args ):
   deleteSpmsiStarStar( mode.vrfName_ )

def handlerMvpnIpv4VrfIPmsiTunnelNoAdvertiseCmd( mode, args ):
   config = configForVrf( mode.vrfName_ )
   config.afMvpnV4VrfIpmsiTunnelDisable = True

def noOrDefaultHandlerMvpnIpv4VrfIPmsiTunnelNoAdvertiseCmd( mode, args ):
   deleteIpmsiTunnelDisable( mode.vrfName_ )

def handlerMvpnIpv4VrfIPmsiTunnelAdvertiseCmd( mode, args ):
   deleteIpmsiTunnelDisable( mode.vrfName_ )

def noOrDefaultHandlerMvpnIpv4VrfIPmsiTunnelAdvertiseCmd( mode, args ):
   config = configForVrf( mode.vrfName_ )
   config.afMvpnV4VrfIpmsiTunnelDisable = True

def handlerMvpnIpv4RsvpP2mpTunnelProfileCmd( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4RsvpP2mpTunnelProfile = args[ 'PROFILE_NAME' ]

def noOrDefaultHandlerIpv4RsvpP2mpTunnelProfileCmd( mode, args ):
   config = configForVrf( mode.vrfName )
   config.afMvpnV4RsvpP2mpTunnelProfile = ''
