# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from CliModel import Dict, Int, Model, Str, Float, Submodel, Bool
from CliPlugin import PciCli
import TableOutput
import Tac
import Tracing
import LazyMount
import re
import time

__defaultTraceHandle__ = Tracing.Handle( "PciCli" )

entityManager = None
pciDeviceConfigDir = PciCli.pciDeviceConfigDir
sysDevices = None
cellDevices = None
pciDeviceStatusDir = None
pcieSwitchStatusDir = None
cellCliRequest = None
sysCliRequest = None

#--------------------------------------------------------------------------------
# show pci
#--------------------------------------------------------------------------------
def getPcieSwitchSmbusErrors():
   return [ ( stat.name, stat.registerAccessErrorCounter )
         for stat in pcieSwitchStatusDir.values() ]

def getPciDevices():
   pciDevices = {}
   for pciDevice in pciDeviceStatusDir.pciDeviceStatus.values():
      pciAddr = pciDevice.addr
      pciId = f"{pciAddr.busNumber:02x}:{pciAddr.device:02x}.{pciAddr.function:x}"
      pciDevices.update( { pciId:pciDevice } )
   return pciDevices

def getPciErrors():
   pciDevices = getPciDevices()
   ret = []
   for pciId in sorted( pciDevices ):
      pciDevice = pciDevices[ pciId ]
      ret.append( ( pciDevice.name, pciId, pciDevice.correctableError,
                  pciDevice.uncorrectableNonFatalError,
                  pciDevice.uncorrectableFatalError ) )
   return ret

def getPciGen():
   pciDevices = getPciDevices()
   ret = []
   for pciId in sorted( pciDevices ):
      pciDevice = pciDevices[ pciId ]
      linkSpeed = pciDevice.linkSpeed
      linkWidth = pciDevice.linkWidth

      # Format the LinkSpeed
      m = re.match( r"(\d+\.?\d*) GT\/s", linkSpeed )
      if m:
         linkSpeed = float( m.group( 1 ) )
      else:
         linkSpeed = 0.0
      
      ret.append( ( pciDevice.name, pciId, linkSpeed, linkWidth ) )
   
   return ret

def updateErrorRequestTimes():
   if cellDevices and not entityManager.locallyReadOnly():
      cellCliRequest.scanErrorRequest = Tac.now()
   if sysDevices and not entityManager.locallyReadOnly():
      sysCliRequest.scanErrorRequest = Tac.now()

def prepShowPci( mode, args ):
   updateErrorRequestTimes()

class PciErrorSum( Model ):
   name = Str( help="PCI device name" )
   correctableErrors = Int( help="Correctable error counter" )
   nonFatalErrors = Int( help="Uncorrectable non-fatal error counter" )
   fatalErrors = Int( help="Uncorrectable fatal error counter" )
   linkSpeed = Float( help="Current link speed of the PCI device in GT/s" )
   linkWidth = Int( help="Current link width of the PCI device" )

class PciErrors( Model ):
   #note, pciId (str) will be used as the key.
   pciIds = Dict( valueType=PciErrorSum, keyType=str,
                  help="Summary of error counters for each PCI device" )
   switchs = Dict( valueType=int,
                   help="Summary of smBus errors on PCIe switch" )

   def render( self ):
      header = ( "Name", "PciId", "CorrErr", "NonFatalErr", "FatalErr",
            "LinkSpeed", "LinkWidth" )
      table = TableOutput.createTable( header )
      left = TableOutput.Format( justify="left" )
      right = TableOutput.Format( justify="right" ) 
      table.formatColumns( left, left, right, right, right, left, left )

      for pciId in sorted(self.pciIds):
         t = self.pciIds[ pciId ]
         speed = ""
         if t.linkSpeed != 0.0: 
            speed = f"{t.linkSpeed:.1f} GT/s"
         
         width = ""
         if t.linkWidth != 0:
            width = f"x{t.linkWidth}"
         table.newRow( t.name, pciId, t.correctableErrors,
               t.nonFatalErrors, t.fatalErrors, speed, width )
      print( table.output() )

      if self.switchs:
         print() 

         header = ( "PcieSwitch", "SMBusERR" )
         table = TableOutput.createTable( header )
         for name in self.switchs:
            table.newRow( name, self.switchs[ name ] )
         print( table.output() )

def pciCmdHandler( mode, args ):
   pciList = getPciErrors()
   pciGenList = getPciGen()
   pciDict = {}
   for line, gen in zip( pciList, pciGenList ):
      pciDict[ line[ 1 ] ] = PciErrorSum(
            name=line[ 0 ],
            correctableErrors=line[ 2 ],
            nonFatalErrors=line[ 3 ],
            fatalErrors=line[ 4 ],
            linkSpeed=gen[ 2 ],
            linkWidth=gen[ 3 ] )

   smbusDict = {}
   if pcieSwitchStatusDir.values():
      smbusErrors = getPcieSwitchSmbusErrors()
      for line in smbusErrors:
         smbusDict[ line[ 0 ] ] = line[ 1 ]

   return PciErrors( pciIds=pciDict, switchs=smbusDict )


#-------------------------------------------------------------------------------
# The "show pci detail" command, in privileged mode.
#-------------------------------------------------------------------------------
def displayPciDetailErrors():
   pciDevices = getPciDevices()
   devSummary = {}
   for pciId, pciDevice in pciDevices.items():
      devSummary[ pciId ] = PciDetailError(
            name=pciDevice.name,
            severity=pciDevice.severity,
            pciErrors={
               # Correctable
               "replayTimerTimeout": pciDevice.replayTimerTimeout,
               "replayNumRollover": pciDevice.replayNumRollover,
               "badTlp": pciDevice.badTlp,
               "badDllp": pciDevice.badDllp,
               "receiverError": pciDevice.receiverError,
               "advisoryNonFatal": pciDevice.advisoryNonFatal,
               "correctedInternalError": pciDevice.correctedInternalError,
               "corrHdrLogOverflow": pciDevice.corrHdrLogOverflow,

               # Uncorrectable
               "surpriseDown": pciDevice.surpriseDown,
               "uncorrInternalError": pciDevice.uncorrInternalError,
               "mcBlockedTlp": pciDevice.mcBlockedTlp,
               "atomicOpEgressBlocked": pciDevice.atomicOpEgressBlocked,
               "tlpPrefixBlocked": pciDevice.tlpPrefixBlocked,
               "poisonedTlpEgressBlocked": pciDevice.poisonedTlpEgressBlocked,
               "poisonedTlp": pciDevice.poisonedTlp,
               "ecrcError": pciDevice.ecrcError,
               "unsupportedRequestError": pciDevice.unsupportedRequestError,
               "completionTimeout": pciDevice.completionTimeout,
               "completerAbort": pciDevice.completerAbort,
               "unexpectedCompletion": pciDevice.unexpectedCompletion,
               "acsViolError": pciDevice.acsViolError,
               "undefinedError": pciDevice.undefinedError,
               "dataLinkProtocolError": pciDevice.dataLinkProtocolError,
               "receiverOverflow": pciDevice.receiverOverflow,
               "flowControlProtocolError": pciDevice.flowControlProtocolError,
               "malformedTlp": pciDevice.malformedTlp,

               # Stat
               "correctableError": pciDevice.correctableError,
               "uncorrectableNonFatalError": pciDevice.uncorrectableNonFatalError,
               "uncorrectableFatalError": pciDevice.uncorrectableFatalError,
            } )
   return PciDetailErrors( devices=devSummary )

class PciDetailError( Model ):
   name = Str( help="Name of pci device" )
   severity = Int( help="Uncorrectable errors severity" )
   pciErrors = Dict( valueType=int, keyType=str,
                  help="Error counts" )

class PciDetailErrors( Model ):
   devices = Dict( keyType=str, valueType=PciDetailError,
         help="Summary of pci device errors" )

   def render( self ):
      for pciId, pciDevice in sorted( self.devices.items() ):
         print( f"{pciDevice.name} {pciId}" )
         print( "-" * 40 )
         print( f"Severity: 0x{pciDevice.severity:x}" )
         print( f"{pciDevice.pciErrors['correctableError']} Correctable Error" )
         print( f" {pciDevice.pciErrors['receiverError']} Receiver Error" )
         print( f" {pciDevice.pciErrors['badTlp']} Bad TLP" )
         print( f" {pciDevice.pciErrors['badDllp']} Bad DLLP" )
         print( f" {pciDevice.pciErrors['replayNumRollover']} "
                "Replay Number Rollover" )
         print( f" {pciDevice.pciErrors['replayTimerTimeout']} "
                "Replay Timer Time-out" )
         print( f" {pciDevice.pciErrors['advisoryNonFatal']} "
                "Advisory Non-Fatal" )
         print( f" {pciDevice.pciErrors['correctedInternalError']} "
                "Corrected Internal Error" )
         print( f" {pciDevice.pciErrors['corrHdrLogOverflow']} Header Log Overflow" )
         print( f"{pciDevice.pciErrors['uncorrectableNonFatalError']} "
                "Uncorrectable Non-Fatal Error" )
         print( f" {pciDevice.pciErrors['poisonedTlp']} Poisoned TLP Received" )
         print( f" {pciDevice.pciErrors['ecrcError']} ECRC Check Failed" )
         print( f" {pciDevice.pciErrors['unsupportedRequestError']} "
                "Unsupported Request" )
         print( f" {pciDevice.pciErrors['completionTimeout']} Completion Time-out" )
         print( f" {pciDevice.pciErrors['completerAbort']} Completion Abort" )
         print( f" {pciDevice.pciErrors['unexpectedCompletion']} "
                "Unexpected Completion" )
         print( f" {pciDevice.pciErrors['acsViolError']} ACS Violation" )
         print( f" {pciDevice.pciErrors['surpriseDown']} Surprise Down Error" )
         print( f" {pciDevice.pciErrors['atomicOpEgressBlocked']} "
                "AtomicOp Egress Blocked" )
         print( f" {pciDevice.pciErrors['uncorrInternalError']} "
                "Uncorrectable Internal Error" )
         print( f" {pciDevice.pciErrors['mcBlockedTlp']} MC Blocked TLP" )
         print( f" {pciDevice.pciErrors['tlpPrefixBlocked']} "
                "TLP Prefix Blocked Error" )
         print( f" {pciDevice.pciErrors['poisonedTlpEgressBlocked']} "
                "Poisoned TLP Egress Blocked" )
         print( f"{pciDevice.pciErrors['uncorrectableFatalError']} "
                "Uncorrectable Fatal Error" )
         print( f" {pciDevice.pciErrors['undefinedError']} Training Error" )
         print( f" {pciDevice.pciErrors['dataLinkProtocolError']} "
                "DLL Protocol Error" )
         print( f" {pciDevice.pciErrors['receiverOverflow']} Receiver Overflow" )
         print( f" {pciDevice.pciErrors['flowControlProtocolError']} "
                "Flow Control Protocol Error" )
         print( f" {pciDevice.pciErrors['malformedTlp']} Malformed Tlp" )

#--------------------------------------------------------------------------------
# show pci detail
#--------------------------------------------------------------------------------
def pciDetailCmdHandler( mode, args ):
   updateErrorRequestTimes()
   return displayPciDetailErrors()

#--------------------------------------------------------------------------------
# show pci errors rates
#--------------------------------------------------------------------------------
def getModelObjFromTacObj( tacObj, modelClass ):
   modelObj = modelClass()
   for key, valueObj in modelClass.__attributes__.items():
      value = getattr( tacObj, key ) if hasattr( tacObj, key) else tacObj.get( key )
      if isinstance( valueObj, Submodel ):
         value = getModelObjFromTacObj( value, valueObj.valueType )
      setattr( modelObj, key, value )
   return modelObj

prettyErrorTypes = {
   "CorrectableError": "Correctable",
   "UncorrectableNonFatalError": "Uncorrectable Non-Fatal",
   "UncorrectableFatalError": "Uncorrectable Fatal"
}

class PciErrorStatsUnit( Model ):
   rate = Float(
      help="Error hits per time unit (errors/second = rate/unitSeconds)" )
   unitSeconds = Int(
      help="Unit's name translated into number of seconds" )
   averagingPeriod = Int(
      help="Last number of seconds over which the estimation was made" )

class PciErrorStatsUnits( Model ):
   second = Submodel( valueType=PciErrorStatsUnit,
      help="Unit containing details about number of errors per second" )
   minute = Submodel( valueType=PciErrorStatsUnit,
      help="Unit containing details about number of errors per minute" )
   hour = Submodel( valueType=PciErrorStatsUnit,
      help="Unit containing details about number of errors per hour" )
   day = Submodel( valueType=PciErrorStatsUnit,
      help="Unit containing details about number of errors per day" )
   week = Submodel( valueType=PciErrorStatsUnit,
      help="Unit containing details about number of errors per week" )

class PciErrorStats( Model ):
   units = Submodel( valueType=PciErrorStatsUnits,
      help="List containing details of all units (from second to week)" )
   masked = Bool(
      help="Denotes whether this error type is currently being masked" )

class PciErrorsStats( Model ):
   name = Str( help="PCI device name" )
   errorStats = Dict( valueType=PciErrorStats, keyType=str,
      help="Error stats (a mapping of error types to their stats)" )

class PciErrorsStatsAll( Model ):
   devices = Dict( valueType=PciErrorsStats, keyType=str,
      help="Devices with recorded errors (a mapping of pciId to their stats)" )

   def getFmtRate( self, unitName, errorStat ):
      if errorStat.masked:
         return "high"
      unit = getattr( errorStat.units, unitName )
      if unit.rate == 0:
         return "0"
      return f"{unit.rate / unit.unitSeconds :.2e}"

   def getHeader( self, unitName, firstErrorStat ):
      unit = getattr( firstErrorStat.units, unitName )
      return f"{unit.averagingPeriod // unit.unitSeconds}-{unitName}"

   def render( self ):
      if not self.devices:
         return
      firstDevice = next( iter( self.devices.values() ) )
      firstErrorStat = next( iter( firstDevice.errorStats.values() ) )
      header = ( "Name", "PCI ID", "Error Type",
                  ( self.getHeader( "second", firstErrorStat ),
                  ( "average (/s)", ) ),
                  ( self.getHeader( "minute", firstErrorStat ),
                  ( "average (/s)", ) ),
                  ( self.getHeader( "hour", firstErrorStat ),
                  ( "average (/s)", ) ),
                  ( self.getHeader( "day", firstErrorStat ),
                  ( "average (/s)", ) ),
                  ( self.getHeader( "week", firstErrorStat ),
                  ( "average (/s)", ) ) )
      table = TableOutput.createTable( header )
      l = TableOutput.Format( justify="left" )
      l.noPadLeftIs( True )
      l.padLimitIs( True )
      r = TableOutput.Format( justify="right" ) 
      r.noPadLeftIs( True )
      r.padLimitIs( True )
      table.formatColumns( l, l, l, r, r, r, r, r, r )
      for pciId, pciRatesError in sorted( self.devices.items() ):
         for errorType, errorStat in pciRatesError.errorStats.items():
            table.newRow( pciRatesError.name,
                          pciId,
                          prettyErrorTypes[ errorType ],
                          self.getFmtRate( "second", errorStat ),
                          self.getFmtRate( "minute", errorStat ),
                          self.getFmtRate( "hour", errorStat ),
                          self.getFmtRate( "day", errorStat ),
                          self.getFmtRate( "week", errorStat ) )
      print( table.output() )

def getErrorsStats( errorTypes, errorsObj ):
   ret = {}
   for errorType in errorTypes:
      if errorType in errorsObj.keys():
         statter = errorsObj[ errorType ].statter
         statsReturn = statter.getStats()
         masker = errorsObj[ errorType ].masker
         value = PciErrorStats(
            units=getModelObjFromTacObj( statsReturn.units, PciErrorStatsUnits ),
            masked=masker.masked if masker else False
         )
         ret.update( { errorType : value } )
   return ret

def pciErrorsRatesCmdHandler( mode, args ):
   updateErrorRequestTimes()
   devices = {}
   for pciId, pciDevice in sorted( getPciDevices().items() ):
      errorStats = getErrorsStats( prettyErrorTypes.keys(), pciDevice.errors )
      if errorStats:
         devices[ pciId ] = PciErrorsStats(
                           name=pciDevice.name,
                           errorStats=errorStats )
   return PciErrorsStatsAll( devices=devices )

#--------------------------------------------------------------------------------
# show pci errors rates detail
#--------------------------------------------------------------------------------
prettyErrorTypesDetail = {
   "CorrectableError": "Correctable Error",
   "ReplayTimerTimeout": "Replay Timer Time-out",
   "ReplayNumRollover": "Replay Number Rollover",
   "BadDllp": "Bad DLLP",
   "BadTlp": "Bad TLP",
   "ReceiverError": "Receiver Error",
   "AdvisoryNonFatal": "Advisory Non-Fatal",
   "CorrectedInternalError": "Corrected Internal Error",
   "CorrHdrLogOverflow": "Header Log Overflow",

   "UncorrectableNonFatalError": "Uncorrectable Non-Fatal Error",
   "PoisonedTlp": "Poisoned TLP Received",
   "EcrcError": "ECRC Check Failed",
   "UnsupportedRequestError": "Unsupported Request",
   "CompletionTimeout": "Completion Time-out",
   "CompleterAbort": "Completion Abort",
   "UnexpectedCompletion": "Unexpected Completion",
   "AcsViolError": "ACS Violation",
   "SurpriseDown": "Surprise Down Error",
   "AtomicOpEgressBlocked": "AtomicOp Egress Blocked",
   "UncorrInternalError": "Uncorrectable Internal Error",
   "McBlockedTlp": "MC Blocked TLP",
   "TlpPrefixBlocked": "TLP Prefix Blocked Error",
   "PoisonedTlpEgressBlocked": "Poisoned TLP Egress Blocked",

   "UncorrectableFatalError": "Uncorrectable Fatal Error",
   "UndefinedError": "Training Error",
   "DataLinkProtocolError": "DLL Protocol Error",
   "ReceiverOverflow": "Receiver Overflow",
   "FlowControlProtocolError": "Flow Control Protocol Error",
   "MalformedTlp": "Malformed Tlp"
}

class PciErrorStatsDetail( Model ):
   units = Submodel( valueType=PciErrorStatsUnits,
      help="List containing details of all units (from second to week)" )
   summary = Str(
      help="Name of the unit that was dynamically selected as human-readable" )
   maximumRatePerSecond = Float(
      help="Maximum error hits per second recorded since boot / clear pci" )
   maximumRateTimestamp = Float(
      help="UNIX timestamp when the maximumRatePerSecond occurred" )
   masked = Bool(
      help="Denotes whether this error type is currently being masked" )
   maskedEver = Bool(
      help="Denotes whether this error type has ever been masked" )

class PciErrorsStatsDetail( Model ):
   name = Str( help="PCI device name" )
   errorStats = Dict( valueType=PciErrorStatsDetail, keyType=str,
      help="Error stats (a mapping of error types to their stats)" )

class PciErrorsStatsDetailAll( Model ):
   devices = Dict( valueType=PciErrorsStatsDetail, keyType=str,
      help="Devices with recorded errors (a mapping of pciId to their stats)" )

   def getFmtRate( self, unitName, errorStat ):
      unit = getattr( errorStat.units, unitName )
      rateString = f"{unit.rate:.2f}" if not errorStat.masked else "high"
      return f"({unit.averagingPeriod // unit.unitSeconds}-{unitName} average): " + \
             f"{rateString} / {unitName}"

   def getFmtSummary( self, unitName, errorStat ):
      if errorStat.masked:
         return "high"
      unit = getattr( errorStat.units, unitName )
      rateString = f"{unit.rate:.2f}"
      return f"{rateString} / {unitName} " + \
             f"({unit.averagingPeriod // unit.unitSeconds}-{unitName} average)"

   def render( self ):
      for pciId, pciRatesError in sorted( self.devices.items() ):
         for errorType, errorStat in pciRatesError.errorStats.items():
            prettyErrorType = prettyErrorTypesDetail[ errorType ]
            prefix = f"{pciRatesError.name} ({pciId}) {prettyErrorType}"
            print( f"{prefix} Rate {self.getFmtRate( 'second', errorStat )}" )
            print( f"{prefix} Rate {self.getFmtRate( 'minute', errorStat )}" )
            print( f"{prefix} Rate {self.getFmtRate( 'hour', errorStat )}" )
            print( f"{prefix} Rate {self.getFmtRate( 'day', errorStat )}" )
            print( f"{prefix} Rate {self.getFmtRate( 'week', errorStat )}" )
            print( f"{prefix} Summary: " + 
                   f"{self.getFmtSummary( errorStat.summary, errorStat )}" )
            maximumRatePerSecondStr = f"{prefix} Maximum Rate Per Second: " + \
                                      f"{errorStat.maximumRatePerSecond:.2f}"
            if errorStat.maximumRatePerSecond and errorStat.maximumRateTimestamp:
               maximumRateTimestampStr = time.strftime(
                  '%Y-%m-%d %H:%M:%S',
                  time.localtime( errorStat.maximumRateTimestamp ) )
               maximumRatePerSecondStr += f" ({maximumRateTimestampStr})"
            print( maximumRatePerSecondStr )
            if errorStat.masked:
               print( f"{prefix} Reporting Status: " + 
                       "Reporting for this error is currently turned off" )
            elif errorStat.maskedEver:
               print( f"{prefix} Reporting Status: " + 
                       "Reporting for this error has previously been turned off" )

def getErrorsStatsDetail( errorTypes, errorsObj ):
   ret = {}
   for errorType in errorTypes:
      if errorType in errorsObj.keys():
         statter = errorsObj[ errorType ].statter
         statsReturn = statter.getStats()
         masker = errorsObj[ errorType ].masker
         value = PciErrorStatsDetail(
            units=getModelObjFromTacObj( statsReturn.units, PciErrorStatsUnits ),
            summary=statsReturn.summary,
            maximumRatePerSecond=statsReturn.maximumRatePerSecond,
            maximumRateTimestamp=statsReturn.maximumRateTimestamp,
            masked=masker.masked if masker else False,
            maskedEver=masker.maskedEver if masker else False
         )
         ret.update( { errorType : value } )
   return ret

def pciErrorsRatesDetailCmdHandler( mode, args ):
   updateErrorRequestTimes()
   devices = {}
   for pciId, pciDevice in sorted( getPciDevices().items() ):
      errorStats = getErrorsStatsDetail( prettyErrorTypesDetail.keys(),
                                         pciDevice.errors )
      if errorStats:
         devices[ pciId ] = PciErrorsStatsDetail(
                           name=pciDevice.name,
                           errorStats=errorStats )
   return PciErrorsStatsDetailAll( devices=devices )

#--------------------------------------------------------------------------------
# show pci tree
#--------------------------------------------------------------------------------
class PciTreeNode( Model ):
   def renderRecurse( self, name, displayStr ):
      if name:
         displayStr += name
      if not self.devices:
         print( displayStr + '\n' )
         return

      length = len( displayStr )
      if name:
         displayStr += '--'
      for childName, child in sorted( self.devices.items() ):
         child.renderRecurse( childName, displayStr )
         if length:
            displayStr = ' ' * length + '+-'

   def render( self ):
      self.renderRecurse( None, '' )

PciTreeNode.__attributes__[ "devices" ] = \
      Dict( valueType=PciTreeNode, help="PCI children of this device",
            optional=True )

def populatePciTree( domainRoot ):
   childBus = domainRoot.childBus
   node = PciTreeNode( devices={} )
   if not childBus:
      return node

   for name, dev in childBus.items():
      node.devices[ name ] = populatePciTree( dev )
   return node

def pciTreeCmdHandler( mode, args ):
   root = PciTreeNode( devices={} )

   for name in cellDevices.pciDeviceConfig.keys():
      if name.startswith( 'DomainRoot' ):
         dev = cellDevices.pciDeviceConfig.get( name )
         root.devices[ name ] = populatePciTree( dev )

   if len( root.devices ) == 0:
      mode.addError( 'Unable to find DomainRoot in Pci tree' )
      return None

   return root

                                                      
#--------------------------------------------------------------------------------
# clear pci
#--------------------------------------------------------------------------------
def waitForClearCounterResponse( requestTime ):
   try:
      Tac.waitFor( lambda: pciDeviceStatusDir.clearCounterResponse > 
                   requestTime, timeout=5.0, 
                   description="clear counter response to be updated",
                   warnAfter=None, maxDelay=0.1, sleep=True )
   except Tac.Timeout:
      print( "Warning: PCIe error counters may not have reset yet" )


def clearPciCmdHandler( mode, args ):
   if cellDevices:
      cellCliRequest.clearCounterRequest = Tac.now()
      waitForClearCounterResponse( cellCliRequest.clearCounterRequest )
   if sysDevices:
      sysCliRequest.clearCounterRequest = Tac.now()
      waitForClearCounterResponse( sysCliRequest.clearCounterRequest )

#------------------------------------------------------
# Plugin method
#------------------------------------------------------

def Plugin( em ):
   global entityManager
   global pciDeviceStatusDir
   global pcieSwitchStatusDir
   global cellDevices, sysDevices
   global cellCliRequest, sysCliRequest
   entityManager = em
   cellId = Cell.cellId()
   pciDeviceStatusDir = LazyMount.mount( em,
         f"cell/{cellId}/hardware/pciDeviceStatusDir",
         "Hardware::PciDeviceStatusDir", "r" )
   pcieSwitchStatusDir = LazyMount.mount( em,
                                         "hardware/pcieSwitch/status/system",
                                         "Tac::Dir", "ri" )
   cellDevices = LazyMount.mount( em,
         f"hardware/cell/{cellId}/pciDeviceMap/config",
         "Hardware::PciDeviceConfigDir", "r" )
   sysDevices = LazyMount.mount( em, "hardware/pciDeviceMap/config",
                                 "Hardware::PciDeviceConfigDir", "r" )
   cellCliRequest = LazyMount.mount(
      em, f"cell/{cellId}/hardware/pciDeviceMap/cliRequest",
      "Hardware::PciDeviceCliRequest",
      "w" )
   sysCliRequest = LazyMount.mount(
      em, "hardware/pciDeviceMap/cliRequest",
      "Hardware::PciDeviceCliRequest", "w" )
