# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import PhyConfigCli
import Tac

def rollOffCmdHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig
   cliConfig = cliCoherentDir.get( mode.intf.name )
   if cliConfig is None:
      cliConfig = cliCoherentDir.newMember( mode.intf.name )
   # pylint: disable=round-builtin # pylint: disable=bad-option-value
   cliConfig.rollOff = round( args[ 'ROLL_OFF' ], PhyConfigCli.rollOffPrecision )
   cliConfig.rollOffConfigured = True

def rollOffCmdNoHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig
   if mode.intf.name in cliCoherentDir:
      cliCoherentDir[ mode.intf.name ].rollOffConfigured = False

def sopFastTrackingCmdHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig
   cliConfig = cliCoherentDir.get( mode.intf.name )
   if cliConfig is None:
      cliConfig = cliCoherentDir.newMember( mode.intf.name )
   cliConfig.sopFastTracking = True

def sopFastTrackingCmdNoHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig
   if mode.intf.name in cliCoherentDir:
      cliCoherentDir[ mode.intf.name ].sopFastTracking = False

# Helper function to return an enum value based on CLI input
def modulationToEnum( modulation ):
   modulationEnum = Tac.Type( "Phy::Coherent::Modulation" )
   return { '16qam': modulationEnum.modulation16Qam,
            '8qam': modulationEnum.modulation8Qam,
            'dp-qpsk': modulationEnum.modulationDpQpsk }.get( modulation,
                  modulationEnum.modulationDefault )

# Helper function to return an enum value based on CLI input
def lineRateToEnum( rate ):
   lineRateEnum = Tac.Type( "Phy::Coherent::LineRate" )
   return { '100g': lineRateEnum.lineRate100g,
            '200g': lineRateEnum.lineRate200g,
            '300g': lineRateEnum.lineRate300g,
            '400g': lineRateEnum.lineRate400g }.get( rate,
                                                      lineRateEnum.lineRateDefault )

def modulationCmdHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig
   cliConfig = cliCoherentDir.get( mode.intf.name )
   if cliConfig is None:
      cliConfig = cliCoherentDir.newMember( mode.intf.name )
   cliConfig.modulation = modulationToEnum( args[ 'MODE' ] )
   if 'RATE' in args:
      cliConfig.lineRate = lineRateToEnum( args[ 'RATE' ] )
   else:
      cliConfig.lineRate = lineRateToEnum( 'default' )

def modulationCmdNoHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig
   cliConfig = cliCoherentDir.get( mode.intf.name )
   if cliConfig:
      cliConfig.modulation = modulationToEnum( 'default' )
      cliConfig.lineRate = lineRateToEnum( 'default' )

def skewCmdHandler( mode, args ):
   cliCoherentDir = PhyConfigCli.phyCliConfigDir.phyCliCoherentConfig

   def dictToEnum( args, enumDict ):
      curKey = ''.join( str( args.get( key, '' ) ) for key in enumDict )
      return enumDict[ curKey ]

   directionEnum = Tac.Type( 'Phy::Coherent::Direction' )
   directionDict = { 'transmitter': directionEnum.directionTransmit,
                     'receiver': directionEnum.directionReceive }

   requestTypeEnum = Tac.Type( 'Phy::Coherent::RequestType' )
   requestTypeDict = { 'compute': requestTypeEnum.skewRequestCompute,
                       'activate': requestTypeEnum.skewRequestActivate,
                       'deactivate': requestTypeEnum.skewRequestDeactivate,
                       'clear': requestTypeEnum.skewRequestClear }

   txIntf = args.get( 'TX_INTF' )
   rxIntfs = args.get( 'RX_INTFS' )
   peerIntfName = getattr( args.get( 'PEER_INTF' ), 'name', '' )
   ethIntfId = Tac.Value( 'Arnet::EthIntfId' )

   direction = dictToEnum( args, directionDict )
   requestType = dictToEnum( args, requestTypeDict )

   # Ensure that calibration is conducted on the same linecard
   if ( txIntf and
        requestType == requestTypeEnum.skewRequestCompute and
        ethIntfId.module( txIntf.name ) != ethIntfId.module( peerIntfName ) ):
      mode.addError( "Computation requires both interfaces on the same linecard" )
      return

   if txIntf:
      intfList = [ txIntf.name ]
   else:
      intfList = list( rxIntfs )

   for intf in intfList:
      # An intf range will automatically add interfaces with lanes other than
      # /1 into the intf list. We want to skip these. Also, if the user entered
      # a single interface with an incorrect lane, inform them this is invalid
      # so they know the command did nothing.
      if ethIntfId.lane( intf ) != 1 or ( peerIntfName and
            ethIntfId.lane( peerIntfName ) != 1 ):
         err = "Unable to perform requested action for lane %d on%s interface %s"
         if len( intfList ) == 1 and ethIntfId.lane( intf ) != 1:
            mode.addError( err % ( ethIntfId.lane( intf ), '', intf ) )
         if peerIntfName and ethIntfId.lane( peerIntfName ) != 1:
            mode.addError( err % ( ethIntfId.lane( peerIntfName ), ' peer',
                                   peerIntfName ) )
         continue

      cliConfig = cliCoherentDir.newMember( intf )

      cliConfig.skewCalibrationRequest = Tac.Value(
            'Phy::Coherent::SkewCalibrationRequest',
            requestType, direction, peerIntfName
         )

      curGen = cliConfig.skewCalibrationGen
      cliConfig.skewCalibrationGen = Tac.Value( 'Fru::GenerationId', curGen + 1 )
