# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliGlobal
from CliPlugin import (
   EthIntfCli,
   IntfCli,
)
from CliPlugin.PlutoCli import (
   gv as plutoCliGv,
   PoePort,
   PoeLinecardMode,
   PoeMode,
   PoePorts,
   PoePortNbrInfo,
   PoePortNeighbors,
)
import LazyMount
import Logging
from PoeLogs import POE_POWER_MAINTAIN_RPR_INCOMPATIBLE
import Tac

DEFAULT_REBOOT_ACTION = 'powerOff'
DEFAULT_INTF_POE_REBOOT_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME = 5
DEFAULT_INTERFACE_SHUTDOWN_ACTION = 'maintain'
DEFAULT_INTF_POE_SHUTDOWN_ACTION = 'unset'
DEFAULT_POE_PRIORITY = 'defaultPriority'
DEFAULT_POE_POWER_LIMIT = Tac.Value( "Units::Watts" ).value
MAX_POE_POWER_LIMIT = 90

gv = CliGlobal.CliGlobal( powerManagerCliConfig=None,
                         lldpStatus=None, powerManager=None )

classMapping = {
   'class0': 15.4, # Watts
   'class1': 4.0,
   'class1D': 8.0,
   'class2': 7.0,
   'class2D': 14.0,
   'class3': 15.4,
   'class3D': 31.0,
   'class4': 30.0,
   'class4D': 60.0,
   'class5': 45.0,
   'class5D': 90.0,
   'class6': 60.0,
   'class7': 75.0,
   'class8': 90.0,
   'invalid': 0.0,
   'unknown': 0.0,
}

def doPoePortDisabled( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.poeEnabled = 'disabled' not in args

def noPoePortDisabled( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.poeEnabled = True

def doPoePortPriorityCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.portPriority = args[ 'PRIORITY' ]

def noPoePortPriorityCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.portPriority = DEFAULT_POE_PRIORITY

def doPoeLimitCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   capacity = MAX_POE_POWER_LIMIT
   if mode.intf.name in plutoCliGv.poeStatus:
      capacity = plutoCliGv.poeStatus[ mode.intf.name ].capacity
   if 'CLASS_NUM' in args:
      powerLimit = classMapping[ args[ 'CLASS_NUM' ] ]
   elif 'NUM_WATTS' in args:
      powerLimit = args[ 'NUM_WATTS' ]
   if 'fixed' in args:
      message = ( "The power limit may exceed what is supported by the device "
                  "and could cause damage to the device." )
      mode.addWarning( message )
      config.fixedLimit = True
   else:
      config.fixedLimit = False
   if powerLimit > capacity:
      message = f"Power limit exceeds capacity of port {mode.intf.name}"
      mode.addWarning( message )
   else:
      config.powerLimit = powerLimit

def noPoeLimitCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.powerLimit = DEFAULT_POE_POWER_LIMIT

def doPoePortLldpDisabled( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.lldpEnabled = 'disabled' not in args

def noPoePortLldpDisabled( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.lldpEnabled = True

def doPoeLegacyDetectCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.legacyDetect = True

def noPoeLegacyDetectCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.legacyDetect = False

def doPoePairsetCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   if '4-pair' in args:
      config.portPairset = 'fourPair'
   elif 'a' in args:
      config.portPairset = 'altA'
   else:
      config.portPairset = 'altB'

def noPoePairsetCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.portPairset = 'fourPair'

def populateEachPoePort( intf ):
   portName = intf
   portPresent = False
   portCapacity = None
   pseEnabled = None
   portPairset = None
   portPriority = None
   lldpEnabled = None
   legacyDetectEnabled = None
   powerLimit = None
   grantedPower = None
   portState = None
   pdClass = None
   power = None
   current = None
   voltage = None
   temperature = None
   misconfigured = False

   if intf in plutoCliGv.poeStatus and plutoCliGv.poeStatus[ intf ]:
      status = plutoCliGv.poeStatus[ intf ]
      config = plutoCliGv.poeCliConfig.newPoeCliConfig( intf )
      portPresent = True
      portCapacity = status.capacity
      pdClass = status.portStatus.pdClass
      if pdClass == "classUnknown":
         pdClass = "unknown"
      if ( status.portStatus.psePoweringStatus == "FourPairPoweringDualSignaturePse"
           and pdClass.startswith( 'class' ) ):
         pdClass = pdClass + 'D'
      pseEnabled = config.poeEnabled
      if status.capacity <= 30.0:
         portPairset = "altA"
      else:
         portPairset = status.portPairset
         if portPairset != config.portPairset:
            misconfigured = True
      portPriority = status.portPriority
      if portPriority == "defaultPriority":
         portPriority = "low"
      lldpEnabled = config.lldpEnabled
      legacyDetectEnabled = config.legacyDetect
      if config.powerLimit == DEFAULT_POE_POWER_LIMIT:
         if pdClass in classMapping:
            powerLimit = min( classMapping[ pdClass ], portCapacity )
         else:
            pdClass = "invalid"
            powerLimit = 0.0
      else:
         powerLimit = config.powerLimit
      portState = status.portStatus.poePortState
      grantedPower = status.pseApprovedPower
      if ( ( plutoCliGv.poeCliConfig.telemetryBasedPoeEnabled
            and portState != "powered" )
           or grantedPower < 0 ):
         grantedPower = 0.0
      power = status.outputPower
      current = status.outputCurrent * 1000
      voltage = status.outputVoltage
      temperature = status.temperature

      primary = None
      alternative = None
      for channelId, cs in status.channelStatus.items():
         channel = PoePort.PoeChannel( pairSet="2-pair",
                                       portState=cs.poePortState,
                                       pdClass=cs.pdClass,
                                       pseLimit=cs.pseLimit,
                                       power=cs.outputPower,
                                       current=cs.outputCurrent * 1000,
                                       voltage=cs.outputVoltage )
         # With PoE port mapping, primary channel has odd channel ID,
         # alternative channel has even channel ID
         if channelId % 2 == 1:
            primary = channel
            if len( status.channelStatus ) == 2:
               channel.pairSet = "Alt A"
         else:
            alternative = channel
            if len( status.channelStatus ) == 2:
               channel.pairSet = "Alt B"

   return PoePort( portName=portName, portPresent=portPresent,
                   portCapacity=portCapacity, pseEnabled=pseEnabled,
                   portPairset=portPairset, portPriority=portPriority,
                   lldpEnabled=lldpEnabled, legacyDetectEnabled=legacyDetectEnabled,
                   powerLimit=powerLimit, grantedPower=grantedPower,
                   portState=portState, pdClass=pdClass, power=power,
                   current=current, voltage=voltage, temperature=temperature,
                   misconfigured=misconfigured, primary=primary,
                   alternative=alternative )

def populatePoePortNbr( poePort ):
   portName = poePort.portName
   portState = poePort.portState
   power = poePort.power
   lldpSystemDesc = []

   if portName in plutoCliGv.poeStatus and plutoCliGv.poeStatus[ portName ]:
      lldpPortStatus = gv.lldpStatus.portStatus.get( portName )
      if lldpPortStatus and lldpPortStatus.remoteSystem:
         for neighbor in lldpPortStatus.remoteSystem.values():
            if neighbor.sysDesc:
               lldpSystemDesc.append( neighbor.sysDesc.value.decode() )

   return PoePortNbrInfo( portState=portState, power=power,
                          lldpSystemDesc=lldpSystemDesc )

def doShowPoe( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detail = 'detail' in args.keys()
   poeIntfNames = [ poeIntf.name for poeIntf in plutoCliGv.poeStatus.values() ]
   if intf or mod:
      intfs = IntfCli.Intf.getAll( mode, intf, mod, intfType=EthIntfCli.EthIntf )
      intfNames = [ i.name for i in intfs if i.name in poeIntfNames ]
   else:
      intfNames = poeIntfNames
   poePorts = { name: populateEachPoePort( name ) for name in intfNames }
   return PoePorts( poePorts=poePorts, detail=detail )

def doShowPoeNeighbor( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   poeIntfNames = [ poeIntf.name for poeIntf in plutoCliGv.poeStatus.values() ]
   if intf or mod:
      intfs = IntfCli.Intf.getAll( mode, intf, mod, intfType=EthIntfCli.EthIntf )
      intfNames = [ i.name for i in intfs if i.name in poeIntfNames ]
   else:
      intfNames = poeIntfNames
   poePorts = { name: populateEachPoePort( name ) for name in intfNames }
   poePortsNbr = { p.portName: populatePoePortNbr( p ) for p in
                   poePorts.values() }
   allocatedPower = float( 0 )
   for consumer in gv.powerManager.powerConsumers:
      device = gv.powerManager.powerConsumers[ consumer ]
      allocatedPower += device.amount
   usedPower = allocatedPower
   if plutoCliGv.poeCliConfig.telemetryBasedPoeEnabled:
      usedPower = sum( poeIntf.outputPower for poeIntf
                      in plutoCliGv.poeStatus.values() )
   budget = min( gv.powerManager.totalPower, gv.powerManagerCliConfig.powerBudget )
   return PoePortNeighbors( poePorts=poePortsNbr,
                            powerBudget=budget,
                            usedPoePower=usedPower,
                            remainingPower=budget - usedPower )

def doPoe( mode, args ):
   childMode = mode.childMode( PoeMode )
   mode.session_.gotoChildMode( childMode )

def noPoe( mode, args ):
   config = plutoCliGv.poeCliConfig
   config.linecardPortPriority.clear()
   config.rebootAction = DEFAULT_REBOOT_ACTION
   config.telemetryBasedPoeEnabled = False

def doRebootActionCmd( mode, args ):
   if 'maintain' in args:
      plutoCliGv.poeCliConfig.rebootAction = 'maintain'
      redSupProtocol = plutoCliGv.sysdbRedSupConfig.protocol
      if plutoCliGv.hwEntMib.chassis and redSupProtocol == 'rpr':
         Logging.log( POE_POWER_MAINTAIN_RPR_INCOMPATIBLE )
   elif 'power-off' in args:
      plutoCliGv.poeCliConfig.rebootAction = 'powerOff'

def noRebootActionCmd( mode, args ):
   plutoCliGv.poeCliConfig.rebootAction = DEFAULT_REBOOT_ACTION

def doPoeRebootActionConfigIfCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   if 'maintain' in args:
      config.rebootAction = 'maintain'
      redSupProtocol = plutoCliGv.sysdbRedSupConfig.protocol
      if plutoCliGv.hwEntMib.chassis and redSupProtocol == 'rpr':
         Logging.log( POE_POWER_MAINTAIN_RPR_INCOMPATIBLE )
   elif 'power-off' in args:
      config.rebootAction = 'powerOff'

def noPoeRebootActionConfigIfCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.rebootAction = DEFAULT_INTF_POE_REBOOT_ACTION

def doInterfaceShutdownActionCmd( mode, args ):
   if 'maintain' in args:
      plutoCliGv.poeCliConfig.intfShutdownAction = 'maintain'
   elif 'power-off' in args:
      plutoCliGv.poeCliConfig.intfShutdownAction = 'powerOff'

def noInterfaceShutdownActionCmd( mode, args ):
   plutoCliGv.poeCliConfig.intfShutdownAction = DEFAULT_INTERFACE_SHUTDOWN_ACTION

def doPoeShutdownActionConfigIfCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   if 'maintain' in args:
      config.intfShutdownAction = 'maintain'
   elif 'power-off' in args:
      config.intfShutdownAction = 'powerOff'

def noPoeShutdownActionConfigIfCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.intfShutdownAction = DEFAULT_INTF_POE_SHUTDOWN_ACTION

def doPoeLinkDownActionConfigIfCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   if 'maintain' in args:
      config.linkDownAction = 'maintain'
   elif 'power-off' in args:
      config.linkDownAction = 'powerOff'
      config.linkDownPoeOffTime = args.get(
                                     'NUM_SECONDS',
                                     DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME )

def noPoeLinkDownActionConfigIfCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newPoeCliConfig( mode.intf.name )
   config.linkDownAction = DEFAULT_INTF_POE_LINK_DOWN_ACTION
   config.linkDownPoeOffTime = DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME

def doPowerBudgetAllocationCmd( mode, args ):
   plutoCliGv.poeCliConfig.telemetryBasedPoeEnabled = 'usage-based' in args

def noPowerBudgetAllocationCmd( mode, args ):
   plutoCliGv.poeCliConfig.telemetryBasedPoeEnabled = False

def doEnterModuleLinecardCmd( mode, args ):
   childMode = mode.childMode( PoeLinecardMode, param=args[ 'MODNAME' ].label )
   mode.session_.gotoChildMode( childMode )
   plutoCliGv.poeCliConfig.newLinecardPortPriority( str( args[ 'MODNAME' ].label ) )

def noEnterModuleLinecardCmd( mode, args ):
   del plutoCliGv.poeCliConfig.linecardPortPriority[ str( args[ 'MODNAME' ].label ) ]

def doPoeLinecardPriorityCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newLinecardPortPriority( str( mode.number ) )
   config.linecardPortPriority = args[ 'PRIORITY' ]

def noPoeLinecardPriorityCmd( mode, args ):
   config = plutoCliGv.poeCliConfig.newLinecardPortPriority( str( mode.number ) )
   config.linecardPortPriority = DEFAULT_POE_PRIORITY

def Plugin( em ):
   gv.powerManagerCliConfig = LazyMount.mount( em,
                           'environment/archer/power/config/powerManagerCliConfig',
                           'PowerManager::PowerManagerCliConfig', 'r' )
   gv.lldpStatus = LazyMount.mount( em,
         'l2discovery/lldp/status/all', 'Lldp::AllStatus', 'r' )
   gv.powerManager = LazyMount.mount( em,
                                   'environment/archer/power/status/powerManager',
                                   'PowerManager::PowerManagerStatus', 'r' )
