# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliGlobal
import Tac
import LazyMount
import Cell
from CliPlugin.PlutoPlatformModuleCli import (
   PlatformModuleStatus,
   PlatformModules,
   Slot,
)

gv = CliGlobal.CliGlobal( cardSlotStatusMount=None,
   entityMib=None,
   powerSlotStatusMount=None,
   fanSlotStatusMount=None,
   platformSlotStatusMount=None )

# -------------------------------------------------------------------------------
# command to power cycle a module
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> power-cycle [ enable checks ]
# -------------------------------------------------------------------------------
def doPowercycleModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   slot.powerCycleEnableChecks = 'checks' in args
   if slot.validNameNotEmpty( mode ):
      if slot.diagsMode:
         mode.addError( f'"{slot.name}" '
                       'cannot be power cycled, already in diags mode' )
         return
      if slot.slotConfig and gv.entityMib.root:
         slotId = slot.slotConfig.slotId
         cardSlot = gv.entityMib.root.cardSlot.get( slotId, None )
         if cardSlot:
            card = cardSlot.card
            if card and card.swappability == "notSwappable":
               mode.addError( f'"{slot.name}" is not hot swappable' )
               return
      try:
         slot.powerCycle()
      except NotImplementedError:
         mode.addError( f'"{slot.name}" not available for power cycle' )

# -------------------------------------------------------------------------------
# command to remove a module
#    simulates physical removal by power cycling and
#        immediately placing in to diags mode ( to prevent rediscovery )
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> remove [ enable checks ]
# -------------------------------------------------------------------------------
def doRemoveModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   slot.powerCycleEnableChecks = 'checks' in args
   if slot.validNameNotEmpty( mode ):
      if slot.diagsMode:
         mode.addError( f'"{slot.name}" cannot be removed, already in diags mode' )
         return
      try:
         slot.inserted = False
      except NotImplementedError:
         mode.addError( f'"{slot.name}" not available for removal' )
      except Tac.Timeout:
         mode.addError( f'Timed out waiting for "{slot.name}" to be removed' )

# -------------------------------------------------------------------------------
# command to place module in to diags mode
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> diags
# -------------------------------------------------------------------------------
def doDiagModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validName( mode ):
      slot.diagsMode = True

# -------------------------------------------------------------------------------
# command to insert a module, take out of diags
#    does not matter if the module reached diags mode
#        being removed or diagsed ( #stillaword )
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> insert
# -------------------------------------------------------------------------------
def doInsertModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validNameNotEmpty( mode ):
      try:
         slot.inserted = True
      except Tac.Timeout:
         mode.addError( f'Timed out waiting for "{slot.name}" to be inserted' )

def doShowPlatformModule( mode, args ):
   modules = {}
   slotName = args.get( "SLOTNAME" )
   if slotName is None or slotName == "all":
      mounts = [ gv.cardSlotStatusMount, gv.fanSlotStatusMount,
                gv.powerSlotStatusMount, gv.platformSlotStatusMount ]
      slotGroups = [ list( m.slotStatus ) for m in mounts if m is not None ]
      slotKeys = [ key for group in slotGroups for key in group ]
      for slotKey in slotKeys:
         if slotKey in gv.powerSlotStatusMount.slotStatus:
            slotKey = f"PowerSupply{slotKey}"
         slot = Slot( slotKey )
         modules[ slot.name ] = PlatformModuleStatus( status=slot.statusString )
   else:
      slot = Slot( slotName )
      if slot.validName( mode ):
         modules[ slot.name ] = PlatformModuleStatus( status=slot.statusString )

   return PlatformModules( modules=modules )

def Plugin( entityManager ):
   gv.powerSlotStatusMount = LazyMount.mount(
      entityManager,
      "hardware/powerSupply/slot/status",
      "Hardware::PowerSupplySlot::Status", "r" )
   gv.fanSlotStatusMount = LazyMount.mount(
      entityManager,
      "hardware/fan/status",
      "Hardware::Fan::Status", "r" )
   if Cell.cellType() == "supervisor":
      gv.cardSlotStatusMount = LazyMount.mount(
         entityManager,
         "hardware/modularSystem/status",
         "Hardware::ModularSystem::Status", "r" )
      gv.entityMib = LazyMount.mount(
         entityManager,
         "hardware/entmib",
         "EntityMib::Status", "r" )
   elif Cell.cellType() == "fixed":
      gv.platformSlotStatusMount = LazyMount.mount(
         entityManager,
         "hardware/platform/slot/status",
         "Hardware::PLSlot::Status", "r" )
