# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel

class ProbeResponderCodeModel( CliModel.Model ):
   description = CliModel.Str( help='Description of code' )
   count = CliModel.Int( help='Number of packets with this code' )

class ProbeResponderModel( CliModel.Model ):
   # Reflection of configuration
   enabled = CliModel.Bool( help='Enabled' )
   allowIfname = CliModel.Bool( help='Allow query by interface name' )
   allowIfindex = CliModel.Bool( help='Allow query by interface index' )
   allowAddr = CliModel.Bool( help='Allow query by interface address' )
   allowRemote = CliModel.Bool( help='Allow query by remote node address' )
   # Status
   requestsReceived = CliModel.Int( help='PROBE requests received' )
   queueTooLong = CliModel.Int( help='PROBE requests dropped due to size of queue' )
   responsesSent = CliModel.Int( help='PROBE responses sent' )
   pktTooShort = CliModel.Int( help='PROBE requests received but too short' )
   incorrectIcmpChecksum = CliModel.Int( help='PROBE requests received with an '
                                              'incorrect ICMP checksum' )
   badExtensionHeader = CliModel.Int( help='PROBE requests received with a '
                                           'malformed ICMP extension header' )
   incorrectExtensionChecksum = CliModel.Int( help='PROBE requests received with '
                                             'an incorrect ICMP extension checksum' )
   badExtensionObject = CliModel.Int( help='PROBE requests received with a bad '
                                           'ICMP extension object' )
   requestTypeNotAllowed = CliModel.Int( help='PROBE requests received with a '
                                              'request type that is not allowed' )
   interfaceLookupFailed = CliModel.Int( help='PROBE requests for which we could '
                                              'not determine the interface' )
   wrongVrfRequested = CliModel.Int( help='PROBE requests which requested an '
                                          'interface in a different VRF' )
   remoteRequestMalformed = CliModel.Int( help='PROBE requests for a remote node '
                                         'but specify an interface name or index' )
   txErrors = CliModel.Int( help='PROBE responses that could not be sent due '
                                 'to a socket error' )
   lastTxError = CliModel.Str( help='The most recent socket error when sending '
                                    'a response' )
   # Histogram
   codes = CliModel.Dict( keyType=int, valueType=ProbeResponderCodeModel,
         help='Counts of response packets per error or non-error code' )

   def render( self ):
      if self.enabled:
         print( "Probe responses are enabled" )
      else:
         print( "Probe responses are disabled" )
         return
      queryTypes = []
      if self.allowIfname:
         queryTypes.append( 'ifname' )
      if self.allowIfindex:
         queryTypes.append( 'ifindex' )
      if self.allowAddr:
         queryTypes.append( 'addr' )
      if self.allowRemote:
         queryTypes.append( 'remote' )
      print( f"Query types allowed: "
             f"{', '.join( queryTypes ) if queryTypes else 'None'}" )
      print()
      # Print all the int fields
      for ( attr, definition ) in self.__attributes__.items():
         if isinstance( definition, CliModel.Int ):
            print( f"{definition.help.rstrip( '.' )}: {getattr( self, attr )}" )
      print( "The most recent socket error when sending a response: "
             f"{self.lastTxError}" )
      if self.codes:
         print()
         print( "Response types sent:" )
         for code in sorted( self.codes ):
            print( f"{self.codes[ code ].description}: {self.codes[ code ].count}" )

class ProbeResponderVrfModel( CliModel.Model ):
   vrfs = CliModel.Dict( keyType=str, valueType=ProbeResponderModel,
                help="Probe responder information for each VRF" )

   def render( self ):
      for vrf, v in self.vrfs.items():
         print( f"VRF {vrf}:" )
         v.render()
