# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import Arnet
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliGlobal
from CliPlugin.VlanCli import Vlan
from CliPlugin.VlanCli import getSwitchIntfConfigEvenIfDisabled
import ConfigMount
import LazyMount
from Vlan import vlanSetToCanonicalString

gv = CliGlobal.CliGlobal( bridgingConfig=None,
                          cliConfig=None,
                          vlanIntfConfigDir=None )

PvlanCliModel = CliDynamicPlugin( "PvlanModel" )

# "[no] private-vlan { isolated|community } primary vlan" handlers
def setPrivateVlan( mode, args ):
   vlanType = args[ 'PVLAN_TYPE' ]
   vlanId = args[ 'VLAN_ID' ]
   mode.vlan.setPrivateVlan( mode, vlanType, vlanId )

def noPrivateVlan( mode, args ):
   mode.vlan.noPrivateVlan( mode )

# "show vlan private-vlan" command handlers
def doShowPrivateVlan( mode, args ):
   vlans = Vlan.getAll( mode )
   ret = PvlanCliModel.VlanPrivateVlan()

   for v in vlans:
      ports = []
      vc = gv.bridgingConfig.vlanConfig.get( v.id )
      primaryVlan = vc.primaryVlan
      if vc and primaryVlan:
         # Get active ports from primary/secondary vlans
         ports = v.activePorts( mode, promoted=False )
         ports += Vlan( primaryVlan ).activePorts( mode, promoted=False )

         interfaces = set()
         for port in ports:
            interfaces.add( port.name )

         interfaces = Arnet.sortIntf( interfaces )

         # Build privateVlans dictionary with primary->secondary vlan lists
         # For example the output will be as below:
         #    3 - (4,isolated,'Et1,Et2'), (5,community,'Et2,Et3')
         #    6 - (7,isolated,'Et4,Et5')

         privateVlan = PvlanCliModel.VlanPrivateVlan.PrivateVlans.PrivateVlan()
         privateVlan.secondaryVlanId = v.id
         privateVlan.vlanType = vc.vlanType
         privateVlan.interfaces = interfaces

         if primaryVlan not in ret.privateVlans:
            ret.privateVlans[ primaryVlan ] = \
               PvlanCliModel.VlanPrivateVlan.PrivateVlans()
         ret.privateVlans[ primaryVlan ].privateVlan.append( privateVlan )

   return ret

# "show interfaces private-vlan mapping" command handlers
def doShowPrivateVlanMapping( mode, args ):
   def convertToPrintForm( mappedVlans ):
      if not mappedVlans:
         return 'NONE'
      elif mappedVlans == '1-4094':
         return 'ALL'
      else:
         return mappedVlans

   print( 'Interface    Secondary Vlans' )
   print( '---------    ---------------' )
   vlans = []
   for name in gv.vlanIntfConfigDir.intfConfig:
      m = re.match( r'Vlan(\d+)$', name )
      vlans.append( int( m.group( 1 ) ) )

   vlans.sort()
   for vlanId in vlans:
      if gv.cliConfig.sviMapping.get( vlanId ):
         mappedPrivateVlans = vlanSetToCanonicalString(
            gv.cliConfig.sviMapping[ vlanId ].secondaryVlan )
      else:
         mappedPrivateVlans = 'ALL'
      print( f'Vlan{str(vlanId):9s}    '
             f'{convertToPrintForm( mappedPrivateVlans ):15s}' )

# -------------------------------------------------------------------------------
# The "[no] switchport pvlan mapping" command, in "config-if" mode.
#
# legacy:
# The "[no] switchport private-vlan mapping" command, in "config-if" mode.
#
#
# The full syntax of this command is:
#
#   switchport pvlan mapping <vlanSet>
#   switchport pvlan mapping add <vlanSet>
#   switchport pvlan mapping remove <vlanSet>
#   {no|default} pvlan mapping [ [add|remove] <vlanSet> ]
# -------------------------------------------------------------------------------
def setPrivateVlanMapping( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   vlanSet = args[ 'VLAN_SET' ]
   vlanOp = args.get( 'OP' )
   if not vlanOp:
      vlanOp = 'set'
   if vlanOp == 'set':
      sic.privateVlanMapping.clear()
      for i in vlanSet:
         sic.privateVlanMapping[ i ] = True
   elif vlanOp == 'add':
      if sic.defaultPrivateVlanMapping:
         for i in range( 1, 4095 ):
            sic.privateVlanMapping[ i ] = True
      for i in vlanSet:
         sic.privateVlanMapping[ i ] = True
   elif vlanOp == 'remove':
      if sic.defaultPrivateVlanMapping:
         for i in range( 1, 4095 ):
            sic.privateVlanMapping[ i ] = True
      for i in vlanSet:
         del sic.privateVlanMapping[ i ]
   sic.defaultPrivateVlanMapping = False

def noPrivateVlanMapping( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.defaultPrivateVlanMapping = True

# "[no|default] switchport trunk private-vlan secondary" command handlers
def setTrunkPrivateVlanMapping( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.trunkPrivateVlanMapping = not CliCommand.isNoOrDefaultCmd( args )

# The "[no] pvlan mapping" command handlers
def setSviPrivateVlanMapping( mode, args ):
   sviMappingState = gv.cliConfig.sviMapping.get( mode.intf.vlanId )
   vlanSet = args[ 'VLAN_SET' ]
   vlanOp = args.get( 'OP' )
   if not vlanOp:
      vlanOp = 'set'
   if not sviMappingState:
      if ( vlanOp == 'add' or
           ( vlanOp == 'set' and len( vlanSet ) == 4094 ) ):
         # nothing to do
         return

      sviMappingState = gv.cliConfig.sviMapping.newMember( mode.intf.vlanId )
      for i in range( 1, 4095 ):
         sviMappingState.secondaryVlan[ i ] = True

   if vlanOp == 'set':
      sviMappingState.secondaryVlan.clear()
      for i in vlanSet:
         sviMappingState.secondaryVlan[ i ] = True
   elif vlanOp == 'add':
      for i in vlanSet:
         sviMappingState.secondaryVlan[ i ] = True
   elif vlanOp == 'remove':
      for i in vlanSet:
         del sviMappingState.secondaryVlan[ i ]
   sviMappingState.defaultSviMapping = False

def noSviPrivateVlanMapping( mode, args ):
   sviMappingState = gv.cliConfig.sviMapping.get( mode.intf.vlanId )
   if sviMappingState:
      sviMappingState.defaultSviMapping = True
   del gv.cliConfig.sviMapping[ mode.intf.vlanId ]

def Plugin( entityManager ):
   mount = LazyMount.mount
   gv.bridgingConfig = mount( entityManager, "bridging/config",
                           "Bridging::Config", "r" )
   gv.cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli",
                                  "Bridging::Input::CliConfig", "w" )
   gv.vlanIntfConfigDir = mount( entityManager, "interface/config/eth/vlan",
                              "Interface::VlanIntfConfigDir", "r" )
