#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import Tac
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from CliPlugin.QosCliCommon import ( setIntfConfig, getIntfStatus, QosProfileMode,
                                     getConfiguredLagMembers, cliBlockingFail,
                                     invertIdToName )
from CliPlugin.QosCliBasicQos import ( pfcStatus, qosConfig, DscpRewriteMode,
                                       qosHwStatus, qosInputConfig, qosSliceHwStatus,
                                       qosStatus, CosToTcProfileMode,
                                       getCurrentGlobalDscpToTcMapping,
                                       getCurrentGlobalExpToTcMapping,
                                       getCurrentGlobalCosToTcMapping,
                                       TcToCosMode, DscpToTcMode,
                                       ExpToTcProfileMode, TcToExpMode )
from CliPlugin.QosCliModel import ( CosToTcProfileAllModel, CosToTcProfileModel,
                                    DscpRewriteMapAllModel, DscpRewriteMapModel,
                                    DscpToTcMapAllModel, DscpToTcMapModel,
                                    IntfDscpPreserveCollectionModel,
                                    IntfTrustModeCollectionModel, IntfTrustModeModel,
                                    IntfTrustModeParametersModel, TcToCosMapAllModel,
                                    TcToCosMapModel, ExpToTcProfileAllModel,
                                    ExpToTcProfileModel, TcDpToExpMapAllModel,
                                    TcDpToExpMapModel )
from CliPlugin import ( EthIntfCli, IntfCli, SubIntfCli, VlanIntfCli )
from QosLib import ( compareAttributeToDefault, isLagPort, isSubIntfId,
                     tcDpToCosSupportedDpValues, tcDpToDscpSupportedDpValues,
                     tcDpToExpSupportedDpValues )
from QosTypes import ( tacCos, tacDropPrecedence, tacDscp,
                       tacExp, tacFeatureName, tacRewriteEnableMode, tacTcDpPair,
                       tacTrafficClass, tacTrustMode, tacTcToCosMapName,
                       tacDscpToTcMapName, tacExpToTcProfileName,
                       tacCosToTcProfileName, tacDscpRewriteMapName,
                       tacTxQueueId, tacPriorityGroup, tacTcDpToExpMapName )

# --------------------------------------------------------------------------------
# Utility functions and handlers follow
# --------------------------------------------------------------------------------
def convertInvalidStrToDefault( value, invalid ):
   if value == invalid:
      return "DEFAULT"
   return f"{str( value ).upper()}"

def getOperTrustMode( intfId ):
   operTrustMode = tacTrustMode.invalid
   lagMem = None
   isLagIntf = isLagPort( intfId )
   isSubIntf = isSubIntfId( intfId )
   if isLagIntf:
      lagMem = getConfiguredLagMembers( intfId )
   # For Lag intfs with no members, pick up operTrustMode from intfConfig
   if isLagIntf and not lagMem:
      if intfId in qosConfig.intfConfig:
         operTrustMode = qosConfig.intfConfig[ intfId ].trustMode
   else:  # For phy intf/lag intf with members, pick up operTrustMode from intfStatus
      intfStatus = getIntfStatus( intfId )
      if intfStatus:
         operTrustMode = intfStatus.trustMode

   if isSubIntf and operTrustMode == tacTrustMode.invalid:
      parentIntfId = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( intfId )
      operTrustMode = getOperTrustMode( parentIntfId )
   return operTrustMode

def showTrustInterface( intf ):
   intfTrustModeModel = IntfTrustModeModel()
   operTrustMode = getOperTrustMode( intf.name )

   # Pick up configured trust mode for interface
   cfgTrustMode = tacTrustMode.invalid
   if intf.name in qosConfig.intfConfig:
      intfConfig = qosConfig.intfConfig[ intf.name ]
      cfgTrustMode = intfConfig.trustMode

   intfTrustModeModel.intfTrustModeParameters = IntfTrustModeParametersModel()
   intfTrustModeModel.intfTrustModeParameters.operationalTrustMode = \
       compareAttributeToDefault( qosHwStatus, "defaultTrustMode", operTrustMode,
                                  tacTrustMode.invalid )
   intfTrustModeModel.intfTrustModeParameters.configuredTrustMode = \
       convertInvalidStrToDefault( cfgTrustMode, tacTrustMode.invalid )
   return intfTrustModeModel

# --------------------------------------------------------------------------------
# Handler of QosInterfacesTrustCmd
# --------------------------------------------------------------------------------
def showInterfacesTrust( mode, args ):
   intf = args.get( 'INTF' )
   intfTrustModeCollection = IntfTrustModeCollectionModel()

   intfType = EthIntfCli.EthIntf
   if qosHwStatus.trustModePerSubInterfaceSupported:
      intfType = ( intfType, SubIntfCli.SubIntf )
   intfs = IntfCli.Intf.getAll( mode, intf, None, intfType=intfType )
   if not intfs:
      return intfTrustModeCollection
   for intf in intfs:
      if not intf.name.startswith( ethOrLagIntfPrefixes ):
         continue
      intfTrustMode = showTrustInterface( intf )
      intfTrustModeCollection.intfTrustModeCollection[ intf.name ] = intfTrustMode

   return intfTrustModeCollection

# --------------------------------------------------------------------------------
# Handler of QosTrustCmd
# --------------------------------------------------------------------------------
def setQosTrustMode( mode, args ):
   timestamp = Tac.now()
   trustMode = args.get( 'MODE', 'None' )
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      prevTrustMode = profile.trustMode
      if CliCommand.isDefaultCmd( args ):
         trustMode = tacTrustMode.invalid
      elif CliCommand.isNoCmd( args ):
         if trustMode in ( tacTrustMode.dscp, tacTrustMode.cos ):
            if ( ( trustMode == tacTrustMode.dscp and
                  prevTrustMode == tacTrustMode.dscp ) or
                  ( trustMode == tacTrustMode.cos and
                  prevTrustMode == tacTrustMode.cos ) ):
               trustMode = tacTrustMode.invalid
            else:
               return
         else:
            trustMode = tacTrustMode.untrusted
      if prevTrustMode == trustMode:
         return
      profile.trustMode = trustMode
   else:
      # Get the current qos config for the intf
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if CliCommand.isDefaultCmd( args ):
         trustMode = tacTrustMode.invalid
      elif CliCommand.isNoCmd( args ):
         if trustMode in ( tacTrustMode.dscp, tacTrustMode.cos ):
            if ( ( trustMode == tacTrustMode.dscp and intfConfig and
                  intfConfig.trustMode == tacTrustMode.dscp ) or
                  ( trustMode == tacTrustMode.cos and intfConfig and
                  intfConfig.trustMode == tacTrustMode.cos ) ):
               trustMode = tacTrustMode.invalid
            else:
               return
         else:
            trustMode = tacTrustMode.untrusted

      if ( trustMode == tacTrustMode.untrusted or
         ( trustMode == tacTrustMode.dscp and
           not pfcStatus.dscpBasedPfcSupported ) ):
         portStatus = None
         for sliceHwStatus in qosSliceHwStatus.values():
            portStatus = sliceHwStatus.portStatus
            if mode.intf.name in portStatus:
               if portStatus[ mode.intf.name ].active:
                  mode.addError( "Cannot change trust mode on "
                                 "priority-flow-control enabled interface" )
                  return
      if not intfConfig:
         prevTrustMode = tacTrustMode.invalid
      else:
         prevTrustMode = intfConfig.trustMode
      if prevTrustMode == trustMode:
         return

      setIntfConfig( mode.intf.name, cfgTrustMode=trustMode )
      if cliBlockingFail( mode, timestamp, tacFeatureName.trustMode, "Trust Mode",
                          mode.intf.name ):
         setIntfConfig( mode.intf.name, cfgTrustMode=prevTrustMode )

# --------------------------------------------------------------------------------
# Handler of QosMapCosToTrafficClassProfilenameCmd
# --------------------------------------------------------------------------------
def configureCosToTcProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   if profileName not in qosInputConfig.cosToTcProfile:
      qosInputConfig.cosToTcProfile.newMember( profileName )
      for cos in range( 0, 8 ):
         # Copy from global/platform default mapping
         qosInputConfig.cosToTcProfile[ profileName ].cosToTcMap[
               cos ] = getCurrentGlobalCosToTcMapping( cos )
   childMode = mode.childMode( CosToTcProfileMode, param=profileName )
   mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# No or default handler of QosMapCosToTrafficClassProfilenameCmd
# --------------------------------------------------------------------------------
def noCosToTcProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   del qosInputConfig.cosToTcProfile[ profileName ]

# --------------------------------------------------------------------------------
# Handler of QosMapExpToTrafficClassProfileNameCmd
# --------------------------------------------------------------------------------
def configureExpToTcProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   if profileName not in qosInputConfig.expToTcProfile:
      qosInputConfig.expToTcProfile.newMember( profileName )
      for exp in range( 0, 8 ):
         # Copy from global/platform default mapping
         qosInputConfig.expToTcProfile[ profileName ].expToTcMap[
               exp ] = getCurrentGlobalExpToTcMapping( exp )
   childMode = mode.childMode( ExpToTcProfileMode, param=profileName )
   mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# No or default handler of QosMapExpToTrafficClassProfileNameCmd
# --------------------------------------------------------------------------------
def noExpToTcProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   del qosInputConfig.expToTcProfile[ profileName ]

# --------------------------------------------------------------------------------
# Handler of QosMapDscpToTrafficClassMapNameCmd
# --------------------------------------------------------------------------------
def createDscpToTcMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   if mapName not in qosInputConfig.dscpToTcNamedMap:
      qosInputConfig.dscpToTcNamedMap.newMember( mapName )
      for dscp in range( tacDscp.min, tacDscp.max + 1 ):
         # Copy from global/platform default mapping
         qosInputConfig.dscpToTcNamedMap[ mapName ].dscpToTcNamedMap[
               dscp ] = getCurrentGlobalDscpToTcMapping( dscp )
   childMode = mode.childMode( DscpToTcMode, param=mapName )
   mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# No or default handler of QosMapDscpToTrafficClassMapNameCmd
# --------------------------------------------------------------------------------
def removeDscpToTcMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   del qosInputConfig.dscpToTcNamedMap[ mapName ]

# --------------------------------------------------------------------------------
# Handler of QosMapDeiToDropPrecedenceCmd
# --------------------------------------------------------------------------------
def setDeiToDp( mode, args ):
   deiList = args.get( 'DEI_VALUE' ) or list( args.get( 'DEI_VALUES' ).values() )
   dp = args.get( 'DROP_PRECEDENCE' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )

   # Get the current deiToDp config
   prevDeiToDpMap = dict( qosInputConfig.deiToDpMap )
   for dei in deiList:
      timestamp = Tac.now()
      if noOrDefaultKw or dp == qosHwStatus.defaultDeiToDpMap[ dei ]:
         deiToDpVal = tacDropPrecedence.invalid
      else:
         deiToDpVal = dp
      qosInputConfig.deiToDpMap[ dei ] = deiToDpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Dei to Dp Map" ):
         # If any of the deiToDp settings fail, revert the entire command
         for deiInner in prevDeiToDpMap:
            qosInputConfig.deiToDpMap[ deiInner ] = prevDeiToDpMap[ deiInner ]
         break


# --------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToTxQueueCmd
# --------------------------------------------------------------------------------
def setTrafficClassToTxQueueMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   txQueueId = ( args.get( 'TXQ_VALUE' ) if 'tx-queue' in args
                                         else args.get( 'UCTXQ_VALUE' ) )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if 7 in tcList and qosHwStatus.tc7ToAnyTxQueueRestricted:
      if noOrDefaultKw:
         # Avoid removing from the list passed to this function
         tcList = [ tc for tc in tcList if tc != 7 ]
      else:
         mode.addError( 'Traffic class 7 to tx-queue mapping cannot be changed' )
         return
   for tc in tcList:
      if noOrDefaultKw or txQueueId == qosHwStatus.defaultTcToTxQueueMap[ tc ]:
         qosInputConfig.tcToTxQueueMap[ tc ] = tacTxQueueId.invalid
      else:
         qosInputConfig.tcToTxQueueMap[ tc ] = txQueueId

# --------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToMcTxQueueCmd
# --------------------------------------------------------------------------------
def setTrafficClassToMcTxQueueMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   txQueueId = args.get( 'MCTXQ_VALUE' )
   for tc in tcList:
      if CliCommand.isNoOrDefaultCmd( args ) or \
            txQueueId == qosHwStatus.defaultTcToMcTxQueueMap[ tc ]:
         qosInputConfig.tcToMcTxQueueMap[ tc ] = tacTxQueueId.invalid
      else:
         qosInputConfig.tcToMcTxQueueMap[ tc ] = txQueueId


# --------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToPriorityGroupCmd
# --------------------------------------------------------------------------------
def setTrafficClassToPriorityGroupMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   pgId = args.get( 'PRIORITYGROUPVALUE' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   for tc in tcList:
      if noOrDefaultKw or int( pgId ) == \
         qosHwStatus.defaultTcToPriorityGroupMap[ tc ]:
         qosInputConfig.tcToPriorityGroupMap[ tc ] = tacPriorityGroup.invalid
      else:
         qosInputConfig.tcToPriorityGroupMap[ tc ] = int( pgId )

# --------------------------------------------------------------------------------
# Handler of QosTrafficClassNameCmd
# --------------------------------------------------------------------------------
def handleTrafficClassName( mode, args ):
   tc = args.get( 'TC_VALUE' )
   tcName = args.get( 'TC_NAME' )
   fgNameToTcMap = qosInputConfig.forwardingGroupNameToTcMap.forwardingGroupNameToTc
   invertIdToName( tc, fgNameToTcMap, tcName )

# --------------------------------------------------------------------------------
# Handler of QosMapToTrafficClassToCosOrExpCmd
# --------------------------------------------------------------------------------
def qosMapToTrafficClassToCosOrExpCmdHandler( mode, args ):
   tc = args.get( 'TC_VALUE' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if 'cos' in args:
      cosList = args.get( 'COS_VALUE' ) or \
            list( args.get( 'COS_VALUES' ).values() )
      setCosToTc( mode, cosList, noOrDefaultKw, tc )
   elif 'exp' in args:
      expList = args.get( 'EXP_VALUE' ) or \
            list( args.get( 'EXP_VALUES' ).values() )
      setExpToTc( mode, expList, noOrDefaultKw, tc )

# --------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToCmd
# --------------------------------------------------------------------------------
def qosMapTrafficClassToCmdHandler( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if 'cos' in args:
      setTrafficClassToCos( mode, tcList, noOrDefaultKw,
            args.get( 'COS' ) )
   elif 'dscp' in args:
      setTrafficClassToDscp( mode, tcList, noOrDefaultKw,
            args.get( 'DSCP' ) )
   elif 'exp' in args:
      setTrafficClassToExp( mode, tcList, noOrDefaultKw,
            args.get( 'EXP' ) )

# --------------------------------------------------------------------------------
# Handler of QosRewriteCosCmd
# --------------------------------------------------------------------------------
def setCosRewrite( mode, args ):
   timestamp = Tac.now()
   prevRewriteConfig = qosInputConfig.cosRewriteEnabled
   if CliCommand.isNoOrDefaultCmd( args ):
      if CliCommand.isDefaultCmd( args ):
         if prevRewriteConfig is tacRewriteEnableMode.rewriteInvalid:
            return
         qosInputConfig.cosRewriteEnabled = tacRewriteEnableMode.rewriteInvalid
      else:  # means 'no'
         if prevRewriteConfig is tacRewriteEnableMode.rewriteDisabled:
            return
         qosInputConfig.cosRewriteEnabled = tacRewriteEnableMode.rewriteDisabled
   else:
      if prevRewriteConfig == tacRewriteEnableMode.rewriteEnabled:
         return
      qosInputConfig.cosRewriteEnabled = tacRewriteEnableMode.rewriteEnabled
   if cliBlockingFail( mode, timestamp, tacFeatureName.cosRewrite, "COS Rewrite" ):
      qosInputConfig.cosRewriteEnabled = prevRewriteConfig

# --------------------------------------------------------------------------------
# Handler of QosCosCmd
# --------------------------------------------------------------------------------
def setQosCosDefault( mode, args ):
   timestamp = Tac.now()
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   defaultCos = args.get( 'COS' )
   # Get the current default Cos value in the intfConfig
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      prevDefaultCos = profile.defaultCos
      if noOrDefault:
         defaultCos = tacCos.invalid
      if prevDefaultCos == defaultCos:
         return
      profile.defaultCos = defaultCos
   else:
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if intfConfig:
         prevDefaultCos = intfConfig.defaultCos
      else:
         prevDefaultCos = tacCos.invalid

      if noOrDefault:
         defaultCos = tacCos.invalid
      if prevDefaultCos == defaultCos:
         return
      setIntfConfig( mode.intf.name, cfgDefaultCos=defaultCos )
      if cliBlockingFail( mode, timestamp, tacFeatureName.defaultCos,
                          "Default Cos", mode.intf.name ):
         setIntfConfig( mode.intf.name, cfgDefaultCos=prevDefaultCos )

# ------------------------------------------------------------------------
# Handler of ShowCosToTcProfile
# ------------------------------------------------------------------------
def showCosToTcProfile( mode, args ):
   profileName = args.get( 'NAME', None )
   cosToTcProfileAllModel = CosToTcProfileAllModel()
   if profileName:
      if profileName in qosStatus.cosToTcProfile:
         profileNames = [ profileName ]
      else:
         profileNames = []
   else:
      profileNames = list( qosStatus.cosToTcProfile.keys() )
   for profileName in profileNames:
      cosToTcProfileModel = CosToTcProfileModel()
      cosToTcMap = qosStatus.cosToTcProfile[ profileName ].cosToTcMap
      for cos in cosToTcMap:
         cosToTcProfileModel.cosToTcMap[ cos ] = cosToTcMap[ cos ]
      cosToTcProfileAllModel.cosToTcProfiles[ profileName ] = cosToTcProfileModel
   return cosToTcProfileAllModel

# -----------------------------------------------------------------------------------
# Handler of ShowTcToCosProfile
# -----------------------------------------------------------------------------------
def showTcToCosProfile( mode, args ):
   mapName = args.get( 'NAME' )
   tcToCosMapAllModel = TcToCosMapAllModel()
   if mapName:
      mapNames = [ mapName ] if mapName in qosStatus.tcToCosNamedMap else []
   else:
      mapNames = qosStatus.tcToCosNamedMap
   for mapName in mapNames:
      tcToCosMapModel = TcToCosMapModel()
      tcToCosNamedMap = qosStatus.tcToCosNamedMap[ mapName ].tcToCosNamedMap
      for tcDpPair in tcToCosNamedMap:
         tcDpIndex = f"tc-{tcDpPair.tc} dp-{tcDpPair.dp}"
         tcToCosMapModel.tcToCosNamedMap[ tcDpIndex ] = tcToCosNamedMap[ tcDpPair ]

      tcToCosMapAllModel.tcToCosNamedMaps[ mapName ] = tcToCosMapModel
   return tcToCosMapAllModel

def getTcToCosMap( mode ):
   return qosInputConfig.tcToCosNamedMap.newMember( mode.mapName )

# -----------------------------------------------------------------------------------
# Handler of TcToCosMapCmd
# -----------------------------------------------------------------------------------
def setTcToCosMap( mode, args ):
   cos = args[ 'COS' ]
   dp = args.get( 'DP' )
   tc = args[ 'TC' ]
   tcToCosNamedMap = getTcToCosMap( mode )
   assert tcToCosNamedMap
   dpList = sorted( tcDpToCosSupportedDpValues() ) if dp is None else [ dp ]
   for dp in dpList:
      tcDpPair = tacTcDpPair( tc, dp )
      tcToCosNamedMap.tcToCosNamedMap[ tcDpPair ] = cos

def getCurrentGlobalTcDpToCosMapping( tc ):
   # While creating a profile we copy from the current global mapping
   # if global mapping does not exist for the cos then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say tc 5 to cos 0 and
   # then goes and does a no tc 5 to cos 0 in the profile.
   cos = qosStatus.tcToCosMap[ tc ]
   if cos == tacCos.invalid:
      cos = qosHwStatus.defaultTcToCosMap[ tc ]
   return cos

# -----------------------------------------------------------------------------------
# No or default handler of TcToCosMapCmd
# -----------------------------------------------------------------------------------
def delTcToCosMap( mode, args ):
   tcToCosNamedMap = getTcToCosMap( mode )
   dp = args.get( 'DP' )
   tc = args[ 'TC' ]
   dpList = sorted( tcDpToCosSupportedDpValues() ) if dp is None else [ dp ]
   for dp in dpList:
      tcDpPair = tacTcDpPair( tc, dp )
      tcToCosNamedMap.tcToCosNamedMap[
         tcDpPair ] = getCurrentGlobalTcDpToCosMapping( tc )

# -----------------------------------------------------------------------------------
# "qos map exp to traffic-class name <profilename>" to enter into
# ExpToTcProfileMode
# -----------------------------------------------------------------------------------
def createTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   if mapName not in qosInputConfig.tcToCosNamedMap:
      qosInputConfig.tcToCosNamedMap.newMember( mapName )
      for tc in range( tacTrafficClass.min, tacTrafficClass.min +
                       qosHwStatus.numTcSupported ):
         for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
            tcDpPair = tacTcDpPair( tc, dp )
            qosInputConfig.tcToCosNamedMap[ mapName ].tcToCosNamedMap[
               tcDpPair ] = getCurrentGlobalTcDpToCosMapping( tc )

   childMode = mode.childMode( TcToCosMode, param=mapName )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultCreateTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   del qosInputConfig.tcToCosNamedMap[ mapName ]

# -----------------------------------------------------------------------------------
# "cos <cos> to traffic-class <tc> in CosToTcProfile mode
# -----------------------------------------------------------------------------------
def getCosToTcProfile( mode ):
   return qosInputConfig.cosToTcProfile.get( mode.profileName )

# --------------------------------------------------------------------------------
# Handler of CosToTrafficClassCmd
# --------------------------------------------------------------------------------
def setCosToTcProfileMap( mode, args ):
   cosValues = args.get( 'COSVALUES' ) or list( args.get( 'COS' ).values() )
   cosToTcProfile = getCosToTcProfile( mode )
   assert cosToTcProfile
   for cos in cosValues:
      cosToTcProfile.cosToTcMap[ cos ] = args[ 'TC_VALUE' ]

# --------------------------------------------------------------------------------
# No or default handler of CosToTrafficClassCmd
# --------------------------------------------------------------------------------
def delCosToTcProfileMap( mode, args ):
   cosValues = args.get( 'COSVALUES' ) or list( args.get( 'COS' ).values() )
   cosToTcProfile = getCosToTcProfile( mode )
   for cos in cosValues:
      # Revert to global/platform default cos-to-tc mapping
      cosToTcProfile.cosToTcMap[ cos ] = getCurrentGlobalCosToTcMapping( cos )

# -----------------------------------------------------------------------------------
# The "[ default|no ] [ mls ] qos map cos <0-7> to traffic-class <tc>" command,
# in "global config" mode.
# -----------------------------------------------------------------------------------
def setCosToTc( mode, cosList, noOrDefaultKw=None, tc=None ):
    # Get the current cosToTcMap config
   prevCosToTcMap = dict( qosInputConfig.cosToTcMap )
   for cos in cosList:
      timestamp = Tac.now()
      if noOrDefaultKw or tc == qosHwStatus.defaultCosToTcMap[ cos ]:
         cosToTcVal = tacTrafficClass.invalid
      else:
         cosToTcVal = tc
      if prevCosToTcMap[ cos ] == cosToTcVal:
         continue
      qosInputConfig.cosToTcMap[ cos ] = cosToTcVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Cos to Tc Map" ):
         # If any of the cosToTc setting fails, revert the entire command
         for cosInner in prevCosToTcMap:
            qosInputConfig.cosToTcMap[ cosInner ] = prevCosToTcMap[ cosInner ]
         break

def setExpToTc( mode, expList, noOrDefaultKw=None, tc=None ):
   # Get the current expToTcMap config
   prevExpToTcMap = dict( qosInputConfig.expToTcMap )
   for exp in expList:
      timestamp = Tac.now()
      if noOrDefaultKw or tc == qosHwStatus.defaultExpToTcMap[ exp ]:
         expToTcVal = tacTrafficClass.invalid
      else:
         expToTcVal = tc
      if prevExpToTcMap[ exp ] == expToTcVal:
         continue
      qosInputConfig.expToTcMap[ exp ] = expToTcVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Exp to Tc Map" ):
         # If any of the expToTc settings fail, revert the entire command
         for expInner in prevExpToTcMap:
            qosInputConfig.expToTcMap[ expInner ] = prevExpToTcMap[ expInner ]
         break

def setTrafficClassToCos( mode, tcList, noOrDefaultKw=None, cos=None ):
    # Get the current tcToCosMap config
   prevTcToCosMap = dict( qosInputConfig.tcToCosMap )
   for tc in tcList:
      timestamp = Tac.now()
      if noOrDefaultKw or cos == qosHwStatus.defaultTcToCosMap[ tc ]:
         tcToCosVal = tacCos.invalid
      else:
         tcToCosVal = cos
      if prevTcToCosMap[ tc ] == tcToCosVal:
         continue
      qosInputConfig.tcToCosMap[ tc ] = tcToCosVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Tc to Cos Map" ):
         # If any of the tcToCos setting fails, revert the entire command
         for tcInner in prevTcToCosMap:
            qosInputConfig.tcToCosMap[ tcInner ] = prevTcToCosMap[ tcInner ]
         break

def setTrafficClassToDscp( mode, tcList, noOrDefaultKw=None, dscp=None ):
   # Get the current tcToDscpMap config
   prevTcToDscpMap = dict( qosInputConfig.tcToDscpMap )
   for tc in tcList:
      timestamp = Tac.now()
      if noOrDefaultKw or dscp == qosHwStatus.defaultTcToDscpMap[ tc ]:
         tcToDscpVal = tacDscp.invalid
      else:
         tcToDscpVal = dscp
      if prevTcToDscpMap[ tc ] == tcToDscpVal:
         continue
      qosInputConfig.tcToDscpMap[ tc ] = tcToDscpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "TC to DSCP Map" ):
         # If any of the tcToDscp setting fails, revert the entire command
         for tcInner in prevTcToDscpMap:
            qosInputConfig.tcToDscpMap[ tcInner ] = prevTcToDscpMap[ tcInner ]
         break

# -----------------------------------------------------------------------------------
# The "qos map traffic-class <tc> to exp <0-7>" command, in "global config" mode.
# -----------------------------------------------------------------------------------
def setTrafficClassToExp( mode, tcList, noOrDefaultKw=None, exp=None ):
   # Get the current tcToDscpMap config
   prevTcToExpMap = dict( qosInputConfig.tcToExpMap )
   for tc in tcList:
      timestamp = Tac.now()
      if noOrDefaultKw or exp == qosHwStatus.defaultTcToExpMap[ tc ]:
         tcToExpVal = tacExp.invalid
      else:
         tcToExpVal = exp
      if prevTcToExpMap[ tc ] == tcToExpVal:
         continue
      qosInputConfig.tcToExpMap[ tc ] = tcToExpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "TC to Exp Map" ):
         # If any of the tcToDscp setting fails, revert the entire command
         for tcInner in prevTcToExpMap:
            qosInputConfig.tcToExpMap[ tcInner ] = prevTcToExpMap[ tcInner ]
         break

# --------------------------------------------------------------------------------
# Handler of QosMapDscpToTrafficClassCmd
# --------------------------------------------------------------------------------
def setDscpToTcDp( mode, args ):
   dscpList = args.get( 'DSCP_VALUE' ) or list( args.get( 'DSCP_VALUES' ).values() )
   tc = args.get( 'TC_VALUE' )
   dp = args.get( 'DROP_PRECEDENCE' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )

   # Get the current dscpToTcMap config
   prevDscpToTcMap = dict( qosInputConfig.dscpToTcMap )
   prevDscpToDpMap = dict( qosInputConfig.dscpToDpMap )
   for dscp in dscpList:
      timestamp = Tac.now()
      if noOrDefaultKw or tc == qosHwStatus.defaultDscpToTcMap[ dscp ]:
         dscpToTcVal = tacTrafficClass.invalid
      else:
         dscpToTcVal = tc
      if not dp or noOrDefaultKw or dp == qosHwStatus.defaultDropPrecedence:
         dscpToDpVal = tacDropPrecedence.invalid
      else:
         dscpToDpVal = dp
      if prevDscpToTcMap[ dscp ] == dscpToTcVal and \
         prevDscpToDpMap[ dscp ] == dscpToDpVal:
         continue
      qosInputConfig.dscpToTcMap[ dscp ] = dscpToTcVal
      qosInputConfig.dscpToDpMap[ dscp ] = dscpToDpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Dscp to Tc/Dp Map" ):
         # If any of the dscpToTc setting fails, revert the entire command
         for dscpInner in prevDscpToTcMap:
            qosInputConfig.dscpToTcMap[ dscpInner ] = prevDscpToTcMap[ dscpInner ]
         for dscpInner in prevDscpToDpMap:
            qosInputConfig.dscpToDpMap[ dscpInner ] = prevDscpToDpMap[ dscpInner ]
         break

# --------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToDscpMapnameCmd
# --------------------------------------------------------------------------------
def configureDscpRewriteMap( mode, args ):
   mapName = args.get( 'MAPNAME' )
   if CliCommand.isNoOrDefaultCmd( args ):
      del qosInputConfig.dscpRewriteMap[ mapName ]
   else:
      if ( len( qosInputConfig.dscpRewriteMap ) >=
           qosHwStatus.numDscpRewriteMapsSupported and
           not mode.session_.startupConfig() ):
         if mapName not in qosInputConfig.dscpRewriteMap:
            mode.addError( "This platform supports only " +
                          f"{qosHwStatus.numDscpRewriteMapsSupported} unique " +
                          "DscpRewriteMaps to be created. Please delete an " +
                          "existing map before creating a new one " )
            return
      qosInputConfig.dscpRewriteMap.newMember( mapName )
      childMode = mode.childMode( DscpRewriteMode, param=mapName )
      mode.session_.gotoChildMode( childMode )

#--------------------------------------------------------------------------------
# Handler of TcToExpMapNameCmd
#--------------------------------------------------------------------------------
def createTcToExpNamedMap( mode, args ):
   mapName = args.get( 'MAP_NAME' )
   if CliCommand.isNoOrDefaultCmd( args ):
      del qosInputConfig.tcDpToExpNamedMap[ mapName ]
   else:
      if mapName not in qosInputConfig.tcDpToExpNamedMap:
         qosInputConfig.tcDpToExpNamedMap.newMember( mapName )

      childMode = mode.childMode( TcToExpMode, param=mapName )
      mode.session_.gotoChildMode( childMode )

#--------------------------------------------------------------------------------
# Handler of TrafficClassToExpCmd
#--------------------------------------------------------------------------------
def createTcDpToExpNamedMap( mode ):
   return qosInputConfig.tcDpToExpNamedMap.get( mode.mapName )

def getTcDpToExpNamedMap( mode ):
   return qosInputConfig.tcDpToExpNamedMap.get( mode.mapName )

def setTcToExpMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   dp = args.get( 'DP' )
   if dp is None:
      dpList = tcDpToExpSupportedDpValues()
   else:
      dpList = [ dp ]
   exp = args[ 'EXP' ]

   tcDpToExpNamedMap = qosInputConfig.tcDpToExpNamedMap.newMember( mode.mapName )
   for tc in tcList:
      for dp in dpList:
         tcDpPair = tacTcDpPair( tc, dp )
         tcDpToExpNamedMap.tcDpToExpMap[ tcDpPair ] = exp

#--------------------------------------------------------------------------------
# No or default handler of TrafficClassToExpCmd
#--------------------------------------------------------------------------------
def delTcToExpMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   dp = args.get( 'DP' )
   if dp is None:
      dpList = tcDpToExpSupportedDpValues()
   else:
      dpList = [ dp ]
   mapName = getTcDpToExpNamedMap( mode )
   if mapName:
      for tc in tcList:
         for dp in dpList:
            tcDpPair = tacTcDpPair( tc, dp )
            del mapName.tcDpToExpMap[ tcDpPair ]

# --------------------------------------------------------------------------------
# Handler of QosRewriteDscpCmd
# --------------------------------------------------------------------------------
def setDscpRewrite( mode, args ):
   timestamp = Tac.now()
   prevRewriteConfig = qosInputConfig.dscpRewriteEnabled
   if CliCommand.isNoOrDefaultCmd( args ):
      if not prevRewriteConfig:
         return
      qosInputConfig.dscpRewriteEnabled = False
   else:
      if prevRewriteConfig:
         return
      qosInputConfig.dscpRewriteEnabled = True

   if cliBlockingFail( mode, timestamp, tacFeatureName.dscpRewrite,
                       "DSCP rewrite" ):
      qosInputConfig.dscpRewriteEnabled = prevRewriteConfig

# --------------------------------------------------------------------------------
# Handler of QosDscpCmd
# --------------------------------------------------------------------------------
def setQosDscpDefault( mode, args ):
   timestamp = Tac.now()
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   defaultDscp = args.get( 'DSCP' )
   # Get the current default Dscp value in the intfConfig
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      prevDefaultDscp = profile.defaultDscp
      if noOrDefault:
         defaultDscp = tacDscp.invalid
      if prevDefaultDscp == defaultDscp:
         return
      profile.defaultDscp = defaultDscp
   else:
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if intfConfig:
         prevDefaultDscp = intfConfig.defaultDscp
      else:
         prevDefaultDscp = tacDscp.invalid

      if noOrDefault:
         defaultDscp = tacDscp.invalid
      if prevDefaultDscp == defaultDscp:
         return
      setIntfConfig( mode.intf.name, cfgDefaultDscp=defaultDscp )
      if cliBlockingFail( mode, timestamp, tacFeatureName.defaultDscp,
                          "Default Dscp", mode.intf.name ):
         setIntfConfig( mode.intf.name, cfgDefaultDscp=prevDefaultDscp )

# -----------------------------------------------------------------------------------
# Handler of ShowDscpToTcProfile
# -----------------------------------------------------------------------------------
def showDscpToTcNamedMap( mode, args ):
   mapName = args.get( 'NAME' )
   dscpToTcMapAllModel = DscpToTcMapAllModel()
   if mapName:
      mapNames = [ mapName ] if mapName in qosStatus.dscpToTcNamedMap else []
   else:
      mapNames = qosStatus.dscpToTcNamedMap
   for mapName in mapNames:
      dscpToTcMapModel = DscpToTcMapModel()
      dscpToTcNamedMap = qosStatus.dscpToTcNamedMap[ mapName ].dscpToTcNamedMap
      for dscp in dscpToTcNamedMap:
         dscpToTcMapModel.dscpToTcNamedMap[ dscp ] = dscpToTcNamedMap[ dscp ]

      dscpToTcMapAllModel.dscpToTcNamedMaps[ mapName ] = dscpToTcMapModel
   return dscpToTcMapAllModel

def getDscpRewriteMap( mode ):
   return qosInputConfig.dscpRewriteMap.get( mode.mapName )

# --------------------------------------------------------------------------------
# Handler of TrafficClassToDscpCmd
# --------------------------------------------------------------------------------
def setTcToDscpMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   dp = args.get( 'DP' )
   if dp is None:
      dpList = tcDpToDscpSupportedDpValues( qosHwStatus )
   else:
      dpList = [ dp ]
   dscp = args[ 'DSCP' ]
   dscpRewriteMap = getDscpRewriteMap( mode )
   assert dscpRewriteMap
   for tc in tcList:
      for dp in dpList:
         tcDpPair = tacTcDpPair( tc, dp )
         dscpRewriteMap.tcToDscpMap[ tcDpPair ] = dscp

# --------------------------------------------------------------------------------
# No or default handler of TrafficClassToDscpCmd
# --------------------------------------------------------------------------------
def delTcToDscpMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or list( args.get( 'TC_LIST' ).values() )
   dp = args.get( 'DP' )
   if dp is None:
      dpList = tcDpToDscpSupportedDpValues( qosHwStatus )
   else:
      dpList = [ dp ]
   mapName = getDscpRewriteMap( mode )
   for tc in tcList:
      for dp in dpList:
         tcDpPair = tacTcDpPair( tc, dp )
         del mapName.tcToDscpMap[ tcDpPair ]

def getDscpToTcMap( mode ):
   return qosInputConfig.dscpToTcNamedMap.get( mode.mapName )

# --------------------------------------------------------------------------------
# Handler of DscpToTcMapCmd
# --------------------------------------------------------------------------------
def setDscpToTcMap( mode, args ):
   dscpToTcNamedMap = getDscpToTcMap( mode )
   assert dscpToTcNamedMap
   dscpList = args.get( 'DSCP_VALUE' ) or list( args.get( 'DSCP_VALUES' ).values() )
   tc = args[ 'TC_VALUE' ]
   for dscp in dscpList:
      dscpToTcNamedMap.dscpToTcNamedMap[ dscp ] = tc

# --------------------------------------------------------------------------------
# No or default handler of DscpToTcMapCmd
# --------------------------------------------------------------------------------
def deleteDscpToTcMap( mode, args ):
   dscpToTcNamedMap = getDscpToTcMap( mode )
   assert dscpToTcNamedMap
   dscpList = args.get( 'DSCP_VALUE' ) or list( args.get( 'DSCP_VALUES' ).values() )
   for dscp in dscpList:
      dscpToTcNamedMap.dscpToTcNamedMap[ dscp ] = \
            getCurrentGlobalDscpToTcMapping( dscp )

# --------------------------------------------------------------------------------
# Handler of QosDscpPreserveCmd
# --------------------------------------------------------------------------------
def applyDscpPreserveMplsEncap( mode, args ):
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      profile.dscpPreserveIpMplsEncapMode = True
   else:
      setIntfConfig( mode.intf.name, cfgDscpPreserveIpMplsEncap=True )

# --------------------------------------------------------------------------------
# No or default handler of QosDscpPreserveCmd
# --------------------------------------------------------------------------------
def deleteDscpPreserveMplsEncap( mode, args ):
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      profile.dscpPreserveIpMplsEncapMode = False
   else:
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if intfConfig:
         setIntfConfig( mode.intf.name, cfgDscpPreserveIpMplsEncap=False )

# --------------------------------------------------------------------------------
# Handler of PoliceRateHigherRateBurstVerTwoCmd
# --------------------------------------------------------------------------------
def policeRateHigherRateBurstVerTwoCmdHandler( mode, args ):
   if 'drop-precedence' in args:
      mode.configurePolicer( mode, no=False, policerMode='trTCM',
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
            policerYellowAction=args[ 'drop-precedence' ],
            pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
            be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
            cmdVersion=2 )
   elif 'dscp' in args:
      dscpValue = args.get( 'DSCP_NAME' ) or args.get( 'DSCP' )
      mode.configurePolicer( mode, no=False, policerMode='trTCM',
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
            policerYellowAction=args[ 'dscp' ], policerYellowValue=dscpValue,
            pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
            be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
            cmdVersion=2 )
   else:
      mode.configurePolicer( mode, no=False, policerMode='trTCM',
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
            pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
            be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
            cmdVersion=2 )

# --------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToDscpCmd
# --------------------------------------------------------------------------------
def showDscpRewriteMaps( mode, args ):
   mapName = args.get( 'MAP' )
   dscpRewriteMapAllModel = DscpRewriteMapAllModel()
   if mapName:
      if mapName in qosStatus.dscpRewriteMap:
         mapNames = [ mapName ]
      else:
         mapNames = []
   else:
      mapNames = list( qosStatus.dscpRewriteMap )
   for mapName in mapNames:
      dscpRewriteMapModel = DscpRewriteMapModel()
      tcToDscpMap = qosStatus.dscpRewriteMap[ mapName ].tcToDscpMap
      for tcDpPair in tcToDscpMap:
         if tcDpPair.dp != tacDropPrecedence.min:
            continue
         dscp = tcToDscpMap[ tcDpPair ]
         if dscp != tacDscp.invalid:
            dscpRewriteMapModel.tcToDscpMap[ tcDpPair.tc ] = dscp

      if qosHwStatus.dscpRewriteDropPrecedenceSupported:
         for tcDpPair in tcToDscpMap:
            dscp = tcToDscpMap[ tcDpPair ]
            if dscp != tacDscp.invalid:
               key = f"tc-{tcDpPair.tc} dp-{tcDpPair.dp}"
               dscpRewriteMapModel.tcDpToDscpMap[ key ] = dscp

      dscpRewriteMapAllModel.dscpRewriteMaps[ mapName ] = dscpRewriteMapModel
   return dscpRewriteMapAllModel

# --------------------------------------------------------------------------------
# Handler of QosInterfacesDscpPreserveCmd
# --------------------------------------------------------------------------------
def showInterfacesDscpPreserve( mode, args ):
   dscpPreserveConfigAllModel = IntfDscpPreserveCollectionModel()
   intfs = args.get( 'INTFS' )
   if not intfs:
      intfType = ( EthIntfCli.EthIntf, SubIntfCli.SubIntf, VlanIntfCli.VlanIntf )
      intfs = IntfCli.Intf.getAll( mode, None, None, intfType=intfType )

   # Create a list of all cfgd ports
   for intf in intfs:
      intf = str( intf )
      if isLagPort( intf ):
         # Case of Port-channel
         intfStatus = getIntfStatus( intf )
         if intfStatus:
            if intfStatus.dscpPreserveMplsIpEncapMode:
               dscpPreserveConfigAllModel.interfaces.append( intf )
         else:
            intfConfig = qosConfig.intfConfig.get( intf )
            if intfConfig:
               if qosConfig.intfConfig[ intf ].dscpPreserveIpMplsEncapMode:
                  dscpPreserveConfigAllModel.interfaces.append( intf )
      else:
         # Case of Ethernet and IntfVlan
         intfStatus = getIntfStatus( intf )
         if intfStatus:
            if intfStatus.dscpPreserveMplsIpEncapMode:
               dscpPreserveConfigAllModel.interfaces.append( intf )

   return dscpPreserveConfigAllModel

# -----------------------------------------------------------------------------------
# "exp <exp> to traffic-class <tc> in ExpToTcProfile mode
# -----------------------------------------------------------------------------------
def getExpToTcProfile( mode ):
   return qosInputConfig.expToTcProfile.get( mode.profileName )

def setExpToTcProfileMap( mode, args ):
   expValues = args.get( 'EXPVALUES' ) or list( args.get( 'EXP' ).values() )
   expToTcProfile = getExpToTcProfile( mode )
   assert expToTcProfile
   for exp in expValues:
      expToTcProfile.expToTcMap[ exp ] = args[ 'TC_VALUE' ]

def delExpToTcProfileMap( mode, args ):
   expValues = args.get( 'EXPVALUES' ) or list( args.get( 'EXP' ).values() )
   expToTcProfile = getExpToTcProfile( mode )
   for exp in expValues:
      # Revert to global/platform default exp-to-tc mapping
      expToTcProfile.expToTcMap[ exp ] = getCurrentGlobalExpToTcMapping( exp )

#--------------------------------------------------------------------------------
# Handler of QosMapTrafficClassToExpCmd
#--------------------------------------------------------------------------------
def showTcDpToExpNamedMaps( mode, args ):
   mapName = args.get( 'MAP' )
   tcDpToExpMapAllModel = TcDpToExpMapAllModel()
   if mapName:
      if mapName in qosStatus.tcDpToExpNamedMap:
         mapNames = [ mapName ]
      else:
         mapNames = []
   else:
      mapNames = list( qosStatus.tcDpToExpNamedMap )
   for mapName in mapNames:
      tcDpToExpMapModel = TcDpToExpMapModel()
      tcDpToExpNamedMap = qosStatus.tcDpToExpNamedMap.get( mapName )
      if tcDpToExpNamedMap:
         tcDpToExpMap = tcDpToExpNamedMap.tcDpToExpMap
         for tcDpPair, exp in tcDpToExpMap.items():
            if exp != tacExp.invalid:
               key = f"tc-{tcDpPair.tc} dp-{tcDpPair.dp}"
               tcDpToExpMapModel.tcDpToExpMap[ key ] = exp

      tcDpToExpMapAllModel.tcDpToExpMaps[ mapName ] = tcDpToExpMapModel
   return tcDpToExpMapAllModel

#-------------------------------------------------------------------------
# qos rewrite traffic-class to exp <mapname>
#-------------------------------------------------------------------------
def applyTcDpToExpMap( mode, args ):
   mapName = args[ 'MAP' ]
   setIntfConfig( mode.intf.name, cfgTcDpToExpMapName=mapName )

def deleteTcDpToExpMap( mode, args ):
   setIntfConfig( mode.intf.name,
         cfgTcDpToExpMapName=tacTcDpToExpMapName.defaultMapName )

# -------------------------------------------------------------------------
# qos rewrite traffic-class to dscp <mapname>
# -------------------------------------------------------------------------
def applyRewriteMap( mode, args ):
   intfConfig = qosInputConfig.intfConfig.newMember( mode.intf.name )
   intfConfig.dscpRewriteMapName = args[ 'MAP' ]

def deleteRewriteMap( mode, args ):
   intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      intfConfig.dscpRewriteMapName = tacDscpRewriteMapName.defaultMapName

# -------------------------------------------------------------------------
# qos map cos to traffic-class <profilename>
# -------------------------------------------------------------------------
def applyCosToTcProfile( mode, args ):
   setIntfConfig( mode.intf.name, cfgCosToTcProfileName=args[ 'MAP' ] )

def removeCosToTcProfile( mode, args ):
   profileName = args.get( 'MAP' )
   intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      shouldDelete = True
      if profileName and intfConfig.cosToTcProfileName != profileName:
         shouldDelete = False

      if shouldDelete:
         setIntfConfig( mode.intf.name,
                     cfgCosToTcProfileName=tacCosToTcProfileName.defaultProfileName )

# -------------------------------------------------------------------------
# qos map exp to traffic-class <profilename>
# -------------------------------------------------------------------------
def applyExpToTcProfile( mode, args ):
   setIntfConfig( mode.intf.name, cfgExpToTcProfileName=args[ 'MAP' ] )

def removeExpToTcProfile( mode, args ):
   profileName = args.get( 'MAP' )
   intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      shouldDelete = True
      if profileName and intfConfig.expToTcProfileName != profileName:
         shouldDelete = False

      if shouldDelete:
         setIntfConfig( mode.intf.name,
                     cfgExpToTcProfileName=tacExpToTcProfileName.defaultProfileName )

def showExpToTcProfile( mode, args ):
   profileName = args.get( 'NAME', None )
   expToTcProfileAllModel = ExpToTcProfileAllModel()
   if profileName:
      if profileName in qosStatus.expToTcProfile:
         profileNames = [ profileName ]
      else:
         profileNames = []
   else:
      profileNames = list( qosStatus.expToTcProfile.keys() )
   for profileName in profileNames:
      expToTcProfileModel = ExpToTcProfileModel()
      expToTcMap = qosStatus.expToTcProfile[ profileName ].expToTcMap
      for exp in expToTcMap:
         expToTcProfileModel.expToTcMap[ exp ] = expToTcMap[ exp ]
      expToTcProfileAllModel.expToTcProfiles[ profileName ] = expToTcProfileModel
   return expToTcProfileAllModel

# -------------------------------------------------------------------------
# qos map dscp to traffic-class <mapname>
# -------------------------------------------------------------------------
def applyDscpToTcMap( mode, args ):
   setIntfConfig( mode.intf.name, cfgDscpToTcMapName=args[ 'MAP_NAME' ] )

def removeDscpToTcMapAttachment( mode, args ):
   mapName = args.get( 'MAP_NAME' )
   intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      shouldDelete = True
      # If the deletion map is not configured on intf, dont remove the map
      if mapName and intfConfig.dscpToTcMapName != mapName:
         shouldDelete = False

      if shouldDelete:
         setIntfConfig( mode.intf.name,
                        cfgDscpToTcMapName=tacDscpToTcMapName.defaultMapName )

def intfTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   setIntfConfig( mode.intf.name, cfgTcToCosMapName=mapName )

def intfRemoveTcToCosMap( mode, args ):
   setIntfConfig( mode.intf.name,
         cfgTcToCosMapName=tacTcToCosMapName.defaultMapName )

def intfCpuTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   setIntfConfig( mode.intf.name, cfgCpuTcToCosMapName=mapName )

def intfCpuRemoveTcToCosMap( mode, args ):
   setIntfConfig( mode.intf.name,
                  cfgCpuTcToCosMapName=tacTcToCosMapName.defaultMapName )
