#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliGlobal
import ConfigMount
import LazyMount
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'QosCliHandler' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

invalidShapeRate = Tac.Value( 'Qos::ShapeRate' )
invalidGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      qosConfig=None,
      qosStatus=None,
      qosHwStatus=None,
      qosSliceHwStatus=None,
      qosAclHwStatus=None,
      qosAclSliceHwStatus=None,
      hwEpochStatus=None,
      cliQosAclConfigReadOnly=None,
      qosAclConfig=None,
      qosAclConfigDir=None,
      qosAclConfigMergeSm=None,
      qosInputConfig=None,
      pfcStatus=None,
      profileConfigDir=None,
      qosSchedulerConfig=None,
   )
)

def startQosAclConfigMergeSm():
   gv.qosAclConfig = Tac.newInstance( "Qos::QosAclConfig" )
   gv.qosAclConfigMergeSm = Tac.newInstance( "Qos::Agent::QosAclConfigMergeSm",
                                          gv.qosAclConfigDir,
                                          gv.cliQosAclConfigReadOnly,
                                          gv.qosAclConfig )

def Plugin( entityManager ):

   gv.qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   gv.qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   gv.qosHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/status/global", "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = \
      "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   gv.qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   gv.qosAclHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   gv.qosAclSliceHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   gv.hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   gv.qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                        "Qos::Input::Config", "w" )
   gv.pfcStatus = LazyMount.mount( entityManager, "dcb/pfc/status", "Pfc::Status",
                                   "r" )
   gv.profileConfigDir = ConfigMount.mount( entityManager, "qos/profile",
                                            "Qos::QosProfileConfigDir", "w" )
   gv.qosSchedulerConfig = ConfigMount.mount( entityManager, "qos/scheduler/config",
                                              "Qos::QosSchedulerConfig", "w" )
   # Set up a single instance of QosAclConfigMergeSm. It needs to mount the Cli
   # config and the input dirs for config from EosSdk. The aggregated view of
   # Qos::QosAclConfig is instantiated in startQosAclConfigMergeSm().
   # Note: cliQosAclConfigReadOnly is used to directly access the cli config entity
   # for use by QosAclConfigMergeSm. This can only be used as an input to that merge
   # SM. cliQosAclConfig, on the other hand is a ConfigMount Proxy that the plugin
   # can write to.
   mg = entityManager.mountGroup()

   gv.cliQosAclConfigReadOnly = mg.mount( "qos/acl/input/cli",
                                       "Qos::Input::AclConfig", "wi" )
   gv.qosAclConfigDir = mg.mount( "qos/acl/input/eosSdkConfig", "Tac::Dir", "ri" )
   mg.close( startQosAclConfigMergeSm )

def getDscpRewriteMap( mode ):
   return gv.qosInputConfig.dscpRewriteMap.get( mode.mapName )

def getDscpToTcMap( mode ):
   return gv.qosInputConfig.dscpToTcNamedMap.get( mode.mapName )

def getTcToCosMap( mode ):
   return gv.qosInputConfig.tcToCosNamedMap.newMember( mode.mapName )
