#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import QosLib
import Tac
from Ark import switchTimeToUtc
from CliPlugin.QosCliCommon import ( counterSupportedEthPhyIntfs,
                                     counterSupportedEthIntfs, setQosProfileConfig )
from CliPlugin.QosCliModel import (fabricIntfName, fabricTokenName, numPriority,
                                   PfcFabricIntfStatusModel, PfcFrameCountersModel,
                                   PfcHistoryCountersModel,
                                   PfcIntfDetailedCountersModel,
                                   PfcIntfHistoryCountersModel,
                                   PfcIntfRangeCountersModel,
                                   PfcIntfRangeDetailedCountersModel,
                                   PfcIntfRangeHistoryCountersModel,
                                   PfcIntfRangeStatusModel,
                                   PfcWatchdogEgressDropCountersModel,
                                   PfcWatchdogIngressDropCountersModel,
                                   PfcWatchdogIntfCountersModel,
                                   PfcWatchdogIntfEgressDropCountersModel,
                                   PfcWatchdogIntfRangeCountersModel,
                                   PfcWatchdogIntfRangeDropCountersModel,
                                   PfcWatchdogTxQueueCountersModel )
from CliPlugin.QosCliPfc import ( PFC_WATCHDOG_INTERMEDIATE_BOUNDARY,
                                 PfcHistoryCounterKey, pfcHistoryCounterStatus,
                                 pfcHistoryStatus, pfcStatus, pfcWdSliceHwStatus,
                                 qosHwStatus, qosInputConfig, qosSliceHwStatus,
                                 isDefaultPfcPortConfig, pfcStatusModelGet,
                                 pfcSupported, statistics, pfcIntfCountersModelGet,
                                 pfcIntfStatusModelGet, pfcWatchdogStatusModelGet,
                                 portPfcConfig, prioritiesToList )
from PfcTypes import tacPfcWatchdogAction


# -----------------------------------------------------------------------------------
# Pfc
# -----------------------------------------------------------------------------------
def prioritiesString( priorities, separator="" ):
   return separator.join( map( str, ( i for i in range( 8 )
                                        if ( 1 << i ) & priorities ) ) )

def hardwareSupports( mode, priorities ):
   if mode.session.disableGuards_:
      return True

   prioritiesSupported = pfcStatus.prioritiesSupported
   hardwareSupported = pfcSupported() and prioritiesSupported
   if hardwareSupported == 0:
      mode.addError( "The hardware does not support priority flow control, "
                     "or forwarding agent is still initializing." )
      return False
   elif prioritiesSupported & priorities == 0:
      mode.session_.addError( "The hardware only supports PFC on priorities",
                              prioritiesString( prioritiesSupported, " " ) )
      return False
   return True

def maybeDeletePortConfig( intfName ):
   intfConfig = qosInputConfig.intfConfig.get( intfName )
   if intfConfig:
      if isDefaultPfcPortConfig( intfConfig.pfcPortConfig ):
         intfConfig.pfcPortConfig = None
      if QosLib.isDefaultIntfConfig( qosHwStatus, intfConfig ):
         del qosInputConfig.intfConfig[ intfName ]

# priority-flow-control mode on
# no priority-flow-control mode
# no priority-flow-control

# -----------------------------------------------------------------------------------
# No or default handler of PriorityFlowControlOnCmd
# -----------------------------------------------------------------------------------
def pfcOff( mode, args ):
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig:
      pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      if pfcPortConfig:
         qosInputConfig.intfConfig[ intf ].pfcPortConfig.enabled = False
      maybeDeletePortConfig( intf )

# -----------------------------------------------------------------------------------
# Handler of PriorityFlowControlOnCmd
# -----------------------------------------------------------------------------------
def pfcOn( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   intf = mode.intf.name
   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]
   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, True, )

   intfConfig.pfcPortConfig.enabled = True

# -----------------------------------------------------------------------------------
# No or default handler of PriorityFlowControlCmd
# -----------------------------------------------------------------------------------
def noPfc( mode, args ):
   """Turn off PFC completely, including priorities."""
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig and \
          qosInputConfig.intfConfig[ intf ].pfcPortConfig:
      qosInputConfig.intfConfig[ intf ].pfcPortConfig = None
   maybeDeletePortConfig( intf )

# -----------------------------------------------------------------------------------
# Handler of PriorityFlowControlPauseWatchdogCmd
# -----------------------------------------------------------------------------------
# enable pfc watchdog on per interface basis
def pfcWatchdogOnOff( mode, args ):
   # by default pfc watchdog is on
   intf = mode.intf.name
   no = CliCommand.isNoCmd( args )
   if intf in qosInputConfig.intfConfig and \
         qosInputConfig.intfConfig[ intf ].pfcPortConfig:
      qosInputConfig.intfConfig[ intf ].pfcPortConfig.watchdogEnabled = not no
   elif no:
      config = qosInputConfig.intfConfig.newMember( intf )
      config.pfcPortConfig = ( intf, False )
      config.pfcPortConfig.watchdogEnabled = False
   maybeDeletePortConfig( intf )

# -----------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogPortActionCmd
# -----------------------------------------------------------------------------------
def configWatchdogPerPortAction( mode, args ):

   intf = mode.intf.name

   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]

   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, False, )

   if 'errdisable' in args:
      intfConfig.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.errdisable
   elif 'drop' in args:
      intfConfig.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.drop
   elif 'notify-only' in args:
      intfConfig.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.notifyOnly

# -----------------------------------------------------------------------------------
# No or default handler of PfCPauseWatchdogPortActionCmd
# -----------------------------------------------------------------------------------
def disableConfigWatchdogPerPortAction( mode, args ):
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig:
      pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      if pfcPortConfig:
         pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.invalid
      maybeDeletePortConfig( intf )

# This function takes the value of the watchdog attribute that conflicts
# with polling-interval and corrects the polling-interval as follows:
# here timeoutVal can either be timeout/recoveryTime
def autoCorrectPfcWatchdogPollingInterval( timeoutVal ):
   # default value is 100 milli seconds
   operPollingInterval = 0.1
   # But polling-interval for watchdog should be at max half of timeoutVal
   # If not, set operPollingInterval to half of timeoutVal for watchdog to work
   if timeoutVal < PFC_WATCHDOG_INTERMEDIATE_BOUNDARY:
      operPollingInterval = ( 0.5 * timeoutVal )
      operPollingInterval = round( operPollingInterval, 3 )
   return operPollingInterval

def addPfcWatchdogWarning( mode, conflictCfgType,
                           timeout, recoveryTime, pollingInterval ):
   if conflictCfgType == "timeout":
      conflictCfgTimeInterval = timeout
      operPollingInterval = autoCorrectPfcWatchdogPollingInterval( timeout )
   elif conflictCfgType == "recovery-time":
      conflictCfgTimeInterval = recoveryTime
      operPollingInterval = autoCorrectPfcWatchdogPollingInterval( recoveryTime )
   warningMessage = f"User configured polling-interval {pollingInterval:.5g} " \
                     f"second(s) is greater than half of {conflictCfgType} " \
                     f"{conflictCfgTimeInterval:.4g} second(s). Setting " \
                     f"polling-interval to {operPollingInterval:.5g} second(s)"
   mode.addWarning( warningMessage )

# Rules for verifying if the config values are in coherence/conflict
# 1. Coherency check when pollingInterval is configured:
#    pollingInterval <= 0.5 * timeout or pollingInterval <= 0.5 * recoveryTime
# 2. No conflict when timeout is 0 even if configured values are incoherent
# 3. If recovery-mode is forced, no conflict even if there is incoherence
#    between recoveryTime and pollingInterval values
# 4. When both timeout and recoveryTime are incoherent with pollingInterval,
#    the we will return config type which has minimum value
def getPfcWatchdogCfgConflictDetails( perPort=False, intf=None, profile=None ):
   if not perPort:
      timeout = qosInputConfig.watchdogTimeout
      recoveryCfg = qosInputConfig.watchdogRecoveryCfg
      recoveryTime = qosInputConfig.watchdogRecoveryCfg.recoveryTime
      forcedRecovery = qosInputConfig.watchdogRecoveryCfg.forcedRecovery
      pollingInterval = qosInputConfig.watchdogPollingInterval
   else:
      # per port timer values
      if intf:
         pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      else:
         pfcPortConfig = profile.pfcPortConfig
      portTimerConfig = pfcPortConfig.portTimerConfig
      timeout = portTimerConfig.portWatchdogTimeout
      recoveryCfg = portTimerConfig.portWatchdogRecoveryCfg
      recoveryTime = recoveryCfg.recoveryTime
      forcedRecovery = recoveryCfg.forcedRecovery
      pollingInterval = portTimerConfig.portWatchdogPollingInterval

   if timeout and pollingInterval:
      if recoveryTime and not forcedRecovery and\
         ( recoveryTime < 2 * pollingInterval ) and ( recoveryTime < timeout ):
         return ( "recovery-time", recoveryCfg, pollingInterval )
      elif timeout < 2 * pollingInterval:
         return ( "timeout", timeout, pollingInterval )

   return None

# Compare the warning details, both before and after a config command,
# and give warning only if there is a change in config values and
def verifyPfcWatchdogConfig( mode, prevConflictDetails, perPort=False, intf=None,
                             profile=None ):
   if not perPort:
      timeout = qosInputConfig.watchdogTimeout
      recoveryTime = qosInputConfig.watchdogRecoveryCfg.recoveryTime
      pollingInterval = qosInputConfig.watchdogPollingInterval
   else:
      if intf:
         pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      else:
         pfcPortConfig = profile.pfcPortConfig
      portTimerConfig = pfcPortConfig.portTimerConfig
      timeout = portTimerConfig.portWatchdogTimeout
      recoveryTime = portTimerConfig.portWatchdogRecoveryCfg.recoveryTime
      pollingInterval = portTimerConfig.portWatchdogPollingInterval

   currConflictDetails = getPfcWatchdogCfgConflictDetails( perPort=perPort,
                                                         intf=intf, profile=profile )
   if currConflictDetails and ( prevConflictDetails != currConflictDetails ):
      conflictCfgType = currConflictDetails[ 0 ]
      addPfcWatchdogWarning( mode, conflictCfgType,
                             timeout, recoveryTime, pollingInterval )

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogDefaultTimeoutCmd
# --------------------------------------------------------------------------------
def configWatchdogTimeout( mode, args ):
   timeout = args.get( 'TIMEOUT' )
   prevConflictDetails = getPfcWatchdogCfgConflictDetails()
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.watchdogTimeout = 0
   elif timeout:
      # verify whether hardware supports all the pfc priorities
      # which is a necessary condition for watchdog to work
      if not hardwareSupports( mode, 0xff ):
         return
      # round the value to a float with 2 digits after the decimal point
      qosInputConfig.watchdogTimeout = round( timeout, 2 )
   verifyPfcWatchdogConfig( mode, prevConflictDetails )

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogDefaultRecoveryTimeCmd
# --------------------------------------------------------------------------------
def configWatchdogRecovery( mode, args ):
   recoveryTime = args.get( 'RECOVERYTIME' )
   prevConflictDetails = getPfcWatchdogCfgConflictDetails()
   recoveryConfig = Tac.Value( "Pfc::WatchdogRecoveryConfig" )
   if CliCommand.isNoOrDefaultCmd( args ):
      recoveryConfig.recoveryTime = 0
      recoveryConfig.forcedRecovery = False
      qosInputConfig.watchdogRecoveryCfg = recoveryConfig
   elif recoveryTime:
      if not hardwareSupports( mode, 0xff ):
         return
      # round the value to a float with 2 digits after the decimal point
      recoveryConfig.recoveryTime = round( recoveryTime, 2 )
      recoveryConfig.forcedRecovery = 'forced' in args
      qosInputConfig.watchdogRecoveryCfg = recoveryConfig
   verifyPfcWatchdogConfig( mode, prevConflictDetails )

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogActionCmd
# --------------------------------------------------------------------------------
def configWatchdogAction( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )

   if noOrDefaultKw or 'errdisable' in args:
      qosInputConfig.watchdogAction = tacPfcWatchdogAction.errdisable
   elif 'drop' in args:
      qosInputConfig.watchdogAction = tacPfcWatchdogAction.drop
   elif 'notify-only' in args:
      qosInputConfig.watchdogAction = tacPfcWatchdogAction.notifyOnly

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogDefaultPollingIntervalCmd
# --------------------------------------------------------------------------------
def configWatchdogPollingInterval( mode, args ):
   interval = args.get( 'INTERVAL' )
   prevConflictDetails = getPfcWatchdogCfgConflictDetails()
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.watchdogPollingInterval = 0
   elif interval:
      if not hardwareSupports( mode, 0xff ):
         return
      # round the value to a float with 3 digits after the decimal point
      qosInputConfig.watchdogPollingInterval = round( interval, 3 )
   verifyPfcWatchdogConfig( mode, prevConflictDetails )

# --------------------------------------------------------------------------------
# Handler of PfcPauseWatchdogHwNDPriorityCmd
# --------------------------------------------------------------------------------
def configWatchdogNonDisruptivePriority( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if noOrDefaultKw:
      qosInputConfig.watchdogNonDisruptivePriorities = 0
      return
   priorityList = args.get( 'PRIORITY' )
   if pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported and\
      len( priorityList ) > pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      mode.addError( 'Maximum number of priorities supported is ' +
            str( pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported ) )
      return
   nonDisruptivePriorities = 0
   for priority in priorityList:
      priorityBit = 1 << priority
      if not hardwareSupports( mode, priorityBit ):
         return
      nonDisruptivePriorities |= priorityBit
   if not nonDisruptivePriorities:
      return
   qosInputConfig.watchdogNonDisruptivePriorities = nonDisruptivePriorities

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogHwNonDisruptivePortOnlyCmd
# --------------------------------------------------------------------------------
def enableWatchdogNonDisruptiveForced( mode, args ):
   enabled = not CliCommand.isNoOrDefaultCmd( args )
   qosInputConfig.forcedNonDisruptiveBehaviour = enabled

# --------------------------------------------------------------------------------
# Handler of PfCPauseWdOverrideActionDropCmd
# --------------------------------------------------------------------------------
def enableWatchdogOverrideAction( mode, args ):
   enabled = not CliCommand.isNoOrDefaultCmd( args )
   qosInputConfig.overrideAction = enabled

# --------------------------------------------------------------------------------
# Handler of PfCPauseWdHwNonDisruptiveActionForwardCmd
# --------------------------------------------------------------------------------
def configWatchdogNonDisruptiveAction( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.watchdogNonDisruptiveAction = tacPfcWatchdogAction.invalid
   else:
      qosInputConfig.watchdogNonDisruptiveAction = tacPfcWatchdogAction.ignorePfc

# --------------------------------------------------------------------------------
# Handler of PriorityFlowControlWatchdogPollingCmd
# --------------------------------------------------------------------------------
def configWatchdogPerPortTimers( mode, args ):
   timeout = args[ 'TIMEOUT' ]
   intf = mode.intf.name
   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]

   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, False, )
   pfcPortConfig = intfConfig.pfcPortConfig

   if not hardwareSupports( mode, 0xff ):
      return

   portTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
   portTimerConfig.portWatchdogTimeout = round( timeout, 2 )

   interval = args.get( 'INTERVAL', 0 )
   portTimerConfig.portWatchdogPollingInterval = round( interval, 3 )

   recoveryCfg = Tac.Value( "Pfc::WatchdogRecoveryConfig" )
   recoveryTime = args.get( 'RECOVERYTIME', 0 )
   recoveryCfg.recoveryTime = round( recoveryTime, 2 )

   recoveryCfg.forcedRecovery = 'forced' in args
   portTimerConfig.portWatchdogRecoveryCfg = recoveryCfg

   portTimerConfig.usePerPortTimerValues = True

   pfcPortConfig.portTimerConfig = portTimerConfig
   verifyPfcWatchdogConfig( mode, None, perPort=True, intf=intf )

# --------------------------------------------------------------------------------
# No or default handler of PriorityFlowControlWatchdogPollingCmd
# --------------------------------------------------------------------------------
def noConfigWatchdogPerPortTimers( mode, args ):
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig:
      pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      if pfcPortConfig:
         portTimerConfig = pfcPortConfig.portTimerConfig
         if portTimerConfig:
            defPortTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
            pfcPortConfig.portTimerConfig = defPortTimerConfig
      maybeDeletePortConfig( intf )

# --------------------------------------------------------------------------------
# Handler of PfCPriorityDefaultCmd
# --------------------------------------------------------------------------------
def configPfcPriorityList( mode, args ):
   priorityList = args.get( 'PRIORITIES' )
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.rxPfcPrioritiesHonored = 0xFF
      return
   rxPfcPrioritiesHonored = 0
   for priority in priorityList:
      priorityBit = 1 << priority
      rxPfcPrioritiesHonored |= priorityBit
   if rxPfcPrioritiesHonored:
      qosInputConfig.rxPfcPrioritiesHonored = rxPfcPrioritiesHonored

# --------------------------------------------------------------------------------
# No or default handler of PriorityFlowControlAllOffCmd
# --------------------------------------------------------------------------------
def pfcGlobalEnabled( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   qosInputConfig.pfcGlobalEnabled = True

# --------------------------------------------------------------------------------
# Handler of PriorityFlowControlAllOffCmd
# --------------------------------------------------------------------------------
def pfcGlobalDisabled( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   qosInputConfig.pfcGlobalEnabled = False

# priority-flow-control priority <0-7> <no-drop|drop>
# no priority-flow-control priority <0-7>

# --------------------------------------------------------------------------------
# Handler of PriorityFlowControlPriorityDropCmd
# --------------------------------------------------------------------------------
def pfcPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   intf = mode.intf.name
   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]

   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, False, )

   pfcPortConfig = intfConfig.pfcPortConfig
   priorities = pfcPortConfig.priorities

   bit = 1 << priority
   if 'drop' in args:
      priorities &= ~bit
   else:
      if not hardwareSupports( mode, bit ):
         return
      priorities |= bit
   intfConfig.pfcPortConfig.priorities = priorities
   maybeDeletePortConfig( intf )

# --------------------------------------------------------------------------------
# No or default handler of PriorityFlowControlPriorityDropCmd
# --------------------------------------------------------------------------------
def noPfcPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig and \
          qosInputConfig.intfConfig[ intf ].pfcPortConfig:
      bit = 1 << priority
      qosInputConfig.intfConfig[ intf ].pfcPortConfig.priorities &= ~bit
      maybeDeletePortConfig( intf )

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogHwMonitorPriorityCmd
# --------------------------------------------------------------------------------
# priority-flow-control pause watchdog hardware monitor priority <priority>
def hwMonitor( mode, args ):
   priorityList = args[ 'PRIORITIES' ]
   if pfcStatus.watchdogMaxHwMonitorPriorities and\
       len( priorityList ) + len( qosInputConfig.hwMonitoredPri ) >\
         pfcStatus.watchdogMaxHwMonitorPriorities:
      mode.addError(
         'Maximum number of hardware monitored priorities supported is ' +
         str( pfcStatus.watchdogMaxHwMonitorPriorities ) )
      return
   for pri in priorityList:
      qosInputConfig.hwMonitoredPri[ pri ] = True

# --------------------------------------------------------------------------------
# No or default handler of PfCPauseWatchdogHwMonitorPriorityCmd
# --------------------------------------------------------------------------------
def noHwMonitor( mode, args ):
   qosInputConfig.hwMonitoredPri.clear()

# ------------------------------------------------------------------------
# QosProfile Mode
# ------------------------------------------------------------------------

# PFC cli commands in qos profile mode
# --------------------------------------------------------------------------------
# Handler of PfCOnProfileCmd
# --------------------------------------------------------------------------------
def qosProfilePfcOn( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig.enabled = True
   else:
      profile.pfcPortConfig = ( "QosProfile", True, )

# --------------------------------------------------------------------------------
# No or default handler of PfCOnProfileCmd
# --------------------------------------------------------------------------------
def qosProfilePfcOff( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig.enabled = False
      setQosProfileConfig( profile )

# --------------------------------------------------------------------------------
# No or default handler of PfCProfileCmd
# --------------------------------------------------------------------------------
def qosProfileNoPfc( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig = None

# --------------------------------------------------------------------------------
# Handler of PfCPauseWatchdogProfileCmd
# --------------------------------------------------------------------------------
def qosProfilePfcWatchdogOnOff( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if CliCommand.isNoCmd( args ):
      if profile.pfcPortConfig:
         profile.pfcPortConfig.watchdogEnabled = False
      else:
         profile.pfcPortConfig = ( "QosProfile", False, )
         profile.pfcPortConfig.watchdogEnabled = False
   else:
      if profile.pfcPortConfig:
         profile.pfcPortConfig.watchdogEnabled = True
         setQosProfileConfig( profile )

# --------------------------------------------------------------------------------
# Handler of PfCPriorityDropProfileCmd
# --------------------------------------------------------------------------------
def qosProfilePfcPriority( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   priority = args[ 'PRIORITY' ]
   profile = mode.qosProfileModeContext.currentEntry_
   if not profile.pfcPortConfig:
      profile.pfcPortConfig = ( "QosProfile", False, )
   bit = 1 << priority
   if 'drop' in args:
      profile.pfcPortConfig.priorities &= ~bit
      profile.fabricPfcDlb &= ~bit
      setQosProfileConfig( profile )
   else:
      if 'dlb' in args:
         profile.fabricPfcDlb |= bit
      elif bit & profile.fabricPfcDlb:
         profile.fabricPfcDlb &= ~bit
      profile.pfcPortConfig.priorities |= bit

# --------------------------------------------------------------------------------
# No or default handler of PfCPriorityDropProfileCmd
# --------------------------------------------------------------------------------
def qosProfileNoPfcPriority( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   priority = args[ 'PRIORITY' ]
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      bit = 1 << priority
      profile.pfcPortConfig.priorities &= ~bit
      if bit & profile.fabricPfcDlb:
         profile.fabricPfcDlb &= ~bit
      setQosProfileConfig( profile )

# --------------------------------------------------------------------------------
# Handler of PfCWatchdogPollingProfileCmd
# --------------------------------------------------------------------------------
def qosProfilePortWatchdogTimer( mode, args ):
   timeout = args[ 'TIMEOUT' ]
   profile = mode.qosProfileModeContext.currentEntry_
   if not profile.pfcPortConfig:
      profile.pfcPortConfig = ( "QosProfile", False, )

   portTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
   portTimerConfig.portWatchdogTimeout = round( timeout, 2 )
   interval = args.get( 'INTERVAL', 0 )
   portTimerConfig.portWatchdogPollingInterval = round( interval, 3 )

   recoveryCfg = Tac.Value( "Pfc::WatchdogRecoveryConfig" )
   recoveryTimeValue = args.get( 'RECOVERYTIME', 0 )
   recoveryCfg.recoveryTime = round( recoveryTimeValue, 2 )

   recoveryCfg.forcedRecovery = 'forced' in args
   portTimerConfig.portWatchdogRecoveryCfg = recoveryCfg
   portTimerConfig.usePerPortTimerValues = True

   profile.pfcPortConfig.portTimerConfig = portTimerConfig
   verifyPfcWatchdogConfig( mode, None, perPort=True, profile=profile )

# --------------------------------------------------------------------------------
# No or default handler of PfCWatchdogPollingProfileCmd
# --------------------------------------------------------------------------------
def qosProfileNoPortWatchdogTimer( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      portTimerConfig = profile.pfcPortConfig.portTimerConfig
      if portTimerConfig:
         defPortTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
         profile.pfcPortConfig.portTimerConfig = defPortTimerConfig
      setQosProfileConfig( profile )

# --------------------------------------------------------------------------------
# Handler of PfCPauseWdPortActionProfileCmd
# --------------------------------------------------------------------------------
def qosProfilePortWatchdogAction( mode, args ):

   profile = mode.qosProfileModeContext.currentEntry_
   if not profile.pfcPortConfig:
      profile.pfcPortConfig = ( "QosProfile", False, )

   if 'errdisable' in args:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.errdisable
   elif 'drop' in args:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.drop
   elif 'notify-only' in args:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.notifyOnly

# --------------------------------------------------------------------------------
# No or default handler of PfCPauseWdPortActionProfileCmd
# --------------------------------------------------------------------------------
def qosProfileNoPortWatchdogAction( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.invalid
      setQosProfileConfig( profile )

#
# -----------------------------------------------------------------------
# SECTION: Priority Flow Control (PFC) Show Command Value Functions
# -----------------------------------------------------------------------
# This section contains definitions of the value functions used for the
# priority flow control show commands.
#
# The value functions make use of helper and model-get routines to
# instantiate and populate the appropriate CLI model objects.
#
# Helper routines:
# . prioritiesToList
# . priArrayToList
# . watchdogActionToEnum
# . getAttrRecursive
# . statistics
# . portPfcConfig
# . portDcbxNote
# . portHwActivationStatus
# . portWatchdogHwStatus
# . txQueueWatchdogSkippable
# . counterSupportedEthIntfs
# . counterSupportedEthPhyIntfs
#
# Sub-Model Get routines:
# . pfcStatusModelGet
# . pfcWatchdogTimerModelGet
# . pfcWatchdogIntfTimerModelGet
# . pfcWatchdogNDPrioritiesModelGet
# . pfcWatchdogNDActionModelGet
# . pfcWatchdogForcedNDModelGet
# . pfcWatchdogOverrideActionModelGet
# . pfcWatchdogStatusModelGet
# . pfcIntfStatusModelGet
# . pfcIntfCountersModelGet
# . pfcPriorityCountersModelGet
# . pfcIntfDetailedCountersModelGet
# . pfcHistoryCountersModelGet
# . pfcIntfHistoryCountersModelGet
# . pfcWatchdogTxQueueCountersModelGet
# . pfcWatchdogIntfCountersModelGet
#
# Model Get routines:
# . pfcFabricIntfStatusModelGet
# . pfcIntfRangeStatusModelGet
# . pfcIntfRangeCountersModelGet
# . pfcIntfRangeDetailedCountersModelGet
# . pfcIntfRangeHistoryCountersModelGet
# . pfcWatchdogIntfRangeCountersModelGet
# . pfcIntfRangeModelGet
#
# Value functions:
# . showPfcFabricStatus
# . showPfcInterfaceStatus
# . showPfcInterfaceCounters
# . showPfcInterfaceDetailedCounters
# . showPfcInterfaceHistoryCounters
# . showPfcWatchdogInterfaceCounters
# . showPfcInterface
# -----------------------------------------------------------------------
#

#
# Obtains the status of PFC Watchdog on the interface hardware.
#
def portWatchdogHwStatus( intfName ):
   for sliceHwStatus in qosSliceHwStatus.values():
      portWatchdogStatus = sliceHwStatus.portWatchdogStatus.get( intfName )

      if portWatchdogStatus:
         return portWatchdogStatus

   # PFC Watchdog in T4/TH4 is currently in StrataMmu and mounted on a different
   # Sysdb path, for other chips , this entity should be empty and for T4/TH4
   # qosSliceHwStatus should not have portWatchdogStatus
   for sliceHwStatus in pfcWdSliceHwStatus.values():
      portWatchdogStatus = sliceHwStatus.portWatchdogStatus.get( intfName )

      if portWatchdogStatus:
         return portWatchdogStatus

   return None

#
# Obtains the status of PFC Watchdog drop counters on the interface hardware.
#
def portWatchdogDropCountersHwStatus( intfName ):
   for sliceHwStatus in qosSliceHwStatus.values():
      portWatchdogStatus = sliceHwStatus.portWatchdogDropCountersStatus.get(
         intfName )
      if portWatchdogStatus:
         return portWatchdogStatus
   for sliceHwStatus in pfcWdSliceHwStatus.values():
      portWatchdogStatus = sliceHwStatus.portWatchdogDropCountersStatus.get(
            intfName )
      if portWatchdogStatus:
         return portWatchdogStatus
   return None

#
# Skips multicast queues as they are not fully supported by the PFC Watchdog.
# Refer bug/133643 for details.
#
def txQueueWatchdogSkippable( txQueue ):
   return txQueue.type == 'mcqueue'

#
# Creates and populates an instance of PfcFrameCountersModel for a priority.
#
def pfcPriorityCountersModelGet( priority, counter, checkpoint ):
   """Return the model for PFC interface counters for a given priority."""

   model = PfcFrameCountersModel()

   model.rxFrames = statistics( counter, checkpoint,
                                [ 'inPfcClassFrames', 'count' ],
                                priority,
                                allowNone=True )

   model.txFrames = statistics( counter, checkpoint,
                                [ 'outPfcClassFrames', 'count' ],
                                priority,
                                allowNone=True )

   return model

#
# Creates and populates an instance of PfcIntfDetailedCountersModel.
#
def pfcIntfDetailedCountersModelGet( intf ):
   """Return the model for PFC detailed counters on a given interface."""

   model = PfcIntfDetailedCountersModel()

   model.priorities = {}

   counter = intf.counter()
   checkpoint = intf.getLatestCounterCheckpoint()

   for p in range( numPriority ):
      model.priorities[ p ] = pfcPriorityCountersModelGet( p, counter, checkpoint )

   return model

#
# Creates and populates an instance of PfcHistoryCountersModel for a priority.
#
def pfcHistoryCountersModelGet( intfId, priority ):
   """Return the model for PFC history counters for a given interface and
      priority."""

   model = PfcHistoryCountersModel()
   key = PfcHistoryCounterKey( intfId, priority )
   pfcHistoryCounter = pfcHistoryCounterStatus.counter.get( key )

   if pfcHistoryCounter is not None:
      model.totalRxPfcOnTransitions = pfcHistoryCounter.totalRxPfcOnTransitions
      model.maxRxPfcOnTimeNs = pfcHistoryCounter.maxRxPfcOnTimeNs
      model.maxRxPfcOnTimestamp = \
            switchTimeToUtc( pfcHistoryCounter.maxRxPfcOnTimestamp )
      model.totalRxPfcOnTimeNs = pfcHistoryCounter.totalRxPfcOnTimeNs

   return model

#
# Creates and populates an instance of PfcIntfHistoryCountersModel.
#
def pfcIntfHistoryCountersModelGet( intf ):
   """Return the model for PFC history counters on a given interface."""

   model = PfcIntfHistoryCountersModel()

   model.priorities = {}

   intfId = intf.status().intfId
   for p in range( numPriority ):
      model.priorities[ p ] = pfcHistoryCountersModelGet( intfId, p )

   return model

#
# Creates and populates an instance of PfcWatchdogTxQueueCountersModel.
#
def pfcWatchdogTxQueueCountersModelGet( txQueue, queueStatus ):
   """Return the model for PFC Watchdog interface counters on a given
      transmit queue."""

   model = PfcWatchdogTxQueueCountersModel()

   model.queueType = 'unicast' if txQueue.type == 'ucqueue' else 'multicast'

   model.stuckCount = (
      queueStatus.stuckInfo.count - queueStatus.stuckInfoOnClear.count )
   model.recoveryCount = (
      queueStatus.recoveryInfo.count - queueStatus.recoveryInfoOnClear.count )

   lastStuck = queueStatus.stuckInfo.lastOccurrence
   lastRecovery = queueStatus.recoveryInfo.lastOccurrence

   model.lastStuckTimestamp = 0.0 if model.stuckCount == 0 else lastStuck
   model.lastRecoveryTimestamp = 0.0 if model.recoveryCount == 0 else lastRecovery

   model.lastStuckTimestampUtc = (
      0.0 if model.stuckCount == 0 else switchTimeToUtc( lastStuck ) )
   model.lastRecoveryTimestampUtc = (
      0.0 if model.recoveryCount == 0 else switchTimeToUtc( lastRecovery ) )

   return model

#
# Creates and populates an instance of PfcWatchdogEgressDropCountersModel.
#
def pfcWatchdogEgressDropCountersModelGet( txQueue, pfcWatchdogIntfDropCounters ):
   """Return the model for PFC Watchdog interface drop counters on a given
      transmit queue."""

   model = PfcWatchdogEgressDropCountersModel()

   egressDropCounters = pfcWatchdogIntfDropCounters.egressDropCounters
   if pfcStatus.watchdogDropCountersSupported:
      model.queueType = 'unicast' if txQueue.type == 'ucqueue' else 'multicast'
      model.outCurrentDrops = egressDropCounters.currentPktsDroppedSnapshot - \
                           egressDropCounters.pktsDroppedSnapshotAtStuck - \
                           egressDropCounters.currentPktsDroppedSnapshotOnClear
      model.outCurrentByteDrops = egressDropCounters.currentBytesDroppedSnapshot - \
                              egressDropCounters.bytesDroppedSnapshotAtStuck - \
                              egressDropCounters.currentBytesDroppedSnapshotOnClear
      model.outAggregateDrops = egressDropCounters.aggregatePktsDropped - \
                             egressDropCounters.aggregatePktsDroppedOnClear + \
                             model.outCurrentDrops
      model.outAggregateByteDrops = egressDropCounters.aggregateBytesDropped - \
                                 egressDropCounters.aggregateBytesDroppedOnClear + \
                                 model.outCurrentByteDrops
   return model

#
# Creates and populates an instance of PfcWatchdogIntfIngressDropCountersModel.
#
def pfcWatchdogIntfIngressDropCountersModelGet( tc, pfcWatchdogIntfDropCounters ):
   """Return the model for PFC Watchdog interface drop counters on a given
      transmit queue."""

   model = PfcWatchdogIngressDropCountersModel()

   ingressDropCounters = pfcWatchdogIntfDropCounters.ingressDropCounters
   if pfcStatus.watchdogDropCountersSupported:
      model.currentDrops = ingressDropCounters.currentPktsDroppedSnapshot - \
                          ingressDropCounters.pktsDroppedSnapshotAtStuck - \
                          ingressDropCounters.currentPktsDroppedSnapshotOnClear
      model.currentByteDrops = ingressDropCounters.currentBytesDroppedSnapshot - \
                               ingressDropCounters.bytesDroppedSnapshotAtStuck - \
                               ingressDropCounters.currentBytesDroppedSnapshotOnClear
      model.aggregateDrops = ingressDropCounters.aggregatePktsDropped - \
                             ingressDropCounters.aggregatePktsDroppedOnClear + \
                             model.currentDrops
      model.aggregateByteDrops = ingressDropCounters.aggregateBytesDropped - \
                                 ingressDropCounters.aggregateBytesDroppedOnClear + \
                                 model.currentByteDrops
   return model

#
# Creates and populates an instance of PfcWatchdogIntfCountersModel.
#
def pfcWatchdogIntfCountersModelGet( intf, portWatchdogStatus ):
   """Return the model for PFC Watchdog counters on a given interface."""

   model = PfcWatchdogIntfCountersModel()

   model.txQueues = {}

   for txQueue, queueStatus in portWatchdogStatus.watchdogQueueStatus.items():
      if txQueueWatchdogSkippable( txQueue ):
         continue

      model.txQueues[ txQueue.id ] = \
         pfcWatchdogTxQueueCountersModelGet( txQueue, queueStatus )

   return model

#
# Creates and populates an instance of PfcWatchdogIntfEgressDropCountersModel.
#
def pfcWatchdogIntfEgressDropCountersModelGet( intf, portWatchdogStatus ):
   """Return the model for PFC Watchdog drop counters on a given interface."""

   model = PfcWatchdogIntfEgressDropCountersModel()

   model.egressCounters = {}

   for txQueue, queueStatus in \
           portWatchdogStatus.watchdogEgressDropCounterStatus.items():
      if txQueueWatchdogSkippable( txQueue ):
         continue

      model.egressCounters[ txQueue.id ] = \
         pfcWatchdogEgressDropCountersModelGet( txQueue, queueStatus )

   for tc, watchdogDropCounterStatus in \
       portWatchdogStatus.watchdogIngressDropCounterStatus.items():
      model.ingressCounters[ tc ] = pfcWatchdogIntfIngressDropCountersModelGet(
            tc, watchdogDropCounterStatus )

   return model

#
# Creates and populates an instance of PfcFabricIntfStatusModel.
#
def pfcFabricIntfStatusModelGet():
   """Return the model for PFC status on the Fabric interface."""

   model = PfcFabricIntfStatusModel()

   model.intfName = fabricTokenName

   model.status = pfcStatusModelGet()
   model.watchdogStatus = pfcWatchdogStatusModelGet()

   #
   # Configuration specific to the Fabric interface.
   #
   portConfig = portPfcConfig( fabricIntfName )

   if not portConfig:
      model.enabled = False
      model.priorities = []
      return model

   model.enabled = portConfig.enabled
   model.priorities = prioritiesToList( portConfig.priorities )

   return model

#
# Creates and populates an instance of PfcIntfRangeStatusModel.
#
def pfcIntfRangeStatusModelGet( intfs ):
   """Return the model for PFC status on a given range of interfaces."""

   model = PfcIntfRangeStatusModel()

   model.status = pfcStatusModelGet()
   model.watchdogStatus = pfcWatchdogStatusModelGet()

   model.interfaceStatuses = {}

   for intf in intfs:
      model.interfaceStatuses[ intf.status().intfId ] = \
                                                      pfcIntfStatusModelGet( intf )

   return model

#
# Creates and populates an instance of PfcIntfRangeCountersModel.
#
def pfcIntfRangeCountersModelGet( intfs ):
   """Return the model for PFC counters on a given range of interfaces."""

   model = PfcIntfRangeCountersModel()

   model.interfaceCounters = {}

   for intf in intfs:
      model.interfaceCounters[ intf.status().intfId ] = \
                                                     pfcIntfCountersModelGet( intf )

   return model

#
# Creates and populates an instance of PfcIntfRangeDetailedCountersModel.
#
def pfcIntfRangeDetailedCountersModelGet( intfs ):
   """Return the model for PFC detailed counters on a given range of interfaces."""
   __revision__ = 3
   model = PfcIntfRangeDetailedCountersModel()

   model.interfaces = {}

   for intf in intfs:
      model.interfaces[ intf.status().intfId ] = \
                                             pfcIntfDetailedCountersModelGet( intf )

   return model

#
# Creates and populates an instance of PfcIntfRangeHistoryCountersModel.
#
def pfcIntfRangeHistoryCountersModelGet( intfs ):
   """Return the model for PFC history counters on a given range of interfaces."""

   model = PfcIntfRangeHistoryCountersModel()

   model.lastClearTimestamp = switchTimeToUtc( pfcHistoryStatus.lastClearTimestamp )
   model.interfaces = {}

   for intf in intfs:
      model.interfaces[ intf.status().intfId ] = \
                                             pfcIntfHistoryCountersModelGet( intf )

   return model

#
# Creates and populates an instance of PfcWatchdogIntfRangeCountersModel.
#
def pfcWatchdogIntfRangeCountersModelGet( intfs ):
   """Return the model for PFC Watchdog counters on a given range of interfaces."""

   model = PfcWatchdogIntfRangeCountersModel()

   model.interfaces = {}
   model.cliOutputIs( bool( intfs ) )

   for intf in intfs:
      portWatchdogStatus = portWatchdogHwStatus( intf.name )

      if not portWatchdogStatus:
         continue

      model.interfaces[ intf.status().intfId ] = \
         pfcWatchdogIntfCountersModelGet( intf, portWatchdogStatus )

   return model

#
# Creates and populates an instance of PfcWatchdogIntfRangeDropCountersModel.
#
def pfcWatchdogIntfRangeDropCountersModelGet( intfs ):
   """Return the model for PFC Watchdog d counters on a given range of interfaces."""

   model = PfcWatchdogIntfRangeDropCountersModel()
   model.interfaces = {}
   model.cliOutputIs( bool( intfs ) )

   model.cliOutputIs( False )

   for intf in intfs:
      portWatchdogStatus = portWatchdogDropCountersHwStatus( intf.name )
      if not portWatchdogStatus:
         continue
      model.cliOutputIs( True )
      model.interfaces[ intf.status().intfId ] = \
         pfcWatchdogIntfEgressDropCountersModelGet( intf, portWatchdogStatus )

   return model

# --------------------------------------------------------------------------------
# Handler of ShowFabricPfc and ShowPfcFabricStatus
# --------------------------------------------------------------------------------
#
# Value function for the PFC fabric status show command.
#
def showPfcFabricStatus( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces Fabric priority-flow-control
      - show priority-flow-control interfaces Fabric status

      Model := PfcFabricIntfStatusModel"""

   return pfcFabricIntfStatusModelGet()

#
# Value function for the PFC interface status show command.
#
def showPfcInterfaceStatus( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control status
      - show priority-flow-control [ interfaces <range> ] status

      Model := PfcIntfRangeStatusModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeStatusModelGet( intfs )

#
# Value function for the PFC interface counters show command.
#
def showPfcInterfaceCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters
      - show priority-flow-control [ interfaces <range> ] counters

      Model := PfcIntfRangeCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeCountersModelGet( intfs )

#
# Value function for the PFC interface detailed counters show command.
#
def showPfcInterfaceDetailedCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters detail
      - show priority-flow-control [ interfaces <range> ] counters detail

      Model := PfcIntfRangeDetailedCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeDetailedCountersModelGet( intfs )

#
# Value function for the PFC interface history counters show command.
#
def showPfcInterfaceHistoryCounters( mode, args ):
   """Return the CLI model for the following command:

      - show priority-flow-control [ interfaces <range> ] counters history

      Model := PfcIntfRangeHistoryCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeHistoryCountersModelGet( intfs )

#
# Value function for the PFC Watchdog interface counters show command.
#
def showPfcWatchdogInterfaceCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters watchdog
      - show priority-flow-control [ interfaces <range> ] counters watchdog

      Model := PfcWatchdogIntfRangeCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcWatchdogIntfRangeCountersModelGet( intfs )

#
# Value function for the PFC Watchdog interface drop counters show command.
#
def showPfcWatchdogInterfaceDropCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters watchdog drop
      - show priority-flow-control [ interfaces <range> ] counters watchdog drop

      Model := PfcWatchdogIntfRangeDropCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcWatchdogIntfRangeDropCountersModelGet( intfs )

#
# Value function for the PFC Watchdog interface drop counters show command.
#
def pfcWatchdogInterfaceDropCountersClear( mode, intf=None, mod=None ):
   """clear watchdog status counters on:
      - clear priority-flow-control counters watchdog drop"""

   intfs = counterSupportedEthIntfs( mode, intf, mod )
   clearCounters = any( portWatchdogDropCountersHwStatus( intf.name )
                        for intf in intfs )
   if clearCounters:
      qosInputConfig.clearWatchdogDropCounters = Tac.now()
#
# Value function for the PFC Watchdog interface stuck/recovery counters show command.
#

def pfcWatchdogInterfaceCountersClear( mode, intf=None, mod=None ):
   """clear watchdog status counters on:
      - clear priority-flow-control counters watchdog"""

   qosInputConfig.clearWatchdogCounters = Tac.now()

# --------------------------------------------------------------------------------
# Handler of ClearPfcCountersWatchdog
# --------------------------------------------------------------------------------
def clearPfcCountersWatchdogHandler( mode, args ):
   if 'drop' in args:
      pfcWatchdogInterfaceDropCountersClear( mode )
   else:
      pfcWatchdogInterfaceCountersClear( mode )

# --------------------------------------------------------------------------------
# Handler of ClearPfcCountersHistory
# --------------------------------------------------------------------------------
def clearPfcCountersHistoryHandler( mode, args ):
   qosInputConfig.clearHistoryCounters = Tac.now()
