#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import LazyMount
import Tac
import Tracing
from QosLib import getSubIntfNum
from QosCliLib import populateShapeRateModel
from QosTypes import tacSchedulerCompensation
from CliPlugin import ( IntfCli, EthIntfCli, SubIntfCli )
from CliPlugin.QosCliCommon import ( populateQosSchedulingGroupsForIntf,
                                     getIntfSubObjectShapeRatePercentReference,
                                     getIntfShapeRatePercentReference,
                                     setIntfConfig, getSubIntfGroupMembership,
                                     getHwShapeRate, populateBwModel,
                                     populateConfiguredSr, getHwGuaranteedBw,
                                     isValidShapeRate, populateConfiguredBw,
                                     isValidGuaranteedBw, getIntfStatus,
                                     getConfiguredLagMembers, isPortChannelIntfId,
                                     QosProfileMode )
from CliPlugin.QosCliModel import ( QosAllSchedulingGroupModel,
                                    QosAllSchedulingHierarchyModel,
                                    IntfSchedulingGroupWithMembersModel,
                                    QosSchedulingHierarchyModel )
from CliPlugin.QosCliScheduling import ( QosSchedulingMode, QosSchedulingPolicyMode,
                                         QosSchedulingIntfMode,
                                         QosSchedulingIntfGroupMode )

__defaultTraceHandle__ = Tracing.Handle( 'QosCliSchedulingHandler' )
t0 = Tracing.trace0
t8 = Tracing.trace8

invalidShapeRate = Tac.Value( 'Qos::ShapeRate' )
invalidGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )

# -----------------------------------------------------------------------------------
# Variables for Qos Scheduling associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
lagInputConfig = None
qosConfig = None
qosStatus = None
qosInputConfig = None
qosSchedulerConfig = None
subIntfHwStatus = None
qosGlobalConfig = None

def goToQosSchedulingIntfMode( mode, args ):
   cliIntf = args[ 'INTFNAME' ]
   if cliIntf.config():
      intfId = cliIntf.config().getRawAttribute( 'intfId' )
   else:
      intfId = Tac.Value( 'Arnet::IntfId', cliIntf.name )
   qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if intfId not in qosSchedulerIntfConfig:
      qosSchedulerIntfConfig.newMember( intfId )

   childMode = mode.childMode( QosSchedulingIntfMode,
                               intfId=intfId )
   mode.session_.gotoChildMode( childMode )

def cleanQosSchedulerGroupConfig( intfName, groupName, members ):
   if intfName in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig[ intfName ]
      del intfConfig.schedulerGroupConfig[ groupName ]
   for m in members:
      if m in qosInputConfig.intfConfig:
         qosInputConfig.intfConfig[ m ].schedulerGroupName = ''

def deleteQosSchedulingIntf( mode, args ):
   cliIntf = args[ 'INTFNAME' ]
   qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if cliIntf.config():
      intfId = cliIntf.config().intfId
   else:
      intfId = Tac.Value( 'Arnet::IntfId', cliIntf.name )
   if intfId in qosSchedulerIntfConfig:
      for groupName, groupConfig in ( qosSchedulerIntfConfig[
            intfId ].group ).items():
         cleanQosSchedulerGroupConfig( intfId, groupName,
                                       set( groupConfig.members ) )
      del qosSchedulerIntfConfig[ intfId ]

def identicalQosSchedulingGroup( group, currentEntry ):
   if not group or not currentEntry or group.policyName != currentEntry.policyName:
      return False
   if len( group.members ) != len( currentEntry.members ):
      return False
   for intfId in currentEntry.members:
      if group.members[ intfId ] != currentEntry.members[ intfId ]:
         return False
   return True

def copyQosSchedulerGroup( newGroup, oldGroup ):
   newGroup.policyName = oldGroup.policyName
   for m in oldGroup.lines:
      oldLine = oldGroup.lines[ m ]
      newLine = newGroup.lines.newMember( m )
      newLine.count = oldLine.count
      for m2 in range( oldLine.count ):
         intfId = oldLine.members[ m2 ]
         assert oldGroup.members[ intfId ] == m
         newGroup.members[ intfId ] = oldGroup.members[ intfId ]
         t8( f"copyQosSchedulerGroup: copy {intfId} at {m},{m2}" )
         newLine.members[ m2 ] = intfId
   newGroup.nextLineIndex = oldGroup.nextLineIndex

class QosSchedulingIntfGroupModeContext():
   def __init__( self, mode, intfId, groupName ):
      self.mode = mode
      self.intf_ = None
      self.intfId_ = intfId
      self.intfName_ = intfId.stringValue
      self.group_ = None
      self.groupName_ = groupName
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.previousEntry_ = None
      qosSchedulingIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
      if intfId not in qosSchedulingIntfConfig:
         return
      qosSchedulingGroupConfig = qosSchedulingIntfConfig[ intfId ].group
      if groupName in qosSchedulingGroupConfig:
         self.group_ = qosSchedulingGroupConfig[ groupName ]
         prevIntfGroup = Tac.newInstance( 'Qos::QosSchedulerGroup',
                                       self.groupName_ )
         copyQosSchedulerGroup( prevIntfGroup, self.group_ )
         self.previousEntry_ = prevIntfGroup

   def copyEditEntry( self ):
      newIntfGroup = Tac.newInstance( 'Qos::QosSchedulerGroup', self.groupName_ )
      copyQosSchedulerGroup( newIntfGroup, self.group_ )
      self.currentEntry_ = newIntfGroup
      return newIntfGroup

   def newEditEntry( self ):
      newIntfGroup = Tac.newInstance( 'Qos::QosSchedulerGroup', self.groupName_ )
      self.currentEntry_ = newIntfGroup

   def groupName( self ):
      return self.groupName_

   def intfName( self ):
      return self.intfName_

   def intfId( self ):
      return self.intfId_

   def currentEntry( self ):
      return self.currentEntry_

   def qosSchedulingIntfGroup( self ):
      return self.group_

   def commit( self ):
      if identicalQosSchedulingGroup( self.group_, self.currentEntry_ ):
         return

      intfConfig = qosInputConfig.intfConfig
      qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
      if self.intfId_ not in qosSchedulerIntfConfig:
         qosSchedulerIntfConfig.newMember( self.intfId_ )
      qosSchedulingGroupConfig = qosSchedulerIntfConfig[ self.intfId_ ].group
      if self.group_ is None:
         groupConfig = qosSchedulingGroupConfig.newMember( self.groupName_ )
      else:
         groupConfig = qosSchedulingGroupConfig.get( self.groupName_ )
      # Because Arnet::IntfId is unhashable, oldMembers will always be a
      # set of strings
      oldMembers = set( groupConfig.members )
      groupConfig.members.clear()
      groupConfig.lines.clear()
      policyChanged = False
      if groupConfig.policyName != self.currentEntry_.policyName:
         groupConfig.policyName = self.currentEntry_.policyName
         policyChanged = True
      commitLineIndex = 0

      # Copy all lines from currentEntry_ to groupConfig, collapsing any empty
      # lines to avoid wrapping around the index space.
      for pendingLineIndex in self.currentEntry_.lines:
         currentLine = self.currentEntry_.lines[ pendingLineIndex ]
         if not currentLine:
            break
         if currentLine.count == 0:
            continue
         newLine = groupConfig.lines.newMember( commitLineIndex )
         newLine.count = currentLine.count
         for memberIndex in currentLine.members:
            # Get the raw Arnet::IntfId value for subIntfId
            subIntfId = currentLine.members.getRawEntry( memberIndex )
            t8( 'commit: groupConfig[', commitLineIndex, '] = currentEntry_[',
                pendingLineIndex, '] memberIndex:', memberIndex,
                'subIntfId:', subIntfId )
            if not subIntfId:
               break
            newLine.members[ memberIndex ] = subIntfId
            groupConfig.members[ subIntfId ] = commitLineIndex
            # Need to use stringValue here to get the correct string; otherwize,
            # the interface name is surrounded by single quotes and will not match.
            oldMembers.discard( subIntfId.stringValue )
         commitLineIndex += 1
      groupConfig.nextLineIndex = commitLineIndex
      self.group_ = groupConfig

      for subIntfId in oldMembers:
         t8( 'Deleting', subIntfId, 'from group config for', self.groupName_ )
         if subIntfId in intfConfig:
            setIntfConfig( subIntfId, cfgSchedulerGroupName="" )
      for subIntfId in groupConfig.members:
         t0( 'create intf config for', subIntfId )
         setIntfConfig( subIntfId, cfgSchedulerGroupName=self.groupName_ )
      if self.intfId_ not in intfConfig:
         intfConfig.newMember( self.intfId_ )
      parentIntfConfig = intfConfig.get( self.intfId_ )
      if self.groupName_ not in parentIntfConfig.schedulerGroupConfig:
         parentIntfConfig.schedulerGroupConfig.newMember(
            self.groupName_, invalidShapeRate, invalidGuaranteedBw )
      if policyChanged:
         policyName = groupConfig.policyName
         policyConfig = ( qosSchedulerConfig.policy.get( policyName )
                          if policyName else None )
         if policyConfig:
            cfgShapeRate = policyConfig.shapeRate
            cfgGuaranteedBw = policyConfig.guaranteedBw
         else:
            # If this policy is deleted or does not exist, apply defaults
            cfgShapeRate = Tac.Value( 'Qos::ShapeRate' )
            cfgGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
         groupIntfConfig = parentIntfConfig.schedulerGroupConfig.get(
            self.groupName_ )
         if groupIntfConfig.shapeRate != cfgShapeRate:
            groupIntfConfig.shapeRate = cfgShapeRate
         if groupIntfConfig.guaranteedBw != cfgGuaranteedBw:
            groupIntfConfig.guaranteedBw = cfgGuaranteedBw

def qosSchedulingIntfGroupConfigCmdHandler( mode, args ):
   groupName = args[ 'GROUPNAME' ]
   intfId = mode.intfId()
   qosSchedulingIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if intfId in qosSchedulingIntfConfig:
      qosSchedulingGroupConfig = qosSchedulingIntfConfig[ intfId ].group
   else:
      qosSchedulingGroupConfig = None
   context = QosSchedulingIntfGroupModeContext( mode, intfId, groupName )
   if qosSchedulingGroupConfig and groupName in qosSchedulingGroupConfig:
      context.copyEditEntry()
   else:
      context.newEditEntry()

   mode.qosSchedulingIntfGroupContext = context
   childMode = mode.childMode( QosSchedulingIntfGroupMode, context=context )
   mode.session_.gotoChildMode( childMode )

def qosSchedulingIntfGroupConfigCmdNoOrDefaultHandler( mode, args ):
   groupName = args[ 'GROUPNAME' ]
   intfId = mode.intfId()
   qosSchedulingIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if intfId in qosSchedulingIntfConfig:
      qosSchedulingGroupConfig = qosSchedulingIntfConfig[ intfId ].group
      if groupName in qosSchedulingGroupConfig:
         members = set( qosSchedulingGroupConfig[ groupName ].members )
         del qosSchedulingGroupConfig[ groupName ]
         cleanQosSchedulerGroupConfig( intfId, groupName, members )

def qosSchedulingPolicyNameCmdHandler( mode, args ):
   policyName = args[ 'POLICYNAME' ]
   context = mode.qosSchedulingIntfGroupModeContext
   group = context.currentEntry_
   group.policyName = policyName

def qosSchedulingPolicyNameCmdNoOrDefaultHandler( mode, args ):
   context = mode.qosSchedulingIntfGroupModeContext
   group = context.currentEntry_
   group.policyName = ""

def qosSchedulingGroupMemberCmdHandler( mode, args ):
   context = mode.qosSchedulingIntfGroupModeContext
   group = context.currentEntry_
   intfs = args[ 'INTFNAME' ]
   lineIndex = group.nextLineIndex
   group.nextLineIndex += 1
   memberLine = group.lines.newMember( lineIndex )
   count = 0
   for _, intf in enumerate( intfs ):
      if intf.config():
         intfId = intf.config().getRawAttribute( 'intfId' )
      else:
         intfId = Tac.Value( 'Arnet::IntfId', intf.name )
      if intfId not in group.members:
         prevGroup = getSubIntfGroupMembership( intfId.stringValue )
         if prevGroup:
            mode.addErrorAndStop(
               f"Subinterface {intfId} is already assigned to group {prevGroup}" )
         group.members[ intfId ] = lineIndex
         memberLine.members[ count ] = intfId
         count += 1
   memberLine.count = count
   if count == 0:
      del group.lines[ lineIndex ]

def qosSchedulingGroupMemberCmdNoOrDefaultHandler( mode, args ):
   context = mode.qosSchedulingIntfGroupModeContext
   group = context.currentEntry_
   if 'INTFNAME' not in args:
      group.members.clear()
      group.lines.clear()
      return
   intfs = args[ 'INTFNAME' ]
   for intf in intfs:
      if intf.config():
         intfId = intf.config().getRawAttribute( 'intfId' )
      else:
         intfId = Tac.Value( 'Arnet::IntfId', intf.name )
      if intfId not in context.currentEntry_.members:
         continue
      m = context.currentEntry_.members[ intfId ]
      del context.currentEntry_.members[ intfId ]
      t8( 'Deleting', intfId, 'which should be on line', m,
          'from group', context.groupName() )
      line = context.currentEntry_.lines[ m ]
      found = False
      for m2 in range( line.count ):
         if found:
            line.members[ m2 - 1 ] = line.members[ m2 ]
            line.members[ m2 ] = Tac.Value( 'Arnet::IntfId' )
            continue
         entry = line.members.getRawEntry( m2 )
         t0( 'Entry', m2, 'is', entry )
         if entry == intfId:
            line.members[ m2 ] = Tac.Value( 'Arnet::IntfId' )
            found = True
            line.count -= 1
      assert found

def identicalQosSchedulingPolicy( policy, currentEntry ):
   if not policy or not currentEntry:
      return False
   return ( policy.guaranteedBw == currentEntry.guaranteedBw and
            policy.shapeRate == currentEntry.shapeRate )

def setSchedulingPolicy( policyName ):
   policyConfig = qosSchedulerConfig.policy.get( policyName )
   if policyConfig:
      cfgShapeRate = policyConfig.shapeRate
      cfgGuaranteedBw = policyConfig.guaranteedBw
   else:
      # If this policy is deleted or does not exist, apply defaults
      cfgShapeRate = Tac.Value( 'Qos::ShapeRate' )
      cfgGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
   for intfId in qosSchedulerConfig.qosSchedulerIntfConfig:
      schIntf = qosSchedulerConfig.qosSchedulerIntfConfig[ intfId ]
      for schGroupName in schIntf.group:
         schGroupIn = schIntf.group[ schGroupName ]
         if schGroupIn.policyName == policyName:
            if intfId not in qosInputConfig.intfConfig:
               qosInputConfig.intfConfig.newMember( intfId )
            if schGroupIn.name not in qosInputConfig.intfConfig[
                  intfId ].schedulerGroupConfig:
               schGroupConfig = qosInputConfig.intfConfig[
                  intfId ].schedulerGroupConfig.newMember( schGroupName,
                                                           cfgShapeRate,
                                                           cfgGuaranteedBw )
            else:
               schGroupConfig = qosInputConfig.intfConfig[
                  intfId ].schedulerGroupConfig.get( schGroupName )
               if schGroupConfig.shapeRate != cfgShapeRate:
                  schGroupConfig.shapeRate = cfgShapeRate
               if schGroupConfig.guaranteedBw != cfgGuaranteedBw:
                  schGroupConfig.guaranteedBw = cfgGuaranteedBw

def copyQosSchedulerPolicy( newPolicy, oldPolicy ):
   newPolicy.guaranteedBw = oldPolicy.guaranteedBw
   newPolicy.shapeRate = oldPolicy.shapeRate

class QosSchedulingPolicyModeContext():
   def __init__( self, mode, policyName ):
      self.mode = mode
      self.policy_ = None
      self.policyName_ = policyName
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.previousEntry_ = None
      qosSchedulingPolicyConfig = qosSchedulerConfig.policy
      if policyName in qosSchedulingPolicyConfig:
         self.policy_ = qosSchedulingPolicyConfig[ policyName ]
         self.policyName_ = policyName
         prevPolicy = Tac.newInstance( 'Qos::QosSchedulerPolicy',
                                       self.policyName_ )
         copyQosSchedulerPolicy( prevPolicy, self.policy_ )
         self.previousEntry_ = prevPolicy

   def copyEditEntry( self ):
      newPolicy = Tac.newInstance( 'Qos::QosSchedulerPolicy', self.policyName_ )
      copyQosSchedulerPolicy( newPolicy, self.policy_ )
      self.currentEntry_ = newPolicy
      return newPolicy

   def newEditEntry( self ):
      newPolicy = Tac.newInstance( 'Qos::QosSchedulerPolicy', self.policyName_ )
      self.currentEntry_ = newPolicy

   def policyName( self ):
      return self.policyName_

   def currentEntry( self ):
      return self.currentEntry_

   def qosSchedulingPolicy( self ):
      return self.policy_

   def commit( self ):
      qosSchedulingPolicyConfig = qosSchedulerConfig.policy
      if identicalQosSchedulingPolicy( self.policy_, self.currentEntry_ ):
         return
      if self.policy_ is None:
         policyConfig = qosSchedulingPolicyConfig.newMember( self.policyName_ )
      else:
         policyConfig = qosSchedulingPolicyConfig.get( self.policyName_ )
      policyConfig.guaranteedBw = self.currentEntry_.guaranteedBw
      policyConfig.shapeRate = self.currentEntry_.shapeRate
      self.policy_ = policyConfig

      setSchedulingPolicy( self.policyName_ )

def gotoQosSchedulingPolicyMode( mode, args ):
   policyName = args[ 'POLICYNAME' ]
   qosSchedulingPolicyConfig = qosSchedulerConfig.policy
   context = QosSchedulingPolicyModeContext( mode, policyName )
   if policyName in qosSchedulingPolicyConfig:
      context.copyEditEntry()
   else:
      context.newEditEntry()

   mode.qosSchedulingPolicyModeContext = context
   childMode = mode.childMode( QosSchedulingPolicyMode, context=context )
   mode.session_.gotoChildMode( childMode )

def deleteQosSchedulingPolicy( mode, args ):
   policyName = args[ 'POLICYNAME' ]
   qosSchedulingPolicyConfig = qosSchedulerConfig.policy
   if policyName in qosSchedulingPolicyConfig:
      del qosSchedulingPolicyConfig[ policyName ]
      setSchedulingPolicy( policyName )

def goToQosSchedulingMode( mode, args ):
   childMode = mode.childMode( QosSchedulingMode )
   mode.session_.gotoChildMode( childMode )

# -----------------------------------------------------------------------------------
# The "show qos scheduling group [ GROUP ] [ INTF ]" command, in "enable" mode.
# -----------------------------------------------------------------------------------
def showQosSchedulingGroup( mode, args ):
   if not subIntfHwStatus.subIntfSchedulingGroupSupported:
      return QosAllSchedulingGroupModel()

   group = args.get( 'GROUP' )
   intf = args.get( 'INTFS' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, intfType=EthIntfCli.EthIntf )
   intfQosScheduling = QosAllSchedulingGroupModel()
   isSubIntfId = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId
   subIntfs = [ intf for intf in qosStatus.intfStatus if isSubIntfId( intf ) ]

   for intf in intfs:
      if not getIntfStatus( intf ):
         continue
      if isPortChannelIntfId( intf.name ):
         # Convert Port-channel1.1 to Ethernet1.1.
         lagMembers = getConfiguredLagMembers( intf.name )
         members = []
         for lagMember in lagMembers:
            members += [ intf.name + '.' + str( getSubIntfNum( subi ) ) for
                         subi in subIntfs if subi.startswith( lagMember ) ]
      else:
         members = [ subi for subi in subIntfs if subi.startswith( intf.name ) ]
      intfSchedulingGroup = populateQosSchedulingGroupsForIntf( intf, group=group,
                                                                members=members )
      if intfSchedulingGroup is not None and \
         not intfSchedulingGroup == IntfSchedulingGroupWithMembersModel():
         intfQosScheduling.insert( intf.name, intfSchedulingGroup )

   return intfQosScheduling

# -----------------------------------------------------------------------------------
# The "show qos scheduling hierarchy [ INTF ]" command, in "enable" mode.
# -----------------------------------------------------------------------------------
def populateSchedulingHierarchy( subIntf, mode ):

   def populateIntfParameters( intf ):
      params = QosSchedulingHierarchyModel.HierarchyModel()
      params.name = intf
      intfConfig = qosConfig.intfConfig.get( intf )
      intfStatus = getIntfStatus( intf )
      if intfConfig and intfStatus:
         statusBw = intfStatus.guaranteedBw
         statusSr = intfStatus.shapeRate

         # configured value
         if isValidGuaranteedBw( intfConfig.guaranteedBw ):
            params.configuredGuaranteedBw = \
               populateConfiguredBw( intfConfig.guaranteedBw, statusBw,
                                     getIntfShapeRatePercentReference( intfStatus ) )

         if isValidShapeRate( intfConfig.shapeRate ):
            params.configuredShapeRate = \
               populateConfiguredSr( intfConfig.shapeRate, statusSr,
                                     getIntfShapeRatePercentReference( intfStatus ) )

         # populate operational value
         opBw = getHwGuaranteedBw( intf )
         if opBw and isValidGuaranteedBw( opBw ):
            params.operationalGuaranteedBw = populateBwModel( opBw )

         opSr = getHwShapeRate( intf )
         if opSr and isValidShapeRate( opSr ):
            params.operationalShapeRate = populateShapeRateModel( opSr )

      return params

   subIntfName = subIntf.name
   intfHierarchyModel = QosSchedulingHierarchyModel()

   # subintf
   subIntfConfig = qosConfig.intfConfig.get( subIntfName )
   if not subIntfConfig:
      return None
   subIntfStatus = getIntfStatus( subIntfName )
   intfHierarchyModel.subinterface = populateIntfParameters( subIntfName )

   # txqueue
   txQueueList = []
   for key, qConfig in subIntfConfig.txQueueConfig.items():
      qHierarchyModel = QosSchedulingHierarchyModel.HierarchyModel()
      qHierarchyModel.name = str( key.id )
      qStatus = subIntfStatus.txQueueStatus.get( key )
      statusBw = None
      statusSr = None
      if qStatus:
         statusBw = qStatus.guaranteedBw
         statusSr = qStatus.shapeRate

      # Configured guaranteed bandwidth
      if isValidGuaranteedBw( qConfig.guaranteedBw ):
         qHierarchyModel.configuredGuaranteedBw = \
            populateConfiguredBw( qConfig.guaranteedBw, statusBw,
                                  getIntfSubObjectShapeRatePercentReference(
                                     subIntfStatus ) )

      # Configured shape rate
      if isValidShapeRate( qConfig.shapeRate ):
         qHierarchyModel.configuredShapeRate = \
            populateConfiguredSr( qConfig.shapeRate, statusSr,
                                  getIntfSubObjectShapeRatePercentReference(
                                     subIntfStatus ) )

      # Operational guaranteed bandwidth
      opBw = getHwGuaranteedBw( subIntfName, txQ=key )
      if opBw and isValidGuaranteedBw( opBw ):
         qHierarchyModel.operationalGuaranteedBw = populateBwModel( opBw )

      # Operational shape rate
      opSr = getHwShapeRate( subIntfName, txQ=key )
      if opSr and isValidShapeRate( opSr ):
         qHierarchyModel.operationalShapeRate = populateShapeRateModel( opSr )

      txQueueList.append( qHierarchyModel )

   if txQueueList:
      intfHierarchyModel.txQueues = txQueueList

   # parent intf
   parentIntfName = subIntfName[ : subIntfName.find( '.' ) ]
   intfConfig = qosConfig.intfConfig.get( parentIntfName )
   intfStatus = getIntfStatus( parentIntfName )
   intfHierarchyModel.parentInterface = populateIntfParameters( parentIntfName )
   if isPortChannelIntfId( parentIntfName ):
      lagMembers = getConfiguredLagMembers( parentIntfName )
      parentIntf = EthIntfCli.EthPhyIntf( lagMembers[ 0 ], mode )
   else:
      parentIntf = EthIntfCli.EthPhyIntf( parentIntfName, mode )
   intfHierarchyModel.parentInterface.interfaceSpeed = parentIntf.bandwidth() * 1000

   # scheduling group
   schedulingGroup = QosSchedulingHierarchyModel.HierarchyModel()

   group = subIntfConfig.schedulerGroupName
   if group and intfConfig and group in intfConfig.schedulerGroupConfig:
      schedulingGroup.name = group
      groupConfig = intfConfig.schedulerGroupConfig[ group ]
      statusBw = None
      statusSr = None
      if intfStatus and group in intfStatus.schedulerGroupStatus:
         statusBw = intfStatus.schedulerGroupStatus[ group ].guaranteedBw
         statusSr = intfStatus.schedulerGroupStatus[ group ].shapeRate

      # Configured guaranteed bandwidth
      if isValidGuaranteedBw( groupConfig.guaranteedBw ):
         schedulingGroup.configuredGuaranteedBw = \
            populateConfiguredBw( groupConfig.guaranteedBw, statusBw,
                                  getIntfSubObjectShapeRatePercentReference(
                                     intfStatus ) )

      # Configured shape rate
      if isValidShapeRate( groupConfig.shapeRate ):
         schedulingGroup.configuredShapeRate = \
            populateConfiguredSr( groupConfig.shapeRate, statusSr,
                                  getIntfSubObjectShapeRatePercentReference(
                                     intfStatus ) )

      # Operational guaranteed bandwidth
      opBw = getHwGuaranteedBw( parentIntfName, group=group )
      if opBw and isValidGuaranteedBw( opBw ):
         schedulingGroup.operationalGuaranteedBw = populateBwModel( opBw )

      # Operational shape rate
      opSr = getHwShapeRate( parentIntfName, group=group )
      if opSr and isValidShapeRate( opSr ):
         schedulingGroup.operationalShapeRate = populateShapeRateModel( opSr )

      intfHierarchyModel.schedulingGroup = schedulingGroup

   return intfHierarchyModel

def showQosSchedulingHierarchy( mode, args ):
   if not subIntfHwStatus.subIntfSchedulingGroupSupported:
      return QosAllSchedulingHierarchyModel()

   allIntfHierarchy = QosAllSchedulingHierarchyModel()
   intf = args.get( 'INTFS' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, intfType=SubIntfCli.SubIntf )

   for intf in intfs:
      if not getIntfStatus( intf ):
         continue
      # We want to ignore et1.1 if et1 is a lag member
      if not isPortChannelIntfId( intf.name ):
         parentIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( intf.name )
         if parentIntf in lagInputConfig.phyIntf:
            continue
      intfHierarchy = populateSchedulingHierarchy( intf, mode )
      if intfHierarchy:
         allIntfHierarchy.interfaces[ intf.name ] = intfHierarchy
   return allIntfHierarchy

def noSchedulerCompensation( mode, args ):
   # resetting to default configuration
   compensation = tacSchedulerCompensation.invalid
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      profile.schedulerCompensation = compensation
   else:
      intf = mode.intf.name
      setIntfConfig( intf, cfgSchedulerCompensation=compensation )

def setSchedulerCompensation( mode, args ):
   compensation = args[ 'BYTES' ]
   minus = 'minus' in args
   if minus:
      compensation = -compensation
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      profile.schedulerCompensation = compensation
   else:
      intf = mode.intf.name
      setIntfConfig( intf, cfgSchedulerCompensation=compensation )

# --------------------------------------------------------------------------------
# Handler of QosNumTxqsPerSubIntfCmd
# --------------------------------------------------------------------------------
def handleNumTxqsPerSubIntf( mode, args ):
   numTxqs = int( args.get( 'NUMTXQS', '4' ) )
   if qosGlobalConfig.numTxqsPerSubIntf != numTxqs:
      qosGlobalConfig.numTxqsPerSubIntf = numTxqs
      mode.addWarning( "Change will take effect only after switch reboot." )

def Plugin( entityManager ):
   global subIntfHwStatus, qosSchedulerConfig, qosStatus, lagInputConfig, \
      qosConfig, qosInputConfig, qosGlobalConfig
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
   qosSchedulerConfig = ConfigMount.mount( entityManager, "qos/scheduler/config",
                                           "Qos::QosSchedulerConfig", "w" )
   qosConfig = LazyMount.mount( entityManager, "qos/config", "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
