#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliCommand
import CliDynamicSymbol
import ConfigMount
import LazyMount
import Tac
import Tracing

from QosCliLib import PMapModelContainer
from QosLib import ( mapTypeToEnum, directionToEnum, coppMapType,
                     copyablePMapTypes, validDpConfigValues, isLagPort,
                     fabricIntfName, invalidShapeRate, invalidGuaranteedBw,
                     fabricTokenName, isSubIntfId )
from QosTypes import ( tacFeatureName, tacTxQueuePriority, tacCMapNm,
                       tacPercent, tacRateUnit, tacBurstUnit, tacPMapNm,
                       tacLatencyThreshold, tacDirection, tacPMapHwPrgmStatus )
from CliPlugin import ( IntfCli, EthIntfCli, SubIntfCli )
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from CliPlugin.QosCliCommon import ( getIntfListFromMode, DYNAMIC_CMAP_EDIT_WARN,
                                     isSubIntfConfigMode, goToTxQueueRangeMode,
                                     setIntfConfig, getIntfStatus, cliBlockingFail,
                                     QosProfileMode, findServicePolicyForIntf,
                                     setQosProfileConfig )
from CliPlugin.QosCliServicePolicy import ( showPMapCurrent, showPMapActive,
                                            showPMapDiff, deleteClassMap,
                                            showCMapDiff, showCMapActive,
                                            showCMapPending, PolicyMapModeQos,
                                            gotoPolicyMapModeCommon, applyProfile,
                                            PolicyMapCopyContext, ClassMapContext,
                                            deletePolicyMap,
                                            PolicyMapClassMode, PolicyMapMode,
                                            policerOutOfRange, setServicePolicy,
                                            waitForCountersUpdate,
                                            QosProfileModeContext )
from CliPlugin.QosCliModel import ( IntfLatencyThresholdCollectionModel,
                                    IntfLatencyThresholdModel,
                                    TxQueueLatencyThresholdModel,
                                    IntfDropThresholdsModel,
                                    TxQueueDropThresholdsModel,
                                    IntfDropThresholdsCollectionModel,
                                    QosProfileConfiguredIntfs,
                                    QosProfileSummaryModel,
                                    PolicyMapAllModel )


__defaultTraceHandle__ = Tracing.Handle( 'QosCliServicePolicyHandler' )
t0 = Tracing.trace0

# -----------------------------------------------------------------------------------
# Variables for Qos Scheduling associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
cliQosAclConfig = None
cliQosAclConfigReadOnly = None
hwEpochStatus = None
qosConfig = None
qosStatus = None
qosAclHwStatus = None
qosHwStatus = None
qosInputConfig = None
qosSliceHwStatus = None
qosAclSliceHwStatus = None
qosAclConfig = None
qosAclConfigDir = None
qosAclConfigMergeSm = None
profileConfigDir = None
qosInputProfileConfig = None

qosCliServicePolicyDynamicSubmodes = CliDynamicSymbol.CliDynamicPlugin(
        "QosCliServicePolicyMode" )

# -----------------------------------------------------------------------------------
# Handlers of ModeletServicePolicyCmd
# -----------------------------------------------------------------------------------
def modeletServicePolicyCmdHandler( mode, args ):
   direction = args.get( 'input' ) or args.get( 'output' )
   setServicePolicy( mode, mapType='qos',
                     pmapName=args[ 'POLICY' ], direction=direction )

def modeletServicePolicyCmdNoOrDefaultHandler( mode, args ):
   direction = args.get( 'input' ) or args.get( 'output' )
   setServicePolicy( mode, noOrDefaultKw=True, mapType='qos',
                     pmapName=args.get( 'POLICY' ), direction=direction )

# -----------------------------------------------------------------------------------
# Handler of QosProfileServicePolicyCmd
# -----------------------------------------------------------------------------------
def qosProfileSetServicePolicy( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   mapType = 'qos'
   pmapName = args.get( 'POLICY' )
   direction = args.get( 'input' ) or args.get( 'output' )
   mapType = mapTypeToEnum( mapType )
   direction = directionToEnum( direction )
   profile = mode.qosProfileModeContext.currentEntry_
   if noOrDefaultKw:
      if profile.servicePolicyConfig:
         key = profile.servicePolicyConfig.key
         if pmapName is None:
            if key.direction == direction and key.type == mapType:
               profile.servicePolicyConfig = None
         elif pmapName == key.pmapName:
            profile.servicePolicyConfig = None
   else:
      key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
      profile.servicePolicyConfig = ( key, )

# -----------------------------------------------------------------------------------
# Handler of PMapModeShowCmd
# -----------------------------------------------------------------------------------
def pmapModeShowCmdHandler( mode, args ):
   if 'diff' in args:
      showPMapDiff( mode )
   elif 'active' in args or 'current' in args:
      showPMapActive( mode )
   else:
      showPMapCurrent( mode )

# -----------------------------------------------------------------------------------
# Handler of PolicyMapCmd
# -----------------------------------------------------------------------------------
def policyMapCmdHandler( mode, args ):
   return PolicyMapModeQos.showPolicyMap( mode, args )

# -----------------------------------------------------------------------------------
# Handler of PolicyMapPmapClassCmapCmd
# -----------------------------------------------------------------------------------
def policyMapPmapClassCmapCmdHandler( mode, args ):
   return PolicyMapClassMode.showPolicyMapClass( mode, args )

# --------------------------------------------------------------------------------
# Handler of PolicyMapTypeQosPmapnameCmd
# --------------------------------------------------------------------------------
def gotoQosPolicyMapMode( mode, args ):
   pmapName = args[ 'PMAP' ]
   mapType = 'qos'
   t0( 'gotoQosPolicyMapMode of policy', pmapName )
   emapType = mapTypeToEnum( mapType )
   if emapType != coppMapType and pmapName == tacPMapNm.coppName:
      # Don't allow copp-system-policy to be configured as QoS map type
      mode.addError( f"{pmapName} cannot be configured as QoS map type" )
      return
   cmapName = tacCMapNm.classDefault
   context = gotoPolicyMapModeCommon( mode, pmapName, mapType, cmapName, None )
   if context:
      mode.pmapContext = context
      childMode = mode.childMode( PolicyMapModeQos, context=context )
      mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# No or default handler of PolicyMapTypeQosPmapnameCmd
# --------------------------------------------------------------------------------
def deleteQosPolicyMap( mode, args ):
   pmapName = args[ 'PMAP' ]
   mapType = 'qos'
   deletePolicyMap( mode, pmapName, mapType, None )

# --------------------------------------------------------------------------------
# Handler of CopyQosPolicyMapCmd
# --------------------------------------------------------------------------------
def copyQosPolicyMap( mode, args ):
   dstPmapName = args[ 'DST_PMAP' ]
   srcPmapName = args[ 'SRC_PMAP' ]
   mapType = 'qos'
   t0(
      f'copyQosPolicyMap: copying parameters from {srcPmapName} to {dstPmapName}' )
   emapType = mapTypeToEnum( mapType )
   assert emapType in copyablePMapTypes()

   if srcPmapName not in cliQosAclConfig.pmapType[ emapType ].pmap:
      mode.addError(
            f'Cannot create {dstPmapName}. ' +
            f'Policy {srcPmapName} does not exist or is not copyable.' )
      return None

   if dstPmapName in cliQosAclConfig.pmapType[ emapType ].pmap:
      mode.addError( f'Policy map {dstPmapName} already exists.' )
      return None

   # Find the source policy-map
   srcPmap = cliQosAclConfig.pmapType[ emapType ].pmap[ srcPmapName ]
   ctx = PolicyMapCopyContext( mode, dstPmapName, srcPmap )

   # Create a new empty policy-map with class-default and copy the
   # src policy-map
   ctx.newEditEntry( cmapName=tacCMapNm.classDefault )
   mode.pmapContext = ctx
   childMode = mode.childMode( PolicyMapModeQos, context=ctx )
   mode.session_.gotoChildMode( childMode )

   return None

# --------------------------------------------------------------------------------
# Handler of ClassMapDropCounterCmd
# --------------------------------------------------------------------------------
def enableQosAclCmapDropCounter( mode, args ):
   cliQosAclConfig.enablePolicerDropCounter = True

# --------------------------------------------------------------------------------
# No or default handler of ClassMapDropCounterCmd
# --------------------------------------------------------------------------------
def disableQosAclCmapDropCounter( mode, args ):
   cliQosAclConfig.enablePolicerDropCounter = False

# --------------------------------------------------------------------------------
# Handler of ClassMapMatchAnyCmapnameCmd
# --------------------------------------------------------------------------------
def gotoQosClassMapMode( mode, args ):
   cmapName = args[ 'CMAP' ]
   mapType = 'qos'
   t0( f'gotoQosClassMapMode {cmapName}' )
   emapType = mapTypeToEnum( mapType )
   context = ClassMapContext( mode, emapType, cmapName, 'match-any' )

   if cmapName in cliQosAclConfig.cmapType[ emapType ].cmap:
      entry = cliQosAclConfig.cmapType[ emapType ].cmap[ cmapName ]
      context.copyEditEntry( entry )
      if entry.dynamic:
         mode.addWarning( DYNAMIC_CMAP_EDIT_WARN )
   else:
      context.newEditEntry()

   mode.cmapContext = context
   childMode = mode.childMode( qosCliServicePolicyDynamicSubmodes.ClassMapModeQos,
                               context=context )
   mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# No or default handler of ClassMapMatchAnyCmapnameCmd
# --------------------------------------------------------------------------------
def classMapMatchAnyCmapnameCmdNoOrDefaultHandler( mode, args ):
   deleteClassMap( mode, args[ 'CMAP' ], 'qos' )

# -----------------------------------------------------------------------------------
# Handler of CmapModeShowCmd
# -----------------------------------------------------------------------------------
def cmapModeShowCmdHandler( mode, args ):
   if 'diff' in args:
      showCMapDiff( mode )
   elif 'active' in args or 'current' in args:
      showCMapActive( mode )
   else:
      showCMapPending( mode )

# -----------------------------------------------------------------------------------
# Handler of ClassMapCmd
# -----------------------------------------------------------------------------------
def classMapCmdHandler( mode, args ):
   return qosCliServicePolicyDynamicSubmodes.ClassMapModeQos.showClassMap(
      mode, args )

# -----------------------------------------------------------------------------------
# The "drop-precedence <1-2> drop-threshold percent <0-100>" command under tx-queue
# -----------------------------------------------------------------------------------
def setTxQueueDropThresholds( mode, txQueueConfig, noOrDefaultKw=None, dp=None,
                              percent=None ):
   if noOrDefaultKw:
      del txQueueConfig.dropThresholds[ dp ]
   else:
      txQueueConfig.dropThresholds[ dp ] = percent

def setDropThresholds( mode, args ):
   dp = args[ 'DP' ]
   percent = args.get( 'PERCENT' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         if noOrDefaultKw:
            return
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      setTxQueueDropThresholds( mode, txQueueConfig, noOrDefaultKw, dp, percent )
      setQosProfileConfig( profile )
   else:
      if ( not mode.session_.startupConfig() and
            dp not in validDpConfigValues( qosHwStatus ) ):
         return

      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if noOrDefaultKw:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if noOrDefaultKw:
                  continue
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
         prevDropThresholds = txQueueConfig.dropThresholds
         setTxQueueDropThresholds( mode, txQueueConfig, noOrDefaultKw, dp, percent )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn,
                             "TxQueue Drop Thresholds", intf ):
            intfConfig = qosInputConfig.intfConfig.get( intf )
            if intfConfig is None:
               intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            for dp in prevDropThresholds:
               txQueueConfig.dropThresholds[ dp ] = prevDropThresholds[ dp ]
            dpList = list( txQueueConfig.dropThresholds )
            for dp in dpList:
               if dp not in prevDropThresholds:
                  del txQueueConfig.dropThresholds[ dp ]
            setIntfConfig( intf )

# --------------------------------------------------------------------------------
# Handler of QosPolicerNameCmd
# --------------------------------------------------------------------------------
def qosPolicerNameCmdHandler( mode, args ):
   configureNamedPolicer( mode, no=False,
         policerName=args[ 'POLICER_NAME' ], cir=int( args[ 'CIR_VALUE' ] ),
         cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args[ 'BC_VALUE' ] ),
         bcUnit=args.get( 'BURST_BC_UNIT' ) )

def configureNamedPolicer( mode, no, policerName, cir=None, cirUnit=None, bc=None,
                           bcUnit=None, shared=True ):
   if no:
      del cliQosAclConfig.namedPolicer[ policerName ]
   else:
      if policerOutOfRange( mode, cir, cirUnit, bc, bcUnit,
                            policerMode='committed' ):
         return
      pir = be = 0
      cirUnit = cirUnit if cirUnit else tacRateUnit.rateUnitbps
      bcUnit = bcUnit if bcUnit else tacBurstUnit.burstUnitBytes

      oldPolicer = None
      policer = None
      bumpVersion = False
      if policerName not in cliQosAclConfig.namedPolicer:
         policer = cliQosAclConfig.namedPolicer.newMember(
                   policerName, cir, bc, pir, be )
         bumpVersion = True
      else:
         oldPolicer = Tac.newInstance( "Qos::PolicerConfig", policerName,
                                       cir, bc, pir, be )
         copyNamedPolicer( cliQosAclConfig.namedPolicer[ policerName ],
                           oldPolicer )

      policer = cliQosAclConfig.namedPolicer[ policerName ]
      policer.named = True
      policer.cir = cir
      policer.bc = bc
      policer.cirUnit = cirUnit
      policer.bcUnit = bcUnit
      # We only support named shared policers using this command.
      policer.shared = True
      if bumpVersion:
         policer.uniqueId = Tac.Value( 'Qos::UniqueId' )
         policer.version += 1
      else:
         if not identicalNamedPolicer( policer, oldPolicer ):
            # We support only named 'shared' policers, and we can only
            # modify rates/burst sizes of this named shared policer.
            # Update the 'paramChangeVersion' for this policer to trigger updating
            # the policer directly in the hw.
            policer.uniqueId = Tac.Value( 'Qos::UniqueId' )
            policer.paramChangeVersion += 1
      # BUG1947473: Should we block waiting for policy maps using this policer
      # to be programmed?
      # In case of rate/burst size changes, oldPolicerConfig has the previous
      # policer configuration.

def copyNamedPolicer( src, dst ):
   dst.named = src.named
   dst.cir = src.cir
   dst.bc = src.bc
   dst.cirUnit = src.cirUnit
   dst.bcUnit = src.bcUnit

def identicalNamedPolicer( p1, p2 ):
   if ( p1.cir != p2.cir or p1.cirUnit != p2.cirUnit or
        p1.bc != p2.bc or p1.bcUnit != p2.bcUnit or
        p1.pir != p2.pir or p1.pirUnit != p2.pirUnit or
        p1.be != p2.be or p1.beUnit != p2.beUnit ):
      return False
   return True

# --------------------------------------------------------------------------------
# No or default handler of QosPolicerNameCmd
# --------------------------------------------------------------------------------
def qosPolicerNameCmdNoOrDefaultHandler( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   configureNamedPolicer( mode, no=noOrDefaultKw,
         policerName=args[ 'POLICER_NAME' ] )

# --------------------------------------------------------------------------------
# Handler of PoliceRateBurstVerTwoCmd
# --------------------------------------------------------------------------------
def policeRateBurstVerTwoCmdHandler( mode, args ):
   mode.configurePolicer( mode, no=False, cir=int( args[ 'CIR' ] ),
         cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args[ 'BURST' ] ),
         bcUnit=args.get( 'BURST_BC_UNIT' ), cmdVersion=2 )

# --------------------------------------------------------------------------------
# Handler of PoliceRateHigherRateBurstVerTwoCmd
# --------------------------------------------------------------------------------
def policeRateHigherRateBurstVerTwoCmdHandler( mode, args ):
   if 'drop-precedence' in args:
      mode.configurePolicer( mode, no=False, policerMode='trTCM',
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
            policerYellowAction=args[ 'drop-precedence' ],
            pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
            be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
            cmdVersion=2 )
   elif 'dscp' in args:
      dscpValue = args.get( 'DSCP_NAME' ) or args.get( 'DSCP' )
      mode.configurePolicer( mode, no=False, policerMode='trTCM',
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
            policerYellowAction=args[ 'dscp' ], policerYellowValue=dscpValue,
            pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
            be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
            cmdVersion=2 )
   else:
      mode.configurePolicer( mode, no=False, policerMode='trTCM',
            cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
            bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
            pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
            be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
            cmdVersion=2 )

# --------------------------------------------------------------------------------
# Handlers of PoliceRateBurstVerOneCmd
# --------------------------------------------------------------------------------
def policeRateBurstVerOneCmdHandler( mode, args ):
   mode.configurePolicer( mode, no=False, cir=int( args[ 'CIR' ] ),
         cirUnit=args[ 'RATE_CIR_UNIT' ], bc=int( args[ 'BC' ] ),
         bcUnit=args[ 'BURST_BC_UNIT' ], cmdVersion=1 )

def policeRateBurstVerOneCmdNoOrDefaultHandler( mode, args ):
   mode.configurePolicer( mode, no=True )

# --------------------------------------------------------------------------------
# Handler of PolicyMapClassSharedPolicerCmd
# --------------------------------------------------------------------------------
def policyMapClassSharedPolicerCmdHandler( mode, args ):
   mode.configurePolicer( mode, no=False, name=args[ 'POLICER_NAME' ] )

def showPolicyMapCountersHelper( mode, emapType, pmapName, direction,
                                 detail, policyMapAllModel, hwStatus ):
   pmapConfigTypePtr = qosAclConfig.pmapType.get( emapType )
   if pmapConfigTypePtr is None:
      return policyMapAllModel
   pmapConfigPtr = pmapConfigTypePtr.pmap.get( pmapName )
   if pmapConfigPtr is None:
      return policyMapAllModel
   for hwStatus_ in hwStatus.values():
      # Static CoPP is not divergent so can break the loop
      # if pmapType is present in any of the slices.
      pmapHwStatusTypePtr = hwStatus_.pmapType.get( emapType )
      if pmapHwStatusTypePtr is not None:
         break
      return policyMapAllModel
   key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", direction, pmapName )
   pmapHwStatusPtr = pmapHwStatusTypePtr.pmap.get( key )
   if pmapHwStatusPtr is None:
      return policyMapAllModel
   if pmapHwStatusPtr.prgmStatus != tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
      return policyMapAllModel
   policyMapsContainer = PMapModelContainer( qosAclConfig, qosStatus,
                                             qosHwStatus, qosSliceHwStatus,
                                             qosAclHwStatus,
                                             qosAclSliceHwStatus, emapType,
                                             direction, hwEpochStatus, None,
                                             policyMapAllModel )
   policyMapsContainer.counterDetail = True
   policyMapsContainer.populatePolicyMapCounters( pmapName )
   return policyMapAllModel

def showPolicyMapCounters( mode, args ):
   pmapName = args.get( 'PMAP', 'copp-system-policy' )
   mapType = args.get( 'copp', 'control-plane' ) or args.get( 'pdp', 'qos' )
   direction = args.get( 'DIRECTION' )
   detail = 'detail' in args
   mapType = 'control-plane' if mapType == 'copp' else mapType
   if 'counters' in args or 'interface' in args:
      return PolicyMapMode.showPolicyMap( mode, args )
   policyMapAllModel = PolicyMapAllModel()
   if qosHwStatus is None:
      return policyMapAllModel
   emapType = mapTypeToEnum( mapType )

   hwStatus = None
   if mapType == 'control-plane' and \
      not qosAclHwStatus.coppDynamicClassSupported:
      hwStatus = qosSliceHwStatus
   else:
      hwStatus = qosAclSliceHwStatus
   pmapConfigTypePtr = qosAclConfig.pmapType.get( emapType )
   if pmapConfigTypePtr is None:
      return policyMapAllModel
   pmapConfigPtr = pmapConfigTypePtr.pmap.get( pmapName )
   if pmapConfigPtr is None:
      return policyMapAllModel

   for hwStatus_ in hwStatus.values():
      # Static CoPP is not divergent so can break the loop
      # if pmapType is present in any of the slices.
      pmapHwStatusTypePtr = hwStatus_.pmapType.get( emapType )
      if pmapHwStatusTypePtr is not None:
         break
      return policyMapAllModel

   if direction is None:
      directions = [ tacDirection.input, tacDirection.output ]
   else:
      directions = [ direction ]

   if mapType == 'control-plane':
      directions = [ tacDirection.input ]

   waitForCountersUpdate( mode, mapType )

   for direction in directions:
      return showPolicyMapCountersHelper( mode, emapType, pmapName, direction,
                                          detail, policyMapAllModel, hwStatus )

def showPMapInterface( mode, args ):
   mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
             args.get( 'pdp', 'qos' )
   intf = args.get( 'INTF' ) or args.get( 'SVI_INTF' ) or args.get( 'VTI_INTF' )
   direction = args.get( 'DIRECTION' )
   intfs = IntfCli.Intf.getAll( mode, intf, None )
   policyMapAllModel = PolicyMapAllModel()
   if not intfs:
      return policyMapAllModel
   emapType = mapTypeToEnum( mapType )
   if mapType not in ( 'qos', 'pdp' ):
      for hwStatus_ in qosSliceHwStatus.values():
         if emapType not in hwStatus_.pmapType:
            return policyMapAllModel
   else:
      for aclHwStatus_ in qosAclSliceHwStatus.values():
         if emapType not in aclHwStatus_.pmapType:
            return policyMapAllModel
   if direction is None:
      directions = [ tacDirection.input ] if mapType == 'pdp' else \
                   [ tacDirection.input, tacDirection.output ]
   else:
      directions = [ directionToEnum( direction ) ]
   if 'counters' in args:
      waitForCountersUpdate( mode, mapType )
   for intf in intfs:
      pmapsFound = findServicePolicyForIntf( intf, emapType, directions )
      pmapDict = {}
      # If a pmap is applied in both the direction we need to set the direction as
      # None
      for pmapName, direction in pmapsFound:
         if pmapName in pmapDict:
            if pmapDict[ pmapName ] != direction:
               pmapDict[ pmapName ] = None
         else:
            pmapDict[ pmapName ] = direction

      for pmapName, direction in pmapDict.items():
         policyMapsContainer = PMapModelContainer( qosConfig, qosAclConfig,
                                                   qosStatus,
                                                   qosHwStatus,
                                                   qosSliceHwStatus,
                                                   qosAclHwStatus,
                                                   qosAclSliceHwStatus,
                                                   emapType, direction,
                                                   hwEpochStatus,
                                                   None,
                                                   policyMapAllModel )
         if 'counters' in args:
            policyMapsContainer.counterDetail = True
         policyMapsContainer.populatePolicyMapInterface( pmapName )
   return policyMapAllModel

def serviceProfileCmdHandler( mode, args ):
   applyProfile( mode, profileName=args[ 'PROFILE' ] )

def serviceProfileNoDefCmdNoOrDefaultHandler( mode, args ):
   applyProfile( mode, profileName=args.get( 'PROFILE' ), noOrDefaultKw=True )

def showQosProfileSummary( mode, args ):
   qosProfileName = args.get( 'PROFILENAME' )
   qosProfileConfig = profileConfigDir.config
   intfToProfileMap = profileConfigDir.intfToProfileMap
   qosProfileSummaryModel = QosProfileSummaryModel()
   profileNames = [ ]
   if qosProfileName:
      if qosProfileName in qosProfileConfig:
         profileNames = [ qosProfileName ]
   else:
      profileNames = list( qosProfileConfig )
   for key in profileNames:
      intfList = [ ]
      for intf in intfToProfileMap:
         intfToProfileMapKey = intfToProfileMap.get( intf )
         if intfToProfileMapKey == key:
            intfName = fabricTokenName if intf == fabricIntfName else intf
            intfList.append( intfName )
      qosProfileSummaryModel.qosProfileApplication[ key ] = \
          QosProfileConfiguredIntfs( intfs=intfList )

   return qosProfileSummaryModel

def serviceVerifyQosHandler( mode, args ):
   qosInputConfig.hwConfigVerificationEnabled = True

def serviceVerifyQosNoHandler( mode, args ):
   qosInputConfig.hwConfigVerificationEnabled = False

# -----------------------------------------------------------------------------------
# The "show qos [mls] interfaces <interface-name> drop-thresholds" command,
# in "enable" mode.
# -----------------------------------------------------------------------------------
def showInterfacesDropThresholds( mode, args ):
   intfDropThresholds = IntfDropThresholdsCollectionModel()
   intfDropThresholds.dropThresholdsSupported = qosHwStatus.\
                                                dropPrecedenceThresholdSupported
   intf = args.get( 'INTF' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, intfType=EthIntfCli.EthIntf )
   if not intfs:
      return intfDropThresholds
   for intf in intfs:
      intfName = intf.name
      if not intfName.startswith( ethOrLagIntfPrefixes ):
         continue
      intfDropThreshold = showDropThresholdInterface( intfName )
      if intfDropThreshold.txQueueDropThresholdsList:
         intfDropThresholds.intfDropThresholdsCollection[
            intfName ] = intfDropThreshold
   return intfDropThresholds


def showDropThresholdInterface( intfName ):
   intfStatus = qosStatus.intfStatus.get( intfName )

   intfDropThresholdsModel = IntfDropThresholdsModel()
   if not qosHwStatus.numTxQueueSupported:
      return intfDropThresholdsModel

   for hwtxqid in range( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         prefix = clitxq.type[ : 2 ].upper()
         txQName = prefix + str( clitxq.id )
         if prefix != 'UC':
            continue
      else:
         txQName = str( clitxq.id )
      txQueueDpThresholds = TxQueueDropThresholdsModel()
      dpThresholdSetting = None
      txQueueDpThresholds.txQueue = txQName
      if isLagPort( intfName ):  # For Lag read from input/config
         if intfName in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intfName ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
            if txQueueConfig:
               if txQueueConfig.dropThresholds:
                  dpThresholdSetting = txQueueConfig.dropThresholds
      elif intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )
         if txQueueStatus and txQueueStatus.dropThresholds:
            dpThresholdSetting = txQueueStatus.dropThresholds
      if dpThresholdSetting:
         txQueueDpThresholds.dropThresholds.update( dpThresholdSetting )
         intfDropThresholdsModel.txQueueDropThresholdsList.append(
            txQueueDpThresholds )

   return intfDropThresholdsModel

def matchMplsTrafficClassCmdHandler( mode, args ):
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( noOrDefault, 'mpls-traffic-class',
                       args.get(
                          'TC' ) or list( args.get( 'TC_RANGE' ).values() ) )

def insertBeforeClassPmapModeCmdHandler( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setClass( no, args[ 'CMAP' ], args.get( 'CLASS' ) )

def queueSetIntfRangeConfigCmdHandler( mode, args ):
   queue = { 'tx-queue': 'TXQSET',
             'uc-tx-queue': 'UCTXQSET',
             'mc-tx-queue': 'MCTXQSET' }
   for key, val in queue.items():
      if key in args:
         goToTxQueueRangeMode( mode, key,
                               list( args.get( val ).values() ) )
         return

def setPriority( mode, args ):
   if CliCommand.isNoCmd( args ):
      priority = tacTxQueuePriority.priorityRoundRobin
   elif not CliCommand.isDefaultCmd( args ) and \
        ( isSubIntfConfigMode( mode ) or
          isinstance( mode.parent_, QosProfileMode ) ):
      # For SubIntfs, publish priorityStrict in case of 'priority strict' cmd.
      priority = tacTxQueuePriority.priorityStrict
   else:
      priority = tacTxQueuePriority.priorityInvalid

   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         if priority == tacTxQueuePriority.priorityInvalid:
            return
         txQueueConfig = profile.txQueueConfig.newMember(
            mode.txQueue, priority, tacPercent.invalid,
            invalidShapeRate, invalidGuaranteedBw )
      else:
         txQueueConfig.priority = priority
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if priority == tacTxQueuePriority.priorityInvalid:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, priority, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if priority == tacTxQueuePriority.priorityInvalid:
                  continue
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, priority, tacPercent.invalid,
                  invalidShapeRate, invalidGuaranteedBw )
            else:
               txQueueConfig.priority = priority
         setIntfConfig( intf )

def handleTxQueueLatencyThreshold( mode, args ):
   latency = args.get( 'LATENCY' )

   latencyThreshold = tacLatencyThreshold()

   # Convert to base unit of nanoseconds
   if args.get( 'microseconds' ):
      latencyThreshold.threshold = latency * 1000
      latencyThreshold.configUnit = 'microseconds'
   elif args.get( 'milliseconds' ):
      latencyThreshold.threshold = latency * 1000000
      latencyThreshold.configUnit = 'milliseconds'

   noOrDefault = latencyThreshold == tacLatencyThreshold()

   def initTxQueueConfig( parent ):
      return parent.txQueueConfig.newMember(
               mode.txQueue,
               tacTxQueuePriority.priorityInvalid,
               tacPercent.invalid,
               invalidShapeRate,
               invalidGuaranteedBw )
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if noOrDefault and not txQueueConfig:
         return

      if not txQueueConfig:
         txQueueConfig = initTxQueueConfig( profile )

      txQueueConfig.latencyThreshold = latencyThreshold
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if noOrDefault:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = initTxQueueConfig( intfConfig )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if noOrDefault:
                  continue
               txQueueConfig = initTxQueueConfig( intfConfig )
         prevLatencyThreshold = txQueueConfig.latencyThreshold
         txQueueConfig.latencyThreshold = latencyThreshold
         setIntfConfig( intf )

         if cliBlockingFail( mode, Tac.now(), tacFeatureName.latencyThreshold,
                             "Latency threshold", intf ):
            txQueueConfig.latencyThreshold = prevLatencyThreshold
            setIntfConfig( intf )

def txQueueLatencyThresholdShowHandler( mode, args ):
   intfLatencyThresholdCollectionModel = IntfLatencyThresholdCollectionModel()
   if qosHwStatus.queueLatencyThresholdOnSubIntfGen3Supported:
      validIntfTypes = ( EthIntfCli.EthIntf, SubIntfCli.SubIntf )
   else:
      validIntfTypes = ( EthIntfCli.EthIntf )
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTERFACE' ), None,
                                intfType=validIntfTypes )

   def genTxqLatencyThresholdModel( intf, txqId ):
      txqLatencyThresholdModel = TxQueueLatencyThresholdModel()
      txq = qosHwStatus.hwTxQueue[ txqId ].txQueue

      prefix = ''
      if qosHwStatus.numMulticastQueueSupported:
         prefix = txq.type[ : 2 ].upper()
         if prefix != 'UC':
            return txqLatencyThresholdModel
      latencyThreshold = tacLatencyThreshold()
      txqLatencyThresholdModel.txQueue = prefix + str( txqId )
      if isLagPort( intf.name ):
         intfConfig = qosInputConfig.intfConfig.get( intf.name )
         if intfConfig:
            txQueueConfig = intfConfig.txQueueConfig.get( txq )
            if txQueueConfig:
               latencyThreshold = txQueueConfig.latencyThreshold
         elif getIntfStatus( intf ):
            # qos profile does not create a "Po" member in  intfConfig
            # pick any member status instead
            intfStatus = getIntfStatus( intf )
            txQueueStatus = intfStatus.txQueueStatus.get( txq )
            if txQueueStatus:
               latencyThreshold = txQueueStatus.latencyThreshold
      else:
         intfStatus = qosStatus.intfStatus.get( intf.name )
         if intfStatus:
            txQueueStatus = intfStatus.txQueueStatus.get( txq )
            if txQueueStatus:
               latencyThreshold = txQueueStatus.latencyThreshold

      txqLatencyThresholdModel.threshold = latencyThreshold.configThreshold()
      txqLatencyThresholdModel.unit = latencyThreshold.configUnitShortName()

      return txqLatencyThresholdModel

   interfaces = intfLatencyThresholdCollectionModel.interfaces
   for intf in intfs:
      if not ( intf.name.startswith( ethOrLagIntfPrefixes ) or
                  ( qosHwStatus.queueLatencyThresholdOnSubIntfGen3Supported and
                  isSubIntfId( intf.name ) ) ):
         continue

      intfLatencyThresholdModel = IntfLatencyThresholdModel()
      txqLatencyThresholdList = intfLatencyThresholdModel.txQueues
      for txqId in range( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
         txqLatencyThresholdList.append( genTxqLatencyThresholdModel( intf,
                                                                      txqId ) )
      interfaces[ intf.name ] = intfLatencyThresholdModel

   return intfLatencyThresholdCollectionModel

def matchDscpCmdHandler( mode, args ):
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   dscpValue = args.get( 'DSCP_NAME' ) or args.get( 'DSCPVALUE' )
   if dscpValue is None:
      dscpValue = list( args.get( 'DSCPRANGE' ).values() )
   ecnValue = args.get( 'ect-ce' ) or args.get( 'non-ect' ) or \
      args.get( 'ce' ) or args.get( 'ect' )
   mode.setMatchValue( noOrDefault, 'dscp', dscpValue, None, ecnValue )

def matchEcnCmdHandler( mode, args ):
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   ecnValue = args.get( 'ect-ce' ) or args.get( 'non-ect' ) or \
      args.get( 'ce' ) or args.get( 'ect' )
   mode.setMatchValue( noOrDefault, 'ecn', None, None, ecnValue )

def noOrDefaultClassMapModeMatchCmdNoOrDefaultHandler( mode, args ):
   matchType = args.get( 'dscp' ) or args.get( 'ecn' ) or \
      args.get( 'mpls-traffic-class', 'mac' )
   mode.setMatchValue( True, matchType, None )

# --------------------------------------------------------------------------------
# Handler of QosProfileProfilenameCmd
# --------------------------------------------------------------------------------
def gotoQosProfileMode( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   qosProfileConfig = profileConfigDir.config
   context = QosProfileModeContext( mode, profileName )
   if profileName in qosProfileConfig:
      context.copyEditEntry()
   else:
      context.newEditEntry()

   mode.qosProfileModeContext = context
   childMode = mode.childMode( QosProfileMode, context=context )
   mode.session_.gotoChildMode( childMode )

# --------------------------------------------------------------------------------
# No or default handler of QosProfileProfilenameCmd
# --------------------------------------------------------------------------------
def deleteQosProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   qosProfileConfig = profileConfigDir.config
   intfToProfileMap = profileConfigDir.intfToProfileMap
   if not profileName in qosProfileConfig:
      return
   if qosInputProfileConfig.servicePolicyConfig:
      key = next( iter( qosInputProfileConfig.servicePolicyConfig ) )
   for intf in intfToProfileMap:
      if intfToProfileMap[ intf ] != profileName:
         continue
      if qosInputProfileConfig.servicePolicyConfig:
         del qosInputProfileConfig.servicePolicyConfig[ key ].intfIds[ intf ]
         if len( qosInputProfileConfig.servicePolicyConfig[ key ].intfIds ) == 0:
            del qosInputProfileConfig.servicePolicyConfig[ key ]
      intfConfig = qosInputProfileConfig.intfConfig.get( intf )
      if intfConfig:
         del qosInputProfileConfig.intfConfig[ intf ]
      ecnIntfCounterConfig = qosInputProfileConfig.ecnIntfCounterConfig.get( intf )
      if ecnIntfCounterConfig:
         del qosInputProfileConfig.ecnIntfCounterConfig[ intf ]
   del qosProfileConfig[ profileName ]

def startQosAclConfigMergeSm():
   global qosAclConfig, qosAclConfigMergeSm
   qosAclConfig = Tac.newInstance( "Qos::QosAclConfig" )
   qosAclConfigMergeSm = Tac.newInstance( "Qos::Agent::QosAclConfigMergeSm",
                                          qosAclConfigDir,
                                          cliQosAclConfigReadOnly,
                                          qosAclConfig )


#--------------------------------------------------------------------------------
# Handler of PerInterface Counter Toggle
#--------------------------------------------------------------------------------
def enableCntIntf( mode, args ):
   mode.addWarning( 'Note that this command will increase TCAM usage and might '
                   'exhaust resources' )
   cliQosAclConfig.perIntfCounter = True

#--------------------------------------------------------------------------------
# No or default handler of PerInterface Counter Toggle 
#--------------------------------------------------------------------------------
def disableCntIntf( mode, args ):
   cliQosAclConfig.perIntfCounter = False

def Plugin( entityManager ):
   global qosHwStatus, qosAclHwStatus, qosSliceHwStatus, qosAclSliceHwStatus, \
   qosConfig, qosStatus, hwEpochStatus, qosInputConfig, cliQosAclConfigReadOnly, \
   cliQosAclConfig, qosAclConfigDir, profileConfigDir, qosInputProfileConfig
   profileConfigDir = ConfigMount.mount( entityManager, "qos/profile",
                                         "Qos::QosProfileConfigDir", "w" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosInputProfileConfig = ConfigMount.mount(
         entityManager, "qos/input/config/qosProfile",
         "Qos::Input::Config", "w" )
   qosAclSliceHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   cliQosAclConfig = ConfigMount.mount( entityManager, "qos/acl/input/cli",
                                        "Qos::Input::AclConfig", "w" )
   mg = entityManager.mountGroup()
   qosAclConfigDir = mg.mount( "qos/acl/input/eosSdkConfig", "Tac::Dir", "ri" )
   cliQosAclConfigReadOnly = mg.mount( "qos/acl/input/cli",
                                       "Qos::Input::AclConfig", "wi" )
   mg.close( startQosAclConfigMergeSm )
