#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import difflib
import io
import sys
from collections import defaultdict

import Cell
import QosLib
import LazyMount
import ConfigMount
import Tac
import Tracing
from CliPlugin import ( EthIntfCli, FabricIntfCli, IntfCli, SubIntfCli )
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from CliPlugin.QosCliCommon import ( showQosInterface, findServicePolicyForIntf,
                                     defaultWeight )
from CliPlugin.QosCliModel import ( INVALID, CosToTcProfileAllModel,
                                   CosToTcProfileModel, DscpRewriteMapAllModel,
                                   DscpRewriteMapModel, DscpToTcMapAllModel,
                                   DscpToTcMapModel, ExpToTcProfileAllModel,
                                   ExpToTcProfileModel, FabricAllQosAllModel,
                                   GlobalQosMapsModel, IntfAllQosAllModel,
                                   IntfQosAllModel, LagSubIntfLoadbalancingModel,
                                   TcToCosMapAllModel, TcToCosMapModel,
                                   IntfServicePolicyQosModel,
                                   TxQueueShapeRateAdaptiveModel,
                                   TcDpToExpMapAllModel, TcDpToExpMapModel,
                                   SchedulerProfileRecapModel,
                                   TxQueueSchedulerMapModel )
from QosLib import ( fabricIntfName, fabricTokenName, qosMapType,
                    tacGuaranteedBwUnit, tacGuaranteedBwVal, tacLatencyThreshold,
                    tacPercent, tacPfcWatchdogAction, tacQueueType,
                    tacSchedulerCompensation, tacShapeRateVal, tacTrustMode,
                    tacTxQueuePriority, isLagPort )
from QosCliLib import PMapModelContainer
from QosTypes import ( tacCos, tacDropPrecedence, tacDscp, tacExp, tacPriorityGroup,
                     tacTrafficClass, tacTxQueueId )

__defaultTraceHandle__ = Tracing.Handle( 'QosShowCommandsHandler' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

SAND_QOS_SLICE_NAME = "SandQos-0"

# -----------------------------------------------------------------------------------
# Variables for Qos Scheduling associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
cliQosAclConfigReadOnly = None
hwEpochStatus = None
lagInputConfig = None
qosConfig = None
qosStatus = None
qosAclHwStatus = None
qosHwStatus = None
qosSliceHwStatus = None
qosAclSliceHwStatus = None
qosAclConfig = None
qosAclConfigDir = None
qosAclConfigMergeSm = None
qosGlobalConfig = None
qosHwStatus = None
qosTmMapping = None
subIntfHwStatus = None

# -----------------------------------------------------------------------------------
# The "show [mls] qos interface [ INTF ] [ detail ]" command, in "enable" mode.
# For now this command is in global config mode showing for all interfaces
# The detail token is available only when the interface-name is fabric.
# -----------------------------------------------------------------------------------
def ShowQosIntfServicePolicy( intf, emapType ):
   intfServicePolicyQosModel = IntfServicePolicyQosModel()
   pmapsFound = findServicePolicyForIntf( intf, emapType )
   for pmapName, direction in pmapsFound:
      policyMapsContainer = PMapModelContainer(
         qosConfig, qosAclConfig, qosStatus, qosHwStatus, qosSliceHwStatus,
         qosAclHwStatus, qosAclSliceHwStatus, QosLib.qosMapType, direction,
         hwEpochStatus, None )
      policyMapsContainer.populatePolicyMapQosInterface(
            pmapName, intfServicePolicyQosModel )
   return intfServicePolicyQosModel

def showInterfacesQos( mode, args ):
   detail = 'detail' in args

   # Handle the fabric show command first
   if 'fabric' in args:
      intfAllQosAll = FabricAllQosAllModel(
         _mixedNumTxqPortsInSystem=bool( qosTmMapping.portTmMap ) )

      intf = FabricIntfCli.FabricIntf( fabricIntfName, mode )
      for sliceHwStatus in sorted( qosSliceHwStatus.values() ):
         intfQosAll = IntfQosAllModel()
         intfQosModel = showQosInterface( intf, sliceHwStatus.name, detail )
         intfServicePolicyQosModel = ShowQosIntfServicePolicy( intf, qosMapType )

         intfQosAll.intfQosModel =  intfQosModel
         intfQosAll.intfServicePolicyQosModel = intfServicePolicyQosModel
         if detail:
            intfAllQosAll.insert( sliceHwStatus.name, intfQosAll )
         else:
            intfAllQosAll.insert( fabricTokenName, intfQosAll )
            break
      return  intfAllQosAll

   intfAllQosAll = IntfAllQosAllModel(
         _mixedNumTxqPortsInSystem=bool( qosTmMapping.portTmMap ) )

   intf = args.get( 'INTF' )
   intfType = ( EthIntfCli.EthIntf, SubIntfCli.SubIntf )
   intfs = IntfCli.Intf.getAll( mode, intf, None, intfType=intfType )
   if not intfs:
      return intfAllQosAll

   for intf in intfs:
      intfQosAll = IntfQosAllModel()
      if not intf.name.startswith( ethOrLagIntfPrefixes ):
         continue
      intfQosModel = showQosInterface( intf, mode=mode )
      intfServicePolicyQosModel = ShowQosIntfServicePolicy( intf, qosMapType )

      intfQosAll.intfQosModel =  intfQosModel
      intfQosAll.intfServicePolicyQosModel = intfServicePolicyQosModel
      intfAllQosAll.insert( intf.name, intfQosAll )

   return intfAllQosAll

# -----------------------------------------------------------------------------------
# The "show [ mls ] qos maps" command, in "enable" mode.
# -----------------------------------------------------------------------------------
def showQosMaps( mode, args ):
   globalQosMaps = GlobalQosMapsModel()

   globalQosMaps.numTcSupported = qosHwStatus.numTcSupported
   globalQosMaps.numTxQueueSupported = qosHwStatus.numTxQueueSupported
   globalQosMaps.numMulticastQueueSupported = qosHwStatus.numMulticastQueueSupported

   globalQosMaps.cosRewriteSupported = qosHwStatus.cosRewriteSupported
   globalQosMaps.cosRewriteDisableSupported = qosHwStatus.cosRewriteDisableSupported
   globalQosMaps.dscpRewriteSupported = qosHwStatus.dscpRewriteSupported
   globalQosMaps.tcToDscpMapSupported = qosHwStatus.tcToDscpMapSupported
   globalQosMaps.expRewriteSupported = qosHwStatus.expRewriteSupported
   globalQosMaps.dscpToDpMapSupported = qosHwStatus.dscpToDpMapSupported
   globalQosMaps.dynamicExpMappingSupported = qosHwStatus.dynamicExpMappingSupported

   globalQosMaps.cosRewriteEnabled = qosStatus.cosRewriteEnabled
   globalQosMaps.dscpRewriteEnabled = qosStatus.dscpRewriteEnabled
   globalQosMaps.tcToPriorityGroupMapSupported = \
      qosHwStatus.tcToPriorityGroupMapSupported
   globalQosMaps.cosToTcProfilePerInterfaceSupported = \
      qosHwStatus.cosToTcPerInterfaceSupported
   globalQosMaps.expToTcProfilePerInterfaceSupported = \
      qosHwStatus.expToTcPerInterfaceSupported
   globalQosMaps.tcToCosMapPerInterfaceSupported = \
      ( qosHwStatus.tcToCosPerInterfaceSupported or
        qosHwStatus.cpuTcToCosPerInterfaceSupported or
        qosHwStatus.cpuTcToCosPerSubInterfaceSupported or
        qosHwStatus.cpuTcToCosPerLagSupported )
   globalQosMaps.dscpRewritePerInterfaceSupported = \
      qosHwStatus.dscpRewritePerInterfaceSupported
   globalQosMaps.pwDecapDscpQosMapSupported = qosHwStatus.pwDecapDscpQosMapSupported
   globalQosMaps.dscpToTcMapPerInterfaceSupported = \
      qosHwStatus.dscpToTcPerInterfaceSupported
   globalQosMaps.dscpToTcMapPerSubInterfaceSupported = \
      qosHwStatus.dscpToTcPerSubInterfaceSupported
   globalQosMaps.deiToDpMapSupported = qosHwStatus.deiPreserveSupported

   def getArrayValues( low, high, array, invalid, defaultArray, defaultVal=None ):
      values = {}
      for index in range( low, high + 1 ):
         if array[ index ] == invalid:
            value = defaultVal if defaultVal is not None else defaultArray[ index ]
         else:
            value = array[ index ]
         if value == invalid:
            value = INVALID
         values[ index ] = value
      return values

   globalQosMaps.cosToTcMap = getArrayValues( 0, 7, qosStatus.cosToTcMap,
           tacTrafficClass.invalid, qosHwStatus.defaultCosToTcMap )

   globalQosMaps.dscpToTcMap = getArrayValues( 0, 63, qosStatus.dscpToTcMap,
           tacTrafficClass.invalid, qosHwStatus.defaultDscpToTcMap )

   if qosHwStatus.deiPreserveSupported:
      globalQosMaps.deiToDpMap = getArrayValues( 0, 1, qosStatus.deiToDpMap,
        tacDropPrecedence.invalid, qosHwStatus.defaultDeiToDpMap )
   if qosHwStatus.dscpToDpMapSupported:
      globalQosMaps.dscpToDpMap = getArrayValues( 0, 63, qosStatus.dscpToDpMap,
        tacDropPrecedence.invalid, [], defaultVal=qosHwStatus.defaultDropPrecedence )
   if qosHwStatus.dynamicExpMappingSupported:
      globalQosMaps.expToTcMap = getArrayValues( 0, 7, qosStatus.expToTcMap,
            tacTrafficClass.invalid, qosHwStatus.defaultExpToTcMap )

   if qosHwStatus.cosRewriteSupported:
      globalQosMaps.tcToCosMap = getArrayValues( 0, qosHwStatus.numTcSupported - 1,
        qosStatus.tcToCosMap, tacCos.invalid, qosHwStatus.defaultTcToCosMap )

   if qosHwStatus.tcToDscpMapSupported:
      globalQosMaps.tcToDscpMap = getArrayValues( 0, qosHwStatus.numTcSupported - 1,
        qosStatus.tcToDscpMap, tacDscp.invalid, qosHwStatus.defaultTcToDscpMap )

   if qosHwStatus.dynamicExpMappingSupported:
      globalQosMaps.tcToExpMap = getArrayValues( 0, qosHwStatus.numTcSupported - 1,
         qosStatus.tcToExpMap, tacExp.invalid, qosHwStatus.defaultTcToExpMap )

   if qosHwStatus.numTxQueueSupported:
      globalQosMaps.tcToTxQueueMap = getArrayValues( 0,
        qosHwStatus.numTcSupported - 1, qosStatus.tcToTxQueueMap,
        tacTxQueueId.invalid, qosHwStatus.defaultTcToTxQueueMap )

      if qosHwStatus.numMulticastQueueSupported:
         globalQosMaps.tcToMcTxQueueMap = getArrayValues( 0,
           qosHwStatus.numTcSupported - 1, qosStatus.tcToMcTxQueueMap,
           tacTxQueueId.invalid, qosHwStatus.defaultTcToMcTxQueueMap )

   if qosHwStatus.tcToPriorityGroupMapSupported:
      globalQosMaps.tcToPriorityGroupMap = getArrayValues( 0,
         qosHwStatus.numTcSupported - 1, qosStatus.tcToPriorityGroupMap,
         tacPriorityGroup.invalid, qosHwStatus.defaultTcToPriorityGroupMap )

   if qosHwStatus.cosToTcPerInterfaceSupported:
      cosToTcProfileAllModel = CosToTcProfileAllModel()
      for profileName in qosStatus.cosToTcProfile:
         cosToTcProfileModel = CosToTcProfileModel()
         cosToTcMap = qosStatus.cosToTcProfile[ profileName ].cosToTcMap
         for cos in cosToTcMap:
            cosToTcProfileModel.cosToTcMap[ cos ] = cosToTcMap[ cos ]
         cosToTcProfileAllModel.cosToTcProfiles[ profileName ] = \
               cosToTcProfileModel
      globalQosMaps.cosToTcProfileAllModel = cosToTcProfileAllModel

   if qosHwStatus.expToTcPerInterfaceSupported:
      expToTcProfileAllModel = ExpToTcProfileAllModel()
      for profileName in qosStatus.expToTcProfile:
         expToTcProfileModel = ExpToTcProfileModel()
         expToTcMap = qosStatus.expToTcProfile[ profileName ].expToTcMap
         for exp in expToTcMap:
            expToTcProfileModel.expToTcMap[ exp ] = expToTcMap[ exp ]
         expToTcProfileAllModel.expToTcProfiles[ profileName ] = \
               expToTcProfileModel
      globalQosMaps.expToTcProfileAllModel = expToTcProfileAllModel

   if ( qosHwStatus.tcToCosPerInterfaceSupported or
        qosHwStatus.cpuTcToCosPerInterfaceSupported or
        qosHwStatus.cpuTcToCosPerSubInterfaceSupported or
        qosHwStatus.cpuTcToCosPerLagSupported ):
      tcToCosMapAllModel = TcToCosMapAllModel()
      for mapName in qosStatus.tcToCosNamedMap:
         tcToCosMapModel = TcToCosMapModel()
         tcToCosNamedMap = qosStatus.tcToCosNamedMap[ mapName ].tcToCosNamedMap
         for tcDpPair in tcToCosNamedMap:
            tcDpIndex = f'tc-{tcDpPair.tc} dp-{tcDpPair.dp}'
            tcToCosMapModel.tcToCosNamedMap[ tcDpIndex ] = \
                                                          tcToCosNamedMap[ tcDpPair ]
         tcToCosMapAllModel.tcToCosNamedMaps[ mapName ] = tcToCosMapModel
      globalQosMaps.tcToCosMapAllModel = tcToCosMapAllModel

   if qosHwStatus.pwDecapDscpQosMapSupported or \
      qosHwStatus.dscpToTcPerInterfaceSupported or \
      qosHwStatus.dscpToTcPerSubInterfaceSupported:
      dscpToTcMapAllModel = DscpToTcMapAllModel()
      for mapName in qosStatus.dscpToTcNamedMap:
         dscpToTcMapModel = DscpToTcMapModel()
         dscpToTcNamedMap = qosStatus.dscpToTcNamedMap[ mapName ].dscpToTcNamedMap
         for dscp in dscpToTcNamedMap:
            dscpToTcMapModel.dscpToTcNamedMap[ dscp ] = dscpToTcNamedMap[ dscp ]

         dscpToTcMapAllModel.dscpToTcNamedMaps[ mapName ] = dscpToTcMapModel
      globalQosMaps.dscpToTcMapAllModel = dscpToTcMapAllModel

   if qosHwStatus.dscpRewritePerInterfaceSupported:
      dscpRewriteMapAllModel = DscpRewriteMapAllModel()
      for mapName in qosStatus.dscpRewriteMap:
         dscpRewriteMapModel = DscpRewriteMapModel()
         tcToDscpMap = qosStatus.dscpRewriteMap[ mapName ].tcToDscpMap

         for tc in range( qosHwStatus.numTcSupported ):
            tcDp = Tac.ValueConst( 'Qos::TcDpPair', tc, tacDropPrecedence.min )
            dscpRewriteMapModel.tcToDscpMap[ tc ] = \
                  tcToDscpMap.get( tcDp, tacDscp.invalid )

         if qosHwStatus.dscpRewriteDropPrecedenceSupported:
            for tc in range( qosHwStatus.numTcSupported ):
               for dp in QosLib.tcDpToDscpSupportedDpValues( qosHwStatus ):
                  tcDp = Tac.ValueConst( 'Qos::TcDpPair', tc, dp )
                  key = f"tc-{tc} dp-{dp}"
                  dscpRewriteMapModel.tcDpToDscpMap[ key ] = \
                     tcToDscpMap.get( tcDp, tacDscp.invalid )

         dscpRewriteMapAllModel.dscpRewriteMaps[ mapName ] = dscpRewriteMapModel
      globalQosMaps.dscpRewriteMapAllModel = dscpRewriteMapAllModel

   if qosHwStatus.expRewriteSupported:
      tcDpToExpMapAllModel = TcDpToExpMapAllModel()
      for mapName in qosStatus.tcDpToExpNamedMap:
         tcDpToExpMapModel = TcDpToExpMapModel()
         tcDpToExpMap = qosStatus.tcDpToExpNamedMap[ mapName ].tcDpToExpMap
         for tcDpPair, exp in tcDpToExpMap.items():
            if exp != tacExp.invalid:
               key = f"tc-{tcDpPair.tc} dp-{tcDpPair.dp}"
               tcDpToExpMapModel.tcDpToExpMap[ key ] = exp
         tcDpToExpMapAllModel.tcDpToExpMaps[ mapName ] = tcDpToExpMapModel
      globalQosMaps.tcDpToExpMapAllModel = tcDpToExpMapAllModel

   return globalQosMaps

# -----------------------------------------------------------------------------------
# Handler of QosProfileModeShowCmd
# -----------------------------------------------------------------------------------
def qosProfileModeShowCmdHandler( mode, args ):
   if 'diff' in args:
      showQosProfileDiff( mode )
   elif 'active' in args:
      showQosProfileActive( mode )
   else:
      showQosProfilePending( mode )

# -----------------------------------------------------------------------------------
# Called by qosProfileModeShowCmdHandler on condition
# -----------------------------------------------------------------------------------
def showQosProfileDiff( mode ):
   activeOutput1 = io.StringIO()
   currentQosProfileConfig = mode.qosProfileModeContext.profile_
   pendingQosProfileConfig = mode.qosProfileModeContext.currentEntry_
   _convertQosProfileToPrintable(
      currentQosProfileConfig,
      output=activeOutput1 )
   activeOutput2 = io.StringIO()
   _convertQosProfileToPrintable(
      pendingQosProfileConfig,
      output=activeOutput2 )
   diff = difflib.unified_diff( activeOutput1.getvalue().splitlines( ),
                                activeOutput2.getvalue().splitlines( ),
                                lineterm='' )
   sys.stdout.write( '\n'.join( list( diff ) ) )
   sys.stdout.write( '\n' )

# -----------------------------------------------------------------------------------
# Called by qosProfileModeShowCmdHandler on condition
# -----------------------------------------------------------------------------------
def showQosProfileActive( mode ):
   mode.showActive()

# -----------------------------------------------------------------------------------
# Called by qosProfileModeShowCmdHandler on condition
# -----------------------------------------------------------------------------------
def showQosProfilePending( mode ):
   profile = mode.qosProfileModeContext.currentEntry_
   _convertQosProfileToPrintable( profile )

def _convertQosProfileToPrintable( currentQosProfile, output=None ):
   if output is None:
      output = sys.stdout
   if currentQosProfile is None:
      return
   output.write( f"qos profile {currentQosProfile.name} \n" )
   if currentQosProfile.trustMode != tacTrustMode.invalid:
      if currentQosProfile.trustMode != tacTrustMode.untrusted:
         output.write( f"   qos trust {currentQosProfile.trustMode}\n" )
      else:
         output.write( "   no qos trust\n" )
   if ( currentQosProfile.defaultCos not in ( qosHwStatus.defaultCos,
                                             tacCos.invalid ) ):
      output.write( f"   qos cos {currentQosProfile.defaultCos}\n" )
   if ( currentQosProfile.defaultDscp not in ( qosHwStatus.defaultDscp,
                                              tacDscp.invalid ) ):
      output.write( f"   qos dscp {currentQosProfile.defaultDscp}\n" )
   schedulerCompensation = currentQosProfile.schedulerCompensation
   if schedulerCompensation != tacSchedulerCompensation.invalid:
      sign = "plus"
      if schedulerCompensation < 0:
         sign = "minus"
         schedulerCompensation = -schedulerCompensation
      output.write( "   tx-scheduler packet size " +
                   f"adjustment {sign} {schedulerCompensation} bytes" )
      output.write( '\n' )
   if currentQosProfile.dscpPreserveIpMplsEncapMode:
      output.write( "   qos rewrite dscp mpls encapsulation disabled\n" )
   shape = currentQosProfile.shapeRate
   if shape and shape.rate != tacShapeRateVal.invalid:
      output.write( f"   shape rate {shape.rate}" )
      if shape.percent != tacPercent.invalid:
         output.write( ' percent' )
      elif QosLib.shapeRateUnitFromEnum( shape.unit ) != 'kbps':
         output.write( f' {QosLib.shapeRateUnitFromEnum( shape.unit )}' )
      if shape.shared:
         output.write( ' shared' )
      output.write( '\n' )
   if subIntfHwStatus.subIntfBandwidthSupported:
      gbw = currentQosProfile.guaranteedBw
      if gbw and gbw.bw != tacGuaranteedBwVal.invalid:
         if gbw.unit == tacGuaranteedBwUnit.guaranteedBwPercent:
            output.write( f'   bandwidth guaranteed percent {gbw.bw}\n' )
         elif gbw.unit == tacGuaranteedBwUnit.guaranteedBwPps:
            output.write( f'   bandwidth guaranteed {gbw.bw} pps\n' )
         else:
            output.write( f'   bandwidth guaranteed {gbw.bw}\n' )
   pfcPortConfig = currentQosProfile.pfcPortConfig
   if pfcPortConfig:
      if pfcPortConfig.enabled:
         output.write( "   priority-flow-control on\n" )
      if not pfcPortConfig.watchdogEnabled:
         output.write( "   no priority-flow-control pause watchdog\n" )
      if pfcPortConfig.priorities:
         _pfcPortConfigPrioritiesPrintable( pfcPortConfig,
            currentQosProfile.fabricPfcDlb, output )
      if pfcPortConfig.portTimerConfig.usePerPortTimerValues:
         _convertPortTimerConfigPrintable( pfcPortConfig, output )
      if pfcPortConfig.watchdogPortAction != tacPfcWatchdogAction.invalid:
         str_value = pfcPortConfig.watchdogPortAction
         to_write = "   priority-flow-control pause "\
            f"watchdog port action {str_value}\n"
         output.write( to_write )
   servicePolicyConfig = currentQosProfile.servicePolicyConfig
   if servicePolicyConfig:
      output.write( "   service-policy type qos " +
                   f"{servicePolicyConfig.key.direction} " +
                   f"{servicePolicyConfig.key.pmapName}\n" )

   if qosHwStatus.numTxQueueSupported:
      for qoshwtxqid in range( qosHwStatus.numTxQueueSupported ):
         if qosHwStatus.hwTxQueue.get( qoshwtxqid ) is None:
            continue
         txQueue = qosHwStatus.hwTxQueue[ qoshwtxqid ].txQueue
         txQueueRenderedOutput = io.StringIO()

         txQueueConfig = currentQosProfile.txQueueConfig.get( txQueue )
         ecnTxQueueCounterCfg = currentQosProfile.ecnTxQueueCounterConfig. \
               get( txQueue.id )
         _txQueueRender( txQueueConfig, ecnTxQueueCounterCfg,
               txQueueRenderedOutput )

         if len( txQueueRenderedOutput.getvalue() ) == 0:
            continue
         output.write( "   " )
         if txQueue.type == tacQueueType.mcq:
            output.write( "mc-tx-queue " )
         elif txQueue.type == tacQueueType.ucq:
            output.write( "uc-tx-queue " )
         elif txQueue.type == tacQueueType.unknown:
            output.write( "tx-queue " )
         output.write( f"{txQueue.id}\n" )
         output.write( txQueueRenderedOutput.getvalue() )

# -----------------------------------------------------------------------------------
# Helper method called by _convertQosProfileToPrintable
# -----------------------------------------------------------------------------------
def _pfcPortConfigPrioritiesPrintable( pfcPortConfig, fabricPfcDlb, output=None ):
   if not output:
      output = sys.stdout
   noDropList = pfcPortConfig.priorities
   dlbEnabledList = fabricPfcDlb
   for noDropId in range( 16 ):
      bit = 1 << noDropId
      if bit & dlbEnabledList:
         output.write( "   priority-flow-control priority " +
                      f"{noDropId} no-drop dlb\n" )
      elif bit & noDropList:
         output.write( f"   priority-flow-control priority {noDropId} no-drop\n" )

# -----------------------------------------------------------------------------------
# Helper method called by _convertQosProfileToPrintable
# -----------------------------------------------------------------------------------
def _convertPortTimerConfigPrintable( pfcPortConfig, output=None ):
   if not output:
      output = sys.stdout
   portTimerConfig = pfcPortConfig.portTimerConfig
   cmdString = 'priority-flow-control pause watchdog port timer'
   timeoutString = f' timeout {portTimerConfig.portWatchdogTimeout:2.2f}'
   cmdString = cmdString + timeoutString
   pollingIntervalValue = "auto" if not \
                          portTimerConfig.portWatchdogPollingInterval else \
                          f"{portTimerConfig.portWatchdogPollingInterval:.3f}"
   pollingIntervalString = f" polling-interval {pollingIntervalValue}"
   cmdString = cmdString + pollingIntervalString
   recoveryCfg = portTimerConfig.portWatchdogRecoveryCfg
   recoveryTimeValue = "0" if not recoveryCfg.recoveryTime else \
                       f"{recoveryCfg.recoveryTime:2.2f}"
   if recoveryCfg.forcedRecovery:
      recoveryTimeValue = recoveryTimeValue + " forced"
   recoveryTimeString = f" recovery-time {recoveryTimeValue}"
   cmdString = cmdString + recoveryTimeString
   output.write( f"   {cmdString}\n" )

# -----------------------------------------------------------------------------------
# Helper method called by _convertQosProfileToPrintable
# -----------------------------------------------------------------------------------
def _txQueueRender( txQueueConfig, ecnTxQueueCounterConfig, output=None ):
   if not output:
      output = sys.stdout
   if txQueueConfig is None:
      if qosHwStatus.ecnCountersSupported and ecnTxQueueCounterConfig:
         output.write( "      random-detect ecn count\n" )
      return
   if txQueueConfig.priority == tacTxQueuePriority.priorityRoundRobin:
      output.write( "      no priority\n" )
   if txQueueConfig.bandwidth != tacPercent.invalid:
      output.write( f"      bandwidth percent {txQueueConfig.bandwidth}\n" )
   if ( txQueueConfig.guaranteedBw and
        txQueueConfig.guaranteedBw.bw != tacGuaranteedBwVal.invalid ):
      output.write( "      bandwidth guaranteed " )
      guaranteedBwUnit = QosLib.guaranteedBwUnitFromEnum(
         txQueueConfig.guaranteedBw.unit )
      if guaranteedBwUnit.lower() == 'percent':
         output.write( f"{guaranteedBwUnit} {txQueueConfig.guaranteedBw.bw}\n" )
      else:
         if guaranteedBwUnit != 'kbps':
            output.write( f"{txQueueConfig.guaranteedBw.bw} {guaranteedBwUnit}\n" )
         else:
            output.write( f"{txQueueConfig.guaranteedBw.bw}\n" )
   if ( txQueueConfig.shapeRate and
        txQueueConfig.shapeRate.rate != tacShapeRateVal.invalid ):
      if txQueueConfig.shapeRate.percent != tacPercent.invalid:
         shapeRateUnit = 'percent'
      else:
         shapeRateUnit = QosLib.shapeRateUnitFromEnum( txQueueConfig.shapeRate.unit )

      if shapeRateUnit == 'kbps':
         shapeRateUnit = ''
      else:
         shapeRateUnit = ' ' + shapeRateUnit
      output.write( "      shape rate " +
                   f"{txQueueConfig.shapeRate.rate}{shapeRateUnit}\n" )

   if qosHwStatus.ecnSupported and txQueueConfig.ecnConfig:
      str_value = _packetDropOrEcnConfigToPrintable( 'ecn', txQueueConfig.ecnConfig )
      output.write( f"      random-detect ecn {str_value}\n" )

   if qosHwStatus.ecnDelayPerTxQSupported and txQueueConfig.ecnDelayConfig:
      output.write( "      random-detect ecn delay threshold " +
                    f"{ txQueueConfig.ecnDelayConfig.maxThd} " +
                    f"{ txQueueConfig.ecnDelayConfig.unit}\n" )

   if qosHwStatus.ecnCountersSupported and ecnTxQueueCounterConfig:
      output.write( "      random-detect ecn count\n" )

   if qosHwStatus.nonEctThdSupported and txQueueConfig.nonEctConfig:
      str_value = _packetDropOrEcnConfigToPrintable(
         "nonEct", txQueueConfig.nonEctConfig )
      output.write( f"      random-detect non-ect {str_value}\n" )

   if ( txQueueConfig.delayEcnEnabled and
        txQueueConfig.ecnDelayThreshold is not None ):
      thresholdConfig = txQueueConfig.ecnDelayThreshold
      minThresh = thresholdConfig.minEcnDelayThreshold().threshold
      if thresholdConfig.unit == 'ecnDelayThresholdInvalidUnit':
         output.write( '      ecn delay\n' )
      elif thresholdConfig.threshold >= minThresh:
         maxThresh = thresholdConfig.maxEcnDelayThreshold().threshold
         if thresholdConfig.threshold <= maxThresh:
            # changing the value to microseconds
            val = thresholdConfig.threshold / 1000
            output.write( f"      ecn delay threshold {int(val)} microseconds\n" )

   if txQueueConfig.weightConfig != defaultWeight:
      output.write( f"      queue length weight {txQueueConfig.weightConfig}\n" )

   if qosHwStatus.wredSupported and txQueueConfig.wredConfig:
      str_value = _packetDropOrEcnConfigToPrintable( "wred",
                                                    txQueueConfig.wredConfig )
      output.write( f"      random-detect drop {str_value}\n" )

   if qosHwStatus.dpWredSupported and txQueueConfig.dpConfig is not None:
      for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
         if dp in txQueueConfig.dpConfig.dpWredConfig:
            dpWredParam = txQueueConfig.dpConfig.dpWredConfig[ dp ]
            str1 = f"      random-detect drop drop-precedence {dp} "
            str2 = f"minimum-threshold {dpWredParam.minThd} {dpWredParam.unit} "
            str3 = f"maximum-threshold {dpWredParam.maxThd} {dpWredParam.unit} "
            str4 = f"drop-probability {dpWredParam.maxDroprate}"
            dpWredPrintable = str1 + str2 + str3 + str4
            if qosHwStatus.wredWeightSupported and \
                  dpWredParam.weight != defaultWeight:
               dpWredPrintable += ' weight ' + str( dpWredParam.weight )
            output.write( dpWredPrintable + '\n' )

   if qosHwStatus.dropPrecedenceThresholdSupported and \
      len( txQueueConfig.dropThresholds ) != 0:
      for dp in txQueueConfig.dropThresholds:
         output.write( f"      drop-precedence {dp} drop-threshold percent " +
                      f"{txQueueConfig.dropThresholds[ dp ]}\n" )

   if qosHwStatus.latencyThresholdSupported and \
      txQueueConfig.latencyThreshold != tacLatencyThreshold():
      output.write( "      latency maximum " +
                   f"{txQueueConfig.latencyThreshold.configThreshold()} " +
                   f"{txQueueConfig.latencyThreshold.configUnit}\n" )

# -----------------------------------------------------------------------------------
# Helper method called by _txQueueRender
# -----------------------------------------------------------------------------------
def _packetDropOrEcnConfigToPrintable( configurationType, dropOrEcnConfig ):
   unit = dropOrEcnConfig.unit
   str1 = f'minimum-threshold {dropOrEcnConfig.minThd} {unit} '
   str2 = f'maximum-threshold {dropOrEcnConfig.maxThd} {unit}'
   dropOrEcnConfigPrintable = str1 + str2
   if qosHwStatus.ecnMarkProbSupported:
      if configurationType == "ecn":
         dropOrEcnConfigPrintable += ( " max-mark-probability " +
                                      f"{dropOrEcnConfig.maxDroprate}" )
   if configurationType == "wred":
      dropOrEcnConfigPrintable += ( " drop-probability " +
                                   f"{dropOrEcnConfig.maxDroprate}" )
   if( ( configurationType == "ecn" and qosHwStatus.ecnWeightSupported ) or
       ( configurationType == "nonEct" and qosHwStatus.nonEctWeightSupported ) or
       ( configurationType == "wred" and qosHwStatus.wredWeightSupported ) ):
      if dropOrEcnConfig.weight != defaultWeight:
         dropOrEcnConfigPrintable += ' weight ' + str( dropOrEcnConfig.weight )
   return dropOrEcnConfigPrintable

# -----------------------------------------------------------------------------------
# Handler for
# -----------------------------------------------------------------------------------
def showTxQueueShapeRateAdaptive( mode, args ):
   txQueueShapeRateAdaptiveModel = TxQueueShapeRateAdaptiveModel()
   txQueueShapeRateAdaptiveModel.configuredMode = qosConfig.txQueueShapeRateAdaptive
   txQueueShapeRateAdaptiveModel.operationalMode = qosStatus.txQueueShapeRateAdaptive
   return txQueueShapeRateAdaptiveModel

#------------------------------------------------------------------------------------
# Handler for ShowInterfaceSchedulerProfile
#------------------------------------------------------------------------------------
def getTxQueueSchProfiles( intfName ):
   sandQos0SliceHwStatus = qosSliceHwStatus[ SAND_QOS_SLICE_NAME ]
   sq0shs = sandQos0SliceHwStatus
   numTxQueueIds = qosHwStatus.numTxQueueSupported

   txqSchMap = {}
   if intfName in sq0shs.perPortTxQueueToSchedulerProfile:
      perPortMap = sq0shs.perPortTxQueueToSchedulerProfile[ intfName ].mapping
      for txQueueId in range( numTxQueueIds ):
         txqSchMap[ txQueueId ] = perPortMap[ txQueueId ]
   return txqSchMap

def showIntfSchedulerProfile( intfName ):
   txQueueSchedulerMapModel = TxQueueSchedulerMapModel()
   txQueueSchedulerMapModel.txQueues = getTxQueueSchProfiles( intfName )
   txQueueSchedulerMapModel.memberIntfs = None
   return txQueueSchedulerMapModel

def showLagIntfSchedulerProfile( lagName, memberIntfs ):
   txQueueSchedulerMapModel = TxQueueSchedulerMapModel()

   # recap current config
   membersCfg = defaultdict( list )
   for mIntf in memberIntfs:
      txqSchMap = getTxQueueSchProfiles( mIntf )
      txqSchTup = tuple( txqSchMap.items() )
      membersCfg[ txqSchTup ].append( mIntf )

   # filter to determine main config
   sortCfg = sorted( membersCfg.items(),
                     key=lambda el: len( el[ 1 ] ),
                     reverse=True )
   if sortCfg:
      mainTxqSchTup, mainIntfs = sortCfg[ 0 ]
      remaining = sortCfg[ 1: ]
      txQueueSchedulerMapModel.txQueues = dict( mainTxqSchTup )

      txQueueSchedulerMapModel.memberIntfs = {}
      for _, remIntfs in remaining:
         for remIntf in remIntfs:
            txQueueSchedulerMapModel.memberIntfs[ remIntf ] = False
      for mainIntf in mainIntfs:
         # Interfaces configured with the main configuration are considered as
         # correctly configured
         txQueueSchedulerMapModel.memberIntfs[ mainIntf ] = True

   return txQueueSchedulerMapModel

def showInterfaceSchedulerProfile( mode, args ):
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTF' ), None, EthIntfCli.EthIntf )

   schedulerProfileRecapModel = SchedulerProfileRecapModel()
   if SAND_QOS_SLICE_NAME not in qosSliceHwStatus:
      return schedulerProfileRecapModel

   lagToIntfs = defaultdict( list )
   for phyIntf, phyIntfVal in lagInputConfig.phyIntf.items():
      if phyIntfVal.lag:
         lagName = phyIntfVal.lag.intfId
         lagToIntfs[ lagName ].append( phyIntf )

   intfInLags = set()

   for intf in intfs:
      intfName = intf.name
      if isLagPort( intfName ):
         memberIntfs = lagToIntfs.get( intfName, [] )
         txQueueSchedulerMapModel = \
                           showLagIntfSchedulerProfile( intfName, memberIntfs )
         intfInLags.update( memberIntfs )
      else:
         txQueueSchedulerMapModel = showIntfSchedulerProfile( intfName )
      if not txQueueSchedulerMapModel.txQueues:
         # If configuration is not found, there is no need to add empty data
         continue
      # Set TxQueueSchedulerMapModel to SchedulerProfileRecapModel
      schedulerProfileRecapModel.interfaces[ intfName ] = txQueueSchedulerMapModel

   # Avoid duplicating data in the output
   for intfName in intfInLags:
      schedulerProfileRecapModel.interfaces.pop( intfName, None )

   return schedulerProfileRecapModel

#------------------------------------------------------------------------------------
# Handler of QosLagSubIntfLoadbalancingCmd
# -----------------------------------------------------------------------------------
def showLagSubIntfLoadbalancing( mode, args ):
   lagSubIntfLoadbalancing = LagSubIntfLoadbalancingModel()
   isConfig = qosGlobalConfig.lagSubIntfLoadbalance
   isOper = False
   for sliceHwStatus in qosSliceHwStatus.values():
      if sliceHwStatus.shapedLagSubIntfLoadbalance:
         isOper = sliceHwStatus.shapedLagSubIntfLoadbalance
         break

   lagSubIntfLoadbalancing.configuredLagSubIntfLoadBalancing = isConfig
   lagSubIntfLoadbalancing.operationalLagSubIntfLoadBalancing = isOper

   return lagSubIntfLoadbalancing

def startQosAclConfigMergeSm():
   global qosAclConfig, qosAclConfigMergeSm
   qosAclConfig = Tac.newInstance( "Qos::QosAclConfig" )
   qosAclConfigMergeSm = Tac.newInstance( "Qos::Agent::QosAclConfigMergeSm",
                                          qosAclConfigDir,
                                          cliQosAclConfigReadOnly,
                                          qosAclConfig )

def Plugin( entityManager ):
   global qosConfig, subIntfHwStatus, qosAclHwStatus, qosAclSliceHwStatus, \
      hwEpochStatus, qosAclConfigDir, cliQosAclConfigReadOnly, qosStatus, \
      qosSliceHwStatus, qosGlobalConfig, qosHwStatus, qosTmMapping, lagInputConfig
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosAclSliceHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )
   qosTmMapping = LazyMount.mount( entityManager, "hardware/tm/mapping",
                                  "Qos::TmMapping", "r" )
   hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
   mg = entityManager.mountGroup()
   qosAclConfigDir = mg.mount( "qos/acl/input/eosSdkConfig", "Tac::Dir", "ri" )
   cliQosAclConfigReadOnly = mg.mount( "qos/acl/input/cli",
                                       "Qos::Input::AclConfig", "wi" )
   mg.close( startQosAclConfigMergeSm )
