# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import List
from CliModel import Bool
from CliModel import Float
from CliModel import Submodel
from IntfModels import Interface
import Tac
import Ark

class ServerInfo( Model ):
   hostname = Str( help="IP address or hostname of the RADIUS server" )
   authport = Int( help="Specifies server authorization port number" )
   acctport = Int( help="Specifies server accounting port number", optional=True )
   dot1xDynAuthEnabled = Bool( help="Specifies if dynamic authorization is enabled",
                            optional=True )
   dynAuthPort = Int( help="Specifies server dynamic authorization port number",
                      optional=True )
   vrf = Str( help="The VRF over which this server is reachable", optional=True )
   tlsport = Int( help="RADIUS server TLS port number", optional=True )
   sslProfile = Str( help="SSL profile for RADIUS server", optional=True )

class RadiusStats( Model ):
   serverInfo = Submodel( valueType=ServerInfo, help="Configured server "
                                                     "information" )
   messagesSent = Int( help="Number of packets sent to the RADIUS server" )
   messagesReceived = Int( help="Number of packets received from the RADIUS "
                                "server" )
   requestsAccepted = Int( help="Number of requests accepted by the RADIUS server" )
   requestsRejected = Int( help="Number of requests rejected by the RADIUS server" )
   requestsTimeout = Int( help="Number of requests to the RADIUS server timed out" )
   requestsRetransmitted = Int( help="Number of requests retransmitted by the "
                                     "RADIUS server" )
   badResponses = Int( help="Number of bad responses from the RADIUS server" )
   connectionErrors = Int( help="Number of connection errors to the RADIUS server" )
   dnsErrors = Int( help="Number of DNS errors by the RADIUS server" )
   coaRequestsReceived = Int( help="Number of CoA requests received from "
                                   "RADIUS server" )
   dmRequestsReceived = Int( help="Number of DM requests received from "
                                  "RADIUS server" )
   coaAckSent = Int( help="Number of CoA Ack responses sent to RADIUS server" )
   dmAckSent = Int( help="Number of DM Ack responses sent to RADIUS server" )
   coaNakSent = Int( help="Number of CoA Nak responses sent to RADIUS server" )
   dmNakSent = Int( help="Number of DM Nak responses sent to RADIUS server" )
   acctStartsSent = Int( help="Number of Accounting Start messages sent to RADIUS "
                              "server" )
   acctInterimUpdatesSent = Int( help="Number of Accounting Interim Update messages "
                                      "sent to RADIUS server" )
   acctStopsSent = Int( help="Number of Accounting Stop messages sent to RADIUS "
                             "server" )

class ServerGroup( Model ):
   serverGroup = Str( help="Configured server-group" )
   members = List( valueType=ServerInfo, help="Mapping of server IP address or "
                                              "hostname to group information" )

class ShowRadius( Model ):
   radiusServers = List( valueType=RadiusStats, help="Mapping of server "
         "IP address or hostname to server information" )
   groups = Dict( keyType=str, valueType=ServerGroup, help="Mapping of group name "
         "to group information", optional=True )
   srcIntf = Dict( valueType=Interface, help="Configured source-interface name" )
   lastCounterClearTime = Float( help="UTC time when the counters were "
                                      "last cleared" )

   def _getVrfString( self, vrf ):
      return f" (VRF {vrf})" if vrf else ""

   def render( self ):
      template = "%26s:%11d"
      for value in self.radiusServers:
         if value.serverInfo.tlsport:
            print( f"RADIUS server             : {value.serverInfo.hostname}, "
                   f"TLS port {value.serverInfo.tlsport}"
                   f"{self._getVrfString( value.serverInfo.vrf )}" )
            print( f"SSL profile               : {value.serverInfo.sslProfile}" )
         else:
            print( f"RADIUS server             : {value.serverInfo.hostname}, "
                   f"authentication port {value.serverInfo.authport}, "
                   f"accounting port {value.serverInfo.acctport}"
                   f"{self._getVrfString( value.serverInfo.vrf )}" )
         if value.serverInfo.dot1xDynAuthEnabled:
            dynAuthPort = value.serverInfo.dynAuthPort
            print( f"Dynamic authorization UDP port: {dynAuthPort}" )
         print( template % ( "Messages sent", value.messagesSent ) )
         print( template % ( "Messages received", value.messagesReceived ) )
         print( template % ( "Requests accepted", value.requestsAccepted ) )
         print( template % ( "Requests rejected", value.requestsRejected ) )
         print( template % ( "Requests timeout", value.requestsTimeout ) )
         print( template % ( "Requests retransmitted",
                             value.requestsRetransmitted ) )
         print( template % ( "Bad responses", value.badResponses ) )
         print( template % ( "Connection errors", value.connectionErrors ) )
         print( template % ( "DNS errors", value.dnsErrors ) )
         print( template % ( "CoA requests received", value.coaRequestsReceived ) )
         print( template % ( "DM requests received", value.dmRequestsReceived ) )
         print( template % ( "CoA ACKs sent", value.coaAckSent ) )
         print( template % ( "DM ACKs sent", value.dmAckSent ) )
         print( template % ( "CoA NAKs sent", value.coaNakSent ) )
         print( template % ( "DM NAKs sent", value.dmNakSent ) )
         print( template % ( "Accounting Starts sent",
                             value.acctStartsSent ) )
         print( template % ( "Interim Updates sent",
                             value.acctInterimUpdatesSent ) )
         print( template % ( "Accounting Stops sent",
                             value.acctStopsSent ) )
         print()

      for serverGroupName, radiusGroup in self.groups.items():
         print( f"{radiusGroup.serverGroup} server-group: {serverGroupName}" )
         for sequence, groupDetails in enumerate( radiusGroup.members ):
            if groupDetails.tlsport:
               print( f"    {sequence:2d}: {groupDetails.hostname}, "
                      f"TLS port {groupDetails.tlsport}"
                      f"{self._getVrfString( groupDetails.vrf )}" )
            else:
               print( f"    {sequence:2d}: {groupDetails.hostname}, "
                      f"authentication port {groupDetails.authport}, "
                      f"accounting port {groupDetails.acctport}"
                      f"{self._getVrfString( groupDetails.vrf )}" )
         if not radiusGroup.members:
            print( "    (server-group is empty)" )
         print()

      print( "Last time counters were cleared:",
             Ark.utcTimeRelativeToNowStr( self.lastCounterClearTime ) )

      if len( self.srcIntf ) > 0:
         print( "\nRADIUS source-interface: Enabled" )
         print( "  RADIUS outgoing packets will be sourced with an IP address " \
                "associated with interface as follows:\n" )
         print( "\tVRF \tsource-interface" )
         print( "\t--- \t----------------" )
         for vrf in sorted( self.srcIntf ):
            print( f"\t{vrf}\t{self.srcIntf[ vrf ]}" )

