#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from AaaPluginLib import hostProtocol
import Ark
import CliDynamicSymbol
import ConfigMount
import HostnameCli
import LazyMount
import ReversibleSecretCli
import Tac
from CliPlugin.RadiusProxyMode import RadiusProxyClientGroupConfigMode
from CliPlugin.RadiusProxyMode import RadiusProxyConfigMode
from CliPlugin.VrfCli import DEFAULT_VRF
from Radius import MAX_KEY_SIZE
from TypeFuture import TacLazyType
from Toggles.RadiusToggleLib import toggleDot1xRadiusProxyMergeV0Enabled

radiusProxyConfig = None
radiusProxyStatus = None
DefaultPorts = TacLazyType( 'RadiusProxy::DefaultPorts' )
TimeoutConstants = TacLazyType( 'RadiusProxy::TimeoutConstants' )
RadiusProxyModel = CliDynamicSymbol.CliDynamicPlugin( "RadiusProxyModel" )

def createClient( mode, clientName, port, acctPort, vrf=DEFAULT_VRF ):
   group = radiusProxyConfig.clientGroup.get( mode.groupName )
   assert group is not None
   spec = Tac.Value( 'RadiusProxy::ClientSpec', clientName=clientName,
                     port=port, acctPort=acctPort, vrf=vrf )
   client = group.client.newMember( spec )
   return client

def radiusProxyCmdHandler( mode, args ):
   radiusProxyConfig.radiusProxyEnabled = True
   childMode = mode.childMode( RadiusProxyConfigMode )
   mode.session_.gotoChildMode( childMode )

def noRadiusProxyCmdHandler( mode, args ):
   radiusProxyConfig.radiusProxyEnabled = False
   radiusProxyConfig.dynAuthEnabled = False
   radiusProxyConfig.clientGroup.clear()
   radiusProxyConfig.key = ReversibleSecretCli.getDefaultSecret()
   radiusProxyConfig.clientMappingTimeout = TimeoutConstants.defaultTimeout

def checkKeySize( mode, key ):
   if key and key.clearTextSize() > MAX_KEY_SIZE:
      mode.addErrorAndStop( f'Maximum key size is { MAX_KEY_SIZE }' )

def radiusProxyClientKeyCmdHandler( mode, args ):
   key = args.get( 'KEY' )
   checkKeySize( mode, key )
   radiusProxyConfig.key = key

def noRadiusProxyClientKeyCmdHandler( mode, args ):
   radiusProxyConfig.key = ReversibleSecretCli.getDefaultSecret()

def clientGroupCmdHandler( mode, args ):
   name = args[ 'NAME' ]
   childMode = mode.childMode( RadiusProxyClientGroupConfigMode,
                               name=name )
   mode.session_.gotoChildMode( childMode )
   groups = radiusProxyConfig.clientGroup
   groups.newMember( name )

def noClientGroupCmdHandler( mode, args ):
   name = args[ 'NAME' ]
   groups = radiusProxyConfig.clientGroup
   del groups[ name ]

def clientToServerGroupCmdHandler( mode, args ):
   clientGroups = radiusProxyConfig.clientGroup
   cg = clientGroups.newMember( mode.groupName )
   sg = args[ 'GROUPS' ]
   # Clear old collection
   cg.serverGroup.clear()
   i = 0
   for s in sg:
      cg.serverGroup[ i ] = s
      i += 1

def noClientToServerGroupCmdHandler( mode, args ):
   clientGroups = radiusProxyConfig.clientGroup
   cg = clientGroups[ mode.groupName ]
   cg.serverGroup.clear()

def clientConfigCmdHandler( mode, args ):
   clientName = args.get( 'HOSTNAME' )
   if clientName:
      HostnameCli.resolveHostname( mode, clientName, doWarn=True )
   else:
      clientName = ( args.get( 'V4ADDR' ) or
                     args.get( 'PREFIX' ) or
                     args.get( 'V6ADDR' ) or
                     args.get( 'PREFIXV6' ) )
   if not isinstance( clientName, str ):
      clientName = clientName.stringValue
   vrf = args.get( 'VRF' )
   key = args.get( 'KEY' )
   checkKeySize( mode, key )
   client = createClient( mode, clientName, DefaultPorts.defaultListenAuthPort,
                          DefaultPorts.defaultListenAcctPort, vrf )
   if key is not None:
      client.key = key

def noClientConfigCmdHandler( mode, args ):
   clientName = args.get( 'HOSTNAME' )
   if clientName:
      HostnameCli.resolveHostname( mode, clientName, doWarn=True )
   else:
      clientName = ( args.get( 'V4ADDR' ) or
                     args.get( 'PREFIX' ) or
                     args.get( 'V6ADDR' ) or
                     args.get( 'PREFIXV6' ) )
   if not isinstance( clientName, str ):
      clientName = clientName.stringValue
   vrf = args.get( 'VRF' )
   group = radiusProxyConfig.clientGroup.get( mode.groupName )
   assert group is not None
   clients = group.client
   spec = Tac.Value( 'RadiusProxy::ClientSpec', clientName=clientName,
                     port=DefaultPorts.defaultListenAuthPort,
                     acctPort=DefaultPorts.defaultListenAcctPort,
                     vrf=vrf )
   del clients[ spec ]

def dynAuthCmdHandler( mode, args ):
   radiusProxyConfig.dynAuthEnabled = True

def noDynAuthCmdHandler( mode, args ):
   radiusProxyConfig.dynAuthEnabled = False

def clientMappingTimeoutCmdHandler( mode, args ):
   radiusProxyConfig.clientMappingTimeout = args.get( 'TIMEOUT' )

def noClientMappingTimeoutCmdHandler( mode, args ):
   radiusProxyConfig.clientMappingTimeout = TimeoutConstants.defaultTimeout

def showClientStats( spec, cgStatus ):
   clientInfo = RadiusProxyModel.ClientInfo()
   clientInfo.hostname = spec.clientName
   clientInfo.authPort = spec.port
   clientInfo.acctPort = spec.acctPort
   clientInfo.vrf = spec.vrf
   if spec in cgStatus.client:
      clientInfo.resolvedIpAddr = cgStatus.client[ spec ].clientIp

   stats = RadiusProxyModel.ClientStats()
   # Initialize the counters to zero
   stats.messagesReceived = 0
   stats.messagesSent = 0
   stats.acceptRespSent = 0
   stats.rejectRespSent = 0
   stats.acctStartsReceived = 0
   stats.acctInterimUpdatesReceived = 0
   stats.acctStopsReceived = 0
   stats.badRequests = 0
   stats.coaRequestsSent = 0
   stats.dmRequestsSent = 0
   stats.coaAcksReceived = 0
   stats.dmAcksReceived = 0
   stats.coaNaksReceived = 0
   stats.dmNaksReceived = 0
   stats.badResponses = 0
   stats.connectionErrors = 0

   counters = cgStatus.counter[ spec ]
   for ip in counters.cpci:
      c = counters.cpci[ ip ]
      stats.messagesReceived += c.authMsgsReceived
      stats.messagesSent += c.authMsgsSent
      stats.acceptRespSent += c.authAcceptsRespSent
      stats.rejectRespSent += c.authRejectsRespSent
      stats.acctStartsReceived += c.acctStartsReceived
      stats.acctInterimUpdatesReceived += c.acctInterimUpdatesReceived
      stats.acctStopsReceived += c.acctStopsReceived
      stats.badRequests += c.badRequests
      stats.coaRequestsSent += c.coaRequestsSent
      stats.dmRequestsSent += c.dmRequestsSent
      stats.coaAcksReceived += c.coaAcksReceived
      stats.dmAcksReceived += c.dmAcksReceived
      stats.coaNaksReceived += c.coaNaksReceived
      stats.dmNaksReceived += c.dmNaksReceived
      stats.badResponses += c.badResponses
      stats.connectionErrors += c.connectionErrors

   stats.clientInfo = clientInfo
   return stats

def showClientGroupInfo( cgStatus ):
   cgInfo = RadiusProxyModel.ClientGroupInfo()
   for spec in cgStatus.counter:
      cgInfo.members.append( showClientStats( spec, cgStatus ) )
   cgInfo.lastCounterClearTime = Ark.switchTimeToUtc( cgStatus.lastCounterClearTime )
   return cgInfo

def showClientGroupCmdHandler( mode, args ):
   ret = RadiusProxyModel.showClientGroupCmd()
   gName = args.get( 'GROUP' )
   if gName is not None:
      if gName not in radiusProxyStatus.clientGroupStatus:
         return ret
      cgStatus = radiusProxyStatus.clientGroupStatus[ gName ]
      ret.groups[ gName ] = showClientGroupInfo( cgStatus )
   else:
      for gName in radiusProxyStatus.clientGroupStatus:
         cgStatus = radiusProxyStatus.clientGroupStatus[ gName ]
         ret.groups[ gName ] = showClientGroupInfo( cgStatus )
   return ret

def clearClientGroupCmdHandler( mode, args ):
   gName = args.get( 'GROUP' )
   currentTime = Tac.now()
   if gName is not None:
      configGroup = radiusProxyConfig.clientGroup.get( gName )
      if configGroup:
         configGroup.lastCounterClearTime = currentTime
   else:
      for gName in radiusProxyConfig.clientGroup:
         configGroup = radiusProxyConfig.clientGroup.get( gName )
         if configGroup:
            configGroup.lastCounterClearTime = currentTime

def showServerInfo( serverId, serverStatus ):
   serverInfo = RadiusProxyModel.ServerInfo()
   radiusServerStatus = radiusProxyStatus.radiusStatus[ serverStatus.spec ]
   serverInfo.serverId = serverId
   serverInfo.hostName = serverStatus.serverName
   if serverStatus.spec.protocol == hostProtocol.protoRadsec:
      serverInfo.tlsPort = serverStatus.port
   else:
      serverInfo.authPort = serverStatus.port
      serverInfo.acctPort = serverStatus.acctPort
   serverInfo.vrf = serverStatus.vrf
   if not serverStatus.serverIp.isAddrZero:
      serverInfo.resolvedIpAddr = serverStatus.serverIp
   if radiusServerStatus.dead:
      serverInfo.status = 'inactive'
      serverInfo.lastDeclaredDead = radiusServerStatus.deadtimerStartTime
   else:
      serverInfo.status = 'active'
   return serverInfo

def showServerGroupInfo( sgStatus ):
   sgInfo = RadiusProxyModel.ServerGroupInfo()
   for serverId, serverStatus in enumerate( sgStatus.serverStatus.values() ):
      sgInfo.members.append( showServerInfo( serverId, serverStatus ) )
   return sgInfo

def showServerGroupCmdHandler( mode, args ):
   ret = RadiusProxyModel.showServerGroupCmd()
   gName = args.get( 'GROUP' )
   for cgStatus in radiusProxyStatus.clientGroupStatus.values():
      for sgStatus in cgStatus.serverGroupStatus.values():
         if gName is not None:
            if gName == sgStatus.name:
               ret.groups[ gName ] = showServerGroupInfo( sgStatus )
               return ret
            else:
               continue
         ret.groups[ sgStatus.name ] = showServerGroupInfo( sgStatus )
   return ret

def Plugin( entityManager ):
   global radiusProxyConfig
   global radiusProxyStatus
   radiusProxyConfig = ConfigMount.mount( entityManager, 'radiusproxy/config',
                                          'RadiusProxy::Config', 'w' )
   if toggleDot1xRadiusProxyMergeV0Enabled():
      radiusProxyStatus = LazyMount.mount( entityManager, 'dot1x/radiusproxy/status',
                                           'RadiusProxy::Status', 'r' )
   else:
      radiusProxyStatus = LazyMount.mount( entityManager, 'radiusproxy/status',
                                           'RadiusProxy::Status', 'r' )
