# Copyright (c) 2011, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#------------------------------------------------------------------------------------
# This module provides Rib ipv6 cli functions that depend on cliribd.
#------------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function
from CliPlugin import IraRouteCommon
from CliPlugin import RibIp6Cli
from CliPlugin.VrfCli import (
   vrfExists,
   DEFAULT_VRF,
)
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import RibCliLib
import Tac

# vrfName is defaulted to None so as to allow RibCliLib.cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
def gatedRoute( mode, prefix, vrfName=None ):
   assert vrfName != ''
   cmd = "show ipv6 route"
   cmd = "%s %s pyformat" % ( cmd, prefix.stringValue )
   out = RibCliLib.cliRibdShowCommand( mode, cmd, output=False,
                                       replaceKernelIntfs=True, vrf=vrfName )

   if not out:
      return {}
   return eval( out.rstrip() ) # pylint: disable=eval-used

def showRouteTag( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if RibIp6Cli.l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( "IPv6 routing table %s does not exist." % vrfName )
      return

   if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
      prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   prefix = prefix.subnet.stringValue
   print( "IPv6 Routing Table - %u entries" %
          len( RibIp6Cli.routing.rStatus( vrfName=vrfName ).route ) )
   routeSorter = IraRouteCommon.showRoutes( RibIp6Cli.routing,
                                            prefix, None, protocol=None,
                                            detail=False,
                                            longerPrefixes=False,
                                            vrfName=vrfName,
                                            host=False,
                                            degradeNhgModel=False,
                                            degradeIp6RouteTypeModel=False,
                                            flattenTunnelOverTunnel=False,
                                    degradeCbfViasModel=False ).tacRoute6Sorter()
   print( routeSorter.legend() )
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Ip6Prefix' ).nullPrefix:
      print( "" )
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print( "" )
      return
   proto = routeSorter.protocolString( lpm )
   tag = ribRoute.get( 'tag' )
   print( "%-5s %s tag %d" % ( proto, lpm, tag ) )
   print( "" )
   if not RibIp6Cli.routing.config( vrfName=vrfName ).routing:
      mode.addWarning( "IPv6 routing not enabled" )

def showRouteAge( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if RibIp6Cli.l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( "IPv6 routing table %s does not exist." % vrfName )
      return

   if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
      prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   prefix = prefix.subnet.stringValue
   print( "IPv6 Routing Table - %u entries" %
          len( RibIp6Cli.routing.rStatus( vrfName=DEFAULT_VRF ).route ) )
   routeSorter = IraRouteCommon.showRoutes(
      RibIp6Cli.routing, prefix, None, protocol=None, detail=False,
      longerPrefixes=False, vrfName=vrfName, host=False, degradeNhgModel=False,
      degradeIp6RouteTypeModel=False,
      flattenTunnelOverTunnel=False, degradeCbfViasModel=False ).tacRoute6Sorter()
   print( routeSorter.legend() )
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Ip6Prefix' ).nullPrefix:
      print( "" )
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print( "" )
      return
   proto = routeSorter.protocolString( lpm )
   age = ribRoute.get( 'time' )
   print( "%-5s %s age %s" % ( proto, lpm, age ) )
   print( "" )
   if not RibIp6Cli.routing.config( vrfName=DEFAULT_VRF ).routing:
      mode.addWarning( "IPv6 routing not enabled" )
