# Copyright (c) 2011, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#------------------------------------------------------------------------------------
# This module provides Rib ipv4 cli functions that depend on cliribd.
#------------------------------------------------------------------------------------

from __future__ import absolute_import, division, print_function
from Arnet import (
   Prefix,
   Subnet,
)
from CliPlugin import IraIpCli
from CliPlugin import IraRouteCommon
from CliPlugin import RibIpCli
from CliPlugin.VrfCli import (
   DEFAULT_VRF,
   vrfExists,
)
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import RibCliLib
import Tac

# vrfName is defaulted to None so as to allow RibCliLib.cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
def gatedRoute( mode, prefix, vrfName=None ):
   assert vrfName != ''
   snet = Subnet( prefix )
   cmd = "show ip route"
   cmd = "%s %s %s pyformat" % ( cmd, snet.address_, snet.mask_ )
   out = RibCliLib.cliRibdShowCommand( mode, cmd, output=False,
                                       replaceKernelIntfs=True, vrf=vrfName )
   if not out or out == 'Rib Agent inactive':
      return {} 
   return eval( out.rstrip() ) # pylint: disable=eval-used

def showRouteTag( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if RibIpCli.l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( "IP routing table %s does not exist." % vrfName )
      return

   if isinstance( prefix, str ):
      prefix = Prefix( prefix )
   prefix = RibIpCli.routing.ip.subnet( prefix )

   routeSorter = IraRouteCommon.showRoutes(
      RibIpCli.routing, prefix, None, protocol=None, detail=False,
      longerPrefixes=False, vrfName=vrfName, host=False, degradeNhgModel=False,
      degradeIp6RouteTypeModel=False,
      flattenTunnelOverTunnel=False, degradeCbfViasModel=False ).tacRouteSorter()
   print( routeSorter.legend() )
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Prefix' ).nullPrefix:
      print( "" )
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print( "" )
      return
   proto = routeSorter.protocolString( lpm )
   tag = ribRoute.get( 'tag' )
   print( "%-5s   %s tag %d" % ( proto, lpm, tag ) )
   print( "" )
   IraIpCli.warnIfRoutingDisabled( mode )
   return

def showRouteAge( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if RibIpCli.l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( " IP routing table %s does not exist." % vrfName )
      return

   if isinstance( prefix, str ):
      prefix = Prefix( prefix )
   prefix = RibIpCli.routing.ip.subnet( prefix )

   routeSorter = IraRouteCommon.showRoutes( RibIpCli.routing, prefix, None,
                                            protocol=None, detail=False,
                                            longerPrefixes=False,
                                            vrfName=vrfName,
                                            host=False,
                                            degradeNhgModel=False,
                                            degradeIp6RouteTypeModel=False,
                                            flattenTunnelOverTunnel=False,
                                    degradeCbfViasModel=False ).tacRouteSorter()
   print( routeSorter.legend() )
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Prefix' ).nullPrefix:
      print( "" )
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print( "" )
      return
   proto = routeSorter.protocolString( lpm )
   fibvias = routeSorter.viasForPrefix( lpm )
   ribvias = ribRoute.get( 'vias', [] )
   i = 0
   tunRoute = False
   for nh in fibvias.via:
      age = None
      hop = str( nh.hop )
      if i == 0 and 'tun' in ribvias[i]['intf']:
         tunRoute = True
      if tunRoute:
         hop = ribvias[i]['intf']
         age = ribvias[i]['time']
      else:
         for via in ribvias:
            if hop == via['nexthop']:
               age = via['time']
               break

      if routeSorter.hasV6Nexthop:
         if i == 0:
            print( "%-5s %s" % ( proto, lpm ) )
         print( "%-5s    via %s, age %s" % ( " " * len( proto ), hop, age ) )
      else:
         if i > 0:
            print( "%-5s   %s via %s, age %s" % ( " " * len( proto ),
                                                  " " * len( lpm.stringValue ),
                                                  hop, age ) )
         else:
            print( "%-5s   %s via %s, age %s" % ( proto, lpm, hop, age ) )
      i += 1
   print( "" )
   IraIpCli.warnIfRoutingDisabled( mode )
   return
