# Copyright (c) 2009, 2010, 2012, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=simplifiable-if-statement
# pylint: disable=inconsistent-return-statements

from functools import partial
import hashlib
import os
import re
import time

import Bunch
import Arnet
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
import CliPlugin.RouteMapCli as Globals
from CliPlugin.RouteMapCliModels import (
   ConfigSanityRouteMaps,
   ExpandedRouteMaps,
   IpLargeCommunityLists,
   IpPrefixEntry,
   IpRemarkEntry,
   IpPrefixList,
   IpPrefixLists,
   IpAsPathEntry,
   IpAsPathList,
   IpAsPathLists,
   IpAsPathSummary,
   IpCommunityLists,
   IpExtCommunityLists,
   PeerFilterEntry,
   PeerFilter,
   PeerFilters,
   PrefixListSummary,
)
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import Logging
import RouteMapLib
from RouteMapLib import (
   isMatchAttrEnabled,
   isSetActionEnabled,
   ipCmd,
   bgpOriginEnum,
   matchPermitEnum,
   metricTypeEnum,
   regexModeEnum,
   isisLevelEnum,
   isisStyleEnum,
   matchAcceptEnum,
   getCommValue,
   getExtCommValue,
   getExtCommLbwDisplayValue,
   getLargeCommValue,
   getExtCommTypeValue,
   getErrorMessageCannotDeleteBecauseUsedBy,
   CommunityType,
   convertRegexIndustryStandard,
   setMatchRuleValue,
   listUrlSchemes,
   listUrlLocalSchemes,
   listUrlRemoteSchemes,
   printRouteMapEntryAttributes,
   printPeerFilter,
   duplicateOperation,
   originAsValidityCliEnum,
   POLICY_COMM_INST_MAX,
   POLICY_COMM_INST_MIN,
   matchContribAggAttrOption,
   AS_PATH_MAX_LENGTH,
   AS_PATH_MAX_LENGTH_GATED,
   PREFIX_SEQNO_MAX_VALUE,
   REMARK_CHARACTER_LIMIT,
   RouteMapMatchOption,
   YangMatchSetOptions,
   routeMapYangSource,
)
from Toggles import RouteMapToggleLib
import Tac
import Tracing
import Url

traceHandle = Tracing.Handle( 'RouteMap' )
#pylint: disable-msg=C0321
#pylint: disable-msg=E1103
#pylint: disable-msg=W0702
#pylint: disable-msg=W0703
#pylint: disable=anomalous-backslash-in-string
t0 = traceHandle.trace0
t2 = traceHandle.trace2
t8 = traceHandle.trace8
#------------------------------------------------------------------------
# Simple CLI tokens
#------------------------------------------------------------------------
communityListType = Tac.Type( "Acl::CommunityListType" )

RouteMapEntryKey = Tac.Type( "Acl::RouteMapEntryKey" )

rmCfgCycleErrMsgBase = 'Route map cycle detected'

def copyTacEntity( destination, source, copyHandler ):
   """
   Function to create a copy of the given Tac::Entity 'source'
   to destination.
   """
   entityCopy = Tac.newInstance( "Cli::Session::EntityCopy" )
   entityCopy.handler = Tac.newInstance(
       "Cli::Session::EntityCopyGlobalSessionHandler" )
   entityCopy.handler.handlerDir = Globals.handlerDir
   entityCopy.handler.handlerTable = Globals.handlerDir.inCopyHandlerTable
   copyHandler.handleEntity( entityCopy, destination, source, "" )

def routeMapsSettingCommunityList( commName ):
   """ @brief  Checks if a community-list is used by a `set community` clause
               in any route-map configured in the running-config.
       @param  [in]   commName    The community name
       @return A set of route-maps (names, seqno) which use the
               policy-construct.
       @note   The runtime complexity is O(N) where N is the number of
               all route-map entries defined.
   """
   routeMapsDeps = set()
   for mapName, routeMap in Globals.mapConfig.routeMap.items():
      for mapSeqno, mapEntry in routeMap.mapEntry.items():
         for attr in ( 'communityAddReplace', 'communityDelete', 'extCommunity',
                       'largeCommunity' ):
            communitySet = getattr( mapEntry, attr )
            if communitySet and commName in communitySet.communityListNameSet:
               routeMapsDeps.add( ( mapName, mapSeqno ) )
   return routeMapsDeps

routeMapMatchOptionByCommunityListType = {
   communityListType.communityStandard: RouteMapMatchOption.matchCommunity,
   communityListType.communityExpanded: RouteMapMatchOption.matchCommunity,
   communityListType.extCommunityStandard: RouteMapMatchOption.matchExtCommunity,
   communityListType.extCommunityExpanded: RouteMapMatchOption.matchExtCommunity,
   communityListType.largeCommunityStandard: RouteMapMatchOption.matchLargeCommunity,
   communityListType.largeCommunityExpanded: RouteMapMatchOption.matchLargeCommunity
}

def routeMapsUsingCommunityList( commName, commListType ):
   """ @brief Checks if a community-list is "used" by any route-map.
       @param [in]  commName     The community-list name.
       @param [in]  commListType The community-list type.
       @return a set of route-maps that depend on that community-list.
   """
   return Globals.routeMapsMatchingOnPolicyConstruct(
      commName, routeMapMatchOptionByCommunityListType[ commListType ] ).union(
         routeMapsSettingCommunityList( commName ) )

def getNumberOfCommunityListEntry( commListName, commListType ):
   """ @return the number of community-list-entries for the community list
                (given name and type).
   """
   counter = 0
   communityList = Globals.aclListConfig.communityList.get( commListName )
   if communityList:
      for commListEntry in communityList.communityListEntry.values():
         if commListEntry.listType == commListType:
            counter += 1
   return counter

def blockCommunityListDeleteIfInUse( mode, commListName, commListType,
                                     seqnoToBeDeleted=None ):
   """ @brief Checks if a community-list can be safely deleted without
              breaking any dependency with a route-map. It prints an
              error message on the current mode stating the community-list
              cannot be deleted (and the list of route-maps currently using it).
       @param [in]  mode         The current mode. Needed to print the error message.
       @param [in]  commListName  The community-list name that one is attempting to
                                 remove.
       @param [in]  commListType  The community-list type that one is attempting to
                                 remove.
       @param [in]  seqnoToBeDeleted  A list of community-list seqno the user is
                                      attempting to delete.
                                      If 'None' it means all seqno are being deleted.
       @return 'True' if the policy construct is used by a route-map and should
               not be deleted.
               'False' if ok to delete.
       @note In case the feature is disabled, the check is simply skipped.
       @note Technically, a community-list cannot be deleted if all the
             following conditions hold:
                - The check is enabled.
                - There exists a route-map with either a match or set clause
                  which targets such community list.
                - No further community list seqno(s) remain in the same list after
                  deletion.
   """
   communityListTypeCapiString = {
      communityListType.communityStandard: 'community-list',
      communityListType.communityExpanded: 'community-list',
      communityListType.extCommunityStandard: 'extcommunity-list',
      communityListType.extCommunityExpanded: 'extcommunity-list',
      communityListType.largeCommunityStandard: 'large-community-list',
      communityListType.largeCommunityExpanded: 'large-community-list'
   }

   if Globals.mapConfig.routeMapPolicyReferenceUnconfiguredError:
      commList = Globals.aclListConfig.communityList.get( commListName )
      if not commList:
         return False  # Community-list does not exists. You can remove it.
      if ( not seqnoToBeDeleted ) or \
         ( len( seqnoToBeDeleted ) == getNumberOfCommunityListEntry(
            commListName, commListType ) ):
         routeMapsDeps = routeMapsUsingCommunityList( commListName, commListType )
         if routeMapsDeps:
            mode.addError( getErrorMessageCannotDeleteBecauseUsedBy(
               communityListTypeCapiString[ commListType ],
               commListName,
               routeMapsDeps ) )
            return True
   return False

def getNumberOfAsPathListEntry( asPathListName ):
   """ @return the number of as-path list entries given the name.
   """
   pathList = Globals.aclListConfig.pathList.get( asPathListName )
   if not pathList:
      return 0
   return len( pathList.pathEntry )

def blockAsPathListDeleteIfInUse( mode, asPathListName, seqnoToBeDeleted=None ):
   """ @brief Checks if an AS-path list can be safely deleted without
              breaking any dependency with a route-map. It prints an
              error message on the current mode stating the AS-path list
              cannot be deleted (and the list of route-maps currently using it).
       @param [in]  mode            The current mode. Needed to print the error
                                    message.
       @param [in]  asPathListName  The AS-path list name that one is attempting to
                                    remove.
       @param [in]  seqnoToBeDeleted  A list of AS-path list seqno the user is
                                      attempting to delete.
                                      If 'None' it means all seqno are being deleted.
       @return 'True' if the policy construct is used by a route-map and should
               not be deleted.
               'False' if ok to delete.
       @note In case the feature is disabled, the check is simply skipped.
   """
   if Globals.mapConfig.routeMapPolicyReferenceUnconfiguredError:
      pathList = Globals.aclListConfig.pathList.get( asPathListName )
      if not pathList:
         return False
      if ( not seqnoToBeDeleted ) or \
         ( len( seqnoToBeDeleted ) == getNumberOfAsPathListEntry( asPathListName ) ):
         routeMapDeps = Globals.routeMapsMatchingOnPolicyConstruct(
            asPathListName, RouteMapMatchOption.matchAsPathList )
         if routeMapDeps:
            mode.addError( getErrorMessageCannotDeleteBecauseUsedBy( 'as-path list',
                                                                     asPathListName,
                                                                     routeMapDeps ) )
            return True
   return False

def blockPrefixListDeleteIfInUse( mode, prefixListName, ipv6 ):
   """ @brief Checks if a 'prefix-list' can be safely deleted without breaking
             any dependency with a route-map.
             Moreover, it prints an error message on the mode (passed as argument)
             in case of dependency.
       @note This check makes distinction between prefix-lists ipv4 and ipv6.
       @return 'True' if the policy construct is used by a route-map and should
               not be deleted.
               'False' if ok to delete.
   """
   if Globals.mapConfig.routeMapPolicyReferenceUnconfiguredError:
      pfxListMap = Globals.aclListConfig.ipv6PrefixList \
         if ipv6 else Globals.aclListConfig.prefixList
      prefixList = pfxListMap.get( prefixListName )
      if not prefixList:
         return False
      routeMapDeps = Globals.routeMapsMatchingOnPolicyConstruct(
         prefixListName,
         RouteMapMatchOption.matchIpv6PrefixList if ipv6
         else RouteMapMatchOption.matchPrefixList )
      if routeMapDeps:
         mode.addError( getErrorMessageCannotDeleteBecauseUsedBy(
            'ipv6 prefix-list' if ipv6 else 'prefix-list',
            prefixListName,
            routeMapDeps ) )
         return True
   return False

def routeMapConfigSanityContainerModel( routeMapConfig ):
   routeMaps = {}
   commTypes = {
      'community': CommunityType.communityTypeStandard,
      'extCommunity': CommunityType.communityTypeExtended,
      'largeCommunity': CommunityType.communityTypeLarge
   }
   allCommLists = {}

   def containsRegexEntries( commList ):
      return any( entry.communityRegex for entry in commList.entries )
   for commTypeName, commType in commTypes.items():
      allCommLists[ commTypeName ] = {}
      for name in Globals.aclListConfig.communityList:
         commList = Globals.communityListByName(
            Globals.aclListConfig, commType, name )

         if commList:
            allCommLists[ commTypeName ][ name ] = {}
            if containsRegexEntries( commList ):
               allCommLists[ commTypeName ][ name ][ "hasRegexpEntries" ] = True

   allPfxLists = {
      'v4': [ name for name in Globals.aclListConfig.prefixList if
               prefixListModelByName(
                  Globals.aclListConfig, listName=name, ipv6=False ) ],
      # pylint: disable-next=unnecessary-comprehension
      'v4access': [ name for name in Globals.aclConfig.config[ 'ip' ].acl ],
      'v6': Globals.aclListConfig.ipv6PrefixList
   }
   allAsPathLists = Globals.aclListConfig.pathList
   allTagSets = Globals.aclListConfig.tagSet

   for name in routeMapConfig.routeMap:
      routeMap = Globals.routeMapByName(
         routeMapConfig.routeMap[ name ], configSanity=True,
         routeMapConfig=routeMapConfig,
         allCommLists=allCommLists,
         allPfxLists=allPfxLists,
         allAsPathLists=allAsPathLists,
         allTagSets=allTagSets )
      if routeMap.entries:
         routeMaps[ name ] = routeMap
   return ConfigSanityRouteMaps( routeMaps=routeMaps )

def getSubRouteMaps( routeMapConfig, mapName, routeMap ):
   routeMaps = {}
   if mapName in routeMapConfig.routeMap:
      for entry in routeMap.entries:
         subRouteMapName = routeMap.entries[ entry ].subRouteMap.name
         if subRouteMapName and subRouteMapName in routeMapConfig.routeMap:
            subRouteMap = Globals.routeMapByName(
               routeMapConfig.routeMap[ subRouteMapName ],
               routeMapConfig=routeMapConfig,
               expanded=True )
            routeMaps[ subRouteMapName ] = subRouteMap
            routeMaps.update( getSubRouteMaps( routeMapConfig, subRouteMapName,
                                               subRouteMap ) )
   return routeMaps

def routeMapExpandedContainerModel( routeMapConfig, mapName=None, pruned=False ):
   routeMaps = {}
   pruned = bool( pruned )
   if mapName is None:
      for name in routeMapConfig.routeMap:
         routeMap = Globals.routeMapByName( routeMapConfig.routeMap[ name ],
                                    routeMapConfig=routeMapConfig, expanded=True )
         if routeMap.entries:
            routeMaps[ name ] = routeMap
   elif mapName in routeMapConfig.routeMap:
      routeMap = Globals.routeMapByName( routeMapConfig.routeMap[ mapName ],
                                 routeMapConfig=routeMapConfig, expanded=True )

      if routeMap.entries:
         routeMaps[ mapName ] = routeMap
         routeMaps.update( getSubRouteMaps( routeMapConfig, mapName, routeMap ) )
   return ExpandedRouteMaps( routeMaps=routeMaps, mapName=mapName, pruned=pruned )

#------------------------------------------------------------------------
# route-map <name> [ statement <name> ] [ permit|deny [<seqno>] ]
#------------------------------------------------------------------------
# Goto 'route-map' config mode. Create a route map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.
def gotoRouteMapMode( mode, args ):
   mapName = args[ 'MAP' ]
   permitAction = args.get( 'ACTION' )
   statementName = args.get( 'STATEMENT' )
   seqNo = args.get( 'SEQUENCE' )

   permit = 'permitMatch'
   seqno = seqNo if seqNo else 10
   context = None
   statementNameFromCli = True
   permitVal = None

   def _getRouteMapSeqno( statementName ):
      # when statementName is not None, a map is present with atleast one entry
      # then iterate through map entries and return ( seqno, permit values ) of
      # the entry matching given statementName. If there is no match return
      # ( -1 & 'permitMatch' ).
      if statementName and mapName in Globals.mapConfig.routeMap and \
         Globals.mapConfig.routeMap[ mapName ].mapEntry:
         for entry in Globals.mapConfig.routeMap[ mapName ].mapEntry.values():
            if entry.statementName == statementName:
               return ( entry.seqno, entry.permit )

         return ( -1, 'permitMatch' )

      # No map or entry yet, use 10 as seqno
      if not mapName in Globals.mapConfig.routeMap or \
         not Globals.mapConfig.routeMap[ mapName ].mapEntry:
         return ( 10, 'permitMatch' )
      # One entry, use it
      if len( Globals.mapConfig.routeMap[ mapName ].mapEntry ) == 1:
         seqNum = list( Globals.mapConfig.routeMap[ mapName ].mapEntry )[ 0 ]
         permitVal = Globals.mapConfig.routeMap[ mapName ].mapEntry[ seqNum ].permit
         return ( seqNum, permitVal )
      # More than one entry, display error
      elif len( Globals.mapConfig.routeMap[ mapName ].mapEntry ) > 1:
         return ( 0, 'permitMatch' )

   if permitAction is not None:
      if permitAction == 'deny':
         permit = 'denyMatch'
      if seqNo:
         if statementName and mapName in Globals.mapConfig.routeMap and \
            Globals.mapConfig.routeMap[ mapName ].mapEntry:
            for entry in Globals.mapConfig.routeMap[ mapName ].mapEntry.values():
               if entry.statementName == statementName and entry.seqno != seqNo:
                  mode.addError( "Statement name %s is already assigned to "
                     "sequence number %d" % ( statementName, entry.seqno ) )
                  return
      else:
         ( seqno, permitVal ) = _getRouteMapSeqno( statementName )
   else:
      ( seqno, permitVal ) = _getRouteMapSeqno( statementName )

   if not seqno:
      mode.addError( "Please specify the route map sequence number" )
      return

   if seqno == -1:
      mode.addError( "Statement name %s does not exist" % statementName )
      return

   if permitAction is None:
      permit = permitVal

   t0( 'gotoRouteMapMode %s, %s, %s, %s' % ( mapName, statementName,
                                             permit, seqno ) )

   if context is None:
      context = Globals.Context( mapName )

   if mapName in Globals.mapConfig.routeMap and \
      seqno in Globals.mapConfig.routeMap[ mapName ].mapEntry:
      entry = Globals.mapConfig.routeMap[ mapName ].mapEntry[ seqno ]

      if statementName and entry.statementName != statementName:
         errorStr = "Statement name %s cannot be changed once it is set" \
                    % ( entry.statementName )
         mode.addError( errorStr )
         return

      context.copyEditEntry( entry, permit )
   else:
      if not statementName:
         statementName = str( seqno )
         statementNameFromCli = False
      context.newEditEntry( permit, seqno, statementName, statementNameFromCli )

   mode.routeMapContext = context
   childMode = mode.childMode( Globals.RouteMapMode, context=context )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------
# route-map <destination_name> { copy|rename } <source_name> [ overwrite ]
#------------------------------------------------------------------------
def copyOrRenameRouteMapMode( mode, args ):
   destMapName = args[ 'DESTINATION' ]
   copyOrRename = args[ 'ACTION' ]
   srcMapName = args[ 'SOURCE' ]
   overwrite = args.get( 'overwrite' )

   newMapToBeAdded = destMapName not in Globals.mapConfig.routeMap
   destMap = None
   srcMap = Globals.mapConfig.routeMap.get( srcMapName )
   if not srcMap:
      errorStr = "Route map %s does not exist" % ( srcMapName )
      mode.addError( errorStr )
      return
   if overwrite or not overwrite and newMapToBeAdded:
      destMap = Globals.mapConfig.routeMap.newMember( destMapName )
   if srcMapName == destMapName:
      return
   if not destMap:
      errorStr = ( "Route map %s already exists. Add 'overwrite'"
                   % ( destMapName ) )
      mode.addError( errorStr )
      return
   routeMapCopyHandler = Globals.inputDir[ 'RouteMap' ].routeMapCopyHandler
   copyTacEntity( destMap, srcMap, routeMapCopyHandler )

   operation = 'Copied'
   if copyOrRename == 'rename':
      del Globals.mapConfig.routeMap[ srcMapName ]
      operation = 'Renamed'
   t0( f'{operation} route map {srcMapName} to {destMapName}' )

#------------------------------------------------------------------------
# no route-map <name> [ statement <name> ] [ permit|deny [<seqno>] ]
#------------------------------------------------------------------------
def deleteRouteMapMode( mode, args ):
   mapName = args[ 'MAP' ]
   seqno = args.get( 'SEQUENCE' )
   statementName = args.get( 'STATEMENT' )

   if seqno is None and statementName and mapName in Globals.mapConfig.routeMap and \
      Globals.mapConfig.routeMap[ mapName ].mapEntry:
      for entry in Globals.mapConfig.routeMap[ mapName ].mapEntry.values():
         if entry.statementName == statementName:
            seqno = entry.seqno

      # When non-existent statement name is provided in CLI just return
      # otherwise route-map will get deleted. This will keep the behavior
      # in parity with when non-existent sequence number is provided
      if seqno is None:
         return

   lastMode = mode.session_.modeOfLastPrompt()
   if isinstance( lastMode, Globals.RouteMapMode ) and \
      lastMode.routeMapContext is not None and \
      lastMode.routeMapContext.mapName() == mapName:

      currentEntry = lastMode.routeMapContext.currentEntry()

      # current sequence discarded, if it is the one being deleted
      if seqno is None or seqno == currentEntry.seqno:
         lastMode.routeMapContext = None

   routeMap = Globals.mapConfig.routeMap.get( mapName )
   if not routeMap:
      return

   def deleteRouteMap( routeMap ):
      # Clean up match tag set workaround cache
      for seqno, entry in routeMap.mapEntry.items():
         if entry.matchTagSetName:
            Globals.maybeClearTagSetCache( entry.matchTagSetName,
                                           routeMap.name, seqno )

      del Globals.mapConfig.routeMap[ mapName ]
      t0( 'Deleted map %s' % mapName )

   if seqno is None:
      # delete the entire map
      deleteRouteMap( routeMap )
      return

   # delete one entry of the map
   entry = routeMap.mapEntry.get( seqno )
   if entry:
      # Clean up match tag set workaround cache
      if entry.matchTagSetName:
         Globals.maybeClearTagSetCache( entry.matchTagSetName, mapName, seqno )

      del routeMap.mapEntry[ seqno ]
      t0( 'Deleted map %s entry %s' % ( mapName, seqno ) )
      # deleted the last entry
      if not routeMap.mapEntry:
         deleteRouteMap( routeMap )
      else:
         routeMap.version += 1

#------------------------------------------------------------------------
# Register top-level CLI commands
#------------------------------------------------------------------------
RouteMapOpModeTypeEnum = Tac.Type( "Routing::RouteMap::RouteMapOpModeType" )

def handlerSetOperationsCmd( mode, args ):
   if 'sequential' in args:
      Globals.mapConfig.routeMapOpMode = \
         RouteMapOpModeTypeEnum.routeMapSetOpModeSequential
   elif 'merged' in args:
      Globals.mapConfig.routeMapOpMode = \
         RouteMapOpModeTypeEnum.routeMapSetOpModeMerged
   else:
      Globals.mapConfig.routeMapOpMode = Globals.mapConfig.routeMapOpModeDefault

def handlerNoSetCommExplicitCmd( mode, args ):
   Globals.mapConfig.noSetCommExplicitMembers = True

def noOrDefaultHandlerNoSetCommExplicitCmd( mode, args ):
   Globals.mapConfig.noSetCommExplicitMembers = False

def handlerShowRouteMapConfigSanityCmd( mode, args ):
   return routeMapConfigSanityContainerModel( Globals.dynamicMapConfig )

def handlerShowRouteMapExpandedCmd( mode, args ):
   return routeMapExpandedContainerModel( Globals.dynamicMapConfig,
                                          mapName=args.get( 'NAME' ),
                                          pruned='pruned' in args )

def handlerShowRouteMapCmd( mode, args ):
   return Globals.routeMapContainerModel(
      Globals.dynamicMapConfig, mapName=args.get( 'NAME' ) )

# for show pending and show diff
def printEntry( mode, entry=None, output=None ):
   context = mode.routeMapContext
   if output is None:
      import sys # pylint: disable=import-outside-toplevel
      output = sys.stdout
   if entry is None:
      entry = context.currentEntry()

   prefix = "   "
   cmdList = []
   output.write( "route-map %s %s %s\n" %
      ( context.mapName(), matchPermitEnum[ entry.permit ], entry.seqno ) )

   printRouteMapEntryAttributes( entry, cmdList )
   for cmd in cmdList:
      output.write( "%s%s\n" % ( prefix, cmd ) )

def showDiff( mode, generateStates ):
   '''
      mode            -- Current Cli Mode
      generateStates  -- A function that accepts 'mode', alongside two
                         write streams, where it writes the old/new states
                         of subject for which the diff is generated. These
                         streams are then used to generate the diff.

      Prints the diff between active and pending configuration states
      of a mode to screen.
   '''
   import difflib # pylint: disable=import-outside-toplevel
   from io import StringIO # pylint: disable=import-outside-toplevel

   pendingOutput, activeOutput = StringIO(), StringIO()
   generateStates( mode, pendingOutput, activeOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print( '\n'.join( diff ) )

def createRouteMapDiffs( mode, pendingOutput, activeOutput ):
   context = mode.routeMapContext
   pendingEntry = context.currentEntry()
   printEntry( mode, pendingEntry, pendingOutput )

   if context.mapName() in Globals.mapConfig.routeMap:
      rtMap = Globals.mapConfig.routeMap[ context.mapName() ]
      configEntry = rtMap.mapEntry[ pendingEntry.seqno ]
      printEntry( mode, configEntry, activeOutput )

def handlerRouteMapModeShowCmd( mode, args ):
   if 'pending' in args:
      return printEntry( mode )
   elif 'diff' in args:
      return showDiff( mode, createRouteMapDiffs )
   return mode.showCurrentRouteMap( args )

def handlerRouteMapModeAbortCmd( mode, args ):
   mode.abort()

# route-map mode description
def setDescription( mode, args ):
   descString = args.get( 'DESCRIPTION' )
   context = mode.routeMapContext
   entry = context.currentEntry()

   # repetitions of a description within the same seqno are
   # not allowed
   if entry.description and descString in list( entry.description.values() ):
      return

   nextId = len(entry.description)
   entry.description[ nextId ] = descString

def noDescription( mode, args ):
   context = mode.routeMapContext
   entry = context.currentEntry()
   entry.description.clear()

#-----------------------------------------------------------------------
# Subroutemap binding rules
#-----------------------------------------------------------------------
def handlerSubRouteMapCmd( mode, args ):
   subRouteMapName = args.get( 'NAME' )
   invert = 'invert-result' in args
   no = CliCommand.isNoOrDefaultCmd( args )
   if no:
      subRouteMapName = ''
      invert = False
   mode.setSubRouteMap( subRouteMapName, subInvert=invert, )
#------------------------------------------------------------------------
# Match and set value binding rules
#------------------------------------------------------------------------
if isMatchAttrEnabled( 'matchAccessList' ):
   def handlerMatchAccessList( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchAccessList',
                           args.get( 'ACL' ) )

if isMatchAttrEnabled( 'matchPrefixList' ):
   def handlerMatchPrefixList( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchPrefixList',
                          args.get( 'NAME' ), invertMatch='invert-result' in args )

if isMatchAttrEnabled( 'matchIpv6PrefixList' ):
   def handlerMatchPrefixListV6( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchIpv6PrefixList',
                          args.get( 'NAME' ), invertMatch='invert-result' in args )

def handlerMatchIpGateway( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchGateway', args[ 'GATEWAY' ] )

if isMatchAttrEnabled( 'matchNextHop' ):
   def handlerMatchNextHop( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchNextHop',
                           args.get( 'NEXTHOP' ) )

if isMatchAttrEnabled( 'matchIpv6NextHop' ):
   def handlerMatchNextHopV6( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchIpv6NextHop',
                           args.get( 'NEXTHOP' ) )

if isMatchAttrEnabled( 'matchNextHopPrefixList' ):
   def handlerMatchNextHopPfxList( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchNextHopPrefixList',
                           args.get( 'NAME' ) )

if isMatchAttrEnabled( 'matchIpv6NextHopPrefixList' ):
   def handlerMatchNextHopPfxListV6( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchIpv6NextHopPrefixList',
                           args.get( 'NAME' ) )

if isMatchAttrEnabled( 'matchResolvedNextHopPrefixList' ):
   def handlerMatchResolvedNextHopPfxList( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchResolvedNextHopPrefixList',
                           args.get( 'NAME' ) )

if isMatchAttrEnabled( 'matchIpv6ResolvedNextHopPrefixList' ):
   def handlerMatchResolvedNextHopPfxListV6( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, 'matchIpv6ResolvedNextHopPrefixList',
                           args.get( 'NAME' ) )

def handlerMatchAsPathName( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchAsPath', args[ 'NAME' ] )

if isMatchAttrEnabled( matchContribAggAttrOption ):
   def handlerMatchAggregateAttributes( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mapName = args.get( 'MAP', '' )
      invert = args.get( 'invert-result' )
      # loop detection doesn't apply when REMOVING configs
      if mapName and not no:
         if Globals.hasRouteMapCfgCycle( mode.routeMapContext.mapName(), mapName ):
            mode.addError( rmCfgCycleErrMsgBase )
            return False
      mode.setMatchValue( no, matchContribAggAttrOption,
                           [ mapName, invert ] )

def handlerMatchCommunityCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   communities = args.pop( 'COMMUNITIES' )
   mode.setMatchValue( no, 'matchCommunity', [ communities ] +
                        list( args.values() ) )

def handlerMatchExtCommunityCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   communities = args.pop( 'COMMUNITIES' )
   mode.setMatchValue( no, 'matchExtCommunity', [ communities ] +
                        list( args.values() ) )

def handlerMatchLargeCommunityCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   communities = args.pop( 'COMMUNITIES' )
   mode.setMatchValue(
      no, 'matchLargeCommunity', [ communities ] + list( args.values() ) )

def handlerMatchCommunityInstances( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   value = Bunch.Bunch()
   value.start = args.get( 'GE' ) if '>=' in args else args.get( 'EQ' )
   value.end = args.get( 'LE' ) if '<=' in args else args.get( 'EQ' )
   if value.start is None:
      value.start = POLICY_COMM_INST_MIN
   if value.end is None:
      value.end = POLICY_COMM_INST_MAX
   if value.start > value.end:
      mode.addError( 'Invalid match clause' )
      return False
   value.invert = 'invert-result' in args
   mode.setMatchValue( no, 'matchCommunityInstances', value )

def handlerMatchDistanceCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchDistance', args[ 'DIST' ] )

def handlerMatchInterfaceCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchInterface', str( args[ 'INTF' ] ) )

def handlerMatchLocalPrefCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchLocalPref', args[ 'BGP_LOCAL_PREF' ] )

def handlerMatchMedCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchMed', args[ 'BGP_MED' ] )

def handlerMatchMetricCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchMetric', args[ 'RT_METRIC' ] )

def handlerMatchMetricTypeCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchMetricType', args[ 'OSPF_TYPE' ] )

def handlerMatchIsisLevelCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchIsisLevel', args[ 'ISIS_LVL' ] )

def handlerMatchProtocolCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchProtocol', args[ 'PROTOCOL' ] )

def handlerMatchTagCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   entry = mode.routeMapContext.currentEntry()
   ruleName = 'matchTag'
   tagSetName = args.get( 'TAG_SET' ) or ''

   # Guards against removing the wrong command
   if no:
      # Ignore when `no match tag` is trying to remove
      # `match tag set` configuration
      if 'ROUTE_TAG' in args and entry.matchTagSetName:
         return
      # Ignore when `no match tag set` is trying to remove
      # `match tag` configuration
      elif 'TAG_SET' in args and not entry.matchTagSetName \
            and ruleName in entry.matchRule:
         return

   if 'TAG_SET' in args:
      tagSetValue = None
      # Retrieve existing tag set and value, if any
      tagSet = Globals.aclListConfig.tagSet.get( tagSetName )
      if tagSet and len( tagSet.tagValue ):
         # Only 1 value is allowed for now
         tagSetValue = tagSet.tagValue[ 0 ]

      # Deleting the configuration
      if no and tagSetName == entry.matchTagSetName:
         entry.matchTagSetName = ''

         # Clean up value if tag set had one
         if tagSetValue is not None:
            mode.setMatchValue( no, ruleName, tagSetValue )

      # Adding the configuration
      elif not no:
         entry.matchTagSetName = tagSetName

         # Update matchTag rule depending on value in tag set
         if tagSetValue is None and ruleName in entry.matchRule:
            del entry.matchRule[ ruleName ]
         elif tagSetValue:
            mode.setMatchValue( False, ruleName, tagSetValue )
   else:
      # match tag <value>, clear matchTagSetName
      entry.matchTagSetName = ''
      mode.setMatchValue( no, ruleName, args[ 'ROUTE_TAG' ] )

def handlerMatchRouterIdCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchRouterId', args[ 'PFX_LIST' ] )

def handlerMatchRouteTypeCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchRouteType', args[ 'RT_TYPE' ] )

def handlerMatchOriginAsCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchOriginAs', args[ 'AS_NUM' ] )

def handlerMatchOriginAsValidityCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchOriginAsValidity', args[ 'VALIDITY' ],
                        invertMatch='invert-result' in args )

def handlerMatchAsCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchPeerAs', args[ 'AS_NUM' ] )

def handlerMatchAsPath( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchAsPathList', args[ 'AS_PATH_LIST' ] )

def handlerMatchAsPathLength( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   asPathLengthRange = args[ 'AS_PATH_LENGTH_EXPR' ]
   if asPathLengthRange is False:
      mode.addError( "Invalid match clause" )
      return
   asPathLengthRange.invert = 'invert-result' in args
   mode.setMatchValue( no, 'matchAsPathLength', asPathLengthRange )

def handlerMatchIsisInstance( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchIsisInstance', args[ 'INSTANCE_ID' ] )

def handlerMatchOspfInstance( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchOspfInstance', args[ 'INSTANCE_ID' ] )

#------------------------------------------------------------------------
# Set commands
#------------------------------------------------------------------------
def handlerSetOriginCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'origin', bgpOriginEnum[ args[ 'ACTION' ] ] )

def handlerSetOriginAsValidityCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'originAsValidity',
         originAsValidityCliEnum[ args[ 'VALIDITY' ] ] )

def handlerSetDistanceCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'distance', args[ 'DIST' ] )

def validateAsPath( mode, asPath, repeat=1 ):
   if ( len( asPath ) * ( repeat ) ) > AS_PATH_MAX_LENGTH:
      mode.addErrorAndStop( "Cannot add more than %d autonomous systems" %
      AS_PATH_MAX_LENGTH )
   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      if len( asPath ) > AS_PATH_MAX_LENGTH_GATED:
         mode.addErrorAndStop( "Multi-agent mode must be configured to add " +
                               "more than %d autonomous systems"
                               % AS_PATH_MAX_LENGTH_GATED )
      if repeat > 1:
         mode.addErrorAndStop( "The 'repeat' option is only supported " +
                               "in multi-agent mode" )

def handlerAsPathPrependCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   asPath = args.get( 'AS_NUM' )
   repeat = args.get( 'REPS', 1 )
   if no:
      func = mode.noSetActionValue
   else:
      func = mode.setActionValue
      validateAsPath( mode, asPath, repeat )
   if repeat > 1:
      asPath.extend( [ 'repeat', repeat ] )
   func( 'asPathPrepend', asPath )

def handlerAsPathReplaceCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   actionValue = args.get( 'none' )
   if actionValue is None:
      actionValue = args.get( 'AS_NUM' )
      repeat = args.get( 'REPS', 1 )
      if not no:
         validateAsPath( mode, actionValue, repeat )
      if repeat > 1:
         actionValue.extend( [ 'repeat', repeat ] )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'asPathReplace', actionValue )

def handlerLastAsCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   lastAsCount = args[ 'REPS' ]
   if ( not no and
         getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd and
         lastAsCount > AS_PATH_MAX_LENGTH_GATED ):
      mode.addErrorAndStop( "Multi-agent mode must be configured to add " +
                              "more than %d autonomous systems"
                              % AS_PATH_MAX_LENGTH_GATED )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'asPathPrependLastAs', lastAsCount )

def handlerSetIpNextHopCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'nextHop', args[ 'NEXTHOP' ] )

def handlerSetIpv6NextHopCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   ipv6NextHop = args[ 'NEXTHOP' ]
   if ipv6NextHop.isLinkLocal:
      mode.addError( 'Link-local IPV6 next hops are prohibited.' )
   else:
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'ipv6NextHop', ipv6NextHop )

def handlerSetIpNextHopPeerAddrCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'nextHopPeerAddr', True )

def handlerSetIpv6NextHopPeerAddrCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'ipv6NextHopPeerAddr', True )

def handlerSetIpNextHopUnchangedCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'nextHopUnchanged', True )

def handlerSetIpv6NextHopUnchangedCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'ipv6NextHopUnchanged', True )

def handlerSetEvpnNextHopUnchangedCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'evpnNextHopUnchanged', True )

def handlerSetOspfBitDnCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'ospfBitDn', True )

def handlerSetOspfv3BitDnCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'ospfv3BitDn', True )

def handlerSetResolutionRibProfileCmd( mode, args ):
   resRibProfileConfig = args[ 'RIBS' ]
   resRibProfileConfig.useDefault = 'system-default' in args
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'resolutionRibProfileConfig', resRibProfileConfig )

def handlerSetLocalPrefCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'localPref', args[ 'BGP_LOCAL_PREF' ] )

def handlerSetWeightCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   weight = args[ 'WEIGHT' ]
   func = mode.noSetActionValue if no else mode.setActionValue
   func( "weight", weight )

def handlerSetBgpBestpathAsPathWeight( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   weight = args[ 'WEIGHT' ]
   func = mode.noSetActionValue if no else mode.setActionValue
   func( "bpAsPathWeight", weight )

def handlerSetMetric( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   metric = args.get( 'METRIC' )
   yangSource = args.get( 'YANG_SOURCE' )
   if yangSource == 'bgp-actions':
      metric.yangSource = routeMapYangSource.yangSourceBgpActions
   elif yangSource == 'ospf-actions':
      metric.yangSource = routeMapYangSource.yangSourceOspfActions
   elif yangSource == 'isis-actions':
      metric.yangSource = routeMapYangSource.yangSourceIsisActions
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'metric', metric )

def handlerSetMetricType( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   metricType = metricTypeEnum.get( args.get( 'TYPE' ) )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'metricType', metricType )

def handlerSetAigpMetricCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   metric = args.get( 'AIGP_METRIC' )
   if not no and metric is None:
      metric = 'igp ' + args[ 'IGP_SET_OPTION' ]
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'aigpMetric', metric )

def handlerSetIsisLevelCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'level', isisLevelEnum[ args[ 'ISIS_LVL' ] ] )

def handlerSetIsisMetricStyleCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   func = mode.noSetActionValue if no else mode.setActionValue
   # for metric style there is only one active enum value
   func( 'isisMetricStyle', isisStyleEnum[ args[ 'wide' ] ] )

def handlerSetTagCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   tag = args.get( 'VALUE', 'auto' )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'tag', tag )

def handlerSetCommunityNoneCmd( mode, args ):
   # Clear any community and communityList entries from communityAddReplace
   # while flagging it as none, remove communityDelete completely, and set the
   # flags accordingly.
   mode.alterModeCommunityDict( "communityAddReplace", "community",
                                 set.intersection, set(),
                                 communityFlag=mode.communityActionToFlag( "none" ),
                                 removeEmpty=False )
   mode.alterModeCommunityDict( "communityAddReplace", "communityListNameSet",
                                 set.intersection, set(),
                                 communityFlag=mode.communityActionToFlag( "none" ),
                                 removeEmpty=False )
   mode.removeCommunitySet( "communityDelete" )
   mode.setCommunityListFlag( False, "communityAddReplace" )

def noOrDefaultHandlerSetCommunityNoneCmd( mode, args ):
   # If we are trying to remove the none community, we want to make sure it
   # exists to be removed first / we aren't removing additive communities.
   if mode.noCommunityActionPossible( "none" ):
      mode.exitNone()

def noOrDefaultHandlerNoSetCommunityCmd( mode, args ):
   action = args.get( 'ACTION' )
   if mode.noCommunityActionPossible( action ):

      # If the user called this method, then we will clear the additive and/or
      # delete communitySets from the entry and switch off the flags for the entry
      if action == "additive":
         communitySetsToClear = [ "communityAddReplace" ]
      elif action == "delete":
         communitySetsToClear = [ "communityDelete" ]
      else: # i.e. action is None (the type not string)

         # Only one of these sets exists (otherwise nocommunityactionpossible
         # would have returned false), but removing non-existent set has no affect
         communitySetsToClear = [ "communityAddReplace", "communityDelete" ]

      # Once we've figured out which communityset(s) to clear, we will have mode
      # clear them and switch off their flag bits.
      for communitySetName in communitySetsToClear:
         mode.removeCommunitySet( communitySetName )

def handleSetCommunityList( mode, no, communityListsList, action=None,
                               communityFilterType=None ):
   actionFlag = mode.communityActionToFlag( action )

   # It may be that in evaluating the actionFlag, it turns out that the user has
   # made an ambiguous command, in which case no action should be taken
   if actionFlag is None:
      return
   elif actionFlag == "commSetDelete":
      fieldString = "communityDelete"
      communitySetObject = mode.getEntry().communityDelete
   elif actionFlag == "commSetFilter":
      fieldString = "communityFilter"
      communitySetObject = mode.getEntry().communityFilter
   else:
      fieldString = "communityAddReplace"
      communitySetObject = mode.getEntry().communityAddReplace

   if no:

      # We also want to make sure that the community set is using the same
      # community type as the command (i.e. community-lists) and that the action
      # the user gave makes sense in the current entry's context
      if mode.usingCommunityList( communitySetObject ) and \
         mode.noCommunityActionPossible( action ):

         # The system will either delete only specified communities, or everything
         # depending on if the "service routing configuration route-map
         # no-set-community explicit-members" command was called
         if Globals.mapConfig.noSetCommExplicitMembers:
            setOperation = set.difference
         else:
            def setOperation( x, y ): return set()
      else:
         return
   else:
      if Globals.mapConfig.routeMapPolicyReferenceUnconfiguredError and \
         ( mode.session_.isInteractive() or mode.session_.isEapiClient() ):
         unconfLists = Globals.getUnconfiguredCommunityLists(
            communityListsList, CommunityType.communityTypeStandard )
         if unconfLists:
            mode.addError( 'community-list "{}" is not configured'.format(
               ', '.join( unconfLists ) ) )
            return

      # Since set commands for community lists overwrite the old set of lists, our
      # operation will "merge" the two sets by dropping the first older one.
      def setOperation( x, y ): return y

      # We also want to ensure that we properly exit none mode (if we're in it)
      mode.exitNone()

      # We also need to handle an atypical behavior vis-a-vis moving default
      # communities over to communityDelete.
      if actionFlag == "commSetDelete":
         mode.handleDefaultToDelete( "communityListNameSet" )

   # We will want to set the communityLists while clearing any communities we had.
   mode.alterModeCommunityDict( fieldString, "communityListNameSet", setOperation,
                                set( communityListsList ), actionFlag,
                                removeEmpty=True,
                                communityFilterType=communityFilterType )

   mode.alterModeCommunityDict( fieldString, "community", set.intersection,
                                set(), communityFlag=None, removeEmpty=True )

   # We will also need to set the useCommunityList flags on.
   mode.setCommunityListFlag( True, fieldString )

def handlerSetCommunityListCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   communityListsList = args.get( 'LIST' )
   action = args.get( 'ACTION' )
   handleSetCommunityList( mode, no, communityListsList,
                           action=action )

def handlerSetCommunityListFilterCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   communityListsList = args.get( 'LIST' )
   action = 'filter'
   communityFilterType = [ 'all' ]
   handleSetCommunityList( mode, no, communityListsList,
                           action=action,
                           communityFilterType=communityFilterType )

def handlerSetCommunityCmd( mode, args ):
   communityValuesList = args.get( 'communityValue' )
   no = CliCommand.isNoOrDefaultCmd( args )
   action = args.get( 'ACTION' )

   # We will convert the community strings into the format used by the TACC model.
   communityValuesList = [ getCommValue( comm ) for comm in communityValuesList ]
   # The current entry's context and the user's action will determine which
   # communitySet is going to be manipulated ( if any )
   actionFlag = mode.communityActionToFlag( action )
   if actionFlag is None:
      return
   elif actionFlag == "commSetDelete":
      fieldString = "communityDelete"
      communitySetObject = mode.getEntry().communityDelete
   else:
      fieldString = "communityAddReplace"
      communitySetObject = mode.getEntry().communityAddReplace

   if no:
      if not mode.usingCommunityList( communitySetObject ) and \
         mode.noCommunityActionPossible( action ):

         if Globals.mapConfig.noSetCommExplicitMembers:
            setOperation = set.difference
         else:
            def setOperation( x, y ): return set()
      else:
         return
   else:
      setOperation = set.union
      mode.exitNone()

   # We need to handle an atypical behavior vis-a-vis moving default
   # communities over to communityDelete.
   if actionFlag == "commSetDelete":
      mode.handleDefaultToDelete( "community" )

   mode.alterModeCommunityDict( fieldString, "community", setOperation,
                                 set( communityValuesList ), actionFlag,
                                 removeEmpty=True )
   mode.alterModeCommunityDict( fieldString, "communityListNameSet",
                                 set.intersection, set(), None,
                                 removeEmpty=True )
   mode.setCommunityListFlag( False, fieldString )

def handlerSetCommunityRemoveCmd( mode, args ):
   communityValuesList = args.get( 'communityValue' )
   action = args.get( 'ACTION' )

   # We will convert the community strings into the format used by the TACC model.
   communityValuesList = [ getCommValue( comm ) for comm in communityValuesList ]
   # The current entry's context and the user's action will determine which
   # communitySet is going to be manipulated ( if any )
   actionFlag = mode.communityActionToFlag( action )
   if actionFlag is None:
      return
   elif actionFlag == "commSetDelete":
      fieldString = "communityDelete"
      communitySetObject = mode.getEntry().communityDelete
   else:
      fieldString = "communityAddReplace"
      communitySetObject = mode.getEntry().communityAddReplace

   if not mode.usingCommunityList( communitySetObject ) and \
      mode.noCommunityActionPossible( action ):
      setOperation = set.difference
   else:
      return

   mode.alterModeCommunityDict( fieldString, "community", setOperation,
                                 set( communityValuesList ), actionFlag,
                                 removeEmpty=True )
   mode.alterModeCommunityDict( fieldString, "communityListNameSet",
                                 set.intersection, set(), None,
                                 removeEmpty=True )
   mode.setCommunityListFlag( False, fieldString )

def handlerSetCommunityListRemoveCmd( mode, args ):
   communityListsList = args.get( 'LIST' )
   action = args.get( 'ACTION' )

   actionFlag = mode.communityActionToFlag( action )

   # It may be that in evaluating the actionFlag, it turns out that the user has
   # made an ambiguous command, in which case no action should be taken
   if actionFlag is None:
      return
   elif actionFlag == "commSetDelete":
      fieldString = "communityDelete"
      communitySetObject = mode.getEntry().communityDelete
   else:
      fieldString = "communityAddReplace"
      communitySetObject = mode.getEntry().communityAddReplace

   # We also want to make sure that the community set is using the same
   # community type as the command (i.e. community-lists) and that the action
   # the user gave makes sense in the current entry's context
   if mode.usingCommunityList( communitySetObject ) and \
      mode.noCommunityActionPossible( action ):
      setOperation = set.difference
   else:
      return

   # We will want to set the communityLists while clearing any communities we had.
   mode.alterModeCommunityDict( fieldString, "communityListNameSet", setOperation,
                                 set( communityListsList ), actionFlag,
                                 removeEmpty=True )
   mode.alterModeCommunityDict( fieldString, "community", set.intersection,
                                 set(), communityFlag=None, removeEmpty=True )

   # We will also need to set the useCommunityList flags on.
   mode.setCommunityListFlag( True, fieldString )

def handlerSetLargeCommunityCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if 'none' in args:
      communityValuesList = [ 'none' ]
   else:
      communityValuesList = args.get( 'COMM', [] )
      action = args.get( 'ACTION' )

      if action is not None:
         communityValuesList += [ action ]

   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'largeCommunity', communityValuesList )

def handleSetLargeCommunityList( mode, no, communityListsList, action=None,
                                 communityFilterType=None ):
   actionFlag = mode.communityActionToFlag( action, "large" )

   # It may be that in evaluating the actionFlag, it turns out that the user has
   # made an ambiguous command, in which case no action should be taken
   if actionFlag is None:
      return
   elif actionFlag == "commSetFilter":
      fieldString = "largeCommunityFilter"
      communitySetObject = mode.getEntry().largeCommunityFilter
   else:
      fieldString = "largeCommunity"
      communitySetObject = mode.getEntry().largeCommunity

   if no:
      # We also want to make sure that the community set is using the same
      # community type as the command (i.e. large-community-lists) and that the
      # action the user gave makes sense in the current entry's context
      if mode.usingCommunityList( communitySetObject ) and \
         mode.noCommunityActionPossible( action, "large" ):

         # The system will either delete only specified communities, or everything
         # depending on if the "service routing configuration route-map
         # no-set-community explicit-members" command was called
         if Globals.mapConfig.noSetCommExplicitMembers:
            setOperation = set.difference
         else:
            def setOperation( x, y ): return set()
      else:
         return
   else:
      if Globals.mapConfig.routeMapPolicyReferenceUnconfiguredError and \
         ( mode.session_.isInteractive() or mode.session_.isEapiClient() ):
         unconfLists = Globals.getUnconfiguredCommunityLists(
            communityListsList, CommunityType.communityTypeLarge )
         if unconfLists:
            mode.addError( 'large-community-list "{}" is not configured'.format(
               ', '.join( unconfLists ) ) )
            return

      # Since set commands for large community lists overwrite the old set of
      # lists, our operation will "merge" the two sets by dropping the first
      # older one.
      def setOperation( x, y ): return y

      # We also want to ensure that we properly exit none mode (if we're in it)
      mode.exitNone()

   # We will want to set the communityLists while clearing any large communities
   # we had.
   mode.alterModeCommunityDict( fieldString, "communityListNameSet", setOperation,
                                set( communityListsList ), actionFlag,
                                removeEmpty=True,
                                communityFilterType=communityFilterType )

   mode.alterModeCommunityDict( fieldString, "community", set.intersection,
                                set(), communityFlag=None, removeEmpty=True )

   # We will also need to set the useCommunityList flags on.
   mode.setCommunityListFlag( True, fieldString )

def handlerSetLargeCommunityListCmd( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      mode.addWarning( 'set large-community large-community-list is only '
                        'supported in multi-agent mode' )

   no = CliCommand.isNoOrDefaultCmd( args )
   communityListsList = args.get( 'LIST' )
   action = args.get( 'ACTION' )
   handleSetLargeCommunityList( mode, no, communityListsList,
                                 action=action )

def handlerSetLargeCommunityListFilterCmd( mode, args ):
   if Globals.l3Config.protocolAgentModel != ProtoAgentModel.multiAgent:
      mode.addWarning( 'set large-community large-community-list is only '
                        'supported in multi-agent mode' )

   no = CliCommand.isNoOrDefaultCmd( args )
   communityListsList = args.get( 'LIST' )
   action = 'filter'
   communityFilterType = [ 'all' ]
   handleSetLargeCommunityList( mode, no, communityListsList,
                                 action=action,
                                 communityFilterType=communityFilterType )

if isSetActionEnabled( 'extCommunity' ):
   def handlerSetExtCommunityNoneCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', [ 'none' ] )

   def noOrDefaultHandlerNoSetExtCommunityCmd( mode, args ):
      extCommAction = args.get( 'ACTION' )
      mode.noSetActionValue( 'extCommunity', [ extCommAction ] if
                              extCommAction else None )

   def handlerSetExtCommunityRtSooCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      extCommAction = args.get( 'ACTION' )
      extCommVal = args[ 'RT_VAL' ] + args[ 'SOO_VAL' ]
      if extCommAction is not None:
         assert extCommAction in [ 'additive', 'delete' ]
         extCommVal += [ extCommAction ]
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', extCommVal )

   def handlerSetExtCommunityLbwDelCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', [ args[ 'LBW_VAL' ], args[ 'delete' ] ] )

   def handlerSetExtCommunityLbwCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', [ args[ 'LBW_VAL' ] ] )

   def handlerSetExtCommunityLbwDivideCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', [ 'lbw divide ' + args[ 'OPTION' ] ] )

   def handlerSetExtCommunityLbwAggregateCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      lbwAgg = 'lbw aggregate'
      lbwVal = args.get( 'LBW_VAL' )
      lbwAgg = lbwAgg + ' ' + lbwVal if lbwVal else lbwAgg
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', [ lbwAgg ] )

   def setExtCommColorHandlerCommon( mode, args ):
      # COLOR_EXP will format community for us in COLOR_VAL
      if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
         mode.addWarning( 'Color extended community is only supported in '
                          'multi-agent mode' )
      no = CliCommand.isNoOrDefaultCmd( args )
      func = mode.noSetActionValue if no else mode.setActionValue
      func( 'extCommunity', args.get( 'COLOR_VAL' ) + args.get( 'ACTION', [] ) )

   def handleSetExtCommunityList( mode, no, communityListsList, action=None,
                                  communityFilterType=None ):
      if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
         mode.addWarning( 'set extcommunity extcommunity-list is only supported in '
                          'multi-agent mode' )
      else:
         actionFlag = mode.communityActionToFlag( action, "extended" )
         # It may be that in evaluating the actionFlag, it turns out that the user
         # has made an ambiguous command, in which case no action should be taken
         if actionFlag is None:
            return
         elif actionFlag == "commSetFilter":
            fieldString = "extCommunityFilter"
            communitySetObject = mode.getEntry().extCommunityFilter
         else:
            fieldString = "extCommunity"
            communitySetObject = mode.getEntry().extCommunity
         if no:
            # We also want to make sure that the community set is using the same
            # community type as the command (i.e. extcommunity-lists) and that the
            # action the user gave makes sense in the current entry's context
            if mode.usingCommunityList( communitySetObject ) and \
               mode.noCommunityActionPossible( action, "extended" ):

               # The system will either delete only specified communities, or
               # everything depending on if the "service routing configuration
               # route-map no-set-community explicit-members" command was called
               if Globals.mapConfig.noSetCommExplicitMembers:
                  setOperation = set.difference
               else:
                  def setOperation( x, y ): return set()
            else:
               return
         else:
            if ( Globals.mapConfig.routeMapPolicyReferenceUnconfiguredError and
               ( mode.session_.isInteractive() or mode.session_.isEapiClient() ) ):
               unconfLists = Globals.getUnconfiguredCommunityLists(
                  communityListsList, CommunityType.communityTypeExtended )
               if unconfLists:
                  mode.addError( 'extcommunity-list "{}" is not configured'.format(
                     ', '.join( unconfLists ) ) )
                  return

            # Since set commands for extcommunity lists overwrite the old set
            # of lists, our operation will "merge" the two sets by dropping
            # the first older one.
            def setOperation( x, y ): return y

            # We also want to ensure that we properly exit none mode (if we're in it)
            mode.exitNone()

         # We will want to set the communityLists while clearing any
         # extcommunities we had.
         mode.alterModeCommunityDict( fieldString, "communityListNameSet",
                                      setOperation, set( communityListsList ),
                                      actionFlag, removeEmpty=True,
                                      communityFilterType=communityFilterType )

         mode.alterModeCommunityDict( fieldString, "community", set.intersection,
                                      set(), communityFlag=None, removeEmpty=True )

         # We will also need to set the useCommunityList flags on.
         mode.setCommunityListFlag( True, fieldString )

   def handlerSetExtCommunityListCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      communityListsList = args.get( 'LIST' )
      extCommAction = args.get( 'ACTION' )
      handleSetExtCommunityList( mode, no, communityListsList,
                                 action=extCommAction )

   def handlerSetExtCommunityListFilterCmd( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      communityListsList = args.get( 'LIST' )
      extCommAction = 'filter'
      communityFilterType = args.get( 'COMM_TYPES' )
      if not communityFilterType:
         communityFilterType = [ 'all' ]
      handleSetExtCommunityList( mode, no, communityListsList,
                                 action=extCommAction,
                                 communityFilterType=communityFilterType )

def handlerSetSegmentIndexCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   index = args[ 'INDEX' ]
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'segmentIndex', index )

def handlerContinueSeqCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   seq = args.get( 'SEQUENCE' )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( 'continueSeq', seq )

def handlerSetIgpMetricCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   igpMetric = args.get( 'VALUE', 'max-metric' )
   func = mode.noSetActionValue if no else mode.setActionValue
   func( "igpMetric", igpMetric )

def setRegexMode( mode, args ):
   regexMode = args.get( 'TYPE' )
   mode.addWarning( "AS path ACLs may get disabled due to invalid regexes "
                    "under new regex mode" )
   Globals.aclListConfig.regexMode = regexModeEnum[ regexMode ]

def noRegexMode( mode, args ):
   Globals.aclListConfig.regexMode = Globals.aclListConfig.regexModeDefault

def nextSeqno( collection, step=10 ):
   seqno = step
   if collection:
      seqno = max( collection ) + step
   return seqno

#-------------------------------------------------------------------------
# AS path list
#
# [no] ip as-path access-list <name> <permit|deny> <regex> [<origin>]
#-------------------------------------------------------------------------
def asPathListHelper( mode, args ):
   listName = args[ 'NAME' ]
   permit = args[ 'ACTION' ]
   asRegex = args[ 'REGEX' ]
   origin = args.get( 'ORIGIN' )
   pathList = Globals.aclListConfig.pathList
   seqno = 1
   if origin is None:
      origin = 'any'

   if listName in pathList:
      asPath = pathList[ listName ]

      # If the as-path access-list was already imported via a URL,
      # simply reset it as the user can either configure an as-path
      # access-list via the URL method or via the regular CLI method.
      if asPath.source != "":
         resetAsPathListImport( mode, asPath )

      seqno = nextSeqno( asPath.pathEntry, step=1 )
   else:
      asPath = pathList.newMember( listName )

   posixRegexMode = Globals.aclListConfig.regexMode == 'regexModePosix'

   # Silently ignore regex entry if it already exists. Hua said
   # we should do the same as ACL instead of overwriting it.

   for pathEntry in asPath.pathEntry.values():
      if asRegex == pathEntry.userRegex:
         return

   helper = Tac.newInstance( "Routing::RouteMap::Helper" )
   validDfaRegex = helper.validateDfaRegex( asRegex )
   validPosixRegex = helper.validatePosixRegex( asRegex )
   if not validDfaRegex and not validPosixRegex:
      mode.addError( "Invalid AS path regular expression" )
      return False

   value = Tac.Value( "Acl::AsPathEntry",
                      seqno,
                      asRegex,
                      matchPermitEnum[ permit ],
                      bgpOriginEnum[ origin ],
                      validDfaRegex,
                      validPosixRegex )
   asPath.pathEntry.addMember( value )
   if ( posixRegexMode and not validPosixRegex ) or \
         ( not posixRegexMode and not validDfaRegex ):
      mode.addWarning( "Invalid regex for current regex mode '%s', "
                       "this as-path ACL is inactive" %
                       regexModeEnum[ Globals.aclListConfig.regexMode ] )
   asPath.version += 1

def removeAsPathListImportFile( asPathList ):
   '''Remove downloaded copy of imported prefix-list'''
   if asPathList.source == "":
      return
   for scheme in listUrlRemoteSchemes:
      if asPathList.source.startswith( scheme ):
         helper = Tac.newInstance( "Acl::AsPathCliHelper" )
         localFile = "%s/%s.cfg" % (
            helper.asPathListEntryLocalDir(), asPathList.name )
         Tac.run( [ "rm", "-f", localFile ], asRoot=True,
                  stdout=Tac.DISCARD, stderr=Tac.DISCARD,
                  ignoreReturnCode=True )
         return

def removeAsPathList( mode, args ):
   listName = args[ 'NAME' ]
   pathList = Globals.aclListConfig.pathList
   if listName in pathList:
      if blockAsPathListDeleteIfInUse( mode, listName ):
         return
      removeAsPathListImportFile( pathList[ listName ] )
      del pathList[ listName ]

def asPathListContainerModel( config, listName=None, filename=None ):
   pathNames = []
   activeIpAsPathLists = {}
   inactiveIpAsPathLists = {}

   if filename is not None:
      return IpAsPathLists( regexMode=regexModeEnum[ config.regexMode ],
                            _filename=filename )

   if listName is not None:
      if listName in config.pathList:
         pathNames = [ listName ]
   else:
      # pylint: disable-next=unnecessary-comprehension
      pathNames = [ name for name in config.pathList ]

   posixRegexMode = config.regexMode == 'regexModePosix'

   for name in pathNames:
      entries = list( config.pathList[ name ].pathEntry.values() )

      ipAsPathEntries = []
      aclActive = True
      for entry in entries:
         permit = matchPermitEnum[ entry.permit ]
         regex = entry.userRegex
         origin = entry.origin
         validRegex = entry.validPosixRegex if posixRegexMode else \
               entry.validDfaRegex
         aclActive = aclActive and validRegex
         valid = None if validRegex else validRegex
         ipAsPathEntries.append( IpAsPathEntry( userRegex=regex, filterType=permit,
                                                origin=origin, valid=valid ) )

      pathListModel = IpAsPathList( ipAsPathEntries=ipAsPathEntries,
                      ipAsPathListSource=config.pathList[ name ].source)

      if aclActive:
         activeIpAsPathLists[ name ] = pathListModel
      else:
         inactiveIpAsPathLists[ name ] = pathListModel

   return IpAsPathLists( regexMode=regexModeEnum[ config.regexMode ],
                         activeIpAsPathLists=activeIpAsPathLists,
                         inactiveIpAsPathLists=inactiveIpAsPathLists )

def resetAsPathListImport( mode, asPath ):
   # This function is only applicable if the aspathlist is
   # imported from a URL
   if asPath.source == "":
      return
   proto, _ = asPath.source.split( ":" )
   if proto + ":" in listUrlSchemes:
      helper = Tac.newInstance( "Acl::AsPathCliHelper" )
      fileName = "%s/%s.cfg" % (helper.asPathListEntryLocalDir(), asPath.name )
      if os.path.exists( fileName ):
         Tac.run( [ "rm", "-f", fileName ], stdout=Tac.DISCARD,
                  stderr=Tac.DISCARD, ignoreReturnCode=True )
      asPath.source = ""
      asPath.sourceTimestamp = 0
      helper = Tac.newInstance( "Acl::AsPathCliHelper" )
      helper.delAsPathListEntries( asPath )

# deletes an entry in the list
def asPathEntryRemover( mode, args ):
   listName = args[ 'NAME' ]
   asPathPermit = args[ 'ACTION' ]
   asPathRegex = args[ 'REGEX' ]
   asPathList = Globals.aclListConfig.pathList
   seqnoToBeDeleted = []
   if listName in asPathList:
      asPath = asPathList.get( listName )
      if not asPath:
         return
      if asPath.source != "":
         resetAsPathListImport( mode, asPath )
         seqnoToBeDeleted = [ entry.seqno for entry in asPath.pathEntry.values() ]
         if blockAsPathListDeleteIfInUse( mode, listName, seqnoToBeDeleted ):
            return
         for seqno in seqnoToBeDeleted:
            del asPath.pathEntry[ seqno ]
         if not asPath.pathEntry:
            del asPathList[ listName ]
      else:
         seqnoToBeDeleted = [ entry.seqno for entry in asPath.pathEntry.values()
                              if entry.userRegex == asPathRegex and
                              matchPermitEnum[ entry.permit ] == asPathPermit ]
         if blockAsPathListDeleteIfInUse( mode, listName, seqnoToBeDeleted ):
            return
         for seqno in seqnoToBeDeleted:
            del asPath.pathEntry[ seqno ]
         if not asPath.pathEntry:
            del asPathList[ listName ]
      asPath.version += 1

def showAsPathList( mode, args ):
   listName = args.get( 'NAME' )
   return asPathListContainerModel( Globals.aclListConfig, listName=listName )

def showAsPathAccessListDetail( mode, args ):
   listName = args.get( 'NAME' )
   if listName not in Globals.aclListConfig.pathList:
      errMsg = "List not found"
      return IpAsPathLists(
         regexMode=regexModeEnum[ Globals.aclListConfig.regexMode ], _errMsg=errMsg )
   asPathAccessList = Globals.aclListConfig.pathList[ listName ]
   if asPathAccessList.source == "":
      mode.addWarning( 'Not an imported as-path access-list' )
      return showAsPathList( mode, args )
   if asPathAccessList.sourceTimestamp == 0:
      errMsg = "Failed to import from source %s" % asPathAccessList.source
      return IpAsPathLists(
         regexMode=regexModeEnum[ Globals.aclListConfig.regexMode ], _errMsg=errMsg )
   helper = Tac.newInstance( "Acl::AsPathCliHelper" )
   isStringRegexMode = "string" == regexModeEnum[ Globals.aclListConfig.regexMode ]
   fileName = helper.showAsPathList( asPathAccessList,
                                     isStringRegexMode )
   if fileName != "":
      return asPathListContainerModel( Globals.aclListConfig, filename=fileName )

def showAsPathAccessListSummary( mode, args ):
   model = IpAsPathSummary()
   listName = args.get( 'NAME' )
   if listName not in Globals.aclListConfig.pathList:
      model._invalidListName = True # pylint: disable=protected-access
      return model
   asPathAccessList = Globals.aclListConfig.pathList[ listName ]
   model.entryCount = len( asPathAccessList.pathEntry )
   model.accessListSource = asPathAccessList.source
   model.sourceTimeStamp = float( asPathAccessList.sourceTimestamp )
   return model

#------------------------------------------------------------------------
# Prefix list
#
# ip prefix-list <name> <permit|deny> <prefix> [<ge> <len> <le> <len>]
# no ip prefix-list <name> seq <seqnum> [<permit|deny> <prefix>]
# no ip prefix-list <name>
#------------------------------------------------------------------------
def getPrefixListEntrySeqno( ipv6, prefixListEntry, permit, prefix, geLen, leLen ):
   for entry in prefixListEntry.values():
      prefixVal = entry.ip6Prefix if ipv6 else entry.prefix.stringValue
      if ( entry.permit == matchPermitEnum [ permit ] and entry.le == leLen and
           entry.ge == geLen and prefixVal == prefix ):
         return entry.seqno
   return None

def getPrefixEntryHashVal( prefix, permit, geLen, leLen ):
   '''
      This method returns the hash-value for a prefix list entry, which
      is used to detect existing entries in a prefix-list config.
   '''
   m = hashlib.md5()
   m.update( str( prefix ).encode() )
   m.update( str( permit ).encode() )
   m.update( str( geLen ).encode() )
   m.update( str( leLen ).encode() )
   return m.hexdigest()

def resetPrefixListImport( mode, pfxList ):
   '''Unmark the pfxList as import list. Remove downloaded copy of the
      entry file'''
   if pfxList.source == "":
      return
   ipv6 = not hasattr( pfxList, "prefixEntry" )
   removeUrlPrefixEntryFile( pfxList, ipv6 )
   pfxList.source = ""
   pfxList.sourceTimestamp = 0
   pfxEntry = pfxList.prefixEntry if not ipv6 else pfxList.ipv6PrefixEntry
   for seqno in list( pfxEntry ):
      del pfxEntry[ seqno ]
   if mode and hasattr( mode, "pendingConfig" ):
      listName = mode.pendingConfig.name
      cl = "Acl::Ipv6PrefixList" if ipv6 else "Acl::PrefixList"
      mode.pendingConfig = Tac.newInstance( cl, listName )

def removeUrlPrefixEntryFile( pfxList, ipv6 ):
   proto, _ = pfxList.source.split( ":" )
   if proto + ":" in listUrlSchemes:
      helper = Tac.newInstance( "Acl::CliHelper" )
      filename = "%s/%s.cfg" % (
         helper.prefixListEntryLocalDir( ipv6 ), pfxList.name )
      if os.path.exists( filename ):
         Tac.run( [ "rm", "-f", filename ], stdout=Tac.DISCARD,
                  stderr=Tac.DISCARD, ignoreReturnCode=True )

def resetPrefixEntryHashVals( mode, pfxlst ):
   if pfxlst.source == "":
      return
   if mode and hasattr( mode, "pendingConfig" ):
      mode.prefixEntryHashVals = set()

def loadPrefixListUrl_( mode, listName, pfxUrl, ipv6, vrfName=None, dupAct=None ):
   '''Load prefix list from a URL source'''
   if isinstance( pfxUrl, str ):
      pfxUrl = Url.parseUrl( pfxUrl,
                             Url.Context( *Url.urlArgsFromMode( mode ) ),
                             fsFunc=lambda fs: fs.scheme in listUrlSchemes )
   assert isinstance( pfxUrl, Url.Url )

   if dupAct is None:
      dupAct = duplicateOperation.duplicateError
   elif dupAct == "ignore":
      dupAct = duplicateOperation.duplicateIgnore
   else: # override
      assert dupAct == "override"
      dupAct = duplicateOperation.duplicateOverride

   def createInactivePfxList():
      pfxLists = Globals.aclListConfig.prefixList if not ipv6 \
         else Globals.aclListConfig.ipv6PrefixList
      if listName not in pfxLists:
         pfxList = pfxLists.newMember( listName )
         pfxList.source = str( pfxUrl )
         pfxList.duplicateHandling = dupAct
         if vrfName:
            pfxList.vrfName = vrfName

   helper = Tac.newInstance( "Acl::CliHelper" )

   failOnError = mode.session_.isInteractive() or mode.session_.isEapiClient()

   listUrlNetworkSchemes = { "http:", "scp:", "https:" }
   if pfxUrl.fs.scheme in listUrlRemoteSchemes:
      if vrfName and ( pfxUrl.fs.scheme in listUrlNetworkSchemes ):
         pfxUrl.setUrlVrfName( vrfName )
      localDir = helper.prefixListEntryLocalDir( ipv6 )
      if not os.path.exists( localDir ):
         os.makedirs( localDir )
      filename = "%s/%s.cfg" % ( localDir, listName )
      try:
         pfxUrl.writeLocalFile( pfxUrl, filename )
      except Exception as e:
         Logging.log( RouteMapLib.PREFIXLIST_IMPORT_FAILED,
                      ipCmd( ipv6 ), listName, pfxUrl,
                      "Failed to download" )
         createInactivePfxList()
         if failOnError:
            mode.addError( "Download prefix list %s from %s failed: %s" % (
               listName, pfxUrl, e ) )
         return not failOnError
   else:
      assert pfxUrl.fs.scheme in listUrlLocalSchemes
      filename = pfxUrl.localFilename()

   if not ipv6:
      pfxNewList = Tac.newInstance( "Acl::PrefixList", listName )
      errorStr = helper.loadPrefixListEntries( pfxNewList, filename, dupAct )
   else:
      pfxNewList = Tac.newInstance( "Acl::Ipv6PrefixList", listName )
      errorStr = helper.loadIpv6PrefixListEntries( pfxNewList, filename, dupAct )
   if errorStr != "":
      Logging.log( RouteMapLib.PREFIXLIST_IMPORT_FAILED,
                   ipCmd( ipv6 ), listName, pfxUrl, errorStr )
      mode.addWarning( "Load prefix list %s failed: %s" % (
         listName, errorStr ) )
      createInactivePfxList()
      return not failOnError

   # Copy list over to Sysdb
   if not ipv6:
      pfxList = Globals.aclListConfig.prefixList[ listName ]  \
         if listName in Globals.aclListConfig.prefixList else \
         Globals.aclListConfig.prefixList.newMember( listName )
      helper.copyPrefixListEntries( pfxList, pfxNewList )
      assert len( pfxList.prefixEntry ) == len( pfxNewList.prefixEntry )
   else:
      pfxList = Globals.aclListConfig.ipv6PrefixList[ listName ] \
         if listName in Globals.aclListConfig.ipv6PrefixList else \
         Globals.aclListConfig.ipv6PrefixList.newMember( listName )
      helper.copyIpv6PrefixListEntries( pfxList, pfxNewList )
      assert len( pfxList.ipv6PrefixEntry ) == len( pfxNewList.ipv6PrefixEntry )
   pfxList.source = str( pfxUrl )
   pfxList.sourceTimestamp = int( time.time() )
   pfxList.duplicateHandling = dupAct
   if vrfName:
      pfxList.vrfName = vrfName
   if not ipv6:
      pfxList.inPrefixListMode = False
   pfxList.version += 1
   Logging.log( RouteMapLib.PREFIXLIST_IMPORT_SUCCEEDED,
                ipCmd( ipv6 ), listName, pfxUrl )
   return True

def loadPrefixListUrl( mode, listName, pfxUrl, dupAct=None ):
   loadPrefixListUrl_( mode, listName, pfxUrl, dupAct=dupAct, ipv6=False )

def loadPrefixListUrlHandler( mode, args ):
   listName = args.get( 'NAME' )
   pfxUrl = args.get( 'URL' )
   dupAct = args.get( 'ACTION' )
   vrfName = args.get( 'VRFNAME' )
   ipv6 = False
   loadPrefixListUrl_( mode, listName, pfxUrl, ipv6, vrfName, dupAct )

def managePrefixListRemarksHandler( mode, args ):
   seqno = args.get( 'SEQNO' )
   comment = args.get( 'COMMENT' )

   # get current mode remark list
   remarkList = mode.pendingRemarkList

   # Check if collision between a remark or a prefix
   if seqno in mode.allRemarkSeqnos or seqno in mode.allPrefixSeqnos:
      mode.addError( f"Duplicate sequence number: {seqno}" )
      return False

   # Check whether sequence number is given. If not generate a default
   if seqno is None:
      seqno = max( nextSeqno( mode.allRemarkSeqnos ),
                  nextSeqno( mode.allPrefixSeqnos ) )
   
   # Check whether the sequence number is greater than the max
   if seqno > PREFIX_SEQNO_MAX_VALUE:
      mode.addError( "Sequence number exhausted" )
      return False

   # Truncate Comment Max Character Limit
   if len( comment ) > REMARK_CHARACTER_LIMIT:
      comment = comment[ : REMARK_CHARACTER_LIMIT ]
   
   # track new seqnos
   mode.modRemarkSeqnos.add( seqno )
   mode.allRemarkSeqnos.add( seqno )
   remarkList.remarkEntries[ seqno ] = comment
   return True

def managePrefixListHandler( mode, args ):
   permit = args[ 'ACTION' ]
   prefix = args[ 'PREFIX' ]
   listName = args.get( 'NAME' )
   geLen = args.get( 'GELEN', 0 )
   leLen = args.get( 'LELEN', 0 )
   eqLen = args.get( 'EQLEN', 0 )
   seqno = args.get( 'SEQNO' )

   items = prefix.split( "/" )
   if eqLen != 0:
      if eqLen < int( items[1] ):
         mode.addError( "Error: eq length %s is less than %s" % ( eqLen, items[1] ) )
         return False
      geLen = eqLen
      leLen = eqLen
   else:
      if geLen != 0 and geLen < int( items[1] ):
         mode.addError( "Error: ge length %s is less than %s"
                        % ( geLen, items[1] ) )
         return False
      if leLen != 0 and leLen < int( items[1] ):
         mode.addError( "Error: le length %s is less than %s"
               % ( leLen, items[1] ) )
         return False
      if leLen != 0 and leLen < geLen:
         mode.addError( "Error: le %s is less than ge %s" % (
               leLen, geLen ) )
         return False

   if isinstance( mode, Globals.IpPrefixListMode ):
      listName = mode.pendingConfig.name

   if listName in Globals.aclListConfig.prefixList:
      prefixList = Globals.aclListConfig.prefixList[ listName ]
      resetPrefixListImport( mode, prefixList )
      resetPrefixEntryHashVals( mode, prefixList )

   if isinstance( mode, Globals.IpPrefixListMode ):
      prefixList = mode.pendingConfig
   else:
      if listName in Globals.aclListConfig.prefixList:
         prefixList = Globals.aclListConfig.prefixList[ listName ]
      else:
         prefixList = Globals.aclListConfig.prefixList.newMember( listName )

   result, hashVal, seqno = checkEntryValidCommon( mode, prefixList.prefixEntry,
                                                   permit, prefix, geLen, leLen,
                                                   seqno )
   if not result:
      return False

   try:
      entry = Tac.Value( "Acl::PrefixEntry",
                         seqno,
                         Arnet.Prefix( prefix ),
                         matchPermitEnum[ permit ],
                         geLen,
                         leLen,
                         eqLen )
   except SystemError:
      mode.addError( 'Not a valid Ip Prefix' )
      return False
   prefixList.prefixEntry.addMember( entry )
   if isinstance( mode, Globals.IpPrefixListMode ):
      mode.allPrefixSeqnos.add( seqno )
      mode.modSeqnos.add( seqno )
      mode.prefixEntryHashVals.add( hashVal )
   else:
      # Notify that prefix list mode changes are committed to sysdb
      prefixList.version += 1

def checkEntryValidCommon( mode, prefixListEntry, permit, prefix, geLen, leLen,
                           seqno ):
   """
   Do the following checks to ensure the new entry we are adding is valid:
   1) Check if another entry already matches the entry we are adding
   2) Check that the new entry does not exhaust the sequence numbers
   3) Check that the new entry does not overwrite an existing rule
   """
   hashVal = ''
   if isinstance( mode, ( Globals.IpPrefixListMode, Globals.Ipv6PrefixListMode ) ):
      t0( 'Duplicate check for hashVal %s' % hashVal )
      ipv6 = isinstance( mode, Globals.Ipv6PrefixListMode )
      arnetPfx = Arnet.Ip6Prefix( prefix ) if ipv6 else Arnet.Prefix( prefix )
      hashVal = getPrefixEntryHashVal( arnetPfx,
                                       matchPermitEnum[ permit ],
                                       geLen, leLen )
      # check collision with remarks
      if seqno in mode.allRemarkSeqnos:
         mode.addError( f"Duplicate sequence number: {seqno}" )
         return ( False, hashVal, seqno )

      if hashVal in mode.prefixEntryHashVals:
         existingEntry = getPrefixListEntrySeqno( ipv6, prefixListEntry, permit,
                                                  prefix, geLen, leLen )
         assert existingEntry is not None
         # Don't throw an error if the entry already exists in the place we wanted to
         # add it
         if existingEntry != seqno:
            mode.addError( Globals.DUP_ENTRY_ERROR_STR.format( existingEntry ) )
         return ( False, hashVal, seqno )
   else:
      ipv6 = False
      existingEntry = getPrefixListEntrySeqno( ipv6, prefixListEntry, permit, prefix,
                                               geLen, leLen )
      if existingEntry is not None:
         # Don't throw an error if the entry already exists in the place we wanted to
         # add it
         if existingEntry != seqno:
            mode.addError( Globals.DUP_ENTRY_ERROR_STR.format( existingEntry ) )
         return ( False, hashVal, seqno )

   if seqno is None:
      seqno = nextSeqno( prefixListEntry )
      # check if there is a larger sequence number in the remarks
      if isinstance( mode, ( Globals.IpPrefixListMode,
                             Globals.Ipv6PrefixListMode ) ):
         seqno = max( seqno, nextSeqno( mode.allRemarkSeqnos ) )
      if seqno > PREFIX_SEQNO_MAX_VALUE:
         mode.addError( "Sequence number exhausted" )
         return ( False, hashVal, seqno )
   else:
      # no overwriting existing rule unless it's the same.
      # If it was the same we should have returned earlier. So just return an
      # error if the seqno specified already exists
      if seqno in prefixListEntry:
         mode.addError( f"Duplicate sequence number: {seqno}" )
         return ( False, hashVal, seqno )
   return ( True, hashVal, seqno )

class PrefixListMaskConstraints:
   """ This object is solely used to test for an exact match
   between masks of a given prefix entry and a cli query
   """
   def __init__( self, eq=None, ge=None, le=None ):
      """
         eq, ge, le -- U32 value denoting mask length
      """
      # pylint: disable-next=singleton-comparison
      def defaultToZero( v ): return v if v != None else 0
      self.eq = defaultToZero( eq )
      self.ge = defaultToZero( ge )
      self.le = defaultToZero( le )
      if self.eq:
         self.le = self.ge = self.eq

   def matches( self, prefixEntry ):
      """
         prefixEntry -- Acl::(Ipv6)PrefixEntry

         Returns True if all masks between
         self and given prefixEntry match
      """
      return prefixEntry.le == self.le and \
             prefixEntry.ge == self.ge and \
             prefixEntry.eq == self.eq

def findPfxEntry( pfxList, targetPfx, cliMasks, permit, ipv6=False ):
   """
      pfxList   -- Acl::PrefixEntry ordered collected, keyed by sequence number
      targetPfx -- "ip/subnet" for ipv4; Arnet::Ip6Prefix for ipv6
      permit    --  Acl::MatchPermit
      cliMasks  -- PrefixListMaskConstraints object
      ipv6      -- boolean denoting ip/ipv6 mode

      Returns the sequence number of the entry on success else None
   """
   if targetPfx is None:
      return None

   if not ipv6:
      addr, mask = targetPfx.split( "/" )
      targetPfx = Tac.newInstance( "Arnet::Prefix", addr, int( mask ) )

   pfxEntryTable = pfxList.ipv6PrefixEntry if ipv6 else pfxList.prefixEntry
   for seq, entry in pfxEntryTable.items():
      pfx = entry.ip6Prefix if ipv6 else entry.prefix
      if pfx == targetPfx:
         # ensure all given and expected masks match
         if cliMasks.matches( entry ) and entry.permit == matchPermitEnum[ permit ]:
            return seq
   return None

def removePrefixEntryHash( mode, pfxEntry, seqno, ipv6 ):
   """
      mode     -- current cli mode
      pfxEntry -- Acl::PrefixEntry object to remove
      seqno    -- U32
      ipv6     -- boolean denoting ip/ipv6 mode

      Removes prefix hash from prefix mode config if it exists.
   """
   pfx = pfxEntry.ip6Prefix if ipv6 else pfxEntry.prefix
   hashVal = getPrefixEntryHashVal( pfx,
                                    pfxEntry.permit,
                                    pfxEntry.ge,
                                    pfxEntry.le )
   if seqno in mode.modSeqnos:
      mode.modSeqnos.remove( seqno )
   if hashVal in mode.prefixEntryHashVals:
      mode.prefixEntryHashVals.remove( hashVal )

def removePrefixListEntry( mode, seqno=None, prefix=None, permit=None,
                           listName=None, ipv6=False, eqLen=None,
                           geLen=None, leLen=None ):
   """
      mode          -- current cli mode
      seqno         -- U32
      prefix        -- "ip/subnet" string for ipv4; Arnet::ip6Prefix for ipv6
      permit        -- Acl::MatchPermit
      listName      -- string
      ipv6          -- boolean denoting ip/ipv6 mode
      eqLen, geLen  -- U32 indicating mask len
      leLen

      Removes a prefix entry from given prefix list keyed by prefix
      or sequence number.
   """
   if ipv6:
      prefixMode = Globals.Ipv6PrefixListMode
      pfxListMap = Globals.aclListConfig.ipv6PrefixList
   else:
      prefixMode = Globals.IpPrefixListMode
      pfxListMap = Globals.aclListConfig.prefixList

   cliMasks = PrefixListMaskConstraints( eqLen, geLen, leLen )

   # get prefix list
   if isinstance( mode, prefixMode ):
      pfxList = mode.pendingConfig
   else:
      pfxList = pfxListMap.get( listName )
      if pfxList is None:
         return

   # If the seqno points to a remark, delete the remark. This functionality exists
   # in the prefix handler because the syntax for this command, no seq [seqno] can
   # only be used by one handler, and is used by the prefix handler. In order to
   # key the user interface consistent, the deletion of remark entries is hosted
   # here.
   if isinstance( mode, prefixMode ) and seqno in mode.allRemarkSeqnos:
      remarkList = mode.pendingRemarkList
      del remarkList.remarkEntries[ seqno ]
      mode.allRemarkSeqnos.remove( seqno )
      return

   # get seqno if not provided
   if seqno is None:
      # only need to check for permit match when no seqno is provided
      seqno = findPfxEntry( pfxList, prefix, cliMasks, permit, ipv6 )

   # delete the entry
   prefixEntryTable = pfxList.ipv6PrefixEntry if ipv6 else pfxList.prefixEntry
   if seqno is not None and seqno in prefixEntryTable:
      pfxEntry = prefixEntryTable[ seqno ]
      del prefixEntryTable[ seqno ]
      if isinstance( mode, prefixMode ):
         mode.allPrefixSeqnos.remove( seqno )
         removePrefixEntryHash( mode, pfxEntry, seqno, ipv6 )
         return

      pfxList.version += 1

   # delete empty list
   if listName and not prefixEntryTable:
      if not ipv6 and pfxList.inPrefixListMode:
         Globals.aclListConfig.prefixListModeCount -= 1
      del pfxListMap[ listName ]

def removePrefixListEntryHandler( mode, args ):
   seqno = args.get( 'SEQNO' )
   prefix = args.get( 'PREFIX' )
   permit = args.get( 'ACTION' )
   listName = args.get( 'NAME' )
   ipv6 = False
   eqLen = args.get( 'EQLEN' )
   geLen = args.get( 'GELEN' )
   leLen = args.get( 'LELEN' )

   # Remove all entries for a given prefix-list name,
   # if listName is the only not None argument.
   if seqno is None and prefix is None and permit is None:
      removePrefixList( mode, listName, ipv6 )
   else:
      removePrefixListEntry( mode, seqno, prefix, permit,
            listName, ipv6, eqLen, geLen, leLen )

def removeIpv6PrefixListEntryHandler( mode, args ):
   seqno = args.get( 'SEQNO' )
   prefix = args.get( 'PREFIX' )
   permit = args.get( 'ACTION' )
   listName = args.get( 'NAME' )
   ipv6 = True
   eqLen = args.get( 'EQLEN' )
   geLen = args.get( 'GELEN' )
   leLen = args.get( 'LELEN' )

   # Remove all entries for a given prefix-list name,
   # if listName is the only not None argument.
   if seqno is None and prefix is None and permit is None:
      pfxListMap = Globals.aclListConfig.ipv6PrefixList
      if listName not in pfxListMap:
         return
      removePrefixListImportFile( pfxListMap[ listName ], ipv6 )
      del pfxListMap[ listName ]
   else:
      removePrefixListEntry( mode, seqno, prefix, permit,
               listName, ipv6, eqLen, geLen, leLen )

def removeIpv6PrefixListHandler( mode, args ):
   listName = args.get( 'NAME' )
   ipv6 = True
   removePrefixList( mode, listName, ipv6 )

def removePrefixListImportFile( pfxList, ipv6 ):
   '''Remove downloaded copy of imported prefix-list'''
   if pfxList.source == "":
      return
   for scheme in listUrlRemoteSchemes:
      if pfxList.source.startswith( scheme ):
         helper = Tac.newInstance( "Acl::CliHelper" )
         localFile = "%s/%s.cfg" % (
            helper.prefixListEntryLocalDir( ipv6 ), pfxList.name )
         Tac.run( [ "rm", "-f", localFile ], asRoot=True,
                  stdout=Tac.DISCARD, stderr=Tac.DISCARD,
                  ignoreReturnCode=True )
         return

#------------------------------------------------------------------------------
# ip prefix-list <destination_name> { copy|rename } <source_name> [ overwrite ]
#------------------------------------------------------------------------------
def copyOrRenamePfxRmkList( mode, destListName, copyOrRename, srcListName,
                            overwrite=None, ipv6=False ):
   pfxListMap = Globals.aclListConfig.ipv6PrefixList \
      if ipv6 else Globals.aclListConfig.prefixList
   destList = None
   srcList = pfxListMap.get( srcListName )

   rmkListMap = Globals.aclListConfig.ipv6RemarkCollection \
      if ipv6 else Globals.aclListConfig.remarkCollection
   rmkDestList = None
   rmkSrcList = None
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      rmkSrcList = rmkListMap.get( srcListName )

   newListToBeAdded = destListName not in pfxListMap and \
                      destListName not in rmkListMap
   if not srcList and not rmkSrcList:
      errorStr = "Prefix list %s does not exist" % ( srcListName )
      mode.addError( errorStr )
      return
   if overwrite or destListName not in pfxListMap:
      destList = pfxListMap.newMember( destListName )
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled() and \
      ( overwrite or destListName not in rmkListMap ):

      rmkDestList = rmkListMap.newMember( destListName )
   if srcListName == destListName:
      return
   if not destList:
      errorStr = ( "Prefix list %s already exists. Add 'overwrite'"
                   % ( destListName ) )
      mode.addError( errorStr )
      return
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled() and not rmkDestList:
      errorStr = ( "Prefix list %s already exists. Add 'overwrite'"
                   % ( destListName ) )
      mode.addError( errorStr )
      return
   prefixListCopyHandler = Tac.newInstance(
         "Routing::RouteMap::%sPrefixListCopyHandler" % ( "Ipv6" if ipv6 else "" ) )
   remarkListCopyHandler = Tac.newInstance(
         "Routing::RouteMap::RemarkListCopyHandler" )
   # changes made to the attribute filter should be replicated in
   # RouteMap/ConfigSessionPlugin/RouteMap.py
   if srcList:
      prefixListCopyHandler.filteredAttribute = ()
      prefixListCopyHandler.filteredAttribute.attributeName[ 'version' ] = True
      copyTacEntity( destList, srcList, prefixListCopyHandler )

   if RouteMapToggleLib.togglePrefixModeRemarksEnabled() and rmkSrcList:
      remarkListCopyHandler.filteredAttribute = ()
      copyTacEntity( rmkDestList, rmkSrcList, remarkListCopyHandler )

   if not ipv6 and newListToBeAdded:
      Globals.aclListConfig.prefixListModeCount += 1
   if copyOrRename == 'rename':
      if not ipv6 and pfxListMap[ srcListName ].inPrefixListMode:
         Globals.aclListConfig.prefixListModeCount -= 1
      removePrefixListImportFile( pfxListMap[ srcListName ], ipv6 )

      if srcListName in pfxListMap:
         del pfxListMap[ srcListName ]

      if RouteMapToggleLib.togglePrefixModeRemarksEnabled() and \
         srcListName in rmkListMap:

         del rmkListMap[ srcListName ]

def copyOrRenamePrefixListHandler( mode, args ):
   destListName = args.get( 'DESTINATION' )
   copyOrRename = args.get( 'ACTION' )
   srcListName = args.get( 'SOURCE' )
   overwrite = args.get( 'overwrite' )
   ipv6 = False
   copyOrRenamePfxRmkList( mode, destListName, copyOrRename,
         srcListName, overwrite, ipv6 )

#--------------------------------------------------------------------------------
# ipv6 prefix-list <destination_name> { copy|rename } <source_name> [ overwrite ]
#--------------------------------------------------------------------------------
def copyOrRenameIpv6PrefixList( mode, args ):
   destIpv6ListName = args.get( 'DESTINATION' )
   copyOrRename = args.get( 'ACTION' )
   srcIpv6ListName = args.get( 'SOURCE' )
   overwrite = args.get( 'overwrite' )
   copyOrRenamePfxRmkList( mode, destIpv6ListName, copyOrRename, srcIpv6ListName,
                               overwrite, ipv6=True )

def removePrefixList( mode, listName, ipv6=False ):
   pfxListMap = Globals.aclListConfig.ipv6PrefixList \
      if ipv6 else Globals.aclListConfig.prefixList
   remarkListMap = Globals.aclListConfig.ipv6RemarkCollection \
      if ipv6 else Globals.aclListConfig.remarkCollection
   if listName not in pfxListMap:
      return
   if blockPrefixListDeleteIfInUse( mode, listName, ipv6 ):
      return
   if not ipv6 and pfxListMap[ listName ].inPrefixListMode:
      Globals.aclListConfig.prefixListModeCount -= 1
   removePrefixListImportFile( pfxListMap[ listName ], ipv6 )
   del pfxListMap[ listName ]
   del remarkListMap[ listName ]

def refreshPrefixList_( mode, listName, ipv6 ):

   pfxLists = Globals.aclListConfig.prefixList if not ipv6 \
      else Globals.aclListConfig.ipv6PrefixList

   def numEntries( pl ):
      return len( pl.prefixEntry ) if not ipv6 \
         else len( pl.ipv6PrefixEntry )

   def dupActStr( pfxList ):
      if pfxList.duplicateHandling == duplicateOperation.duplicateError:
         return None
      if pfxList.duplicateHandling == duplicateOperation.duplicateOverride:
         return "override"
      assert pfxList.duplicateHandling == duplicateOperation.duplicateIgnore
      return "ignore"

   if not listName:
      for name, pfxList in pfxLists.items():
         if pfxList.source != "":
            dupAct = dupActStr( pfxList )
            if loadPrefixListUrl_( mode, name, pfxList.source, ipv6,
                                   dupAct=dupAct ):
               print( "refreshed %s prefix-list %s source %s" % (
                  ipCmd( ipv6 ), name, pfxList.source ) )
               print( "Num entries: %d" % numEntries( pfxList ) )
   elif listName in pfxLists:
      pfxList = pfxLists[ listName ]
      if pfxList.source == "":
         mode.addError( "prefix list %s does not have a URL configured" % (
            listName ) )
         return False
      dupAct = dupActStr( pfxList )
      if loadPrefixListUrl_( mode, listName, pfxList.source, ipv6,
                             dupAct=dupAct ):
         print( "refreshed %s prefix-list %s source %s" % (
            ipCmd( ipv6 ), listName, pfxList.source ) )
         print( "Num entries: %d" % numEntries( pfxList ) )
   else:
      mode.addError( "prefix-list %s not found" % listName )
      return False
   return True

def prefixListModelFromEntries( prefixes, remarks, listName, isIpv6,
                                duplicateHandling=None ):
   ipPrefixEntries = []
   # extract the prefixes
   for seq, entry in prefixes.items():
      permitVal = matchPermitEnum[ entry.permit ]
      if ( isIpv6 ): # pylint: disable=superfluous-parens
         prefixVal = entry.ip6Prefix.stringValue
      else:
         prefixVal = entry.prefix.stringValue

      geLen = None if entry.eq != 0 else entry.ge
      leLen = None if entry.eq != 0 else entry.le
      eqLen = None if entry.eq == 0 else entry.eq

      ipPrefixEntries.append( IpPrefixEntry( seqno=entry.seqno,
                                             filterType=permitVal,
                                             prefix=Arnet.IpGenPrefix( prefixVal ),
                                             ge=geLen, le=leLen, eq=eqLen ) )

   # extract all the remarks
   ipRemarkEntries = []
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      for seq, remark in remarks.items():
         ipRemarkEntries.append( IpRemarkEntry( seqno=seq, remark=remark ) )

   return IpPrefixList( ipPrefixEntries=ipPrefixEntries,
                        ipRemarkEntries=ipRemarkEntries,
                        duplicateHandling=duplicateHandling )

def prefixListModelByName( config, listName, ipv6=False ):
   if ipv6:
      if listName in config.ipv6PrefixList:
         ipv6PrefixList = config.ipv6PrefixList.get( listName, {} )
         ipv6RemarkList = config.ipv6RemarkCollection.get( listName, {} )
         if ipv6PrefixList.source != "":
            return IpPrefixList( ipPrefixEntries=[],
                                 ipRemarkEntries=[],
                                 ipPrefixListSource=ipv6PrefixList.source )
         # if the list exists, then pass the entries into the model
         dupHandler = None
         ipv6PrefixEntries = {}
         ipv6RemarkEntries = {}
         if ipv6PrefixList:
            dupHandler = ipv6PrefixList.duplicateHandling
            ipv6PrefixEntries = ipv6PrefixList.ipv6PrefixEntry
         if ipv6RemarkList and RouteMapToggleLib.togglePrefixModeRemarksEnabled():
            ipv6RemarkEntries = ipv6RemarkList.remarkEntries
         return prefixListModelFromEntries(
            ipv6PrefixEntries, ipv6RemarkEntries, listName,
            True, duplicateHandling=dupHandler )
      return None

   if listName in config.prefixList:
      prefixList = config.prefixList.get( listName, {} )
      remarkList = config.remarkCollection.get( listName, {} )
      if prefixList.source != "":
         return IpPrefixList( ipPrefixEntries=[],
                              ipRemarkEntries=[],
                              ipPrefixListSource=prefixList.source )

      # if the list exists, then pass the entries into the model
      dupHandler = None
      prefixEntries = {}
      remarkEntries = {}
      if prefixList:
         dupHandler = prefixList.duplicateHandling
         prefixEntries = prefixList.prefixEntry
      if remarkList and RouteMapToggleLib.togglePrefixModeRemarksEnabled():
         remarkEntries = remarkList.remarkEntries
      return prefixListModelFromEntries(
         prefixEntries, remarkEntries, listName, False,
         duplicateHandling=dupHandler )

   return None

def prefixListContainerModel( config, listName=None, ipv6=False, filename=None ):
   ipPrefixLists = {}

   if filename is not None:
      return IpPrefixLists( _isIpv6=ipv6, _filename=filename )

   if listName is None:
      prefixList = config.ipv6PrefixList if ipv6 else config.prefixList
      for name in prefixList:
         prefixListModel = prefixListModelByName( config, listName=name, ipv6=ipv6 )
         if prefixListModel:
            ipPrefixLists[ name ] = prefixListModel
   else:
      prefixListModel = prefixListModelByName( config, listName=listName, ipv6=ipv6 )
      if prefixListModel:
         ipPrefixLists[ listName ] = prefixListModel

   inPrefixListMode = config.prefixListModeCount > 0
   return IpPrefixLists( _isIpv6=ipv6, _inPrefixListMode=inPrefixListMode,
                         ipPrefixLists=ipPrefixLists )

def showPrefixList( mode, args ):
   listName = args.get( 'NAME' )
   filename = args.get( 'filename' )
   return prefixListContainerModel( Globals.aclListConfig, listName=listName,
                                    filename=filename )

def showPrefixListDetail_( mode, listName, ipv6 ):
   '''Show presumably large number of prefix list entries imported
   from a URL source'''
   if ( not ipv6 and not listName in Globals.aclListConfig.prefixList ) or \
      ( ipv6 and not listName in Globals.aclListConfig.ipv6PrefixList ):
      errMsg = "List not found"
      return IpPrefixLists( _errMsg=errMsg )
   pfxList = Globals.aclListConfig.prefixList[ listName ] if not ipv6 \
      else Globals.aclListConfig.ipv6PrefixList[ listName ]
   if pfxList.source == "":
      # Not a URL-loaded prefix list
      print( "! Not an imported prefix-list" )
      args = { 'NAME': listName }
      if not ipv6:
         return showPrefixList( mode, args )
      else:
         return showIpv6PrefixList( mode, args )
   if pfxList.sourceTimestamp == 0:
      # Failed to load
      errMsg = "Failed to import from source %s" % pfxList.source
      return IpPrefixLists( _errMsg=errMsg )
   helper = Tac.newInstance( "Acl::CliHelper" )
   filename = helper.showPrefixList( pfxList ) if not ipv6 \
      else helper.showIpv6PrefixList( pfxList )
   args = { 'NAME': listName, 'filename': filename }
   if not ipv6:
      return showPrefixList( mode, args )
   else:
      return showIpv6PrefixList( mode, args )

def showPrefixListDetail( mode, args ):
   listName = args.get( 'NAME' )
   return showPrefixListDetail_( mode, listName, ipv6=False )

def showPrefixListSummary_( mode, listName, ipv6 ):
   model = PrefixListSummary()
   if ( not ipv6 and not listName in Globals.aclListConfig.prefixList ) or \
      ( ipv6 and not listName in Globals.aclListConfig.ipv6PrefixList ):
      model._invalidListName = True # pylint: disable=protected-access
      return model
   pfxList = Globals.aclListConfig.prefixList.get( listName ) if not ipv6 \
      else Globals.aclListConfig.ipv6PrefixList.get( listName )
   rmkList = Globals.aclListConfig.remarkCollection.get( listName ) if not ipv6 \
      else Globals.aclListConfig.ipv6RemarkCollection.get( listName )

   pfxCount, rmkCount = 0, 0
   if pfxList:
      pfxCount = len( pfxList.prefixEntry ) if not ipv6 \
         else len( pfxList.ipv6PrefixEntry )
   if rmkList and RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      rmkCount = len( rmkList.remarkEntries )

   model.entryCount = pfxCount + rmkCount
   model.pfxListSource = pfxList.source
   model.sourceTimeStamp = float( pfxList.sourceTimestamp )
   return model

def showPrefixListSummary( mode, args ):
   listName = args.get( 'NAME' )
   return showPrefixListSummary_( mode, listName, ipv6=False )

def resequencePfxRmkListHelper( mode, pfxList, start,
                                inc, ipv6, rmkList=None ):
   """
      pfxList   -- Acl::PrefixList for ipv4, Acl::Ipv6PrefixList for ipv6
      rmkList   -- Acl::RemarkList
      start     -- starting sequence number
      inc       -- displacement between sequences
      ipv6      -- boolean denoting ipv4/ipv6 mode


      builds copies of the prefix and remark lists, renumbering all sequences
      starting from "start" with a step of "inc" between them
   """
   t0( 'Creating a resequenced copy of the prefix list' )
   pfxListName = "resequencedList"
   pfxListType = "Acl::Ipv6PrefixList" if ipv6 else "Acl::PrefixList"
   pfxEntryType = "Acl::Ipv6PrefixEntry" if ipv6 else "Acl::PrefixEntry"
   pfxEntries = pfxList.ipv6PrefixEntry if ipv6 else pfxList.prefixEntry
   pfxResequencedList = Tac.newInstance( pfxListType, pfxListName )
   pfxResequencedEntries = pfxResequencedList.ipv6PrefixEntry if ipv6 \
                                  else pfxResequencedList.prefixEntry

   rmkEntries = {}
   rmkResequencedList = {}
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      rmkListName = "resequencedList"
      rmkListType = "Acl::RemarkList"
      rmkEntries = rmkList.remarkEntries
      rmkResequencedList = Tac.newInstance( rmkListType, rmkListName )
      rmkResequencedEntries = rmkResequencedList.remarkEntries

   # clear and update the set of modified sequence numbers
   mode.modSeqnos.clear()
   mode.modRemarkSeqnos.clear()
   mode.allRemarkSeqnos.clear()
   mode.allPrefixSeqnos.clear()

   # get seq no
   rmkSeq = list( rmkEntries )
   pfxSeq = list( pfxEntries )

   # iterate across both remarks and prefixes, assigning the new seqno to
   # the next lowest seqno
   i, j = 0, 0
   stepCount = 0
   while i < len( pfxSeq ) or j < len( rmkSeq ):
      seqno = start + ( inc * stepCount )
      # if out of remarks, assign prefixes and vice-versa.
      if j == len( rmkSeq ):
         entry = pfxEntries[ pfxSeq[ i ] ]
         prefix = entry.ip6Prefix if ipv6 else entry.prefix
         resequencedEntry = Tac.Value( pfxEntryType,
                                       seqno,
                                       prefix,
                                       entry.permit,
                                       entry.ge,
                                       entry.le,
                                       entry.eq )
         pfxResequencedEntries.addMember( resequencedEntry )
         mode.modSeqnos.add( seqno )
         mode.allPrefixSeqnos.add( seqno )
         i += 1
      elif i == len( pfxSeq ):
         rmkResequencedEntries[ seqno ] = rmkEntries[ rmkSeq[ j ] ]
         mode.modRemarkSeqnos.add( seqno )
         mode.allRemarkSeqnos.add( seqno )
         j += 1
      elif pfxSeq[ i ] > rmkSeq[ j ]:
         rmkResequencedEntries[ seqno ] = rmkEntries[ rmkSeq[ j ] ]
         mode.modRemarkSeqnos.add( seqno )
         mode.allRemarkSeqnos.add( seqno )
         j += 1
      else:
         entry = pfxEntries[ pfxSeq[ i ] ]
         prefix = entry.ip6Prefix if ipv6 else entry.prefix
         resequencedEntry = Tac.Value( pfxEntryType,
                                       seqno,
                                       prefix,
                                       entry.permit,
                                       entry.ge,
                                       entry.le,
                                       entry.eq )
         pfxResequencedEntries.addMember( resequencedEntry )
         mode.modSeqnos.add( seqno )
         mode.allPrefixSeqnos.add( seqno )
         i += 1
      stepCount += 1

   return pfxResequencedList, rmkResequencedList

def resequencePrefixList( mode, start=None, inc=None, ipv6=False ):
   """
      start     -- starting sequence number
      inc       -- displacement between sequences
      ipv6      -- boolean denoting ipv4/ipv6 mode

      callback method to renumber all sequences in a prefix list
   """
   start = 10 if start is None else int( start )
   inc = 10 if inc is None else int( inc )
   prefixListMode = Globals.Ipv6PrefixListMode if ipv6 else Globals.IpPrefixListMode
   pfxListConfig = Globals.aclListConfig.ipv6PrefixList \
      if ipv6 else Globals.aclListConfig.prefixList

   if isinstance( mode, prefixListMode ):
      prefixList = mode.pendingConfig
      remarkList = mode.pendingRemarkList
      listName = prefixList.name
      prefixEntries = prefixList.ipv6PrefixEntry if ipv6 else prefixList.prefixEntry
      remarkEntries = remarkList.remarkEntries
      # Check if prefix list is empty
      if not prefixEntries and not remarkEntries:
         t0( 'Prefix list and remark list are empty, nothing to resequence' )
         return True
      # Ensure last sequence number does not go out of range
      prefixListLen = len( prefixEntries )
      remarkListLen = len( remarkEntries )
      lastSeq = start + ( prefixListLen + remarkListLen - 1 ) * inc
      errorMsg = ( "Last sequence is beyond max "
                   "allowed sequence number of %d" % PREFIX_SEQNO_MAX_VALUE )
      if lastSeq > PREFIX_SEQNO_MAX_VALUE:
         mode.addError( errorMsg )
         return False

      reseqPrefixList, reseqRemarkList = resequencePfxRmkListHelper( mode,
                                                                     prefixList,
                                                                     start, inc,
                                                                     ipv6,
                                                                     remarkList )

      # reset url imported prefix list
      if listName in pfxListConfig:
         listSource = pfxListConfig[ listName ].source
         if listSource:
            urlPrefixList = pfxListConfig[ mode.pendingConfig.name ]
            resetPrefixListImport( mode, urlPrefixList )

   # update pending config to point to a resequenced copy of the prefix list
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      mode.pendingRemarkList = reseqRemarkList
   mode.pendingConfig = reseqPrefixList
   t0( 'Prefix list resequence completed successfully' )
   return True

def resequencePrefixListHandler( mode, args ):
   start = args.get( 'START' )
   inc = args.get( 'INC' )
   ipv6 = False
   resequencePrefixList( mode, start, inc, ipv6 )

def resequenceIpv6PrefixListHandler( mode, args ):
   start = args.get( 'START' )
   inc = args.get( 'INC' )
   ipv6 = True
   resequencePrefixList( mode, start, inc, ipv6 )

def handlerRefreshIpPrefixList( mode, args ):
   refreshPrefixList_( mode, args.get( 'LISTNAME' ), ipv6=False )

def communityStandardListHelper( listName, communityPermit, commType,
                                 communityValues=None ):
   # right now we only create a community list if a commset is given
   assert type( commType ) == CommunityType # pylint: disable=unidiomatic-typecheck

   if communityValues is None:
      return
   if commType == CommunityType.communityTypeLarge:
      listType = communityListType.largeCommunityStandard
   elif commType == CommunityType.communityTypeExtended:
      listType = communityListType.extCommunityStandard
   elif commType == CommunityType.communityTypeStandard:
      listType = communityListType.communityStandard
   else:
      assert False, "Unrecognized community type: %s" % ( commType )
   # get or create community list
   communityList = Globals.aclListConfig.communityList
   if listName in communityList:
      community = communityList[ listName ]
      seqno =  max( community.communityListEntry ) + 1
   else:
      community = communityList.newMember( listName )
      seqno = 0

   newEntry = community.communityListEntry.newMember( seqno )
   # translate communities
   for value in communityValues:
      if commType == CommunityType.communityTypeLarge:
         newEntry.community[ getLargeCommValue( value ) ] = True
      elif commType == CommunityType.communityTypeExtended:
         if value == 'lbw any':
            lbwAny = getExtCommTypeValue( 'lbwAny' )
            newEntry.community[ lbwAny ] = True
         elif 'lbw range' in value:
            valueSplit = value.split()
            newEntry.community[ getExtCommValue( 'lbw range ' + \
                                                 valueSplit[ 2 ] ) ] = True
            newEntry.community[ getExtCommValue( 'lbw range ' + \
                                                 valueSplit[ 3 ] ) ] = True
            newEntry.lbwDisplay[ 0 ] = 'range ' + getExtCommLbwDisplayValue( 'lbw ' \
                                                                  + valueSplit[ 2 ] )
            newEntry.lbwDisplay[ 1 ] = 'range ' + getExtCommLbwDisplayValue( 'lbw ' \
                                                                  + valueSplit[ 3 ] )
         else:
            newEntry.community[ getExtCommValue( value ) ] = True
            if 'lbw' in value:
               newEntry.lbwDisplay[ 0 ] = getExtCommLbwDisplayValue( value )
      elif commType == CommunityType.communityTypeStandard:
         newEntry.community[ getCommValue( value ) ] = True
      else:
         assert False, "Unrecognized community type: %s" % ( commType )
   newEntry.permit = matchPermitEnum[ communityPermit ]
   newEntry.listType = listType
   internetVal = getCommValue( "internet" )
   hasDeny = False
   hasPermit = False
   hasMultiCommStatement = False
   newCommunities = set( newEntry.community )
   for idx, entry in community.communityListEntry.items():
      communities = set( entry.community )
      if len( communities ) > 1:
         hasMultiCommStatement = True
      if idx == seqno:
         continue
      if entry.permit == matchPermitEnum[ "permit" ]:
         hasPermit = True
      else:
         hasDeny = True
      if ( ( entry.permit == newEntry.permit ) and
           ( entry.listType == listType ) and
           ( communities == newCommunities ) ):
         del community.communityListEntry[ seqno ]
         return
   newEntry.version = 1
   if communityPermit == "permit":
      if hasMultiCommStatement:
         if seqno == 0:
            community.matchSetOptions = YangMatchSetOptions.yangMatchAll
            return
      elif not hasDeny:
         community.matchSetOptions = YangMatchSetOptions.yangMatchAny
         return
      elif internetVal in newCommunities and not hasPermit:
         community.matchSetOptions = YangMatchSetOptions.yangMatchInvert
         return
   community.matchSetOptions = YangMatchSetOptions.yangMatchUndefined

def communityExpandedListHelper( mode, listName, communityPermit, commType,
                                 commRegex ):
   assert type( commType ) == CommunityType # pylint: disable=unidiomatic-typecheck
   if commType == CommunityType.communityTypeLarge:
      listType = communityListType.largeCommunityExpanded
   elif commType == CommunityType.communityTypeExtended:
      listType = communityListType.extCommunityExpanded
   elif commType == CommunityType.communityTypeStandard:
      listType = communityListType.communityExpanded
   else:
      assert False, "Unrecognized community type: %s" % ( commType )
   communityList = Globals.aclListConfig.communityList
   convertedRegex = convertRegexIndustryStandard( commRegex )
   if not Globals.gv.routeMapHelper.isPosixRegexValid( convertedRegex ):
      mode.addError( 'Bad regular expression' )
      return
   if listName in communityList:
      community = communityList[ listName ]
      seqno = max( community.communityListEntry ) + 1
   else:
      community = communityList.newMember( listName )
      seqno = 0

   for entry in community.communityListEntry.values():
      if ( ( matchPermitEnum[ entry.permit ] == communityPermit ) and
           ( entry.commRegex == convertedRegex ) and
           ( entry.listType == listType ) ):
         return
   # add the entry into the community list
   newEntry = community.communityListEntry.newMember( seqno )
   newEntry.permit = matchPermitEnum[ communityPermit ]
   newEntry.commRegex = convertedRegex
   newEntry.commRegexDisplay = commRegex
   newEntry.listType = listType
   newEntry.version = 1
   
   # set matchSetOptions, "matchSetOptions = ALL" is not supported
   if communityPermit == "permit" and ( seqno == 0 or
         community.matchSetOptions == YangMatchSetOptions.yangMatchAny ):
      community.matchSetOptions = YangMatchSetOptions.yangMatchAny
   else:
      community.matchSetOptions = YangMatchSetOptions.yangMatchUndefined

def communityListEntryRemover( mode, listName, listType, communityPermit,
                               communityValuesOrRegex ):

   commValues = None
   commRegex = None

   if listName not in Globals.aclListConfig.communityList:
      return

   if listType == communityListType.largeCommunityStandard:
      if ( communityPermit is None and
           not communityValuesOrRegex and
           listName in Globals.aclListConfig.communityList ):
         # handling 'no ip ...extcommunity-list ... NAME' to remove entire list
         if blockCommunityListDeleteIfInUse( mode, listName, listType ):
            return
         Globals.aclListConfig.communityList[ listName ].communityListEntry.clear()
         del Globals.aclListConfig.communityList[ listName ]
         return

      commValues = [ getLargeCommValue( value ) for value in communityValuesOrRegex ]
   elif listType == communityListType.extCommunityStandard:
      if ( communityPermit is None and
           not communityValuesOrRegex and
           listName in Globals.aclListConfig.communityList ):
         # handling 'no ip ...extcommunity-list ... NAME' to remove entire list
         if blockCommunityListDeleteIfInUse( mode, listName, listType ):
            return
         Globals.aclListConfig.communityList[ listName ].communityListEntry.clear()
         del Globals.aclListConfig.communityList[ listName ]
         return

      newCommValuesOrRegex = []
      lbwAny = False
      for value in communityValuesOrRegex:
         valueSplit = value.split()
         if len(valueSplit) == 4 and valueSplit[ 0 ] == 'lbw' and \
            valueSplit[ 1 ] == 'range':
            rangeValueFirst = 'lbw range ' + valueSplit[ 2 ]
            rangeValueSecond = 'lbw range ' + valueSplit[ 3 ]
            newCommValuesOrRegex.append(rangeValueFirst)
            newCommValuesOrRegex.append(rangeValueSecond)
         elif valueSplit[ 0 ] == 'lbw' and valueSplit[ 1 ] == 'any':
            lbwAny = True
         else:
            newCommValuesOrRegex.append(value)
      commValues = [ getExtCommValue( value ) for value in newCommValuesOrRegex ]
      if lbwAny:
         commValues.append( getExtCommTypeValue( 'lbwAny' ) )
   elif listType == communityListType.communityStandard:
      if ( communityPermit is None and
           not communityValuesOrRegex and
           listName in Globals.aclListConfig.communityList ):
         # handling 'no ip ...community-list ... NAME' to remove entire list
         if blockCommunityListDeleteIfInUse( mode, listName, listType ):
            return
         Globals.aclListConfig.communityList[ listName ].communityListEntry.clear()
         del Globals.aclListConfig.communityList[ listName ]
         return

      commValues = [ getCommValue( value ) for value in communityValuesOrRegex ]
   elif listType in ( communityListType.largeCommunityExpanded,
                      communityListType.extCommunityExpanded,
                      communityListType.communityExpanded ):
      if ( communityPermit is None and
           communityValuesOrRegex is None and
           listName in Globals.aclListConfig.communityList ):
         # handling 'no ip ...community-list ... NAME' to remove entire list
         if blockCommunityListDeleteIfInUse( mode, listName, listType ):
            return
         Globals.aclListConfig.communityList[ listName ].communityListEntry.clear()
         del Globals.aclListConfig.communityList[ listName ]
         return
      commRegex = communityValuesOrRegex

   def compareValuesOrRegex( entry ):
      if commRegex:
         return entry.commRegexDisplay == commRegex
      if commValues:
         return sorted( entry.community ) == sorted( commValues )
      return False

   communityList = Globals.aclListConfig.communityList
   seqnoToBeDeleted = []
   if listName in communityList:
      community = communityList[ listName ]
      for entry in community.communityListEntry.values():
         if (( matchPermitEnum[ entry.permit ] == communityPermit ) and
             ( entry.listType == listType ) and
             ( compareValuesOrRegex( entry ) )):
            seqnoToBeDeleted.append( entry.seqno )
      if blockCommunityListDeleteIfInUse( mode, listName, listType,
                                          seqnoToBeDeleted ):
         return
      for seqno in seqnoToBeDeleted:
         del community.communityListEntry[ seqno ]
      if len ( community.communityListEntry ) == 0:
         del communityList[ listName ]

def manageCommunityStandardList( mode, listName, communityPermit,
                                 communityValues=None, negate=None ):
   if negate:
      communityListEntryRemover( mode, listName,
                                 communityListType.communityStandard,
                                 communityPermit, communityValues )
   else:
      communityStandardListHelper( listName, communityPermit,
                                   CommunityType.communityTypeStandard,
                                   communityValues=communityValues )

def manageCommunityExpandedList( mode, listName,
                                 communityPermit, commRegex, negate=None ):
   if negate:
      communityListEntryRemover( mode, listName,
                                 communityListType.communityExpanded,
                                 communityPermit, commRegex )
   else:
      communityExpandedListHelper( mode, listName, communityPermit,
                                   CommunityType.communityTypeStandard, commRegex )

def communityListContainerModel( config, commType, listName=None ):
   assert type( commType ) == CommunityType # pylint: disable=unidiomatic-typecheck
   communityLists = {}
   if listName is None:
      for name in config.communityList:
         communityList = Globals.communityListByName( config, commType, name )
         if communityList:
            communityLists[ name ] = communityList
   else:
      communityList = Globals.communityListByName( config, commType, listName )
      if communityList:
         communityLists[ listName ] = communityList

   if commType == CommunityType.communityTypeStandard:
      return IpCommunityLists( ipCommunityLists=communityLists )
   elif commType == CommunityType.communityTypeLarge:
      return IpLargeCommunityLists( ipLargeCommunityLists=communityLists )
   elif commType == CommunityType.communityTypeExtended:
      return IpExtCommunityLists( ipExtCommunityLists=communityLists )
   else:
      assert False, "Unrecognized community type: %s" % ( commType )

def ipCommunityListCmdHandler( mode, args ):
   manageCommunityStandardList(
      mode=mode, listName=args[ 'NAME' ],
      communityPermit=args.get( 'ACTION' ),
      communityValues=args.get( 'communityValue' ),
      negate=CliCommand.isNoOrDefaultCmd( args ) )

def handlerShowIpCommunityListCmd( mode, args ):
   return communityListContainerModel( Globals.aclListConfig,
                                       CommunityType.communityTypeStandard,
                                       listName=args.get( 'NAME' ) )

def ipCommunityListRegexpCmdHandler( mode, args ):
   manageCommunityExpandedList(
      mode=mode, listName=args[ 'NAME' ],
      communityPermit=args.get( 'ACTION' ),
      commRegex=args.get( 'REGEX' ), negate=CliCommand.isNoOrDefaultCmd( args ) )

def manageExtCommunityStandardList( mode, listName, communityPermit,
                                    extCommListVal=None, negate=None ):
   if negate:
      communityListEntryRemover( mode, listName,
                                 communityListType.extCommunityStandard,
                                 communityPermit, extCommListVal )
   else:
      communityStandardListHelper( listName, communityPermit,
                                   CommunityType.communityTypeExtended,
                                   communityValues=extCommListVal )

def manageExtCommunityExpandedList( mode, listName, communityPermit,
                                    commRegex, negate=None ):
   if negate:
      communityListEntryRemover( mode, listName,
                                 communityListType.extCommunityExpanded,
                                 communityPermit, commRegex )
   else:
      communityExpandedListHelper( mode, listName, communityPermit,
                                   CommunityType.communityTypeExtended, commRegex )

def isRange( lbwVal ):
   valueSplit = lbwVal.split()
   if len( valueSplit ) == 4 and valueSplit[ 0 ] == 'lbw' and \
      valueSplit[ 1 ] == 'range':
      return True
   else:
      return False

def isValidRange( lbwVal ):
   valueSplit = lbwVal.split()
   if ':' in valueSplit[ 2 ] and ':' in valueSplit[ 3 ]:
      lbw1Split = valueSplit[ 2 ].split( ':' )
      lbw2Split = valueSplit[ 3 ].split( ':' )
      # check for K,M,G
      bwre = re.compile( r'(\d+|\d+\.\d+)(K|M|G)?$' )
      m1 = bwre.match( lbw1Split[ 1 ] )
      m2 = bwre.match( lbw2Split[ 1 ] )
      multiplier = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
      if m1.groups()[1] and m1.groups()[1] in multiplier:
         lbw1_f = float( m1.groups()[0] ) * multiplier[m1.groups()[1]]
      else:
         lbw1_f = float( m1.groups()[0] )
      if m2.groups()[1] and m2.groups()[1] in multiplier:
         lbw2_f = float( m2.groups()[0] ) * multiplier[m2.groups()[1]]
      else:
         lbw2_f = float( m2.groups()[0] )
      if lbw1Split[ 0 ] != lbw2Split[ 0 ]:
         return False
      elif lbw1_f >= lbw2_f:
         return False
      else:
         return True
   else:
      return False

def ipExtCommunityListRegexpCmdHandler( mode, args ):
   manageExtCommunityExpandedList(
      mode=mode, listName=args[ 'NAME' ],
      communityPermit=args.get( 'ACTION' ),
      commRegex=args.get( 'REGEX' ), negate=CliCommand.isNoOrDefaultCmd( args ) )

def ipExtCommunityListCmdHandler( mode, args ):
   extComms = []
   extComms += args.get( 'RT_VAL', [] )
   extComms += args.get( 'SOO_VAL', [] )
   extComms += args.get( 'LBW_VAL', [] )
   color = args.get( 'COLOR_VAL', [] )
   esi = args.get( 'ESI_VAL', [] )
   ospfRouterId = args.get( 'ROUTER_ID', [] )
   if ( color and getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent ):
      mode.addWarning( 'Color extended community is only supported in '
                       'multi-agent mode' )
   if ( esi and getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent ):
      mode.addWarning( 'ES-Import extended community is only supported in '
                       'multi-agent mode' )
   if ( ospfRouterId and
        getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent ):
      mode.addWarning( 'OSPF Router ID extended community is only supported in '
                       'multi-agent mode' )
      ospfRouterId = []
   extComms += color
   extComms += esi
   extComms += ospfRouterId

   manageExtCommunityStandardList(
      mode=mode, listName=args[ 'NAME' ],
      communityPermit=args.get( 'ACTION' ),
      extCommListVal=extComms, negate=CliCommand.isNoOrDefaultCmd( args ) )

def handlerShowIpExtCommunityListCmd( mode, args ):
   return communityListContainerModel( Globals.aclListConfig,
                                       CommunityType.communityTypeExtended,
                                       listName=args.get( 'NAME' ) )

def manageLargeCommunityStandardList( mode, listName, communityPermit,
                                      largeCommListVal=None, negate=None ):
   if negate:
      communityListEntryRemover( mode, listName,
                                 communityListType.largeCommunityStandard,
                                 communityPermit, largeCommListVal )
   else:
      communityStandardListHelper( listName, communityPermit,
                                   commType=CommunityType.communityTypeLarge,
                                   communityValues=largeCommListVal )

def manageLargeCommunityExpandedList( mode, listName, communityPermit,
                                      commRegex, negate=None ):
   if negate:
      communityListEntryRemover( mode, listName,
                                 communityListType.largeCommunityExpanded,
                                 communityPermit, commRegex )
   else:
      communityExpandedListHelper( mode, listName, communityPermit,
                                   CommunityType.communityTypeLarge, commRegex )

def ipLargeCommunityListCmdHandler( mode, args ):
   manageLargeCommunityStandardList(
      mode=mode, listName=args[ 'NAME' ],
      communityPermit=args.get( 'ACTION' ),
      largeCommListVal=args.get( 'LARGECOMMS' ),
      negate=CliCommand.isNoOrDefaultCmd( args ) )

def ipLargeCommunityListRegexpCmdHandler( mode, args ):
   manageLargeCommunityExpandedList(
      mode=mode, listName=args[ 'NAME' ],
      communityPermit=args.get( 'ACTION' ),
      commRegex=args.get( 'REGEX' ), negate=CliCommand.isNoOrDefaultCmd( args ) )

def handlerShowIpLargeCommunityListCmd( mode, args ):
   return communityListContainerModel( Globals.aclListConfig,
                                       CommunityType.communityTypeLarge,
                                       listName=args.get( 'NAME' ) )

def gotoIpPfxListMode( mode, args ):
   listName = args[ 'NAME' ]
   childMode = mode.childMode( Globals.IpPrefixListMode,
                               listName=listName )
   mode.session_.gotoChildMode( childMode )

def generateIpPfxListModel( pfxListEntries, rmkListEntries, pfxAttr ):
   """
      pfxListEntries -- Dict of seqno and Acl::(Ipv6)PrefixEntry objects
      rmkListEntries -- Dict of seqno and remarks (strings)
      pfxAttr        -- Prefix entry prefix attribute name

      Generates an IpPrefixList model object.
   """
   pfxListEntryModels = [
         IpPrefixEntry( seqno=seqno,
                        filterType=matchPermitEnum[ e.permit ],
                        prefix=getattr( e, pfxAttr ).stringValue,
                        ge=e.ge,
                        le=e.le,
                        eq=e.eq or None )
         for seqno, e in pfxListEntries.items()
      ]

   rmkListEntryModels = []
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      rmkListEntryModels = [
         IpRemarkEntry( seqno=seqno, remark=remark )
         for seqno, remark in rmkListEntries.items()
      ]

   return IpPrefixList( ipPrefixEntries=pfxListEntryModels,
                        ipRemarkEntries=rmkListEntryModels )

def printPfxListStates( mode, pendingOutput, activeOutput, ipv6=False ):
   """
      mode          -- Current CLI mode
      pendingOutput -- pending output write stream
      activeOutput  -- active output write stream
      ipv6          -- boolean indicating ip/ipv6 mode

      Writes current active and pending prefix list states to
      activeOutput and pendingOutput
   """
   if ipv6:
      pfx, modeListNameAttr = "ip6Prefix", "pfxListName"
      pfxListMapAttr, pfxEntryAttr = "ipv6PrefixList", "ipv6PrefixEntry"
      rmkListMapAttr, rmkEntryAttr = "ipv6RemarkCollection", "remarkEntries"
   else:
      pfx, modeListNameAttr = "prefix", "listName"
      pfxListMapAttr, pfxEntryAttr = "prefixList", "prefixEntry"
      rmkListMapAttr, rmkEntryAttr = "remarkCollection", "remarkEntries"

   activePfxListMap = getattr( Globals.aclListConfig, pfxListMapAttr )
   listName = getattr( mode, modeListNameAttr )
   activePfxList = activePfxListMap.get( listName )
   activePfxListEntries = getattr( activePfxList, pfxEntryAttr, {} )

   activeRmkListEntries = {}
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      activeRmkListMap = getattr( Globals.aclListConfig, rmkListMapAttr )
      activeRmkList = activeRmkListMap.get( listName )
      activeRmkListEntries = getattr( activeRmkList, rmkEntryAttr, {} )

   # render active prefix list to activeOutput stream
   if activePfxListEntries or activeRmkListEntries:
      pfxListModel = generateIpPfxListModel( activePfxListEntries,
                                             activeRmkListEntries,
                                             pfx )
      pfxListModel.renderPrefixList( listName, isIpv6=ipv6, output=activeOutput,
                                     inPrefixListMode=True )
   # render pending config to pendingOutput stream
   printPendingPfxList( mode, pendingOutput, ipv6 )

def printPendingPfxList( mode, output=None, ipv6=False ):
   """
      mode   -- Current CLI mode
      output -- Output stream to write to
      ipv6   -- boolean indicating ip/ipv6 mode

      Renders expected final state of prefix list configuration to output stream
   """
   if ipv6:
      pfxEntries = mode.pendingConfig.ipv6PrefixEntry
      pfx, listName = "ip6Prefix", "pfxListName"
   else:
      pfxEntries = mode.pendingConfig.prefixEntry
      pfx, listName = "prefix", "listName"

   rmkEntries = {}
   if RouteMapToggleLib.togglePrefixModeRemarksEnabled():
      rmkEntries = mode.pendingRemarkList.remarkEntries

   pfxListModel = generateIpPfxListModel( pfxEntries, rmkEntries, pfx )
   pfxListModel.renderPrefixList( getattr( mode, listName ),
                                  isIpv6=ipv6,
                                  output=output,
                                  inPrefixListMode=True )

def handlerIpShowPendingPfxListCmd( mode, args ):
   printPendingPfxList( mode )

def handlerIpShowDiffPfxListCmd( mode, args ):
   showDiff( mode, printPfxListStates )

def handlerIpPrefixListModeAbortCmd( mode, args ):
   mode.abort()

def convertInPrefixListMode( mode ):
   for listName in Globals.aclListConfig.prefixList:
      prefixList = Globals.aclListConfig.prefixList[ listName ]
      if not prefixList.inPrefixListMode:
         prefixList.inPrefixListMode = True
         Globals.aclListConfig.prefixListModeCount += 1

def gotoIpv6PfxListMode( mode, args ):
   ipv6ListName = args[ 'NAME' ]
   childMode = mode.childMode( Globals.Ipv6PrefixListMode,
                              ipv6PfxListName=ipv6ListName )
   mode.session_.gotoChildMode( childMode )

ipv6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixValidMatcher( "Ipv6 Prefix",
   overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
ipv6prefixListMaskLenMatcher = CliMatcher.IntegerMatcher( 1, 128,
   helpdesc='Mask length' )

def loadIpv6PrefixListUrlHandler( mode, args ):
   ipv6ListName = args.get( 'NAME' )
   pfxUrl = args.get( 'URL' )
   dupAct = args.get( 'ACTION' )
   vrfName = args.get( 'VRFNAME' )
   loadPrefixListUrl_( mode, ipv6ListName, pfxUrl, ipv6=True,
                       vrfName=vrfName, dupAct=dupAct )

def manageIpv6PrefixList( mode, permit, ip6Prefix,
                          geLen=None, leLen=None, eqLen=None, seqno=None ):
   if ip6Prefix is None:
      mode.addError( "Invalid prefix." )
      return False
   prefixLen = ip6Prefix.len
   geLen = 0 if geLen is None else int( geLen )
   leLen = 0 if leLen is None else int( leLen )
   eqLen = 0 if eqLen is None else int( eqLen )

   if eqLen != 0:
      if eqLen < int( prefixLen ):
         mode.addError( "Error: eq length %s is less than %s" % (
            eqLen, prefixLen ) )
         return False
      geLen = eqLen
      leLen = eqLen
   else:
      if geLen != 0 and geLen < int( prefixLen ):
         mode.addError( "Error: ge length %s is less than %s" % (
               geLen, prefixLen ) )
         return False
      if leLen != 0 and leLen < int( prefixLen ):
         mode.addError( "Error: le length %s is less than %s" % (
               leLen, prefixLen ) )
         return False
      if leLen != 0 and leLen < geLen:
         mode.addError( "Error: le %s is less than ge %s" % (
               leLen, geLen ) )
         return False

   if mode.pendingConfig.name in Globals.aclListConfig.ipv6PrefixList:
      prefixList = Globals.aclListConfig.ipv6PrefixList[ mode.pendingConfig.name ]
      resetPrefixListImport( mode, prefixList )
      resetPrefixEntryHashVals( mode, prefixList )

   prefixList = mode.pendingConfig

   result, hashVal, seqno = checkEntryValidCommon( mode, prefixList.ipv6PrefixEntry,
                                                   permit, ip6Prefix, geLen, leLen,
                                                   seqno )
   if not result:
      return False

   Ip6Prefix = Tac.Type( 'Arnet::Ip6Prefix' )
   try:
      entry = Tac.Value( "Acl::Ipv6PrefixEntry",
                         seqno,
                         Ip6Prefix(ip6Prefix.address, ip6Prefix.len),
                         matchPermitEnum[ permit ],
                         geLen,
                         leLen,
                         eqLen)
   except SystemError:
      mode.addError( 'Not a valid IPv6 Prefix' )
      return False
   if isinstance( mode, Globals.Ipv6PrefixListMode ):
      mode.allPrefixSeqnos.add( seqno )
   prefixList.ipv6PrefixEntry.addMember( entry )
   mode.modSeqnos.add( seqno )
   mode.prefixEntryHashVals.add( hashVal )

def manageIpv6PrefixListHandler( mode, args ):
   permit = args[ 'ACTION' ]
   ip6Prefix = args[ 'PREFIX' ]
   geLen = args.get( 'GELEN' )
   leLen = args.get( 'LELEN' )
   eqLen = args.get( 'EQLEN' )
   seqno = args.get( 'SEQNO' )
   manageIpv6PrefixList( mode, permit, ip6Prefix, geLen, leLen, eqLen, seqno )

def removeIpv6PrefixList( mode, ipv6ListName ):
   # Delete entire prefix list from global config mode
   if ipv6ListName in Globals.aclListConfig.ipv6PrefixList:
      pfxList = Globals.aclListConfig.ipv6PrefixList[ ipv6ListName ]
      removePrefixListImportFile( pfxList, ipv6=True )
      del Globals.aclListConfig.ipv6PrefixList[ ipv6ListName ]

def showIpv6PrefixList( mode, args ):
   ipv6ListName = args.get( 'NAME' )
   filename = args.get( 'filename' )
   return prefixListContainerModel( Globals.aclListConfig, listName=ipv6ListName,
                                    ipv6=True, filename=filename )

def showIpv6PrefixListDetail( mode, args ):
   ipv6ListName = args.get( 'NAME' )
   return showPrefixListDetail_( mode, ipv6ListName, ipv6=True )

def showIpv6PrefixListSummary( mode, args ):
   ipv6ListName = args.get( 'NAME' )
   return showPrefixListSummary_( mode, ipv6ListName, ipv6=True )

def handlerIpv6ShowPendingPfxListCmd( mode, args ):
   printPendingPfxList( mode, ipv6=True )

def handlerIpv6ShowDiffPfxListCmd( mode, args ):
   showDiff( mode, partial( printPfxListStates, ipv6=True ) )

def handlerRefreshIpv6PrefixList( mode, args ):
   refreshPrefixList_( mode, args.get( 'LISTNAME' ), ipv6=True )

def handlerIpv6PrefixListModeAbortCmd( mode, args ):
   mode.abort()

def gotoPeerFilterMode( mode, args ):
   filterName = args.get( 'NAME' )
   # Enter config sub mode for peer filters
   childMode = mode.childMode( Globals.PeerFilterMode,
                               filterName=filterName )
   mode.session_.gotoChildMode( childMode )

def managePeerFilter( mode, matchName, matchValue, accept, seqno=None ):
   # Specify a match rule
   pFilter = mode.pendingConfig
   if seqno is None:
      seqno = nextSeqno( pFilter.mapEntry )
      if seqno > 65535:
         mode.addError( 'Sequence numbers exhausted' )
         return False

   # Find or create pending entry to update...
   if seqno in mode.pendingConfig.mapEntry:
      entry = mode.pendingConfig.mapEntry[ seqno ]
   else:
      entry = mode.pendingConfig.mapEntry.newMember( seqno )
   mode.modSeqnos.add( seqno )

   # Delete any existing rules
   entry.matchRule.clear()

   # Write new rule
   entry.permit = matchPermitEnum[ 'permit' if accept == 'accept' else 'deny' ]
   rule = Tac.Value( 'Routing::RouteMap::MatchRule', matchName )
   setMatchRuleValue( rule, matchValue )
   entry.matchRule.addMember( rule )

def peerFilterEntryRemover( mode, args ):
   seqno = args[ 'SEQ' ]
   # Remove a match rule
   pFilter = mode.pendingConfig
   if not pFilter:
      return
   if seqno in pFilter.mapEntry:
      del pFilter.mapEntry[ seqno ]
      if ( seqno in mode.modSeqnos ): # pylint: disable=superfluous-parens
         mode.modSeqnos.remove( seqno )

def handlerManagePeerFilterCmd( mode, args ):
   managePeerFilter( mode, 'matchAsRange', args[ 'AS_RANGE' ], args[ 'RESULT' ],
         seqno=args.get( 'SEQ' ) )

def showActivePeerFilter( mode ):
   # Handle sub mode's 'show active' command
   pFilter = Globals.peerFilterConfig.peerFilter.get( mode.filterName )
   if pFilter:
      print( 'peer-filter %s' % ( pFilter.name ) )
      printPeerFilter( pFilter )

def showPendingPeerFilter( mode ):
   # Handle sub mode's 'show pending' command
   pFilter = mode.pendingConfig
   print( 'peer-filter %s' % ( pFilter.name ) )
   printPeerFilter( pFilter )

def showDiffPeerFilter( mode ):
   # Handle sub mode's 'show diff' command
   import difflib # pylint: disable=import-outside-toplevel
   from io import StringIO # pylint: disable=import-outside-toplevel
   activeOutput = StringIO()
   pFilter = Globals.peerFilterConfig.peerFilter.get( mode.filterName )
   if pFilter:
      activeOutput.write( 'peer-filter %s\n' % ( pFilter.name ) )
      printPeerFilter( pFilter, output=activeOutput )

   pendingOutput = StringIO()
   pFilter = mode.pendingConfig
   pendingOutput.write( 'peer-filter %s\n' % ( pFilter.name ) )
   printPeerFilter( pFilter, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print( '\n'.join( list( diff ) ) )

def handlerPeerFilterModeConfigCmd( mode, args ):
   if 'active' in args:
      return showActivePeerFilter( mode )
   elif 'pending' in args:
      return showPendingPeerFilter( mode )
   elif 'diff' in args:
      return showDiffPeerFilter( mode )
   assert False, 'Unknown type'
   return None

def setPeerFilterDescription( mode, args ):
   descString = args.get( 'DESCRIPTION' )
   # Add a description to a filter
   pFilter = mode.pendingConfig

   # repetitions of a description are not allowed
   if pFilter.description and descString in list( pFilter.description.values() ):
      return

   nextId = len( pFilter.description )
   pFilter.description[ nextId ] = descString

def noPeerFilterDescription( mode, args ):
   # Remove all descriptions from a filter
   pFilter = mode.pendingConfig
   pFilter.description.clear()

def handlerPeerFilterModeAbortCmd( mode, args ):
   mode.abort()

# no peer-filter <name>
def removePeerFilter( mode, args ):
   filterName = args.get( 'NAME' )
   if filterName in Globals.peerFilterConfig.peerFilter:
      del Globals.peerFilterConfig.peerFilter[ filterName ]

# show peer-filter [<name>]
def filterByName( peerFilter ):
   description = []
   for descKey in peerFilter.description:
      description.append( peerFilter.description[ descKey ] )
   peerFilterEntries = {}
   for seqno in sorted( peerFilter.mapEntry ):
      entry = peerFilter.mapEntry[ seqno ]
      accept = matchAcceptEnum[ entry.permit ]
      matchRules = Globals.getMatchRules( entry, routeCtx=False )
      peerFilterEntries[ seqno ] = PeerFilterEntry( filterType=accept,
         matchRules=matchRules )
   return PeerFilter( description=description, entries=peerFilterEntries )

def peerFilterContainerModel( config, filterName=None ):
   filters = {}
   if filterName is None:
      for name in config.peerFilter:
         filters[ name ] = filterByName( config.peerFilter[ name ] )
   elif filterName in config.peerFilter:
      filters[ filterName ] = filterByName( config.peerFilter[ filterName ] )
   return PeerFilters( peerFilters=filters )

def showPeerFilter( mode, args ):
   filterName = args.get( 'NAME' )
   return peerFilterContainerModel( Globals.peerFilterConfig, filterName=filterName )

def loadAsPathAccessListUrl( mode, listName, asPathUrl, dupAct=None ):
   '''Load as-path access-list from a URL source'''
   if isinstance( asPathUrl, str ):
      asPathUrl = Url.parseUrl( asPathUrl,
                                Url.Context( *Url.urlArgsFromMode( mode ) ),
                                fsFunc=lambda fs: fs.scheme in listUrlSchemes )
   assert isinstance( asPathUrl, Url.Url )

   if dupAct is None:
      dupAct = duplicateOperation.duplicateError
   elif dupAct == "ignore":
      dupAct = duplicateOperation.duplicateIgnore
   else: # override
      assert dupAct == "override"
      dupAct = duplicateOperation.duplicateOverride

   def createInactiveAsPathList():
      asPathAccessLists = Globals.aclListConfig.pathList
      if listName not in asPathAccessLists:
         asPathList = asPathAccessLists.newMember( listName )
         asPathList.source = str( asPathUrl )
         asPathList.duplicateHandling = dupAct

   helper = Tac.newInstance( "Acl::AsPathCliHelper" )

   # if the url specified is a remote URL, it first
   # is downloaded to the router and stored. The filename used is
   # listName.cfg

   failOnError = mode.session_.isInteractive() or mode.session_.isEapiClient()

   if asPathUrl.fs.scheme in listUrlRemoteSchemes:
      localDir = helper.asPathListEntryLocalDir()
      if not os.path.exists( localDir ):
         os.makedirs( localDir )
      filename = "%s/%s.cfg" % ( localDir, listName )
      try:
         asPathUrl.writeLocalFile( asPathUrl, filename )
      except Exception as e:
         print( "Exception hit: failed to download" )
         Logging.log( RouteMapLib.ASPATH_ACCESS_LIST_IMPORT_FAILED,
                      listName, asPathUrl,
                      "Failed to download" )
         createInactiveAsPathList()
         if failOnError:
            mode.addError( "Download as-path access-list %s from %s failed: %s" % (
               listName, asPathUrl, e ) )
         return not failOnError
   else:
      # If the url is a local file, just use it. There is no
      # need to download.
      assert asPathUrl.fs.scheme in listUrlLocalSchemes
      filename = asPathUrl.localFilename()

   # Create a new as-path access list and then call C++ to parse
   # the url file. If the load was successful, its copied to sysdb.
   posixRegexMode = Globals.aclListConfig.regexMode == 'regexModePosix'
   asPathNewList = Tac.newInstance( "Acl::AsPathList", listName )
   asPathNewList.source = str( asPathUrl )

   asPathLoadRetStatus = Tac.newInstance( "Acl::AsPathUrlLoadReturnStatus" )

   helper.loadAsPathListEntries( asPathNewList, filename, dupAct,
                                 posixRegexMode, asPathLoadRetStatus )
   errorStr = asPathLoadRetStatus.errorStr
   warningOnly = asPathLoadRetStatus.warningOnly

   if errorStr != "":
      Logging.log( RouteMapLib.ASPATH_ACCESS_LIST_IMPORT_FAILED,
                   listName, asPathUrl, errorStr )
      if failOnError:
         mode.addError( "Load as-path access-list %s failed: %s" % (
            listName, errorStr ) )
      createInactiveAsPathList()
      return not failOnError

   if warningOnly:
      mode.addWarning( "Some invalid regexes for current regex mode '%s' "
                       "are present. The as-path ACL '%s' is inactive." %
                       ( regexModeEnum[ Globals.aclListConfig.regexMode ],
                       listName ) )

   # Copy list over to Sysdb
   asPathList = Globals.aclListConfig.pathList[ listName ]  \
      if listName in Globals.aclListConfig.pathList else \
      Globals.aclListConfig.pathList.newMember( listName )

   # Save the source for show and refresh purposes
   asPathList.source = str( asPathUrl )
   asPathList.sourceTimestamp = int( time.time() )
   asPathList.duplicateHandling = dupAct

   helper.copyAsPathListEntries( asPathList, asPathNewList )

   # The version number is incremented so that ribd can consume the commit
   asPathList.version += 1
   Logging.log( RouteMapLib.ASPATH_ACCESS_LIST_IMPORT_SUCCEEDED,
                listName, asPathUrl )
   return True

# For a url based as-path list, refresh the given listName.
# If listName is none, then refresh all url based as-path lists

def refreshAsPathAccessListUrl( mode, args ):
   listName = args.get( 'NAME' )
   asPathLists = Globals.aclListConfig.pathList

   def numAsPathAclEntries( listName ):
      return len( listName.pathEntry )

   def dupActStr( asPathList ):
      if asPathList.duplicateHandling == duplicateOperation.duplicateError:
         return None
      if asPathList.duplicateHandling == duplicateOperation.duplicateOverride:
         return "override"
      assert asPathList.duplicateHandling == duplicateOperation.duplicateIgnore
      return "ignore"

   if listName:
      if listName in asPathLists:
         aclAsPathList = asPathLists[ listName ]
         dupAct = dupActStr( aclAsPathList )
         if aclAsPathList.source == "":
            mode.addError( "as-path access list %s does not have a URL configured" %(
                            listName ) )
            return False
         if loadAsPathAccessListUrl( mode, listName, aclAsPathList.source, dupAct ):
            print ( "refreshed ip as-path access-list %s source %s" % (
               listName, aclAsPathList.source ) )
            print( "Num entries: %d" % numAsPathAclEntries( aclAsPathList ) )
      else:
         mode.addError( "as-path access-list %s not found" % ( listName ) )
         return False
   else:
      for name, aclAsPathList in asPathLists.items():
         dupAct = dupActStr( aclAsPathList )
         if aclAsPathList.source:
            if loadAsPathAccessListUrl( mode, name, aclAsPathList.source, dupAct ):
               print( "refreshed ip as-path access-list %s source %s" % (
                  name, aclAsPathList.source ) )
               print( "Num entries: %d" % numAsPathAclEntries( aclAsPathList ) )
   return True

def handlerLoadAsPathAccessListCmd( mode, args ):
   listName = args.get( 'NAME' )
   asPathUrl = args.get( 'URL' )
   dupAct = args.get( 'ACTION' )
   loadAsPathAccessListUrl( mode, listName, asPathUrl, dupAct )

def handlerNoLoadAsPathAccessListCmd( mode, args ):
   Globals.aclListConfig.pathList.clear()

def handlerTagSet( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   tagSetName = args[ 'TAG_SET' ]
   tagSetValue = args.get( 'TAG_VALUE' )
   tagSet = Globals.aclListConfig.tagSet.get( tagSetName )

   # Cache value to see if anything changed
   oldTagValue = None
   if tagSet and len( tagSet.tagValue ) != 0:
      oldTagValue = tagSet.tagValue[ 0 ]

   if no:
      if tagSet:
         tagSet.tagValue.clear()
         if len( tagSet.routeMapEntryCache ) == 0:
            # Only delete when there are no route map entry using
            # this tag list
            del Globals.aclListConfig.tagSet[ tagSetName ]
   else:
      if not tagSet:
         tagSet = Globals.aclListConfig.newTagSet( tagSetName )

      # Update tag value, only 1 tag value is supported
      if len( tagSet.tagValue ) == 0:
         tagSet.tagValue.push( tagSetValue )
      else:
         tagSet.tagValue[ 0 ] = tagSetValue

   # Value has changed, update all referred route map entry in the cache,
   # assuming they have `match tag list <tagSetName>` configured
   if tagSet and oldTagValue != tagSetValue:
      t8( f'Tag list value has changed to {tagSetValue}' )
      ruleName = 'matchTag'
      for key in tagSet.routeMapEntryCache:
         routeMap = Globals.mapConfig.routeMap.get( key.routeMapName )
         if not routeMap:
            # This shouldn't happen, the cache is not up to date for some reason
            t0( f'Could not find route-map {key.routeMapName}' )
            continue

         mapEntry = routeMap.mapEntry.get( key.seqno )
         if not mapEntry:
            # This shouldn't happen, the cache is not up to date for some reason
            t0( f'Could not find RouteMap {key.routeMapName} sequence {key.seqno}' )
            continue

         # Update matchTag rule within map entry
         rule = mapEntry.matchRule.get( ruleName )
         if tagSetValue is None:
            if rule:
               del mapEntry.matchRule[ ruleName ]
               mapEntry.bumpVersionId()
         else:
            rule = Tac.Value( 'Routing::RouteMap::MatchRule', ruleName )
            setMatchRuleValue( rule, tagSetValue )
            mapEntry.matchRule.addMember( rule )
            mapEntry.bumpVersionId()
