#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.RouterGeneralCli import (
      generalConfig,
      ucmpStaticVrfConfigDir,
      RouterGeneralVrfFilteredRibMode,
      getOrCreateFilteredRibCfg,
)

import Tac
import CliCommand

staticConfig = Tac.Type( "Routing::Rib::RoutingProtocol" ).staticConfig
AddressFamily = Tac.Type( "Arnet::AddressFamily" )

def setHfecOnResolvingStatic( mode, args ):
   generalConfig.hfecOnResolvingProto.add( staticConfig )

def noHfecOnResolvingStatic( mode, args ):
   generalConfig.hfecOnResolvingProto.remove( staticConfig )

def bfdMapAdminDownDownHandler( mode, args ):
   generalConfig.bfdMapAdminDownDown = True

def noBfdMapAdminDownDownHandler( mode, args ):
   generalConfig.bfdMapAdminDownDown = False

def setIpOverIpHfec ( mode, args ):
   generalConfig.ipOverIpHfec = True

def noIpOverIpHfec ( mode, args ):
   generalConfig.ipOverIpHfec = False

#--------------------------------------------------------------------------------
# "[ no | default ] route (static|user) ucmp forwarding \
#                   [ fec maximum-size MAXUCMP [ deviation DEVIATION percent ] ]"
#--------------------------------------------------------------------------------
def ucmpModeCmdHandler( mode, args ):
   if mode.vrfName not in ucmpStaticVrfConfigDir.vrfConfig:
      ucmpStaticVrfConfigDir.vrfConfig.newMember( mode.vrfName )
   ucmpConfig = ucmpStaticVrfConfigDir.vrfConfig[ mode.vrfName ]
   ucmpConfig.ucmpEnabled = 'isTrue'
   ucmpConfig.maxUcmp = args.get( 'MAXUCMP', ucmpConfig.maxUcmpInvalid )
   ucmpConfig.ucmpDeviation = args.get( 'DEVIATION',
                                        ucmpConfig.ucmpDeviationInvalid )
def ucmpModeCmdNoDefaultHandler( mode, args ):
   del ucmpStaticVrfConfigDir.vrfConfig[ mode.vrfName ]

def _routeTrackNextEpoch():
   return generalConfig.routeTrackConfig.epoch + 1

RouteTrackConfig = Tac.Type( 'Routing::General::RouteTrackConfig' )

def routeTrackHandler( mode, args ):
   prefix = args[ 'PREFIX' ]
   if not prefix.validAsPrefix:
      mode.addError( f'{prefix} is not a valid prefix' )
      return
   if ( generalConfig.routeTrackConfig.vrfName == args[ 'VRF' ] and
        generalConfig.routeTrackConfig.prefix == prefix ):
      return
   generalConfig.routeTrackConfig = RouteTrackConfig( args[ 'VRF' ], prefix,
                                                      _routeTrackNextEpoch() )

def routeTrackNoCmdHandler( mode, args ):
   if ( not generalConfig.routeTrackConfig.vrfName and
        not generalConfig.routeTrackConfig.prefix ):
      return
   generalConfig.routeTrackConfig = RouteTrackConfig.emptyWithEpoch(
      _routeTrackNextEpoch() )

def clearTrackingHandler( mode, args ):
   config = generalConfig.routeTrackConfig
   generalConfig.routeTrackConfig = RouteTrackConfig(
      config.vrfName,
      config.prefix,
      _routeTrackNextEpoch() )

#--------------------------------------------------------------------------------
# "[ no | default ] unicast-rib ( ipv4 | ipv6 ) RIB_NAME"
# in "router general" VRF submode
#--------------------------------------------------------------------------------
def filteredRibHandler( mode, args ):
   if 'ipv4' in args:
      af = AddressFamily.ipv4
   else:
      af = AddressFamily.ipv6
   filteredRibName = args[ 'RIB_NAME' ]

   vrfName = None
   filteredRib = None
   # Check if there is an existing filteredRib of the same name
   for vrfConfig in generalConfig.vrfConfig.values():
      for ribName in vrfConfig.filteredRib:
         if ribName == filteredRibName:
            vrfName = vrfConfig.name
            filteredRib = vrfConfig.filteredRib[ ribName ]
            break

   # Only enter the submode if there is no existing filtered rib of the same name
   # but in different VRF or in different AF. 
   if ( not filteredRib or
        ( filteredRib and vrfName == mode.vrfName and filteredRib.af == af ) ):
      # Create the filtered Rib config here so we have an empty filtered rib entry
      # even if the user immediately exits the submode
      getOrCreateFilteredRibCfg( mode.vrfName, af, filteredRibName )
      childMode = mode.childMode( RouterGeneralVrfFilteredRibMode,
                                  vrfName=mode.vrfName,
                                  addrFamily=af,
                                  filteredRibName=filteredRibName )
      mode.session_.gotoChildMode( childMode )
   else:
      mode.addError( 'There is an existing unicast-rib with same name' )

def filteredRibNoCmdHandler( mode, args ):
   if 'ipv4' in args:
      af = AddressFamily.ipv4
   else:
      af = AddressFamily.ipv6
   filteredRibName = args[ 'RIB_NAME' ]
   vrfConfig = generalConfig.vrfConfig.get( mode.vrfName )
   filteredRib = vrfConfig.filteredRib.get( filteredRibName )
   if filteredRib and filteredRib.af == af:
      del vrfConfig.filteredRib[ filteredRibName ]

#--------------------------------------------------------------------------------
# "[ no | default ] match rcf FUNCTION
# in "router general" VRF unicast-rib submode
#--------------------------------------------------------------------------------
def filteredRibMatchRcfHandler( mode, args ):
   # Currently only supports 1 'match rcf FUNCTION' config per Filtered RIB,
   # an empty rcfName indicates there is no match rcf config for the RIB.
   if CliCommand.isNoOrDefaultCmd( args ):
      rcfName = ''
   else:
      # The config model stores the function name, but it is entered with
      # parenthesis. Strip them off before storing.
      rcfName = args[ 'FUNCTION' ][ : -2 ]
   mode.updateRcfName( rcfName )
