# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
from CliPlugin.RoutingBgpCli import (
   configForVrf,
   bgpNeighborConfig
)
import CliPlugin.RoutingBgpNeighborCli as Globals
from BgpLib import (
   afActivationAttrs,
)
from IpLibConsts import DEFAULT_VRF
import Tac

bgpConfig = None
bgpVrfConfigDir = None

MetricOutStateEnum = Tac.Type( "Routing::Bgp::MetricOutState" )

def delNeighborConfigIfDefault( peerKey, vrfName ):
   '''Function to delete neighborConfig if all attributes have default
   values

   If all the attributes of a neighborConfig are default, then the
   neighbor should be deleted. This can happen as the result of a
   setAttr call, if the value it is set to is the default. In other
   words, it's possible for a set call to have the effect of deleting
   a neighbor config.

   This should be called after any set or clear operation on a
   NeighborConfig attribute.
   '''
   config = configForVrf( vrfName )
   peerConfig = bgpNeighborConfig( peerKey, vrfName, create=False )
   if not peerConfig or peerConfig.isPeerGroup or peerConfig.isPeerGroupPeer:
      return
   allAttrs = set( i.name for i in peerConfig.tacType.attributeQ if i.writable )
   attrsWithPresent = set( i for i in allAttrs
                           if hasattr( peerConfig, i + 'Present' ) )
   attrsWithPresent.add( 'timers' )
   attrsWithPresent.add( 'metricOut' )
   for attr in attrsWithPresent:
      if attr == 'metricOut':
         if ( getattr( peerConfig, 'metricOutState' ) !=
              getattr( peerConfig, 'metricOutStateDefault' ) ):
            return
      elif getattr( peerConfig, attr + 'Present' ):
         return
   if peerConfig.commandTagId != peerConfig.commandTagIdDefault:
      return
   for attr in afActivationAttrs:
      if getattr( peerConfig, attr ) != Globals.PeerAfStateEnum.afDefault:
         return
   if ( peerConfig.apSend or peerConfig.apSendIPv4Uni or
        peerConfig.apSendIPv6Uni or
        peerConfig.apSendIPv4LabeledUni or
        peerConfig.apSendIPv6LabeledUni or
        peerConfig.apSendEvpn or
        peerConfig.apSendVpnV4 or peerConfig.apSendVpnV6 or
        peerConfig.apSendDps ):
      return
   if peerConfig.maxRoutesAfiSafiConfig:
      return
   if peerConfig.maxAcceptedRoutesAfiSafiConfig:
      return
   if peerConfig.maxAdvRoutesAfiSafiConfig:
      return
   if ( peerConfig.routeTargetExportFilterDisabled or
        peerConfig.routeTargetExportFilterDisabledAf ):
      return
   if ( peerConfig.missingPolicyActionIn !=
        getattr( peerConfig, 'missingPolicyActionInvalid' ) or
        peerConfig.missingPolicyActionOut !=
        getattr( peerConfig, 'missingPolicyActionInvalid' ) or
        peerConfig.missingPolicyActionAfIn or
        peerConfig.missingPolicyActionAfOut ):
      return
   if ( peerConfig.rtMembershipDefaultRouteTarget !=
        peerConfig.rtMembershipDefaultRouteTargetDefault or
        peerConfig.rtMembershipDefaultRouteTargetEncoding !=
        peerConfig.rtMembershipDefaultRouteTargetEncodingDefault ):
      return
   del config.neighborConfig[ peerConfig.key ]

def resetPresents( config ):
   def isDefault( attr ):
      if getattr( config, attr ) == getattr( config, attr + 'Default' ):
         return True
      return False
   allAttrs = set( i.name for i in config.tacType.attributeQ if i.writable )
   attrsWithPresent = set( i for i in allAttrs
                           if hasattr( config, i + 'Present' ) )
   attrsWithPresent.add( 'timers' )
   attrsWithPresent.add( 'metricOut' )
   # By default, BGP requests BFD sessions without specifying intervals
   # This default behavior is configured by setting bfdIntervalConfigPresent to False
   # Therefore, bfdIntervalConfig is always non-default when it is present
   attrsWithPresent.remove( 'bfdIntervalConfig' )

   for attr in attrsWithPresent:
      if attr == 'metricOut':
         if isDefault( attr ):
            setattr( config, 'metricOutState',
                     MetricOutStateEnum.metricNotConfigured )
      elif getattr( config, attr + 'Present' ):
         if attr == 'timers':
            if isDefault( 'keepaliveTime' ) and isDefault( 'holdTime' ):
               setattr( config, attr + 'Present', False )
         else:
            if isDefault( attr ):
               setattr( config, attr + 'Present', False )

   for attr in afActivationAttrs:
      if isDefault( attr ):
         setattr( config, attr, Globals.PeerAfStateEnum.afDefault )

def handlerConfigBgpNoEqualsDefaultCmd( mode, args ):
   # If ned-mode is already true, we don't want to cause churn in the system
   if bgpConfig.noEqualsDefaultMode:
      return
   bgpConfig.noEqualsDefaultMode = True
   # For all BGP peers, reset all explicit-no attributes to default values
   for peer in bgpConfig.neighborConfig.values():
      resetPresents( peer )
      delNeighborConfigIfDefault( peer.key, vrfName=DEFAULT_VRF )
   for vrfConfig in bgpVrfConfigDir.vrfConfig.values():
      for peer in vrfConfig.neighborConfig.values():
         resetPresents( peer )
         delNeighborConfigIfDefault( peer.key, vrfName=vrfConfig.name )

def noOrDefaultHandlerConfigBgpNoEqualsDefaultCmd( mode, args ):
   bgpConfig.noEqualsDefaultMode = False

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global bgpVrfConfigDir

   bgpConfig = ConfigMount.mount( entityManager, 'routing/bgp/config',
                                 'Routing::Bgp::Config', 'w' )
   bgpVrfConfigDir = ConfigMount.mount( entityManager, 'routing/bgp/vrf/config',
                                       'Routing::Bgp::VrfConfigDir', 'w' )
