#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in
# pylint: disable=consider-using-f-string

import re
import Arnet
from ArnetLib import (
   bgpFormatAsn,
   formatRd,
)
import BasicCliUtil
from BgpLib import (
   addressFamilies,
   bgpConfigAttrsAfMap,
   cleanupAsnConfig,
   NoOrDefault,
   routeTargetToExtCommU64Value,
   vpnAfTypeMapInv,
   updateLsImportCfg,
   updateLsProducerRoleCfg,
   removeLsImportCfg,
   setLsProtoRoleCfg,
   YangSourceEnum,
   YangSources,
)
import CliCommand
from CliParser import AlreadyHandledError
from CliPlugin import AclCli
from CliPlugin import IraIpCli
from CliPlugin import MplsCli
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.IraVrfCli import addAgentVrfEntry
from CliPlugin.RoutingBgpCli import (
   addPathSendConfig,
   BgpCmdBaseClass,
   BgpNEDCmdBaseClass,
   bgpNeighborConfig,
   configForVrf,
   configModeCmdForAf,
   conflictingRcfConfigs,
   conflictingRouteInstallMapConfigs,
   conflictingRouteMapConfigs,
   conflictingRouteTargetConfigs,
   deleteBgpDefaultVrfConfig,
   deleteRouterBgpMacVrfHook,
   deleteRouterBgpUcmpVrfHook,
   deleteRouterBgpVrfHook,
   deleteRouterBgpAfiSafiHook,
   emitWarningIfOrrNotSupportedInRibdMode,
   emitWarningIfRfdNotSupportedInRibdMode,
   TpFieldSetIpv4Mode,
   TpFieldSetIpv6Mode,
   VspFieldSetIpv4Mode,
   VspFieldSetIpv6Mode,
   getBgpRouteDistinguisherInput,
   getBgpRouteDistinguisherRemoteDomainInput,
   getConflictingAttrsForAf,
   getExplicitDefaultTristate,
   getExplicitDefaultValue,
   getTimeInSeconds,
   getVpnTypeAttr,
   haveConflictingRedistInternal,
   maybeWarnConflictingRedist,
   MissingPolicyConfig,
   redistributeConfig,
   RouterBgpBaseAfEvpnMode,
   RouterBgpBaseAfIpMulticastMode,
   RouterBgpBaseAfIpv6MulticastMode,
   RouterBgpBaseAfIpUniMode,
   RouterBgpBaseAfIpv6UniMode,
   RouterBgpBaseAfLabelV4Mode,
   RouterBgpBaseAfLabelV6Mode,
   RouterBgpBaseAfLinkStateMode,
   RouterBgpBaseAfSrTeMode,
   RouterBgpBaseAfDpsMode,
   RouterBgpBaseMode,
   RouterBgpDefaultVrfAfMode,
   RouterBgpDefaultVrfMode,
   RouterBgpOrrPositionMode,
   RouterBgpRfdPolicyMode,
   RouterBgpRouteDistinguisherMode,
   RouterBgpVrfAfIpMulticastMode,
   RouterBgpVrfAfIpMode,
   RouterBgpVrfAfIp6Mode,
   RouterBgpVrfAfIpv6MulticastMode,
   RouterBgpVrfMode,
   setServiceAclCommon,
   switchRedistInternalConfigToAfMode,
   switchRedistInternalConfigToAllAfMode,
   TrafficPolicyFieldSetMappingsMode,
   VrfSelectionPolicyFieldSetMappingsMode,
   ucmpConfigForVrf,
   validateRouteTargetCommand,
   vpnTypeTokenAllowedInMode,
)
import CliPlugin.RoutingBgpInstanceCli as Globals
from CliPlugin.RoutingBgpInstanceCli import (
   nhResRibsConfigHelper,
   resetBgpAfModeConfig,
   setNhResRibsUdpTunnelConfig,
)
import ConfigMount
from IpLibConsts import (
   DEFAULT_VRF,
   VRFNAMES_RESERVED,
)
from IpLibTypes import ProtocolAgentModelType
import LazyMount
from ProtoEnums import protoDict
from RouteMapLib import (
   isAsdotConfigured,
   getCommValue,
)
import Tac
from TypeFuture import TacLazyType

AggregateAddressType = TacLazyType( 'Routing::Bgp::AggregateAddress' )
AsnNotation = TacLazyType( 'Routing::AsnNotation' )
AsNumKey = TacLazyType( "Arnet::AutonomousSystem" )
AutoAggregateDomainType = TacLazyType( 'Routing::Bgp::AutoAggregateDomain' )
ExtendedNextHopCapabilityEnum = TacLazyType(
   "Routing::Bgp::ExtendedNextHopCapability" )
PeerFieldSetFilter = TacLazyType( 'Routing::Bgp::PeerFieldSetFilter' )
RedistInternalStateEnum = TacLazyType( "Routing::Bgp::RedistInternalState" )
ReflectedRouteAttrStateEnum = TacLazyType( "Routing::Bgp::ReflectedRouteAttrState" )
GrHelperRestartTimeClassEnum = TacLazyType(
   "Routing::Bgp::GrHelperRestartTimeClass" )
LuLabelLocalTerminationModeEnum = TacLazyType(
   "Routing::Bgp::LuLabelLocalTerminationMode" )
LabelRangeInfo = TacLazyType( 'Mpls::LabelRangeInfo' )
RfdPolicyType = TacLazyType( "Routing::Bgp::RfdPolicy" )
VrfLocalLabelConfig = TacLazyType( "Routing::Bgp::VrfLocalLabelConfig" )

rdAutoInputDir = None
rdConfigInputDir = None
asnConfig = None
aclStatus = None
aclCheckpoint = None
ucmpVrfConfigDir = None
bgpVrfConfigDir = None
lsImportConfig = None
lsRoleConfig = None

def updateFieldSetBindings(
      coll, fsName, commPresent=None, community=None, peerType=None, groups=None ):
   bindings = coll[ fsName ]
   ipv6 = bindings.ipv6

   # For unspecified args, copy old value
   if commPresent is None:
      commPresent = bindings.communityPresent
   if community is None:
      community = bindings.community
   if peerType is None:
      peerType = bindings.peerType
   if groups is None:
      groups = bindings.groupFilter

   if commPresent is False:
      community = Tac.Value( 'Acl::CommunityValue', 0, 0, 0 )

   if commPresent or ( peerType != PeerFieldSetFilter.inactive ) or groups:
      # Mutually exclusive: only one can be set
      assert ( commPresent ^
               ( peerType != PeerFieldSetFilter.inactive ) ^
               bool( groups ) )

   if not isinstance( groups, dict ):
      # Tacc "sets" are really maps
      groups = { name: True for name in groups }

   coll[ fsName ] = Tac.Value( 'Routing::Bgp::FieldSetBindings', ipv6=ipv6,
                               communityPresent=commPresent, community=community,
                               peerType=peerType, groupFilter=groups )

PEERS_AND_COMMUNITY_ERROR_MSG = \
   ( '\'peers ...\' and \'community ...\' are ' + 'mutually exclusive' )
PEERS_ND_VRF_ERROR_MSG = '\'peers ...\' is not supported in non-default VRF'

def getModuleFunction( functionName ):
   # Returns a function by name from the current module (file).
   # Throws if function does not exist.
   function = globals()[ functionName ]
   if not callable( function ):
      raise NotImplementedError
   return function

# RoutingBgpCli.BgpCmdBaseClass
def BgpCmdBaseClass_handler( commandName, mode, args ):
   BgpCmdBaseClass.callHandler(
      getModuleFunction( commandName + "_handleNormal" ),
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

def BgpCmdBaseClass_noOrDefaultHandler( commandName, mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

# RoutingBgpCli.BgpNEDCmdBaseClass
def BgpNEDCmdBaseClass_handler( commandName, mode, args ):
   BgpNEDCmdBaseClass.callHandler(
      getModuleFunction( commandName + "_handleNormal" ),
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

def BgpNEDCmdBaseClass_noOrDefaultHandler( commandName, mode, args ):
   BgpNEDCmdBaseClass.callNoOrDefaultHandler(
      getModuleFunction( commandName + "_handleNoOrDefault" ),
      mode,
      args )

# class SetLocalAsCmd( BgpCmdBaseClass ):
def SetLocalAsCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asNumber = args[ 'AS_NUM' ]

def SetLocalAsCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   # AS_NUM is optional in the 'no' form of the command, but if it is provided it
   # must match the currently configured AS_NUM.
   asNumber = args.get( 'AS_NUM' )
   if not asNumber or asNumber == config.asNumber:
      config.asNumber = 0
   elif config.asNumber:
      mode.addError( "local AS number %s does not match configuration (%s)" %
                     ( asNumber, config.asNumber ) )

def SetLocalAsCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetLocalAsCmd", mode, args )

def SetLocalAsCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetLocalAsCmd", mode, args )

# class SetFecSkipIarCmd( CliCommandClass ):
def SetFecSkipIarCmd_setBgpFecSkipIar( mode, eventType ):
   config = configForVrf( mode.vrfName )
   if eventType == 'route-changes':
      config.fecSkipIarDivergenceDetection = 'isTrue'
      config.fecSkipIarTracking = 'isInvalid'
   elif eventType == 'drain-undrain':
      # Deprecated using hidden keyword. CL/14166452
      config.fecSkipIarDivergenceDetection = 'isInvalid'
      config.fecSkipIarTracking = 'isInvalid'
   elif eventType == 'all':
      mode.addWarning( 'To make this command effective, clear all the '
         'routes by issuing "clear ip bgp *"' )
      config.fecSkipIarDivergenceDetection = 'isInvalid'
      config.fecSkipIarTracking = 'isTrue'
   else:
      assert False, 'Invalid token %s' % eventType

def SetFecSkipIarCmd_noBgpFecSkipIar( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if config.fecSkipIarTracking == 'isTrue':
      mode.addWarning( 'To make this command effective, clear all the '
         'routes by issuing "clear ip bgp *"' )
   if noOrDefault == NoOrDefault.DEFAULT:
      config.fecSkipIarDivergenceDetection = 'isInvalid'
      config.fecSkipIarTracking = 'isInvalid'
   else:
      config.fecSkipIarDivergenceDetection = 'isFalse'
      config.fecSkipIarTracking = 'isFalse'

def SetFecSkipIarCmd_handler( mode, args ):
   SetFecSkipIarCmd_setBgpFecSkipIar( mode, eventType=args.get( 'IAR_EVENT' ) )

def SetFecSkipIarCmd_noHandler( mode, args ):
   SetFecSkipIarCmd_noBgpFecSkipIar( mode, NoOrDefault.NO )

def SetFecSkipIarCmd_defaultHandler( mode, args ):
   SetFecSkipIarCmd_noBgpFecSkipIar( mode, NoOrDefault.DEFAULT )

# class SetFecDrainUndrainIarCmd( CliCommandClass ):
def SetFecDrainUndrainIarCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecSkipIarDrainUndrain = 'isFalse'
   if 'new-route' in args:
      config.fecDelayNewRouteDuringIar = 'isTrue'
   else:
      config.fecDelayNewRouteDuringIar = 'isInvalid'

def SetFecDrainUndrainIarCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecSkipIarDrainUndrain = 'isTrue'
   config.fecDelayNewRouteDuringIar = 'isFalse'

def SetFecDrainUndrainIarCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecSkipIarDrainUndrain = 'isInvalid'
   config.fecDelayNewRouteDuringIar = 'isInvalid'

# class SetFecPeerInitIarCmd( CliCommandClass ):
def SetFecPeerInitIarCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecSkipIarPeerInit = 'isFalse'

def SetFecPeerInitIarCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecSkipIarPeerInit = 'isTrue'

def SetFecPeerInitIarCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecSkipIarPeerInit = 'isInvalid'

# class SetIarTimeoutCmd( BgpCmdBaseClass ):
def SetIarTimeoutCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fecIarTimeout = args.get( 'TIMEOUT' )

def SetIarTimeoutCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if noOrDefault == NoOrDefault.NO:
      # to disable the timeout, set to 0
      config.fecIarTimeout = config.fecIarTimeoutDisabled
   else:
      config.fecIarTimeout = config.fecIarTimeoutDefault

def SetIarTimeoutCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetIarTimeoutCmd", mode, args )

def SetIarTimeoutCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetIarTimeoutCmd", mode, args )

# class RouteTargetExportFilterDisabledCmd( BgpCmdBaseClass ):
def RouteTargetExportFilterDisabledCmd_setFilterDisabled(
      dummyModeForStubCommandHandler, vpnNlriType, value ):
   config = configForVrf( DEFAULT_VRF )
   if value:
      config.routeTargetExportFilterDisabled[ vpnNlriType ] = True
   else:
      del config.routeTargetExportFilterDisabled[ vpnNlriType ]

def RouteTargetExportFilterDisabledCmd_handleNoOrDefault( mode, args, noOrDefault ):
   vpnNlriType = args[ 'VPN_NLRI_TYPE' ]
   RouteTargetExportFilterDisabledCmd_setFilterDisabled(
      None, vpnNlriType, noOrDefault == NoOrDefault.NO )

def RouteTargetExportFilterDisabledCmd_handleNormal( mode, args ):
   # The BgpCmdBaseClass will always call _handleNoOrDefault with NoOrDefault.NO
   # as the 'disabled' keyword in the syntax of this command is non-optional.
   # However, to make the linter happy with this class inheritance, we need to
   # have a stub for _handleNormal.
   pass

def RouteTargetExportFilterDisabledCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetExportFilterDisabledCmd", mode, args )

def RouteTargetExportFilterDisabledCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RouteTargetExportFilterDisabledCmd", mode, args )

# class SetNeighborDefaultSendCommunity( BgpNEDCmdBaseClass ):
def SetNeighborDefaultSendCommunity_setNeighborDefaultSendCommunity(
      mode, standard=False, large=False, extended=False ):
   config = configForVrf( mode.vrfName )
   config.sendCommunityPresent = True
   config.sendCommunity = not ( standard or large or extended )
   config.sendStandardCommunity = standard or config.sendStandardCommunityDefault
   config.sendLargeCommunity = large or config.sendLargeCommunityDefault
   config.sendExtendedCommunity = extended or config.sendExtendedCommunityDefault

def SetNeighborDefaultSendCommunity_noNeighborDefaultSendCommunity(
      mode, isDefault, standard=False, large=False, extended=False ):
   config = configForVrf( mode.vrfName )
   sendCommunityOnly = not ( standard or large or extended )
   if isDefault:
      if sendCommunityOnly:
         config.sendCommunity = config.sendCommunityDefault
      if standard or sendCommunityOnly:
         config.sendStandardCommunity = config.sendStandardCommunityDefault
      if large or sendCommunityOnly:
         config.sendLargeCommunity = config.sendLargeCommunityDefault
      if extended or sendCommunityOnly:
         config.sendExtendedCommunity = config.sendExtendedCommunityDefault
      config.sendCommunityPresent = ( config.sendCommunity or
                                       config.sendStandardCommunity or
                                       config.sendLargeCommunity or
                                       config.sendExtendedCommunity )
   else:
      if sendCommunityOnly:
         config.sendCommunity = False
      if standard or sendCommunityOnly:
         config.sendStandardCommunity = False
      if large or sendCommunityOnly:
         config.sendLargeCommunity = False
      if extended or sendCommunityOnly:
         config.sendExtendedCommunity = False
      config.sendCommunityPresent = True

def SetNeighborDefaultSendCommunity_handleNormal( mode, args ):
   SetNeighborDefaultSendCommunity_setNeighborDefaultSendCommunity(
      mode, standard='standard' in args,
      large='large' in args,
      extended='extended' in args )

def SetNeighborDefaultSendCommunity_handleNoOrDefault( mode, args, noOrDefault ):
   SetNeighborDefaultSendCommunity_noNeighborDefaultSendCommunity(
      mode, noOrDefault == NoOrDefault.DEFAULT,
      standard='standard' in args,
      large='large' in args,
      extended='extended' in args )

def SetNeighborDefaultSendCommunity_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "SetNeighborDefaultSendCommunity", mode, args )

def SetNeighborDefaultSendCommunity_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "SetNeighborDefaultSendCommunity", mode, args )

# class NeighborDefaultIndexRoutesPeerDisabledCmd( CliCommandClass ):
def NeighborDefaultIndexRoutesPeerDisabledCmd_handler( mode, args ):
   Globals.bgpConfig.indexRoutesPeerDisabled = True

def NeighborDefaultIndexRoutesPeerDisabledCmd_noOrDefaultHandler( mode, args ):
   Globals.bgpConfig.indexRoutesPeerDisabled = False

# class RouterBgpCmd( CliCommandClass ):
def RouterBgpCmd_handler( mode, args ):
   IraIpCli.warnIfRoutingDisabled( mode )
   asNumber = args[ 'AS_NUMBER' ]
   if Globals.bgpConfig.asNumber != 0 and Globals.bgpConfig.asNumber != asNumber:
      mode.addError( "BGP is already running with AS number %s"
                     % bgpFormatAsn( Globals.bgpConfig.asNumber ) )
      return
   if asNumber == 23456:
      mode.addError( "Configuring ASN %d for BGP is disallowed" % 23456 )
      return
   # selective rollback may leave behind stale ucmp config
   if Globals.bgpConfig.asNumber == 0:
      ucmpVrfConfigDir.vrfConfig.clear()
   Globals.bgpConfig.asNumber = asNumber
   addAgentVrfEntry( DEFAULT_VRF, "Bgp" )
   childMode = mode.childMode( RouterBgpBaseMode, asNumber=asNumber )
   mode.session_.gotoChildMode( childMode )

def RouterBgpCmd_noOrDefaultHandler( mode, args ):
   vrfNames = [ DEFAULT_VRF ] + list( bgpVrfConfigDir.vrfConfig )

   # cleanup all bgp vrf service ACL config
   childMode = mode.childMode(
      RouterBgpBaseMode, asNumber=Globals.bgpConfig.asNumber )
   for v in vrfNames:
      vrfMode = RouterBgpDefaultVrfMode if v == DEFAULT_VRF else RouterBgpVrfMode
      subMode = childMode.childMode( vrfMode, vrfName=v )
      for aclType in [ 'ip', 'ipv6' ]:
         setServiceAclCommon( subMode, aclType, no=True )

   # delete all the vrf configs
   for v in vrfNames:
      for hook in deleteRouterBgpVrfHook.extensions():
         hook( v )

   # cleanup asnConfig
   cleanupAsnConfig( asnConfig )

   # cleanup mac vrf config
   for hook in deleteRouterBgpMacVrfHook.extensions():
      hook()

   # delete all the vrf ucmp configs
   for v in ucmpVrfConfigDir.vrfConfig:
      for hook in deleteRouterBgpUcmpVrfHook.extensions():
         hook( v )

   # cleanup ucmp vrf related config
   for hook in deleteRouterBgpUcmpVrfHook.extensions():
      hook( DEFAULT_VRF )

# class ClearAllDisabledCmd( CliCommandClass ):
def ClearAllDisabledCmd_handler( mode, args ):
   Globals.bgpConfig.clearAllDisabled = True

def ClearAllDisabledCmd_noOrDefaultHandler( mode, args ):
   Globals.bgpConfig.clearAllDisabled = False

# class SetFibSyncBatchSizeCmd( CliCommandClass ):
def SetFibSyncBatchSizeCmd_handler( mode, args ):
   batchSize = args.get( 'BATCH_SIZE' )
   config = configForVrf( mode.vrfName )
   fibSyncBatchSize = int( batchSize ) if batchSize else 0
   config.fibSync = Tac.Value( 'Routing::Bgp::FibSync', True, fibSyncBatchSize )

def SetFibSyncBatchSizeCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.fibSync = config.fibSyncDefault

# class BgpLuRibCmd( CliCommandClass ):
def BgpLuRibCmd_handler( mode, args ):
   rib = args.get( 'rib' )
   tunnelRmName = args.get( 'TUNNEL_RM_NAME', '' )
   ipRmName = args.get( 'IP_RM_NAME', '' )
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent and \
      not ( mode.session.inConfigSession() or mode.session.startupConfig() ):
      # in ribd mode interactive session, setting route-map filter or
      # use ipAndTunnel type is not supported.
      if rib == 'ipAndTunnel':
         mode.addError( "Labeled-unicast routes installed in both Tunnel"
                        " RIB and IP RIB is only supported on multi-agent"
                        " mode" )
         return
      if tunnelRmName or ipRmName:
         mode.addError( "Filtering the labeled-unicast RIB by route-map"
                        " is only supported on multi-agent mode" )
         return
   bgpLuRib = Tac.Value( "Routing::Bgp::BgpLuRib", rib, tunnelRmName, ipRmName )
   configForVrf( mode.vrfName ).bgpLuRib = bgpLuRib

def BgpLuRibCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   bgpLuRib = Tac.Value( "Routing::Bgp::BgpLuRib", config.bgpLuRib.ribDefault,
                           config.bgpLuRib.rmNameDefault,
                           config.bgpLuRib.rmNameDefault )
   config.bgpLuRib = bgpLuRib

# class ShutdownCmd( CliCommandClass ):
def ShutdownCmd_handler( mode, args ):
   if not BasicCliUtil.confirm( mode,
                                 "You are attempting to shutdown BGP. "
                                 "Are you sure you want to shutdown? [confirm]" ):
      return
   config = configForVrf( mode.vrfName )
   config.shutdownMsg = args.get( 'MESSAGE', config.shutdownMsgDefault )
   config.shutdown = True

def ShutdownCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.shutdownMsg = config.shutdownMsgDefault
   config.shutdown = False

# class LogNeighborChangesCmd( CliCommandClass ):
def LogNeighborChangesCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.logNeighborChanges = getExplicitDefaultTristate(
      mode, 'logNeighborChanges' )

def LogNeighborChangesCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.logNeighborChanges = 'isFalse'

def LogNeighborChangesCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.logNeighborChanges = 'isInvalid'

# class ClientToClientReflectionCmd( CliCommandClass ):
def ClientToClientReflectionCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.rrDisabled = config.rrDisabledDefault

def ClientToClientReflectionCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.rrDisabled = True

def ClientToClientReflectionCmd_defaultHandler( mode, args ):
   ClientToClientReflectionCmd_handler( mode, args )

# class SetClusterIdCmd( CliCommandClass ):
def SetClusterIdCmd_handler( mode, args ):
   ipAddr = args[ 'CLUSTER_ID' ]
   config = configForVrf( mode.vrfName )
   config.bgpClusterId = ipAddr

def SetClusterIdCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.bgpClusterId = config.bgpClusterIdDefault

# class BgpAggregateCommunityInheritanceLooseCmd( CliCommandClass ):
def BgpAggregateCommunityInheritanceLooseCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.aggregateCommInheritLoose = True

def BgpAggregateCommunityInheritanceLooseCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.aggregateCommInheritLoose = config.aggregateCommInheritLooseDefault

# class SetAllowAsCmd( CliCommandClass ):
def SetAllowAsCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.allowAs = args.get( 'COUNT', config.allowAsEnabledDefault )

def SetAllowAsCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.allowAs = config.allowAsDefault

# class MaintenanceReceiverRouteMapCmd( CliCommandClass ):
def MaintenanceReceiverRouteMapCmd_handler( mode, args ):
   localPref = args.get( 'LOCAL_PREF', Globals.bgpConfig.maintRecvLocalPrefDefault )
   disabled = 'disabled' in args
   Globals.bgpConfig.maintRecvLocalPref = localPref
   Globals.bgpConfig.maintRecvRouteMapDisabled = disabled

def MaintenanceReceiverRouteMapCmd_noOrDefaultHandler( mode, args ):
   MaintenanceReceiverRouteMapCmd_handler( mode, args )

# class BgpRedistInternalCmd( CliCommandClass ):
def BgpRedistInternalCmd_setBgpRedistInternalCommon( mode, no=False ):
   # Prevent configuration of bgp redistribute-internal in both
   # router-bgp and router-bgp-af levels.
   config = configForVrf( mode.vrfName )
   addrFamily = mode.addrFamily
   if haveConflictingRedistInternal( mode, config, addrFamily=addrFamily ):
      return

   attr = bgpConfigAttrsAfMap[ 'bgpRedistInternal' ].get( addrFamily )
   if addrFamily == 'all':
      switchRedistInternalConfigToAllAfMode( config )
   else:
      switchRedistInternalConfigToAfMode( config )

   if no:
      val = RedistInternalStateEnum.disableRedistInt
   else:
      val = RedistInternalStateEnum.enableRedistInt

   setattr( config, attr, val )

def BgpRedistInternalCmd_handler( mode, args ):
   BgpRedistInternalCmd_setBgpRedistInternalCommon( mode )

def BgpRedistInternalCmd_noHandler( mode, args ):
   BgpRedistInternalCmd_setBgpRedistInternalCommon( mode, no=True )

def BgpRedistInternalCmd_defaultHandler( mode, args ):
   BgpRedistInternalCmd_setBgpRedistInternalCommon( mode )

# class BgpSetTimerCmd( CliCommandClass ):
def BgpSetTimerCmd_handler( mode, args ):
   keepalive = args.get( 'KEEPALIVE' )
   hold = args.get( 'HOLD_TIME', 0 )
   minHold = args.get( 'MIN_HOLD_TIME' )
   sendFailureHold = args.get( 'SEND_FAILURE_HOLD_TIME' )
   config = configForVrf( mode.vrfName )

   if keepalive is not None:
      if hold < keepalive:
         mode.addError( 'keepalive time cannot be greater than hold time' )
         return
      if hold != 0 and keepalive == 0:
         mode.addError(
            'keepalive time cannot be zero when hold time is not zero' )
         return
      if minHold is not None and hold != 0 and hold < minHold:
         mode.addError(
            'Hold time cannot be less than minimum hold time.' )
         return
      config.keepaliveTime = getExplicitDefaultValue(
         mode, 'keepaliveTime', keepalive )
      config.holdTime = getExplicitDefaultValue(
         mode, 'holdTime', hold )
   else:
      # Unconditionally set as invalid for all vrfs
      # For default vrf, we always set attributes as invalid. In this case set
      # them as invalid for non-default vrfs as well. Setting value to default
      # sets system default values. Setting to invalid causes it to be inherited
      # from default vrf. We want inheritance hence set as invalid
      config.keepaliveTime = config.keepaliveTimeInvalid
      config.holdTime = config.holdTimeInvalid

   if minHold is not None:
      config.minHoldTime = getExplicitDefaultValue(
      mode, 'minHoldTime', minHold )
   else:
      # Set invalid if it doesn't exits
      config.minHoldTime = config.minHoldTimeInvalid

   if sendFailureHold is not None:
      config.sendFailureHoldTime = getExplicitDefaultValue(
      mode, 'sendFailureHoldTime', sendFailureHold )
   else:
      # Set invalid if it doesn't exits
      config.sendFailureHoldTime = config.sendFailureHoldTimeInvalid

def BgpSetTimerCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   # The "no timers bgp" command operates differently from the
   # "default timers bgp..." command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   if mode.vrfName == DEFAULT_VRF:
      config.keepaliveTime = config.keepaliveTimeInvalid
      config.holdTime = config.holdTimeInvalid
      config.minHoldTime = config.minHoldTimeInvalid
      config.sendFailureHoldTime = config.sendFailureHoldTimeInvalid
   else:
      config.keepaliveTime = config.keepaliveTimeDefault
      config.holdTime = config.holdTimeDefault
      config.minHoldTime = config.minHoldTimeDefault
      config.sendFailureHoldTime = config.sendFailureHoldTimeDefault

def BgpSetTimerCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.keepaliveTime = config.keepaliveTimeInvalid
   config.holdTime = config.holdTimeInvalid
   config.minHoldTime = config.minHoldTimeInvalid
   config.sendFailureHoldTime = config.sendFailureHoldTimeInvalid

# class SetGracefulRestartCmd( CliCommandClass ):
def SetGracefulRestartCmd_configureGlobalGracefulRestart(
      mode, enable=True, noOrDefault=None, addrFamily='all' ):
   config = configForVrf( mode.vrfName )
   if addrFamily == 'ipv4 multicast':
      mode.addError( 'Graceful restart is currently not supported for IPv4 '
                     'multicast mode' )
      return
   if addrFamily == 'ipv6 multicast':
      mode.addError( 'Graceful restart is currently not supported for IPv6 '
                     'multicast mode' )
      return

   triStateVal = None
   if enable:
      triStateVal = 'isTrue'
   else:
      assert noOrDefault
      triStateVal = (
            'isInvalid' if noOrDefault == NoOrDefault.DEFAULT else 'isFalse'
      )

   if addrFamily == 'all':
      if triStateVal == 'isFalse':
         triStateVal = getExplicitDefaultTristate( mode, 'gracefulRestart' )

   attr = bgpConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
   setattr( config, attr, triStateVal )

def SetGracefulRestartCmd_noHandler( mode, args ):
   SetGracefulRestartCmd_configureGlobalGracefulRestart(
         mode,
         enable=False,
         noOrDefault=NoOrDefault.NO,
         addrFamily=mode.addrFamily )

def SetGracefulRestartCmd_defaultHandler( mode, args ):
   SetGracefulRestartCmd_configureGlobalGracefulRestart(
         mode,
         enable=False,
         noOrDefault=NoOrDefault.DEFAULT,
         addrFamily=mode.addrFamily )

def SetGracefulRestartCmd_handler( mode, args ):
   SetGracefulRestartCmd_configureGlobalGracefulRestart(
         mode,
         enable=True,
         noOrDefault=None,
         addrFamily=mode.addrFamily )

# class SetGracefulRestartTimesCmd( CliCommandClass ):
def SetGracefulRestartTimesCmd_setGrRestartTime(
      mode, restartTime, noOrDefault=None ):
   config = configForVrf( mode.vrfName )
   if noOrDefault:
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.grRestartTime = config.grRestartTimeInvalid
      else:
         config.grRestartTime = config.grRestartTimeDefault
   else:
      config.grRestartTime = getExplicitDefaultValue(
            mode,
            'grRestartTime',
            restartTime )

def SetGracefulRestartTimesCmd_setGrStalepathTime(
      mode, stalepathTime, noOrDefault=None ):
   config = configForVrf( mode.vrfName )
   if noOrDefault:
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.grStalepathTime = config.grStalepathTimeInvalid
      else:
         config.grStalepathTime = config.grStalepathTimeDefault
   else:
      config.grStalepathTime = getExplicitDefaultValue(
            mode,
            'grStalepathTime',
            stalepathTime )

def SetGracefulRestartTimesCmd_noOrDefaultGracefulRestartTimes( mode,
                                       noOrDefault,
                                       restartTime=None,
                                       stalepathTime=None ):
   if restartTime:
      SetGracefulRestartTimesCmd_setGrRestartTime(
            mode,
            None,
            noOrDefault=noOrDefault )
   elif stalepathTime:
      SetGracefulRestartTimesCmd_setGrStalepathTime(
            mode,
            None,
            noOrDefault=noOrDefault )

def SetGracefulRestartTimesCmd_noHandler( mode, args ):
   SetGracefulRestartTimesCmd_noOrDefaultGracefulRestartTimes(
         mode,
         NoOrDefault.NO,
         restartTime=args.get( 'restart-time' ),
         stalepathTime=args.get( 'stalepath-time' ) )

def SetGracefulRestartTimesCmd_defaultHandler( mode, args ):
   SetGracefulRestartTimesCmd_noOrDefaultGracefulRestartTimes(
         mode,
         NoOrDefault.DEFAULT,
         restartTime=args.get( 'restart-time' ),
         stalepathTime=args.get( 'stalepath-time' ) )

def SetGracefulRestartTimesCmd_handler( mode, args ):
   if args.get( 'restart-time' ):
      SetGracefulRestartTimesCmd_setGrRestartTime(
            mode,
            args.get( 'SECONDS' ) )
   elif args.get( 'stalepath-time' ):
      SetGracefulRestartTimesCmd_setGrStalepathTime(
            mode,
            args.get( 'SECONDS' ) )

# class SetGracefulRestartHelperCmd( CliCommandClass ):
def SetGracefulRestartHelperCmd_setGrHelper(
      mode, extRestartTime, timeClass, llgrHelper, routeMap, grOptional ):
   config = configForVrf( mode.vrfName )
   config.grHelper = getExplicitDefaultTristate( mode, 'grHelper' )
   if extRestartTime:
      extRestartTime = getTimeInSeconds( extRestartTime, timeClass )
      config.grHelperRestartTime = getExplicitDefaultValue(
            mode, 'grHelperRestartTime', extRestartTime )
      config.grHelperRestartTimeClass = timeClass
   else:
      config.grHelperRestartTime = config.grHelperRestartTimeInvalid
      config.grHelperRestartTimeClass = config.grHelperRestartTimeClassInvalid
   if llgrHelper:
      config.llgrHelper = 'isTrue'
   else:
      config.llgrHelper = 'isInvalid'
   if routeMap:
      config.grHelperRouteMap = \
            Tac.Type( "Ark::TristateString" ).valueSet( routeMap )
   else:
      config.grHelperRouteMap = config.grHelperRouteMapInvalid
   if grOptional:
      config.grHelperRmGrOptional = 'isTrue'
   else:
      config.grHelperRmGrOptional = 'isInvalid'

def SetGracefulRestartHelperCmd_noGrHelper( mode ):
   config = configForVrf( mode.vrfName )
   config.grHelper = 'isFalse'
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   if mode.vrfName == DEFAULT_VRF:
      config.grHelperRestartTime = config.grHelperRestartTimeInvalid
      config.llgrHelper = 'isInvalid'
      config.grHelperRouteMap = config.grHelperRouteMapInvalid
      config.grHelperRestartTimeClass = config.grHelperRestartTimeClassInvalid
      config.grHelperRmGrOptional = config.grHelperRmGrOptionalInvalid
   else:
      config.grHelperRestartTime = config.grHelperRestartTimeDefault
      config.llgrHelper = config.llgrHelperDefault
      config.grHelperRouteMap = config.grHelperRouteMapDefault
      config.grHelperRestartTimeClass = config.grHelperRestartTimeClassDefault
      config.grHelperRmGrOptional = config.grHelperRmGrOptionalDefault

def SetGracefulRestartHelperCmd_defaultGrHelper( mode ):
   config = configForVrf( mode.vrfName )
   config.grHelper = 'isInvalid'
   config.grHelperRestartTime = config.grHelperRestartTimeInvalid
   config.llgrHelper = 'isInvalid'
   config.grHelperRouteMap = config.grHelperRouteMapInvalid
   config.grHelperRestartTimeClass = config.grHelperRestartTimeClassInvalid
   config.grHelperRmGrOptional = config.grHelperRmGrOptionalInvalid

def SetGracefulRestartHelperCmd_handler( mode, args ):
   timeClass = getattr( GrHelperRestartTimeClassEnum,
         args.get( 'UNITS', 'invalid' ) )
   SetGracefulRestartHelperCmd_setGrHelper( mode,
                                             args.get( 'RESTART_TIME' ),
                                             timeClass,
                                             ( 'long-lived' in args ),
                                             args.get( 'MAP_NAME' ),
                                             ( 'optional' in args ) )

def SetGracefulRestartHelperCmd_noHandler( mode, args ):
   SetGracefulRestartHelperCmd_noGrHelper( mode )

def SetGracefulRestartHelperCmd_defaultHandler( mode, args ):
   SetGracefulRestartHelperCmd_defaultGrHelper( mode )

# class BgpHitlessRestartDisabledCmd( CliCommandClass ):
def BgpHitlessRestartDisabledCmd_handler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   if ( getEffectiveProtocolModel( mode.session.mode ) ==
         ProtocolAgentModelType.ribd ):
      mode.addWarning( 'Routing protocols model multi-agent must be '
                        'configured for restart hitless disabled to take effect' )
   config = configForVrf( mode.vrfName )
   if 'disabled' in args:
      config.bgpHitlessRestartDisabled = True
   else:
      config.bgpHitlessRestartDisabled = config.bgpHitlessRestartDisabledDefault

def BgpHitlessRestartDisabledCmd_noOrDefaultHandler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   config = configForVrf( mode.vrfName )
   config.bgpHitlessRestartDisabled = config.bgpHitlessRestartDisabledDefault

# class BgpEnforceAsFirstCmd( CliCommandClass ):
def BgpEnforceAsFirstCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.enforceFirstAs = getExplicitDefaultTristate( mode, 'enforceFirstAs' )

def BgpEnforceAsFirstCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.enforceFirstAs = 'isFalse'

def BgpEnforceAsFirstCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.enforceFirstAs = 'isInvalid'

# class SetAsPathMultiPathRelaxCmd( CliCommandClass ):
def SetAsPathMultiPathRelaxCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asPathMultiPathRelax = getExplicitDefaultTristate(
         mode,
         'asPathMultiPathRelax' )
   config.asPathMultiPathRelaxMatch = args.get(
         'ASPATH_LEN',
         config.asPathMultiPathRelaxMatchDefault )

def SetAsPathMultiPathRelaxCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asPathMultiPathRelax = 'isFalse'
   config.asPathMultiPathRelaxMatch = config.asPathMultiPathRelaxMatchInvalid

def SetAsPathMultiPathRelaxCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asPathMultiPathRelax = 'isInvalid'
   config.asPathMultiPathRelaxMatch = config.asPathMultiPathRelaxMatchDefault

# class IgnoreAsPathCmd( CliCommandClass ):
def IgnoreAsPathCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.ignoreAsPathLen = 'isTrue'

def IgnoreAsPathCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.ignoreAsPathLen = getExplicitDefaultTristate( mode, 'ignoreAsPathLen' )

def IgnoreAsPathCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.ignoreAsPathLen = 'isInvalid'

# class BgpBestPathDomainPath( CliCommandClass ):
def BgpBestPathDomainPath_handler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   config = configForVrf( mode.vrfName )
   config.includeDomainPathLen = 'isTrue'

def BgpBestPathDomainPath_noHandler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   config = configForVrf( mode.vrfName )
   config.includeDomainPathLen = getExplicitDefaultTristate(
      mode, 'includeDomainPathLen' )

def BgpBestPathDomainPath_defaultHandler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   config = configForVrf( mode.vrfName )
   config.includeDomainPathLen = 'isInvalid'

# class SetSkipNhIgpCostCmd( CliCommandClass ):
def SetSkipNhIgpCostCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.skipNextHopIgpCost = 'isTrue'

def SetSkipNhIgpCostCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.skipNextHopIgpCost = (
         getExplicitDefaultTristate( mode, 'skipNextHopIgpCost' )
   )

def SetSkipNhIgpCostCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.skipNextHopIgpCost = 'isInvalid'

# class BgpBestPathOriginAsValidityCmd( BgpCmdBaseClass ):
def BgpBestPathOriginAsValidityCmd_setOriginAsValidity(
      mode, includeOriginAsValidity ):
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "RPKI origin AS validation is only supported in "
                        "multi-agent mode" )
   config = configForVrf( mode.vrfName )
   config.includeOriginAsValidity = includeOriginAsValidity

def BgpBestPathOriginAsValidityCmd_handleNormal( mode, args ):
   BgpBestPathOriginAsValidityCmd_setOriginAsValidity( mode, 'isTrue' )

def BgpBestPathOriginAsValidityCmd_handleNoOrDefault( mode, args, noOrDefault ):
   includeOriginAsValidity = \
      'isInvalid' if noOrDefault == NoOrDefault.DEFAULT else 'isFalse'
   BgpBestPathOriginAsValidityCmd_setOriginAsValidity(
      mode, includeOriginAsValidity )

def BgpBestPathOriginAsValidityCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "BgpBestPathOriginAsValidityCmd", mode, args )

def BgpBestPathOriginAsValidityCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "BgpBestPathOriginAsValidityCmd", mode, args )

# class SetAddpathInstallCmd( CliCommandClass ):
def SetAddpathInstallCmd_handler( mode, args ):
   ecmpPrimary = ( 'ecmp-primary' in args )
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'all':
      config.picIPv4Uni = True
      config.picEcmpPrimaryIPv4Uni = ecmpPrimary
   else:
      attr = bgpConfigAttrsAfMap[ 'pic' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, True )
      attr = bgpConfigAttrsAfMap[ 'picEcmpPrimary' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, ecmpPrimary )

def SetAddpathInstallCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'all':
      config.picIPv4Uni = False
      config.picEcmpPrimaryIPv4Uni = False
   else:
      attr = bgpConfigAttrsAfMap[ 'pic' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, False )
      attr = bgpConfigAttrsAfMap[ 'picEcmpPrimary' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, False )

# class SetAddPathReceiveCmd( CliCommandClass ):
def SetAddPathReceiveCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'all':
      config.apRecv = getExplicitDefaultTristate( mode, 'apRecv' )
   else:
      attr = bgpConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, 'isTrue' )

def SetAddPathReceiveCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, 'isFalse' )

def SetAddPathReceiveCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, 'isInvalid' )

# class SetAddpathSendCmd( CliCommandClass ):
def SetAddpathSendCmd_setAddpathSend(
      mode, app='appNone', pathLimit=0, noOrDefault=None ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = configForVrf( vrfName )
   sendConfig = addPathSendConfig( config, addrFamily )

   if not noOrDefault:
      assert app != 'appNone'
      newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app,
                              enable=True, pathLimit=pathLimit )
      if app not in sendConfig:
         sendConfig.clear()
      sendConfig.addMember( newConfig )
   elif noOrDefault == NoOrDefault.NO:
      # if app is appAny then show run will show the no form of this
      # command with "any" at the end for backward compatibility
      assert app == 'appNone' or app == 'appAny'
      # The following setting is to support config inheritance
      newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app,
                              enable=False )
      if app not in sendConfig:
         sendConfig.clear()
      sendConfig.addMember( newConfig )
   elif noOrDefault == NoOrDefault.DEFAULT:
      sendConfig.clear()

def SetAddpathSendCmd_handler( mode, args ):
   app = 'appNone'
   pathLimit = 0
   # For backward compatibility if "any" is specified then the
   # no form of this command will show "any"
   if 'any' in args:
      app = 'appAny'
   elif 'backup' in args:
      app = 'appBackup'
   elif 'limit' in args:
      if 'ecmp' in args:
         app = 'appEcmp'
      else:
         app = 'appLimit'
      pathLimit = args[ 'LIMIT' ]
   elif 'ecmp' in args:
      app = 'appEcmp'
   SetAddpathSendCmd_setAddpathSend(
         mode,
         app=app,
         pathLimit=pathLimit )

def SetAddpathSendCmd_noHandler( mode, args ):
   app = 'appNone'
   # For backward compatibility if "any" is specified then the
   # no form of this command will show "any"
   if 'ANY_HIDDEN' in args or 'any' in args:
      app = 'appAny'
   SetAddpathSendCmd_setAddpathSend(
         mode,
         app=app,
         noOrDefault=NoOrDefault.NO )

def SetAddpathSendCmd_defaultHandler( mode, args ):
   SetAddpathSendCmd_setAddpathSend(
         mode,
         noOrDefault=NoOrDefault.DEFAULT )

# class BgpRouteInstallMapCmd( BgpCmdBaseClass ):
def BgpRouteInstallMapCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   # prevent configuration from both globally and at af levels
   if conflictingRouteInstallMapConfigs( mode, args[ 'MAP_NAME' ], config,
                                          mode.addrFamily ):
      return
   attr = bgpConfigAttrsAfMap[ 'routeInstallMap' ].get( mode.addrFamily )
   setattr( config, attr, args[ 'MAP_NAME' ] )

def BgpRouteInstallMapCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'routeInstallMap' ].get( mode.addrFamily )
   defaultVal = getattr( config, 'routeInstallMapDefault' )
   setattr( config, attr, defaultVal )

def BgpRouteInstallMapCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "BgpRouteInstallMapCmd", mode, args )

def BgpRouteInstallMapCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "BgpRouteInstallMapCmd", mode, args )

# class SetNetworkCmd( BgpCmdBaseClass ):
def SetNetworkCmd_setNetworkCommon(
      mode, prefix, restricted=False, mapName='', rcfName='', no=False ):
   if ':' in str( prefix ):
      if mode.addrFamily == 'ipv4':
         mode.addErrorAndStop( "IPv6 networks may not be specified in IPv4 mode" )
      if mode.addrFamily == 'ipv4 multicast':
         mode.addErrorAndStop( "IPv6 networks may not be specified in IPv4 " +
                                 "multicast mode" )
      if mode.addrFamily == 'ipv4 labeled-unicast':
         mode.addErrorAndStop( "IPv6 networks may not be specified in IPv4 " +
                                 "labeled-unicast mode" )

   if '.' in str( prefix ):
      if mode.addrFamily == 'ipv6':
         mode.addErrorAndStop( "IPv4 networks may not be specified in IPv6 mode" )
      if mode.addrFamily == 'ipv6 labeled-unicast':
         mode.addErrorAndStop( "IPv4 networks may not be specified in IPv6 " +
                                 "labeled-unicast mode" )

   if prefix is None:
      mode.addError( "Invalid network wildcard mask" )
      return

   vrfName = mode.vrfName
   addrFamily = mode.addrFamily

   if addrFamily in [ '', 'ipv4', 'ipv6' ]:
      addrFamily = 'all'
   prefix = str( prefix )
   config = configForVrf( vrfName )
   attrName = bgpConfigAttrsAfMap[ 'networkList' ].get( addrFamily )
   networkList = getattr( config, attrName )

   if no:
      del networkList[ Arnet.IpGenPrefix( prefix ) ]
   else:
      networkList.addMember( Tac.Value( 'Routing::Bgp::Network',
                                    network=Arnet.IpGenPrefix( prefix ),
                                    restricted=( restricted and
                                                addrFamily in [ '', 'ipv4' ] ),
                                    routeMap=mapName,
                                    rcf=rcfName ) )

def SetNetworkCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION', '' )
   # Trim the () off the end of the RCF function
   rcfName = rcfName[ : -2 ] if rcfName else rcfName
   SetNetworkCmd_setNetworkCommon(
      mode,
      args.get( 'prefix' ),
      mapName=args.get( 'MAP_NAME', '' ),
      rcfName=rcfName )

def SetNetworkCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetNetworkCmd_setNetworkCommon(
      mode,
      args.get( 'prefix' ),
      no=True )

def SetNetworkCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNetworkCmd", mode, args )

def SetNetworkCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNetworkCmd", mode, args )

# class SetInstanceAspathCmpIncNhCmd( CliCommandClass ):
def SetInstanceAspathCmpIncNhCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asPathCmpIncNh = 'isTrue'

def SetInstanceAspathCmpIncNhCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asPathCmpIncNh = getExplicitDefaultTristate( mode, 'asPathCmpIncNh' )

def SetInstanceAspathCmpIncNhCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.asPathCmpIncNh = 'isInvalid'

# class SetBestPathTieBreakBase( object ):
SetBestPathTieBreakBase_tbAttrMap = {
   'cluster-list-length': 'ecmpTbOnClusterListLen',
   'originator-id': 'ecmpTbOnOrigId',
   'router-id': 'ecmpTbOnRouterId',
   'age': 'ecmpTbOnAge'
}

def SetBestPathTieBreakBase_setTieBreak( mode, tieBreak ):
   config = configForVrf( mode.vrfName )
   setattr( config, SetBestPathTieBreakBase_tbAttrMap[ tieBreak ], 'isTrue' )

def SetBestPathTieBreakBase_noTieBreak( mode, tieBreak, noOrDefault ):
   config = configForVrf( mode.vrfName )
   attr = SetBestPathTieBreakBase_tbAttrMap[ tieBreak ]
   if noOrDefault == NoOrDefault.DEFAULT:
      val = 'isInvalid'
   else:
      val = getExplicitDefaultTristate( mode, attr )
   setattr( config, attr, val )

# class SetBestPathTieBreakCmd( CliCommandClass, SetBestPathTieBreakBase ):
def SetBestPathTieBreakCmd_handler( mode, args ):
   SetBestPathTieBreakBase_setTieBreak( mode, args[ 'TIE_BREAK' ] )

def SetBestPathTieBreakCmd_noHandler( mode, args ):
   SetBestPathTieBreakBase_noTieBreak(
         mode,
         args[ 'TIE_BREAK' ],
         NoOrDefault.NO )

def SetBestPathTieBreakCmd_defaultHandler( mode, args ):
   SetBestPathTieBreakBase_noTieBreak(
         mode,
         args[ 'TIE_BREAK' ],
         NoOrDefault.DEFAULT )

# class SetEcmpTbDeprecatedCmd( CliCommandClass, SetBestPathTieBreakBase ):
def SetEcmpTbDeprecatedCmd_handler( mode, args ):
   SetBestPathTieBreakBase_setTieBreak(
         mode,
         args[ 'TIE_BREAK' ] )

def SetEcmpTbDeprecatedCmd_noHandler( mode, args ):
   SetBestPathTieBreakBase_noTieBreak(
         mode,
         args[ 'TIE_BREAK' ],
         NoOrDefault.NO )

def SetEcmpTbDeprecatedCmd_defaultHandler( mode, args ):
   SetBestPathTieBreakBase_noTieBreak(
         mode,
         args[ 'TIE_BREAK' ],
         NoOrDefault.DEFAULT )

# class SetBestPathEcmpFastCmd( CliCommandClass ):
def SetBestPathEcmpFastCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.ecmpFast = getExplicitDefaultTristate( mode, 'ecmpFast' )

def SetBestPathEcmpFastCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.ecmpFast = 'isInvalid'

def SetBestPathEcmpFastCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.ecmpFast = 'isFalse'

# class SetInstanceAdvertiseInactive( CliCommandClass ):
def SetInstanceAdvertiseInactive_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.advertiseInactive = 'isTrue'

def SetInstanceAdvertiseInactive_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.advertiseInactive = getExplicitDefaultTristate( mode, 'advertiseInactive' )

def SetInstanceAdvertiseInactive_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.advertiseInactive = 'isInvalid'

# class SetAggregateAddressBaseCmd( BgpCmdBaseClass ):
# class SetAggregateAddressCmd( SetAggregateAddressBaseCmd ):
# class SetIPv4MulticastAggregateAddressCmd( SetAggregateAddressBaseCmd ):
# class SetIPv6MulticastAggregateAddressCmd( SetAggregateAddressBaseCmd ):
def SetAggregateAddressBaseCmd_getAggregateListAttr( addrFamily ):
   # separate config for ipv4 multicast, ipv6 multicast and
   # ipv4/ipv6 unicast aggregation
   attrName = 'aggregateList'
   if addrFamily == 'ipv4 multicast':
      attrName = 'aggregateAfV4MulticastList'
   elif addrFamily == 'ipv6 multicast':
      attrName = 'aggregateAfV6MulticastList'
   return attrName

def SetAggregateAddressBaseCmd_setAggregateAddress(
      mode,
      address,
      asset=False,
      summaryonly=False,
      advertiseonly=False,
      attrmap=None,
      attrRcf=None,
      matchmap=None ):
   if address is None or str( address ) == '':
      mode.addError( 'Invalid address wildcard mask' )
      return

   if not summaryonly:
      summaryonly = AggregateAddressType.summaryOnlyDefault
   if not asset:
      asset = AggregateAddressType.asSetDefault
   if not attrmap:
      attrmap = AggregateAddressType.attrMapDefault
   if not attrRcf:
      attrRcf = AggregateAddressType.attrRcfDefault
   if not matchmap:
      matchmap = AggregateAddressType.matchMapDefault
   if not advertiseonly:
      advertiseonly = AggregateAddressType.advertiseOnlyDefault

   aggregateListAttr = getattr( configForVrf( mode.vrfName ),
                                 SetAggregateAddressBaseCmd_getAggregateListAttr(
                                    mode.addrFamily ) )
   # If RCF present we need to strip the () from the end of the function name
   attrRcf = attrRcf.replace( "()", "" )
   aggregateListAttr.addMember(
      Tac.Value(
         'Routing::Bgp::AggregateAddress',
         address=Arnet.IpGenPrefix( str( address ) ),
         summaryOnly=summaryonly,
         asSet=asset,
         attrMap=attrmap,
         attrRcf=attrRcf,
         matchMap=matchmap,
         advertiseOnly=advertiseonly ) )

def SetAggregateAddressBaseCmd_noAggregateAddress( mode, address ):
   aggregateListAttr = getattr( configForVrf( mode.vrfName ),
                                    SetAggregateAddressBaseCmd_getAggregateListAttr(
                                    mode.addrFamily ) )
   del aggregateListAttr[ Arnet.IpGenPrefix( str( address ) ) ]

def SetAggregateAddressBaseCmd_handleNormal( mode, args ):
   if mode.addrFamily in [ 'ipv4 multicast', 'ipv6 multicast' ] and (
         getEffectiveProtocolModel( mode.session.mode ) ==
         ProtocolAgentModelType.ribd ):
      mode.addWarning( 'Routing protocols model multi-agent must be '
                       'configured for multicast aggregation' )
   SetAggregateAddressBaseCmd_setAggregateAddress(
         mode,
         args[ 'PREFIX' ],
         asset=args.get( 'as-set', False ),
         summaryonly=args.get( 'summary-only', False ),
         advertiseonly=args.get( 'advertise-only', False ),
         attrmap=args.get( 'ATTR_MAP' ),
         attrRcf=args.get( 'ATTR_FUNCTION' ),
         matchmap=args.get( 'MATCH_MAP' ) )

def SetAggregateAddressBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetAggregateAddressBaseCmd_noAggregateAddress(
      mode, args[ 'PREFIX' ] )

def SetAggregateAddressBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetAggregateAddressBaseCmd", mode, args )

def SetAggregateAddressBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetAggregateAddressBaseCmd", mode, args )

# class SetInstanceAutoAggDomainCmd( BgpCmdBaseClass ):
def SetInstanceAutoAggDomainCmd_setAutoAggDomain(
      mode,
      prefix,
      autoAggDomainArgs=None,
      autoAggGroupByArgs=None ):
   prefix = Arnet.IpGenPrefix( str( prefix ) )
   if prefix is None or str( prefix ) == '':
      mode.addError( "Invalid prefix" )
      return

   argsDict = dict( autoAggDomainArgs if autoAggDomainArgs else [] )
   asset = argsDict.get( 'asset', AutoAggregateDomainType.asSetDefault )
   attrmap = argsDict.get( 'attrmap', AutoAggregateDomainType.attrMapDefault )
   matchmap = argsDict.get( 'matchmap', AutoAggregateDomainType.matchMapDefault )

   groupByDict = dict( autoAggGroupByArgs if autoAggGroupByArgs else [] )
   groupByAsPath = 'groupByAsPath' in groupByDict
   groupByLocPref = 'groupByLocPref' in groupByDict
   groupByMed = 'groupByMed' in groupByDict
   groupByCommunity = 'groupByCommunity' in groupByDict
   groupByExtCommunity = 'groupByExtCommunity' in groupByDict

   configForVrf( mode.vrfName ).autoAggDomainList.addMember(
      AutoAggregateDomainType( prefix, asSet=asset,
                                 attrMap=attrmap, matchMap=matchmap,
                                 groupByAsPath=groupByAsPath,
                                 groupByLocPref=groupByLocPref,
                                 groupByMed=groupByMed,
                                 groupByCommunity=groupByCommunity,
                                 groupByExtCommunity=groupByExtCommunity ) )

def SetInstanceAutoAggDomainCmd_noAutoAggDomain( mode, prefix ):
   config = configForVrf( mode.vrfName )
   del config.autoAggDomainList[ Arnet.IpGenPrefix( str( prefix ) ) ]

def SetInstanceAutoAggDomainCmd_handleNormal( mode, args ):
   SetInstanceAutoAggDomainCmd_setAutoAggDomain(
         mode,
         args[ 'PREFIX' ],
         autoAggDomainArgs=args.get( 'DOMAIN_ARGS', {} ),
         autoAggGroupByArgs=args.get( 'GROUP_ARGS', {} )
   )

def SetInstanceAutoAggDomainCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetInstanceAutoAggDomainCmd_noAutoAggDomain( mode, args[ 'PREFIX' ] )

def SetInstanceAutoAggDomainCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceAutoAggDomainCmd", mode, args )

def SetInstanceAutoAggDomainCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceAutoAggDomainCmd", mode, args )

# class SetInstanceRouterIdCmd( BgpCmdBaseClass ):
def SetInstanceRouterIdCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.routerId = args[ 'ROUTER_ID' ]

def SetInstanceRouterIdCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   config.routerId = config.routerIdDefault

def SetInstanceRouterIdCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceRouterIdCmd", mode, args )

def SetInstanceRouterIdCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceRouterIdCmd", mode, args )

# class SetInstanceBgpDistanceCmd( CliCommandClass ):
def SetInstanceBgpDistanceCmd_distanceBgp(
      mode,
      bgpExternalDistance,
      bgpInternalDistance=None,
      bgpLocalDistance=None ):
   # The cli prevents bgpInternalDistance and bgpLocalDistance
   # being set to None independently. Either both have values, or both
   # are none
   if bgpInternalDistance is None and bgpLocalDistance is None:
      bgpInternalDistance = bgpExternalDistance
      bgpLocalDistance = bgpExternalDistance
   config = configForVrf( mode.vrfName )
   setVal = True
   if ( mode.vrfName == DEFAULT_VRF and
         bgpExternalDistance == config.externalDistanceDefault and
         bgpInternalDistance == config.internalDistanceDefault and
         bgpLocalDistance == config.localDistanceDefault ):
      setVal = False
   eD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                     pref=bgpExternalDistance, isSet=setVal )
   iD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                     pref=bgpInternalDistance, isSet=setVal )
   lD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                     pref=bgpLocalDistance, isSet=setVal )
   config.externalDistance = eD
   config.internalDistance = iD
   config.localDistance = lD

def SetInstanceBgpDistanceCmd_noDistanceBgp( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   # The "no distance bgp" command operates differently from the
   # "default distance bgp..." command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
      setVal = False
   else:
      setVal = True
   eD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                     pref=config.externalDistanceDefault, isSet=setVal )
   iD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                     pref=config.internalDistanceDefault, isSet=setVal )
   lD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                     pref=config.localDistanceDefault, isSet=setVal )
   config.externalDistance = eD
   config.internalDistance = iD
   config.localDistance = lD

def SetInstanceBgpDistanceCmd_handler( mode, args ):
   SetInstanceBgpDistanceCmd_distanceBgp(
         mode,
         args[ 'EXTERNAL' ],
         bgpInternalDistance=args.get( 'INTERNAL' ),
         bgpLocalDistance=args.get( 'LOCAL' ) )

def SetInstanceBgpDistanceCmd_noHandler( mode, args ):
   SetInstanceBgpDistanceCmd_noDistanceBgp( mode, NoOrDefault.NO )

def SetInstanceBgpDistanceCmd_defaultHandler( mode, args ):
   SetInstanceBgpDistanceCmd_noDistanceBgp( mode, NoOrDefault.DEFAULT )

# class SetInstanceListenLimitRangeBaseClass( object ):
def SetInstanceListenLimitRangeBaseClass_setBgpListenLimit( mode, bgpLimit ):
   config = configForVrf( mode.vrfName )
   config.listenLimit = getExplicitDefaultValue(
         mode,
         'listenLimit',
         bgpLimit )

def SetInstanceListenLimitRangeBaseClass_noBgpListenLimit( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   # The "no bgp listen limit..." command operates differently from
   # the "default bgp listen limit..." command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
      config.listenLimit = config.listenLimitInvalid
   else:
      config.listenLimit = config.listenLimitDefault

def SetInstanceListenLimitRangeBaseClass_setBgpListenRange(
      mode, prefix, pgKey, peerSpecifier, includeRouterId ):
   config = configForVrf( mode.vrfName )
   prefix = Arnet.IpGenPrefix( str( prefix ) )
   assert pgKey.group is not None, "The peer group key is of the wrong type"
   peergroupName = pgKey.group
   for pg in config.listenRange:
      # Prevent configuration of identical or overlapping listenRanges
      for existingPrefix in config.listenRange.get( pg ).prefixList:
         if prefix == existingPrefix and \
            config.listenRange.get( pg ).peergroupName == peergroupName:
            continue
         if prefix.overlaps( existingPrefix ):
            mode.addError( 'Cannot configure range %s while %s'
                                 ' is configured against %s.'
                                 % ( prefix, existingPrefix, pg ) )
            return
   listenRangeConfig = config.listenRange.get( peergroupName )
   if not listenRangeConfig:
      listenRangeConfig = config.listenRange.newMember( peergroupName )

   peerSpec = Tac.Value( 'Routing::Bgp::PeerSpecifier', prefix )
   if includeRouterId and getEffectiveProtocolModel( mode ) != \
      ProtocolAgentModelType.multiAgent and \
      not ( mode.session.inConfigSession() or mode.session.startupConfig() ):
      mode.addError( "peer-id is only supported in multi-agent mode" )
      return
   peerSpec.includeRouterId = includeRouterId
   if isinstance( peerSpecifier, str ):
      peerSpec.hasAsn = False
      peerSpec.peerFilter = peerSpecifier
   else:
      peerSpec.hasAsn = True
      peerSpec.asn = peerSpecifier
   listenRangeConfig.prefixList[ prefix ] = peerSpec
   bgpNeighborConfig( pgKey, vrfName=mode.vrfName )

def SetInstanceListenLimitRangeBaseClass_noBgpListenRange( mode, prefix, pgKey ):
   config = configForVrf( mode.vrfName )
   prefix = Arnet.IpGenPrefix( str( prefix ) )
   assert pgKey.group is not None, "The peer group key is of the wrong type"
   peergroupName = pgKey.group
   if config.listenRange.get( peergroupName ):
      del config.listenRange[ peergroupName ].prefixList[ prefix ]
      if not config.listenRange.get( peergroupName ).prefixList:
         del config.listenRange[ peergroupName ]

# class SetInstanceListenLimitDeprecatedCmd():
def SetInstanceListenLimitDeprecatedCmd_handler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_setBgpListenLimit(
         mode,
         args[ 'LIMIT' ] )

def SetInstanceListenLimitDeprecatedCmd_noHandler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_noBgpListenLimit( mode, NoOrDefault.NO )

def SetInstanceListenLimitDeprecatedCmd_defaultHandler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_noBgpListenLimit(
         mode,
         NoOrDefault.DEFAULT )

# class SetInstanceDynamicPeerMaxLimit( CliCommandClass ):
def SetInstanceDynamicPeerMaxLimit_handler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_setBgpListenLimit(
         mode,
         args[ 'LIMIT' ] )

def SetInstanceDynamicPeerMaxLimit_noHandler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_noBgpListenLimit( mode, NoOrDefault.NO )

def SetInstanceDynamicPeerMaxLimit_defaultHandler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_noBgpListenLimit(
         mode,
         NoOrDefault.DEFAULT )

# class SetInstanceListenRangePrefixPgCmd( CliCommandClass ):
def SetInstanceListenRangePrefixPgCmd_handler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_setBgpListenRange(
         mode,
         args[ 'PREFIX' ],
         args[ 'PGKEY' ],
         args[ 'PEER_SPEC' ],
         'router-id' in args )

def SetInstanceListenRangePrefixPgCmd_noOrDefaultHandler( mode, args ):
   SetInstanceListenLimitRangeBaseClass_noBgpListenRange(
         mode,
         args[ 'PREFIX' ],
         args[ 'PGKEY' ] )

# class SetIpv4UnicastCmd( CliCommandClass ):
def SetIpv4UnicastCmd_setIPv4Unicast( mode, overIpv6=False ):
   config = configForVrf( mode.vrfName )
   if overIpv6:
      # IPv4 unicast over IPv6 is disabled by default
      config.defaultV4UniOverV6 = 'isTrue'
   else:
      # IPv4 unicast is enabled by default
      config.defaultV4Uni = getExplicitDefaultTristate( mode, 'defaultV4Uni' )

def SetIpv4UnicastCmd_noDefaultIPv4Unicast( mode, noOrDefault, overIpv6=False ):
   config = configForVrf( mode.vrfName )
   if noOrDefault == NoOrDefault.DEFAULT:
      if overIpv6:
         config.defaultV4UniOverV6 = 'isInvalid'
      else:
         config.defaultV4Uni = 'isInvalid'
   else:
      if overIpv6:
         config.defaultV4UniOverV6 = getExplicitDefaultTristate(
                                             mode, 'defaultV4UniOverV6' )
      else:
         config.defaultV4Uni = 'isFalse'

def SetIpv4UnicastCmd_handler( mode, args ):
   SetIpv4UnicastCmd_setIPv4Unicast( mode, overIpv6=( 'transport' in args ) )

def SetIpv4UnicastCmd_defaultHandler( mode, args ):
   SetIpv4UnicastCmd_noDefaultIPv4Unicast(
         mode,
         NoOrDefault.DEFAULT,
         overIpv6=( 'transport' in args ) )

def SetIpv4UnicastCmd_noHandler( mode, args ):
   SetIpv4UnicastCmd_noDefaultIPv4Unicast(
         mode,
         NoOrDefault.NO,
         overIpv6=( 'transport' in args ) )

# class SetIpv6UnicastCmd( CliCommandClass ):
def SetIpv6UnicastCmd_setIPv6Unicast( mode ):
   # IPv6 unicast is disabled by default
   config = configForVrf( mode.vrfName )
   config.defaultV6Uni = 'isTrue'

def SetIpv6UnicastCmd_noDefaultIPv6Unicast( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if noOrDefault == NoOrDefault.DEFAULT:
      config.defaultV6Uni = 'isInvalid'
   else:
      config.defaultV6Uni = getExplicitDefaultTristate( mode, 'defaultV6Uni' )

def SetIpv6UnicastCmd_handler( mode, args ):
   SetIpv6UnicastCmd_setIPv6Unicast( mode )

def SetIpv6UnicastCmd_noHandler( mode, args ):
   SetIpv6UnicastCmd_noDefaultIPv6Unicast( mode, NoOrDefault.NO )

def SetIpv6UnicastCmd_defaultHandler( mode, args ):
   SetIpv6UnicastCmd_noDefaultIPv6Unicast( mode, NoOrDefault.DEFAULT )

# class MissingPolicyCmd( CliCommandClass ):
def MissingPolicyCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   mpCfg = MissingPolicyConfig( args )
   if ( mode.vrfName == DEFAULT_VRF and
         mpCfg.actionType == config.missingPolicyActionDefault ):
      actionType = config.missingPolicyActionInvalid
   else:
      actionType = mpCfg.actionType
   setattr( config, mpCfg.actionAttr, actionType )
   if mpCfg.addressFamilyAll:
      setattr( config, mpCfg.addrFamilyAllAttr, 'isTrue' )
   else:
      setattr( config, mpCfg.addrFamilyAllAttr, 'isInvalid' )
   if mpCfg.includeSubRm:
      setattr( config, mpCfg.includeSubAttr, 'isTrue' )
   else:
      setattr( config, mpCfg.includeSubAttr, 'isInvalid' )
   if mpCfg.includePfxList:
      setattr( config, mpCfg.includePfxListAttr, 'isTrue' )
   else:
      setattr( config, mpCfg.includePfxListAttr, 'isInvalid' )
   if mpCfg.includeCommList:
      setattr( config, mpCfg.includeCommListAttr, 'isTrue' )
   else:
      setattr( config, mpCfg.includeCommListAttr, 'isInvalid' )

   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      if mpCfg.addressFamilyAll:
         mode.addWarning( "Routing protocols model multi-agent must be "
                           "configured for bgp missing-policy address-family "
                           "all configuration" )
      if mpCfg.includeSubRm or mpCfg.includePfxList or mpCfg.includeCommList:
         mode.addWarning( "Routing protocols model multi-agent must be "
                           "configured for bgp missing-policy include options" )

def MissingPolicyCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   mpCfg = MissingPolicyConfig( args )
   actionType = config.missingPolicyActionDefault
   addressFamilyAll = getExplicitDefaultTristate( mode, mpCfg.addrFamilyAllAttr )
   inclSubRm = getExplicitDefaultTristate( mode, mpCfg.includeSubAttr )
   inclPfxList = getExplicitDefaultTristate( mode, mpCfg.includePfxListAttr )
   inclCommList = getExplicitDefaultTristate( mode, mpCfg.includeCommListAttr )
   if mode.vrfName == DEFAULT_VRF or CliCommand.isDefaultCmd( args ):
      actionType = config.missingPolicyActionInvalid
      addressFamilyAll = 'isInvalid'
      inclSubRm = 'isInvalid'
      inclPfxList = 'isInvalid'
      inclCommList = 'isInvalid'
   setattr( config, mpCfg.actionAttr, actionType )
   setattr( config, mpCfg.addrFamilyAllAttr, addressFamilyAll )
   setattr( config, mpCfg.includeSubAttr, inclSubRm )
   setattr( config, mpCfg.includePfxListAttr, inclPfxList )
   setattr( config, mpCfg.includeCommListAttr, inclCommList )

# class MissingPolicyAfSharedCmd( CliCommandClass ):
def MissingPolicyAfSharedCmd_handler( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                        "configured for bgp missing-policy configuration "
                        "under address family mode" )
   config = configForVrf( mode.vrfName )
   mpCfg = MissingPolicyConfig( args, af=True )
   actionAfConfig = getattr( config, mpCfg.actionAttr )
   includeSubAfConfig = getattr( config, mpCfg.includeSubAttr )
   includePfxAfConfig = getattr( config, mpCfg.includePfxListAttr )
   includeCommAfConfig = getattr( config, mpCfg.includeCommListAttr )
   af = Tac.Value( "Routing::Bgp::AfiSafi", *addressFamilies[ mode.addrFamily ] )
   actionAfConfig[ af ] = mpCfg.actionType
   includeSubAfConfig[ af ] = 'isTrue' if mpCfg.includeSubRm else 'isFalse'
   includePfxAfConfig[ af ] = 'isTrue' if mpCfg.includePfxList else 'isFalse'
   includeCommAfConfig[ af ] = 'isTrue' if mpCfg.includeCommList else 'isFalse'

def MissingPolicyAfSharedCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   mpCfg = MissingPolicyConfig( args, af=True )
   actionAfConfig = getattr( config, mpCfg.actionAttr )
   includeSubAfConfig = getattr( config, mpCfg.includeSubAttr )
   includePfxAfConfig = getattr( config, mpCfg.includePfxListAttr )
   includeCommAfConfig = getattr( config, mpCfg.includeCommListAttr )
   af = Tac.Value( "Routing::Bgp::AfiSafi", *addressFamilies[ mode.addrFamily ] )
   del actionAfConfig[ af ]
   del includeSubAfConfig[ af ]
   del includePfxAfConfig[ af ]
   del includeCommAfConfig[ af ]

# class SetInstanceAutoLocalAddrCmd( CliCommandClass ):
def SetInstanceAutoLocalAddrCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.autoLocalAddr = 'isTrue'

def SetInstanceAutoLocalAddrCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.autoLocalAddr = getExplicitDefaultTristate(
         mode,
         'autoLocalAddr' )

def SetInstanceAutoLocalAddrCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.autoLocalAddr = 'isInvalid'

# class SetInstanceControlPlaneFilterDefaultAllowCmd( BgpCmdBaseClass ):
def SetInstanceControlPlaneFilterDefaultAllowCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.cpFilterDefaultAllow = True

def SetInstanceControlPlaneFilterDefaultAllowCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   config.cpFilterDefaultAllow = False

def SetInstanceControlPlaneFilterDefaultAllowCmd_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "SetInstanceControlPlaneFilterDefaultAllowCmd", mode, args )

def SetInstanceControlPlaneFilterDefaultAllowCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "SetInstanceControlPlaneFilterDefaultAllowCmd", mode, args )

# class SetTransportAfMssCmd( BgpCmdBaseClass ):
def SetTransportAfMssCmd_setTcpMaxSegSize( mode, mss, ipv4 ):
   config = configForVrf( mode.vrfName )
   if ipv4:
      config.sockMaxSegSizeIpv4 = mss
   else:
      config.sockMaxSegSizeIpv6 = mss

def SetTransportAfMssCmd_noTcpMaxSegSize( mode, ipv4 ):
   config = configForVrf( mode.vrfName )
   if ipv4:
      config.sockMaxSegSizeIpv4 = config.sockMaxSegSizeInvalid
   else:
      config.sockMaxSegSizeIpv6 = config.sockMaxSegSizeInvalid

def SetTransportAfMssCmd_handleNormal( mode, args ):
   SetTransportAfMssCmd_setTcpMaxSegSize(
         mode,
         args[ 'MSS' ],
         ( 'ipv4' in args ) )

def SetTransportAfMssCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetTransportAfMssCmd_noTcpMaxSegSize( mode, ( 'ipv4' in args ) )

def SetTransportAfMssCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetTransportAfMssCmd", mode, args )

def SetTransportAfMssCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetTransportAfMssCmd", mode, args )

# class SetTransportListenPortCmd( BgpCmdBaseClass ):
def SetTransportListenPortCmd_setListenPort( mode, listenPort ):
   config = configForVrf( mode.vrfName )
   config.listenPort = listenPort

def SetTransportListenPortCmd_noListenPort( mode ):
   config = configForVrf( mode.vrfName )
   config.listenPort = config.listenPortInvalid

def SetTransportListenPortCmd_handleNormal( mode, args ):
   SetTransportListenPortCmd_setListenPort( mode, args[ 'PORT' ] )

def SetTransportListenPortCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetTransportListenPortCmd_noListenPort( mode )

def SetTransportListenPortCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetTransportListenPortCmd", mode, args )

def SetTransportListenPortCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetTransportListenPortCmd", mode, args )

# class SetTransportQosDscpCmd( BgpCmdBaseClass ):
def SetTransportQosDscpCmd_setDscp( mode, dscp ):
   Globals.bgpConfig.dscp = dscp

def SetTransportQosDscpCmd_noDscp( mode ):
   Globals.bgpConfig.dscp = Globals.bgpConfig.dscpDefault

def SetTransportQosDscpCmd_handleNormal( mode, args ):
   SetTransportQosDscpCmd_setDscp( mode, args[ 'DSCP' ] )

def SetTransportQosDscpCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetTransportQosDscpCmd_noDscp( mode )

def SetTransportQosDscpCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetTransportQosDscpCmd", mode, args )

def SetTransportQosDscpCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetTransportQosDscpCmd", mode, args )

# class SetTransportPmtudCmd( BgpCmdBaseClass ):
def SetTransportPmtudCmd_setPathMtuDiscovery( mode ):
   config = configForVrf( mode.vrfName )
   config.pathMtuDiscovery = 'isTrue'
   config.pathMtuDiscoverySet = True

def SetTransportPmtudCmd_disabledPathMtuDiscovery( mode ):
   config = configForVrf( mode.vrfName )
   config.pathMtuDiscovery = 'isFalse'
   config.pathMtuDiscoverySet = True

def SetTransportPmtudCmd_defaultPathMtuDiscovery( mode ):
   config = configForVrf( mode.vrfName )
   config.pathMtuDiscovery = config.pathMtuDiscoveryDefault
   config.pathMtuDiscoverySet = False

def SetTransportPmtudCmd_handleNormal( mode, args ):
   SetTransportPmtudCmd_setPathMtuDiscovery( mode )

def SetTransportPmtudCmd_handleNoOrDefault( mode, args, noOrDefault ):
   if noOrDefault == NoOrDefault.DEFAULT:
      SetTransportPmtudCmd_defaultPathMtuDiscovery( mode )
   else:
      SetTransportPmtudCmd_disabledPathMtuDiscovery( mode )

def SetTransportPmtudCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetTransportPmtudCmd", mode, args )

def SetTransportPmtudCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetTransportPmtudCmd", mode, args )

# class SetTransportSockBufferCmd( BgpCmdBaseClass ):
def SetTransportSockBufferCmd_noSockBuffer( mode ):
   config = configForVrf( mode.vrfName )
   config.sockBufferSize = None

def SetTransportSockBufferCmd_setSockBuffer( mode, size ):
   config = configForVrf( mode.vrfName )
   config.sockBufferSize = \
            Tac.newInstance( 'Arnet::TcpSockBufferSize', size, size )

def SetTransportSockBufferCmd_handleNormal( mode, args ):
   SetTransportSockBufferCmd_setSockBuffer( mode, args[ 'SIZE' ] )

def SetTransportSockBufferCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetTransportSockBufferCmd_noSockBuffer( mode )

def SetTransportSockBufferCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetTransportSockBufferCmd", mode, args )

def SetTransportSockBufferCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetTransportSockBufferCmd", mode, args )

# class SetInstanceConfedIdCmd( CliCommandClass ):
def SetInstanceConfedIdCmd_setConfedId( mode, asNumber ):
   config = configForVrf( mode.vrfName )
   config.confedId = asNumber

def SetInstanceConfedIdCmd_noConfedId( mode ):
   config = configForVrf( mode.vrfName )
   config.confedId = 0

def SetInstanceConfedIdCmd_handler( mode, args ):
   SetInstanceConfedIdCmd_setConfedId( mode, args.get( 'AS_NUM' ) )

def SetInstanceConfedIdCmd_noOrDefaultHandler( mode, args ):
   SetInstanceConfedIdCmd_noConfedId( mode )

# class SetInstanceConfedPeersCmd( CliCommandClass ):
def SetInstanceConfedPeersCmd_setConfedPeers( mode, asIter ):
   config = configForVrf( mode.vrfName )
   for asn in asIter:
      asnKey = AsNumKey( asn )
      config.confedPeer[ asnKey ] = True

def SetInstanceConfedPeersCmd_noConfedPeers( mode, asIter ):
   config = configForVrf( mode.vrfName )
   for asn in asIter:
      asnKey = AsNumKey( asn )
      del config.confedPeer[ asnKey ]

def SetInstanceConfedPeersCmd_handler( mode, args ):
   SetInstanceConfedPeersCmd_setConfedPeers( mode, args.get( 'AS_NUMS' ) )

def SetInstanceConfedPeersCmd_noOrDefaultHandler( mode, args ):
   SetInstanceConfedPeersCmd_noConfedPeers( mode, args.get( 'AS_NUMS' ) )

# class SetInstanceAsnNotationCmd( BgpCmdBaseClass ):
def SetInstanceAsnNotationCmd_handleNormal( mode, args ):
   asnConfig.asnNotation = args[ 'ASN_NOTATION' ]

def SetInstanceAsnNotationCmd_handleNoOrDefault( mode, args, noOrDefault ):
   asnConfig.asnNotation = asnConfig.asnNotationDefault

def SetInstanceAsnNotationCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceAsnNotationCmd", mode, args )

def SetInstanceAsnNotationCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceAsnNotationCmd", mode, args )

# class BgpAfServiceAclMixin( object ):
def BgpAfServiceAclMixin_setServiceAcl( mode, aclName=None ):
   setServiceAclCommon( mode, aclName=aclName )

def BgpAfServiceAclMixin_noServiceAcl( mode, aclName=None ):
   setServiceAclCommon( mode, no=True, aclName=aclName )

def BgpAfServiceAclMixin_setServiceAclV6( mode, aclName=None ):
   setServiceAclCommon( mode, aclType='ipv6', aclName=aclName )

def BgpAfServiceAclMixin_noServiceAclV6( mode, aclName=None ):
   setServiceAclCommon( mode, aclType='ipv6', no=True, aclName=aclName )

# class SetInstanceServiceAclCmd( BgpAfServiceAclMixin, BgpCmdBaseClass ):
def SetInstanceServiceAclCmd_handleNormal( mode, args ):
   BgpAfServiceAclMixin_setServiceAcl( mode, aclName=args[ 'ACL' ] )

def SetInstanceServiceAclCmd_handleNoOrDefault( mode, args, noOrDefault ):
   BgpAfServiceAclMixin_noServiceAcl( mode, aclName=args.get( 'ACL' ) )

def SetInstanceServiceAclCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceServiceAclCmd", mode, args )

def SetInstanceServiceAclCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceServiceAclCmd", mode, args )

# class SetInstanceServiceAclV6Cmd( BgpAfServiceAclMixin, BgpCmdBaseClass ):
def SetInstanceServiceAclV6Cmd_handleNormal( mode, args ):
   BgpAfServiceAclMixin_setServiceAclV6( mode, aclName=args[ 'ACL6' ] )

def SetInstanceServiceAclV6Cmd_handleNoOrDefault( mode, args, noOrDefault ):
   BgpAfServiceAclMixin_noServiceAclV6( mode, aclName=args.get( 'ACL6' ) )

def SetInstanceServiceAclV6Cmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceServiceAclV6Cmd", mode, args )

def SetInstanceServiceAclV6Cmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceServiceAclV6Cmd", mode, args )

# class SetMaxPathsCmd( CliCommandClass ):
def SetMaxPathsCmd_noMaxNumPaths( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   # The "no maximum-paths" command operates differently from the
   # "default maximum-paths" command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets maxPaths to the system default,
   # which will be 1, and for maxEcmp will be depend on the hardware.
   if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
      config.maxPaths = config.maxPathsInvalid
      config.maxEcmp = config.maxEcmpInvalid
   else:
      config.maxPaths = config.maxPathsDefault
      config.maxEcmp = Globals.routingHardwareStatus.maxLogicalProtocolEcmp
   config.maxPathsYangSource.clear()

def SetMaxPathsCmd_maxNumPaths( mode, maxPaths, maxEcmp, yangSources ):
   def handleMaxPathsYangSource( enumVal, add ):
      if add:
         config.maxPathsYangSource.add( enumVal )
      else:
         config.maxPathsYangSource.remove( enumVal )

   config = configForVrf( mode.vrfName )
   currMaxEcmp = config.maxEcmp
   if ( ( maxEcmp and maxPaths > maxEcmp ) or
         ( maxEcmp is None and
            currMaxEcmp != config.maxEcmpInvalid and
            0 < currMaxEcmp < maxPaths ) ):
      mode.addError(
         "maximum-paths should be less than or equal to max ECMP" )
      return
   if config.maxPaths != maxPaths:
      config.maxPaths = getExplicitDefaultValue( mode, 'maxPaths', maxPaths )
   if maxEcmp and config.maxEcmp != maxEcmp:
      config.maxEcmp = getExplicitDefaultValue( mode, 'maxEcmp', maxEcmp )
   handleMaxPathsYangSource( YangSourceEnum.yangSourceGlobal,
                             yangSources.sourceGlobalEnabled )
   handleMaxPathsYangSource( YangSourceEnum.yangSourceIpv4Unicast,
                             yangSources.sourceIpv4UnicastEnabled )
   handleMaxPathsYangSource( YangSourceEnum.yangSourceIpv6Unicast,
                             yangSources.sourceIpv6UnicastEnabled )
   # maxEcmp and source OpenConfig are not supported together, so configuring one
   # should clear the state of the other.
   if yangSources.count( True ) > 0:
      config.maxEcmp = config.maxEcmpInvalid



def SetMaxPathsCmd_handler( mode, args ):
   SetMaxPathsCmd_maxNumPaths(
         mode,
         args.get( 'NUM_PATHS' ),
         args.get( 'NUM_ECMP_RTS' ),
         YangSources( 'global' in args,
                      'ipv4-unicast' in args,
                      'ipv6-unicast' in args ) )

def SetMaxPathsCmd_noHandler( mode, args ):
   SetMaxPathsCmd_noMaxNumPaths( mode, NoOrDefault.NO )

def SetMaxPathsCmd_defaultHandler( mode, args ):
   SetMaxPathsCmd_noMaxNumPaths( mode, NoOrDefault.DEFAULT )

# class SetVrfVpnDefaultExportCmd( BgpCmdBaseClass ):
def SetVrfVpnDefaultExportCmd_validateVpnDefaultExportCommand( mode, vpnType ):
   ''' This function validates the route target command to check if the vpnType
         is allowed in the mode '''
   if not vpnTypeTokenAllowedInMode( vpnType, mode ):
      mode.addError( '%s argument is not allowed under address-family %s' % (
                     vpnType, mode.addrFamily ) )
      raise AlreadyHandledError()

def SetVrfVpnDefaultExportCmd_conflictingVpnDefaultExportConfigs(
      mode, config, addrFamily, vpnType ):
   ''' This function detects conflicting configuration 
         for "default-route export <vpnType> <rm>" commands
   '''
   defaultExportAttrs = bgpConfigAttrsAfMap[ 'defaultExport' ]
   conflictingAttrs = getConflictingAttrsForAf( defaultExportAttrs, addrFamily )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   for otherAf, c in conflictingAttrs:
      if vpnAf in getattr( config, c ):
         errorMsg = ( "Cannot configure default-route %s in mode '%s' "
                        "while it is configured in mode '%s'"
                        % ( vpnType, configModeCmdForAf( addrFamily ),
                        configModeCmdForAf( otherAf ) ) )
         mode.addError( errorMsg )
         raise AlreadyHandledError()

def SetVrfVpnDefaultExportCmd_setVpnDefaultExport(
      mode, vpnType, defaultExportArgs ):
   ''' Handler for "default-route export <vpnType> (<rm>|<rcf>)" commands '''

   defaultExportDict = dict( defaultExportArgs if defaultExportArgs else [] )
   rm = defaultExportDict.get( "mapName" )
   rcf = defaultExportDict.get( "rcfName" )
   if rcf is not None:
      # If RCF present we need to strip the () from the end of the function name
      rcf = rcf.replace( '()', '' )
   always = defaultExportDict.get( "always" )
   addrFamily = mode.addrFamily

   SetVrfVpnDefaultExportCmd_validateVpnDefaultExportCommand( mode, vpnType )

   config = configForVrf( mode.vrfName )
   SetVrfVpnDefaultExportCmd_conflictingVpnDefaultExportConfigs(
         mode, config, addrFamily, vpnType )

   attr = bgpConfigAttrsAfMap[ 'defaultExport' ].get( addrFamily )
   defaultExportCfg = getattr( config, attr )
   vpnAf = vpnAfTypeMapInv[ vpnType ]

   # As we only allow one default originate policy to be in place at any given time
   # We must check to what type of policy is being set so we can set the other type
   # back to its default value first. Otherwise it is possible for both route-map and
   # Rcf policy to exist briefly at the same time when propagating the changes.
   # If no policyName is set, we don't care about order as we set both route-map and
   # Rcf back to their default values.
   defaultExportVpnCfg = defaultExportCfg.get( vpnAf )
   if defaultExportVpnCfg is None:
      defaultExportVpnCfg = defaultExportCfg.newMember( vpnAf )
   if rm is None and rcf is None:
      defaultExportVpnCfg.routeMap = defaultExportVpnCfg.routeMapDefault
      defaultExportVpnCfg.rcf = defaultExportVpnCfg.rcfDefault
   elif rcf:
      defaultExportVpnCfg.routeMap = defaultExportVpnCfg.routeMapDefault
      defaultExportVpnCfg.rcf = rcf
   else:
      defaultExportVpnCfg.rcf = defaultExportVpnCfg.rcfDefault
      defaultExportVpnCfg.routeMap = rm
   defaultExportVpnCfg.always = always

def SetVrfVpnDefaultExportCmd_noVpnDefaultExport( mode, vpnType ):
   ''' Handler for "default-route export <vpnType> (<rm>|<rcf)" commands '''
   config = configForVrf( mode.vrfName )
   addrFamily = mode.addrFamily

   attr = bgpConfigAttrsAfMap[ 'defaultExport' ].get( addrFamily )
   defaultExportCfg = getattr( config, attr )
   vpnAf = vpnAfTypeMapInv[ vpnType ]

   del defaultExportCfg[ vpnAf ]

def SetVrfVpnDefaultExportCmd_handleNormal( mode, args ):
   SetVrfVpnDefaultExportCmd_setVpnDefaultExport(
         mode, args[ 'VPN' ], args.get( 'EXPORT_DEF_ORIGIN' ) )

def SetVrfVpnDefaultExportCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetVrfVpnDefaultExportCmd_noVpnDefaultExport( mode, args[ 'VPN' ] )

def SetVrfVpnDefaultExportCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetVrfVpnDefaultExportCmd", mode, args )

def SetVrfVpnDefaultExportCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetVrfVpnDefaultExportCmd", mode, args )

# class SetInstanceAlwaysCompareMedCmd( CliCommandClass ):
def SetInstanceAlwaysCompareMedCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.alwaysCompareMed = 'isTrue'

def SetInstanceAlwaysCompareMedCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.alwaysCompareMed = getExplicitDefaultTristate( mode, 'alwaysCompareMed' )

def SetInstanceAlwaysCompareMedCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.alwaysCompareMed = 'isInvalid'

# class SetBestPathMedCmd( CliCommandClass ):
def SetBestPathMedCmd_setBestPathMed(
      mode, missingAsWorst, confed, noOrDefault=None ):
   config = configForVrf( mode.vrfName )
   if noOrDefault:
      if missingAsWorst and confed:
         if noOrDefault == NoOrDefault.DEFAULT:
            config.medConfed = 'isInvalid'
            config.medConfedMissingAsWorst = 'isInvalid'
         else:
            config.medConfed = getExplicitDefaultTristate( mode, 'medConfed' )
            config.medConfedMissingAsWorst = getExplicitDefaultTristate(
                  mode,
                  'medConfedMissingAsWorst' )
      elif missingAsWorst:
         if noOrDefault == NoOrDefault.DEFAULT:
            config.medMissingAsWorst = 'isInvalid'
         else:
            config.medMissingAsWorst = getExplicitDefaultTristate(
                  mode,
                  'medMissingAsWorst' )
      elif confed:
         if noOrDefault == NoOrDefault.DEFAULT:
            config.medConfed = 'isInvalid'
            config.medConfedMissingAsWorst = 'isInvalid'
         else:
            config.medConfed = getExplicitDefaultTristate( mode, 'medConfed' )
            if config.medConfedMissingAsWorst == 'isTrue':
               config.medConfedMissingAsWorst = getExplicitDefaultTristate(
                     mode,
                     'medConfedMissingAsWorst' )
   else:
      if missingAsWorst and confed:
         config.medConfed = 'isTrue'
         config.medConfedMissingAsWorst = 'isTrue'
      elif missingAsWorst:
         config.medMissingAsWorst = 'isTrue'
      elif confed:
         config.medConfed = 'isTrue'
         config.medConfedMissingAsWorst = 'isInvalid'

def SetBestPathMedCmd_handler( mode, args ):
   SetBestPathMedCmd_setBestPathMed(
         mode,
         ( 'missing-as-worst' in args ),
         ( 'confed' in args ) )

def SetBestPathMedCmd_noHandler( mode, args ):
   SetBestPathMedCmd_setBestPathMed(
         mode,
         ( 'missing-as-worst' in args ),
         ( 'confed' in args ),
         noOrDefault=NoOrDefault.NO )

def SetBestPathMedCmd_defaultHandler( mode, args ):
   SetBestPathMedCmd_setBestPathMed(
         mode,
         ( 'missing-as-worst' in args ),
         ( 'confed' in args ),
         noOrDefault=NoOrDefault.DEFAULT )

# class SetConvergenceTimeCmd( CliCommandClass ):
def SetConvergenceTimeCmd_setConvergenceTime( mode, seconds, slowPeer=False ):
   config = configForVrf( mode.vrfName )
   if slowPeer:
      config.convergenceIdlePeerTime = getExplicitDefaultValue(
            mode, 'convergenceIdlePeerTime', seconds )
   else:
      config.convergenceTime = getExplicitDefaultValue(
            mode, 'convergenceTime', seconds )

def SetConvergenceTimeCmd_noConvergenceTime( mode, noOrDefault, slowPeer=False ):
   # The "no bgp convergence time" command operates differently from the
   # "default bgp..." command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   config = configForVrf( mode.vrfName )
   if slowPeer:
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.convergenceIdlePeerTime = config.convergenceIdlePeerTimeInvalid
      else:
         config.convergenceIdlePeerTime = config.convergenceIdlePeerTimeDefault
   else:
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.convergenceTime = config.convergenceTimeInvalid
      else:
         config.convergenceTime = config.convergenceTimeDefault

def SetConvergenceTimeCmd_handler( mode, args ):
   SetConvergenceTimeCmd_setConvergenceTime(
         mode,
         args.get( 'SECONDS' ),
         slowPeer=( 'slow-peer' in args ) )

def SetConvergenceTimeCmd_noHandler( mode, args ):
   SetConvergenceTimeCmd_noConvergenceTime(
         mode,
         NoOrDefault.NO,
         slowPeer=( 'slow-peer' in args ) )

def SetConvergenceTimeCmd_defaultHandler( mode, args ):
   SetConvergenceTimeCmd_noConvergenceTime(
         mode,
         NoOrDefault.DEFAULT,
         slowPeer=( 'slow-peer' in args ) )

# class SetUpdateWaitForConvergenceCmd( CliCommandClass ):
def SetUpdateWaitForConvergenceCmd_setWaitForConvergence( mode ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'convergenceNoSync' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, 'isFalse' )

def SetUpdateWaitForConvergenceCmd_noWaitForConvergence( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'convergenceNoSync' ].get( mode.addrFamily )
   if attr:
      if noOrDefault == NoOrDefault.DEFAULT:
         setattr( config, attr, 'isInvalid' )
      else:
         if mode.addrFamily == 'all':
            setattr(
                  config,
                  attr,
                  getExplicitDefaultTristate(
                        mode,
                        'convergenceNoSync' ) )
         else:
            setattr( config, attr, 'isTrue' )

def SetUpdateWaitForConvergenceCmd_handler( mode, args ):
   SetUpdateWaitForConvergenceCmd_setWaitForConvergence( mode )

def SetUpdateWaitForConvergenceCmd_noHandler( mode, args ):
   SetUpdateWaitForConvergenceCmd_noWaitForConvergence(
         mode,
         NoOrDefault.NO )

def SetUpdateWaitForConvergenceCmd_defaultHandler( mode, args ):
   SetUpdateWaitForConvergenceCmd_noWaitForConvergence(
         mode,
         NoOrDefault.DEFAULT )

# class SetInstancePeerMacResolutionTimeoutCmd( CliCommandClass ):
def SetInstancePeerMacResolutionTimeoutCmd_noBgpPeerMacResolutionTimeout(
      mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   # The "no peer-mac..." command operates differently from
   # the "default peer-mac..." command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
      config.peerMacResolutionTimeout = config.peerMacResolutionTimeoutInvalid
   else:
      config.peerMacResolutionTimeout = config.peerMacResolutionTimeoutDefault

def SetInstancePeerMacResolutionTimeoutCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.peerMacResolutionTimeout = getExplicitDefaultValue(
         mode,
         'peerMacResolutionTimeout',
         args[ 'TIMEOUT' ] )

def SetInstancePeerMacResolutionTimeoutCmd_noHandler( mode, args ):
   SetInstancePeerMacResolutionTimeoutCmd_noBgpPeerMacResolutionTimeout(
         mode,
         NoOrDefault.NO )

def SetInstancePeerMacResolutionTimeoutCmd_defaultHandler( mode, args ):
   SetInstancePeerMacResolutionTimeoutCmd_noBgpPeerMacResolutionTimeout(
         mode,
         NoOrDefault.DEFAULT )

# class HostRoutesFibDirectInstallCmd( CliCommandClass ):
def HostRoutesFibDirectInstallCmd_handler( mode, args ):
   pass

def HostRoutesFibDirectInstallCmd_noOrDefaultHandler( mode, args ):
   HostRoutesFibDirectInstallCmd_handler( mode, args )

# class SetInstancePreserveAttrsCmd( CliCommandClass ):
def SetInstancePreserveAttrsCmd_setPreserveAttrs( mode, always=False ):
   config = configForVrf( mode.vrfName )
   if always:
      config.reflectedRouteAttr = ReflectedRouteAttrStateEnum.alwaysPreserved
   else:
      config.reflectedRouteAttr = (
            ReflectedRouteAttrStateEnum.preservedExceptRouteMap
      )

def SetInstancePreserveAttrsCmd_handler( mode, args ):
   SetInstancePreserveAttrsCmd_setPreserveAttrs(
         mode,
         always=( 'always' in args ) )

def SetInstancePreserveAttrsCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.reflectedRouteAttr = ReflectedRouteAttrStateEnum.notPreserved

def SetInstancePreserveAttrsCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.reflectedRouteAttr = config.reflectedRouteAttrDefault

# class SetInstanceDefaultMetricCmd( BgpCmdBaseClass ):
def SetInstanceDefaultMetricCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.defaultMetric = args[ 'METRIC' ]
   config.defaultMetricPresent = True

def SetInstanceDefaultMetricCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   config.defaultMetric = config.defaultMetricDefault
   config.defaultMetricPresent = False

def SetInstanceDefaultMetricCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceDefaultMetricCmd", mode, args )

def SetInstanceDefaultMetricCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceDefaultMetricCmd", mode, args )

# class UcmpModeCmd( CliCommandClass ):
def UcmpModeCmd_setUcmp( mode, ucmpMode, size=None, deviation=None ):
   ucmpConfig = ucmpConfigForVrf( mode.vrfName )
   # Historically, we stored a ucmp "mode" number with a mode of "0"
   # indicating that UCMP is disabled. The CLI only supports a single mode
   # ( "ucmp mode 1 ..." ) and we have no plans to introduce new modes in the
   # future. So we ignore the ucmpMode value here and simply "enable" UCMP.
   ucmpConfig.ucmpEnabled = 'isTrue'
   if size:
      ucmpConfig.maxUcmp = size
   else:
      ucmpConfig.maxUcmp = ucmpConfig.maxUcmpInvalid

   if deviation:
      ucmpConfig.ucmpDeviation = deviation
   else:
      ucmpConfig.ucmpDeviation = ucmpConfig.ucmpDeviationInvalid

def UcmpModeCmd_noUcmp( mode, noOrDefault ):
   ucmpConfig = ucmpConfigForVrf( mode.vrfName )
   # The "no ucmp mode ..." command operates differently from
   # the "default ucmp mode..." command for the non-default VRF.
   # The 'default' option is the same as unconfigured, which will cause
   # the non-default vrf to use the value configured in the default vrf.
   # The 'no' option explicitly sets parameters to the system defaults.
   if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
      ucmpConfig.ucmpEnabled = 'isInvalid'
      ucmpConfig.maxUcmp = ucmpConfig.maxUcmpInvalid
      ucmpConfig.ucmpDeviation = ucmpConfig.ucmpDeviationInvalid
   else:
      ucmpConfig.ucmpEnabled = 'isFalse'
      ucmpConfig.maxUcmp = ucmpConfig.maxUcmpDefault
      ucmpConfig.ucmpDeviation = ucmpConfig.ucmpDeviationDefault

def UcmpModeCmd_handler( mode, args ):
   UcmpModeCmd_setUcmp(
         mode, args[ 'MODE' ],
         size=args.get( 'MAX' ),
         deviation=args.get( 'DEV' ) )

def UcmpModeCmd_noHandler( mode, args ):
   UcmpModeCmd_noUcmp( mode, NoOrDefault.NO )

def UcmpModeCmd_defaultHandler( mode, args ):
   UcmpModeCmd_noUcmp( mode, NoOrDefault.DEFAULT )

# class UcmpLinkBandwidthCmd( CliCommandClass ):
def UcmpLinkBandwidthCmd_setUcmpLinkBandwidth(
      mode,
      recursive=False,
      recursiveUserWeight=False,
      encodeWeight=False,
      delay=None ):
   config = ucmpConfigForVrf( mode.vrfName )
   if recursive:
      if recursiveUserWeight:
         config.ucmpLinkbwRecursiveUserWeight = 'isTrue'
         # As we do not want the base command (just 'recursive') to save separately
         # from a command with tokens, we keep them mutually exclusive.
         config.ucmpLinkbwRecursive = 'isInvalid'
         if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.ribd:
            msg = 'The command is only supported in the multi-agent routing ' \
                  'protocol model.'
            mode.addWarning( msg )
      else:
         config.ucmpLinkbwRecursive = 'isTrue'
         # See above comment about keeping these configs mutually exclusive.
         config.ucmpLinkbwRecursiveUserWeight = 'isInvalid'
         if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.ribd:
            msg = 'The command is only supported in the ribd routing protocol model.'
            mode.addWarning( msg )
   elif encodeWeight:
      config.ucmpLinkbwWeighted = 'isTrue'
   elif delay is not None: # Check against None here as delay may be 0
      config.ucmpLinkbwDelay = getExplicitDefaultValue(
         mode, 'ucmpLinkbwDelay', delay, useUcmpConfig=True )

def UcmpLinkBandwidthCmd_noUcmpLinkBandwidth(
      mode,
      noOrDefault,
      recursive=False,
      encodeWeight=False,
      delay=False ):
   config = ucmpConfigForVrf( mode.vrfName )
   setDefault = mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT
   if encodeWeight:
      config.ucmpLinkbwWeighted = 'isInvalid' if setDefault else 'isFalse'
   elif recursive:
      config.ucmpLinkbwRecursive = 'isInvalid' if setDefault else 'isFalse'
      # All per-protocol LBW inheritance tokens will be deactivated along with the
      # ribd-exclusive function's config.
      config.ucmpLinkbwRecursiveUserWeight = 'isInvalid' if setDefault else 'isFalse'
   elif delay:
      config.ucmpLinkbwDelay = config.ucmpLinkbwDelayInvalid if setDefault \
         else config.ucmpLinkbwDelayDefault

def UcmpLinkBandwidthCmd_handler( mode, args ):
   UcmpLinkBandwidthCmd_setUcmpLinkBandwidth(
         mode,
         recursive=( 'recursive' in args ),
         # 'user weight' ordering/presence should be enforced by syntax
         recursiveUserWeight=( 'user' in args ),
         encodeWeight=( 'encoding-weighted' in args ),
         delay=args.get( 'DELAY' ) )

def UcmpLinkBandwidthCmd_noHandler( mode, args ):
   UcmpLinkBandwidthCmd_noUcmpLinkBandwidth(
         mode,
         NoOrDefault.NO,
         recursive=( 'recursive' in args ),
         encodeWeight=( 'encoding-weighted' in args ),
         delay=( 'update-delay' in args ) )

def UcmpLinkBandwidthCmd_defaultHandler( mode, args ):
   UcmpLinkBandwidthCmd_noUcmpLinkBandwidth(
         mode,
         NoOrDefault.DEFAULT,
         recursive=( 'recursive' in args ),
         encodeWeight=( 'encoding-weighted' in args ),
         delay=( 'update-delay' in args ) )

# class UcmpFecThresholdCmd( CliCommandClass ):
def UcmpFecThresholdCmd_setUcmpFecThresholds(
      mode, ucmpFecTrigPercentage=None, ucmpFecClrPercentage=None ):
   config = ucmpConfigForVrf( mode.vrfName )
   if ( ucmpFecTrigPercentage != config.ucmpTriggerFecThresholdInvalid and
         ucmpFecClrPercentage != config.ucmpClearFecThresholdInvalid and
         ucmpFecTrigPercentage < ucmpFecClrPercentage ):
      mode.addError(
            'clear threshold percentage needs to be less than '
            'or equal to trigger threshold percentage '
      )
      return
   config.ucmpTriggerFecThreshold = ucmpFecTrigPercentage
   config.ucmpClearFecThreshold = ucmpFecClrPercentage

def UcmpFecThresholdCmd_noUcmpFecThresholds( mode, noOrDefault ):
   config = ucmpConfigForVrf( mode.vrfName )
   if config:
      if noOrDefault == NoOrDefault.DEFAULT:
         # by default, ucmp thresholds are enabled with default values
         config.ucmpTriggerFecThreshold = config.ucmpTriggerFecThresholdDefault
         config.ucmpClearFecThreshold = config.ucmpClearFecThresholdDefault
      else:
         config.ucmpTriggerFecThreshold = config.ucmpTriggerFecThresholdInvalid
         config.ucmpClearFecThreshold = config.ucmpClearFecThresholdInvalid

def UcmpFecThresholdCmd_handler( mode, args ):
   UcmpFecThresholdCmd_setUcmpFecThresholds(
         mode,
         ucmpFecTrigPercentage=args[ 'PERCENT' ],
         ucmpFecClrPercentage=args[ 'CLR_PERCENT' ] )

def UcmpFecThresholdCmd_noHandler( mode, args ):
   UcmpFecThresholdCmd_noUcmpFecThresholds( mode, NoOrDefault.NO )

def UcmpFecThresholdCmd_defaultHandler( mode, args ):
   UcmpFecThresholdCmd_noUcmpFecThresholds( mode, NoOrDefault.DEFAULT )

# class SetInstanceNexthopUnchangedCmd( CliCommandClass ):
def SetInstanceNexthopUnchangedCmd_noNexthopUnchanged( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'nexthopUnchanged' ].get( mode.addrFamily )
   if attr:
      if noOrDefault == NoOrDefault.DEFAULT:
         setattr( config, attr, 'isInvalid' )
      else:
         setattr( config, attr, 'isFalse' )

def SetInstanceNexthopUnchangedCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'nexthopUnchanged' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, 'isTrue' )

def SetInstanceNexthopUnchangedCmd_noHandler( mode, args ):
   SetInstanceNexthopUnchangedCmd_noNexthopUnchanged( mode, NoOrDefault.NO )

def SetInstanceNexthopUnchangedCmd_defaultHandler( mode, args ):
   SetInstanceNexthopUnchangedCmd_noNexthopUnchanged( mode, NoOrDefault.DEFAULT )

# class SetInstanceNexthopSelfCmd( CliCommandClass ):
def SetInstanceNexthopSelfCmd_setNexthopSelf( mode ):
   if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.ribd:
      msg = 'Address family next-hop-self only supported in multi-agent mode'
      mode.addWarning( msg )
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'nextHopSelf' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, 'isTrue' )

def SetInstanceNexthopSelfCmd_noNexthopSelf( mode, noOrDefault ):
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ 'nextHopSelf' ].get( mode.addrFamily )
   if attr:
      if noOrDefault == NoOrDefault.DEFAULT:
         setattr( config, attr, 'isInvalid' )
      else:
         setattr( config, attr, 'isFalse' )

def SetInstanceNexthopSelfCmd_handler( mode, args ):
   SetInstanceNexthopSelfCmd_setNexthopSelf( mode )

def SetInstanceNexthopSelfCmd_noHandler( mode, args ):
   SetInstanceNexthopSelfCmd_noNexthopSelf( mode, NoOrDefault.NO )

def SetInstanceNexthopSelfCmd_defaultHandler( mode, args ):
   SetInstanceNexthopSelfCmd_noNexthopSelf( mode, NoOrDefault.DEFAULT )

# class SetInstanceNextHopAfIpv6Cmd( BgpCmdBaseClass ):
def SetInstanceNextHopAfIpv6Cmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'ipv4':
      config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isEnabled
   elif mode.addrFamily == 'ipv6':
      mode.addError(
            'IPv6 nexthop address family may not be specified in IPv6 mode' )
   else:
      raise ValueError()

def SetInstanceNextHopAfIpv6Cmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'ipv4':
      config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isDisabled
   elif mode.addrFamily == 'ipv6':
      mode.addError(
            'IPv6 nexthop address family may not be specified in IPv6 mode' )
   else:
      raise ValueError()

def SetInstanceNextHopAfIpv6Cmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceNextHopAfIpv6Cmd", mode, args )

def SetInstanceNextHopAfIpv6Cmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceNextHopAfIpv6Cmd", mode, args )

# class AigpSessionAfCommand( CliCommandClass ):
def AigpSessionAfCommand_setAigpSessionAf( mode, peerType, val ):
   # Note that the default state is enabled on ibgp/confed, disabled on ebgp.
   # The attributes reflect this.
   aigpAttrMap = {
      'ibgp': 'aigpSessionIbgpDisabled',
      'confederation': 'aigpSessionConfedDisabled',
      'ebgp': 'aigpSessionEbgpEnabled',
   }
   config = configForVrf( mode.vrfName )
   attrname = aigpAttrMap[ peerType ]
   attr = bgpConfigAttrsAfMap[ attrname ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, val )

def AigpSessionAfCommand_handler( mode, args ):
   # The attributes are enabling/disabling depending on the peer type.
   # Set the TristateBool value appropriately.
   value = 'isTrue' if args[ 'PEER_TYPE' ] == 'ebgp' else 'isFalse'
   AigpSessionAfCommand_setAigpSessionAf( mode, args[ 'PEER_TYPE' ], value )

def AigpSessionAfCommand_noHandler( mode, args ):
   value = 'isFalse' if args[ 'PEER_TYPE' ] == 'ebgp' else 'isTrue'
   AigpSessionAfCommand_setAigpSessionAf( mode, args[ 'PEER_TYPE' ], value )

def AigpSessionAfCommand_defaultHandler( mode, args ):
   AigpSessionAfCommand_setAigpSessionAf( mode, args[ 'PEER_TYPE' ], 'isInvalid' )

# class NexthopResolutionDisabledCmd( CliCommandClass ):
NexthopResolutionDisabledCmd_afiSafisSupported = {
   'ipv4': 'afV4NexthopResolutionDisabled',
   'ipv6': 'afV6NexthopResolutionDisabled',
   'vpn-ipv4': 'afVpnV4NexthopResolutionDisabled',
   'vpn-ipv6': 'afVpnV6NexthopResolutionDisabled',
   'evpn': 'afEvpnNexthopResolutionDisabled',
}

NexthopResolutionDisabledCmd_vpnAfiSafis = [ 'vpn-ipv4', 'vpn-ipv6', 'evpn' ]

def NexthopResolutionDisabledCmd_setNextHopResolutionDisabled(
      afiSafi, vrfName=DEFAULT_VRF ):
   if afiSafi in NexthopResolutionDisabledCmd_vpnAfiSafis:
      assert vrfName == DEFAULT_VRF
   config = configForVrf( vrfName )
   setattr( config, NexthopResolutionDisabledCmd_afiSafisSupported[
      afiSafi ], True )

def NexthopResolutionDisabledCmd_noNextHopResolutionDisabled(
      afiSafi, vrfName=DEFAULT_VRF ):
   if afiSafi in NexthopResolutionDisabledCmd_vpnAfiSafis:
      assert vrfName == DEFAULT_VRF
   config = configForVrf( vrfName )
   setattr( config, NexthopResolutionDisabledCmd_afiSafisSupported[
      afiSafi ], False )

def NexthopResolutionDisabledCmd_handler( mode, args ):
   NexthopResolutionDisabledCmd_setNextHopResolutionDisabled(
      mode.addrFamily, mode.vrfName )

def NexthopResolutionDisabledCmd_noOrDefaultHandler( mode, args ):
   NexthopResolutionDisabledCmd_noNextHopResolutionDisabled(
      mode.addrFamily, mode.vrfName )

# class NextHopOriginateTunnelCmd( CliCommand.CliCommandClass )
def setNhOrigTunnelProfileEncapTypeAndPolicyName( mode, args, encapType,
                                                  policyName ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.nhOrigTunnelProfileEncapTypeVpnV4 = encapType
      config.nhOrigTunnelProfilePolicyNameVpnV4 = policyName
   elif mode.addrFamily == 'vpn-ipv6':
      config.nhOrigTunnelProfileEncapTypeVpnV6 = encapType
      config.nhOrigTunnelProfilePolicyNameVpnV6 = policyName
   elif mode.addrFamily == 'ipv4':
      tc = Tac.Value( 'Routing::Bgp::OriginateTunnelConfig',
                      encapType, policyName )
      config.afV4NexthopOriginateTunnelConfig = tc
   elif mode.addrFamily == 'ipv6':
      tc = Tac.Value( 'Routing::Bgp::OriginateTunnelConfig',
                      encapType, policyName )
      config.afV6NexthopOriginateTunnelConfig = tc
   else:
      raise ValueError()

def noNhOrigTunnelProfileEncapTypeAndPolicyName( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.nhOrigTunnelProfileEncapTypeVpnV4 = \
            config.nhOrigTunnelProfileEncapTypeVpnV4Default
      config.nhOrigTunnelProfilePolicyNameVpnV4 = \
            config.nhOrigTunnelProfilePolicyNameVpnV4Default
   elif mode.addrFamily == 'vpn-ipv6':
      config.nhOrigTunnelProfileEncapTypeVpnV6 = \
            config.nhOrigTunnelProfileEncapTypeVpnV6Default
      config.nhOrigTunnelProfilePolicyNameVpnV6 = \
            config.nhOrigTunnelProfilePolicyNameVpnV6Default
   elif mode.addrFamily == 'ipv4':
      config.afV4NexthopOriginateTunnelConfig = \
         config.afV4NexthopOriginateTunnelConfigDefault
   elif mode.addrFamily == 'ipv6':
      config.afV6NexthopOriginateTunnelConfig = \
         config.afV4NexthopOriginateTunnelConfigDefault
   else:
      raise ValueError()

def NextHopOriginateTunnelCmd_handler( mode, args ):
   policyName = args[ 'POLICYNAME' ]
   setNhOrigTunnelProfileEncapTypeAndPolicyName( mode, args, 'udp', policyName )

def NextHopOriginateTunnelCmd_noOrDefaultHandler( mode, args ):
   noNhOrigTunnelProfileEncapTypeAndPolicyName( mode, args )

# class RedistributeBaseCmd( object ):
def RedistributeBaseCmd_setRedistribute(
      mode,
      proto,
      intext=None,
      mapName=None,
      rcfName=None,
      mType=0,
      isisLevel=None,
      leaked=False ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   if addrFamily == 'ipv4 multicast':
      if proto not in [ 'protoDirect', 'protoStatic', 'protoIsis', 'protoOspf',
                        'protoOspfNssa', 'protoOspfAse', 'protoOspf3',
                        'protoOspf3Nssa', 'protoOspf3Ase', 'protoAttachedHost',
                        'protoBgp' ]:
         raise ValueError()
   config = configForVrf( vrfName )
   if proto == 'protoBgpAggregate' and mapName:
      mode.addWarning( 'The route-map argument is no longer supported for '
                        'aggregates and has no effect' )

   if proto == 'protoEosSdk' and getEffectiveProtocolModel( mode ) != \
            ProtocolAgentModelType.multiAgent:
      mode.addWarning( "'redistribute %s' in mode '%s' is "
            "only supported in multi-agent model" %
            ( protoDict[ proto ], configModeCmdForAf( addrFamily ) ) )

   if addrFamily in [ 'ipv4', 'ipv6' ] and \
         proto in [ 'protoDirect', 'protoStatic', 'protoOspf', 'protoRip',
               'protoAttachedHost', 'protoDynamic', 'protoBgp' ]:
      # for proto == 'protoBgp', leaked will be True
      if getEffectiveProtocolModel( mode ) != \
            ProtocolAgentModelType.multiAgent:
         mode.addWarning( "'redistribute %s' in mode 'address-family %s' is "
               "only supported in multi-agent model" %
               ( protoDict[ proto ], addrFamily ) )
   if not intext:
      intext = ''
   elif intext == 'external':
      if proto == 'protoOspf':
         proto = 'protoOspfAse'
      elif proto == 'protoOspf3':
         proto = 'protoOspf3Ase'
   elif intext == 'nssa-external':
      if proto == 'protoOspf':
         proto = 'protoOspfNssa'
      elif proto == 'protoOspf3':
         proto = 'protoOspf3Nssa'

   level = 0
   if proto == 'protoIsis':
      if isisLevel == 'level-1':
         level = 1
      elif isisLevel == 'level-1-2':
         level = 3
      else:
         level = 2
      maybeWarnConflictingRedist( mode, config, proto, addrFamily )

   elif proto == 'protoOspf3' or proto == 'protoOspf3Nssa' or \
         proto == 'protoOspf3Ase':
      maybeWarnConflictingRedist( mode, config, proto, addrFamily )

   redistConfig = redistributeConfig( addrFamily, config )
   assert not ( mapName and rcfName )  # Only one of them can be enabled/applied
   redistribute = Tac.Value( 'Routing::Bgp::Redistribute',
                              proto, routeType=intext, metricType=mType,
                              routeMap=( mapName or '' ),
                              rcf=( rcfName or '' ), isisLevel=level,
                              includeLeaked=leaked )
   redistConfig.addMember( redistribute )

def RedistributeBaseCmd_noRedistribute(
      mode, proto, intext='', noOrDefault=NoOrDefault.DEFAULT ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = configForVrf( vrfName )
   originalProto = proto
   if not intext:
      intext = ''
   if intext == 'external':
      if proto == 'protoOspf':
         proto = 'protoOspfAse'
      elif proto == 'protoOspf3':
         proto = 'protoOspf3Ase'
   elif intext == 'nssa-external':
      if proto == 'protoOspf':
         proto = 'protoOspfNssa'
      elif proto == 'protoOspf3':
         proto = 'protoOspf3Nssa'

   redistConfig = redistributeConfig( addrFamily, config )
   if noOrDefault == NoOrDefault.DEFAULT:
      del redistConfig[ proto ]
   else:
      redistribute = Tac.Value( 'Routing::Bgp::Redistribute',
            proto, routeType=intext,
            includeLeaked=( proto == 'protoBgp' ),
            redistEnabled=False )
      redistConfig.addMember( redistribute )
      if addrFamily in [ 'ipv4', 'ipv6' ]:
         if getEffectiveProtocolModel( mode ) != \
               ProtocolAgentModelType.multiAgent:
            mode.addWarning( "'redistribute %s disabled' in mode "
                  "'address-family %s' is only supported in multi-agent model" %
                  ( protoDict[ originalProto ], addrFamily ) )

# class RedistributeMcastCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeMcastCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args.get( 'proto' ),
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeMcastCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args.get( 'proto' ) )

def RedistributeMcastCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeMcastCmd", mode, args )

def RedistributeMcastCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeMcastCmd", mode, args )

# class RedistributeMcast6Cmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeMcast6Cmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args.get( 'proto' ),
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeMcast6Cmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args.get( 'proto' ) )

def RedistributeMcast6Cmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeMcast6Cmd", mode, args )

def RedistributeMcast6Cmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeMcast6Cmd", mode, args )

# class RouteInputBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
# class RouteInputMcastV4UnicastCmd( RouteInputBaseCmd ):
# class RouteInputMcastV6UnicastCmd( RouteInputBaseCmd ):
def RouteInputBaseCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      # Remove the trailing () characters
      rcfName = rcfName.rstrip( '()' )
   # No need to specify the safi 'unicast' now as that is the only
   # one supported for now
   RedistributeBaseCmd_setRedistribute( mode, 'protoBgp', rcfName=rcfName )

def RouteInputBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, 'protoBgp' )

def RouteInputBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteInputBaseCmd", mode, args )

def RouteInputBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteInputBaseCmd", mode, args )

# class ConnStatRedistributeProtoBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
# class ConnStatRedistributeProtoCmd( ConnStatRedistributeProtoBaseCmd ):
# class ConnStatRedistributeProtoAfCmd( ConnStatRedistributeProtoBaseCmd ):
def ConnStatRedistributeProtoBaseCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      # Remove the trailing () characters
      rcfName = re.sub( r'\(\)', '', rcfName )
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         mapName=args.get( 'MAP_NAME' ),
         rcfName=rcfName,
         leaked=( 'leaked' in args ) )

def ConnStatRedistributeProtoBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute(
      mode, args[ 'proto' ], noOrDefault=noOrDefault )

def ConnStatRedistributeProtoBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "ConnStatRedistributeProtoBaseCmd", mode, args )

def ConnStatRedistributeProtoBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "ConnStatRedistributeProtoBaseCmd", mode, args )

# class RedistributeProtoCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeProtoCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeProtoCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args[ 'proto' ] )

def RedistributeProtoCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeProtoCmd", mode, args )

def RedistributeProtoCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeProtoCmd", mode, args )

# class RedistributeRipAfCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeRipAfCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeRipAfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args[ 'proto' ],
         noOrDefault=noOrDefault )

def RedistributeRipAfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeRipAfCmd", mode, args )

def RedistributeRipAfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeRipAfCmd", mode, args )

# class RedistributeAttachedHostAfCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeAttachedHostAfCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeAttachedHostAfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args[ 'proto' ],
         noOrDefault=noOrDefault )

def RedistributeAttachedHostAfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeAttachedHostAfCmd", mode, args )

def RedistributeAttachedHostAfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeAttachedHostAfCmd", mode, args )

# class RedistributeDynamicPolicyCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeDynamicPolicyCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      if rcfName.endswith( '()' ):
         # Remove the trailing () characters
         rcfName = rcfName[ : -2 ]
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         mapName=args.get( 'MAP_NAME' ),
         rcfName=rcfName )

def RedistributeDynamicPolicyCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args[ 'proto' ] )

def RedistributeDynamicPolicyCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeDynamicPolicyCmd", mode, args )

def RedistributeDynamicPolicyCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeDynamicPolicyCmd", mode, args )

# class RedistributeDynamicPolicyAfCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeDynamicPolicyAfCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      if rcfName.endswith( '()' ):
         # Remove the trailing () characters
         rcfName = rcfName[ : -2 ]
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         mapName=args.get( 'MAP_NAME' ),
         rcfName=rcfName )

def RedistributeDynamicPolicyAfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args[ 'proto' ],
         noOrDefault=noOrDefault )

def RedistributeDynamicPolicyAfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeDynamicPolicyAfCmd", mode, args )

def RedistributeDynamicPolicyAfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeDynamicPolicyAfCmd", mode, args )

# class RedistributeBgpBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
# class RedistributeBgpCmd( RedistributeBgpBaseCmd ):
# class RedistributeBgpAfCmd( RedistributeBgpBaseCmd ):
def RedistributeBgpBaseCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         'protoBgp',
         mapName=args.get( 'MAP_NAME' ),
         leaked=True )

def RedistributeBgpBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, 'protoBgp',
         noOrDefault=noOrDefault )

def RedistributeBgpBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeBgpBaseCmd", mode, args )

def RedistributeBgpBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeBgpBaseCmd", mode, args )

# class RedistributeIsisBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
# class RedistributeIsisCmd( RedistributeIsisBaseCmd ):
# class RedistributeIsisAfCmd( RedistributeIsisBaseCmd ):
def RedistributeIsisBaseCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      if rcfName.endswith( '()' ):
         # Remove the trailing () characters
         rcfName = rcfName[ : -2 ]
   RedistributeBaseCmd_setRedistribute(
         mode,
         'protoIsis',
         isisLevel=args.get( 'isisLevel' ),
         mapName=args.get( 'MAP_NAME' ),
         rcfName=rcfName,
         leaked=( 'leaked' in args ) )

def RedistributeIsisBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, 'protoIsis',
         noOrDefault=noOrDefault )

def RedistributeIsisBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeIsisBaseCmd", mode, args )

def RedistributeIsisBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeIsisBaseCmd", mode, args )

# class RedistributeDhcpCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeDhcpCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         'protoDhcp',
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeDhcpCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, 'protoDhcp' )

def RedistributeDhcpCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeDhcpCmd", mode, args )

def RedistributeDhcpCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeDhcpCmd", mode, args )

# class RedistributeOspfMatchIntExtLeakedBaseCmd( BgpCmdBaseClass ):
# class RedistributeOspfOrOspf3MatchIntExtLeakedCmd():
# class RedistributeOspf3MatchIntExtLeakedCmd():
# class RedistributeOspfMatchIntExtLeakedCmd():
def RedistributeOspfMatchIntExtLeakedBaseCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         intext=args.get( args.get( 'intext' ) ),
         mapName=args.get( 'MAP_NAME' ),
         leaked=( 'leaked' in args ) )

def RedistributeOspfMatchIntExtLeakedBaseCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute(
         mode,
         args[ 'proto' ],
         intext=args.get( args.get( 'intext' ) ),
         noOrDefault=noOrDefault )

def RedistributeOspfMatchIntExtLeakedBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeOspfMatchIntExtLeakedBaseCmd", mode, args )

def RedistributeOspfMatchIntExtLeakedBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RedistributeOspfMatchIntExtLeakedBaseCmd", mode, args )

# class RedistributeOspfMatchNssaExtLeakedBaseCmd( BgpCmdBaseClass ):
# class RedistributeOspfOrOspf3MatchNssaExtLeakedCmd():
# class RedistributeOspf3MatchNssaExtLeakedCmd():
# class RedistributeOspfMatchNssaExtLeakedCmd():
def RedistributeOspfMatchNssaExtLeakedBaseCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         intext='nssa-external',
         mapName=args.get( 'MAP_NAME' ),
         mType=args.get( 'NSSA_TYPE', 0 ),
         leaked=( 'leaked' in args ) )

def RedistributeOspfMatchNssaExtLeakedBaseCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute(
         mode,
         args[ 'proto' ],
         intext='nssa-external',
         noOrDefault=noOrDefault )

def RedistributeOspfMatchNssaExtLeakedBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeOspfMatchNssaExtLeakedBaseCmd", mode, args )

def RedistributeOspfMatchNssaExtLeakedBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RedistributeOspfMatchNssaExtLeakedBaseCmd", mode, args )

# class RedistributeOspfMatchIntExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeOspfMatchIntExtCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         intext=args.get( 'intext' ),
         mapName=args.get( 'MAP_NAME' ) )

def RedistributeOspfMatchIntExtCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute(
         mode,
         args[ 'proto' ],
         intext=args.get( 'intext' ) )

def RedistributeOspfMatchIntExtCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeOspfMatchIntExtCmd", mode, args )

def RedistributeOspfMatchIntExtCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RedistributeOspfMatchIntExtCmd", mode, args )

# class RedistributeOspf3MatchNssaExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
def RedistributeOspf3MatchNssaExtCmd_handleNormal( mode, args ):
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         intext='nssa-external',
         mapName=args.get( 'MAP_NAME' ),
         mType=args.get( 'NSSA_TYPE', 0 ) )

def RedistributeOspf3MatchNssaExtCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute(
         mode,
         args[ 'proto' ],
         intext='nssa-external' )

def RedistributeOspf3MatchNssaExtCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeOspf3MatchNssaExtCmd", mode, args )

def RedistributeOspf3MatchNssaExtCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RedistributeOspf3MatchNssaExtCmd", mode, args )

# class UserRedistributeBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
# class UserRedistributeCmd( UserRedistributeBaseCmd ):
# class UserRedistributeAfCmd( UserRedistributeBaseCmd ):
def UserRedistributeBaseCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      # Remove the trailing () characters
      rcfName = rcfName.rstrip( '()' )
   RedistributeBaseCmd_setRedistribute(
         mode,
         args[ 'proto' ],
         rcfName=rcfName )

def UserRedistributeBaseCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeBaseCmd_noRedistribute( mode, args[ 'proto' ],
         noOrDefault=noOrDefault )

def UserRedistributeBaseCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "UserRedistributeBaseCmd", mode, args )

def UserRedistributeBaseCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "UserRedistributeBaseCmd", mode, args )

# class RedistributeTunnelCmd( BgpCmdBaseClass ):
def RedistributeTunnelCmd_setRedistribute( mode, proto, rcfName=None ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = configForVrf( vrfName )
   redistConfig = redistributeConfig( addrFamily, config )
   redistribute = Tac.Value( 'Routing::Bgp::Redistribute',
                              proto, rcf=( rcfName or '' ) )
   redistConfig.addMember( redistribute )

def RedistributeTunnelCmd_noRedistribute( mode, proto ):
   vrfName = mode.vrfName
   addrFamily = mode.addrFamily
   config = configForVrf( vrfName )
   redistConfig = redistributeConfig( addrFamily, config )
   del redistConfig[ proto ]

def RedistributeTunnelCmd_handleNormal( mode, args ):
   rcfName = args.get( 'FUNCTION' )
   if rcfName:
      if rcfName.endswith( '()' ):
         # Remove the trailing () characters
         rcfName = rcfName[ : -2 ]
   RedistributeTunnelCmd_setRedistribute( mode,
                                             args.get( 'proto' ),
                                             rcfName=rcfName )

def RedistributeTunnelCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RedistributeTunnelCmd_noRedistribute( mode, args.get( 'proto' ) )

def RedistributeTunnelCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RedistributeTunnelCmd", mode, args )

def RedistributeTunnelCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RedistributeTunnelCmd", mode, args )

# class SetInstanceRouteDistinguisher( BgpCmdBaseClass ):
def SetInstanceRouteDistinguisher_handleNormal( mode, args ):
   rd = args[ 'RD' ]
   check = Tac.Value( 'Ira::RdAssignmentChecker',
                        LazyMount.force( rdAutoInputDir ),
                        LazyMount.force( rdConfigInputDir ),
                        mode.vrfName, rd )
   if not check.acceptable:
      errorMsg = check.errorMessage.replace( rd,
                     formatRd( rd, isAsdotConfigured( asnConfig ) ) )
      mode.addError( errorMsg )
      return
   bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
   bgpRouteDistinguisherInput.routeDistinguisher[ mode.vrfName ] = rd
   config = configForVrf( mode.vrfName )
   config.rdAll = False

   # For default VRF, write to bgp config as well. IRA doesn't publish VRF status
   # for default VRF, so the config written above at
   # ip/vrf/routeDistinguisherInputDir/config for default VRF will not be read
   # by BGP. We still write there for default VRF so that RdAssignmentChecker
   # can be used to check that the RD is not already in use.
   if mode.vrfName == DEFAULT_VRF:
      Globals.bgpConfig.routeDistinguisher = rd

def SetInstanceRouteDistinguisher_handleNoOrDefault( mode, args, noOrDefault ):
   bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
   del bgpRouteDistinguisherInput.routeDistinguisher[ mode.vrfName ]
   config = configForVrf( mode.vrfName )
   config.routeDistinguisher = 'INVALID'
   config.rdAll = False
   # For default VRF, write to bgp config as well. IRA doesn't publish VRF status
   # for default VRF, so the config written above at
   # ip/vrf/routeDistinguisherInputDir/config for default VRF will not be read
   # by BGP. We still write there for default VRF so that RdAssignmentChecker
   # can be used to check that the RD is not already in use.
   if mode.vrfName == DEFAULT_VRF:
      Globals.bgpConfig.routeDistinguisher = 'INVALID'

def SetInstanceRouteDistinguisher_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceRouteDistinguisher", mode, args )

def SetInstanceRouteDistinguisher_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetInstanceRouteDistinguisher", mode, args )

# class SetInstanceRouteDistinguisherRemoteDomain( BgpCmdBaseClass ):
def SetInstanceRouteDistinguisherRemoteDomain_handleNormal( mode, args ):
   rd = args[ 'RD' ]
   check = Tac.Value( 'Ira::RdAssignmentChecker',
                        LazyMount.force( rdAutoInputDir ),
                        LazyMount.force( rdConfigInputDir ),
                        mode.vrfName, rd )
   if not check.acceptable:
      errorMsg = check.errorMessage.replace( rd,
                     formatRd( rd, isAsdotConfigured( asnConfig ) ) )
      mode.addError( errorMsg )
      return
   rdRemoteDomainConfig = getBgpRouteDistinguisherRemoteDomainInput()
   rdRemoteDomainConfig.routeDistinguisher[ mode.vrfName ] = rd
   if 'all' in args:
      config = configForVrf( mode.vrfName )
      config.autoRd = False
      config.rdAll = True
      if mode.vrfName == DEFAULT_VRF:
         config.routeDistinguisher = rd
      rdConfig = getBgpRouteDistinguisherInput()
      rdConfig.routeDistinguisher[ mode.vrfName ] = rd
   elif 'remote' in args:
      config = configForVrf( mode.vrfName )
      config.rdAll = False

def SetInstanceRouteDistinguisherRemoteDomain_handleNoOrDefault(
      mode, args, noOrDefault ):
   rdRemoteDomainConfig = getBgpRouteDistinguisherRemoteDomainInput()
   del rdRemoteDomainConfig.routeDistinguisher[ mode.vrfName ]
   config = configForVrf( mode.vrfName )
   config.rdAll = False
   if 'all' in args:
      rdConfig = getBgpRouteDistinguisherInput()
      del rdConfig.routeDistinguisher[ mode.vrfName ]
      if mode.vrfName == DEFAULT_VRF:
         config.routeDistinguisher = 'INVALID'

def SetInstanceRouteDistinguisherRemoteDomain_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetInstanceRouteDistinguisherRemoteDomain", mode, args )

def SetInstanceRouteDistinguisherRemoteDomain_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "SetInstanceRouteDistinguisherRemoteDomain", mode, args )

# class RdAutoCommand( CliCommandClass ):
def RdAutoCommand_handler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   config = configForVrf( mode.vrfName )
   config.autoRd = True

def RdAutoCommand_noOrDefaultHandler( mode, args ):
   assert mode.vrfName == DEFAULT_VRF
   config = configForVrf( mode.vrfName )
   config.autoRd = False

# class RouteTargetImportExportBaseClass( object ):
def RouteTargetImportExportBaseClass_setRouteTargetCommon(
      mode, rt, attrName, vpnType=None, domain=None, warn=True ):
   ''' Handler for "route-target import/export <rt>" commands '''
   # Note: Pass warn=False to suppress warnings as in the case of 'both' keyword
   if not validateRouteTargetCommand( mode, vpnType, warn=warn ):
      return
   config = configForVrf( mode.vrfName )
   if conflictingRouteTargetConfigs( mode, config, mode.addrFamily, vpnType ):
      return
   extComm = routeTargetToExtCommU64Value( rt )
   if domain != 'remote':
      # add the regular RT
      vpnTypeAttr = getVpnTypeAttr( config, attrName, mode.addrFamily, vpnType )
      vpnTypeAttr[ extComm ] = True
   if domain in ( 'remote', 'all' ):
      # add the remote domain RT
      remoteAttrName = attrName + 'RemoteDomain'
      vpnTypeAttr = getVpnTypeAttr( config, remoteAttrName, mode.addrFamily,
            vpnType )
      vpnTypeAttr[ extComm ] = True
      if domain == 'remote':
         return
      # set rtAll flag
      rtAllAttrName = attrName + 'All'
      vpnTypeAttr = getVpnTypeAttr( config, rtAllAttrName, mode.addrFamily, vpnType )
      vpnTypeAttr[ extComm ] = True

def RouteTargetImportExportBaseClass_noRouteTargetCommon(
      mode, rt, attrName, vpnType=None, domain=None, warn=True ):
   ''' Handler for "no route-target import/export <rt>" commands" '''
   # Note: Pass warn=False to suppress warnings as in the case of 'both' keyword
   if not validateRouteTargetCommand( mode, vpnType, warn=warn ):
      return

   config = configForVrf( mode.vrfName )
   if rt is not None:
      extComm = routeTargetToExtCommU64Value( rt )
   if domain != 'remote':
      # delete the regular RT and rtAll flag
      vpnTypeAttr = getVpnTypeAttr( config, attrName, mode.addrFamily, vpnType )
      rtAllAttrName = attrName + 'All'
      rtAllAttr = getVpnTypeAttr( config, rtAllAttrName, mode.addrFamily, vpnType )
      if rt is not None:
         del vpnTypeAttr[ extComm ]
         if rtAllAttr is not None:
            del rtAllAttr[ extComm ]
         if domain != 'all':
            return
      else:
         vpnTypeAttr.clear()
         if rtAllAttr is not None:
            rtAllAttr.clear()
            domain = 'all'

   if domain in ( 'remote', 'all' ):
      # delete the remote domain RT
      attrName += 'RemoteDomain'
      vpnTypeAttr = getVpnTypeAttr( config, attrName, mode.addrFamily, vpnType )
      if rt is None:
         vpnTypeAttr.clear()
      else:
         del vpnTypeAttr[ extComm ]

# class MplsVpnVrfLocalLabelCmd( BgpCmdBaseClass ):
def MplsVpnVrfLocalLabelCmd_handleNormal( mode, args ):
   label = args.get( 'LABEL' )
   staticBase = MplsCli.labelRangeValue( LabelRangeInfo.rangeTypeStatic ).base
   labelIndex = label - staticBase
   config = configForVrf( mode.vrfName )
   value = VrfLocalLabelConfig( LabelRangeInfo.rangeTypeStatic, label, labelIndex )
   conflictingVrf = None
   for vrfName, vrfConfig in bgpVrfConfigDir.vrfConfig.items():
      if ( vrfName != mode.vrfName and 
           ( vrfConfig.vrfLocalLabelAfVpnV4 == value or 
              vrfConfig.vrfLocalLabelAfVpnV6 == value ) ): 
         conflictingVrf = vrfName
         break
   if conflictingVrf:
      mode.addError( "Cannot configure vrf-label %s for VRF %s, "
                     "while it is configured in VRF %s" % \
                           ( label, mode.vrfName, conflictingVrf ) )
      return
   if args.get( 'vpn-ipv4' ):
      config.vrfLocalLabelAfVpnV4 = value
   elif args.get( 'vpn-ipv6' ):
      config.vrfLocalLabelAfVpnV6 = value

def MplsVpnVrfLocalLabelCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if args.get( 'vpn-ipv4' ):
      config.vrfLocalLabelAfVpnV4 = config.vrfLocalLabelAfVpnV4Default
   elif args.get( 'vpn-ipv6' ):
      config.vrfLocalLabelAfVpnV6 = config.vrfLocalLabelAfVpnV6Default

def MplsVpnVrfLocalLabelCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "MplsVpnVrfLocalLabelCmd", mode, args )

def MplsVpnVrfLocalLabelCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "MplsVpnVrfLocalLabelCmd", mode, args )

# class RouteTargetImportExportVpnRouteCmd( BgpCmdBaseClass ):
def RouteTargetImportExportVpnRouteCmd_handleNormal( mode, args ):
   RouteTargetImportExportBaseClass_setRouteTargetCommon(
         mode,
         args[ 'RT' ],
         'routeTarget' + args[ 'IMPORT_EXPORT' ].title(),
         vpnType=args.get( 'VPN_TYPE' ) )

def RouteTargetImportExportVpnRouteCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetImportExportBaseClass_noRouteTargetCommon(
         mode,
         args.get( 'RT' ),
         'routeTarget' + args[ 'IMPORT_EXPORT' ].title(),
         vpnType=args.get( 'VPN_TYPE' ) )

def RouteTargetImportExportVpnRouteCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetImportExportVpnRouteCmd", mode, args )

def RouteTargetImportExportVpnRouteCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RouteTargetImportExportVpnRouteCmd", mode, args )

# class RouteTargetBothRouteCmd( BgpCmdBaseClass ):
def RouteTargetBothRouteCmd_setBothRouteTarget( mode, rt ):
   RouteTargetImportExportBaseClass_setRouteTargetCommon(
         mode,
         rt,
         'routeTargetImport' )
   RouteTargetImportExportBaseClass_setRouteTargetCommon(
         mode,
         rt,
         'routeTargetExport' )

def RouteTargetBothRouteCmd_noBothRouteTarget( mode, rt ):
   RouteTargetImportExportBaseClass_noRouteTargetCommon(
         mode,
         rt,
         'routeTargetImport',
         warn=( not mode.addrFamily == 'all' ) )
   RouteTargetImportExportBaseClass_noRouteTargetCommon(
         mode,
         rt,
         'routeTargetExport',
         warn=( not mode.addrFamily == 'all' ) )

def RouteTargetBothRouteCmd_handleNormal( mode, args ):
   RouteTargetBothRouteCmd_setBothRouteTarget( mode, args[ 'RT' ] )

def RouteTargetBothRouteCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetBothRouteCmd_noBothRouteTarget( mode, args.get( 'RT' ) )

def RouteTargetBothRouteCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetBothRouteCmd", mode, args )

def RouteTargetBothRouteCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetBothRouteCmd", mode, args )

# class RouteTargetImportExportRouteMapCmd( BgpCmdBaseClass ):
def RouteTargetImportExportRouteMapCmd_setRTRouteMapCommon(
      mode, rm, attrName, vpnType ):
   ''' Handler for "route-target import/export route-map" commands '''
   if not validateRouteTargetCommand( mode, vpnType, warn=True ):
      return
   config = configForVrf( mode.vrfName )
   if conflictingRouteMapConfigs( mode, config, mode.addrFamily, vpnType ):
      return
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   routeMapConfig = getattr( config, attr )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   routeMapConfig[ vpnAf ] = rm

def RouteTargetImportExportRouteMapCmd_noRTRouteMapCommon(
      mode, rm, attrName, vpnType ):
   ''' Handler for "no route-target import/export route-map" commands" '''
   assert vpnType is not None
   if not validateRouteTargetCommand( mode, vpnType, warn=False ):
      return
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   routeMapConfig = getattr( config, attr )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   rmEntry = routeMapConfig.get( vpnAf )
   if rmEntry and ( not rm or rmEntry == rm ):
      del routeMapConfig[ vpnAf ]

def RouteTargetImportExportRouteMapCmd_handleNormal( mode, args ):
   RouteTargetImportExportRouteMapCmd_setRTRouteMapCommon(
         mode,
         args[ 'MAP' ],
         'routeMap' + args[ 'IMPORT_EXPORT' ].title(),
         args[ 'VPN_TYPE' ] )

def RouteTargetImportExportRouteMapCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetImportExportRouteMapCmd_noRTRouteMapCommon(
         mode,
         args.get( 'MAP' ),
         'routeMap' + args[ 'IMPORT_EXPORT' ].title(),
         args[ 'VPN_TYPE' ] )

def RouteTargetImportExportRouteMapCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetImportExportRouteMapCmd", mode, args )

def RouteTargetImportExportRouteMapCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RouteTargetImportExportRouteMapCmd", mode, args )

# class RouteTargetExportRcfCmd( BgpCmdBaseClass ):
def RouteTargetExportRcfCmd_setRTRcfCommon(
      mode, rcfVpn, rcfVrf, attrName, vpnType ):
   ''' Handler for "route-target export rcf" commands '''
   if not validateRouteTargetCommand( mode, vpnType, warn=True ):
      return
   config = configForVrf( mode.vrfName )
   if conflictingRcfConfigs( mode, config, mode.addrFamily, vpnType ):
      return
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   # If there is no filter function applied the below replace() will just return
   # the default value '' else it will return the filter functions name
   rcfVrfFilterFuncName = rcfVrf.replace( '()', '' )
   rcfFuncName = rcfVpn.replace( '()', '' )
   rcfConfig = getattr( config, attr )
   rcfMainAndFilterConfig = Tac.Value( 'Routing::Bgp::RcfMainAndFilterConfig',
                                       rcfFuncName, rcfVrfFilterFuncName )
   rcfConfig[ vpnAf ] = rcfMainAndFilterConfig

def RouteTargetExportRcfCmd_noRTRcfCommon( mode, attrName, vpnType ):
   ''' Handler for "no route-target export rcf" commands" '''
   assert vpnType is not None
   if not validateRouteTargetCommand( mode, vpnType, warn=False ):
      return
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   rcfConfig = getattr( config, attr )
   if rcfConfig.get( vpnAf ):
      del rcfConfig[ vpnAf ]

def RouteTargetExportRcfCmd_handleNormal( mode, args ):
   RouteTargetExportRcfCmd_setRTRcfCommon(
         mode,
         args[ 'VPN_FUNCTION' ],
         args.get( 'VRF_FUNCTION', '' ),
         'rcfExport',
         args[ 'VPN_TYPE' ] )

def RouteTargetExportRcfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetExportRcfCmd_noRTRcfCommon(
         mode,
         'rcfExport',
         args[ 'VPN_TYPE' ] )

def RouteTargetExportRcfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetExportRcfCmd", mode, args )

def RouteTargetExportRcfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetExportRcfCmd", mode, args )

# class RouteTargetExportEvpnRcfCmd( BgpCmdBaseClass ):
def RouteTargetExportEvpnRcfCmd_handleNormal( mode, args ):
   RouteTargetExportRcfCmd_setRTRcfCommon(
         mode,
         args[ 'VPN_FUNCTION' ],
         '',
         'rcfExport',
         'evpn' )

def RouteTargetExportEvpnRcfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetExportRcfCmd_noRTRcfCommon(
         mode,
         'rcfExport',
         'evpn' )

def RouteTargetExportEvpnRcfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetExportEvpnRcfCmd", mode, args )

def RouteTargetExportEvpnRcfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetExportEvpnRcfCmd", mode, args )

# class RouteTargetImportRcfCmd( BgpCmdBaseClass ):
def RouteTargetImportRcfCmd_setRTRcfCommon(
      mode, rcfVrf, rcfVpn, attrName, vpnType ):
   ''' Handler for "route-target import rcf" commands '''
   if not validateRouteTargetCommand( mode, vpnType, warn=True ):
      return
   config = configForVrf( mode.vrfName )
   if conflictingRcfConfigs( mode, config, mode.addrFamily, vpnType ):
      return
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   # If there is no filter function applied the below replace() will just return
   # the default value '' else it will return the filter functions name
   rcfVpnFilterFuncName = rcfVpn.replace( '()', '' )
   rcfFuncName = rcfVrf.replace( '()', '' )
   rcfConfig = getattr( config, attr )
   rcfMainAndFilterConfig = Tac.Value( 'Routing::Bgp::RcfMainAndFilterConfig',
                                       rcfFuncName, rcfVpnFilterFuncName )
   rcfConfig[ vpnAf ] = rcfMainAndFilterConfig

def RouteTargetImportRcfCmd_noRTRcfCommon( mode, attrName, vpnType ):
   ''' Handler for "no route-target import rcf" commands" '''
   assert vpnType is not None
   if not validateRouteTargetCommand( mode, vpnType, warn=False ):
      return
   config = configForVrf( mode.vrfName )
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   rcfConfig = getattr( config, attr )
   if rcfConfig.get( vpnAf ):
      del rcfConfig[ vpnAf ]

def RouteTargetImportRcfCmd_handleNormal( mode, args ):
   RouteTargetImportRcfCmd_setRTRcfCommon(
         mode,
         args[ 'VRF_FUNCTION' ],
         args.get( 'VPN_FUNCTION', '' ),
         'rcfImport',
         args[ 'VPN_TYPE' ] )

def RouteTargetImportRcfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetImportRcfCmd_noRTRcfCommon(
         mode,
         'rcfImport',
         args[ 'VPN_TYPE' ] )

def RouteTargetImportRcfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetImportRcfCmd", mode, args )

def RouteTargetImportRcfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetImportRcfCmd", mode, args )

# class RouteTargetImportEvpnRcfCmd( BgpCmdBaseClass ):
def RouteTargetImportEvpnRcfCmd_handleNormal( mode, args ):
   RouteTargetImportRcfCmd_setRTRcfCommon(
         mode,
         args[ 'VRF_FUNCTION' ],
         '',
         'rcfImport',
         'evpn' )

def RouteTargetImportEvpnRcfCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetImportRcfCmd_noRTRcfCommon(
         mode,
         'rcfImport',
         'evpn' )

def RouteTargetImportEvpnRcfCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetImportEvpnRcfCmd", mode, args )

def RouteTargetImportEvpnRcfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetImportEvpnRcfCmd", mode, args )

# class RouteTargetExportBaseCmd( object ):
def RouteTargetExportBaseCmd_setRTExportImportedRoute( mode, vpnType=None ):
   ''' Handler for "route-target export imported-route" commands '''
   if not validateRouteTargetCommand( mode, vpnType ):
      return
   config = configForVrf( mode.vrfName )
   if conflictingRouteTargetConfigs( mode, config, mode.addrFamily, vpnType ):
      return
   attrName = 'allowImportedRouteToExport'
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   allowImportedRouteToExport = getattr( config, attr )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   allowImportedRouteToExport[ vpnAf ] = True

def RouteTargetExportBaseCmd_noRTExportImportedRoute( mode, vpnType=None ):
   ''' Handler for "no route-target export imported-route" commands '''
   if not validateRouteTargetCommand( mode, vpnType ):
      return
   config = configForVrf( mode.vrfName )
   attrName = 'allowImportedRouteToExport'
   attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
   allowImportedRouteToExport = getattr( config, attr )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   del allowImportedRouteToExport[ vpnAf ]

# class RouteTargetExportEvpnCmd( RouteTargetExportBaseCmd, BgpCmdBaseClass ):
def RouteTargetExportEvpnCmd_handleNormal( mode, args ):
   RouteTargetExportBaseCmd_setRTExportImportedRoute( mode, vpnType='evpn' )

def RouteTargetExportEvpnCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetExportBaseCmd_noRTExportImportedRoute( mode, vpnType='evpn' )

def RouteTargetExportEvpnCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetExportEvpnCmd", mode, args )

def RouteTargetExportEvpnCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetExportEvpnCmd", mode, args )

# class RouteTargetExportAfVpnCmd( RouteTargetExportBaseCmd, BgpCmdBaseClass ):
def RouteTargetExportAfVpnCmd_handleNormal( mode, args ):
   RouteTargetExportBaseCmd_setRTExportImportedRoute(
         mode,
         vpnType=args[ 'VPN_TYPE' ] )

def RouteTargetExportAfVpnCmd_handleNoOrDefault( mode, args, noOrDefault ):
   RouteTargetExportBaseCmd_noRTExportImportedRoute(
         mode,
         vpnType=args[ 'VPN_TYPE' ] )

def RouteTargetExportAfVpnCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouteTargetExportAfVpnCmd", mode, args )

def RouteTargetExportAfVpnCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouteTargetExportAfVpnCmd", mode, args )

# class RouteTargetImportExportEvpnRemoteDomainRouteCmd( BgpCmdBaseClass ):
def RouteTargetImportExportEvpnRemoteDomainRouteCmd_handleNormal( mode, args ):
   domain = args.get( 'remote', None ) or args.get( 'all' )
   RouteTargetImportExportBaseClass_setRouteTargetCommon( mode,
         args[ 'RT' ], 'routeTarget' + args[ 'IMPORT_EXPORT' ].title(),
         vpnType='evpn', domain=domain )

def RouteTargetImportExportEvpnRemoteDomainRouteCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   domain = args.get( 'remote', None ) or args.get( 'all', None )
   RouteTargetImportExportBaseClass_noRouteTargetCommon( mode,
         args.get( 'RT' ), 'routeTarget' + args[ 'IMPORT_EXPORT' ].title(),
         vpnType='evpn', domain=domain )

def RouteTargetImportExportEvpnRemoteDomainRouteCmd_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "RouteTargetImportExportEvpnRemoteDomainRouteCmd", mode, args )

def RouteTargetImportExportEvpnRemoteDomainRouteCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RouteTargetImportExportEvpnRemoteDomainRouteCmd", mode, args )

# class ClearBgpAclCounters( CliCommandClass ):
def ClearBgpAclCounters_handler( mode, args ):
   aclFilter = None
   if 'ipv4' in args:
      aclFilter = 'ip'
   elif 'ipv6' in args:
      aclFilter = 'ipv6'
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclFilter )

# class RouterBgpAddrFamilyCommand( BgpCmdBaseClass ):
def RouterBgpAddrFamilyCommand_gotoAfMode( mode, af ):
   if ( af == 'evpn' and
         ( getEffectiveProtocolModel( mode.session.mode ) ==
            ProtocolAgentModelType.ribd ) ):
      mode.addWarning( 'Routing protocols model multi-agent must be '
                        'configured for EVPN address-family' )

   # Map address-family from args to the appropriate goto/delete methods.
   afModeMap = {
      'ipv4': RouterBgpBaseAfIpUniMode,
      'ipv6': RouterBgpBaseAfIpv6UniMode,
      'ipv4 labeled-unicast': RouterBgpBaseAfLabelV4Mode,
      'ipv6 labeled-unicast': RouterBgpBaseAfLabelV6Mode,
      'ipv4 multicast': RouterBgpBaseAfIpMulticastMode,
      'ipv6 multicast': RouterBgpBaseAfIpv6MulticastMode,
      'ipv4 sr-te': RouterBgpBaseAfSrTeMode,
      'ipv6 sr-te': RouterBgpBaseAfSrTeMode,
      'evpn': RouterBgpBaseAfEvpnMode,
   }
   childMode = mode.childMode( afModeMap.get( af ), addrFamily=af )
   mode.session_.gotoChildMode( childMode )

def RouterBgpAddrFamilyCommand_deleteAfMode( mode, af ):
   resetBgpAfModeConfig( Globals.bgpConfig, af, mode.vrfName )
   # On delete, address family may have plugin-specific write-mounts to reset.
   # Evpn currently needs this hook as '[ no | default ] address-family evpn'
   # cannot be moved to Evpn package until all commands registered in
   # RouterBgpBaseAfEvpnMode are moved from BgpCommon to Evpn.
   for hook in deleteRouterBgpAfiSafiHook.extensions():
      hook( af )

def RouterBgpAddrFamilyCommand_handleNormal( mode, args ):
   RouterBgpAddrFamilyCommand_gotoAfMode( mode, args[ 'af' ] )

def RouterBgpAddrFamilyCommand_handleNoOrDefault( mode, args, noOrDefault ):
   RouterBgpAddrFamilyCommand_deleteAfMode( mode, args[ 'af' ] )

def RouterBgpAddrFamilyCommand_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpAddrFamilyCommand", mode, args )

def RouterBgpAddrFamilyCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpAddrFamilyCommand", mode, args )

# class RouterBgpLinkStateCommand( BgpCmdBaseClass ):
def RouterBgpLinkStateCommand_handleNormal( mode, args ):
   if ( getEffectiveProtocolModel( mode.session.mode ) ==
         ProtocolAgentModelType.ribd ):
      mode.addWarning( 'Routing protocols model multi-agent must be '
                        'configured for BGP Link State address-family' )
   childMode = mode.childMode( RouterBgpBaseAfLinkStateMode )
   mode.session_.gotoChildMode( childMode )

def RouterBgpLinkStateCommand_handleNoOrDefault( mode, args, noOrDefault ):
   resetBgpAfModeConfig( Globals.bgpConfig, 'link-state', mode.vrfName )
   lsImportConfig.protoConfig.clear()
   lsRoleConfig.protoConfig.clear()

def RouterBgpLinkStateCommand_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpLinkStateCommand", mode, args )

def RouterBgpLinkStateCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpLinkStateCommand", mode, args )

# class RouterBgpDpsCommand( BgpCmdBaseClass ):
def RouterBgpDpsCommand_handleNormal( mode, args ):
   if ( getEffectiveProtocolModel( mode.session.mode ) ==
         ProtocolAgentModelType.ribd ):
      mode.addWarning( 'Routing protocols model multi-agent must be '
                        'configured for BGP path-selection address-family' )
   childMode = mode.childMode( RouterBgpBaseAfDpsMode )
   mode.session_.gotoChildMode( childMode )

def RouterBgpDpsCommand_handleNoOrDefault( mode, args, noOrDefault ):
   resetBgpAfModeConfig( Globals.bgpConfig, 'path-selection', mode.vrfName )

def RouterBgpDpsCommand_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpDpsCommand", mode, args )

def RouterBgpDpsCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpDpsCommand", mode, args )

# class RouterBgpVrfAfCommand( BgpCmdBaseClass ):
def RouterBgpVrfAfCommand_handleNormal( mode, args ):
   if mode.vrfName == DEFAULT_VRF:
      NewMode = RouterBgpDefaultVrfAfMode
   elif args.get( 'ipv4' ):
      NewMode = RouterBgpVrfAfIpMode
   else:
      NewMode = RouterBgpVrfAfIp6Mode

   childMode = mode.childMode(
      NewMode, addrFamily=args[ 'af' ], vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def RouterBgpVrfAfCommand_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   resetBgpAfModeConfig( config, args[ 'af' ], mode.vrfName )

def RouterBgpVrfAfCommand_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpVrfAfCommand", mode, args )

def RouterBgpVrfAfCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpVrfAfCommand", mode, args )

# class RouterBgpVrfAfMulticastCommand( BgpCmdBaseClass ):
def RouterBgpVrfAfMulticastCommand_handleNormal( mode, args ):
   addrFamily = args[ 'af' ] + ' multicast'
   newMode = RouterBgpVrfAfIpMulticastMode
   if args.get( 'ipv6' ):
      newMode = RouterBgpVrfAfIpv6MulticastMode
   childMode = mode.childMode( newMode, addrFamily=addrFamily, vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def RouterBgpVrfAfMulticastCommand_handleNoOrDefault( mode, args, noOrDefault ):
   addrFamily = args[ 'af' ]
   config = configForVrf( mode.vrfName )
   addrFamily = addrFamily + ' multicast'
   resetBgpAfModeConfig( config, addrFamily, mode.vrfName )

def RouterBgpVrfAfMulticastCommand_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpVrfAfMulticastCommand", mode, args )

def RouterBgpVrfAfMulticastCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpVrfAfMulticastCommand", mode, args )

# class BgpVrfCmd( CliCommandClass ):
def BgpVrfCmd_validateVrfNameLength( mode, vrfName ):
   maxLength = Tac.Type( "L3::VrfName" ).maxLength
   if len( vrfName ) > maxLength:
      mode.addErrorAndStop(
         "'%s' too long: must be no more than %d characters"
         % ( vrfName, maxLength ) )
   return True

def BgpVrfCmd_handler( mode, args ):
   vrfName = args[ 'VRFNAME' ]
   if ( vrfName in VRFNAMES_RESERVED and vrfName != DEFAULT_VRF ):
      mode.addError( "vrf name %s is reserved." % vrfName )
      return

   assert BgpVrfCmd_validateVrfNameLength( mode, vrfName )

   # TODO: check number of vrfs and warn
   # TODO: check vrfCapability and warn
   # TODO: check if vrf is configured or not and warn

   IraIpCli.warnIfRoutingDisabled( mode, vrfName )
   vrfMode = ( RouterBgpDefaultVrfMode if vrfName == DEFAULT_VRF else
                  RouterBgpVrfMode )
   childMode = mode.childMode( vrfMode, vrfName=vrfName )
   mode.session.gotoChildMode( childMode )

def BgpVrfCmd_noOrDefaultHandler( mode, args ):
   vrfName = args[ 'VRFNAME' ]

   if not BgpVrfCmd_validateVrfNameLength( mode, vrfName ):
      return

   # cleanup bgp vrf service ACL config
   vrfMode = ( RouterBgpDefaultVrfMode if vrfName == DEFAULT_VRF else
               RouterBgpVrfMode )
   childMode = mode.childMode( vrfMode, vrfName=vrfName )
   for aclType in [ 'ip', 'ipv6' ]:
      setServiceAclCommon( childMode, aclType, no=True )

   if vrfName == DEFAULT_VRF:
      deleteBgpDefaultVrfConfig()
      return
   assert vrfName not in VRFNAMES_RESERVED
   if vrfName in bgpVrfConfigDir.vrfConfig:
      for hook in deleteRouterBgpVrfHook.extensions():
         hook( vrfName )
   if vrfName in ucmpVrfConfigDir.vrfConfig:
      for hook in deleteRouterBgpUcmpVrfHook.extensions():
         hook( vrfName )

# class SetBestpathCompatibleEbgpIbgp( CliCommandClass ):
def SetBestpathCompatibleEbgpIbgp_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.compatibleEBgpIBgp = 'isTrue'

def SetBestpathCompatibleEbgpIbgp_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.compatibleEBgpIBgp = (
         getExplicitDefaultTristate( mode, 'compatibleEBgpIBgp' ) )

def SetBestpathCompatibleEbgpIbgp_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.compatibleEBgpIBgp = 'isInvalid'

# class BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd( CliCommandClass ):
def BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.nexthopResolutionNhgTunnelEcmpDisabled = 'isTrue'

def BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.nexthopResolutionNhgTunnelEcmpDisabled = 'isFalse'

def BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd_defaultHandler(
      mode, args ):
   config = configForVrf( mode.vrfName )
   config.nexthopResolutionNhgTunnelEcmpDisabled = 'isInvalid'

# class SetSkipAigpMetricCmd( CliCommandClass ):
def SetSkipAigpMetricCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.skipAigpMetric = 'isTrue'

def SetSkipAigpMetricCmd_noHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.skipAigpMetric = ( getExplicitDefaultTristate( mode, 'skipAigpMetric' ) )

def SetSkipAigpMetricCmd_defaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.skipAigpMetric = 'isInvalid'

# class SkipRibBestpathCommand( CliCommandClass ):
def SkipRibBestpathCommand_setSkipRibBestpath(
      afiSafi, skipRib=False, skipBestpath=False, vrfName=DEFAULT_VRF ):
   config = configForVrf( vrfName )
   if afiSafi == 'ipv4':
      config.skipRibIPv4Uni = skipRib
      config.skipBestpathIPv4Uni = skipBestpath
   elif afiSafi == 'ipv6':
      config.skipRibIPv6Uni = skipRib
      config.skipBestpathIPv6Uni = skipBestpath
   else:
      if afiSafi == 'flow-spec ipv4':
         config.skipRibIPv4Flowspec = skipRib
         config.skipBestpathIPv4Flowspec = skipBestpath
      elif afiSafi == 'flow-spec ipv6':
         config.skipRibIPv6Flowspec = skipRib
         config.skipBestpathIPv6Flowspec = skipBestpath

def SkipRibBestpathCommand_handler( mode, args ):
   SkipRibBestpathCommand_setSkipRibBestpath(
      mode.addrFamily, skipRib=True, skipBestpath='bestpath-selection' in args,
      vrfName=mode.vrfName )

def SkipRibBestpathCommand_noOrDefaultHandler( mode, args ):
   SkipRibBestpathCommand_setSkipRibBestpath(
      mode.addrFamily, skipRib=False, skipBestpath=False, vrfName=mode.vrfName )

# class SetVpnExportPathsMethod( BgpCmdBaseClass ):
def SetVpnExportPathsMethod_setVpnExportPathsMethod( mode, layer, method ):
   value = ( 'isTrue' if method == 'best' else 'isFalse' )
   config = configForVrf( mode.vrfName )
   if layer is None or layer == 'layer-3':
      config.vpnExportMethodL3Best = value
   if layer is None or layer == 'layer-2':
      config.vpnExportMethodL2Best = value

def SetVpnExportPathsMethod_noVpnExportPathsMethod( mode, layer, value ):
   config = configForVrf( mode.vrfName )
   if layer is None or layer == 'layer-3':
      config.vpnExportMethodL3Best = value
   if layer is None or layer == 'layer-2':
      config.vpnExportMethodL2Best = value

def SetVpnExportPathsMethod_handleNormal( mode, args ):
   SetVpnExportPathsMethod_setVpnExportPathsMethod(
         mode, args.get( 'LAYER' ), args[ 'METHOD' ] )

def SetVpnExportPathsMethod_handleNoOrDefault( mode, args, noOrDefault ):
   value = ( 'isFalse' if ( noOrDefault == NoOrDefault.NO ) else 'isInvalid' )
   SetVpnExportPathsMethod_noVpnExportPathsMethod( mode, args.get( 'LAYER' ), value )

def SetVpnExportPathsMethod_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetVpnExportPathsMethod", mode, args )

def SetVpnExportPathsMethod_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetVpnExportPathsMethod", mode, args )

# class ConfigLsImportCmd( CliCommandClass ):
ConfigLsImportCmd_kwProtoMap = {
   'isis': 'protoIsis',
}

def ConfigLsImportCmd_handler( mode, args ):
   proto = ConfigLsImportCmd_kwProtoMap[ args[ 'PROTOCOL' ] ]
   if proto not in [ 'protoIsis' ]:
      raise ValueError()

   instName = args.get( 'INST' )
   identifier = args.get( 'ID' )

   twoWayCheckDisabled = ( 'link' in args ) and \
                         ( 'connectivity-check' in args ) and \
                         ( 'disabled' in args )

   updateLsImportCfg( lsImportConfig, proto, instName, identifier=identifier,
                      twoWayCheckDisabled=twoWayCheckDisabled )
   updateLsProducerRoleCfg( lsRoleConfig, proto, True )

def ConfigLsImportCmd_noOrDefaultHandler( mode, args ):
   proto = ConfigLsImportCmd_kwProtoMap[ args.get( 'PROTOCOL' ) ]
   instName = args.get( 'INST' )
   if removeLsImportCfg( lsImportConfig, proto, instName=instName ):
      # All import instances have been removed, so remove the producer role as well.
      updateLsProducerRoleCfg( lsRoleConfig, proto, False )

# class ConfigLsRoleCmd( CliCommandClass ):
ConfigLsRoleCmd_kwProtoMap = {
   'isis': 'protoIsis',
}

def ConfigLsRoleCmd_handler( mode, args ):
   proto = ConfigLsRoleCmd_kwProtoMap[ args[ 'PROTOCOL' ] ]
   if proto not in [ 'protoIsis' ]:
      raise ValueError()
   producerEnabled = proto in lsImportConfig.protoConfig
   setLsProtoRoleCfg( lsRoleConfig, proto,
                      explicitRoleEnabled=True,
                      propagatorEnabled=True,
                      consumerEnabled=False,
                      producerEnabled=producerEnabled )

def ConfigLsRoleCmd_noOrDefaultHandler( mode, args ):
   proto = ConfigLsRoleCmd_kwProtoMap[ args.get( 'PROTOCOL' ) ]
   producerEnabled = proto in lsImportConfig.protoConfig
   setLsProtoRoleCfg( lsRoleConfig, proto,
                      explicitRoleEnabled=False,
                      propagatorEnabled=False,
                      consumerEnabled=False,
                      producerEnabled=producerEnabled )

# class ConfigLsImportBgpEpeCmd( CliCommandClass ):
# BGP EPE (AID11677) supports importing from a single instance only.
ConfigLsImportBgpEpeCmd_instName = 'peer segments'
ConfigLsImportBgpEpeCmd_proto = 'protoBgp'

def ConfigLsImportBgpEpeCmd_handler( mode, args ):
   proto = ConfigLsImportBgpEpeCmd_proto
   instName = ConfigLsImportBgpEpeCmd_instName
   updateLsImportCfg( lsImportConfig, proto, instName )
   updateLsProducerRoleCfg( lsRoleConfig, proto, True )

def ConfigLsImportBgpEpeCmd_noOrDefaultHandler( mode, args ):
   proto = ConfigLsImportBgpEpeCmd_proto
   instName = ConfigLsImportBgpEpeCmd_instName
   removeLsImportCfg( lsImportConfig, proto, instName=instName )
   updateLsProducerRoleCfg( lsRoleConfig, proto, False )

# class ConfigLsImportDpsCmd( CliCommandClass ):
# DPS supports importing from a single default instance only.
ConfigLsImportDpsCmd_instName = 'default'
ConfigLsImportDpsCmd_proto = 'protoDps'

def ConfigLsImportDpsCmd_handler( mode, args ):
   proto = ConfigLsImportDpsCmd_proto
   instName = ConfigLsImportDpsCmd_instName
   updateLsImportCfg( lsImportConfig, proto, instName )
   updateLsProducerRoleCfg( lsRoleConfig, proto, True )

def ConfigLsImportDpsCmd_noOrDefaultHandler( mode, args ):
   proto = ConfigLsImportDpsCmd_proto
   instName = ConfigLsImportDpsCmd_instName
   removeLsImportCfg( lsImportConfig, proto, instName=instName )
   updateLsProducerRoleCfg( lsRoleConfig, proto, False )

# class ConfigLsRoleDpsCmd( CliCommandClass ):
ConfigLsRoleDpsCmd_proto = 'protoDps'

def ConfigLsRoleDpsCmd_handler( mode, args ):
   proto = ConfigLsRoleDpsCmd_proto
   setLsProtoRoleCfg( lsRoleConfig, proto, True, # explicitRole
                      'propagator' in args,
                      'consumer' in args,
                      proto in lsImportConfig.protoConfig )

def ConfigLsRoleDpsCmd_noOrDefaultHandler( mode, args ):
   proto = ConfigLsRoleDpsCmd_proto
   setLsProtoRoleCfg( lsRoleConfig, proto, False, # explicitRole
                      False, # propagator
                      False, # consumer
                      proto in lsImportConfig.protoConfig )

# class ResolutionRibsCmd( CliCommandClass ):
def ResolutionRibsCmd_handler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )
   setNhResRibsUdpTunnelConfig( mode.addrFamily, args, vrfName=mode.vrfName )

def ResolutionRibsCmd_noOrDefaultHandler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )
   setNhResRibsUdpTunnelConfig( mode.addrFamily, args, no=True,
                                vrfName=mode.vrfName )

# class ResolutionRibsLuCmd( CliCommandClass ):
def ResolutionRibsLuCmd_handler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

def ResolutionRibsLuCmd_noOrDefaultHandler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )

# class Resolution6peRibsCmd( CliCommandClass ):
def Resolution6peRibsCmd_handler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

def Resolution6peRibsCmd_noOrDefaultHandler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )

# class ResolutionMplsRibsCmd( CliCommandClass ):
def ResolutionMplsRibsCmd_handler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

def ResolutionMplsRibsCmd_noOrDefaultHandler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )

# class ResolutionVxlanRibsCmd( CliCommandClass ):
def ResolutionVxlanRibsCmd_handler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

def ResolutionVxlanRibsCmd_noOrDefaultHandler( mode, args ):
   nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )

# class SetNexthopLuOriginateLfibBackupUninstallDelayCmd( BgpCmdBaseClass ):
def SetNexthopLuOriginateLfibBackupUninstallDelayCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.lfibUninstallDelay = args[ 'UNINSTALL_DELAY' ]
   mode.addWarning(
      "Make sure that this delay value is large enough to cover "
      "the time it takes the network to reconverge after a peer failure." )

def SetNexthopLuOriginateLfibBackupUninstallDelayCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if noOrDefault == NoOrDefault.NO:
      config.lfibUninstallDelay = 0
   else:
      config.lfibUninstallDelay = config.lfibUninstallDelayDefault

def SetNexthopLuOriginateLfibBackupUninstallDelayCmd_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "SetNexthopLuOriginateLfibBackupUninstallDelayCmd", mode, args )

def SetNexthopLuOriginateLfibBackupUninstallDelayCmd_noOrDefaultHandler(
      mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "SetNexthopLuOriginateLfibBackupUninstallDelayCmd", mode, args )

# class SetNhLuOrigLfibBackupIpFwdCmd( BgpCmdBaseClass ):
def SetNhLuOrigLfibBackupIpFwdCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   config.nhLuOrigLfibBackupIpFwd = 'isTrue'

def SetNhLuOrigLfibBackupIpFwdCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if noOrDefault == NoOrDefault.NO:
      config.nhLuOrigLfibBackupIpFwd = 'isFalse'
   else:
      config.nhLuOrigLfibBackupIpFwd = 'isInvalid'

def SetNhLuOrigLfibBackupIpFwdCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetNhLuOrigLfibBackupIpFwdCmd", mode, args )

def SetNhLuOrigLfibBackupIpFwdCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetNhLuOrigLfibBackupIpFwdCmd", mode, args )

# class SetEpeNexthopOriginateCmd( BgpCmdBaseClass ):
def SetEpeNexthopOriginateCmd_setEpeNexthopOriginateCommon(
      mode, nexthop, no=False ):
   if nexthop.af == 'ipv6':
      if mode.addrFamily == 'ipv4 labeled-unicast':
         mode.addErrorAndStop( "IPv6 next hop may not be specified in IPv4 mode" )
      elif nexthop.isLinkLocal:
         mode.addErrorAndStop( "Link local address may not be specified" )
   else:
      if mode.addrFamily == 'ipv6 labeled-unicast':
         mode.addErrorAndStop( "IPv4 next hop may not be specified in IPv6 mode" )

   vrfName = mode.vrfName
   config = configForVrf( vrfName )
   if mode.addrFamily == 'ipv4 labeled-unicast':
      attrName = 'epeNexthopAfV4LabeledUni'
   else:
      attrName = 'epeNexthopAfV6LabeledUni'
   nexthopSet = getattr( config, attrName )

   if no:
      del nexthopSet[ nexthop ]
   else:
      nexthopSet.add( nexthop )

def SetEpeNexthopOriginateCmd_handleNormal( mode, args ):
   SetEpeNexthopOriginateCmd_setEpeNexthopOriginateCommon(
         mode,
         args[ 'ADDR' ] )

def SetEpeNexthopOriginateCmd_handleNoOrDefault( mode, args, noOrDefault ):
   SetEpeNexthopOriginateCmd_setEpeNexthopOriginateCommon(
         mode,
         args[ 'ADDR' ],
         no=True )

def SetEpeNexthopOriginateCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetEpeNexthopOriginateCmd", mode, args )

def SetEpeNexthopOriginateCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetEpeNexthopOriginateCmd", mode, args )

# class SetEpeNexthopOriginateLfibBackupIpFwdCmd( BgpCmdBaseClass ):
def SetEpeNexthopOriginateLfibBackupIpFwdCmd_setEpeNexthopLfibBackupIpFwd(
      mode, nexthop, no=False ):
   if nexthop.af == 'ipv6':
      if mode.addrFamily == 'ipv4 labeled-unicast':
         mode.addErrorAndStop( "IPv6 next hop may not be specified in IPv4 mode" )
      elif nexthop.isLinkLocal:
         mode.addErrorAndStop( "Link local address may not be specified" )
   else:
      if mode.addrFamily == 'ipv6 labeled-unicast':
         mode.addErrorAndStop( "IPv4 next hop may not be specified in IPv6 mode" )

   vrfName = mode.vrfName
   config = configForVrf( vrfName )
   if mode.addrFamily == 'ipv4 labeled-unicast':
      attrName = 'epeNexthopLfibBackupIpFwdAfV4LabeledUni'
   else:
      attrName = 'epeNexthopLfibBackupIpFwdAfV6LabeledUni'
   nexthopList = getattr( config, attrName )

   if no == NoOrDefault.DEFAULT:
      del nexthopList[ nexthop ]
   elif no == NoOrDefault.NO:
      nexthopList[ nexthop ] = False
   else:
      nexthopList[ nexthop ] = True

def SetEpeNexthopOriginateLfibBackupIpFwdCmd_handleNormal( mode, args ):
   SetEpeNexthopOriginateLfibBackupIpFwdCmd_setEpeNexthopLfibBackupIpFwd(
         mode,
         args[ 'ADDR' ] )

def SetEpeNexthopOriginateLfibBackupIpFwdCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   SetEpeNexthopOriginateLfibBackupIpFwdCmd_setEpeNexthopLfibBackupIpFwd(
         mode,
         args[ 'ADDR' ],
         no=noOrDefault )

def SetEpeNexthopOriginateLfibBackupIpFwdCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetEpeNexthopOriginateLfibBackupIpFwdCmd", mode, args )

def SetEpeNexthopOriginateLfibBackupIpFwdCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "SetEpeNexthopOriginateLfibBackupIpFwdCmd", mode, args )

# class RouterBgpOrrPositionCmd( BgpNEDCmdBaseClass ):
def RouterBgpOrrPositionCmd_handleNormal( mode, args ):
   emitWarningIfOrrNotSupportedInRibdMode( mode )
   positionName = args[ 'POSITION_NAME' ]
   childMode = mode.childMode( RouterBgpOrrPositionMode,
                                 orrPositionName=positionName )
   mode.session.gotoChildMode( childMode )
   if positionName not in Globals.bgpConfig.orrNamedPosition:
      position = Tac.Value( 'Routing::Bgp::OrrPositionConfig', positionName )
      Globals.bgpConfig.orrNamedPosition.addMember( position )

def RouterBgpOrrPositionCmd_handleNoOrDefault( mode, args, noOrDefault ):
   positionName = args[ 'POSITION_NAME' ]
   del Globals.bgpConfig.orrNamedPosition[ positionName ]

def RouterBgpOrrPositionCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "RouterBgpOrrPositionCmd", mode, args )

def RouterBgpOrrPositionCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler( "RouterBgpOrrPositionCmd", mode, args )

# class RouterBgpOrrPositionPrefixCmd( BgpNEDCmdBaseClass ):
def RouterBgpOrrPositionPrefixCmd_writeNewPosition( positionName, prefixes ):
   newOrrPosition = Tac.Value( 'Routing::Bgp::OrrPositionConfig', positionName )
   # Cannot use priority zero for a collection in a Nominal, thus start=1
   for priority, pfx in enumerate( prefixes, start=1 ):
      newOrrPosition.prefixByPriority[ priority ] = pfx
   Globals.bgpConfig.orrNamedPosition.addMember( newOrrPosition )

def RouterBgpOrrPositionPrefixCmd_handleNormal( mode, args ):
   orrPosition = Globals.bgpConfig.orrNamedPosition[ mode.orrPositionName ]

   prefixes = list( orrPosition.prefixByPriority.values() )
   maxPrefixes = 10
   if len( prefixes ) >= maxPrefixes:
      mode.addError( "Can configure at most %d IGP prefixes for ORR position %r"
                     % ( maxPrefixes, mode.orrPositionName ) )
      return

   prefix = args[ 'PREFIX' ]
   position = args.get( 'POSITION' )

   # We will support a smallish number of prefixes, so use a simple
   # algorithm where we have a python ordered list, prefixes, that
   # we update, and then write into the ordered collection with
   # incrementing priority values for each prefix.
   if position is not None:
      refPrefix = args[ 'REF_PREFIX' ]
      if not refPrefix in prefixes:
         mode.addError( "IGP prefix '%s' is not configured" % ( refPrefix ) )
         return
      # If an existing prefix is specified with before/after, it
      # works as a move, so remove any existing entry.
      try:
         # Prefix already present, remove it and re-add in new
         # position below.
         prefixes.remove( prefix )
      except ValueError:
         # Not already present, so an addition.
         pass
      for idx, existingPrefix in enumerate( prefixes ):
         if refPrefix != existingPrefix:
            continue
         if position == 'after':
            idx += 1
         prefixes.insert( idx, prefix )
         break
   else:
      if prefix in prefixes:
         # Do not allow addition of duplicate.
         mode.addWarning( "IGP prefix '%s' is already configured for "
                           "ORR position %r" % ( prefix, mode.orrPositionName ) )
         return
      # Insert at the end
      prefixes.append( prefix )
   # Write the new priority to prefix mapping.
   RouterBgpOrrPositionPrefixCmd_writeNewPosition( mode.orrPositionName, prefixes )

def RouterBgpOrrPositionPrefixCmd_handleNoOrDefault( mode, args, noOrDefault ):
   prefix = args[ 'PREFIX' ]
   orrPosition = Globals.bgpConfig.orrNamedPosition[ mode.orrPositionName ]
   prefixes = list( orrPosition.prefixByPriority.values() )
   try:
      prefixes.remove( prefix )
   except ValueError:
      # Prefix is not present, nothing to do.
      return
   # Write the new priority to prefix mapping.
   RouterBgpOrrPositionPrefixCmd_writeNewPosition( mode.orrPositionName, prefixes )

def RouterBgpOrrPositionPrefixCmd_handler( mode, args ):
   BgpNEDCmdBaseClass_handler( "RouterBgpOrrPositionPrefixCmd", mode, args )

def RouterBgpOrrPositionPrefixCmd_noOrDefaultHandler( mode, args ):
   BgpNEDCmdBaseClass_noOrDefaultHandler(
      "RouterBgpOrrPositionPrefixCmd", mode, args )

# Classes for TrafficPolicy, VrfSelectionPolicy FieldSetMappingsCommand
def FieldSetMappingsCommand_handler( mode, args, mapCtx='tp' ):
   assert mapCtx in [ 'tp', 'vsp' ]
   modeClass = TrafficPolicyFieldSetMappingsMode if mapCtx == 'tp' else \
      VrfSelectionPolicyFieldSetMappingsMode
   attrName = mapCtx + 'FieldSetMappingDisabled'

   childMode = mode.childMode( modeClass, mapCtx=mapCtx, vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )
   config = configForVrf( mode.vrfName )
   if getattr( config, attrName ) == 'isInvalid':
      setattr( config, attrName, 'isFalse' )

def FieldSetMappingsCommand_noOrDefaultHandler( mode, args, mapCtx='tp' ):
   assert mapCtx in [ 'tp', 'vsp' ]
   config = configForVrf( mode.vrfName )
   setattr( config, mapCtx + 'FieldSetMappingDisabled', 'isInvalid' )
   getattr( config, mapCtx + 'FieldSetMappingBindingsIpv4' ).clear()
   getattr( config, mapCtx + 'FieldSetMappingBindingsIpv6' ).clear()

# class TrafficPolicyFieldSetMappingsCommand( CliCommandClass ):
def TrafficPolicyFieldSetMappingsCommand_handler( mode, args ):
   FieldSetMappingsCommand_handler( mode, args, mapCtx='tp' )

def TrafficPolicyFieldSetMappingsCommand_noOrDefaultHandler( mode, args ):
   FieldSetMappingsCommand_noOrDefaultHandler( mode, args, mapCtx='tp' )

# class VrfSelectionPolicyFieldSetMappingsCommand( CliCommandClass ):
def VrfSelectionPolicyFieldSetMappingsCommand_handler( mode, args ):
   FieldSetMappingsCommand_handler( mode, args, mapCtx='vsp' )

def VrfSelectionPolicyFieldSetMappingsCommand_noOrDefaultHandler( mode, args ):
   FieldSetMappingsCommand_noOrDefaultHandler( mode, args, mapCtx='vsp' )

# Class for TrafficPolicy, VrfSelectionPolicy FieldSetMappingsDisabledCmd
def FieldSetMappingsDisabledCmd_handler( mode, args, mapCtx='tp', enable=True ):
   assert mapCtx in [ 'tp', 'vsp' ]
   config = configForVrf( mode.vrfName )
   value = 'isTrue' if enable else 'isFalse'
   setattr( config, mapCtx + 'FieldSetMappingDisabled', value )

# class TpFieldSetMappingsDisabledCmd( CliCommandClass ):
def TpFieldSetMappingsDisabledCmd_handler( mode, args ):
   FieldSetMappingsDisabledCmd_handler( mode, args, mapCtx='tp', enable=True )

def TpFieldSetMappingsDisabledCmd_noOrDefaultHandler( mode, args ):
   FieldSetMappingsDisabledCmd_handler( mode, args, mapCtx='tp', enable=False )

# class VspFieldSetMappingsDisabledCmd( CliCommandClass ):
def VspFieldSetMappingsDisabledCmd_handler( mode, args ):
   FieldSetMappingsDisabledCmd_handler( mode, args, mapCtx='vsp', enable=True )

def VspFieldSetMappingsDisabledCmd_noOrDefaultHandler( mode, args ):
   FieldSetMappingsDisabledCmd_handler( mode, args, mapCtx='vsp', enable=False )

# class FieldSetCommand( CliCommandClass ):
def FieldSetCommand_handler( mode, args ):
   mapCtx = mode.mapCtx
   assert mapCtx in [ 'tp', 'vsp' ]
   ipv6 = 'ipv6' in args
   if ipv6:
      fsName = args[ 'FSNAME6' ]
      fsMode = TpFieldSetIpv6Mode if mapCtx == 'tp' else VspFieldSetIpv6Mode
      collAttrName = mapCtx + 'FieldSetMappingBindingsIpv6'
   else:
      fsName = args[ 'FSNAME' ]
      fsMode = TpFieldSetIpv4Mode if mapCtx == 'tp' else VspFieldSetIpv4Mode
      collAttrName = mapCtx + 'FieldSetMappingBindingsIpv4'
   childMode = mode.childMode( fsMode, fsName=fsName, mapCtx=mapCtx,
                               vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )
   config = configForVrf( mode.vrfName )
   coll = getattr( config, collAttrName )
   bindings = coll.get( fsName )
   if not bindings:
      coll[ fsName ] = Tac.Value( 'Routing::Bgp::FieldSetBindings', ipv6=ipv6 )

def FieldSetCommand_noOrDefaultHandler( mode, args ):
   mapCtx = mode.mapCtx
   assert mapCtx in [ 'tp', 'vsp' ]
   ipv6 = 'ipv6' in args
   if ipv6:
      fsName = args[ 'FSNAME6' ]
      collAttrName = mapCtx + 'FieldSetMappingBindingsIpv6'
   else:
      fsName = args[ 'FSNAME' ]
      collAttrName = mapCtx + 'FieldSetMappingBindingsIpv4'
   config = configForVrf( mode.vrfName )
   coll = getattr( config, collAttrName )
   del coll[ fsName ]

# class FieldSetCommunityCmd( CliCommandClass ):
def FieldSetCommunityCmd_handler( mode, args ):
   mapCtx = mode.mapCtx
   assert mapCtx in [ 'tp', 'vsp' ]
   config = configForVrf( mode.vrfName )
   ipv6, fsName = mode.getContext()
   attrName = mapCtx + 'FieldSetMappingBindingsIpv'
   attrName += '6' if ipv6 else '4'
   coll = getattr( config, attrName )

   if ( mapCtx == 'tp' and
        ( coll[ fsName ].peerType != PeerFieldSetFilter.inactive or
          coll[ fsName ].groupFilter ) ):
      mode.addError( PEERS_AND_COMMUNITY_ERROR_MSG )
      return

   updateFieldSetBindings( coll, fsName, commPresent=True,
                           community=getCommValue( args[ 'communityValue' ] ) )

def FieldSetCommunityCmd_noOrDefaultHandler( mode, args ):
   mapCtx = mode.mapCtx
   assert mapCtx in [ 'tp', 'vsp' ]
   config = configForVrf( mode.vrfName )
   ipv6, fsName = mode.getContext()
   attrName = mapCtx + 'FieldSetMappingBindingsIpv'
   attrName += '6' if ipv6 else '4'
   coll = getattr( config, attrName )
   updateFieldSetBindings( coll, fsName, commPresent=False )

# class FieldSetPeersCmd( CliCommandClass ):
def FieldSetPeersCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   ( ipv6, fsName ) = mode.getContext()
   coll = ( config.tpFieldSetMappingBindingsIpv6 if ipv6 else
            config.tpFieldSetMappingBindingsIpv4 )
   if coll[ fsName ].communityPresent:
      mode.addError( PEERS_AND_COMMUNITY_ERROR_MSG )
   elif mode.vrfName != 'default':
      mode.addError( PEERS_ND_VRF_ERROR_MSG )
   else:
      if 'ibgp' in args:
         peerType = PeerFieldSetFilter.ibgp
         groups = frozenset()
      elif 'ebgp' in args:
         peerType = PeerFieldSetFilter.ebgp
         groups = frozenset()
      elif 'group' in args:
         peerType = PeerFieldSetFilter.inactive
         groups = args[ 'GROUPS' ]
      else:
         assert False, "no peer type"
      # Do not set commPresent=False because the error checking above prevents us
      # from getting here if a community is present
      updateFieldSetBindings( coll, fsName, peerType=peerType, groups=groups )

def FieldSetPeersCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   ( ipv6, fsName ) = mode.getContext()
   coll = ( config.tpFieldSetMappingBindingsIpv6 if ipv6 else
            config.tpFieldSetMappingBindingsIpv4 )
   updateFieldSetBindings( coll, fsName, peerType=PeerFieldSetFilter.inactive,
                           groups={} )

# class RouterBgpRfdPolicyCmd( BgpCmdBaseClass ):
def RouterBgpRfdPolicyCmd_handleNormal( mode, args ):
   emitWarningIfRfdNotSupportedInRibdMode( mode )
   rfdPolicyName = args[ 'RFD_POLICY_NAME' ]
   childMode = mode.childMode( RouterBgpRfdPolicyMode, rfdPolicyName=rfdPolicyName )
   mode.session.gotoChildMode( childMode )

def RouterBgpRfdPolicyCmd_handleNoOrDefault( mode, args, noOrDefault ):
   rfdPolicyName = args[ 'RFD_POLICY_NAME' ]
   del Globals.bgpConfig.rfdPolicy[ rfdPolicyName ]

def RouterBgpRfdPolicyCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpRfdPolicyCmd", mode, args )

def RouterBgpRfdPolicyCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpRfdPolicyCmd", mode, args )

# class RfdPenaltyThresholdConfigCmd( BgpCmdBaseClass ):
def RfdPenaltyThresholdConfigCmd_handleNormal( mode, args ):
   if args[ "REUSE" ] >= args[ "SUPPRESSION" ]:
      mode.addError( "Suppression threshold must be greater than reuse threshold" )
      return
   if args[ "SUPPRESSION" ] >= args[ "CEILING" ]:
      mode.addError( "Maximum threshold must be greater than suppression threshold" )
      return
   pendingConfig = mode.pendingConfig
   pendingConfig.reuseThreshold = args[ "REUSE" ]
   pendingConfig.suppressThreshold = args[ "SUPPRESSION" ]
   pendingConfig.ceilingThreshold = args[ "CEILING" ]

def RfdPenaltyThresholdConfigCmd_handleNoOrDefault( mode, args, noOrDefault ):
   pendingConfig = mode.pendingConfig
   rfdDefaultParams = mode.rfdDefaultParams
   pendingConfig.reuseThreshold = rfdDefaultParams.reuseThresholdDefault
   pendingConfig.suppressThreshold = rfdDefaultParams.suppressThresholdDefault
   pendingConfig.ceilingThreshold = rfdDefaultParams.ceilingThresholdDefault

def RfdPenaltyThresholdConfigCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RfdPenaltyThresholdConfigCmd", mode, args )

def RfdPenaltyThresholdConfigCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RfdPenaltyThresholdConfigCmd", mode, args )

# class RfdPenaltyChangeConfigCmd( BgpCmdBaseClass ):
def RfdPenaltyChangeConfigCmd_handleNormal( mode, args ):
   pendingConfig = mode.pendingConfig
   if "PENALTY" in args:
      pendingConfig.penaltyAttrUpdate = args[ "PENALTY" ]
   else:
      pendingConfig.penaltyAttrUpdate = 0

def RfdPenaltyChangeConfigCmd_handleNoOrDefault( mode, args, noOrDefault ):
   pendingConfig = mode.pendingConfig
   rfdDefaultParams = mode.rfdDefaultParams
   pendingConfig.penaltyAttrUpdate = rfdDefaultParams.penaltyAttrUpdateDefault

def RfdPenaltyChangeConfigCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RfdPenaltyChangeConfigCmd", mode, args )

def RfdPenaltyChangeConfigCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RfdPenaltyChangeConfigCmd", mode, args )

# class RfdPenaltyWithdrawalConfigCmd( BgpCmdBaseClass ):
def RfdPenaltyWithdrawalConfigCmd_handleNormal( mode, args ):
   pendingConfig = mode.pendingConfig
   pendingConfig.penaltyUnreach = args[ "PENALTY" ]

def RfdPenaltyWithdrawalConfigCmd_handleNoOrDefault( mode, args, noOrDefault ):
   pendingConfig = mode.pendingConfig
   rfdDefaultParams = mode.rfdDefaultParams
   pendingConfig.penaltyUnreach = rfdDefaultParams.penaltyUnreachDefault

def RfdPenaltyWithdrawalConfigCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RfdPenaltyWithdrawalConfigCmd", mode, args )

def RfdPenaltyWithdrawalConfigCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RfdPenaltyWithdrawalConfigCmd", mode, args )

# class RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd( BgpCmdBaseClass ):
def RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd_handleNormal( mode, args ):
   pendingConfig = mode.pendingConfig
   if "PENALTY" in args:
      pendingConfig.penaltyReach = args[ "PENALTY" ]
   else:
      pendingConfig.penaltyReach = 0

def RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   pendingConfig = mode.pendingConfig
   rfdDefaultParams = mode.rfdDefaultParams
   pendingConfig.penaltyReach = rfdDefaultParams.penaltyReachDefault

def RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd_handler( mode, args ):
   BgpCmdBaseClass_handler(
      "RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd", mode, args )

def RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd_noOrDefaultHandler(
      mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd", mode, args )

# class RfdPenaltyDecayHalfLifeConfigCmd( BgpCmdBaseClass ):
def RfdPenaltyDecayHalfLifeConfigCmd_handleNormal( mode, args ):
   pendingConfig = mode.pendingConfig
   if "minutes" in args:
      pendingConfig.halfLife = args[ "HALF_LIFE" ] * 60
      pendingConfig.halfLifeUnitIsMinute = True
   elif "seconds" in args:
      pendingConfig.halfLife = args[ "HALF_LIFE" ]
      pendingConfig.halfLifeUnitIsMinute = False

def RfdPenaltyDecayHalfLifeConfigCmd_handleNoOrDefault( mode, args, noOrDefault ):
   pendingConfig = mode.pendingConfig
   rfdDefaultParams = mode.rfdDefaultParams
   pendingConfig.halfLife = rfdDefaultParams.halfLifeDefault
   pendingConfig.halfLifeUnitIsMinute = False

def RfdPenaltyDecayHalfLifeConfigCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RfdPenaltyDecayHalfLifeConfigCmd", mode, args )

def RfdPenaltyDecayHalfLifeConfigCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "RfdPenaltyDecayHalfLifeConfigCmd", mode, args )

def RfdSuppressionIgnoreConfigCmd_handleNormal( mode, args ):
   pendingConfig = mode.pendingConfig
   pendingConfig.suppressionIgnoredPrefixListV4Uni = \
      args.get( 'IPV4_PREFIX_LIST',
                RfdPolicyType.suppressionIgnoredPrefixListV4UniDefault )
   pendingConfig.suppressionIgnoredPrefixListV6Uni = \
      args.get( 'IPV6_PREFIX_LIST',
                RfdPolicyType.suppressionIgnoredPrefixListV6UniDefault )

def RfdSuppressionIgnoreConfigCmd_handleNoOrDefault( mode, args, noOrDefault ):
   pendingConfig = mode.pendingConfig
   pendingConfig.suppressionIgnoredPrefixListV4Uni = \
      RfdPolicyType.suppressionIgnoredPrefixListV4UniDefault
   pendingConfig.suppressionIgnoredPrefixListV6Uni = \
      RfdPolicyType.suppressionIgnoredPrefixListV6UniDefault

def handlerRfdSuppressionIgnoreConfigCmd( mode, args ):
   BgpCmdBaseClass_handler( "RfdSuppressionIgnoreConfigCmd", mode, args )

def noOrDefaultHandlerRfdSuppressionIgnoreConfigCmd( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RfdSuppressionIgnoreConfigCmd", mode, args )

# class RouterBgpRouteDistinguisherCmd
def RouterBgpRouteDistinguisherCmd_handleNormal( mode, args ):
   childMode = mode.childMode( RouterBgpRouteDistinguisherMode )
   mode.session.gotoChildMode( childMode )

def RouterBgpRouteDistinguisherCmd_handleNoOrDefault( mode, args, noOrDefault ):
   Globals.bgpConfig.rdAutoUnifiedConfig =\
         Globals.bgpConfig.rdAutoUnifiedConfigDefault

def RouterBgpRouteDistinguisherCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "RouterBgpRouteDistinguisherCmd", mode, args )

def RouterBgpRouteDistinguisherCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "RouterBgpRouteDistinguisherCmd", mode, args )

# class RouteDistinguisherAssignmentAutoRangeConfigCmd
def RouteDistinguisherAssignmentAutoRangeConfigCmd_handleNormal( mode, args ):
   rdAutoUnifiedConfig = Tac.nonConst( Globals.bgpConfig.rdAutoUnifiedConfig )
   rdAutoUnifiedConfig.assignedNumberStart = args[ 'MIN' ]
   rdAutoUnifiedConfig.assignedNumberEnd = args[ 'MAX' ]
   Globals.bgpConfig.rdAutoUnifiedConfig = rdAutoUnifiedConfig

def RouteDistinguisherAssignmentAutoRangeConfigCmd_handleNoOrDefault( mode, args,
      noOrDefault ):
   defaultRdAutoUnifiedConfig = Globals.bgpConfig.rdAutoUnifiedConfigDefault
   rdAutoUnifiedConfig = Tac.nonConst( Globals.bgpConfig.rdAutoUnifiedConfig )
   rdAutoUnifiedConfig.assignedNumberStart = \
         defaultRdAutoUnifiedConfig.assignedNumberStart
   rdAutoUnifiedConfig.assignedNumberEnd = \
         defaultRdAutoUnifiedConfig.assignedNumberEnd
   Globals.bgpConfig.rdAutoUnifiedConfig = rdAutoUnifiedConfig

def handlerRouteDistinguisherAssignmentAutoRangeConfigCmd( mode, args ):
   BgpCmdBaseClass_handler( "RouteDistinguisherAssignmentAutoRangeConfigCmd", mode,
         args )

def noOrDefaultHandlerRouteDistinguisherAssignmentAutoRangeConfigCmd( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
         "RouteDistinguisherAssignmentAutoRangeConfigCmd", mode, args )

# class RouteDistinguisherAssignmentAutoAfiConfigCmd
def RouteDistinguisherAssignmentAutoAfiConfigCmd_handleNormal( mode, args ):
   rdAutoUnifiedConfig = Tac.nonConst( Globals.bgpConfig.rdAutoUnifiedConfig )
   rdAutoAfi = Tac.nonConst( rdAutoUnifiedConfig.rdAutoAfi )
   if 'l2-evpn' in args:
      rdAutoAfi.l2evpn = True
   rdAutoUnifiedConfig.rdAutoAfi = rdAutoAfi
   Globals.bgpConfig.rdAutoUnifiedConfig = rdAutoUnifiedConfig

def RouteDistinguisherAssignmentAutoAfiConfigCmd_handleNoOrDefault( mode, args,
      noOrDefault ):
   rdAutoUnifiedConfig = Tac.nonConst( Globals.bgpConfig.rdAutoUnifiedConfig )
   rdAutoAfi = Tac.nonConst( rdAutoUnifiedConfig.rdAutoAfi )
   if 'l2-evpn' in args:
      rdAutoAfi.l2evpn = False
   rdAutoUnifiedConfig.rdAutoAfi = rdAutoAfi
   Globals.bgpConfig.rdAutoUnifiedConfig = rdAutoUnifiedConfig

def handlerRouteDistinguisherAssignmentAutoAfiConfigCmd( mode, args ):
   BgpCmdBaseClass_handler( "RouteDistinguisherAssignmentAutoAfiConfigCmd", mode,
         args )

def noOrDefaultHandlerRouteDistinguisherAssignmentAutoAfiConfigCmd( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
         "RouteDistinguisherAssignmentAutoAfiConfigCmd", mode, args )

# class LfibEntryInstallationSkippedCmd( BgpCmdBaseClass ):
def LfibEntryInstallationSkippedCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'ipv4 labeled-unicast':
      config.lfibSkipIPv4LabeledUni = True
   elif mode.addrFamily == 'ipv6 labeled-unicast':
      config.lfibSkipIPv6LabeledUni = True
   else:
      raise ValueError()

def LfibEntryInstallationSkippedCmd_handleNoOrDefault( mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'ipv4 labeled-unicast':
      config.lfibSkipIPv4LabeledUni = False
   elif mode.addrFamily == 'ipv6 labeled-unicast':
      config.lfibSkipIPv6LabeledUni = False
   else:
      raise ValueError()

def LfibEntryInstallationSkippedCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "LfibEntryInstallationSkippedCmd", mode, args )

def LfibEntryInstallationSkippedCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "LfibEntryInstallationSkippedCmd", mode, args )

# class LabelLocalTerminationExplicitNullCmd( BgpCmdBaseClass ):
def LabelLocalTerminationExplicitNullCmd_handleNormal( mode, args ):
   config = configForVrf( mode.vrfName )
   v4 = 'ipv4' in mode.addrFamily
   if 'explicit-null' in args:
      nullLabel = LuLabelLocalTerminationModeEnum.explicitNull
   else:
      nullLabel = LuLabelLocalTerminationModeEnum.implicitNull
   if v4:
      config.labelLocalTerminationModeAfV4LabeledUni = nullLabel
   else:
      config.labelLocalTerminationModeAfV6LabeledUni = nullLabel

def LabelLocalTerminationExplicitNullCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'ipv6 labeled-unicast':
      config.labelLocalTerminationModeAfV6LabeledUni = \
            LuLabelLocalTerminationModeEnum.implicitNull
   elif mode.addrFamily == 'ipv4 labeled-unicast':
      config.labelLocalTerminationModeAfV4LabeledUni = \
            LuLabelLocalTerminationModeEnum.implicitNull

def LabelLocalTerminationExplicitNullCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "LabelLocalTerminationExplicitNullCmd", mode, args )

def LabelLocalTerminationExplicitNullCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler(
      "LabelLocalTerminationExplicitNullCmd", mode, args )

# class SetEpePeerSetCmd( BgpCmdBaseClass ):
def SetEpePeerSetCmd_handleEpePeerSet(
      mode, peerSetName, communityValue, noOrDefault=False ):
   vrfName = mode.vrfName
   config = configForVrf( vrfName )
   addrFamily = mode.addrFamily
   if mode.addrFamily != 'ipv4' and mode.addrFamily != 'ipv6':
      raise ValueError()
   attrName = bgpConfigAttrsAfMap[ 'epePeerSetConfig' ].get( addrFamily )
   peerSetColl = getattr( config, attrName )

   # NoOrDefault.DEFAULT:
   #   'default peer-set <set name> [community <number>]' being executed
   #   'no peer-set <set name> [community <number>]' being executed
   # NoOrDefault.NO:
   #   even though the peer-set command not support "disabled" option, we
   #   still handle it as the same as the deletion case
   if noOrDefault == NoOrDefault.DEFAULT or noOrDefault == NoOrDefault.NO:
      del peerSetColl[ peerSetName ]
      return
   peerSet = peerSetColl.get( peerSetName )
   if peerSet is None:
      peerSet = peerSetColl.newMember( peerSetName )

   peerSet.communityPresent = communityValue is not None
   peerSet.community = Tac.Value( 'Acl::CommunityValue', 0, 0, 0 ) \
      if communityValue is None \
      else getCommValue( communityValue )

def SetEpePeerSetCmd_handleNormal( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      msg = "EPE peer set feature is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )
   SetEpePeerSetCmd_handleEpePeerSet( mode,
      args[ 'PEER_SET' ], args.get( 'communityValue', None ) )

def SetEpePeerSetCmd_handleNoOrDefault( mode, args, noOrDefault ):
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      msg = "EPE peer set feature is only supported in multi-agent mode"
      if mode.session.isInteractive():
         mode.addError( msg )
         return
      mode.addWarning( msg )
   SetEpePeerSetCmd_handleEpePeerSet( mode,
      args[ 'PEER_SET' ], args.get( 'communityValue', None ), noOrDefault )

def SetEpePeerSetCmd_handler( mode, args ):
   BgpCmdBaseClass_handler( "SetEpePeerSetCmd", mode, args )

def SetEpePeerSetCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass_noOrDefaultHandler( "SetEpePeerSetCmd", mode, args )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global rdAutoInputDir
   global rdConfigInputDir
   global asnConfig
   global aclStatus
   global aclCheckpoint
   global ucmpVrfConfigDir
   global bgpVrfConfigDir
   global lsImportConfig
   global lsRoleConfig

   rdAutoInputDir = LazyMount.mount(
      entityManager, 'ip/vrf/routeDistinguisherInputDir/auto', 'Tac::Dir', 'ri' )
   rdConfigInputDir = ConfigMount.mount(
      entityManager, 'ip/vrf/routeDistinguisherInputDir/config', 'Tac::Dir', 'w' )
   asnConfig = ConfigMount.mount(
      entityManager, 'routing/bgp/asn/config', 'Routing::AsnConfig', 'w' )
   aclStatus = LazyMount.mount(
      entityManager, 'acl/status/all', 'Acl::Status', 'r' )
   aclCheckpoint = LazyMount.mount(
      entityManager, 'acl/checkpoint', 'Acl::CheckpointStatus', 'w' )
   bgpVrfConfigDir = ConfigMount.mount(
      entityManager, 'routing/bgp/vrf/config', 'Routing::Bgp::VrfConfigDir', 'w' )
   ucmpVrfConfigDir = ConfigMount.mount(
      entityManager, 'routing/ucmp/bgp/vrf/config',
      'Routing::Ucmp::VrfUcmpConfigDir', 'w' )

   lsImportConfig = ConfigMount.mount(
      entityManager, 'routing/bgp/lsImportConfig',
      'Routing::Bgp::LinkStateImportConfig', 'w' )

   def deleteBgpLsImportCmd( vrfName ):
      if vrfName == DEFAULT_VRF and lsImportConfig:
         lsImportConfig.protoConfig.clear()
   deleteRouterBgpVrfHook.addExtension( deleteBgpLsImportCmd )

   lsRoleConfig = ConfigMount.mount(
      entityManager, 'routing/bgp/lsRoleConfig',
      'Routing::Bgp::LinkStateRoleConfig', 'w' )

   def deleteBgpLsRoleCmd( vrfName ):
      if vrfName == DEFAULT_VRF and lsRoleConfig:
         lsRoleConfig.protoConfig.clear()
   deleteRouterBgpVrfHook.addExtension( deleteBgpLsRoleCmd )
