#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
from IpLibConsts import DEFAULT_VRF
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
   AmiResponseException,
   showRibCapiCommand,
   EmptyResponseException,
   cliRibdShowCommand,
   ribdBgpShowCmdDict
)
from CliPlugin.VrfCli import (
   VrfExecCmdDec,
   generateVrfCliModel,
)
from CliPlugin import BgpCliModels
from CliPlugin import VrfCli
from BgpLib import (
   createKey,
   doLogClearIpBgp
)
import Tac
import os
import six
import sys

bgpConfig = None
bgpClearReq = None
bgpClearResp = None
bgpTestControlReq = None
bgpVrfClearReqDir = None
bgpVrfClearRespDir = None

updateErrorsVrfModel = generateVrfCliModel(
   BgpCliModels.BgpNeighborUpdateError,
   "Per VRF BGP Neighbor Update-Error list" )

def showRibDamiCommand():
   return ribdBgpShowCmdDict.get( 'showRibDamiCommand' )

def _doShowIpBgpUpdateErrors( mode, ipv6, vrfName, addr=None ):
   args = {}
   cmd = 'MIO_DGET_BGP_UPDATE_ERROR'
   args[ 'ipv6' ] = ipv6
   if addr is not None:
      if ipv6 == 1:
         args[ 'addrv6' ] = addr.stringValue
      else:
         args[ 'addrv4' ] = addr
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   try:
      return showRibCapiCommand()( mode, BgpCliModels.BgpNeighborUpdateError, cmd,
                                   args, clientName='BGP', skipNullResponses=False )
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowIpBgpUpdateErrors' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames,
                cliModel=updateErrorsVrfModel )
def doShowIpBgpUpdateErrors( mode, addr=None, vrfName=None ):
   return _doShowIpBgpUpdateErrors( mode, 0, vrfName, addr )

@ArBgpShowOutput( 'doShowIpBgpUpdateErrors' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames,
                cliModel=updateErrorsVrfModel )
def doShowIp6BgpUpdateErrors( mode, addr=None, vrfName=None ):
   return _doShowIpBgpUpdateErrors( mode, 1, vrfName, addr )

def handlerShowIpBgpNeighborsUpdateErrorsCmd( mode, args ):
   return doShowIpBgpUpdateErrors( mode, addr=args.get( 'ADDRESS' ),
                                   vrfName=args.get( 'VRF' ) )

def handlerShowIp6BgpNeighborsUpdateErrorsCmd( mode, args ):
   return doShowIp6BgpUpdateErrors( mode, addr=args.get( 'ADDRESS' ),
                                    vrfName=args.get( 'VRF' ) )

# pylint: disable=consider-using-f-string
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames )
@ArBgpShowOutput( 'doClearIpBgp' )
def doClearIpBgp( mode, transportAddrFamily=None, peer=None, soft=False,
                  direction=None, vrfName=None, routerId=None,
                  errorsOrCounters=None, resetMsg='' ):

   if ( not peer and bgpConfig.clearAllDisabled and not errorsOrCounters ):
      mode.addError( "Cannot clear all peering sessions because "
                     "clear all commands are disabled" )
      return
   #
   # Log what we are attempting to do
   #
   doLogClearIpBgp( mode, transportAddrFamily, peer, soft, direction,
                    vrfName,
                    errorsOrCounters == 'errors',
                    errorsOrCounters == 'counters',
                    resetMsg )

   #
   # transportBasedClear is True in "clear ip(v6) bgp neighbor *" command.
   #       In this case only the Address family specific peers are hard reset.
   # transportBasedClear is False in "clear ip(v6) bgp *" command.
   #        In this case all peers are hard reset.
   # we derive the transportBasedClear from transportAddrFamily which is set only
   # in 'clear ip(v6) bgp neighbor *' command
   #
   transportBasedClear = transportAddrFamily is not None

   # if peer is specified, reset that peer.
   # if peer is not specified reset all peers if its 'clear ip(v6) bgp *'.
   # if peer is not specified reset only transportAddrFamily specific peers if
   # the command is 'clear ip(v6) bgp neighbor *'.
   if peer:
      peerKey = peer
   elif transportAddrFamily is None or transportAddrFamily == 'ip':
      peerKey = createKey( '0.0.0.0' )
   elif transportAddrFamily == 'ipv6':
      peerKey = createKey( '::' )

   k = "%d,%d" % ( os.getpid(), Tac.now() )
   if direction == 'in':
      d = 'peerClearSoftIn'
   elif direction == 'out':
      d = 'peerClearSoftOut'
   else:
      d = 'peerClearSoftBoth'
   req = Tac.Value( 'Routing::Bgp::PeerClearRequest', peerKey,
                     bool( peer is None ),
                     bool( soft or direction ), d,
                     transportBasedClear,
                     bool( errorsOrCounters == 'errors' ),
                     bool( errorsOrCounters == 'counters' ),
                     resetMsg )
   bcReq = None
   bcResp = None
   if vrfName != DEFAULT_VRF:
      bcReq = bgpVrfClearReqDir.clearRequestDir.newMember( vrfName )
      bcResp = bgpVrfClearRespDir.newEntity(
         'Routing::Bgp::ClearResponseDir', vrfName )
      description = 'BGP clear request to complete for VRF %s' % vrfName
      warningPost = ' for VRF %s' % vrfName
   else:
      bcReq = bgpClearReq
      bcResp = bgpClearResp
      description = 'BGP clear request to complete'
      warningPost = ''
   if ( bcReq is None ) or ( bcResp is None ):
      return
   bcReq.bgpPeerClearRequest[ k ] = req
   try:
      # If we are running as part of a BGP Cohab CLI clear test, we will not get a
      # response as the agent won't be running.  Timeout immediately to speed
      # up the testing.
      if 'BGP_COHAB_CLI_CLEAR_TEST' in os.environ:
         timeout = 0
      else:
         timeout = 5.0

      Tac.waitFor( lambda: k in bcResp.bgpPeerClearResponse,
                   description=description,
                   warnAfter=None, sleep=True, maxDelay=0.1, timeout=timeout )
   except Tac.Timeout:
      sessions = 'sessions%s' % warningPost \
            if peer is None else 'session%s' % warningPost
      mode.addWarning( "BGP %s may not have reset yet" % sessions )

   del bcReq.bgpPeerClearRequest[ k ]

def handlerClearIpBgpCmd( mode, args ):
   soft = "soft" in args
   vrfName = args.get( "VRF" )
   direction = args.get( 'DIRECTION' )
   errorsOrCounters = args.get( 'STATISTICS' )
   resetMsg = args.get( "MESSAGE", '' )
   routerId = args.get( 'ROUTERID' )

   doClearIpBgp( mode, peer=args[ 'PEER' ], soft=soft,
                 direction=direction, vrfName=vrfName, routerId=routerId,
                 errorsOrCounters=errorsOrCounters, resetMsg=resetMsg )
# pylint: enable=consider-using-f-string

def handlerClearIpBgpTransportCmd( mode, args ):
   vrfName = args.get( "VRF" )
   transportAddrFamily = None
   if "ip" in args:
      transportAddrFamily = "ip"
   elif "ipv6" in args:
      transportAddrFamily = "ipv6"
   resetMsg = args.get( "MESSAGE", "" )

   doClearIpBgp( mode, vrfName=vrfName,
                 transportAddrFamily=transportAddrFamily,
                 resetMsg=resetMsg )

# -------------------------------------------------------------------------------
# "show bgp instance
#     (hidden options) [internal [dump-duplicates]]
#     (hidden options) [internal brib count [detail]]
#     (hidden options) [internal events tracking start [advertise-threshold
#                       <rib-add-threshold> <ribout-advertise-threshold>]]
#     (hidden options) [internal events tracking stop]
#     (hidden options) [internal events tracking summary]
#     [vrf <vrfName>]"
# -------------------------------------------------------------------------------

# Handle internal events tracking commands
# The 'start' and 'show' commands are handled inline by directly updating relevant
# attributes in Sysdb.
# The 'summary' command is handled by cliribd
def doShowBgpEventsTracking( mode, internal, vrfName ):
   if internal.get( 'start' ):
      if internal.get( 'advertise-threshold' ):
         numRibRoutes = internal.get( 'numRibRoutes' )
         numRibOutRoutes = internal.get( 'numRibOutRoutes' )
         bgpTestControlReq.ribAddThreshold = int( numRibRoutes )
         bgpTestControlReq.ribOutPathAdvertiseThreshold = int( numRibOutRoutes )
      print( 'Starting BGP Events Tracking' )
      bgpTestControlReq.eventsTrackingEnabled = True
      return

   if internal.get( 'stop' ):
      print( 'Stopping BGP Events Tracking' )
      bgpTestControlReq.eventsTrackingEnabled = False
      bgpTestControlReq.ribAddThreshold = 0
      bgpTestControlReq.ribRemoveThreshold = 0
      bgpTestControlReq.ribOutPathAdvertiseThreshold = 0
      bgpTestControlReq.ribOutPathWithdrawThreshold = 0
      return

   if internal.get( 'summary' ) or internal.get( 'detail' ):
      cmd = 'show bgp events-tracking-summary'
      # gated 'summary' command doesn't summarize events tracking data for all
      # the vrfs. Instead prints data for individual vrf.
      if not vrfName:
         vrfName = 'all'
      cliRibdShowCommand()( mode, cmd, clientName="BGP", vrf=vrfName )

cliShowBgpInstanceVrfModel = generateVrfCliModel( BgpCliModels.ShowBgpInstanceModel,
   "per VRF bgp instance information models.", revision=4 )
# vrfName is defaulted to None so as to allow cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.

@ArBgpShowOutput( 'doShowBgpInstance' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=cliShowBgpInstanceVrfModel )
def doShowBgpInstance( mode, internal=None, vrfName=None ):
   if internal:
      if internal.get( 'events' ) and internal.get( 'tracking' ):
         doShowBgpEventsTracking( mode, internal, vrfName )
      return

   cmd = 'show bgp instance'
   if mode.session_.outputFormat_ == 'text':
      cliRibdShowCommand()( mode, cmd, clientName="BGP", vrf=vrfName )
   else:
      mode.addError( 'This is an unconverted command.' )

# -------------------------------------------------------------------------------
# "show bgp update-group
#     (index | A.B.C.D | A:B:C:D:E:F:G:H | LL_ADDR)
#     [vrf <vrfName>]"
# -------------------------------------------------------------------------------
updateGroupVrfModel = generateVrfCliModel( BgpCliModels.BgpUpdateGroups,
                                           "Per VRF BGP update group",
                                           uncheckedModel=True )

@ArBgpShowOutput( 'doShowBgpUpdateGroup' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=updateGroupVrfModel )
def doShowBgpUpdateGroup( mode, opt=None, vrfName=None ):
   cmd = 'MIO_DGET_BGP_UPDATE_GROUP'
   args = {}

   if opt is not None:
      if isinstance( opt, six.integer_types ):
         args[ 'ug-index' ] = opt
      elif ':' in opt.gatedRepr():
         args[ 'addrv6' ] = opt.gatedRepr()
      else:
         args[ 'addrv4' ] = opt.gatedRepr()
   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if mode.session_.outputFormat_ == "text":
      args[ 'fastpath' ] = 1
      return showRibDamiCommand()( sys.stdout.fileno(), cmd, args, clientName='BGP' )
   else:
      try:
         args[ 'fastpath' ] = 0
         return showRibCapiCommand()( mode, BgpCliModels.BgpUpdateGroups,
                                      cmd, args, clientName='BGP',
                                      skipNullResponses=False )
      except ( EmptyResponseException(), AmiResponseException() ):
         return None

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global bgpClearReq, bgpClearResp
   global bgpVrfClearRespDir, bgpVrfClearReqDir
   global bgpTestControlReq
   bgpConfig = LazyMount.mount( entityManager, 'routing/bgp/config',
                                'Routing::Bgp::Config', 'w' )
   bgpClearReq = LazyMount.mount( entityManager, 'routing/bgp/clear/request',
                                  'Routing::Bgp::ClearRequestDir', 'w' )
   bgpClearResp = LazyMount.mount( entityManager, 'routing/bgp/clear/response',
                                   'Routing::Bgp::ClearResponseDir', 'r' )
   bgpVrfClearReqDir = LazyMount.mount( entityManager,
                                        'routing/bgp/clear/vrf/request',
                                        'Routing::Bgp::VrfClearRequestDir', 'w' )
   bgpVrfClearRespDir = LazyMount.mount( entityManager,
                                         'routing/bgp/clear/vrf/response',
                                         'Tac::Dir', 'r' )
   bgpTestControlReq = LazyMount.mount( entityManager,
                                        'routing/bgp/testControlRequest',
                                        'Routing::Bgp::TestControlRequestDir', 'w' )
