#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import Ark
from Arnet import Prefix, Ip6Prefix
import ast
import Cell
import CliGlobal
from CliPlugin.RoutingBgpCli import (
      bgpNeighborConfig,
      configForVrf,
)
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput
from CliPlugin.RpkiCli import (
      getCacheNames,
      gv,
      RpkiCacheConfigMode,
      RpkiOriginValidationMode,
      RpkiTransportTcpConfigMode,
      RpkiTransportTlsConfigMode,
)
from CliDynamicSymbol import CliDynamicPlugin
from CliSavePlugin.RpkiCliSave import (
      isDefaultPortInUseBasedOnTransportType,
)
from EosRpkiLib import (
      getCacheMessage,
      getCacheUnusedMessage,
      getPartialTableRemovalNotSupportedMessage,
      getRoaTableMessage,
)
from IpLibConsts import DEFAULT_VRF
from io import StringIO
import LazyMount
import sys
import SmashLazyMount
import Tac
import Tracing
from TypeFuture import TacLazyType
import Toggles.RpkiToggleLib

# pkgdeps: rpm RpkiLib-lib

traceHandle = Tracing.Handle( 'RpkiCli' )
t0 = traceHandle.trace0

# Using CliGlobal to ensure that the variables are set only once, thus avoiding race
# conditions
handlerGv = CliGlobal.CliGlobal( rpkiStatistics=None,
                                 rpkiRoaStore=None,
                                 rpkiCacheCounterSmash=None,
                                 rpkiCacheCounterSnapshotSmash=None,
                                 cacheCounterDir=None,
                                 asnConfig=None )

RpkiCliModels = CliDynamicPlugin( "RpkiCliModels" )

def doEnterRpkiCacheConfigMode( mode, args ):
   cacheName = args[ 'NAME' ]
   rpkiDefaults = Tac.Type( 'Rpki::RpkiDefaults' )
   invalidCacheNames = Tac.Type( 'Rpki::RpkiInvalidCacheName' )
   if len( cacheName ) > rpkiDefaults.cacheNameMaxSize:
      mode.addError( "Cache name cannot exceed " +
                     f"{rpkiDefaults.cacheNameMaxSize} characters" )
      return
   if cacheName in invalidCacheNames.attributes:
      mode.addError( f"{cacheName} is a keyword and cannot be used for a cache " +
                     "name" )
      return
   childMode = mode.childMode( RpkiCacheConfigMode, cacheName=cacheName )
   mode.session_.gotoChildMode( childMode )

def noEnterRpkiCacheConfigMode( mode, args ):
   cacheName = args[ 'NAME' ]
   del gv.rpkiConfig.cacheConfig[ cacheName ]

def doRpkiHostVrfPortCmd( mode, args ):
   if mode.rpkiCacheConfig.transportConfig.transportAuthenticationType == 'tls':
      defaultPort = mode.rpkiCacheConfig.tlsPortDefault
   else:
      # TCP and to be supported secure transports use tcpPortDefault as default
      # port except SSH
      defaultPort = mode.rpkiCacheConfig.tcpPortDefault
   port = args.get( 'PORT', defaultPort )
   mode.rpkiCacheConfig.port = port
   vrf = args.get( 'VRF', DEFAULT_VRF )
   mode.rpkiCacheConfig.vrf = vrf
   ipAddrOrHost = args[ 'IP_OR_HOST' ]
   mode.rpkiCacheConfig.host = ipAddrOrHost

def noRpkiHostVrfPortCmd( mode, args ):
   # Set the default value
   if mode.rpkiCacheConfig.transportConfig.transportAuthenticationType == 'tls':
      mode.rpkiCacheConfig.port = mode.rpkiCacheConfig.tlsPortDefault
   else:
      # TCP and to be supported secure transports use tcpPortDefault as default
      # port except SSH
      mode.rpkiCacheConfig.port = mode.rpkiCacheConfig.tcpPortDefault
   mode.rpkiCacheConfig.vrf = DEFAULT_VRF
   mode.rpkiCacheConfig.host = ''

def doRpkiRefreshIntervalCmd( mode, args ):
   seconds = args.get( 'SECONDS', 0 )
   mode.rpkiCacheConfig.refreshInterval = seconds

def doRpkiRetryIntervalCmd( mode, args ):
   seconds = args.get( 'SECONDS', 0 )
   mode.rpkiCacheConfig.retryInterval = seconds

def doRpkiExpireIntervalCmd( mode, args ):
   seconds = args.get( 'SECONDS', 0 )
   mode.rpkiCacheConfig.expireInterval = seconds

def doRpkiPreferenceCmd( mode, args ):
   preference = args[ 'VALUE' ]
   mode.rpkiCacheConfig.preference = preference

def noRpkiPreferenceCmd( mode, args ):
   # Set the default value
   mode.rpkiCacheConfig.preference = mode.rpkiCacheConfig.preferenceDefault

def doRpkiLocalInterfaceCmd( mode, args ):
   intf = args[ 'INTF' ]
   mode.rpkiCacheConfig.sourceIntf = intf.name

def noRpkiLocalInterfaceCmd( mode, args ):
   mode.rpkiCacheConfig.sourceIntf = mode.rpkiCacheConfig.sourceIntfDefault

def doRpkiRoaTableCmd( mode, args ):
   roaTableNames = args[ 'NAME' ]
   roaTableConfig = Tac.newInstance( 'Rpki::CacheConfig::RoaTableConfig' )
   for roaTableName in roaTableNames:
      roaTableConfig.roaTableNames.add( roaTableName )
   mode.rpkiCacheConfig.roaTableConfig = roaTableConfig

def noRpkiRoaTableCmd( mode, args ):
   # Either no argument should be specified or all the tables names assigned to
   # the cache server must be present in the argument for the command to work
   roaTableNames = args.get( 'NAME' )
   if roaTableNames is not None:
      partialTableRemove = not (
              set( mode.rpkiCacheConfig.roaTableConfig.roaTableNames ) ==
              set( roaTableNames ) )
      if partialTableRemove:
         mode.addError( getPartialTableRemovalNotSupportedMessage() )
         return
   rpkiDefaults = Tac.Type( 'Rpki::RpkiDefaults' )
   roaTableConfig = Tac.newInstance( 'Rpki::CacheConfig::RoaTableConfig' )
   roaTableConfig.roaTableNames.add( rpkiDefaults.defaultRoaTableName )
   mode.rpkiCacheConfig.roaTableConfig = roaTableConfig

def updateTransportConfig( currentTransportConfig,
                           transportAuthenticationType=None,
                           tcpKeepaliveOptions=None,
                           sslProfileName=None ):
   '''
   Helper method that will set only the non none attribute passed as parameter and
   retain the values of other parameters in the transportConfig.
   '''
   if transportAuthenticationType is None:
      transportAuthenticationType = \
         currentTransportConfig.transportAuthenticationType
   if tcpKeepaliveOptions is None:
      tcpKeepaliveOptions = currentTransportConfig.tcpKeepaliveOptions
   if sslProfileName is None:
      sslProfileName = currentTransportConfig.sslProfileName
   transportConfig = Tac.Value( 'Rpki::RpkiTransportConfig',
                                transportAuthenticationType, tcpKeepaliveOptions,
                                sslProfileName )
   return transportConfig

transportConfigObject = TacLazyType( 'Rpki::RpkiTransportConfig' )

def doEnterRpkiTransportTcpConfigMode( mode, args ):
   # This submode is mutually exclusive with 'transport tls' and 'transport ssh'
   # (TODO) so if the current auth type is not tcp, the config is overridden with
   # defaults.
   childMode = mode.childMode( RpkiTransportTcpConfigMode )
   currentTransportConfig = childMode.rpkiCacheConfig.transportConfig
   if currentTransportConfig.transportAuthenticationType != childMode.tcp:
      if isDefaultPortInUseBasedOnTransportType( childMode.rpkiCacheConfig ):
         childMode.rpkiCacheConfig.port = childMode.rpkiCacheConfig.tcpPortDefault
      transportConfig = updateTransportConfig(
         currentTransportConfig,
         transportAuthenticationType=childMode.tcp,
         tcpKeepaliveOptions=currentTransportConfig.tcpKeepaliveDefault,
         sslProfileName=transportConfigObject.sslProfileNameDefault )
      childMode.rpkiCacheConfig.transportConfig = transportConfig
   mode.session_.gotoChildMode( childMode )

def noEnterRpkiTransportTcpConfigMode( mode, args ):
   # Restore default TCP transport Config
   mode.rpkiCacheConfig.transportConfig = \
         transportConfigObject.cacheTransportConfigDefault

def doTcpKeepaliveCommand( mode, args ):
   idleTime = args[ 'IDLE_TIME' ]
   probeInterval = args[ 'PROBE_INTERVAL' ]
   probeCount = args[ 'PROBE_COUNT' ]
   keepalive = Tac.Value( 'Arnet::TcpKeepaliveOptions',
                          idleTime, probeInterval, probeCount )
   transportConfig = updateTransportConfig(
      mode.rpkiCacheConfig.transportConfig,
      tcpKeepaliveOptions=keepalive )
   mode.rpkiCacheConfig.transportConfig = transportConfig

def noTcpKeepaliveCommand( mode, args ):
   keepaliveDefault = transportConfigObject.tcpKeepaliveDefault
   transportConfig = updateTransportConfig(
      mode.rpkiCacheConfig.transportConfig,
      tcpKeepaliveOptions=keepaliveDefault )
   mode.rpkiCacheConfig.transportConfig = transportConfig

def doEnterRpkiTransportTlsConfigMode( mode, args ):
   childMode = mode.childMode( RpkiTransportTlsConfigMode )
   currentTransportConfig = childMode.rpkiCacheConfig.transportConfig
   if currentTransportConfig.transportAuthenticationType != childMode.tls:
      if isDefaultPortInUseBasedOnTransportType( childMode.rpkiCacheConfig ):
         childMode.rpkiCacheConfig.port = childMode.rpkiCacheConfig.tlsPortDefault
      transportConfig = updateTransportConfig(
         currentTransportConfig,
         tcpKeepaliveOptions=currentTransportConfig.tcpKeepaliveDefault,
         transportAuthenticationType=childMode.tls )
      childMode.rpkiCacheConfig.transportConfig = transportConfig
   mode.session_.gotoChildMode( childMode )

def noEnterRpkiTransportTlsConfigMode( mode, args ):
   mode.rpkiCacheConfig.transportConfig = \
      transportConfigObject.cacheTransportConfigDefault

def doRpkiTlsSslProfileCmd( mode, args ):
   sslProfileName = args[ 'PROFILE_NAME' ]
   currentTransportConfig = mode.rpkiCacheConfig.transportConfig
   transportConfig = updateTransportConfig( currentTransportConfig,
                                            sslProfileName=sslProfileName )
   mode.rpkiCacheConfig.transportConfig = transportConfig

def noRpkiTlsSslProfileCmd( mode, args ):
   currentTransportConfig = mode.rpkiCacheConfig.transportConfig
   sslProfileNameDefault = transportConfigObject.sslProfileNameDefault
   transportConfig = updateTransportConfig( currentTransportConfig,
                                            sslProfileName=sslProfileNameDefault )
   mode.rpkiCacheConfig.transportConfig = transportConfig

@ArBgpShowOutput( 'RpkiShowRoas', arBgpModeOnly=True )
def doRpkiShowRoas( mode, args ):
   prefix = None
   asdot = handlerGv.asnConfig.isAsdotConfigured()
   # If we pass an unmounted LazyMount entity to the SM constructor, it will hang
   # when it attempts to mount it.  So we force the mount first to prevent this.
   gv.rpkiStatus.force()
   gv.roaTableStatusDir.force()
   if 'ipv6' in args:
      showRoaSm = Tac.newInstance( 'Rpki::RpkiShowRoa6Sm', handlerGv.rpkiRoaStore,
                                   asdot, gv.rpkiStatus, gv.roaTableStatusDir )
      prefix = args.get( 'PREFIX6' )
      if prefix:
         showRoaSm.prefix = Ip6Prefix( prefix )
   else:
      showRoaSm = Tac.newInstance( 'Rpki::RpkiShowRoa4Sm', handlerGv.rpkiRoaStore,
                                   asdot, gv.rpkiStatus, gv.roaTableStatusDir )
      prefix = args.get( 'PREFIX4' )
      if prefix:
         showRoaSm.prefix = Prefix( prefix )

   # cacheFilter and filterCompare are used to
   # determine what rows in the store to skip over
   # cFilter of 0 means that there will be no cache filtering

   # when cFilter & roa->cachesReported() == filterCompare, we skip that roa/ row
   cFilter = 0

   if 'cache' in args:
      cacheName = args.get( 'NAME' )
      # if the user is filtering the store based on a specified cache
      if cacheName:
         status = gv.rpkiStatus.cacheStatus.get( cacheName )
         if status:
            cFilter = cFilter | gv.rpkiStatus.cacheStatus[ cacheName ].cacheBit
         else:
            # Treating this as a cli error, because the c++helper is not invoked
            # and there will not be any output to validate against the CliModel.
            mode.addError( getCacheMessage( cacheName ) )
            return None
      if 'missing' in args:
         # if the user is filtering the store based on the absence from any cache
         if not cacheName:
            cacheList = getCacheNames( mode )
            for cacheName in cacheList:
               if gv.rpkiStatus.cacheStatus[ cacheName ].cacheState == 'synced':
                  cFilter = cFilter | gv.rpkiStatus.cacheStatus[ cacheName ].cacheBit
         showRoaSm.filterCompare = cFilter
   showRoaSm.cacheFilter = cFilter

   if 'table' in args:
      tableName = args.get( 'TABLENAME' )
      if tableName:
         tableStatus = gv.roaTableStatusDir.roaTableStatus.get( tableName )
         if tableStatus:
            showRoaSm.specifiedTable = tableName
         else:
            # Treating this as a cli error, because the c++helper is not invoked
            # and there will not be any output to validate against the CliModel.
            mode.addError( getRoaTableMessage( tableName ) )
            return None

   showRoaSm.sortRoas()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   showRoaSm.printRoas( fd, fmt )
   return RpkiCliModels.RpkiShowRoaModel

@ArBgpShowOutput( 'RpkiShowRoaSummary', arBgpModeOnly=True )
def doRpkiShowRoaSummary( mode, args ):
   model = RpkiCliModels.RpkiShowRoaSummaryModel()

   allCacheStatus = gv.rpkiStatus.cacheStatus

   if not allCacheStatus:
      msg = getCacheMessage( None )
      mode.addMessage( msg )

   for cacheName, status in allCacheStatus.items():
      summaryModel = RpkiCliModels.CacheRoaSummaryModel()
      summaryModel.ipv4RoaCount = status.ip4RoaCount
      summaryModel.ipv6RoaCount = status.ip6RoaCount

      model.caches[ cacheName ] = summaryModel

   for roaTable in gv.roaTableStatusDir.roaTableStatus.values():
      summaryModel = RpkiCliModels.TableRoaSummaryModel()
      summaryModel.ipv4RoaCount = roaTable.ip4RoaCount
      summaryModel.ipv6RoaCount = roaTable.ip6RoaCount
      model.tables[ roaTable.roaTableName ] = summaryModel

   model.total = RpkiCliModels.CacheRoaSummaryModel()
   model.total.ipv4RoaCount = len( handlerGv.rpkiRoaStore.ip4Roa )
   model.total.ipv6RoaCount = len( handlerGv.rpkiRoaStore.ip6Roa )
   return model

@ArBgpShowOutput( 'RpkiShowRoaTableSummary', arBgpModeOnly=True )
def doRpkiShowRoaTableSummary( mode, args ):
   model = RpkiCliModels.RpkiShowRoaTableSummaryModel()

   allRoaTableNames = gv.roaTableStatusDir.roaTableStatus
   if not allRoaTableNames:
      msg = getRoaTableMessage( None )
      mode.addMessage( msg )
      return model
   allCacheNames = getCacheNames( mode )
   allCacheNames = sorted( allCacheNames )

   for tableName in allRoaTableNames:
      summaryModel = RpkiCliModels.RoaTableSummaryModel()
      cacheList = []

      for cacheName in allCacheNames:
         if cacheName not in gv.rpkiStatus.cacheStatus:
            # Only up to 32 caches are created.
            # Skip if cache does not exist in cacheStatus.
            continue
         if gv.roaTableStatusDir.roaTableStatus[ tableName ].roaTableBit &\
            gv.rpkiStatus.cacheStatus[ cacheName ].destinationRoaTablesBit:
            cacheList.append( cacheName )

      summaryModel.caches = cacheList
      summaryModel.status = gv.roaTableStatusDir.roaTableStatus[ tableName ].state
      model.roaTables[ tableName ] = summaryModel

   return model

def createSubmodel( cacheName, status, mode, model, detail ):
   connectionData = handlerGv.rpkiStatistics.cacheConnectionData.get( cacheName )
   if not connectionData:
      return
   submodel = RpkiCliModels.CacheModel()
   submodel.host = status.host
   submodel.port = status.port
   submodel.vrf = status.vrf
   submodel.refreshInterval = status.effectiveRefreshInterval
   submodel.retryInterval = status.effectiveRetryInterval
   submodel.expireInterval = status.effectiveExpireInterval
   submodel.preference = status.preference
   submodel.state = status.cacheState
   submodel.setProtocolVersion( Tac.enumValue( 'Rpki::RpkiProtocolVersion',
                                               status.protocolVersion ) )
   submodel.sessionId = status.sessionId
   submodel.serialNumber = status.serialNumber
   submodel.lastUpdateSync = Ark.switchTimeToUtc(
      connectionData.lastUpdateSyncTimestamp )
   submodel.lastFullSync = Ark.switchTimeToUtc(
      connectionData.lastFullSyncTimestamp )
   submodel.entries = status.ip4RoaCount + status.ip6RoaCount
   model.caches[ cacheName ] = submodel
   submodel.deleted = cacheName not in gv.rpkiConfig.cacheConfig
   submodel.activeSince = Ark.switchTimeToUtc(
      connectionData.lastConnectionTimestamp )
   # setReasonInactive should be called after deleted is set
   submodel.setReasonInactive( connectionData.errorReason,
                               connectionData.lastConnectionTimestamp )
   submodel.transportProtocol = status.transportConfig.transportAuthenticationType
   submodel.lastSerialQuery = Ark.switchTimeToUtc(
            connectionData.lastSerialQueryTimestamp )
   submodel.lastResetQuery = Ark.switchTimeToUtc(
      connectionData.lastResetQueryTimestamp )
   submodel.setTransportInformation( status.transportConfig )
   roaTables = []
   for name in sorted( status.appliedRoaTableName ):
      if name != 'default':
         roaTables.append( name )
   if 'default' in status.appliedRoaTableName:
      roaTables.insert( 0, 'default' )
   submodel.roaTables = roaTables
   if detail:
      detailModel = RpkiCliModels.CacheDetailModel()
      tcpKeepaliveOptions = status.transportConfig.tcpKeepaliveOptions
      detailModel.tcpKeepaliveIdleTime = tcpKeepaliveOptions.idleTime
      detailModel.tcpKeepaliveProbeInterval = tcpKeepaliveOptions.probeInterval
      detailModel.tcpKeepaliveProbeCount = tcpKeepaliveOptions.probeCount
      detailModel.lastConfigChange = Ark.switchTimeToUtc(
         connectionData.lastConfigChangeTimestamp )
      detailModel.lastConnectionError = Ark.switchTimeToUtc(
         connectionData.lastConnectionErrorTimestamp )
      detailModel.lastProtocolError = Ark.switchTimeToUtc(
         connectionData.lastErrorTimestamp )
      detailModel.overrideConfiguredIntervalValues = \
         status.overrideConfiguredIntervalValues
      submodel.detail = detailModel
      if submodel.deleted:
         # No need to gather TCP information for deleted cache
         return
      tcpModel = None
      buff = StringIO()
      cmd = f'TCP_INFORMATION {cacheName}'
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            'Rpki', 'RpkiCli', cmd,
                                            stringBuff=buff )
      try:
         data = ast.literal_eval( buff.getvalue() )
         if 'tcpStats' in data:
            tcpModel = RpkiCliModels.TcpInformationModel()
            tcpStats = data[ 'tcpStats' ]
            tcpModel.setAttrFromDict( tcpStats )
            detailModel.tcpInformation = tcpModel
      except SyntaxError:
         t0( 'Error: Unable to retrieve TCP socket information correctly for',
             cacheName )

@ArBgpShowOutput( 'RpkiShowCache', arBgpModeOnly=True )
def doRpkiShowCache( mode, args ):
   model = RpkiCliModels.RpkiShowCacheModel()
   model.caches = {}
   detail = 'detail' in args

   cacheName = args.get( 'NAME' )
   if cacheName:
      status = gv.rpkiStatus.cacheStatus.get( cacheName )
      if status:
         createSubmodel( cacheName, status, mode, model, detail )
      elif cacheName in gv.rpkiConfig.cacheConfig:
         mode.addMessage( getCacheUnusedMessage( [ cacheName ] ) )
      else:
         mode.addMessage( getCacheMessage( cacheName ) )
      return model

   # Check if there are no caches
   if not gv.rpkiStatus.cacheStatus and not gv.rpkiConfig.cacheConfig:
      mode.addMessage( getCacheMessage( None ) )

   # Display all caches with a status
   for cacheName, status in gv.rpkiStatus.cacheStatus.items():
      createSubmodel( cacheName, status, mode, model, detail )

   # display unused caches (cacheConfig but no cacheStatus)
   unusedCaches = [ cacheName for cacheName in gv.rpkiConfig.cacheConfig
                    if cacheName not in gv.rpkiStatus.cacheStatus ]
   if unusedCaches:
      mode.addMessage( getCacheUnusedMessage( unusedCaches ) )
   return model

@ArBgpShowOutput( 'RpkiShowCachePduCounters', arBgpModeOnly=True )
def doRpkiShowCachePduCounters( mode, args ):
   caches = args.get( 'CACHES' )
   model = RpkiCliModels.RpkiShowCacheCounterModel()
   model.caches = {}

   if not caches:
      msg = getCacheMessage( args.get( 'NAME' ) )
      mode.addMessage( msg )

   for cache in caches:
      counterModel = RpkiCliModels.CacheCounterModel()
      if Toggles.RpkiToggleLib.toggleRpkiCounterSysdbEnabled():
         pduCounter = handlerGv.cacheCounterDir.cacheCounter[ cache ].pduCounter
      else:
         # show the difference between current and snapshot counters
         current = handlerGv.rpkiCacheCounterSmash.counter[ cache ].pduCounter
         snapshot = handlerGv.rpkiCacheCounterSnapshotSmash.\
         counterSnapshot[ cache ].pduCounter
         pduCounter = current - snapshot

      counterModel.serialNotify = pduCounter.serialNotify
      counterModel.serialQuery = pduCounter.serialQuery
      counterModel.resetQuery = pduCounter.resetQuery
      counterModel.ipv4Prefix = pduCounter.ipv4Prefix
      counterModel.ipv6Prefix = pduCounter.ipv6Prefix
      counterModel.endOfData = pduCounter.endOfData
      counterModel.cacheReset = pduCounter.cacheReset
      counterModel.cacheResponse = pduCounter.cacheResponse

      model.caches[ cache.stringValue ] = counterModel

   return model

@ArBgpShowOutput( 'RpkiShowCachePduCountersErrors', arBgpModeOnly=True )
def doRpkiShowCachePduCountersErrors( mode, args ):
   caches = args.get( 'CACHES' )
   model = RpkiCliModels.RpkiShowCacheErrorCounterModel()
   model.cachesRx = {}
   model.cachesTx = {}

   if not caches:
      msg = getCacheMessage( args.get( 'NAME' ) )
      mode.addMessage( msg )

   for cache in caches:
      if Toggles.RpkiToggleLib.toggleRpkiCounterSysdbEnabled():
         pduCounter = handlerGv.cacheCounterDir.cacheCounter[ cache ].pduErrorCounter
      else:
         # show the difference between current and snapshot counters
         current = handlerGv.rpkiCacheCounterSmash.counter[ cache ].pduErrorCounter
         snapshot = handlerGv.rpkiCacheCounterSnapshotSmash.\
         counterSnapshot[ cache ].pduErrorCounter
         pduCounter = current - snapshot
      # tx
      txErrorPdu = RpkiCliModels.CacheTxErrorCounterModel()

      txErrorPdu.corruptData = pduCounter.txCorruptData
      txErrorPdu.internalError = pduCounter.txInternalError
      txErrorPdu.unsupportedProtocol = pduCounter.txUnsupportedProtocol
      txErrorPdu.unsupportedPdu = pduCounter.txUnsupportedPdu
      txErrorPdu.unexpectedProtocol = pduCounter.txUnexpectedProtocol
      txErrorPdu.other = pduCounter.txOther
      txErrorPdu.withdrawalUnknown = pduCounter.txWithdrawalUnknown
      txErrorPdu.duplicateAnnouncement = pduCounter.txDuplicateAnnouncement

      # rx
      rxErrorPdu = RpkiCliModels.CacheRxErrorCounterModel()

      rxErrorPdu.corruptData = pduCounter.rxCorruptData
      rxErrorPdu.internalError = pduCounter.rxInternalError
      rxErrorPdu.unsupportedProtocol = pduCounter.rxUnsupportedProtocol
      rxErrorPdu.unsupportedPdu = pduCounter.rxUnsupportedPdu
      rxErrorPdu.unexpectedProtocol = pduCounter.rxUnexpectedProtocol
      rxErrorPdu.other = pduCounter.rxOther
      rxErrorPdu.noData = pduCounter.rxNoData
      rxErrorPdu.invalidRequest = pduCounter.rxInvalidRequest
      rxErrorPdu.pduTimeout = pduCounter.pduTimerExpire

      model.cachesTx[ cache.stringValue ] = txErrorPdu
      model.cachesRx[ cache.stringValue ] = rxErrorPdu

   return model

def doRpkiClearCacheCmd( mode, args ):
   cacheName = args.get( 'NAME' )
   cmd = 'CLEAR_CACHE'
   if cacheName:
      cmd = cmd + f' {cacheName}'
   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         'Rpki', 'RpkiCli', cmd )

def doRpkiClearCacheCounterCmd( mode, args ):
   cmd = 'CLEAR_COUNTER'
   name = args.get( 'NAME' )
   if name:
      cmd += f' {name}'
   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         'Rpki', 'RpkiCli', cmd )

def setTristateRpkiOriginValidationMethod( bgpConfig, attrName, method ):
   setattr( bgpConfig, attrName,
            Tac.Value( "Routing::Bgp::TristateRpkiOriginValidationMethod",
                       value=method,
                       isSet=True ) )

def clearTristateRpkiOriginValidationMethod( bgpConfig, attrName ):
   default = getattr( bgpConfig, attrName + 'Default' )
   setattr( bgpConfig, attrName,
            Tac.Value( "Routing::Bgp::TristateRpkiOriginValidationMethod",
                       value=default,
                       isSet=False ) )

def setTristateRpkiOriginValidationSend( bgpConfig, attrName ):
   setattr( bgpConfig, attrName, 'isTrue' )

def clearTristateRpkiOriginValidationSend( bgpConfig, attrName ):
   setattr( bgpConfig, attrName, 'isInvalid' )

ovMethodEnum = Tac.Type( "Routing::Bgp::Rpki::OriginValidationMethod" )

def ovMethod( args ):
   if 'local' in args:
      return ovMethodEnum.ovLocal
   elif 'community' in args:
      return ovMethodEnum.ovCommunity
   elif 'prefer-community' in args:
      return ovMethodEnum.ovPreferCommunity
   elif 'disabled' in args:
      return ovMethodEnum.ovDisabled
   else:
      assert False, f"Unknown method {args}"
      return None  # keep pylint happy

def setOriginValidationMethod( self, args ):
   bgpConfig = configForVrf( self.vrfName )
   method = ovMethod( args )
   if args.get( 'ibgp' ):
      attrName = 'rpkiIbgpOvMethod'
   elif args.get( 'ebgp' ):
      attrName = 'rpkiEbgpOvMethod'
   elif args.get( 'redistributed' ):
      attrName = 'rpkiRedistOvMethod'
   setTristateRpkiOriginValidationMethod( bgpConfig, attrName, method )

def noOriginValidationMethod( self, args ):
   bgpConfig = configForVrf( self.vrfName )
   if args.get( 'ibgp' ):
      attrName = 'rpkiIbgpOvMethod'
   elif args.get( 'ebgp' ):
      attrName = 'rpkiEbgpOvMethod'
   elif args.get( 'redistributed' ):
      attrName = 'rpkiRedistOvMethod'
   clearTristateRpkiOriginValidationMethod( bgpConfig, attrName )

def setOriginValidationSend( self, args ):
   bgpConfig = configForVrf( self.vrfName )
   if args.get( 'ibgp' ):
      attrName = 'rpkiIbgpOvSend'
   elif args.get( 'ebgp' ):
      attrName = 'rpkiEbgpOvSend'
   setTristateRpkiOriginValidationSend( bgpConfig, attrName )

def noOriginValidationSend( self, args ):
   bgpConfig = configForVrf( self.vrfName )
   if args.get( 'ibgp' ):
      attrName = 'rpkiIbgpOvSend'
   elif args.get( 'ebgp' ):
      attrName = 'rpkiEbgpOvSend'
   clearTristateRpkiOriginValidationSend( bgpConfig, attrName )

def setOriginValidationRoutemap( self, args ):
   bgpConfig = configForVrf( self.vrfName )
   if args.get( 'redistributed' ):
      bgpConfig.rpkiRedistOvRouteMap = args[ 'ROUTEMAP' ]
   else:
      bgpConfig.rpkiOvRouteMap = args[ 'ROUTEMAP' ]

def noOriginValidationRoutemap( self, args ):
   bgpConfig = configForVrf( self.vrfName )
   if args.get( 'redistributed' ):
      bgpConfig.rpkiRedistOvRouteMap = bgpConfig.rpkiRedistOvRouteMapDefault
   else:
      bgpConfig.rpkiOvRouteMap = bgpConfig.rpkiOvRouteMapDefault

def doEnterBgpBaseRpkiOriginValidationMode( mode, args ):
   childMode = mode.childMode( RpkiOriginValidationMode )
   mode.session_.gotoChildMode( childMode )

def noEnterBgpBaseRpkiOriginValidationMode( mode, args ):
   bgpConfig = configForVrf( mode.vrfName )
   # Set all vars back to the defaults.
   clearTristateRpkiOriginValidationMethod( bgpConfig, 'rpkiEbgpOvMethod' )
   clearTristateRpkiOriginValidationSend( bgpConfig, 'rpkiEbgpOvSend' )
   clearTristateRpkiOriginValidationMethod( bgpConfig, 'rpkiIbgpOvMethod' )
   clearTristateRpkiOriginValidationSend( bgpConfig, 'rpkiIbgpOvSend' )
   bgpConfig.rpkiOvRouteMap = bgpConfig.rpkiOvRouteMapDefault
   clearTristateRpkiOriginValidationMethod( bgpConfig, 'rpkiRedistOvMethod' )
   bgpConfig.rpkiRedistOvRouteMap = bgpConfig.rpkiRedistOvRouteMapDefault

def doBgpNeighborRpkiRecvOriginValidation( mode, args ):
   config = bgpNeighborConfig( args[ 'PEER' ], vrfName=mode.vrfName )
   config.rpkiOvMethod = ovMethod( args )
   config.rpkiOvMethodPresent = True

def noBgpNeighborRpkiRecvOriginValidation( mode, args ):
   config = bgpNeighborConfig( args[ 'PEER' ], create=False,
                               vrfName=mode.vrfName )
   if config:
      config.rpkiOvMethod = config.rpkiOvMethodDefault
      config.rpkiOvMethodPresent = False

def doBgpNeighborRpkiCommunitySend( mode, args ):
   config = bgpNeighborConfig( args[ 'PEER' ], vrfName=mode.vrfName )
   config.rpkiOvSend = 'disabled' not in args
   config.rpkiOvSendPresent = True

def noBgpNeighborRpkiCommunitySend( mode, args ):
   config = bgpNeighborConfig( args[ 'PEER' ], create=False,
                               vrfName=mode.vrfName )
   if config:
      config.rpkiOvSend = config.rpkiOvSendDefault
      config.rpkiOvSendPresent = False

def Plugin( entityManager ):
   handlerGv.rpkiStatistics = LazyMount.mount( entityManager,
                                     Cell.path( 'routing/rpki/cache/statistics' ),
                                     'Rpki::CacheStatisticsDir', 'r' )
   roaMountPath = Tac.Type( 'Rpki::RpkiRoaStore' ).smashPath
   handlerGv.rpkiRoaStore = \
      SmashLazyMount.mount( entityManager,
                            roaMountPath,
                            'Rpki::RpkiRoaStore',
                            SmashLazyMount.mountInfo( 'reader' ) )
   cacheCounterPath = Tac.Type( 'Rpki::AllRpkiCacheCounterDir' ).smashPath
   handlerGv.rpkiCacheCounterSmash = \
      SmashLazyMount.mount( entityManager,
                            cacheCounterPath,
                            'Rpki::AllRpkiCacheCounterDir',
                            SmashLazyMount.mountInfo( 'reader' ) )
   counterSnapshotPath = Tac.Type(
      'Rpki::AllRpkiCacheCounterSnapshotDir' ).smashPath
   handlerGv.rpkiCacheCounterSnapshotSmash = \
      SmashLazyMount.mount( entityManager,
                            counterSnapshotPath,
                            'Rpki::AllRpkiCacheCounterSnapshotDir',
                            SmashLazyMount.mountInfo( 'reader' ) )
   handlerGv.asnConfig = LazyMount.mount( entityManager, 'routing/bgp/asn/config',
                                'Routing::AsnConfig', 'r' )
   handlerGv.cacheCounterDir = LazyMount.mount( entityManager,
                                     Cell.path( 'routing/rpki/cache/counter' ),
                                     'Rpki::CacheCounterDir', 'r' )
