#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin
from CliModel import cliPrinted
from CliPlugin.ArBgpCli import ArBgpAsyncCliCommand
from CliPlugin.BgpCliHelperCli import convertPeerAddr
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
   summaryVrfModel,
)
from IpLibConsts import DEFAULT_VRF

ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )
BgpCliHelperCliHandler = CliDynamicPlugin( "BgpCliHelperCliHandler" )
BgpCliHelperCommand = BgpCliHelperCliHandler.BgpCliHelperCommand
RtMembershipCliModels = CliDynamicPlugin( "RtMembershipCliModels" )

@ArBgpShowOutput( 'doShowBgpRtMembershipSummary', arBgpModeOnly=True )
def ShowBgpRtMembershipSummary_handler( mode, args ):
   cmd = ArBgpAsyncCliCommand( mode, 'show summary', vrfName=DEFAULT_VRF,
                                 nlriAfiSafi='rt-membership' )
   cmd.run()
   return cliPrinted( summaryVrfModel )

@ArBgpShowOutput( 'doShowRtMembershipFloodlist', arBgpModeOnly=True )
def ShowRtMembershipFloodlist_handler( mode, args ):
   cmd = ArBgpAsyncCliCommand( mode, 'show rt-membership floodlist' )
   if 'ROUTE_TARGET' in args:
      cmd.addParam( 'routeTarget', args[ 'ROUTE_TARGET' ] )
   cmd.run()
   return cliPrinted( RtMembershipCliModels.ShowRtMembershipFloodlistModel )

@ArBgpShowOutput( 'doShowBgpNeighborsRtMembershipFilter', arBgpModeOnly=True )
def ShowNeighborRtMembershipFilter_handler( mode, args ):
   kwargs = { 'peerAddr': args.get( 'PEER' ) }
   convertPeerAddr( kwargs )
   cmd = ArBgpAsyncCliCommand( mode, 'show neighbors rt-membership filter',
                               **kwargs )
   cmd.run()
   return cliPrinted( RtMembershipCliModels.ShowRtMembershipNeighborFilterModel )

@ArBgpShowOutput( 'doShowBgpRtMembership', arBgpModeOnly=True )
def ShowBgpRtMembership_handler( mode, args ):
   return BgpCliHelperCommand(
      mode,
      'show bgp rt-membership',
      afi='rt-membership',
      detail=args.get( 'detail' ),
      model=RtMembershipCliModels.rtMembershipRoutesVrfModel
      ).run()

@ArBgpShowOutput( 'doShowBgpPrefixRtMembership', arBgpModeOnly=True )
def ShowBgpPrefixRtMembership_handler( mode, args ):
   rt = args.get( 'ROUTE_TARGET' )
   routeTarget = '0x0000000000000000' if rt[ 0 ] == '0' else rt[ 0 ]
   rtPrefixLength = '' if len( rt ) != 2 else rt[ 1 ]
   return BgpCliHelperCommand(
      mode,
      'show bgp rt-membership',
      afi='rt-membership',
      routeTarget=routeTarget,
      rtPrefixLength=rtPrefixLength,
      detail=args.get( 'detail' ),
      model=RtMembershipCliModels.rtMembershipRoutesVrfModel
      ).run()

@ArBgpShowOutput( 'doShowBgpOriginAsRtMembership', arBgpModeOnly=True )
def ShowBgpOriginAsRtMembership_handler( mode, args ):
   return BgpCliHelperCommand(
      mode,
      'show bgp rt-membership',
      afi='rt-membership',
      originAs=args.get( 'AS_NUM' ),
      detail=args.get( 'detail' ),
      model=RtMembershipCliModels.rtMembershipRoutesVrfModel
      ).run()

@ArBgpShowOutput( 'doShowBgpNeighborsRtMembership', arBgpModeOnly=True )
def ShowBgpNeighborsRtMembership_handler( mode, args ):
   peerAddr = { 'peerAddr': args.get( 'PEER' ) }
   convertPeerAddr( peerAddr )
   return BgpCliHelperCommand(
      mode,
      'show bgp rt-membership',
      afi='rt-membership',
      peerAddr=peerAddr.get( 'peerAddr' ),
      llIntf=peerAddr.get( 'llIntf' ),
      routeType=args.get( 'ROUTE_TYPE' ),
      model=RtMembershipCliModels.rtMembershipRoutesVrfModel
      ).run()

def getRtNlriAttrs( args, originAs, rt, rtPrefixLength ):
   # We always need to get the user supplied value for the Nlri String
   # If the user gives an as dot number like 100.100, the following logic
   # will make sure the nlri string contains 100.100 instead of 100*U32MAX+100
   originAs = args.get( 'AS_DOT_NUM', originAs )
   # Make sure we always represent any variant of the 0/0 rt prefix as 0/0
   if rt.startswith( "0x0" ) and int( rt, 16 ) == 0 and rtPrefixLength == "0":
      rt = "0"
   nlriAttrs = {
      'originAs': str( originAs ),
      'routeTarget': f"{rt}/{rtPrefixLength}",
   }
   return nlriAttrs

def getRtAdditionalParams( originAs, rt, rtPrefixLength ):
   if rt == "0":
      rt = "0x0000000000000000"
   additionalParams = {
       'originAs': str( originAs ),
       'routeTarget': rt,
       'rtPrefixLength': rtPrefixLength,
   }
   return additionalParams

def validateNlriAttrs( mode, nlriAttrs ):
   originAs = nlriAttrs[ 'originAs' ]
   routeTarget = nlriAttrs[ 'routeTarget' ]
   if originAs == 'none' and routeTarget != '0/0':
      mode.addError( f"Invalid route-target {routeTarget} for origin-as {originAs}" )
      return False
   return True

@ArBgpShowOutput( 'doShowBgpDebugPolicyRtMembership', arBgpModeOnly=True )
def policyDebugRtMembershipHandler( mode, args ):
   originAs = args[ 'AS_NUM' ]
   rtPrefix = args[ 'ROUTE_TARGET' ]
   # rtPrefix is a tuple where the first part is the route target, and
   # the second part is optionally the prefix length
   routeTarget = rtPrefix[ 0 ]
   # Check if the prefix length is present, if not use the default value of 64
   rtPrefixLength = '64' if len( rtPrefix ) != 2 else rtPrefix[ 1 ]

   nlriType = 'rtMembership'
   nlriAttrs = getRtNlriAttrs( args, originAs, routeTarget, rtPrefixLength )
   additionalParams = getRtAdditionalParams( originAs, routeTarget, rtPrefixLength )
   if not validateNlriAttrs( mode, nlriAttrs ):
      return None

   return ArBgpCliHandler.policyDebugHandlerCommon( mode, args, nlriType, nlriAttrs,
      additionalParams=additionalParams )
