# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import Ip4Address
from CliModel import (
      Bool,
      DeferredModel,
      Dict,
      Enum,
      GeneratorDict,
      Int,
      List,
      Submodel,
      Str,
)
from CliPlugin.BgpCliModels import (
      BgpPeerAddress,
      BgpRoutePeerEntry,
      originValues,
      REASON_NOT_BEST_LIST,
      RouteTarget,
)
from CliPlugin.VrfCli import generateVrfCliModel

##########
# Models for "show bgp rt-membership floodlist"
##########

class ShowRtMembershipFloodlistRouteTargetModel( DeferredModel ):
   peers = List( valueType=BgpPeerAddress,
                 help='RT membership VRF and route target peer floodlist' )

class ShowRtMembershipFloodlistVrfModel( DeferredModel ):
   routeTargets = Dict(
      keyType=RouteTarget,
      valueType=ShowRtMembershipFloodlistRouteTargetModel,
      help='RT membership VRF floodlist information, keyed by route target' )

class ShowRtMembershipFloodlistModel( DeferredModel ):
   vrfs = Dict( valueType=ShowRtMembershipFloodlistVrfModel,
                help='RT membership floodlist information, keyed by VRF name' )

##########
# Models for "show bgp neighbors <nbr> rt-membership filter"
##########

class ShowRtMembershipNeighborFilterPeerModel( DeferredModel ):
   routeTargets = List( valueType=RouteTarget,
                        help='RT membership VRF and neighbor route target filter' )

class ShowRtMembershipNeighborFilterVrfModel( DeferredModel ):
   peers = Dict(
      keyType=BgpPeerAddress,
      valueType=ShowRtMembershipNeighborFilterPeerModel,
      help='RT membership VRF neighbor filter information, keyed by peer address' )

class ShowRtMembershipNeighborFilterModel( DeferredModel ):
   vrfs = Dict( valueType=ShowRtMembershipNeighborFilterVrfModel,
                help='RT membership neighbor filter information, keyed by VRF name' )

class RtMembershipRouteASPathEntry( DeferredModel ):
   asPath = Str( optional=True, help='AS path string (if absent,  '
                 'then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 'Confed-External', 'Local',
                               'Invalid' ),
                      help='AS path type: '
                            'Internal - originated by I-BGP '
                            'External - originated by E-BGP '
                            'Confed-External - originated by a E-BGP confederation '
                            'Local - originated locally '
                            'Invalid - AS path is invalid' )

class RtMembershipRouteTypeEntry ( DeferredModel ):
   stale = Bool( default=False, help='Route is stale' )
   valid = Bool( default=False, help='Route is valid' )
   suppressed = Bool( default=False,
                      help='Route is suppressed from entering the Rib' )
   active = Bool( default=False, help='Route is active' )
   backup = Bool( default=False, help='Route is backup' )
   ecmpHead = Bool( default=False, help='Route is the ECMP head' )
   ecmp = Bool( default=False, help='Route is an ECMP route' )
   ucmp = Bool( optional=True, default=False, help='Route is an UCMP route' )
   ecmpContributor = Bool( default=False, help='Route contributes to ECMP' )
   atomicAggregator = Bool( default=False, help='Route is an atomic-aggregate' )
   queued = Bool( default=False, help='Route is queued for advertisement' )
   localAgg = Bool( optional=True, help='Route is locally aggregated' )
   origin = Enum( optional=True, values=originValues, help='Route origin' )
   waitForConvergence = Bool( optional=True,
                              help='Route is pending best path selection' )

class RtMembershipRouteDetailEntry( DeferredModel ):
   communities = List( valueType=str, help='List of communities for this route' )
   extCommunities = List( valueType=str,
                          help='List of extended communities for this route' )
   extCommunityListRaw = List(
      valueType=int, help='List of raw extended communities for this route' )
   largeCommunities = List( valueType=str,
                            help='List of large communities for this route' )
   rxSafi = Str( optional=True, help='Received SAFI of this route' )
   origin = Enum( values=( 'Igp', 'Egp', 'Incomplete' ), help='Route origin' )
   originator = Ip4Address( optional=True,
                            help='Router ID of the originator of this route' )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True,
                         help='Peer information for the route' )
   fromRRClient = Bool( default=False,
                        help='Route received from route reflector client' )
   reasonNotBestpath = Enum( optional=True, values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class RtMembershipRoutePath( DeferredModel ):
   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=RtMembershipRouteASPathEntry,
                           help='AS path information' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=RtMembershipRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   tag = Int( optional=True, help='Tag for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received."
                          "Only returned with detail output" )
   routeDetail = Submodel( valueType=RtMembershipRouteDetailEntry, optional=True,
                           help='Route details' )

class RtMembershipRouteTargetPrefix( DeferredModel ):
   routeTarget = Int( help='The Route Target prefix value' )
   maskLength = Int( help='The number of bits (0-64) in the Route Target prefix' )

class RtMembershipRouteKeyDetail( DeferredModel ):
   originAs = Str( help='The ASN of the Autonomous System that originated the '
                        'Route Target prefix' )
   routeTargetPrefix = Submodel( valueType=RtMembershipRouteTargetPrefix,
                                 help='The Route Target prefix' )
   nlriLength = Int( help='The number of bits (0-96) in the overall Route '
                          'Target membership NLRI' )

class RtMembershipRouteEntry( DeferredModel ):
   routeKeyDetail = Submodel( valueType=RtMembershipRouteKeyDetail, help='NLRI '\
                              'details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   rtMembershipRoutePaths = List( valueType=RtMembershipRoutePath,
         help='List of BGP RT MEMBERSHIP route ECMP paths' )

# Revision 2 may include routes with omitted Origin AS, not present in revision 1.
# Revision 1 schema is not different: originAs is a string in both cases.
# There is no `degrade` method, since output is generated by
# `ShowBgpRtMembership` TACC class.
class RtMembershipRoutes( DeferredModel ):
   __revision__ = 2
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   rtMembershipRoutes = GeneratorDict( keyType=str, valueType=RtMembershipRouteEntry,
      help='Dictionary of BGP RT MEMBERSHIP route entries indexed by the route key' )

# Wrap RtMembershipRoutes model under "vrfs" key
rtMembershipRoutesVrfModel = generateVrfCliModel( RtMembershipRoutes,
                                                  "RT MEMBERSHIP route summary",
                                                  revision=2 )
