# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Dict
from CliModel import Int
from CliModel import Model
from CliModel import Str
import TableOutput

#---------------------------------------------------------------------------------
# show hardware system spi-chips
#---------------------------------------------------------------------------------
class SpiChip( Model ):
   chipId = Int( help='Chip JEDEC ID', optional=True )
   chipModel = Str( help='Chip model', optional=True )
   chipName = Str( help='Chip name' )
   chipSize = Int( help='Chip capacity in megabits', optional=True )
   chipVendor = Str( help='Chip vendor', optional=True )
   slot = Str( help='Slot containing chip' )

class SpiChips( Model ):
   chips = Dict( keyType=str, valueType=SpiChip,
                 help='SPI flash chips on this system, keyed by slot and chip name' )

   def render( self ):
      if not self.chips:
         return

      table = TableOutput.createTable( ( 'Slot', 'Chip Name', 'Chip ID',
                                         'Chip Vendor', 'Chip Model',
                                         'Chip Size (Mb)' ) )

      justifyLeft = TableOutput.Format( justify="left" )
      justifyLeft.padLimitIs( True )
      justifyRight = TableOutput.Format( justify="right" )
      justifyRight.padLimitIs( True )
      table.formatColumns( justifyLeft, justifyLeft, justifyRight, justifyLeft,
                           justifyLeft, justifyRight )

      for chipKey in Arnet.sortIntf( self.chips ):
         chip = self.chips[ chipKey ]
         chipIdString = format( chip.chipId, "#08x" ) if chip.chipId else "n/a"
         chipVendor = chip.chipVendor if chip.chipVendor else "n/a"
         chipModel = chip.chipModel if chip.chipModel else "n/a"
         chipSize = chip.chipSize if chip.chipSize else "n/a"
         table.newRow( chip.slot, chip.chipName, chipIdString, chipVendor, chipModel,
                       chipSize )

      print( table.output() )
