# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
   Dict,
   Int,
   Model,
   Submodel
)
from CliPlugin.SfeFlowCacheWalkerCliLib import (
   CounterPrefix,
   WalkerCounterLib
)
from collections import OrderedDict
from natsort import natsorted

renderOrder = ( CounterPrefix.general,
               CounterPrefix.flowOffload,
               CounterPrefix.ipfix )

class Counter( Model ):
   counters = Dict( keyType=str, valueType=int, help='Key and value map' )

class WalkerCounter( Model ):
   walkerCounters = Dict( keyType=str,
                   valueType=Counter,
                   help='Flow cache walker counters per feature' )

class WalkerStatus( Model ):
   bucketStart = Int( help='Bucket start index' )
   bucketEnd = Int( help='Bucket end index' )
   lastScanDuration = Int( help="Last scan duration in milliseconds" )
   scanCount = Int( help='Scan count' )

def renderCounter( wc ):
   cLib = WalkerCounterLib()
   for gPrefix in renderOrder:
      gCounter = wc.walkerCounters.get( gPrefix, None )
      if gCounter and gCounter.counters:
         for cName, cVal in natsorted( gCounter.counters.items() ):
            cliText = cLib.allCounters[ gPrefix ][ cName ].cliText
            if gPrefix == CounterPrefix.general:
               print( f'{cliText.capitalize().strip()}: {cVal}' )
            else:
               key = f'{gPrefix} {cliText.lower()}'.strip()
               print( f'{key}: {cVal}' )

# ---------------------------------------------------------------------------------
# show agent sfe threads flow cache scan counters
#
# Purged count: 44
# IPFIX export count: 440
# IPFIX failed export count: 44
# ---------------------------------------------------------------------------------
class ModelShowCounters( Model ):
   counter = Submodel( WalkerCounter,
                        help='Flow cache walker counters per feature' )
   def render( self ):
      renderCounter( self.counter )

# ---------------------------------------------------------------------------------
# show agent sfe threads flow cache scan counters detail [thread <N>]
#
# Thread: 1
# Purged count: 22
# IPFIX export count: 220
# IPFIX failed export count: 22
# Flow offload flush count: 2
# Flow offload heart beat flow count: 1100

# Thread: 2
# Purged count: 22
# IPFIX export count: 220
# IPFIX failed export count: 22
# Flow offload flush count: 2
# Flow offload heart beat flow count: 1200
# ---------------------------------------------------------------------------------
class ModelShowCountersDetail( Model ):
   counterDetail = Dict( keyType=int,
                         valueType=WalkerCounter,
                         help='Flow cache walker counters per thread' )

   def render( self ):
      for threadId, walkerCounter in sorted( self.counterDetail.items() ):
         print( f'Thread: {threadId}' )
         renderCounter( walkerCounter )
         print( '' )

# ---------------------------------------------------------------------------------
# show agent sfe threads flow cache scan status [thread <N>]
#
# Thread: 1
# Bucket start: 1
# Bucket end: 524288
# Last scan duration: 1020 milliseconds
# Scan count: 11

# Thread: 2
# Bucket start: 524289
# Bucket end: 1048576
# Last scan duration: 1420 milliseconds
# Scan count: 11
# ---------------------------------------------------------------------------------
class ModelShowStatus( Model ):
   status = Dict( keyType=int, valueType=WalkerStatus,
                  help='Flow cache walker status per thread' )

   def render( self ):
      for threadId, ws in sorted( self.status.items() ):
         print( f'Thread: {threadId}' )

         stat = OrderedDict()
         stat[ 'Bucket start index' ] = ws.bucketStart
         stat[ 'Bucket end index' ] = ws.bucketEnd
         stat[ 'Last scan duration' ] = f'{ws.lastScanDuration} milliseconds'
         stat[ 'Scan count' ] = ws.scanCount

         for sName, sVal in stat.items():
            print( f'{sName.capitalize()}: {sVal}' )
         print( '' )
