#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
   Ip4Address,
   IpGenericAddress,
)
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.BgpCliModels import (
   BgpRouteASPathEntry,
   REASON_NOT_BEST_LIST,
)
from CliModel import (
   DeferredModel,
   Dict,
   Enum,
   Int,
   List,
   Str,
   Submodel,
)
from Toggles.PseudowireToggleLib import toggleVplsBgpSignalingEnabled

BgpVpnModels = CliDynamicPlugin( "BgpVpnModels" )
BgpVpnRouteDetailEntry = BgpVpnModels.BgpVpnRouteDetailEntry
BgpVpnRouteTypeEntry = BgpVpnModels.BgpVpnRouteTypeEntry

# This enum list should be a duplicate of the BgpVplsPlugin::VplsVeFwdTgtStatus enum
# defined in the BgpVpls package and MUST be kept in perfect sync with it. Ideally
# we would just use the actual enum in the CAPI model with something like:
#   FwdTgtStatus = TacLazyType( "Routing::Bgp::BgpVplsPlugin::VplsVeFwdTgtStatus" )
#   ...
#   error = Enum( values=FwdTgtStatus.attributes )
# However that will create a dependency between the CAPI model and libArBgp.so which
# is not allowed due to the size of libArBgp.so (this is enforced by Umbrella/
# CliLazyHandlersResolvableTest.py). To get around this, we need to define a
# duplicate of the enum to be used by the CAPI model.
VplsVeFwdTgtStatusEnum = [
   'unknownPeStatus',
   'validPeStatus',
   'missingLocalLabelBlock',
   'missingRemoteLabelBlock',
   'overlappingRemoteLabelBlocks',
   'invalidRemoteLabelBlock',
   'duplicatedRemoteVeIds',
   'multipleRemoteVeIds',
   'missingLayer2InfoExtComm',
   'tooManyLayer2InfoExtComms',
   'encapTypeNotSupported',
   'invalidRemoteVeId',
   'featureNotSupported',
]

class BgpVplsAdGroupMember( DeferredModel ):
   nextHop = IpGenericAddress( help="Remote PE next hop" )

class BgpVplsAdGroup( DeferredModel ):
   l2VpnId = Int( help="L2 VPN identifier" )
   rd = Str( help="VPLS group route distinguisher" )
   importRts = List( valueType=int, help="Route targets to import" )
   exportRts = List( valueType=int, help="Route targets to export" )
   localPeAddr = Int( help="Local PE address", optional=True )
   remotePeers = Dict( valueType=BgpVplsAdGroupMember,
                       help="Remote peers indexed by PE address" )

class BgpVplsAdInstance( DeferredModel ):
   groups = Dict(
      valueType=BgpVplsAdGroup,
      help="BGP VPLS auto discovered groups indexed by group name" )

class BgpVplsAdInstances( DeferredModel ):
   instances = Dict(
      valueType=BgpVplsAdInstance,
      help="BGP VPLS auto discovered instances indexed by instance name" )

class BgpVplsVeGroupMember( DeferredModel ):
   vplsVeId = Int( help="VPLS Edge identifier" )
   encapLabel = Int( help="Encapsulation label", optional=True )
   decapLabel = Int( help="Decapsulation label", optional=True )
   error = Enum( values=VplsVeFwdTgtStatusEnum,
                 optional=True, help="Remote peer negotiation error" )

class BgpVplsVeGroupRemoteVeId( DeferredModel ):
   remotePeers = Dict( valueType=BgpVplsVeGroupMember,
                       help="Remote peers indexed by PE address" )

class BgpVplsVeGroup( DeferredModel ):
   rd = Str( help="VPLS group route distinguisher" )
   importRts = List( valueType=int, help="Route targets to import" )
   exportRts = List( valueType=int, help="Route targets to export" )
   rcfImports = Str( optional=True, help="RCF function to apply on import" )
   rcfExports = Str( optional=True, help="RCF function to apply on export" )
   vplsVeIdLocal = Int( help="Local VPLS Edge identifier" )
   vplsVeIdsRemote = Dict( keyType=int,
                          valueType=BgpVplsVeGroupRemoteVeId,
                          help="Remote VE-IDs advertised by the remote peers" )

class BgpVplsVeInstance( DeferredModel ):
   groups = Dict(
      valueType=BgpVplsVeGroup,
      help="BGP VPLS signaling groups indexed by group name" )

class BgpVplsVeInstances( DeferredModel ):
   instances = Dict(
      valueType=BgpVplsVeInstance,
      help="BGP VPLS signaling instances indexed by instance name" )

class BgpVplsRouteKeyDetail( DeferredModel ):
   if toggleVplsBgpSignalingEnabled():
      nlriType = Enum( values=( "autoDiscovery", "signaling", ), help="NLRI type" )
      vplsVeId = Int( optional=True, help="VPLS Edge identifier" )
      vplsVeBlockOffset = Int(
         optional=True,
         help="The VE ID of the first PE that can use a label from the block" )
   else:
      nlriType = Enum( values=( "autoDiscovery", ), help="NLRI type" )
   rd = Str( optional=True, help="Route distinguisher" )
   peAddr = Int( optional=True, help="PE address" )

class BgpVplsRouteDetailEntry( BgpVpnRouteDetailEntry ):
   vplsVeLabelBase = Int( optional=True,
                          help="The first label in the VPLS VE block" )
   vplsVeBlockSize = Int( optional=True,
                          help="The number of labels in the VPLS VE block" )
   rxPathId = Int( optional=True, help='Received path ID of this route' )
   txPathId = Int( optional=True, help='Advertised path ID of this route' )

class BgpVplsPath( DeferredModel ):
   asPathEntry = Submodel( valueType=BgpRouteASPathEntry,
                           help="AS path information" )
   # TODO: BUG982065
   # Currently the nextHop attribute will be missing if the path is policy rejected.
   # However we should update this model to have a new attribute that explicitly
   # shows if a path was rejected by policy.
   nextHop = IpGenericAddress( optional=True, help="Next hop address" )
   med = Int( optional=True, help="Multi exit discriminator" )
   localPreference = Int( optional=True, help="Local preference" )
   weight = Int( optional=True, help="Weight" )
   tag = Int( optional=True, help="Tag" )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received" )

   routeType = Submodel( valueType=BgpVpnRouteTypeEntry, help="Route type" )
   routeDetail = Submodel( valueType=BgpVplsRouteDetailEntry, optional=True,
                           help="Route details" )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help="Reason route was not selected as BGP best path" )

class BgpVplsRoute( DeferredModel ):
   routeKeyDetail = Submodel( valueType=BgpVplsRouteKeyDetail, help="NLRI details" )
   totalPaths = Int( help="Total number of paths for this route" )
   paths = List( valueType=BgpVplsPath, help="BGP VPLS route paths" )

class BgpVplsRoutes( DeferredModel ):
   vrf = Str( help="VRF name" )
   routerId = Ip4Address( help="BGP Router Identity" )
   asn = Int( help="Autonomous System Number" )
   routes = Dict( valueType=BgpVplsRoute,
                  help="BGP VPLS routes indexed by route key" )
