# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
import CliDynamicSymbol
import CliGlobal
from dataclasses import dataclass
import LazyMount
import re
import Tracing

gv = CliGlobal.CliGlobal(
   hwLedConfig=None,
   intfToLedMapping=None,
   allPortStatus=None,
)

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

HardwareLedSettings = \
   CliDynamicSymbol.LazyCallback( "ShowHardwareLedModel.HardwareLedSettings" )
HardwareLedSetting = \
   CliDynamicSymbol.LazyCallback( "ShowHardwareLedModel.HardwareLedSetting" )
VirtualLed = \
   CliDynamicSymbol.LazyCallback( "ShowHardwareLedModel.VirtualLed" )

@dataclass
class DisplayLed:
   name: str
   color: str

class HwLedInfo:
   slot: int
   ledNum: int
   seq: list[ DisplayLed ]
   status: str

def showHardwareLedHandler( mode, args ):
   hwLeds = gv.hwLedConfig.leds
   if not hwLeds:
      mode.addError( "No LEDs found" )
      return None

   # The `ledInfoMap` will contain all of the information necessary to display a
   # hardware LED and all of its corresponding virtual LEDs. For every hardware
   # LED, there is at least one virtual/display LED. The key is the hardware LED's
   # name as it is in the hardware led config table in Sysdb. All the handler needs
   # to do is populate this map with the relevant information, and then the Model
   # will use that information for its display.
   ledInfoMap: dict[ str, HwLedInfo ] = {}

   # STEP 1: Create mapping from LED name -> associated interfaces. Note that this
   # will include both active interfaces and inactive/subsumed interfaces.
   intfToLed = gv.intfToLedMapping.intfToLedName
   ledToIntfs: dict[ str, list[ str ] ] = {}
   for ( intf, led ) in intfToLed.items():
      led = internalLedToExternal( led )
      if led not in ledToIntfs:
         ledToIntfs[ led ] = []
      ledToIntfs[ led ].append( intf )

   # STEP 2: Populate the table with its keys, as well as each hardware LED's slot
   # and LED number. Additionally, create a list of all of the hardware LEDs, which
   # will then be sorted. This is useful when all of the interfaces an LED is
   # associated with are subsumed so that we can backtrack and find the most recent
   # unsubsumed interface.
   allLedsList: list[ str ] = []
   for led in hwLeds:
      allLedsList.append( led )
      ledInfoMap[ led ] = HwLedInfo()
      slot, ledNum = getSlotAndLed( led, ledToIntfs )
      ledInfoMap[ led ].slot = slot
      ledInfoMap[ led ].ledNum = ledNum
      ledInfoMap[ led ].seq = []

   allLedsList = ArPyUtils.naturalsorted( allLedsList )

   # STEP 3: Populate the table with its associated virtual LEDs. For non-interfaces,
   # this will be a single LED. For interfaces, it can be either a single LED

   # Check for whether there is a status before dereferencing.
   # Then, make sure the interface is not subsumed. Note that status.active is
   # equivalent to "not subsumed", meaning that if active, the interface is not
   # subsumed and is displayed in commands such as `show interface status`.
   #
   # The variable `enabledStateReason`, inside of
   # interface/config/eth/phy/all/intfConfig, is (at time of implementation)
   # always "inactive" iff the interface is subsumed, and so it could also be
   # used as part of the check. As this variable and status.active are bijective,
   # there is no point in doing so at this time.
   #
   # For a table of interface states and the values of relevant variables to
   # determine subsumation status, see r/488956.
   for i, led in enumerate( allLedsList ):
      config = hwLeds[ led ]
      if intfs := ledToIntfs.get( led ):
         # Case: The LED is associated with interfaces.
         if ( not config.lightSettingDir or
            len( config.lightSettingDir.lightSetting ) == 0 ):
            # Case 1/2: The LightSettingDir is not used, so default to
            # the LightSetting.
            color = getColor( config.lightSetting )
            disp = DisplayLed( name=led, color=color )
            ledInfoMap[ led ].seq.append( disp )
         else:
            # Case 2/2: The LightSettingDir is used. If all of the interfaces are
            # in-active/subsumed, then the virtual LED name is that of the previous
            # interface. All non-subsumed interfaces will otherwise be in the
            # sequence list.
            intfs = ArPyUtils.naturalsorted( intfs )
            idx = 0
            for intf in intfs:
               intfStatus = gv.allPortStatus.intfStatus.get( intf )
               if not intfStatus or not intfStatus.active:
                  continue
               color = getColor( config.lightSettingDir.lightSetting[ idx ] )
               disp = DisplayLed( name=intf, color=color )
               ledInfoMap[ led ].seq.append( disp )
               idx = idx + 1
            if not ledInfoMap[ led ].seq:
               # If nothing has been added, then all interfaces must have been
               # subsumed. Using the sorted list of all LEDs, find the first LED
               # which was not subsumed. It is looking like we may need to iterate
               # over the list instead of the map.

               # If it is guaranteed that at least one interface per port is active,
               # and that the interface grouping is properly aligned, then it is
               # true that the active interface must be on an LED, because otherwise
               # the subsumation would not be aligned.
               ledIdx = i
               prevIntf = led
               intfStatus = gv.allPortStatus.intfStatus.get( prevIntf )
               while not intfStatus.active:
                  ledIdx -= 1
                  prevIntf = allLedsList[ ledIdx ]
                  intfStatus = gv.allPortStatus.intfStatus.get( prevIntf )
               color = "n/a"
               disp = DisplayLed( name=prevIntf, color=color )
               ledInfoMap[ led ].seq.append( disp )

         # Use sequence of virtual LED states to determine overall LED status
         seq = ledInfoMap[ led ].seq
         if all( vled.color == "n/a" for vled in seq ):
            ledInfoMap[ led ].status = "off"
         # Solid: All virtual LEDs are the same, or first is on and rest are off,
         #        and beacon is disabled.
         elif all( vled.color == seq[ 0 ].color for vled in seq ) or \
              all( vled.color == "n/a" for vled in seq[ 1 : ] ) and \
              ( not config.lightSetting.flashRate and
                not config.lightSettingDir.beacon ):
            ledInfoMap[ led ].status = "solid"
         else:
            ledInfoMap[ led ].status = "flashing"
      else:
         # Case: The LED is *not* associated with interfaces.
         color = getColor( config.lightSetting )
         disp = DisplayLed( name=led, color=color )
         ledInfoMap[ led ].seq.append( disp )
         ledInfoMap[ led ].status = "off" if color == "n/a" else "solid"

   # For each physical LED, populate the corresponding Model in
   # "ShowHardwareLedModel.py". Only add the components which are relevant.
   result = HardwareLedSettings()
   desired = args.get( "NAME" )
   if desired:
      desired = desired.lower()
   for led, ledInfo in ledInfoMap.items():
      if desired and not noncontiguousSubstring( desired, led.lower() ):
         continue
      entry = HardwareLedSetting()
      entry.slot = ledInfo.slot
      entry.led = ledInfo.ledNum
      entry.status = ledInfo.status

      isSolid = all( ( virt.color in ( "n/a", ledInfo.seq[ 0 ].color ) )
                     for virt in ledInfo.seq )

      for virt in ledInfo.seq:
         vled = VirtualLed()
         vled.name = virt.name
         if isSolid:
            # Necessary in case of GXXX case (Green, Off, Off, Off). We want to
            # make each virtual LED the actual color it is supposed to be, not the
            # LedPolicy stuff.
            vled.color = ledInfo.seq[ 0 ].color
         else:
            vled.color = virt.color
         entry.seq.append( vled )

      result.systemLeds[ led ] = entry

   return result

# -----------------------------------------------------------------------------
# Helper functions
# -----------------------------------------------------------------------------

# TODO: Currently, the internal scheme of naming LEDs associated with interfaces
# is to have <intf type><slot>-<# of LED, from 1 to N>. So if Ethernet37 had two
# LEDs, they would be Ethernet37-1 and Ethernet37-2. However, externally, the LED
# number depends on the number of lanes per LED. So if Ethernet37 had eight lanes,
# the LEDs are named Ethernet37/1 and Ethernet37/5. Note that there is also a change
# from a dash to a slash. Although it works at the moment, this second system does
# not scale well--with 16 lanes and 4 LEDs, it would produce LED numbers of 1, 5, 9,
# and 13, which isn't particularly clear. It is also confusing when the number of
# lanes per LED changes.
#
# There is a plan to change the style of the LED names that are published to the
# internal scheme, but until that happens, there needs to be a method that translates
# from one naming scheme to the other.
def internalLedToExternal( led: str ) -> str:
   '''Converts the internal LED naming scheme to the external scheme. E.g. converts
   Ethernet3/1 to Ethernet3-1 and Ethernet4/5 to Ethernet4-2.'''
   if led.endswith( "-1" ):
      led = led[ : -2 ] + "/1"
   elif led.endswith( "-2" ):
      led = led[ : -2 ] + "/5"
   elif led.endswith( "-3" ):
      led = led[ : -2 ] + "/9"
   elif led.endswith( "-4" ):
      led = led[ : -2 ] + "/13"
   else:
      t0( "Warning: found unexpected internal LED name." )
   return led

def noncontiguousSubstring( pattern: str, name: str ) -> bool:
   i, j = 0, 0
   while i < len( pattern ) and j < len( name ):
      if pattern[ i ] == name[ j ]:
         i += 1
      j += 1
   return i == len( pattern )

def getColor( lightSetting ) -> str:
   '''From the LightSetting, determine the color of the LED. Returns a string.'''
   ls = lightSetting
   if ls.red and ls.green and ls.blue:
      color = "white"
   elif ( ls.red and ls.green ) or ls.yellow:
      color = "amber"
   elif ls.red and ls.blue:
      color = "purple"
   elif ls.red:
      color = "red"
   elif ls.green:
      color = "green"
   elif ls.blue:
      color = "blue"
   else:
      color = "n/a"
   return color

def getSlotAndLed( ledName: str, ledToIntfs ) -> tuple[ int, int ]:
   '''From the LED name, determine its slot and LED number. Returns a tuple of
   two integers.'''
   # Cases:
   # - old Ethernet ports (e.g. Eth3/4, Eth6/1/1)
   #     -> regex for slot, intfToLedName for LED
   #     -> r"(\d+)\/(\d+)$"
   # - Fans (e.g. Fan3/2, Fan7)
   #     -> Slot/LED = (X, Y) or (X, 1) respectively
   #     -> r"Fan(\d+)"
   # - Linecards (e.g. Linecard6)
   #     -> regex for slot, 1 for LED
   #     -> r"Linecard(\d+)"
   # - MultiFanStatusX
   #     -> n/a for slot, 1 for LED
   #     -> did not match prev
   # - MultiPowerSupplyX
   #     -> n/a for slot, 1 for LED
   #     -> did not match prev
   intfReg = re.compile( r"(\d+)-(\d+)$" )
   fanXYReg = re.compile( r"^Fan(\d+)\/(\d+)" )
   fanXReg = re.compile( r"^Fan(\d+)" )
   linecardReg = re.compile( r"^Linecard(\d+)" )
   lastIntReg = re.compile( r"(\d+)$" )
   slot = "-1"
   led = "1"
   if ledName in ledToIntfs:
      arbIntf = ledToIntfs[ ledName ][ 0 ]
      ledRealName = gv.intfToLedMapping.intfToLedName[ arbIntf ]
      # Real name is in style "Ethernet47-2" or "Ethernet20"
      if match := intfReg.search( ledRealName ):
         slot, led = match.group( 1 ), match.group( 2 )
      elif match := lastIntReg.search( ledRealName ):
         slot = match.group( 1 )
   elif match := fanXYReg.search( ledName ):
      slot, led = match.group( 1 ), match.group( 2 )
   elif match := fanXReg.search( ledName ):
      slot = match.group( 1 )
   elif match := linecardReg.search( ledName ):
      slot = match.group( 1 )

   # Need to cast from string back to integer.
   return int( slot ), int( led )

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.hwLedConfig = LazyMount.mount( entityManager, "hardware/led/config",
                                  "Hardware::Led::LedSystemConfigDir", "r" )
   gv.intfToLedMapping = LazyMount.mount( entityManager,
                                       "hardware/led/intfToLedMapping",
                                       "Hardware::Led::IntfToLedMapping", "r" )
   gv.allPortStatus = LazyMount.mount( entityManager, "interface/status/eth/phy/all",
                                 "Interface::AllEthPhyIntfStatusDir", "r" )
