#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
from CliModel import (
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
)
import TableOutput

class VirtualLed( Model ):
   name = Str( help="name of the led getting printed" )
   color = Enum( values=[ "n/a", "red", "green", "blue",
                          "amber", "purple", "white" ],
                 help="the physical color of the LED" )

class HardwareLedSetting( Model ):
   slot = Int( help="slot location of LED" )
   led = Int( help="position of the LED on a slot" )
   seq = List( valueType=VirtualLed,
               help="the LEDs to be displayed, in order (if necessary) of flashing" )
   status = Enum( values=[ "off", "solid", "flashing" ],
                  help="status of the LED" )

class HardwareLedSettings( Model ):
   systemLeds = Dict( keyType=str, valueType=HardwareLedSetting,
                      help="A mapping of system LED to its settings." )

   def render( self ):
      headers = ( "Name", "Slot", "LED", "Color", "Status", "Sequence" )

      table = TableOutput.createTable( headers )
      formatNameColumn = TableOutput.Format( justify="left" )
      formatSlotColumn = TableOutput.Format( justify="right" )
      formatLedColumn = TableOutput.Format( justify="right" )
      formatColorColumn = TableOutput.Format( justify="left" )
      formatStatusColumn = TableOutput.Format( justify="left" )
      formatSequenceColumn = TableOutput.Format( justify="left" )
      cols = [ formatNameColumn, formatSlotColumn, formatLedColumn,
               formatColorColumn, formatStatusColumn, formatSequenceColumn ]

      # Remove additional column paddings
      for col in cols:
         col.padLimitIs( True )

      table.formatColumns( *cols )

      for ledName in ArPyUtils.naturalsorted( self.systemLeds ):
         setting = self.systemLeds[ ledName ]
         slot = str( setting.slot ) if setting.slot != -1 else "n/a"
         led = setting.led
         status = setting.status

         if len( setting.seq ) == 1:
            name = setting.seq[ 0 ].name
            color = setting.seq[ 0 ].color
            table.newRow( name, slot, str( led ),
                          color, status, "n/a" )
         else:
            for i, virt in enumerate( setting.seq, 1 ):
               if status == "flashing":
                  seq = f"flash {i}"
               else:
                  seq = "n/a"
               table.newRow( virt.name,
                             slot,
                             str( led ),
                             virt.color,
                             status,
                             seq )

      print( table.output() )
