#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import datetime

import CliModel

class SnmpDebugHashedIdentModel( CliModel.Model ):
   hashedIdentifiers = CliModel.Dict(
      keyType=str,
      valueType=str,
      help="A mapping of actual identifier to hashed identifier" )

   def render( self ):
      hashedHeader = "Hashed Identifier"
      print( f"{hashedHeader:40} Actual Identifier" )
      for actual, hashed in self.hashedIdentifiers.items():
         print( f"{hashed:40} {actual}" )

class SnmpSentNotificationVariable( CliModel.Model ):
   oid = CliModel.Str( help='The OID of the variable' )
   value = CliModel.Str( help='The value corresponding to the oid' )

class SnmpSentNotification( CliModel.Model ):
   oid = CliModel.Str( help='The notification that was sent' )
   when = CliModel.Float( help='When the notification was sent (UTC Seconds)' )
   variables = CliModel.List( valueType=SnmpSentNotificationVariable, help="A list "
         "of variables for this notification" )

class SnmpSentNotificationsModel( CliModel.Model ):
   _snmpEnabled = CliModel.Bool( help='SNMP is enabled' )
   notifications = CliModel.List( valueType=SnmpSentNotification, help="A list of "
         "notifications that were sent" )

   def render( self ):
      if not self._snmpEnabled:
         # snmpEnabled() has already printed an error
         return

      print( "Sent date/time       Notification" )
      print( "   variable = value" )
      print()
      for notification in self.notifications:
         fmt = datetime.datetime.fromtimestamp( notification.when ).strftime(
                                                               '%Y-%m-%d %H:%M:%S' )
         # pylint: disable-next=consider-using-f-string
         print( "%-20s %s" % ( fmt, notification.oid ) )
         for variable in notification.variables:
            # pylint: disable-next=consider-using-f-string
            print( "   %s = %s" % ( variable.oid, variable.value ) )
         print()
