# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import os
import re
import sys
import copy

import BasicCli, HostnameCli
import CliCommand
import CliMatcher
from CliPlugin import AclCli
from CliPlugin import ConfigMgmtMode
from CliPlugin import FileCli
from CliPlugin.Ssh import HostKeyAlgoExpression
from CliPlugin.Ssh import HOSTKEY_ALGORITHM_HIDDEN_ALIASES
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
from CliPlugin.AaaCli import sshkeyContentMatcher
import CliParser
import ConfigMount
import Tac
from socket import IPPROTO_TCP
import AclCliLib
import DscpCliLib
from CliMode.VrfConfig import VrfConfigMode
from CliMode.SshTunnel import SshTunnelMode
from CliMode.SshTunnel import SshTunnelVrfMode
from CliMode.SshUser import ( SshUserMode, SshKeyMode, SshPrincipalMode )
from CliMode.SshAuthenticationX509 import SshAuthenticationX509Mode
import LazyMount
import Logging
import SysMgrLib
import SshCertLib
from Url import UrlMatcher
import tempfile
from AaaPluginLib import primarySshKeyId
from SshAlgorithms import CIPHERS, SUPPORTED_CIPHERS, FIPS_CIPHERS, \
   MACS, SUPPORTED_MACS, FIPS_MACS, KEXS, SUPPORTED_KEXS, FIPS_KEXS, \
   HOSTKEYS, SUPPORTED_HOSTKEYS, FIPS_HOSTKEYS
from SysMgrLib import authenCliTokenToTacType, updateAuthenticationMethods

# ------------------------------
# Ssh config commands
#
# From config mode
#    management ssh    - enter ssh config mode
#    idle-timeout x - set the idle session timeout to x minutes
#    authentication mode [ password | keyboard-interactive ]
#          - make sshd enter specified mode, mutually exclusive with the
#            authentication protocol command and deprecated in favor of it
#    authentication protocol
#            ( { password | keyboard-interactive | public-key } )
#              |
#            ( { multi-factor ( password | keyboard-interactive | public-key ) } )
#          - enables the specified methods of authentication
#    ciphers SSH_CIPHER1 ... SSH_CIPHERN - set SSH exclusive cipher list
#    [ no | default ] ciphers - clear filter and use all supported ciphers
#    [ no | default ] authentication - return sshd to keyboard-interactive mode
#    [ no | default ] shutdown - disables or enables sshd
#    [ no | default ] qos dscp <0-63> - set dscp value in IP header
#    vrf VRF - enter ssh vrf config mode
#
# From ssh vrf config mode
#    [ no | default ] shutdown - disables or enables sshd in vrf
#
#    For example:
#    management ssh
#       A
#       vrf x
#          B
#
#    A\B             B default             B shutdown            B no shutdown
#    A default       enabled\enabled       enabled\disabled      enabled\enabled
#    A shutdown      disabled\disabled     disabled\disabled     disabled\enabled
#    A no shutdown   enabled\enabled       enabled\disabled      enabled\enabled

REKEYUNITS = {
      "kbytes" : "Kilobyte(s)",
      "mbytes" : "Megabyte(s)",
      "gbytes" : "Gigabyte(s)",
}

REKEYTIMEUNITS = {
   "seconds": "Second(s)",
   "minutes": "Minute(s)",
   "hours": "Hour(s)",
   "days": "Day(s)",
   "weeks": "Week(s)",
}

TCPFORWARDINGSETTINGS = {
   "all" : "Allow TCP forwarding for both local and remote",
   "local" : "Allow local forwarding only",
   "remote" : "Allow remote forwarding only",
}

serverPort = Tac.Type( "Mgmt::Ssh::ServerPort" )
sshTunnel = Tac.newInstance( "Mgmt::Ssh::TunnelConfig", "arastra" )
SshConstants = Tac.Type( "Mgmt::Ssh::Constants" )
AuthenMethod = Tac.Type( "Mgmt::Ssh::AuthenMethod" )
KeyType = Tac.Type( "Mgmt::Ssh::KeyType" )

SECURITY_SSH_TUNNEL_CONFIGURED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_CONFIGURED",
              severity=Logging.logInfo,
              fmt="SSH tunnel %s from local TCP port %s to "
              "%s:%s via %s@%s is fully configured.",
              explanation="An SSH tunnel was just configured and "
              "will try to connect.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SECURITY_SSH_TUNNEL_UNCONFIGURED = Logging.LogHandle(
              "SECURITY_SSH_TUNNEL_UNCONFIGURED",
              severity=Logging.logInfo,
              fmt="SSH tunnel %s from local TCP port %s to %s:%s via %s@%s"
              " has been unconfigured.",
              explanation="An SSH tunnel to a remote server just stopped running"
              " and will not be automatically restarted since it was terminated"
              " by a command.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

sshConfig = None
localUserConfig = None
aclConfig = None
aclCpConfig = None
dscpConfig = None
sslConfig = None

sshShowKwMatcher = CliMatcher.KeywordMatcher( 'ssh',
      helpdesc='Show SSH status and statistics' )
hostKeyMatcher = CliCommand.setCliExpression(
      data={ kw: helpstr
             for kw, helpstr in HOSTKEYS.items()
             if kw in SUPPORTED_HOSTKEYS },
      hiddenData=HOSTKEY_ALGORITHM_HIDDEN_ALIASES,
      name='HOSTKEYALGOS' )
hostkeyKwMatcher = CliMatcher.KeywordMatcher( 'hostkey',
                                     helpdesc='Set SSH hostkey related options' )
showKeyKwMatcher = CliMatcher.KeywordMatcher( 'key', alternates=[ 'hostkey' ],
                                     helpdesc='Show sshd key information' )
authPrincipalsCmdMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir(
                               SshConstants.authPrincipalsCmdDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='Authorized principals command filename',
                            helpname='WORD' )
trustedCaMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir( SshConstants.caKeysDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='CA public key filename',
                            helpname='WORD' )
hostCertMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir( SshConstants.hostCertsDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='Switch\'s SSH hostkey cert filename',
                            helpname='WORD' )
revokeListMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir(
                                             SshConstants.revokeListsDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='Revoked SSH user keys filename',
                            helpname='WORD' )
thousandIntMatcher = CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Set Value' )
connectionKwMatcher = CliMatcher.KeywordMatcher( 'connection',
                                          helpdesc='Settings for SSH connections' )
authKwMatcher = CliMatcher.KeywordMatcher( 'authentication',
                                          helpdesc='Change authentication settings' )

SshOptions = Tac.Type( "Mgmt::Ssh::Options" )
# ssh-options which will be used with ssh-keys and ssh-principals
SSHOPTIONS = {
      SshOptions.agentForwarding: "Enable authentication agent forwarding",
      SshOptions.certAuthority: "Listed key is a certification authority",
      SshOptions.command: "Execute \"command\" during authentication",
      SshOptions.environment: "Add provided string to the environment during " +
         "authentication",
      SshOptions.expiryTime: "Expiry time of the key",
      SshOptions.patternFrom: "Specify PATTERNS, see ssh_config for more details",
      SshOptions.noAgentForwarding: "Forbid agent forwarding",
      SshOptions.noPortForwarding: "Forbid TCP forwarding",
      SshOptions.noPty: "Prevent TTY allocation",
      SshOptions.noUserRc: "Prevent execution on ~/.ssh/rc",
      SshOptions.noX11Forwarding: "Forbid X11 forwarding",
      SshOptions.permitListen: "Permit listening only on specified host and port",
      SshOptions.permitOpen: "Connect only to specified host and port",
      SshOptions.portForwarding: "Enable port forwarding",
      SshOptions.principals: "Specify allowed principals for certificate " +
         "authentication",
      SshOptions.pty: "Permit PTY allocation",
      SshOptions.noTouchRequired: "Demonstration of user presence not required",
      SshOptions.verifyRequired: "User needs to verify signatures made with this " +
         "key",
      SshOptions.restrict: "Enable all restrictions",
      SshOptions.tunnel: "Force a tun device on the server",
      SshOptions.userRc: "Enable execution of ~/.ssh/rc",
      SshOptions.x11Forwarding: "Permit X11 forwarding",
}

def _listdir( path ):
   try:
      return os.listdir( path )
   except OSError:
      return []

class SshTunnelConfigModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self, mode )
      self.config()

   def config( self ):
      holdingConfig = sshConfig
      if self.mode.vrfConfigName_ and self.mode.vrfConfigName_ != DEFAULT_VRF:
         holdingConfig = sshConfig.vrfConfig.newMember( self.mode.vrfConfigName_ )
      tunnel = holdingConfig.tunnel.newMember( self.mode.tunnelName_ )
      if not self.mode.vrfConfigName_:
         tunnel.configuredInSshMode = True
      return tunnel

   def setLocalPort( self, args ):
      self.config().localPort = args[ 'PORT' ]

   def defaultLocalPort( self, args ):
      self.config().localPort = serverPort.invalid

   def setRemote( self, args ):
      self.config().remoteHost = args[ 'REMOTE_HOST' ]
      self.config().remotePort = args[ 'REMOTE_PORT' ]

   def defaultRemote( self, args ):
      self.config().remoteHost = ""
      self.config().remotePort = serverPort.invalid

   def setSshServer( self, args ):
      # Do a sanity check on the hostname or IP address that the user
      # entered. If it doesn't appear to be legal, print a warning,
      # but don't reject the entry. (The specified hostname may not
      # yet have been configured in DNS, for example.)
      sshServerAddress = args[ 'ADDR' ]
      sshServerUsername = args[ 'USER' ]
      sshServerPort = args.get( 'PORT' )
      HostnameCli.resolveHostname( self.mode, sshServerAddress,
                                   doWarn=True )
      self.config().sshServerAddress = sshServerAddress
      self.config().sshServerUsername = sshServerUsername
      if sshServerPort:
         self.config().sshServerPort = sshServerPort
      else:
         self.config().sshServerPort = serverPort.defaultPort

   def defaultSshServer( self, args ):
      self.config().sshServerAddress = ""
      self.config().sshServerUsername = ""
      self.config().sshServerPort = serverPort.invalid

   def setServerAliveInterval( self, args ):
      self.config().serverAliveInterval = args[ 'SECONDS' ]

   def defaultServerAliveInterval( self, args ):
      self.config().serverAliveInterval = sshTunnel.serverAliveIntervalDefault

   def setServerAliveMaxLost( self, args ):
      self.config().serverAliveMaxLost = args[ 'SECONDS' ]

   def defaultServerAliveMaxLost( self, args ):
      self.config().serverAliveMaxLost = sshTunnel.serverAliveMaxLostDefault

   def setShutdown( self, args=None ):
      Logging.log( SECURITY_SSH_TUNNEL_UNCONFIGURED,
                   self.config().name,
                   str( self.config().localPort ),
                   self.config().remoteHost,
                   str( self.config().remotePort ),
                   self.config().sshServerUsername,
                   self.config().sshServerAddress )
      self.config().enable = False

   def disableShutdown( self, args=None ):
      Logging.log( SECURITY_SSH_TUNNEL_CONFIGURED,
                   self.config().name,
                   str( self.config().localPort ),
                   self.config().remoteHost,
                   str( self.config().remotePort ),
                   self.config().sshServerUsername,
                   self.config().sshServerAddress )
      self.config().enable = True

   def setUnlimitedRestarts( self, args ):
      self.config().unlimitedRestarts = True

   def defaultUnlimitedRestarts( self, args ):
      self.config().unlimitedRestarts = False

#-----------------------------------------------------------------------------------
# [ no | default ] local port PORT
#-----------------------------------------------------------------------------------
class LocalPort( CliCommand.CliCommandClass ):
   syntax = 'local port PORT'
   noOrDefaultSyntax = 'local port ...'
   data = {
            'local': 'Configure the local port for the tunnel',
            'port': 'Set the port to use',
            'PORT': CliMatcher.IntegerMatcher( serverPort.min, serverPort.max,
                                               helpdesc='Number of the port to '
                                                        'use' ),
          }
   handler = SshTunnelConfigModelet.setLocalPort
   noOrDefaultHandler = SshTunnelConfigModelet.defaultLocalPort

SshTunnelConfigModelet.addCommandClass( LocalPort )

#-----------------------------------------------------------------------------------
# [ no | default ] server-alive interval SECONDS
#-----------------------------------------------------------------------------------
class ServerAliveInterval( CliCommand.CliCommandClass ):
   syntax = 'server-alive interval SECONDS'
   noOrDefaultSyntax = 'server-alive interval ...'
   data = {
            'server-alive': 'Set SSH ServerAlive options',
            'interval': 'Time period ( in seconds ) to send SSH keep-alive packets',
            'SECONDS': thousandIntMatcher
          }
   handler = SshTunnelConfigModelet.setServerAliveInterval
   noOrDefaultHandler = SshTunnelConfigModelet.defaultServerAliveInterval

SshTunnelConfigModelet.addCommandClass( ServerAliveInterval )

#-----------------------------------------------------------------------------------
# [ no | default ] server-alive count-max SECONDS
#-----------------------------------------------------------------------------------
class ServerAliveCountMax( CliCommand.CliCommandClass ):
   syntax = 'server-alive count-max SECONDS'
   noOrDefaultSyntax = 'server-alive count-max ...'
   data = {
            'server-alive': 'Set SSH ServerAlive options',
            'count-max': ( 'Number of SSH keep-alive packets that can be lost '
                           'before the connection is assumed dead' ),
            'SECONDS': thousandIntMatcher
          }
   handler = SshTunnelConfigModelet.setServerAliveMaxLost
   noOrDefaultHandler = SshTunnelConfigModelet.defaultServerAliveMaxLost

SshTunnelConfigModelet.addCommandClass( ServerAliveCountMax )

#-----------------------------------------------------------------------------------
# [ no | default ] remote host REMOTE_HOST port REMOTE_PORT
#-----------------------------------------------------------------------------------
class TunnelRemoteHost( CliCommand.CliCommandClass ):
   syntax = 'remote host REMOTE_HOST port REMOTE_PORT'
   noOrDefaultSyntax = 'remote ...'
   data = {
            'remote': 'Configure the remote options for the tunnel',
            'host': 'Set the host to direct the tunnel to',
            'REMOTE_HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'port': 'Set the port to use',
            'REMOTE_PORT': CliMatcher.IntegerMatcher( serverPort.min,
                                                      serverPort.max,
                                                      helpdesc='Number of the port '
                                                               'to use' ),
          }
   handler = SshTunnelConfigModelet.setRemote
   noOrDefaultHandler = SshTunnelConfigModelet.defaultRemote

SshTunnelConfigModelet.addCommandClass( TunnelRemoteHost )

#-----------------------------------------------------------------------------------
# [ no | default ] ssh-server ADDR user USER
#                  [ port PORT ]
#-----------------------------------------------------------------------------------
class SshServer( CliCommand.CliCommandClass ):
   syntax = 'ssh-server ADDR user USER [ port PORT ]'
   noOrDefaultSyntax = 'ssh-server ...'
   data = {
            'ssh-server': 'Configure the remote ssh-server to connect to',
            'ADDR': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'user': 'Username to login with',
            'USER': CliMatcher.PatternMatcher( '.+', helpname='WORD',
                                               helpdesc='Login name for ssh '
                                                        'server' ),
            'port': 'Port for SSH server (default 22)',
            'PORT': CliMatcher.IntegerMatcher( serverPort.min,
                                               serverPort.max,
                                               helpdesc='Number of the port to '
                                                        'use' ),
          }
   handler = SshTunnelConfigModelet.setSshServer
   noOrDefaultHandler = SshTunnelConfigModelet.defaultSshServer

SshTunnelConfigModelet.addCommandClass( SshServer )

#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class TunnelShutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   defaultSyntax = 'shutdown'
   noSyntax = 'shutdown'
   data = {
            'shutdown': 'Disable SSH Tunnel',
          }
   handler = SshTunnelConfigModelet.setShutdown
   defaultHandler = SshTunnelConfigModelet.setShutdown
   noHandler = SshTunnelConfigModelet.disableShutdown

SshTunnelConfigModelet.addCommandClass( TunnelShutdown )

#-----------------------------------------------------------------------------------
# [ no | default ] unlimited-restarts
#-----------------------------------------------------------------------------------
class TunnelUnlimitedRestarts( CliCommand.CliCommandClass ):
   syntax = 'unlimited-restarts'
   noOrDefaultSyntax = 'unlimited-restarts'
   data = {
            'unlimited-restarts': ( 'Ignore restart tokens and allow unlimited '
                                    'restarts' ),
          }
   handler = SshTunnelConfigModelet.setUnlimitedRestarts
   noOrDefaultHandler = SshTunnelConfigModelet.defaultUnlimitedRestarts
   hidden = True

SshTunnelConfigModelet.addCommandClass( TunnelUnlimitedRestarts )

class SshMainTunnelConfigMode( SshTunnelMode, BasicCli.ConfigModeBase ):
   name = "SSH Main Tunnel Configuration"

   def __init__( self, parent, session, tunnelName ):
      self.tunnelName_ = tunnelName
      self.vrfConfigName_ = None
      SshTunnelMode.__init__( self, tunnelName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SshMainTunnelConfigMode.addModelet( SshTunnelConfigModelet )

class SshVrfTunnelConfigMode( SshTunnelVrfMode, BasicCli.ConfigModeBase ):
   name = "SSH VRF Tunnel Configuration"

   def __init__( self, parent, session, tunnelName, vrfConfig ):
      vrfConfigName = vrfConfig.vrfName
      self.tunnelName_ = tunnelName
      self.vrfConfigName_ = vrfConfigName
      SshTunnelVrfMode.__init__( self, ( vrfConfigName, tunnelName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SshVrfTunnelConfigMode.addModelet( SshTunnelConfigModelet )

class SshUserConfigModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self, mode )
      # Ensure that the tunnel sysdb path exists
      self.config().userModeEntered = True

   def config( self ):
      return sshConfig.user.newMember( self.mode.userName_ )

   def setSshUserTcpForwarding( self, args ):
      self.config().userTcpForwarding = args[ 'FORWARDING_SETTING' ]

   def noSshUserTcpForwarding( self, args ):
      self.config().userTcpForwarding = self.config().userTcpForwardingDefault

#-----------------------------------------------------------------------------------
# [ no | default ] tcp forwarding SETTING in User submode
#-----------------------------------------------------------------------------------
class UserTcpForwardingSetting( CliCommand.CliCommandClass ):
   syntax = 'tcp forwarding FORWARDING_SETTING'
   noOrDefaultSyntax = 'tcp forwarding'

   data = {
            'tcp': 'Configure SSH server TCP settings for user',
            'forwarding': 'TCP Forwarding option',
            'FORWARDING_SETTING': CliMatcher.EnumMatcher( TCPFORWARDINGSETTINGS )
          }
   handler = SshUserConfigModelet.setSshUserTcpForwarding
   noOrDefaultHandler = SshUserConfigModelet.noSshUserTcpForwarding

SshUserConfigModelet.addCommandClass( UserTcpForwardingSetting )

class SshUserConfigMode( SshUserMode, BasicCli.ConfigModeBase ):
   name = "SSH User Specification Configuration"

   def __init__( self, parent, session, userName ):
      self.userName_ = userName
      SshUserMode.__init__( self, userName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def config( self ):
      return sshConfig.user.newMember( self.userName_ )

SshUserConfigMode.addModelet( SshUserConfigModelet )

class SshKeyConfigMode( SshKeyMode, BasicCli.ConfigModeBase ):
   name = "SSH Authorized Key Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, userName, keyName ):
      self.userName_ = userName
      self.keyName_ = keyName
      SshKeyMode.__init__( self, ( self.userName_, self.keyName_ ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config()

   def config( self ):
      userConfig = sshConfig.user[ self.userName_ ]
      return userConfig.sshAuthKeys.newMember( self.keyName_ )

class SshPrincipalConfigMode( SshPrincipalMode, BasicCli.ConfigModeBase ):
   name = "SSH Authorized Principal Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, userName, principal ):
      self.userName_ = userName
      self.principal_ = principal
      SshPrincipalMode.__init__( self, ( self.userName_, self.principal_ ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config()

   def config( self ):
      userConfig = sshConfig.user[ self.userName_ ]
      return userConfig.sshAuthPrincipals.newMember( self.principal_ )

class SshOptionsConfigModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      self.mode_ = mode
      CliParser.Modelet.__init__( self, mode )
      self.config()

   def config( self ):
      return self.mode_.config()

   def setSshOptions( self, args ):
      # Validate all ssh-options
      sshOption = args[ 'OPTION' ]
      sshValue = args.get( 'VALUE', '' )

      # Check the syntax of the options
      # The below options does not have values
      if ( sshOption in ( SshOptions.agentForwarding, SshOptions.certAuthority,
         SshOptions.noAgentForwarding, SshOptions.noPortForwarding,
         SshOptions.noPty, SshOptions.noUserRc, SshOptions.noX11Forwarding,
         SshOptions.portForwarding, SshOptions.pty, SshOptions.noTouchRequired,
         SshOptions.verifyRequired, SshOptions.restrict, SshOptions.userRc,
         SshOptions.x11Forwarding ) and sshValue != "" ):
         self.mode.addError( f"SSH option {sshOption} does not have a value" )
         return

      # The below options need values
      if ( sshOption in ( SshOptions.command, SshOptions.patternFrom,
         SshOptions.permitListen, SshOptions.permitOpen, SshOptions.principals,
         SshOptions.tunnel )
         and sshValue == "" ):
         self.mode.addError( f"SSH option {sshOption} needs a value" )
         return

      # The below options have values which can be validated
      # expiry-time YYYYMMDD or YYYYMMDDHHMM[SS]
      expiryTimeFormat = re.compile( r'^(\d){8}$|^(\d){12}$|^(\d){14}$' )
      if sshOption == SshOptions.expiryTime and not expiryTimeFormat.match(
            sshValue ):
         self.mode.addError( f"Invalid value {sshValue} for option expiry-time, "
                            "valid formats YYYYMMDD or YYYYMMDDHHMM[SS]" )
         return

      # environment NAME=VALUE
      environmentFormat = re.compile( r'(\S)+=(\S)+' )
      if sshOption == SshOptions.environment and not environmentFormat.match(
            sshValue ):
         self.mode.addError( f"Invalid format {sshValue} for option environment, "
                            "valid format \"NAME=VALUE\"" )
         return

      if sshOption == SshOptions.tunnel and not sshValue.isdigit():
         self.mode.addError( f"Invalid value {sshValue} for option tunnel,"
               " value must be a number" )
         return

      # Some options are mutually exclusive, ensure we don't have such violations
      mutuallyExclusiveOptions = [
         [ SshOptions.agentForwarding, SshOptions.noAgentForwarding ],
         [ SshOptions.portForwarding, SshOptions.noPortForwarding ],
         [ SshOptions.pty, SshOptions.noPty ],
         [ SshOptions.userRc, SshOptions.noUserRc ],
         [ SshOptions.x11Forwarding, SshOptions.noX11Forwarding ]
      ]

      for option in mutuallyExclusiveOptions:
         if ( ( sshOption == option[ 0 ] and
               option[ 1 ] in self.config().sshOptions ) or
            ( sshOption == option[ 1 ] and
             option[ 0 ] in self.config().sshOptions ) ):
            self.mode.addError( f"SSH option {option[0]} and {option[1]} "
                                "are mutually exclusive" )
            return

      # All validations done, update config
      optionEntity = Tac.Value( "Mgmt::Ssh::SshOptions", sshOption )

      # Only ssh-option "environment" can have multiple values so retain
      # previous values if present
      if ( sshOption == SshOptions.environment and
            sshOption in self.config().sshOptions ):
         for value in self.config().sshOptions[ sshOption ].sshValues:
            optionEntity.sshValues.add( value )

      if sshValue:
         optionEntity.sshValues.add( sshValue )

      self.config().sshOptions.addMember( optionEntity )

   def noSshOptions( self, args ):
      sshOption = args.get( 'OPTION', '' )
      sshValue = args.get( 'VALUE', '' )
      if sshOption and sshOption in self.config().sshOptions:
         # Check if value is specified and cleanup required
         optionConfig = self.config().sshOptions[ sshOption ]
         if ( sshValue and sshValue in optionConfig.sshValues and
               len( optionConfig.sshValues ) > 1 ):
            # Create a new entity
            newOption = Tac.Value( "Mgmt::Ssh::SshOptions", sshOption )
            # Copy existing values except the one to be deleted
            for value in optionConfig.sshValues:
               if value != sshValue:
                  newOption.sshValues.add( value )

            self.config().sshOptions.addMember( newOption )
            return

         # All values removed or value not specified, remove the sshOption
         del self.config().sshOptions[ sshOption ]

# -----------------------------------------------------------------------------------
# [ no | default ] option OPTION [ VALUE ] in key submode
# -----------------------------------------------------------------------------------
class SshOptionSetting( CliCommand.CliCommandClass ):
   syntax = 'option OPTION [ VALUE ]'
   noOrDefaultSyntax = 'option OPTION [ VALUE ]'
   data = {
            'option': 'Configure SSH-options for a specific SSH authorized key',
            'OPTION': CliMatcher.EnumMatcher( SSHOPTIONS ),
            'VALUE': CliMatcher.QuotedStringMatcher(
               pattern=r'[A-Za-z0-9_\-=*?.,!:\[\]\"/]+',
               helpdesc='SSH options value, multi-word values must be quoted',
               helpname='LINE' )
          }
   handler = SshOptionsConfigModelet.setSshOptions
   noOrDefaultHandler = SshOptionsConfigModelet.noSshOptions

SshOptionsConfigModelet.addCommandClass( SshOptionSetting )

def generalSetKnownHost( config, host, keyAlgo, publicKey, cliMode ):
   mappedKeyAlgo = SysMgrLib.cliKeyTypeToTac[ keyAlgo ]
   configuredInSshMode = cliMode == SshConfigMode
   config.addKnownHost( Tac.Value( "Mgmt::Ssh::KnownHost",
                                    host,
                                    mappedKeyAlgo,
                                    publicKey,
                                    configuredInSshMode ) )

def generalNoKnownHost( config, host ):
   del config.knownHost[ host ]

def validateSshAuthorizedKey( key ):
   ''' Use ssh-keygen to validate the entered key is a valid
   '''
   try:
      with tempfile.NamedTemporaryFile( mode='w' ) as f:
         f.write( key + '\n' )
         f.flush()
         try:
            Tac.run( [ '/usr/bin/ssh-keygen', '-l', '-f', f.name ],
                     stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         except Tac.SystemCommandError:
            return False
   except OSError:
      return False
   return True

def updatePublishedAuthKey( userConfig, deletedKey ):
   '''
      If the deleted key is the published key, then publish a new one
      if present.
      Note: The user config key collection should be accurate before
            calling this function. Any keys to be deleted should be deleted.
   '''
   if deletedKey == userConfig.publishedSshAuthKey:
      userConfig.publishedSshAuthKey = ""
      for key in userConfig.sshAuthKeys:
         candidateKey = userConfig.sshAuthKeys[ key ].keyContents
         if candidateKey != "":
            userConfig.publishedSshAuthKey = candidateKey
            break

#----------------------------------------------------------------
# SshConfigMode
#----------------------------------------------------------------
class SshConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = "Ssh configuration"

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "ssh" )
      self.aclConfig_ = aclConfig
      self.aclCpConfig_ = aclCpConfig
      self.dscpConfig_ = dscpConfig
      self.userName_ = None
      self.keyName_ = None

   def config( self ):
      return sshConfig

   def localUserConfig( self ):
      return localUserConfig

   def _setIdleTimeout( self, timeout ):
      to = Tac.Value( "Mgmt::SessionTimeout::IdleSessionTimeout" )
      to.timeout = timeout
      self.config().idleTimeout = to

   def setIdleTimeout( self, args ):
      self._setIdleTimeout( args[ 'IDLETIME' ] * 60 )

   def noIdleTimeout( self, args=None ):
      self._setIdleTimeout( self.config().idleTimeout.defaultTimeout )

   def setAuthenticationMode( self, args ):
      self.config().legacyAuthenticationModeSet = True
      self.config().authenticationMethodList.clear()

      newAuthenticationMethods = [ [ AuthenMethod.publicKey ] ]
      if "password" in args:
         newAuthenticationMethods.append( [ AuthenMethod.password ] )
      if "keyboard-interactive" in args:
         newAuthenticationMethods.append( [ AuthenMethod.keyboardInteractive ] )
      updateAuthenticationMethods( self.config(), newAuthenticationMethods )

   def defaultAuthenticationMode( self, args=None ):
      defaultArgs = { "keyboard-interactive": "keyboard-interactive" }
      self.setAuthenticationMode( defaultArgs )

   def setAuthenticationProtocol( self, args ):
      options = args.get( 'OPTIONS' )
      multiFactorEnabled = 'multi-factor' in args
      self.config().legacyAuthenticationModeSet = False

      if multiFactorEnabled:
         newMethods =\
         [ [ authenCliTokenToTacType[ opt ] for opt in optionSequence ]
          for optionSequence in options ]

      else:
         newMethods = [ [ authenCliTokenToTacType[ opt ] ] for opt in options[ 0 ] ]

      updateAuthenticationMethods( self.config(), newMethods )

   def defaultAuthenticationProtocol( self, args=None ):
      defaultAuthenticationMethods = [
         [ AuthenMethod.keyboardInteractive ],
         [ AuthenMethod.publicKey ] ]
      updateAuthenticationMethods( self.config(), defaultAuthenticationMethods )

   def gotoSshAuthenticationX509Mode( self, args ):
      """Handler for 'authentication x509'"""
      childMode = self.childMode( SshAuthenticationX509ConfigMode )
      self.session_.gotoChildMode( childMode )

   def noSshAuthenticationX509Mode( self, args ):
      """Handler for '( no | default ) authentication x509'"""
      sshConfig.x509ProfileName = ''
      sshConfig.x509UsernameDomainOmit = False

   def checkAlgorithms( self, requested, fipsCompliant ):
      """
      Compares the requested algorithms against the list of FIPS compliant
      counterparts and prints a warning message for algorithms that are
      not compliant.

      Only a warning message is printed and no further actions are taken.
      """
      if self.config().fipsRestrictions:
         for algo in requested:
            # Check if its a named key, if yes extract the algo field
            # /persist/secure/ssh/keys/<algo>/<file-name>
            if '/persist/secure/ssh/keys/' in algo:
               algo = SshCertLib.getAlgoFromKeyPath( algo )
            if algo not in fipsCompliant:
               self.addWarning(
                     f"{algo} is disabled when using FIPS algorithms." )

   def checkSupported( self, requested, supported ):
      for algo in requested:
         if algo not in supported:
            self.addWarning(
                  f"{algo} is not supported." )
   def setCiphers( self, args ):
      """
      Sets the cipher filter to allow only those in the iterable ciphers.
      """
      ciphers = args[ 'CIPHERS' ]
      self.checkAlgorithms( ciphers, FIPS_CIPHERS )
      self.checkSupported( ciphers, SUPPORTED_CIPHERS )
      self.config().cipher = ' '.join( ciphers )

   def noCiphers( self, args ):
      """
      Clear the SSH cipher filter list. Accept all ciphers again.
      """
      self.config().cipher = ''

   def defaultCiphers( self, args=None ):
      """
      Set cipher list to default safe ciphers.
      """
      self.config().cipher = self.config().cipherDefault

   def setKeyExchange( self, args ):
      """
      Sets the kex filter to allow only those in the iterable kex.
      """
      kexs = args[ 'KEX' ]
      self.checkAlgorithms( kexs, FIPS_KEXS )
      self.checkSupported( kexs, SUPPORTED_KEXS )
      self.config().kex = ' '.join( kexs )

   def noKeyExchange( self, args ):
      """
      Clear the SSH kex filter list. Accept all kex methods again.
      """
      self.config().kex = ''

   def defaultKeyExchange( self, args=None ):
      """
      Set the kex filter list to safe defaults.
      """
      self.config().kex = self.config().kexDefault

   def setMac( self, args ):
      """
      Sets the mac filter to allow only those in the iterable mac.
      """
      macs = args[ 'MAC' ]
      self.checkAlgorithms( macs, FIPS_MACS )
      self.checkSupported( macs, SUPPORTED_MACS )
      self.config().mac = ' '.join( macs )

   def noMac( self, args ):
      """
      Clear the SSH mac filter list. Accept all mac algorithms again.
      """
      self.config().mac = ''

   def defaultMac( self, args=None ):
      """
      Set the mac list to the default safe MACs
      """
      self.config().mac = self.config().macDefault

   def setKnownHost( self, args ):
      host = args[ 'HOST' ]
      keyAlgo = args[ 'KEYALGO' ]
      publicKey = args[ 'PUB_KEY' ]
      generalSetKnownHost( self.config(), host, keyAlgo, publicKey,
                           cliMode=SshConfigMode )

   def noKnownHost( self, args ):
      host = args[ 'HOST' ]
      knownHosts = self.config().knownHost
      if not host in knownHosts or not knownHosts[ host ].configuredInSshMode:
         return
      generalNoKnownHost( self.config(), host )

   def _setServerPort( self, port ):
      self.config().serverPort = port
      for aclType in ( 'ip', 'ipv6' ):
         for serviceAclVrfConfig in\
               self.aclCpConfig_.cpConfig[ aclType ].serviceAcl.values():
            serviceConfig = serviceAclVrfConfig.service.get( 'ssh' )
            if serviceConfig:
               serviceConfig.ports = str( port )
      self.updateDscpRules()

   def setServerPort( self, args ):
      self._setServerPort( args[ 'PORT' ] )

   def noServerPort ( self, args=None ):
      self._setServerPort( serverPort.defaultPort )

   def fipsRestrictions( self, args ):
      self.config().fipsRestrictions = True
      self.checkAlgorithms( self.config().cipher.split(), FIPS_CIPHERS )
      self.checkAlgorithms( self.config().mac.split(), FIPS_MACS )
      self.checkAlgorithms( self.config().kex.split(), FIPS_KEXS )
      self.checkAlgorithms( self.config().hostkey.split(), FIPS_HOSTKEYS )

   def noFipsRestrictions( self, args=None ):
      self.config().fipsRestrictions = False

   def setRekeyData( self, args ):
      """
      Set the rekey data limit.
      """
      amount = args[ 'AMOUNT' ]
      rekeyUnit = args[ 'REKEYUNIT' ]

      if not SysMgrLib.checkRekeyDataLimit( amount, rekeyUnit ):
         self.addError( "Rekey data frequency must be less than 4GB." )
         return
      self.config().rekeyDataAmount = amount
      self.config().rekeyDataUnit = rekeyUnit

   def defaultRekeyData( self, args=None ):
      self.config().rekeyDataAmount = self.config().rekeyDataAmountDefault
      self.config().rekeyDataUnit = self.config().rekeyDataUnitDefault

   def setRekeyTime( self, args ):
      """
      Set the rekey time limit.
      """
      self.config().rekeyTimeLimit = args[ 'TIMELIMIT' ]
      self.config().rekeyTimeUnit = args[ 'REKEYTIMEUNIT' ]

   def defaultRekeyTime( self, args=None ):
      """
      By default have a time limit of 0 (i.e., no rekeying based on time).
      """
      self.config().rekeyTimeLimit = self.config().rekeyTimeLimitDefault
      self.config().rekeyTimeUnit = self.config().rekeyTimeUnitDefault

   def setHostKeyAlgorithms( self, args ):
      """
      Set the HostKey filter to allow the given HostKey Algorithm
      """
      hostKeySet = set( args.get( 'HOSTKEYALGOS', () ) )
      if 'HOSTKEYALGOS' in args:
         # Filter for the legacy hostkeys.
         for keyAlgo, keyAlgoAlias in SysMgrLib.legacyKeyTypeToAliasedParams.items():
            if keyAlgo in hostKeySet:
               hostKeySet.remove( keyAlgo )
               hostKeySet.add( keyAlgoAlias[ 0 ] )

      # Ensure we can't enter two keys of the same algorithm,
      # either default or named keys
      knownAlgos = copy.deepcopy( hostKeySet )

      for f in args.get( 'FILENAME', () ):
         # Extract the algorithm and ensure its not already listed
         # We cannot have two or more host keys with the same algorithm
         keyAlgo = SshCertLib.getAlgoFromKeyPath( f.pathname )
         if keyAlgo in knownAlgos:
            self.addError( f"Cannot add two or more host keys of type {keyAlgo}" )
            return
         hostKeySet.add( f.realFilename_ )
         knownAlgos.add( keyAlgo )

      self.checkAlgorithms( knownAlgos, FIPS_HOSTKEYS )
      self.checkSupported( knownAlgos, SUPPORTED_HOSTKEYS )
      self.config().hostkey = " ".join( sorted( hostKeySet ) )

   def defaultHostKeyAlgorithms( self, args=None ):
      """
      Set the HostKey filters to allow all HostKey Algorithms
      """
      self.config().hostkey = self.config().hostkeyDefault

   def _setLoginGraceTime( self, loginGraceTime ):
      """
      Set the time period allowed for completing a login.
      """
      loginTimeout = Tac.Value( "Mgmt::SessionTimeout::SuccessfulLoginTimeout" )
      loginTimeout.timeout = loginGraceTime
      self.config().successfulLoginTimeout = loginTimeout

   def setLoginGraceTime( self, args ):
      """
      Set the time period allowed for completing a login.
      """
      self._setLoginGraceTime( args[ 'TIMEOUT' ] )

   def noLoginGraceTime( self, args ):
      """
      Sets the time period allowed for completing a login to
      infinite.
      """
      self._setLoginGraceTime( 0 )

   def defaultLoginGraceTime( self, args ):
      """
      Set the time period allowed for completing a login back
      to the default.
      """
      self._setLoginGraceTime( self.config().successfulLoginTimeout.defaultTimeout )

   def setLogLevel( self, args ):
      """Set the log level for SSHD"""
      self.config().logLevel = args[ 'LOGLEVEL' ]

   def noLogLevel( self, args=None ):
      """Restore the log level to default for SSHD"""
      self.config().logLevel = self.config().logLevelDefault

   def enableLoggingTarget( self, args=None ):
      """Enable SSH/SSHD system logging"""
      self.config().loggingTargetEnabled = True

   def disableLoggingTarget( self, args=None ):
      """Disable SSH/SSHD system logging """
      self.config().loggingTargetEnabled = False

   def checkHostKeys( self, args ):
      """
      Turn on strictly checking host keys.
      """
      self.config().enforceCheckHostKeys = True

   def noCheckHostKeys( self, args=None ):
      """
      Turn off strictly checking host keys.
      """
      self.config().enforceCheckHostKeys = False

   def setPermitEmptyPasswords( self, args ):
      if 'auto' in args:
         val = 'automatic'
      elif 'permit' in args:
         val = 'permit'
      elif 'deny' in args:
         val = 'deny'
      self.config().permitEmptyPasswords = val

   def setPermitEmptyPasswordsDefault( self, args=None ):
      self.config().permitEmptyPasswords = \
          self.config().permitEmptyPasswordsDefault

   def setClientAliveInterval( self, args ):
      self.config().clientAliveInterval = args[ 'INTERVAL' ]

   def setClientAliveIntervalDefault( self, args=None ):
      self.config().clientAliveInterval = \
         self.config().clientAliveIntervalDefault

   def setClientAliveCountMax( self, args ):
      self.config().clientAliveCountMax = args[ 'SECONDS' ]

   def setClientAliveCountMaxDefault( self, args=None ):
      self.config().clientAliveCountMax = \
         self.config().clientAliveCountMaxDefault

   def shutdown( self, args ):
      self.config().serverState = "disabled"

   def noShutdown( self, args=None ):
      self.config().serverState = "enabled"

   def verifyDns( self, args ):
      self.config().verifyDns = True

   def noVerifyDns( self, args=None ):
      self.config().verifyDns = False

   def setIpAcl( self, args ):
      aclName = args[ 'ACL_NAME' ]
      vrfName = args.get( 'VRF' )
      AclCliLib.setServiceAcl( self, 'ssh', IPPROTO_TCP,
                               self.aclConfig_, self.aclCpConfig_, aclName,
                               'ip', vrfName, port=[ self.config().serverPort ] )

   def noIpAcl( self, args=None ):
      if args:
         aclName = args.get( 'ACL_NAME' )
         vrfName = args.get( 'VRF' )
      else:
         aclName = None
         vrfName = None
      AclCliLib.noServiceAcl( self, 'ssh', self.aclConfig_, self.aclCpConfig_,
                              aclName, 'ip', vrfName )

   def setIp6Acl( self, args ):
      aclName = args[ 'ACL_NAME' ]
      vrfName = args.get( 'VRF' )
      AclCliLib.setServiceAcl( self, 'ssh', IPPROTO_TCP,
                               self.aclConfig_, self.aclCpConfig_, aclName,
                               'ipv6', vrfName, port=[ self.config().serverPort ] )

   def noIp6Acl( self, args=None ):
      if args:
         aclName = args.get( 'ACL_NAME' )
         vrfName = args.get( 'VRF' )
      else:
         aclName = None
         vrfName = None
      AclCliLib.noServiceAcl( self, 'ssh', self.aclConfig_, self.aclCpConfig_,
                              aclName, 'ipv6', vrfName )

   def gotoSshMainTunnelConfigMode( self, args ):
      tunnelName = args[ 'TUNNELNAME' ]
      childMode = self.childMode( SshMainTunnelConfigMode, tunnelName=tunnelName )
      self.session_.gotoChildMode( childMode )

   def noMainSshTunnel( self, args ):
      tunnelName = args[ 'TUNNELNAME' ]
      if not tunnelName in sshConfig.tunnel or \
         not sshConfig.tunnel[ tunnelName ].configuredInSshMode:
         return
      childMode = self.childMode( SshMainTunnelConfigMode, tunnelName=tunnelName )
      childMode.modeletMap[ SshTunnelConfigModelet ].setShutdown()
      del sshConfig.tunnel[ tunnelName ]

   def updateDscpRules( self ):
      dscpValue = self.config().dscpValue

      if not dscpValue:
         del self.dscpConfig_.protoConfig[ 'ssh' ]
         return

      protoConfig = self.dscpConfig_.newProtoConfig( 'ssh' )
      ruleColl = protoConfig.rule
      ruleColl.clear()

      for vrf in itertools.chain( [ DEFAULT_VRF ], self.config().vrfConfig ):
         # Traffic to external ssh server.
         DscpCliLib.addDscpRule( ruleColl, '0.0.0.0', 22, False, vrf, 'tcp',
                                 dscpValue )
         DscpCliLib.addDscpRule( ruleColl, '::', 22, False, vrf, 'tcp', dscpValue,
                                 v6=True )

         # Traffic from internal ssh server.
         DscpCliLib.addDscpRule( ruleColl, '0.0.0.0', self.config().serverPort, True,
                                 vrf, 'tcp', dscpValue )
         DscpCliLib.addDscpRule( ruleColl, '::', self.config().serverPort, True, vrf,
                                 'tcp', dscpValue, v6=True )

   def setDscp( self, args ):
      self.config().dscpValue = args[ 'DSCP' ]
      self.updateDscpRules()

   def noDscp( self, args=None ):
      self.config().dscpValue = self.config().dscpValueDefault
      self.updateDscpRules()

   def setConnectionLimit( self, args ):
      config = self.config()
      config.connLimit = args.get( 'LIMIT', config.connLimitDefault )

   def setPerHostConnectionLimit( self, args ):
      config = self.config()
      config.perHostConnLimit = args.get( 'LIMIT', config.perHostConnLimitDefault )

   def setAuthPrincipalsCmd( self, args ):
      self.config().authPrincipalsCmdFile = args.get( 'FILENAME', '' )
      self.config().authPrincipalsCmdArgs = args.get( 'ARGUMENTS', '' )

   def setCaKeyFile( self, args ):
      self.config().caKeyFiles.clear()
      for fileName in args[ 'FILENAME' ]:
         self.config().caKeyFiles.add( fileName )

   def noCaKeyFile( self, args ):
      files = args.get( 'FILENAME' )
      if files:
         for fileName in files:
            self.config().caKeyFiles.remove( fileName )
      else:
         self.config().caKeyFiles.clear()

   def setHostCertFile( self, args ):
      self.config().hostCertFiles.clear()
      certFiles = args[ 'FILENAME' ]
      hostCertKeyTypes = SshCertLib.hostCertsByKeyTypes( certFiles )
      for keyType, certs in hostCertKeyTypes.items():
         # If the keyType is invalid, it's probably because cert doesn't exist,
         # since we validate the host certs when they are copied to the fs.
         # Don't need to complain now; user can copy the file later.
         if keyType == KeyType.invalid:
            continue
         if len( certs ) > 1:
            warnMsg = ( "Host certificates %s have the same key type: %s."
                        " Only one of these certificates will have an effect on"
                        " the SSH config."
                        " Please remove the conflicting certificates." )
            self.addWarning( warnMsg % ( ", ".join( certs ), keyType ) )
      for fileName in certFiles:
         self.config().hostCertFiles.add( fileName )

   def noHostCertFile( self, args ):
      files = args.get( 'FILENAME' )
      if files:
         for fileName in files:
            self.config().hostCertFiles.remove( fileName )
      else:
         self.config().hostCertFiles.clear()

   def setRevokedUserKeysFile( self, args ):
      self.config().revokedUserKeysFiles.clear()
      for fileName in args[ 'FILENAME' ]:
         self.config().revokedUserKeysFiles.add( fileName )

   def noRevokedUserKeysFile( self, args ):
      files = args.get( 'FILENAME' )
      if files:
         for fileName in files:
            self.config().revokedUserKeysFiles.remove( fileName )
      else:
         self.config().revokedUserKeysFiles.clear()

   def gotoSshUserConfigMode( self, args ):
      self.userName_ = args[ 'USER' ]
      childMode = self.childMode( SshUserConfigMode, userName=self.userName_ )
      self.session_.gotoChildMode( childMode )

   def noSshUser( self, args ):
      user = args[ 'USER' ]

      if user in self.config().user:
         SysMgrLib.defaultSshUserConfig( user, self.config() )

   def gotoSshKeyConfigMode( self, args ):
      self.keyName_ = args[ 'KEYNAME' ]
      # Flag which triggers usage of the new CLI command
      sshConfig.useNewSshCliKey = True
      if self.userName_ and self.keyName_:
         childMode = self.childMode( SshKeyConfigMode, userName=self.userName_,
                                    keyName=self.keyName_ )
         self.session_.gotoChildMode( childMode )

   def noSshKey( self, args ):
      keyName = args[ 'KEYNAME' ]

      # Non-existent key
      if keyName not in self.config().sshAuthKeys:
         return

      # Note down the key before deleting it
      deletedKey = self.config().sshAuthKeys[ keyName ].keyContents
      del self.config().sshAuthKeys[ keyName ]

      # If the deleted key is the published key, publish a different one
      updatePublishedAuthKey( self.config(), deletedKey )

   def addSshKeyContent( self, args ):
      keyContents = args[ 'SSHKEY' ]
      userConfig = sshConfig.user[ self.userName_ ]
      if self.userName_ and self.keyName_ and keyContents:
         if not validateSshAuthorizedKey( keyContents ):
            self.addError( 'Unrecognized ssh key' )
            return
         # Add the key contents to the SshAuthKeys collection
         self.config().keyContents = keyContents

         # Update the published key if
         # 1. Modified key is the primary key or
         # 2. No published key present, update the modified key as published key.
         #    Don't update the published key if one is already present.
         if ( self.keyName_ == primarySshKeyId or
               userConfig.publishedSshAuthKey == "" ):
            userConfig.publishedSshAuthKey = keyContents

   def noSshKeyContent( self, args ):
      clearedKey = self.config().keyContents
      self.config().keyContents = ""

      # If the cleared key is the published key, publish a different one
      updatePublishedAuthKey( sshConfig.user[ self.userName_ ], clearedKey )

   def gotoSshPrincipalConfigMode( self, args ):
      principal = args[ 'PRINCIPAL' ]
      # Flag which triggers usage of the new CLI command
      sshConfig.useNewSshCliPrincipal = True
      if self.userName_ and principal:
         childMode = self.childMode( SshPrincipalConfigMode, userName=self.userName_,
                                    principal=principal )
         self.session_.gotoChildMode( childMode )

   def noSshPrincipal( self, args ):
      principal = args[ 'PRINCIPAL' ]

      # Non-existent principal
      if principal not in self.config().sshAuthPrincipals:
         return

      del self.config().sshAuthPrincipals[ principal ]

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [ no | default ] connection limit LIMIT
#-----------------------------------------------------------------------------------
class SshSessionLimit( CliCommand.CliCommandClass ):
   syntax = 'connection limit LIMIT'
   noOrDefaultSyntax = syntax.replace( 'LIMIT', '...' )
   data = {
            'connection': connectionKwMatcher,
            'limit': 'Set maximum number of SSH sessions',
            'LIMIT': CliMatcher.IntegerMatcher( 0, 100,
                                                  helpdesc='Maximum number of '
                                                           'SSH connections' ),
          }
   handler = SshConfigMode.setConnectionLimit
   noOrDefaultHandler = SshConfigMode.setConnectionLimit

SshConfigMode.addCommandClass( SshSessionLimit )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [ no | default ] connection per-host LIMIT
#-----------------------------------------------------------------------------------
class SshPerHostLimit( CliCommand.CliCommandClass ):
   syntax = 'connection per-host LIMIT'
   noOrDefaultSyntax = 'connection per-host ...'
   data = {
            'connection': connectionKwMatcher,
            'per-host': 'Set maximum number of SSH sessions from a single host',
            'LIMIT': CliMatcher.IntegerMatcher( 1, 20,
                                                  helpdesc='Maximum number of '
                                                           'SSH connections' ),
          }
   handler = SshConfigMode.setPerHostConnectionLimit
   noOrDefaultHandler = SshConfigMode.setPerHostConnectionLimit

SshConfigMode.addCommandClass( SshPerHostLimit )

#-----------------------------------------------------------------------------------
# [ no | default ] known-hosts HOST KEYALGOEXPR PUB_KEY
#-----------------------------------------------------------------------------------
knownHostsConfigKwMatcher = CliMatcher.KeywordMatcher( 'known-hosts',
                                          alternates=[ 'known-host' ],
                                          helpdesc='SSH Known Hosts public keys' )

class KnownHosts( CliCommand.CliCommandClass ):
   syntax = 'known-hosts HOST KEYALGOEXPR PUB_KEY'
   noOrDefaultSyntax = 'known-hosts HOST ...'
   data = {
            'known-hosts': knownHostsConfigKwMatcher,
            'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'PUB_KEY': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9+/=]+',
                                                  helpdesc='Base 64 string',
                                                  helpname='STRING' ),
          }
   handler = SshConfigMode.setKnownHost
   noOrDefaultHandler = SshConfigMode.noKnownHost
   hidden = True

SshConfigMode.addCommandClass( KnownHosts )

#-----------------------------------------------------------------------------------
# RFE40658 : Allow precise configuration of SSH allowed ciphers
# [ no | default ] cipher CIPHERS
#-----------------------------------------------------------------------------------
CiphersExpression = CliCommand.setCliExpression( CIPHERS, name='CIPHERS' )
class Cipher( CliCommand.CliCommandClass ):
   syntax = 'cipher CIPHEREXPR'
   noSyntax = 'cipher ...'
   defaultSyntax = 'cipher ...'
   data = {
            'cipher': 'Exclusive list of cryptographic ciphers for sshd to speak',
            'CIPHEREXPR': CiphersExpression
          }
   handler = SshConfigMode.setCiphers
   noHandler = SshConfigMode.noCiphers
   defaultHandler = SshConfigMode.defaultCiphers

SshConfigMode.addCommandClass( Cipher )

#-----------------------------------------------------------------------------------
# Hidden command to deal with legacy upgreades that use 'ciphers'
# [ no | default ] ciphers CIPHERS
#-----------------------------------------------------------------------------------
class Ciphers( CliCommand.CliCommandClass ):
   syntax = 'ciphers CIPHEREXPR'
   data = {
            'ciphers': 'Exclusive list of cryptographic ciphers for sshd to speak',
            'CIPHEREXPR': CiphersExpression
          }
   handler = SshConfigMode.setCiphers
   hidden = True

SshConfigMode.addCommandClass( Ciphers )

#-----------------------------------------------------------------------------------
# [ no | default ] key-exchange KEXEXPR
#-----------------------------------------------------------------------------------
class KeyExchange( CliCommand.CliCommandClass ):
   syntax = 'key-exchange KEXEXPR'
   noSyntax = 'key-exchange ...'
   defaultSyntax = 'key-exchange ...'
   data = {
            'key-exchange': ( 'Exclusive list of key-exchange methods for sshd '
                              'to speak' ),
            'KEXEXPR': CliCommand.setCliExpression( KEXS, name='KEX' )
          }
   handler = SshConfigMode.setKeyExchange
   noHandler = SshConfigMode.noKeyExchange
   defaultHandler = SshConfigMode.defaultKeyExchange

SshConfigMode.addCommandClass( KeyExchange )

#-----------------------------------------------------------------------------------
# [ no | default ] mac MAC
#-----------------------------------------------------------------------------------
class MacCmd( CliCommand.CliCommandClass ):
   syntax = 'mac MACEXPR'
   noSyntax = 'mac ...'
   defaultSyntax = 'mac ...'
   data = {
            'mac': 'Exclusive list of MAC algorithms for sshd to speak',
            'MACEXPR': CliCommand.setCliExpression( MACS, name='MAC' )
          }
   handler = SshConfigMode.setMac
   noHandler = SshConfigMode.noMac
   defaultHandler = SshConfigMode.defaultMac

SshConfigMode.addCommandClass( MacCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] idle-timeout IDLETIME
#-----------------------------------------------------------------------------------
class IdleTimeout( CliCommand.CliCommandClass ):
   syntax = 'idle-timeout IDLETIME'
   noOrDefaultSyntax = 'idle-timeout ...'
   data = {
            'idle-timeout': 'Set idle session timeout(minutes)',
            'IDLETIME': CliMatcher.IntegerMatcher( 0, 86400,
                                                   helpdesc='Idle session timeout '
                                                            'in minutes' )
          }
   handler = SshConfigMode.setIdleTimeout
   noOrDefaultHandler = SshConfigMode.noIdleTimeout

SshConfigMode.addCommandClass( IdleTimeout )
#-----------------------------------------------------------------------------------
# [ no | default ] authentication mode ( password | keyboard-interactive )
#-----------------------------------------------------------------------------------
class AuthenticationModeCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication mode ( password | keyboard-interactive )'
   noOrDefaultSyntax = 'authentication mode ...'
   data = {
            'authentication': authKwMatcher,
            'mode': 'Change authentication mode',
            'password': 'Makes sshd enter password mode',
            'keyboard-interactive': 'Makes sshd enter keyboard-interactive mode',
          }
   handler = SshConfigMode.setAuthenticationMode
   noOrDefaultHandler = SshConfigMode.defaultAuthenticationMode
   hidden = True

SshConfigMode.addCommandClass( AuthenticationModeCmd )
#-----------------------------------------------------------------------------------
# [ no | default ]
#       authentication protocol [ multi-factor ]
#       ( { password | keyboard-interactive | public-key } )
#           |
#       ( { multi-factor ( password | keyboard-interactive | public-key) } )
#-----------------------------------------------------------------------------------
class AuthenticationProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication protocol OPTIONS | { multi-factor OPTIONS }'
   noOrDefaultSyntax = 'authentication protocol ...'
   data = {
            'authentication': authKwMatcher,
            'protocol': 'Change which authentication protocols are enabled',
            'multi-factor': 'Enable multi factor authentication methods',
            'OPTIONS': CliCommand.SetEnumMatcher( {
               'password': 'Enable password authentication',
               'keyboard-interactive': 'Enable keyboard-interactive authentication',
               'public-key': 'Enable public-key authentication'
            }, inIteration=True )
          }
   handler = SshConfigMode.setAuthenticationProtocol
   noOrDefaultHandler = SshConfigMode.defaultAuthenticationProtocol

SshConfigMode.addCommandClass( AuthenticationProtocolCmd )
#-----------------------------------------------------------------------------------
# [ no | default ] authentication empty-passwords ( auto | permit | deny )
#-----------------------------------------------------------------------------------
class AuthenticationEmptyPasswords( CliCommand.CliCommandClass ):
   syntax = 'authentication empty-passwords ( auto | permit | deny )'
   noOrDefaultSyntax = 'authentication empty-passwords ...'
   data = {
            'authentication': authKwMatcher,
            'empty-passwords': 'Whether to allow empty passwords',
            'auto': 'Auto-determine whether to allow empty passwords (default)',
            'permit': 'Allow empty passwords',
            'deny': 'Do not allow empty passwords',
          }
   handler = SshConfigMode.setPermitEmptyPasswords
   noOrDefaultHandler = SshConfigMode.setPermitEmptyPasswordsDefault

SshConfigMode.addCommandClass( AuthenticationEmptyPasswords )
#-----------------------------------------------------------------------------------
# [ no | default ] authentication x509
#-----------------------------------------------------------------------------------
class AuthenticationX509Mode( CliCommand.CliCommandClass ):
   syntax = 'authentication x509'
   noOrDefaultSyntax = syntax
   data = {
           'authentication': authKwMatcher,
           'x509': "Configure authentication with X509 certificates",
          }
   handler = SshConfigMode.gotoSshAuthenticationX509Mode
   noOrDefaultHandler = SshConfigMode.noSshAuthenticationX509Mode

SshConfigMode.addCommandClass( AuthenticationX509Mode )
#-----------------------------------------------------------------------------------
# [ no | default ] client-alive interval INTERVAL
#-----------------------------------------------------------------------------------
class ClientAliveInterval( CliCommand.CliCommandClass ):
   syntax = 'client-alive interval INTERVAL'
   noOrDefaultSyntax = 'client-alive interval ...'
   data = {
            'client-alive': 'Set SSH ClientAlive options',
            'interval': 'Time period ( in seconds ) to send SSH keep-alive packets',
            'INTERVAL': thousandIntMatcher
          }
   handler = SshConfigMode.setClientAliveInterval
   noOrDefaultHandler = SshConfigMode.setClientAliveIntervalDefault

SshConfigMode.addCommandClass( ClientAliveInterval )
#-----------------------------------------------------------------------------------
# [ no | default ] client-alive count-max SECONDS
#-----------------------------------------------------------------------------------
class ClientAliveCountMax( CliCommand.CliCommandClass ):
   syntax = 'client-alive count-max SECONDS'
   noOrDefaultSyntax = 'client-alive count-max ...'
   data = {
            'client-alive': 'Set SSH ClientAlive options',
            'count-max': ( 'Number of keep-alive packets that can be sent without '
                           'a response before the connection is assumed dead' ),
            'SECONDS': thousandIntMatcher
          }
   handler = SshConfigMode.setClientAliveCountMax
   noOrDefaultHandler = SshConfigMode.setClientAliveCountMaxDefault

SshConfigMode.addCommandClass( ClientAliveCountMax )
#-----------------------------------------------------------------------------------
# [ no | default ] rekey frequency AMOUNT REKEYUNIT
#-----------------------------------------------------------------------------------
class RekeyFrequency( CliCommand.CliCommandClass ):
   syntax = 'rekey frequency AMOUNT REKEYUNIT'
   noOrDefaultSyntax = 'rekey frequency ...'
   data = {
            'rekey': 'When to rekey ssh connection',
            'frequency': 'rekey upon meeting criteria',
            'AMOUNT': CliMatcher.IntegerMatcher( 1, sys.maxsize,
                                       helpdesc='Amount of data before rekeying' ),
            'REKEYUNIT': CliMatcher.EnumMatcher( REKEYUNITS )
          }
   handler = SshConfigMode.setRekeyData
   noOrDefaultHandler = SshConfigMode.defaultRekeyData

SshConfigMode.addCommandClass( RekeyFrequency )
#-----------------------------------------------------------------------------------
# [ no | default ] rekey interval TIMELIMIT REKEYTIMEUNIT
#-----------------------------------------------------------------------------------
class RekeyInterval( CliCommand.CliCommandClass ):
   syntax = 'rekey interval TIMELIMIT REKEYTIMEUNIT'
   noOrDefaultSyntax = 'rekey interval ...'
   data = {
            'rekey': 'When to rekey ssh connection',
            'interval': 'rekey after alloted time',
            'TIMELIMIT': CliMatcher.IntegerMatcher( 0, 99999,
                                       helpdesc='Amount of time before rekeying' ),
            'REKEYTIMEUNIT': CliMatcher.EnumMatcher( REKEYTIMEUNITS )
          }
   handler = SshConfigMode.setRekeyTime
   noOrDefaultHandler = SshConfigMode.defaultRekeyTime

SshConfigMode.addCommandClass( RekeyInterval )
#-----------------------------------------------------------------------------------
# [ no | default ] hostkey client strict-checking
#-----------------------------------------------------------------------------------
class HostkeyClient( CliCommand.CliCommandClass ):
   syntax = 'hostkey client strict-checking'
   noOrDefaultSyntax = 'hostkey client strict-checking ...'
   data = {
            'hostkey': hostkeyKwMatcher,
            'client': 'hostkey settings for ssh connections from the switch',
            'strict-checking': 'Enforce strict host key checking',
          }
   handler = SshConfigMode.checkHostKeys
   noOrDefaultHandler = SshConfigMode.noCheckHostKeys

SshConfigMode.addCommandClass( HostkeyClient )
#-----------------------------------------------------------------------------------
# [ no | default ] hostkey server [ { HOSTKEYALGOS | FILENAME } ]
#-----------------------------------------------------------------------------------
hostkeyServerKwMatcher = CliMatcher.KeywordMatcher( 'server',
                                     helpdesc='Switch\'s SSH hostkey settings' )

class HostkeyServer( CliCommand.CliCommandClass ):
   syntax = 'hostkey server [ { HOSTKEYALGOS | FILENAME } ]'
   noOrDefaultSyntax = 'hostkey server ...'
   data = {
            'hostkey': hostkeyKwMatcher,
            'server' : hostkeyServerKwMatcher,
            'HOSTKEYALGOS': hostKeyMatcher,
            'FILENAME': UrlMatcher( fsFunc=lambda fs: fs.scheme == 'ssh-key:',
               helpdesc='SSH Private key file name',
               acceptSimpleFile=False )
          }
   handler = SshConfigMode.setHostKeyAlgorithms
   noOrDefaultHandler = SshConfigMode.defaultHostKeyAlgorithms

SshConfigMode.addCommandClass( HostkeyServer )
#-----------------------------------------------------------------------------------
# [ no | default ] server-port PORT
#-----------------------------------------------------------------------------------
class ServerPort( CliCommand.CliCommandClass ):
   syntax = 'server-port PORT'
   noOrDefaultSyntax = 'server-port ...'
   data = {
            'server-port': 'Change server port',
            'PORT': CliMatcher.IntegerMatcher( serverPort.min, serverPort.max,
                                               helpdesc='Number of the port to '
                                                        'use' ),
          }
   handler = SshConfigMode.setServerPort
   noOrDefaultHandler = SshConfigMode.noServerPort

SshConfigMode.addCommandClass( ServerPort )
#-----------------------------------------------------------------------------------
# [ no | default ] fips restrictions
#-----------------------------------------------------------------------------------
class FipsRestrictions( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = 'fips restrictions ...'
   data = {
            'fips': 'FIPS settings',
            'restrictions': 'Use FIPS compliant algorithms',
          }
   handler = SshConfigMode.fipsRestrictions
   noOrDefaultHandler = SshConfigMode.noFipsRestrictions

SshConfigMode.addCommandClass( FipsRestrictions )

#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class Shutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown ...'
   data = {
            'shutdown': 'Disable sshd'
          }
   handler = SshConfigMode.shutdown
   noOrDefaultHandler = SshConfigMode.noShutdown

SshConfigMode.addCommandClass( Shutdown )

# -----------------------------------------------------------------------------------
# [ no | default ] verify dns
# -----------------------------------------------------------------------------------
class VerifyDns( CliCommand.CliCommandClass ):
   syntax = 'verify dns'
   noOrDefaultSyntax = 'verify dns ...'
   data = {
            'verify': 'Set options related to sshd verification',
            'dns': 'Configure SSH daemon verify DNS check'
          }
   handler = SshConfigMode.verifyDns
   noOrDefaultHandler = SshConfigMode.noVerifyDns

SshConfigMode.addCommandClass( VerifyDns )

#-----------------------------------------------------------------------------------
# [ no | default ] login timeout TIMEOUT
#-----------------------------------------------------------------------------------
class LoginTimeout( CliCommand.CliCommandClass ):
   syntax = 'login timeout TIMEOUT'
   noOrDefaultSyntax = 'login timeout ...'
   data = {
            'login': 'Set options related to logging in',
            'timeout': 'Set the time allowed for a user to log in via ssh',
            'TIMEOUT': CliMatcher.IntegerMatcher( 1, 600,
                                            helpdesc='time allowed ( in seconds ).' )
          }
   handler = SshConfigMode.setLoginGraceTime
   noHandler = SshConfigMode.noLoginGraceTime
   defaultHandler = SshConfigMode.defaultLoginGraceTime

SshConfigMode.addCommandClass( LoginTimeout )
#-----------------------------------------------------------------------------------
# [ no | default ] log-level LOGLEVEL
#-----------------------------------------------------------------------------------
LOG_LEVELS = ( 'quiet', 'fatal', 'error', 'info', 'verbose',
               'debug', 'debug1', 'debug2', 'debug3' )
LOG_LEVELS_HELPDESC = { i: i.upper() + ' log level' for i in LOG_LEVELS }

class LogLevel( CliCommand.CliCommandClass ):
   syntax = 'log-level LOGLEVEL'
   noOrDefaultSyntax = 'log-level ...'
   data = {
            'log-level': 'Configure SSH daemon logging level',
            'LOGLEVEL': CliMatcher.EnumMatcher( LOG_LEVELS_HELPDESC )
          }
   handler = SshConfigMode.setLogLevel
   noOrDefaultHandler = SshConfigMode.noLogLevel

SshConfigMode.addCommandClass( LogLevel )
#-----------------------------------------------------------------------------------
# [ no | default ] logging target system
#-----------------------------------------------------------------------------------
class LoggingTarget( CliCommand.CliCommandClass ):
   syntax = 'logging target system'
   noOrDefaultSyntax = syntax
   data = {
            'logging': 'Configure SSH system logging',
            'target': 'Specify SSH/SSHD service syslog target',
            'system': 'Set SSH log messages destination to system log buffer'
          }
   handler = SshConfigMode.enableLoggingTarget
   noOrDefaultHandler = SshConfigMode.disableLoggingTarget

SshConfigMode.addCommandClass( LoggingTarget )
#-----------------------------------------------------------------------------------
# [ no | default ] ip access-group NAME [ vrf VRF ] in
#-----------------------------------------------------------------------------------
vrfWithNameExprFactory = VrfExprFactory(
      helpdesc='Configure the VRF in which to apply the access control list',
      inclDefaultVrf=True )
class IpAccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ip': 'SSH IP configuration',
            'access-group': 'Configure access control list',
            'ACL_NAME': AclCli.standardIpAclNameMatcher,
            'VRF': vrfWithNameExprFactory,
            'in': 'Inbound packets'

          }
   handler = SshConfigMode.setIpAcl
   noOrDefaultHandler = SshConfigMode.noIpAcl

SshConfigMode.addCommandClass( IpAccessGroup )
#-----------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group NAME [ vrf VRF ] in
#-----------------------------------------------------------------------------------
class Ipv6AccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ipv6': 'SSH IPv6 configuration',
            'access-group': 'Configure access control list',
            'ACL_NAME': AclCli.standardIp6AclNameMatcher,
            'VRF': vrfWithNameExprFactory ,
            'in': 'Inbound packets'

          }
   handler = SshConfigMode.setIp6Acl
   noOrDefaultHandler = SshConfigMode.noIp6Acl

SshConfigMode.addCommandClass( Ipv6AccessGroup )
#-----------------------------------------------------------------------------------
# [ no | default ] tunnel TUNNELNAME
#-----------------------------------------------------------------------------------
class Tunnel( CliCommand.CliCommandClass ):
   syntax = 'tunnel TUNNELNAME'
   noOrDefaultSyntax = 'tunnel TUNNELNAME ...'
   data = {
            'tunnel': 'manage named SSH tunnel',
            'TUNNELNAME': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                        helpdesc='SSH tunnel name', helpname='WORD' )
          }
   handler = SshConfigMode.gotoSshMainTunnelConfigMode
   noOrDefaultHandler = SshConfigMode.noMainSshTunnel
   hidden = True

SshConfigMode.addCommandClass( Tunnel )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [no | default] authorized-principals command FILE [ ARGUMENTS ]
#-----------------------------------------------------------------------------------
class SshAuthPrincipalsCmd( CliCommand.CliCommandClass ):
   syntax = 'authorized-principals command FILENAME [ ARGUMENTS ]'
   noOrDefaultSyntax = 'authorized-principals command ...'
   data = {
            'authorized-principals': 'Configure the authorized principals settings',
            'command' : 'Configure the authorized principals command',
            'FILENAME': authPrincipalsCmdMatcher,
            'ARGUMENTS': CliMatcher.StringMatcher(
               helpname='ARGUMENTS',
               helpdesc='Arguments to pass to the authorized principals command'
            )
          }
   handler = SshConfigMode.setAuthPrincipalsCmd
   noOrDefaultHandler = handler

SshConfigMode.addCommandClass( SshAuthPrincipalsCmd )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [no | default] trusted-ca key public FILE1 [FILE2 ...]
#-----------------------------------------------------------------------------------
class SshCaPublicKeyFile( CliCommand.CliCommandClass ):
   syntax = "trusted-ca key public { FILENAME }"
   noOrDefaultSyntax = "trusted-ca key public [ { FILENAME } ]"
   data = {
            'trusted-ca': 'Configure trusted CA',
            'key'       : 'Configure CA public key',
            'public'    : 'Configure CA public key file',
            'FILENAME': trustedCaMatcher,
          }
   handler = SshConfigMode.setCaKeyFile
   noOrDefaultHandler = SshConfigMode.noCaKeyFile

SshConfigMode.addCommandClass( SshCaPublicKeyFile )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [no | default] hostkey server cert FILE1 [FILE2 ...]
#-----------------------------------------------------------------------------------
class SshHostCertFile( CliCommand.CliCommandClass ):
   syntax = "hostkey server cert { FILENAME }"
   noOrDefaultSyntax = "hostkey server cert [ { FILENAME } ]"
   data = {
            'hostkey'   : hostkeyKwMatcher,
            'server'    : hostkeyServerKwMatcher,
            'cert'      : 'Configure switch\'s hostkey cert file',
            'FILENAME': hostCertMatcher,
          }
   handler = SshConfigMode.setHostCertFile
   noOrDefaultHandler = SshConfigMode.noHostCertFile

SshConfigMode.addCommandClass( SshHostCertFile )

#-----------------------------------------------------------------------
# (config-mgmt-ssh) [no|default] user-keys revoke-list FILE
#-----------------------------------------------------------------------
class SshRevokedKeysFile( CliCommand.CliCommandClass ):
   syntax = "user-keys revoke-list { FILENAME }"
   noOrDefaultSyntax = "user-keys revoke-list [ { FILENAME } ]"
   data = {
            'user-keys'   : 'SSH user keys\' settings',
            'revoke-list' : 'Configure revoked SSH user keys file',
            'FILENAME'  : revokeListMatcher,
          }
   handler = SshConfigMode.setRevokedUserKeysFile
   noOrDefaultHandler = SshConfigMode.noRevokedUserKeysFile

SshConfigMode.addCommandClass( SshRevokedKeysFile )

#-----------------------------------------------------------------------------------
# [ no | default ] username user
#-----------------------------------------------------------------------------------
class UserConfig( CliCommand.CliCommandClass ):
   syntax = 'username USER'
   noOrDefaultSyntax = 'username USER ...'
   data = {
            'username': 'Enter SSH user specific configuration submode',
            'USER': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                    helpdesc='SSH user name', helpname='WORD' )
          }
   handler = SshConfigMode.gotoSshUserConfigMode
   noOrDefaultHandler = SshConfigMode.noSshUser

SshConfigMode.addCommandClass( UserConfig )

# -----------------------------------------------------------------------------------
# [ no | default ] ssh-key KEYNAME
# -----------------------------------------------------------------------------------
class SshKeyConfig( CliCommand.CliCommandClass ):
   syntax = 'ssh-key KEYNAME'
   noOrDefaultSyntax = syntax
   data = {
            'ssh-key': 'Enter SSH authorized key specific configuration submode',
            'KEYNAME': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                    helpdesc='SSH key name', helpname='WORD' )
          }
   handler = SshConfigMode.gotoSshKeyConfigMode
   noOrDefaultHandler = SshConfigMode.noSshKey

SshUserConfigMode.addCommandClass( SshKeyConfig )

# -----------------------------------------------------------------------------------
# [ no | default ] public-key SSHKEY
# -----------------------------------------------------------------------------------
class SshKeyContentConfig( CliCommand.CliCommandClass ):
   syntax = 'public-key SSHKEY'
   noOrDefaultSyntax = 'public-key ...'
   data = {
            'public-key': 'Configure an SSH public key for the user',
            'SSHKEY': sshkeyContentMatcher,
          }
   handler = SshConfigMode.addSshKeyContent
   noOrDefaultHandler = SshConfigMode.noSshKeyContent

SshKeyConfigMode.addCommandClass( SshKeyContentConfig )

SshKeyConfigMode.addModelet( SshOptionsConfigModelet )

# -----------------------------------------------------------------------------------
# [ no | default ] ssh-principal PRINCIPAL
# -----------------------------------------------------------------------------------
class SshPrincipalConfig( CliCommand.CliCommandClass ):
   syntax = 'ssh-principal PRINCIPAL'
   noOrDefaultSyntax = syntax
   data = {
            'ssh-principal':
               'Enter SSH authorized principal specific configuration submode',
            'PRINCIPAL': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='SSH authorized principal',
               helpname='WORD' )
          }
   handler = SshConfigMode.gotoSshPrincipalConfigMode
   noOrDefaultHandler = SshConfigMode.noSshPrincipal

SshUserConfigMode.addCommandClass( SshPrincipalConfig )

SshPrincipalConfigMode.addModelet( SshOptionsConfigModelet )

#-----------------------------------------------------------------------------------
# [ no | default ] qos dscp DSCP
#-----------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( SshConfigMode, SshConfigMode.setDscp,
                                   SshConfigMode.noDscp )

#-------------------------------------------------------------------------------
# "vrf VRF" config mode
#-------------------------------------------------------------------------------
class SshVrfConfigMode( VrfConfigMode, BasicCli.ConfigModeBase ):
   name = "SSH VRF Configuration"

   def __init__( self, parent, session, vrfName ):
      VrfConfigMode.__init__( self, ( vrfName, "ssh", sshConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoSshVrfTunnelConfigMode( self, args ):
      tunnelName = args[ 'TUNNELNAME' ]
      childMode = self.childMode( SshVrfTunnelConfigMode,
                                  tunnelName=tunnelName,
                                  vrfConfig=self.config() )
      self.session_.gotoChildMode( childMode )

   def noSshVrfTunnel( self, args ):
      self._noSshVrfTunnel( args[ 'TUNNELNAME' ] )

   def _noSshVrfTunnel( self, tunnelName ):
      conf = sshConfig if self.vrfName_ == DEFAULT_VRF else self.config()
      if tunnelName not in conf.tunnel or \
         conf.tunnel[ tunnelName ].configuredInSshMode:
         return
      childMode = self.childMode( SshVrfTunnelConfigMode,
                                  tunnelName=tunnelName,
                                  vrfConfig=self.config() )
      childMode.modeletMap[ SshTunnelConfigModelet ].setShutdown()
      del conf.tunnel[ tunnelName ]

   def setKnownHost( self, args ):
      host = args[ 'HOST' ]
      keyAlgo = args[ 'KEYALGO' ]
      publicKey = args[ 'PUB_KEY' ]
      conf = sshConfig if self.vrfName_ == DEFAULT_VRF else self.config()
      generalSetKnownHost( conf, host, keyAlgo, publicKey,
                           cliMode=SshVrfConfigMode )

   def noKnownHost( self, args ):
      host = args[ 'HOST' ]
      conf = sshConfig if self.vrfName_ == DEFAULT_VRF else self.config()
      if not host in conf.knownHost or conf.knownHost[ host ].configuredInSshMode:
         return
      generalNoKnownHost( conf, host )

#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class VrfConfigShutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noSyntax = 'shutdown ...'
   defaultSyntax = 'shutdown ...'
   data = {
            'shutdown': 'Disable sshd',
          }
   @staticmethod
   def handler( mode, args ):
      SshVrfConfigMode.shutdown( mode )

   @staticmethod
   def noHandler( mode, args ):
      SshVrfConfigMode.noShutdown( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      SshVrfConfigMode.defaultShutdown( mode )

SshVrfConfigMode.addCommandClass( VrfConfigShutdown )

#-----------------------------------------------------------------------------------
# [ no | default ] tunnel TUNNELNAME
#-----------------------------------------------------------------------------------
class VrfConfigTunnel( CliCommand.CliCommandClass ):
   syntax = 'tunnel TUNNELNAME'
   noOrDefaultSyntax = 'tunnel TUNNELNAME ...'
   data = {
            'tunnel': 'manage named SSH tunnel',
            'TUNNELNAME': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                       helpdesc='SSH tunnel name', helpname='WORD' ),
          }
   handler = SshVrfConfigMode.gotoSshVrfTunnelConfigMode
   noOrDefaultHandler = SshVrfConfigMode.noSshVrfTunnel

SshVrfConfigMode.addCommandClass( VrfConfigTunnel )

#-----------------------------------------------------------------------------------
# [ no | default ] known-hosts HOST KEYALGOEXPR PUB_KEY
#-----------------------------------------------------------------------------------
class VrfKnownHosts( CliCommand.CliCommandClass ):
   syntax = 'known-hosts HOST KEYALGOEXPR PUB_KEY'
   noOrDefaultSyntax = 'known-hosts HOST ...'
   data = {
            'known-hosts': 'SSH Known Hosts public keys',
            'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'PUB_KEY': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9+/=]+',
                                                      helpdesc='Base 64 string',
                                                      helpname='STRING' ),
          }
   handler = SshVrfConfigMode.setKnownHost
   noOrDefaultHandler = SshVrfConfigMode.noKnownHost

SshVrfConfigMode.addCommandClass( VrfKnownHosts )

#-----------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#-----------------------------------------------------------------------------------
class VrfMode( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
            'VRF': VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                   inclDefaultVrf=True ),
          }
   handler = "SshHandler.gotoSshVrfConfigMode"
   noOrDefaultHandler = "SshHandler.noSshVrfConfigMode"

SshConfigMode.addCommandClass( VrfMode )

#----------------------------------------------------------------
# SshAuthenticationX509ConfigMode
#----------------------------------------------------------------
class SshAuthenticationX509ConfigMode( SshAuthenticationX509Mode,
                                       BasicCli.ConfigModeBase ):
   name = "SSH Authentication X509 Configuration"

   def __init__( self, parent, session ):
      SshAuthenticationX509Mode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def config( self ):
      return sshConfig

   def setServerSslProfile( self, args ):
      """Handler for 'ssl profile PROFILENAME'"""
      sshConfig.x509ProfileName = args[ 'PROFILENAME' ]

   def noServerSslProfile( self, args ):
      """Handler for '( no | default ) ssl profile PROFILENAME'"""
      sshConfig.x509ProfileName = ''

   def usernameDomainOmit( self, args ):
      """Handler for 'username domain omit'"""
      sshConfig.x509UsernameDomainOmit = True

   def noUsernameDomainOmit( self, args ):
      """Handler for '( no | default ) username domain omit'"""
      sshConfig.x509UsernameDomainOmit = False

#-----------------------------------------------------------------------------------
# [no | default] server ssl profile [PROFILENAME]
#-----------------------------------------------------------------------------------
class ServerSslProfile( CliCommand.CliCommandClass ):
   syntax = 'server ssl profile [ PROFILENAME ]'
   noOrDefaultSyntax = 'server ssl profile ...'
   data = {
            'server': 'Configure server',
            'ssl': 'Configure SSL related options',
            'profile': 'Configure SSL profile',
            'PROFILENAME': CliMatcher.DynamicNameMatcher(
               lambda mode: sslConfig.profileConfig,
               'Profile name' ),
          }
   handler = SshAuthenticationX509ConfigMode.setServerSslProfile
   noOrDefaultHandler = SshAuthenticationX509ConfigMode.noServerSslProfile

SshAuthenticationX509ConfigMode.addCommandClass( ServerSslProfile )

#-----------------------------------------------------------------------------------
# [no | default ] username domain omit
#-----------------------------------------------------------------------------------
class UsernameDomainOmit( CliCommand.CliCommandClass ):
   syntax = 'username domain omit'
   noOrDefaultSyntax = syntax
   data = {
            'username': ( 'Configure matching the SSH username to a presented'
                          ' X509 certificate\'s Common Name and SAN' ),
            'domain': ( 'Configure the treatment of the domain name in a'
                        ' presented X509 certificate\'s Common Name and SAN' ),
            'omit': ( 'Ignore the domain name in a presented X509 certificate\'s'
                      ' Common Name and SAN when matching them against the logging'
                      ' in SSH username' )
          }
   handler = SshAuthenticationX509ConfigMode.usernameDomainOmit
   noOrDefaultHandler = SshAuthenticationX509ConfigMode.noUsernameDomainOmit

SshAuthenticationX509ConfigMode.addCommandClass( UsernameDomainOmit )

def verifyHook( mode, hashName ):
   if hashName == 'md5' and sshConfig.fipsRestrictions:
      mode.addError( "MD5 is not allowed when using FIPS algorithms." )
      return False
   return True

def Plugin( entityManager ):
   global sshConfig, localUserConfig
   global aclConfig, aclCpConfig
   global dscpConfig
   global sslConfig
   sshConfig = ConfigMount.mount( entityManager, "mgmt/ssh/config",
                                  "Mgmt::Ssh::Config", "w" )
   localUserConfig = ConfigMount.mount( entityManager, "security/aaa/local/config",
                                    "LocalUser::Config", "w" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   dscpConfig = ConfigMount.mount( entityManager,  "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
   sslConfig = LazyMount.mount( entityManager, "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config", "r" )
   FileCli.verifyHook.addExtension( verifyHook )
