# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Str
import SshCertLib
import SysMgrLib

MAX_NAME_LEN = 29

class SshHostKey( Model ):
   hostKey = Str( help="The public host key of the switch" )
   metadata = Dict(
      help="Algorithm-specific metadata for ssh host key",
      keyType=str, valueType=str, optional=True )

   def render( self ):
      out = ""
      if self.hostKey:
         out = self.hostKey
      if self.hostKey:
         if self.metadata:
            for paramName, paramValue in self.metadata.items():
               out += f" {paramName}={paramValue}"
      print( out )

class TrustedCaKeys( Model ):
   keys = List( valueType=str, help="List of trusted CA public keys" )

   def render( self ):
      for key in self.keys:
         print( key )

class HostCertificates( Model ):
   certificates = Dict( keyType=str, valueType=str,
                        help="A Mapping between configured host certificate"
                             " filenames and the raw certificate" )
   def render( self ):
      for certFile in self.certificates:
         try:
            print( SshCertLib.validateHostCert( certFile ) )
         except SshCertLib.SshHostCertError:
            pass

class RevokedKeys( Model ):
   keys = List( valueType=str, help="List of revoked public keys" )

   def render( self ):
      for key in self.keys:
         print( key )

class SshOption( Model ):
   sshValues = List( help='SSH key option values', valueType=str )

class SshAuthKey( Model ):
   keyContents = Str( help='SSH key contents' )
   sshOptions = Dict( help='SSH key options, maps an SSH-option name to '
                           'a list of SSH-option values',
                      keyType=str, valueType=SshOption )

class SshAuthPrincipal( Model ):
   sshOptions = Dict( help='SSH principal options, maps an SSH-option name to '
                           'a list of SSH-option values',
                      keyType=str, valueType=SshOption )

class SshUserConfig( Model ):
   sshAuthKeys = Dict( help='SSH user key configuration, maps an SSH-key name to '
                            'SshAuthKey type',
                       keyType=str, valueType=SshAuthKey )
   sshAuthPrincipals = Dict( help='SSH user principal configuration, maps an '
                                  'SSH-principal name to SshAuthPrincipal type',
                             keyType=str, valueType=SshAuthPrincipal )

class SshUsers( Model ):
   users = Dict( help='List of SSH Users', keyType=str, valueType=SshUserConfig )

   def render( self ):
      outputStr = ""
      keyNameFmt = 'SSH Key: %s\n'
      principalFmt = 'SSH Principal: %s\n'
      keyContentsFmt = 'Public Key: %s\n'

      for username in sorted( self.users ):
         outputStr += f"User: {username}\n"
         keyConfig = self.users[ username ].sshAuthKeys
         principalConfig = self.users[ username ].sshAuthPrincipals
         for key in sorted( keyConfig ):
            outputStr += keyNameFmt % key
            if keyConfig[ key ].keyContents:
               outputStr += keyContentsFmt % keyConfig[ key ].keyContents
            outputStr = SysMgrLib.outputOptions( key, keyConfig, outputStr )
         for principal in sorted( principalConfig ):
            outputStr += principalFmt % principal
            outputStr = SysMgrLib.outputOptions( principal, principalConfig,
                  outputStr )
         outputStr += "\n"
      if outputStr:
         # Avoid the additional new line added to separate users
         outputStr = outputStr[ : -1 ]
         print( outputStr )
