#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import ConfigMount
import Tracing
from CliMode.Swag import (
   SwagMode,
   SwagTopoMode,
   SwagTopoMemberMode,
)
from CliPlugin import (
   FabricChannelIntfCli,
   VlanCli
)
from CliPlugin.SwagCliLib import (
   memberMatcher,
   memberIdTokenMatcher,
   memberIdValueMatcher,
   swagNonMember,
)

import Swag
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'Swag::SwagCli' )
t0 = traceHandle.trace0

SwagConfig = TacLazyType( 'SwagAgent::Config' )
gv = CliGlobal.CliGlobal( swagConfig=None )

#-------------------------------------------------------------------------------
# config-switch-aggregation mode
#-------------------------------------------------------------------------------
class ConfigSwagMode( SwagMode, BasicCli.ConfigModeBase ):
   name = "SWAG configuration"

   def __init__( self, parent, session ):
      SwagMode.__init__( self, param=None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def handleMemberId( self, args ):
      memberId = args.get( 'ID', None )
      if Swag.isMember():
         t0( 'member id ID is unsupported on members; ignoring' )
         memberId = None
      gv.swagConfig.memberId = memberId

#-------------------------------------------------------------------------------
# (config-switch-aggregation)# member id <ID>
#-------------------------------------------------------------------------------
class SwagMemberId( CliCommand.CliCommandClass ):
   """
   This command is used to configure the SWAG member ID when in candidate mode. It is
   not available once the device is a SWAG member as changing member ID requires
   reprovisioning.
   """
   syntax = 'member id ID'
   noOrDefaultSyntax = 'member id ...'
   data = {
      'member': memberMatcher,
      'id': memberIdTokenMatcher,
      'ID': memberIdValueMatcher,
   }

   handler = ConfigSwagMode.handleMemberId
   noOrDefaultHandler = handler

ConfigSwagMode.addCommandClass( SwagMemberId )

#-------------------------------------------------------------------------------
# (config-switch-aggregation)# control network vlan <VLAN_ID>
#-------------------------------------------------------------------------------
vlanMatcher = CliCommand.guardedKeyword(
   'vlan', helpdesc="Configure VLAN",
   guard=swagNonMember )

class SwagVlan( CliCommand.CliCommandClass ):
   """
   This command is used to configure the SWAG control network VLAN when in
   candidate mode. It is not available once the device is a SWAG member as
   changing VLAN ID requires reprovisioning.
   """
   syntax = 'control network vlan VLAN_ID'
   noOrDefaultSyntax = 'control network vlan...'
   data = {
      'control': 'Configure switch aggregation control-plane',
      'network': 'Configure switch aggregation control network',
      'vlan': vlanMatcher,
      'VLAN_ID': VlanCli.vlanIdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      if no:
         vlanId = None
      else:
         if Swag.isMember():
            t0( 'Configuring control network vlan is unsupported on members;'
                ' ignoring' )
            vlanId = None
         else:
            vlanId = args[ 'VLAN_ID' ].id

      if vlanId == gv.swagConfig.vlan:
         return
      warningStr = ( 'Control network VLAN must be the same on all switches in the'
                     ' aggregation group.\nSave configuration and reload for the'
                     ' VLAN change to take effect' )
      mode.addWarning( warningStr )
      gv.swagConfig.vlan = vlanId

   noOrDefaultHandler = handler

ConfigSwagMode.addCommandClass( SwagVlan )

#-------------------------------------------------------------------------------
# config-switch-aggregation-topo mode
#-------------------------------------------------------------------------------
class ConfigSwagTopoMode( SwagTopoMode, BasicCli.ConfigModeBase ):
   name = "SWAG topology configuration"

   def __init__( self, parent, session ):
      SwagTopoMode.__init__( self, param=None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# (config-switch-aggregation)# topology
#-------------------------------------------------------------------------------
class SwagTopo( CliCommand.CliCommandClass ):
   syntax = 'topology'
   noOrDefaultSyntax = syntax
   data = {
      'topology': 'Configure switch aggregation topology',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ConfigSwagTopoMode )
      mode.session_.gotoChildMode( childMode )
      t0( 'handleTopoMode' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.swagConfig.topologyConfig.topoConfig.clear()

ConfigSwagMode.addCommandClass( SwagTopo )

#-------------------------------------------------------------------------------
# config-switch-aggregation-topo-member mode
#-------------------------------------------------------------------------------
class ConfigSwagTopoMemberMode( SwagTopoMemberMode, BasicCli.ConfigModeBase ):
   name = "SWAG topology member configuration"

   def __init__( self, parent, session, memberId ):
      SwagTopoMemberMode.__init__( self, memberId )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# (config-switch-aggregation-topo)# member <ID>
#-------------------------------------------------------------------------------
topologyMemberMatcher = CliMatcher.KeywordMatcher(
   'member', helpdesc="Configure switch aggregation member" )

class SwagTopoMember( CliCommand.CliCommandClass ):
   syntax = 'member ID'
   noOrDefaultSyntax = syntax
   data = {
      'member': topologyMemberMatcher,
      'ID': memberIdValueMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      memberId = args[ 'ID' ]
      gv.swagConfig.topologyConfig.newTopoConfig( memberId )
      childMode = mode.childMode( ConfigSwagTopoMemberMode, memberId=memberId )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      memberId = args[ 'ID' ]
      if memberId in gv.swagConfig.topologyConfig.topoConfig:
         del gv.swagConfig.topologyConfig.topoConfig[ memberId ]

ConfigSwagTopoMode.addCommandClass( SwagTopoMember )

#-------------------------------------------------------------------------------
# (config-switch-aggregation-topo-member-<ID>)#
# interface FCINTF_ID member <NEIGHBOR_ID>
#-------------------------------------------------------------------------------
memberNeighMatcher = CliMatcher.KeywordMatcher(
   'member', helpdesc="Configure adjacent SWAG members for this SWAG member" )

class SwagTopoMemberIntf( CliCommand.CliCommandClass ):
   syntax = 'interface FCINTF_ID member NEIGHBOR_ID'
   noOrDefaultSyntax = syntax
   data = {
      'interface': 'Configure fabric channel interface',
      'FCINTF_ID': FabricChannelIntfCli.fabricChannelIntfMatcher,
      'member': memberNeighMatcher,
      'NEIGHBOR_ID': memberIdValueMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      newNeighbor = args[ 'NEIGHBOR_ID' ]
      if newNeighbor == mode.memberId:
         mode.addErrorAndStop( f'neighbor member {newNeighbor} should be different'
                               f' from current member {mode.memberId}' )

      newIntf = args[ 'FCINTF_ID' ].name
      topoConfig = gv.swagConfig.topologyConfig.topoConfig[ mode.memberId ]

      no = CliCommand.isNoOrDefaultCmd( args )
      if no:
         for oldNeighbor, oldIntf in topoConfig.neighbor.items():
            if oldNeighbor == newNeighbor and oldIntf == newIntf:
               # found the intf<->neighbor pair to be deleted
               del topoConfig.neighbor[ oldNeighbor ]
               return
      else:
         for oldNeighbor, oldIntf in topoConfig.neighbor.items():
            if oldIntf == newIntf and oldNeighbor != newNeighbor:
               # the Fabric-Channel used to connect to a different neighbor
               # remove the old neighbor
               del topoConfig.neighbor[ oldNeighbor ]
               break
         topoConfig.neighbor[ newNeighbor ] = newIntf

   noOrDefaultHandler = handler

ConfigSwagTopoMemberMode.addCommandClass( SwagTopoMemberIntf )

#-----------------------------------------------------------
# Have the ConfigAgent mount all needed state from sysdb
#-----------------------------------------------------------
def Plugin( entityManager ):
   gv.swagConfig = ConfigMount.mount( entityManager, SwagConfig.mountPath,
                                      "SwagAgent::Config", "w" )
