#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliGlobal
import ConfigMount
import LazyMount
import Tracing
from CliPlugin import IntfCli
from CliPlugin.LagCliLib import FabricChannelIntfId
from CliDynamicSymbol import CliDynamicPlugin
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'Swag::SwagCli' )
t0 = traceHandle.trace0

swagDynamicSubModes = CliDynamicPlugin( "SwagConfigCli" )

gv = CliGlobal.CliGlobal( swagConfig=None,
                          ethLagIntfConfigDir=None )
SwagConfig = TacLazyType( 'SwagAgent::Config' )

#-----------------------------------------------------------
# (config)# switch aggregation
#-----------------------------------------------------------
def swagCommandHandler( mode, args ):
   childMode = mode.childMode( swagDynamicSubModes.ConfigSwagMode )
   mode.session_.gotoChildMode( childMode )

def swagCommandDefaultHandler( mode, args ):
   gv.swagConfig.memberId = None
   gv.swagConfig.vlan = None
   # Not updating intfConfig as that's controlled by 'channel-group
   # fabric ...' configuration and not part of swag mode config.

# -------------------------------------------------------------------------------
# (config-if-Fc<MEMBER_ID/CHANNEL_ID>)# peer member id <PEER_ID>
# -------------------------------------------------------------------------------
def removeFcIntfConfigs( fcIntf ):
   if not FabricChannelIntfId.isFabricChannelIntfId( fcIntf ):
      return
   memberId = FabricChannelIntfId.memberId( fcIntf )
   topoConfig = gv.swagConfig.topologyConfig.topoConfig.get( memberId, None )
   if topoConfig:
      for peerId, intf in topoConfig.neighbor.items():
         if fcIntf == intf:
            # currently there is only one Fabric-Channel between a pair of
            # members
            del topoConfig.neighbor[ peerId ]
            break
      if not topoConfig.neighbor:
         del gv.swagConfig.topologyConfig.topoConfig[ memberId ]

def fcIntfCommandHandler( mode, args ):
   fcIntf = mode.intf.name
   memberId = FabricChannelIntfId.memberId( fcIntf )
   no = CliCommand.isNoOrDefaultCmd( args )
   # currently, for a SWAG source member, there is one-to-one
   # relationship between Fabric-Channel and peer member id
   if no:
      t0( 'remove intf', fcIntf )
      removeFcIntfConfigs( fcIntf )
   else:
      peerId = args[ 'PEER_ID' ]
      if peerId == memberId:
         mode.addErrorAndStop( f'peer member {peerId} should be different'
                               f' from current member {memberId}' )
      topoConfig = gv.swagConfig.topologyConfig.newTopoConfig( memberId )
      t0( 'create topoConfig for', fcIntf )
      for oldPeer, oldIntf in topoConfig.neighbor.items():
         if oldIntf == fcIntf or oldPeer == peerId:
            # the Fabric-Channel used to connect to a different peer
            # remove the old peer
            del topoConfig.neighbor[ oldPeer ]
            break
      topoConfig.neighbor[ peerId ] = fcIntf

class FcIntfConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing entries from
   swagConfig.topologyConfig.topoConfig when the associated
   Fabric-Channel interface is removed."""

   def setDefault( self ):
      removeFcIntfConfigs( self.intf_.name )

#-----------------------------------------------------------
# Have the ConfigAgent mount all needed state from sysdb
#-----------------------------------------------------------
def Plugin( entityManager ):
   gv.swagConfig = ConfigMount.mount( entityManager,
                                      SwagConfig.mountPath,
                                      'SwagAgent::Config',
                                      'w' )
   gv.ethLagIntfConfigDir = LazyMount.mount( entityManager,
                                             'interface/config/eth/lag',
                                             'Interface::EthLagIntfConfigDir',
                                             'r' )
   # Register a dependent class to perform customized actions when a
   # Fabric-Channel is deleted.
   priority = 10 # arbitrarily chosen priority.
   IntfCli.Intf.registerDependentClass( FcIntfConfigCleaner,
                                        priority )
