# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
      Ip4Address,
      MacAddress,
)
from CliModel import (
      Dict,
      Enum,
      Int,
      Model,
)
from IntfModels import Interface
import TableOutput
import Tac

Role = Tac.Type( 'Swag::Role' )

class MemberStatus( Model ):
   role = Enum( values=Role.attributes,
                help='Role of member' )
   ipv4Address = Ip4Address( help='Member IP address' )
   macAddress = MacAddress( help='Member MAC address' )

class SwagSummary( Model ):
   """
   show switch aggregation summary
   """
   candidateId = Int( help='Identifier for this member', optional=True )
   activeSupervisorId = Int( help='Identifier for the active supervisor in the SWAG',
                             optional=True )
   vlan = Int( help='Control network VLAN', optional=True )
   members = Dict( keyType=int,
                   valueType=MemberStatus,
                   help='SWAG member information' )

   def render( self ):
      '''
      #show switch aggregation summary
      Active supervisor: 2
      Control network VLAN: 4090
      Number of members: 4

      Member Role         IP Address   MAC Address
      ------ ---------- ------------  --------------
           1 supervisor    127.1.0.1  021c.a700.0001
           2 supervisor    127.1.0.2  021c.a700.0002
           3     worker    127.1.0.3  021c.a700.0003
          96     worker   127.1.0.96  021c.a700.0060
      '''
      roleForText = {
            Role.supervisorSwitch: 'supervisor',
            Role.supervisorServer: 'supervisor',
            Role.candidate: Role.candidate,
            Role.worker: Role.worker,
      }
      if self.candidateId is not None:
         print( f'Candidate: {self.candidateId}' )
      if self.activeSupervisorId is not None:
         print( f'Active supervisor: {self.activeSupervisorId}' )
      if self.vlan is not None:
         print( f'Control network VLAN: {self.vlan}' )
      print( f'Number of members: {len(self.members)}' )
      if self.members:
         headers = ( 'Member', 'Role', 'IP Address', 'MAC Address' )
         table = TableOutput.createTable( headers )
         right = TableOutput.Format( justify='right' )
         right.padLimitIs( True )

         left = TableOutput.Format( justify='left' )
         left.noPadLeftIs( True )
         left.padLimitIs( True )

         table.formatColumns( right, left, left, left )
         for memberId in sorted( self.members ):
            status = self.members[ memberId ]
            table.newRow( memberId, roleForText[ status.role ],
                          status.ipv4Address, status.macAddress.displayString )
         print( table.output() )

class ReachabilityIntfStatus( Model ):
   interface = Interface( help='Name of the interface' )
   status = Enum( values=( 'configured', 'pending', 'installed', 'failed', 'stale' ),
                  help='Installation status of reachability information' )

   @classmethod
   def factory_( cls, mid, egressIntf, reachabilityIntfStatus ):
      model = cls()

      if egressIntf is None:
         # Generate a "stale" entry when PD has yet to uninstall an entry which has
         # been deconfigured.
         model.status = 'stale'
         model.interface = reachabilityIntfStatus.intfId
         return model

      # Determine the status. If PD has not acknowledged the request from PI, the
      # status is 'configured'. Otherwise, the PD "reachabilityState" is
      # authoritative.
      model.status = 'configured'
      model.interface = egressIntf

      if reachabilityIntfStatus is None:
         return model

      if egressIntf == reachabilityIntfStatus.intfId:
         state = reachabilityIntfStatus.reachabilityState
         if state == 'reachabilityPending':
            model.status = 'pending'
         elif state == 'reachabilityFailed':
            model.status = 'failed'
         elif state == 'reachabilitySucceeded':
            model.status = 'installed'
         else:
            assert False, f'Unexpected reachabilityState={state}'
      return model

class MemberReachability( Model ):
   reachability = Dict( keyType=int,
         valueType=ReachabilityIntfStatus,
         help='Reachability to every SWAG member' )

   @classmethod
   def factory_( cls, memberReachability, memberReachabilityStatus ):
      merged = {}
      if memberReachability is not None:
         for mid, egressIntf in memberReachability.egressIntf.items():
            ris = None
            if memberReachabilityStatus is not None:
               ris = memberReachabilityStatus.reachabilityState.get( mid )
            merged[ mid ] = ( egressIntf, ris )
      if memberReachabilityStatus is not None:
         for mid, reachabilityIntfStatus in \
               memberReachabilityStatus.reachabilityState.items():
            intfId = None
            if memberReachability is not None:
               intfId = memberReachability.egressIntf.get( mid )
            merged[ mid ] = ( intfId, reachabilityIntfStatus )

      model = cls()
      for mid, ( intfId, reachabilityIntfStatus ) in merged.items():
         model.reachability[ mid ] = ReachabilityIntfStatus.factory_(
                  mid,
                  intfId,
                  reachabilityIntfStatus )

      return model

class SwagReachability( Model ):
   """
   show switch aggregation reachability
   """
   memberReachabilities = Dict(
         keyType=int,
         valueType=MemberReachability,
         help='Reachability for each SWAG member to every other SWAG member' )

   @classmethod
   def factory_( cls, memberReachabilityDir, memberReachabilityStatusDir ):
      merged = {}
      for mid in set( memberReachabilityDir ) | set( memberReachabilityStatusDir ):
         merged[ mid ] = ( memberReachabilityDir.get( mid ),
                           memberReachabilityStatusDir.get( mid ) )

      model = cls()
      for mid, ( mr, mrs ) in merged.items():
         model.memberReachabilities[
               int( mid ) ] = MemberReachability.factory_( mr, mrs )
      return model

   def render( self ):
      '''
      #show switch aggregation reachability
      Src ID Dest ID Interface           Status
      ------ ------- ------------------- ------------
      1      2       Fabric-Channel3/100  configured
      1      3       Fabric-Channel3/100  pending
      1      96      Fabric-Channel3/100  installed
      2      1       Fabric-Channel2/100  failed
      2      3       Fabric-Channel2/100  installed
      2      96      Fabric-Channel2/100  installed
      3      1       Fabric-Channel3/100  installed
      3      2       Fabric-Channel3/200  installed
      3      96      Fabric-Channel3/200  installed
      96     1       Fabric-Channel96/100 installed
      96     2       Fabric-Channel96/200 installed
      96     3       Fabric-Channel96/100 installed
      '''
