# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from ArnetModel import IpGenericAddress
from CliModel import Model, Int, Dict, Bool
from IntfModels import Interface
import LazyMount
import TableOutput

twampLightStatus = None

class TwampLightReflectorCountersPerIp( Model ):
   # TODO: BUG856901 Add dropped packet statistics
   rx = Int( help="Packets received on reflector" )
   tx = Int( help="Packets transferred from reflector" )

   def addToTable( self, ipAddr, intf, table ):
      table.newRow( ipAddr,
                    intf,
                    self.rx,
                    self.tx )

   @staticmethod
   def create( ipIntfAddr ):
      res = TwampLightReflectorCountersPerIp()
      res.rx = ipIntfAddr.rxPkt
      res.tx = ipIntfAddr.txPkt

      return res

class TwampLighSenderCountersPerIp( Model ):
   # TODO: BUG856901 Add dropped packet statistics
   tx = Int( help="Packets received on sender" )
   rx = Int( help="Packets transferred from sender" )

   def addToTable( self, ipAddr, intf, table ):
      table.newRow( ipAddr,
                    intf,
                    self.rx,
                    self.tx )

   @staticmethod
   def create( ipIntfAddr ):
      res = TwampLighSenderCountersPerIp()
      res.rx = ipIntfAddr.rxPkt
      res.tx = ipIntfAddr.txPkt

      return res

class TwampLightCountersPerInterface( Model ):
   reflectors = Dict( keyType=IpGenericAddress,
                      valueType=TwampLightReflectorCountersPerIp,
                      help="Twamp Light reflector counters, keyed by IP Address" )
   senders = Dict( keyType=IpGenericAddress,
                   valueType=TwampLighSenderCountersPerIp,
                   help="Twamp Light sender counters, keyed by IP Address" )
   dropTotal = Int( help="Twamp Light drop counters", default=0 )
   dropInvalid = Int( help="Invalid packet drop counters", default=0 )
   dropTooSmall = Int( help="Too small packet drop counters", default=0 )
   dropWrongDstPort = Int( help="Wrong dst port packet drop counters", default=0 )
   dropInternalError = Int( help="Internal error packet drop counters", default=0 )
   dropWrongEthType = Int( help="Wrong EtherType packet drop counters", default=0 )
   dropUnknownIP = Int( help="Unknown IP packet drop counters", default=0 )
   dropIncorrectTimestamps = Int( help="Incorrect timestamps packet drop counters",
                                  default=0 )

class TwampLightCounters( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=TwampLightCountersPerInterface,
                      help="Twamp Light counters, keyed by Interface" )
   _renderReflector = Bool( help="Show only reflector statistics" )
   _renderSender = Bool( help="Show only sender statistics" )
   _renderDrop = Bool( help="Show only drop statistics" )
   _renderDetailedDrop = Bool( help="Show detailed drop statistics" )
   _renderAll = Bool( help="Show all statistics" )

   def render( self ):
      firstColumnFormat = TableOutput.Format( justify="left", minWidth=12 )
      firstColumnFormat.noPadLeftIs( True )
      firstColumnFormat.padLimitIs( True )
      interfaceColumnFormat = TableOutput.Format( justify="left", minWidth=11 )
      interfaceColumnFormat.noPadLeftIs( True )
      interfaceColumnFormat.padLimitIs( True )
      pktColumnFormat = TableOutput.Format( justify="right", minWidth=8 )
      pktColumnFormat.padLimitIs( True )
      dropPktColumnFormat = TableOutput.Format( justify="right", minWidth=15 )
      dropPktColumnFormat.padLimitIs( True )

      if self._renderSender or self._renderAll:
         # This adds table headings if both tables are to be displayed
         if self._renderAll:
            print( "Sender statistics\n" )
         senderTable = TableOutput.createTable(
            [ 'IP Address', 'Interface', 'Rx Pkts', 'Tx Pkts' ] )
         senderTable.formatColumns( firstColumnFormat,
                                    interfaceColumnFormat,
                                    pktColumnFormat,
                                    pktColumnFormat )
         for intf in Arnet.sortIntf( self.interfaces ):
            for addr in self.interfaces[ intf ].senders:
               counters = self.interfaces[ intf ].senders[ addr ]
               counters.addToTable( addr, intf, senderTable )
         print( senderTable.output() )

         # This adds additional whitespace to separate tables
         if self._renderAll:
            print( "" )

      if self._renderReflector or self._renderAll:
         # This adds table headings if both tables are to be displayed
         if self._renderAll:
            print( "Reflector statistics\n" )
         reflectorTable = TableOutput.createTable(
            [ 'IP Address', 'Interface', 'Rx Pkts', 'Tx Pkts' ] )
         reflectorTable.formatColumns( firstColumnFormat,
                                       interfaceColumnFormat,
                                       pktColumnFormat,
                                       pktColumnFormat )
         for intf in Arnet.sortIntf( self.interfaces ):
            for addr in self.interfaces[ intf ].reflectors:
               counters = self.interfaces[ intf ].reflectors[ addr ]
               counters.addToTable( addr, intf, reflectorTable )
         print( reflectorTable.output() )

         # This adds additional whitespace to separate tables
         if self._renderAll:
            print( "" )

      if self._renderDetailedDrop:
         firstIntf = True
         for intf in Arnet.sortIntf( self.interfaces ):
            if not firstIntf:
               # Additional blank line before every intf except the first one
               print( "" )
            else:
               firstIntf = False
            print( f"Interface: {intf}" )
            print( f"Invalid TWAMP packet: {self.interfaces[ intf ].dropInvalid}" )
            print( f"Packet too small: {self.interfaces[ intf ].dropTooSmall}" )
            print( f"Wrong dst port: {self.interfaces[ intf ].dropWrongDstPort}" )
            print( f"Internal error: {self.interfaces[ intf ].dropInternalError}" )
            print( f"Wrong EtherType: {self.interfaces[ intf ].dropWrongEthType}" )
            print( f"Unknown IP address: {self.interfaces[ intf ].dropUnknownIP}" )
            print( "Incorrect timestamps: "
                   f"{self.interfaces[ intf ].dropIncorrectTimestamps}" )
      elif self._renderDrop or self._renderAll:
         if self._renderAll:
            print( "Dropped packets statistics\n" )
         dropTable = TableOutput.createTable(
            [ 'Interface', 'Rx Dropped Pkts' ] )
         dropTable.formatColumns( interfaceColumnFormat,
                                  dropPktColumnFormat )
         for intf in Arnet.sortIntf( self.interfaces ):
            dropTable.newRow( intf, self.interfaces[ intf ].dropTotal )
         print( dropTable.output() )

   @staticmethod
   def create( mode, args ):
      renderAll = False
      renderSenderOnly = 'sender' in args
      renderReflectorOnly = 'reflector' in args
      renderDropOnly = 'drop' in args
      renderDropDetailed = 'detail' in args
      if not renderSenderOnly and not renderReflectorOnly and not renderDropOnly:
         renderAll = True
      res = TwampLightCounters( _renderReflector=renderReflectorOnly,
                                _renderSender=renderSenderOnly,
                                _renderDrop=renderDropOnly,
                                _renderDetailedDrop=renderDropDetailed,
                                _renderAll=renderAll )
      for intfStatus in twampLightStatus.intfStatus.values():
         res.interfaces[ intfStatus.intfId ] = TwampLightCountersPerInterface()
         if renderDropOnly or renderAll:
            res.interfaces[ intfStatus.intfId ].dropTotal =\
               intfStatus.getTotalDroppedPktCount()
            res.interfaces[ intfStatus.intfId ].dropInvalid =\
               intfStatus.droppedPktInvalid
            res.interfaces[ intfStatus.intfId ].dropTooSmall =\
               intfStatus.droppedPktTooSmall
            res.interfaces[ intfStatus.intfId ].dropWrongDstPort =\
               intfStatus.droppedPktWrongDstPort
            res.interfaces[ intfStatus.intfId ].dropInternalError =\
               intfStatus.droppedPktInternalError
            res.interfaces[ intfStatus.intfId ].dropWrongEthType =\
               intfStatus.droppedPktWrongEthType
            res.interfaces[ intfStatus.intfId ].dropUnknownIP =\
               intfStatus.droppedPktUnknownIP
            res.interfaces[ intfStatus.intfId ].dropIncorrectTimestamps =\
               intfStatus.droppedPktIncorrectTimestamps
         for addr, ipIntfAddr in intfStatus.reflectorIpAddress.items():
            if renderReflectorOnly or renderAll:
               twampLightReflectorCounters =\
                  TwampLightReflectorCountersPerIp.create( ipIntfAddr )
               res.interfaces[ intfStatus.intfId ].reflectors[ addr ] =\
                  twampLightReflectorCounters
         for addr, ipIntfAddr in intfStatus.senderIpAddress.items():
            if renderSenderOnly or renderAll:
               twampLightSenderCounters =\
                  TwampLighSenderCountersPerIp.create( ipIntfAddr )
               res.interfaces[ intfStatus.intfId ].senders[ addr ] =\
                  twampLightSenderCounters

      return res

def Plugin( entityManager ):
   global twampLightStatus

   twampLightStatus = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/status',
      'Twamp::Light::Status', 'r' )
