# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from ArnetModel import IpGenericAddress
from CliModel import Model, Int, Dict, Str, Float
from CliPlugin.TwampLightShowCommands import gv
from IntfModels import Interface
import LazyMount
import TableOutput
import Smash
import Tac

twampLightStatus = None
twampLightConfig = None
ethPhyIntfConfigDir = None
ethLagIntfConfigDir = None

def doMount( path, entryType, mountInfo, shmemEm=None ):
   if not shmemEm:
      shmemEm = gv.shmemEm
   mountGroup = shmemEm.getMountGroup()
   entity = mountGroup.doMount(
      path, entryType, mountInfo )
   mountGroup.doClose()
   return entity

def getProfileName( monitorTarget ):
   if not monitorTarget:
      return ""
   for profileName in twampLightConfig.senderIdMap.keys():
      if twampLightConfig.senderIdMap[ profileName ] == monitorTarget.senderId:
         return profileName
   return ""

class Result( Model ):
   intfId = Interface( help="Name of the interface" )
   oneWayMinDelay = Int( help="One way min delay in microseconds", default=0 )
   twoWayMinDelay = Int( help="Two way min delay in microseconds", default=0 )

   def addToTable( self, ipAddr, table ):
      table.newRow( ipAddr,
                    self.intfId.stringValue,
                    self.oneWayMinDelay,
                    self.twoWayMinDelay )

   @staticmethod
   def create( result ):
      res = Result()
      res.intfId = result.txIntfId
      res.oneWayMinDelay = result.oneWayMinDelay
      res.twoWayMinDelay = result.twoWayMinDelay

      return res

class ResultDetailed( Model ):
   intfId = Interface( help="Name of the interface" )
   intfDescription = Str( help="Interface description" )
   senderProfile = Str( help="Sender profile name" )
   samplingRate = Float( help="Effective sampling rate" )
   pktTx = Int( help="Packets sent", default=0 )
   pktRx = Int( help="Packets received", default=0 )
   oneWayMinDelay = Int( help="One way min delay in microseconds", default=0 )
   twoWayMinDelay = Int( help="Two way min delay in microseconds", default=0 )
   oneWayMaxDelay = Int( help="One way max delay in microseconds", default=0 )
   twoWayMaxDelay = Int( help="Two way max delay in microseconds", default=0 )
   oneWayAvgDelay = Int( help="One way avg delay in microseconds", default=0 )
   twoWayAvgDelay = Int( help="Two way avg delay in microseconds", default=0 )
   oneWayDelayVariation = Int( help="One way min variation in microseconds",
                               default=0 )
   twoWayDelayVariation = Int( help="Two way min variation in microseconds",
                               default=0 )

   def addToTable( self, table ):
      table.newRow( "One-way delay",
                    self.oneWayMinDelay,
                    self.oneWayMaxDelay,
                    self.oneWayAvgDelay,
                    self.oneWayDelayVariation )
      table.newRow( "Two-way delay",
                    self.twoWayMinDelay,
                    self.twoWayMaxDelay,
                    self.twoWayAvgDelay,
                    self.twoWayDelayVariation )

   @staticmethod
   def create( result ):
      res = ResultDetailed()
      res.intfId = result.txIntfId
      intfConfig = ethPhyIntfConfigDir.intfConfig.get( result.txIntfId )
      if not intfConfig:
         intfConfig = ethLagIntfConfigDir.intfConfig.get( result.txIntfId )
      res.intfDescription = intfConfig.description if intfConfig else ""
      res.samplingRate = twampLightStatus.intfStatus[ res.intfId ].txDelay
      intfStatus = twampLightStatus.intfStatus.get( result.txIntfId )
      if intfStatus:
         senderIpAddressStatus = intfStatus.senderIpAddress.get( result.address )
         if senderIpAddressStatus:
            res.pktRx = senderIpAddressStatus.rxPkt
            res.pktTx = senderIpAddressStatus.txPkt
      res.oneWayMinDelay = result.oneWayMinDelay
      res.twoWayMinDelay = result.twoWayMinDelay
      res.oneWayMaxDelay = result.oneWayMaxDelay
      res.twoWayMaxDelay = result.twoWayMaxDelay
      res.oneWayAvgDelay = result.oneWayAvgDelay
      res.twoWayAvgDelay = result.twoWayAvgDelay
      res.oneWayDelayVariation = result.oneWayDelayVariation
      res.twoWayDelayVariation = result.twoWayDelayVariation

      return res

class ResultsModel( Model ):
   addresses = Dict( keyType=IpGenericAddress,
                     valueType=Result,
                     help="TWAMP light results, keyed by address" )

   def render( self ):
      ipAddressColumnFormat = TableOutput.Format( justify="left", minWidth=12 )
      ipAddressColumnFormat.noPadLeftIs( True )
      ipAddressColumnFormat.padLimitIs( True )
      interfaceColumnFormat = TableOutput.Format( justify="left", minWidth=11 )
      interfaceColumnFormat.noPadLeftIs( True )
      interfaceColumnFormat.padLimitIs( True )
      minDelayColumnFormat = TableOutput.Format( justify="right", minWidth=15 )
      minDelayColumnFormat.padLimitIs( True )

      table = TableOutput.createTable( [ 'IP Address', 'Interface',
                                          'One Way\nMinimum Delay',
                                          'Two Way\nMinimum Delay' ] )
      table.formatColumns( ipAddressColumnFormat,
                           interfaceColumnFormat,
                           minDelayColumnFormat,
                           minDelayColumnFormat )
      for address in Arnet.sortIntf( self.addresses ):
         result = self.addresses[ address ]
         result.addToTable( address, table )
      print( "Unit: microseconds\n" )
      print( table.output() )

   @staticmethod
   def create( mode, args ):
      filterIpAddress = args.get( 'IP' )

      res = ResultsModel()
      for path in gv.dirMounter.mountedPaths.path:
         mountedResult = doMount( path, 'Twamp::Light::Result',
                                  Smash.mountInfo( 'reader' ) )
         for item in mountedResult.monitorResult.values():
            if not item.valid:
               continue
            if ( filterIpAddress == item.address ) or not filterIpAddress:
               result = Result.create( item )
               res.addresses[ item.address ] = result
      return res

class DetailedResultsModel( Model ):
   addressesDetailed = Dict( keyType=IpGenericAddress,
                             valueType=ResultDetailed,
                             help="TWAMP light detailed results, keyed by address" )
   def render( self ):
      for address in Arnet.sortIntf( self.addressesDetailed ):
         resultDetailed = self.addressesDetailed[ address ]
         print( f"IP address: {address}" )
         print( f"Interface: {resultDetailed.intfId.stringValue}" )
         print( f"Description: {resultDetailed.intfDescription}" )
         print( f"Sender profile name: {resultDetailed.senderProfile}" )
         samplingRateStr = "Effective sampling rate: " +\
            f"1 packet every {resultDetailed.samplingRate} second"
         if resultDetailed.samplingRate > 1.0:
            samplingRateStr += "s"
         print( samplingRateStr )
         print( f"Packets sent: {resultDetailed.pktTx}" )
         print( f"Packets received: {resultDetailed.pktRx}" )
         measurementColumnFormat =\
            TableOutput.Format( justify="left", minWidth=12 )
         measurementColumnFormat.noPadLeftIs( True )
         measurementColumnFormat.padLimitIs( True )
         minColumnFormat = TableOutput.Format( justify="right", minWidth=4 )
         minColumnFormat.padLimitIs( True )
         varianceColumnFormat = TableOutput.Format( justify="right", minWidth=9 )
         varianceColumnFormat.padLimitIs( True )

         table = TableOutput.createTable( [ 'Measurement', 'Min', 'Max',
                                             'Avg', 'Variance' ] )
         table.formatColumns( measurementColumnFormat,
                              minColumnFormat,
                              minColumnFormat,
                              minColumnFormat,
                              varianceColumnFormat )
         resultDetailed.addToTable( table )
         print( "\nUnit: microseconds\n" )
         print( table.output() )

   @staticmethod
   def create( mode, args ):
      filterIpAddress = args.get( 'IP' )

      res = DetailedResultsModel()
      for path in gv.dirMounter.mountedPaths.path:
         mountedResult = doMount( path, 'Twamp::Light::Result',
                                  Smash.mountInfo( 'reader' ) )
         inputPath = "monitor/twamp/twampLight/input/"
         clientName = path.split( '/' )[ -1 ]
         newPath = inputPath + clientName
         clientMonitorTargets = doMount(
            newPath, 'Twamp::Light::ClientMonitorTargets',
            Smash.mountInfo( 'shadow' ) )
         for item in mountedResult.monitorResult.values():
            monitorTargetKey = Tac.Value( 'Twamp::Light::MonitorTargetKey',
                                           item.address )
            monitorTarget =\
               clientMonitorTargets.monitorTarget.get( monitorTargetKey )
            senderProfileName = getProfileName( monitorTarget )
            if not item.valid:
               continue
            if ( filterIpAddress == item.address ) or not filterIpAddress:
               resultDetailed = ResultDetailed.create( item )
               resultDetailed.senderProfile = senderProfileName

               senderProfile =\
                  twampLightConfig.senderProfile.get( senderProfileName )
               if senderProfile.offset >= senderProfile.significance:
                  resultDetailed.senderProfile +=\
                     " (invalid: Profile configuration is invalid)"

               res.addressesDetailed[ item.address ] = resultDetailed

      return res

def Plugin( entityManager ):
   global twampLightStatus
   global twampLightConfig
   global ethPhyIntfConfigDir
   global ethLagIntfConfigDir

   twampLightStatus = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/status',
      'Twamp::Light::Status', 'r' )
   twampLightConfig = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/config',
      'Twamp::Light::Config', 'r' )
   ethPhyIntfConfigDir = LazyMount.mount(
      entityManager, 'interface/config/eth/phy/all',
      'Interface::AllEthPhyIntfConfigDir', 'r' )
   ethLagIntfConfigDir = LazyMount.mount(
      entityManager, 'interface/config/eth/lag',
      'Interface::EthLagIntfConfigDir', 'r' )
