# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Tac
import LazyMount
import EthIntfLib
from CliDynamicSymbol import CliDynamicPlugin
import CliPlugin.XcvrAllStatusDir
from XcvrLib import isCmisTransceiver, getSlotId
from TypeFuture import TacLazyType
import CliGlobal

CmisCdbTrigger = TacLazyType( "Xcvr::CmisCdbTrigger" )
CmisFwAccessMethod = TacLazyType( "Xcvr::CmisFwAccessMethod" )
CmisFwPasswordType = TacLazyType( "Xcvr::CmisFwPasswordType" )
XcvrPresence = TacLazyType( "Xcvr::XcvrPresence" )

xcvrCdbCapsModel = CliDynamicPlugin( 'XcvrCdbCapabilitiesModel' )
gv = CliGlobal.CliGlobal( xcvrStatusDir=None, cmisFwStatusSliceDir=None,
                          cmisCdbCapabilitiesSliceDir=None,
                          xcvrCentralStatus=None )

# Mappings between the TAC Enum values used by this command, and the enum values
# defined in the CLI model file
cdbCmdTriggers = {
   CmisCdbTrigger.cdbTriggerCmdIdWrite: "commandIdWrite",
   CmisCdbTrigger.cdbTriggerFullCmdWrite: "fullCommandWrite"
}

fwPswdTypes = {
   CmisFwPasswordType.fwPasswordNotApplicable: "notApplicable",
   CmisFwPasswordType.fwVendorPassword: "vendorPassword",
   CmisFwPasswordType.fwVendorPasswordSequence: "vendorPasswordSequence",
   CmisFwPasswordType.fwMsaPassword: "msaPassword"
}

fwAccessMethods = {
   CmisFwAccessMethod.fwAccessNotSupported: "none",
   CmisFwAccessMethod.fwAccessLpl: "lpl",
   CmisFwAccessMethod.fwAccessEpl: "epl",
   CmisFwAccessMethod.fwAccessLplAndEpl: "lplAndEpl"
}

# Helper class used for retrieving the necessary data for CLI
class CmisCdbCliHelper():
   def __init__( self, xcvrStatus ):
      self.xcvrStatus = xcvrStatus
      self.xcvrName = xcvrStatus.name if xcvrStatus else ""

   def sliceName( self ):
      isModular = Cell.cellType() == "supervisor"
      if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId(
         self.xcvrName ):
         return EthIntfLib.sliceName( self.xcvrName )
      else:
         return "FixedSystem"

   def getCdbCaps( self ):
      cmisCdbCapsDir = gv.cmisCdbCapabilitiesSliceDir[ self.sliceName() ]
      return cmisCdbCapsDir.cmisCdbCapabilities.get( self.xcvrName )

   def getFwCaps( self ):
      fwStatusDir = gv.cmisFwStatusSliceDir[ self.sliceName() ]
      fwStatus = fwStatusDir.cmisFwStatus.get( self.xcvrName )
      if not fwStatus:
         return None
      return fwStatus.firmwareCapabilities

def _populateModuleCdbCaps( cdbCapsDetailsModel, cdbCaps ):
   '''
   Helper function that populates the CLI model with module-wide CDB capabilities
   from Sysdb.
   '''
   moduleCdbCapsInfo = xcvrCdbCapsModel.TransceiverCdbCapabilitiesModuleInfo()
   moduleCdbCapsInfo.cdbInstancesSupported = Tac.enumValue(
      "Xcvr::CmisCdbSupport", cdbCaps.cdbInstancesSupported )
   moduleCdbCapsInfo.backgroundModeSupport = cdbCaps.cdbBackgroundModeSupported
   moduleCdbCapsInfo.autoPagingSupport = cdbCaps.cdbAutoPagingSupported
   moduleCdbCapsInfo.maxEplPagesSupported = Tac.enumValue( "Xcvr::CmisCdbEplSupport",
                                                           cdbCaps.cdbMaxEplPages )
   moduleCdbCapsInfo.maxEplTransactionSize = cdbCaps.maxEplTransactionSize
   moduleCdbCapsInfo.maxLplTransactionSize = cdbCaps.maxLplTransactionSize
   moduleCdbCapsInfo.cdbCommandExecutionTrigger = cdbCmdTriggers.get(
      cdbCaps.cdbCommandTrigger )
   moduleCdbCapsInfo.maxCompletionTime = cdbCaps.cdbMaxCompletionTime
   moduleCdbCapsInfo.cdbMaxBusyTime = cdbCaps.getCdbMaxBusyTime()
   for cmd, cmdSup in cdbCaps.cdbCmdSupport.items():
      cmdStr = f"{cmd:0{4}X}h"
      moduleCdbCapsInfo.cdbModuleCommandsSupported[ cmdStr ] = cmdSup
   cdbCapsDetailsModel.moduleCdbCapabilitiesAdvertised = moduleCdbCapsInfo

def _populateFwCdbCaps( cdbCapsDetailsModel, fwCaps ):
   '''
   Helper function that populates the CLI model with firmware management capabilities
   from Sysdb.
   '''
   fwCdbCapsInfo = xcvrCdbCapsModel.TransceiverCdbCapabilitiesFirmwareManagement()
   # We still want to pass an empty model even if we don't have capabilities, but
   # only in the case when there is an override applied. The caller of this function
   # is the one figuring out whether we need a model or not. Here we just populate
   # the model.
   if fwCaps is not None:
      fwCdbCapsInfo.firmwarePasswordType = fwPswdTypes.get( fwCaps.fwPasswordType )
      fwCdbCapsInfo.firmwareImageReadbackSupport = fwCaps.imageReadbackSupported
      fwCdbCapsInfo.skipFirmwareErasedBlock = fwCaps.skipErasedBlocks
      fwCdbCapsInfo.firmwareCopyCommandSupport = fwCaps.fwCopyCmdSupported
      fwCdbCapsInfo.firmwareAbortCommandSupport = fwCaps.fwAbortCmdSupported
      fwCdbCapsInfo.firmwareStartCommandPayloadSize = fwCaps.startCmdPayloadSize
      fwCdbCapsInfo.firmwareErasedByteMarker = fwCaps.erasedByte
      fwCdbCapsInfo.firmwareMaxEplTransactionSize = fwCaps.fwMaxEplTransactionSize
      fwCdbCapsInfo.firmwareMaxLplTransactionSize = fwCaps.fwMaxLplTransactionSize
      fwCdbCapsInfo.firmwareWriteMethod = fwAccessMethods.get( fwCaps.fwWriteMethod )
      fwCdbCapsInfo.firmwareReadMethod = fwAccessMethods.get( fwCaps.fwReadMethod )
      fwCdbCapsInfo.firmwareHitlessRestartSupported = \
         fwCaps.hitlessFwRestartSupported
      fwCdbCapsInfo.firmwareStartCommandMaxDuration = fwCaps.startMaxDuration
      fwCdbCapsInfo.firmwareAbortCommandMaxDuration = fwCaps.abortMaxDuration
      fwCdbCapsInfo.firmwareWriteCommandMaxDuration = fwCaps.fwWriteMaxDuration
      fwCdbCapsInfo.firmwareCompleteCommandMaxDuration = fwCaps.completeMaxDuration
      fwCdbCapsInfo.firmwareCopyCommandMaxDuration = fwCaps.copyMaxDuration
   cdbCapsDetailsModel.firmwareManagementCdbCapabilitiesAdvertised = fwCdbCapsInfo

def _populateCdbOverrides( cdbCapsDetailsModel, ovProfile ):
   '''
   Helper function that populates the CLI model with generic CDB and FW-management
   CDB override data from Sysdb.
   '''
   cdbOverrides = xcvrCdbCapsModel.TransceiverCdbCapabilitiesModuleInfo()
   fwCdbOverrides = xcvrCdbCapsModel.TransceiverCdbCapabilitiesFirmwareManagement()
   # Generic CDB override data
   if ovProfile.cdbInstancesSupported is not None:
      cdbOverrides.cdbInstancesSupported = Tac.enumValue(
         "Xcvr::CmisCdbSupport", ovProfile.cdbInstancesSupported )
   cdbOverrides.backgroundModeSupport = ovProfile.cdbBackgroundModeSupported
   cdbOverrides.autoPagingSupport = ovProfile.cdbAutoPagingSupported
   if ovProfile.cdbMaxEplPages is not None:
      cdbOverrides.maxEplPagesSupported = Tac.enumValue( "Xcvr::CmisCdbEplSupport",
                                                         ovProfile.cdbMaxEplPages )
   cdbOverrides.maxEplTransactionSize = ovProfile.maxEplTransactionSize
   cdbOverrides.maxLplTransactionSize = ovProfile.maxLplTransactionSize
   cdbOverrides.cdbCommandExecutionTrigger = cdbCmdTriggers.get(
      ovProfile.cdbCommandTrigger )
   cdbOverrides.maxCompletionTime = ovProfile.cdbMaxCompletionTime
   cdbOverrides.cdbMaxBusyTime = ovProfile.cdbMaxBusyTime
   for cmd, cmdSup in ovProfile.cdbCmdSupport.items():
      cmdStr = f"{cmd.cmd:0{4}X}h"
      cdbOverrides.cdbModuleCommandsSupported[ cmdStr ] = cmdSup

   # FW-management CDB override data
   fwCdbOverrides.firmwarePasswordType = fwPswdTypes.get( ovProfile.fwPasswordType )
   fwCdbOverrides.firmwareImageReadbackSupport = ovProfile.imageReadbackSupported
   fwCdbOverrides.skipFirmwareErasedBlock = ovProfile.skipErasedBlocks
   fwCdbOverrides.firmwareCopyCommandSupport = ovProfile.fwCopyCmdSupported
   fwCdbOverrides.firmwareAbortCommandSupport = ovProfile.fwAbortCmdSupported
   fwCdbOverrides.firmwareStartCommandPayloadSize = ovProfile.startCmdPayloadSize
   fwCdbOverrides.firmwareErasedByteMarker = ovProfile.erasedByte
   fwCdbOverrides.firmwareMaxEplTransactionSize = ovProfile.fwMaxEplTransactionSize
   fwCdbOverrides.firmwareMaxLplTransactionSize = ovProfile.fwMaxLplTransactionSize
   fwCdbOverrides.firmwareWriteMethod = fwAccessMethods.get(
      ovProfile.fwWriteMethod )
   fwCdbOverrides.firmwareReadMethod = fwAccessMethods.get( ovProfile.fwReadMethod )
   fwCdbOverrides.firmwareHitlessRestartSupported = \
      ovProfile.hitlessFwRestartSupported
   fwCdbOverrides.firmwareStartCommandMaxDuration = ovProfile.startMaxDuration
   fwCdbOverrides.firmwareAbortCommandMaxDuration = ovProfile.abortMaxDuration
   fwCdbOverrides.firmwareWriteCommandMaxDuration = ovProfile.fwWriteMaxDuration
   fwCdbOverrides.firmwareCompleteCommandMaxDuration = ovProfile.completeMaxDuration
   fwCdbOverrides.firmwareCopyCommandMaxDuration = ovProfile.copyMaxDuration

   cdbCapsDetailsModel.moduleCdbCapabilitiesOverride = cdbOverrides
   cdbCapsDetailsModel.firmwareManagementCdbCapabilitiesOverride = fwCdbOverrides

def _populateCdbCapabilities( model, slotNum, xcvrStatus, cdbCaps, fwCaps ):
   if cdbCaps is None:
      # If we don't have any CDB capabilities to show, we're done here.
      return
   cdbCapsDetails = xcvrCdbCapsModel.TransceiverCdbCapabilitiesSlotDetail()
   cdbCapsDetails.serialNumber = xcvrStatus.vendorInfo.vendorSn
   _populateModuleCdbCaps( cdbCapsDetails, cdbCaps )
   isOverrideApplied = ( gv.xcvrCentralStatus is not None and
                         gv.xcvrCentralStatus.cmisCdbOverride is not None and
                         gv.xcvrCentralStatus.cmisCdbOverride.overrideProfilesDir and
                         cdbCaps.overrideProfile is not None )
   if fwCaps is not None or isOverrideApplied:
      _populateFwCdbCaps( cdbCapsDetails, fwCaps )
   if isOverrideApplied:
      profile = gv.xcvrCentralStatus.cmisCdbOverride.overrideProfilesDir[
         cdbCaps.overrideProfile ]
      _populateCdbOverrides( cdbCapsDetails, profile )
   model.ports[ slotNum ] = cdbCapsDetails

def handleShowXcvrCdbCapabilities( mode, args ):
   # Grab all the slots we're interested in
   slotNumToXcvrName = {}
   for xcvrName in gv.xcvrStatusDir.xcvrStatus:
      slotNumToXcvrName[ getSlotId( xcvrName ) ] = xcvrName
   slots = args.get( 'SLOTS' )
   if slots:
      xcvrNameRange = [ slotNumToXcvrName[ getSlotId( slot ) ] for slot in slots ]
   else:
      xcvrNameRange = gv.xcvrStatusDir.xcvrStatus
   # Create the model
   model = xcvrCdbCapsModel.TransceiverCdbCapabilitiesSlotCollection()
   for xcvrName in xcvrNameRange:
      xcvrStatus = gv.xcvrStatusDir.xcvrStatus[ xcvrName ]
      if ( xcvrStatus is None or xcvrStatus.presence != XcvrPresence.xcvrPresent or
           not isCmisTransceiver( xcvrStatus ) ):
         # Only CMIS transceivers are supported in this command
         continue
      slotNum = getSlotId( xcvrName )
      cmisCdbCliHelper = CmisCdbCliHelper( xcvrStatus )
      cdbCaps = cmisCdbCliHelper.getCdbCaps()
      fwCaps = cmisCdbCliHelper.getFwCaps()
      _populateCdbCapabilities( model, slotNum, xcvrStatus, cdbCaps, fwCaps )
   return model

def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.cmisFwStatusSliceDir = LazyMount.mount( em,
      "hardware/xcvr/status/cmis/cdbFirmware/slice", "Tac::Dir", "ri" )
   gv.cmisCdbCapabilitiesSliceDir = LazyMount.mount( em,
      "hardware/xcvr/status/cmis/cdbCapabilities/slice", "Tac::Dir", "ri" )
   gv.xcvrCentralStatus = LazyMount.mount( em,
      "hardware/xcvr/central/status",
      "XcvrCentralAgent::XcvrCentralStatus", "ri" )
