# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Tac
import ConfigMount
import LazyMount
import EthIntfLib
from CliDynamicSymbol import CliDynamicPlugin
import CliPlugin.XcvrAllStatusDir
from XcvrLib import isCmisTransceiver, getSlotId
from TypeFuture import TacLazyType
import CliGlobal

XCFM = CliDynamicPlugin( 'XcvrCmisFwModel' )

gv = CliGlobal.CliGlobal( xcvrStatusDir=None, xcvrCmisCdbConfigCliSlices=None,
                          cmisFwStatusSliceDir=None,
                          cmisCdbCapabilitiesSliceDir=None )

# FwRequests make a roundtrip. The Cli writes to xcvrCmisCdbCli which triggers
# CmisCliSanitizerSm. After sanitizerSm determines its validity, it writes back to
# CmisFwStatus. fwRequestTimeout is the max time the Cli will wait for the
# sanitizerSm to write back to CmisFwStatus. fwRequestMaxDelay is the time between
# polls
cmisFwRequestTimeout = 2
cmisFwRequestMaxDelay = 0.1

CmisCdbFwCommandType = TacLazyType( "Xcvr::CmisCdbFwCommandType" )
CmisCdbFailResult = TacLazyType( "Xcvr::CmisCdbFailResult" )
CmisCdbInProgressResult = TacLazyType( "Xcvr::CmisCdbInProgressResult" )
CmisCdbStatus = TacLazyType( "Xcvr::CmisCdbStatus" )
CmisCdbSuccessResult = TacLazyType( "Xcvr::CmisCdbSuccessResult" )
CmisCdbSupport = TacLazyType( "Xcvr::CmisCdbSupport" )
CmisFwCliSanitizerResult = TacLazyType( "Xcvr::CmisFwCliSanitizerResult" )
CmisFwRequestType = TacLazyType( "Xcvr::CmisFwRequestType" )
CmisFwUpdateState = TacLazyType( "Xcvr::CmisFwUpdateState" )
XcvrPresence = TacLazyType( "Xcvr::XcvrPresence" )

# ------------------------------------------------------
# Common Helpers
# ------------------------------------------------------

xcvrNotPresentErrorMsg = "No transceiver in slot {slotNum}"
slotNotSupportedErrorMsg = "Firmware commands are not supported on slot {slotNum}"
timeoutErrorMsg = "Slot {slotNum}: Timeout"
sanitizerResponseErrorMsgs = {
   CmisFwCliSanitizerResult.cmisFwXcvrNotInitialized:
      ( "Firmware update is not yet available as the transceiver in slot "
        "{slotNum} is initializing" ),
   CmisFwCliSanitizerResult.cmisFwXcvrNoFwUpgradeSupport:
      ( "Firmware update is not supported on slot {slotNum}" ),
   CmisFwCliSanitizerResult.cmisFwXcvrNoFwAbort:
      ( "Firmware update abort is not supported by the transceiver in slot "
        "{slotNum}" ),
   CmisFwCliSanitizerResult.cmisFwXcvrUpgradeInProgress:
      "Firmware update in progress by the transceiver in slot {slotNum}",
   CmisFwCliSanitizerResult.cmisFwCliCmdDisabled:
      "This command is disabled for the transceiver in slot {slotNum}",
}
errorIsDir = ( "Error finding firmware binary {fileUrl} (Is a directory: "
               "'{filePath}')" )
errorFileNotPresent = ( "Error finding firmware binary {fileUrl} (No such file "
                       "or directory: '{filePath}')" )

def _toUtcTimestamp( timestamp ):
   '''
   Converts a timestamp taken with Tac.now() to UTC.

   Parameters
   ----------
   timestamp : float
   '''
   if timestamp == 0:
      # If the timestamp has never been set, we will just pass 0 to the model and
      # render it as "never"
      return timestamp
   return timestamp + Tac.utcNow() - Tac.now()

def _sanitizerSmCompleted( sanitizerStatus, lastRequest ):
   '''
   Determines if sanitizerSm has finished by checking if sanitizerStatus has been
   updated after the last request was made. Returns True if sanitizerSm has finished

   Parameters
   -------------
   sanitizerStatus : Xcvr::CmisFwCliSanitizerStatus
   lastRequest : float
      Timestamp of when the request was last made
   '''
   if not sanitizerStatus:
      return False
   lastSanitizerResultTimestamp = sanitizerStatus.lastSanitizerResultTimestamp

   return lastSanitizerResultTimestamp >= lastRequest

def _allSanitizerSmCompleted( slotNumToFwStatus, slotNumToRequestTimestamp ):
   '''
   Determines if sanitizerSm has finished for all slots where a firmware upgrade
   request has been made. Returns True if sanitizerSm has finished for all slots

   Parameters
   ----------
   slotNumToFwStatus : Dict
      Maps slotNum to fwStatus
   slotNumToRequestTimestamp : Dict
      Maps slotNum to the timestamp of when the last request was made
   '''
   return all( _sanitizerSmCompleted( fwStatus.firmwareCliSanitizerStatus,
               slotNumToRequestTimestamp[ slotNum ] ) for slotNum, fwStatus
               in slotNumToFwStatus.items() )

def _getSlotNumToXcvrNameMapping():
   '''
   Creates a dictionary that maps from slotNum->XcvrName. This is done by looking at
   the xcvrStatuses that currently exist.

   Returns
   -------
   slotNumToXcvrName : Dict
   '''
   slotNumToXcvrName = {}
   for xcvrName in gv.xcvrStatusDir.xcvrStatus:
      slotNumToXcvrName[ getSlotId( xcvrName ) ] = xcvrName
   return slotNumToXcvrName

class CmisFwCliHelper:
   def __init__( self, mode, xcvrStatus, slotNum ):
      '''
      Helper Class that contains functionalities common to all firmware upgrade CLI
      commands

      Parameters
      ----------
      mode : BasicCliModes.EnableMode
      xcvrStatus : Xcvr::XcvrNewStatus
      slotNum : str
         ex. "3" for fixed systems or "1/3" for modular systems
      '''
      self.mode = mode
      self.xcvrStatus = xcvrStatus
      self.xcvrName = xcvrStatus.name
      self.slotNum = slotNum

   def sliceName( self ):
      '''
      Returns the slice name that corresponds with the xcvrName
      '''
      isModular = Cell.cellType() == "supervisor"
      if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId(
         self.xcvrName ):
         sliceName = EthIntfLib.sliceName( self.xcvrName )
      else:
         sliceName = "FixedSystem"
      return sliceName

   def checkXcvrPresent( self ):
      '''
      Check if the Xcvr corresponding to the xcvrStatus is present in the slot. If it
      is, return True. Otherwise, print out an error message and return False.
      '''
      if self.xcvrStatus.presence != XcvrPresence.xcvrPresent:
         self.mode.addError( xcvrNotPresentErrorMsg.format( slotNum=self.slotNum ) )
         return False
      else:
         return True

   def checkIsCmis( self ):
      '''
      Check if the Xcvr corresponding to the xcvrStatus is CMIS. If it is, return
      True. Otherwise, print out an error message and return False.
      '''
      if isCmisTransceiver( self.xcvrStatus ):
         return True
      else:
         self.mode.addError(
            slotNotSupportedErrorMsg.format( slotNum=self.slotNum ) )
         return False

   def getFwStatus( self ):
      cmisFwStatusDir = gv.cmisFwStatusSliceDir[ self.sliceName() ]
      cmisFwStatus = cmisFwStatusDir.cmisFwStatus.get( self.xcvrName )
      return cmisFwStatus

   def checkFwStatusExists( self ):
      '''
      Check that fwStatus exists. If it does, return fwStatus. Otherwise, print out
      an error message and return None.
      '''
      cmisFwStatus = self.getFwStatus()
      if cmisFwStatus is None:
         self.mode.addError( sanitizerResponseErrorMsgs[
            CmisFwCliSanitizerResult.cmisFwXcvrNoFwUpgradeSupport ].format(
               slotNum=self.slotNum ) )
      return cmisFwStatus

   def createFwRequest( self, fwRequestType, imageFilePath=None ):
      '''
      Create a fwRequest based on fwRequestType

      Parameters
      ----------
      fwRequestType : Xcvr::CmisFwRequestType
         type of request to create
      imageFilePath : str | None
         Sets Xcvr::CmisFwManageRequest to this file path if it is not None. Note
         that this path is the real linux file path rather than with "file:/" or
         "flash:/" prefix

      Returns
      -------
      fwRequest : Xcvr::CmisFwManagementRequest
         the created request
      '''
      fwRequest = Tac.Value( "Xcvr::CmisFwManagementRequest" )
      fwRequest.requestType = fwRequestType
      fwRequest.lastRequest = Tac.now()
      if imageFilePath:
         fwRequest.imageFilePath = imageFilePath
      return fwRequest

   def getXcvrCmisCdbCli( self ):
      '''
      Get the xcvrCmisCdbCli corresponding to the xcvrName. If it doesn't exist yet,
      create it.

      Returns
      -------
      xcvrCmisCdbCli : Xcvr::XcvrCmisCdbCli
      '''
      xcvrCmisCdbConfigCliDir = gv.xcvrCmisCdbConfigCliSlices[ self.sliceName() ]
      xcvrCmisCdbCli = xcvrCmisCdbConfigCliDir.xcvrCmisCdbCli.get( self.xcvrName )
      if xcvrCmisCdbCli is None:
         xcvrCmisCdbCli = xcvrCmisCdbConfigCliDir.newXcvrCmisCdbCli( self.xcvrName )
      return xcvrCmisCdbCli

def makeCmisFwRequests( mode, slotRange, cmisFwRequestType, fwBinaryRealFilename ):
   '''
   Creates a CmisFwRequest for each slot within a slotRange and populates the request
   with the requestType and the file path (if it has one). Each request will make
   a round trip with the sanitizer SM and will wait until each request is returned
   with a status. This function is also responsible for printing out error messages
   for slots that don't meet the requirements/ timeouts.

   Parameters
   ----------
   mode : BasicCliModes.EnableMode
      This is used to print out error messages
   slotRange : MultiRangeRule.IntfList
   cmisFwRequestType : Xcvr::CmisFwManagementRequest
   fwBinaryRealFilename : str | None
      The linux file path. For example, "/tmp/cmisFwUpdateTestImg.bin". If None, the
      created fwRequests will not contain a file path.
   '''
   slotNumToXcvrName = _getSlotNumToXcvrNameMapping()
   # This dict maps from slotNum to the time when the request is made. It is used
   # to check when sanitizerSm finishes
   slotNumToRequestTimestamp = {}
   slotNumToFwStatus = {}
   for slot in slotRange:
      slotNum = getSlotId( slot )
      xcvrName = slotNumToXcvrName[ slotNum ]
      xcvrStatus = gv.xcvrStatusDir.xcvrStatus[ xcvrName ]
      cmisFwCliHelper = CmisFwCliHelper( mode, xcvrStatus, slotNum )

      # Check that the xcvr has the required capabilities
      if not cmisFwCliHelper.checkXcvrPresent():
         continue
      if not cmisFwCliHelper.checkIsCmis():
         continue
      fwStatus = cmisFwCliHelper.checkFwStatusExists()
      if not fwStatus:
         continue

      xcvrCmisCdbCli = cmisFwCliHelper.getXcvrCmisCdbCli()

      # Determine which fwRequest to make by looking at the command entered
      fwRequest = cmisFwCliHelper.createFwRequest(
         cmisFwRequestType,
         imageFilePath=fwBinaryRealFilename )

      slotNumToFwStatus[ slotNum ] = fwStatus
      slotNumToRequestTimestamp[ slotNum ] = fwRequest.lastRequest
      xcvrCmisCdbCli.cmisFwManagementRequest = fwRequest

   # Wait until all slots that a request was made on had finished - or until
   # timeout
   ds = "firmware requests to finish"
   try:
      Tac.waitFor( lambda: _allSanitizerSmCompleted( slotNumToFwStatus,
                      slotNumToRequestTimestamp ),
                      sleep=True,
                      description=ds,
                      timeout=cmisFwRequestTimeout,
                      maxDelay=cmisFwRequestMaxDelay )
   except Tac.Timeout:
      # Slots that timed out are handled below.
      pass

   for slotNum, fwStatus in slotNumToFwStatus.items():
      sanitizerStatus = fwStatus.firmwareCliSanitizerStatus
      # Check if slot has timeout. If it did, print an error and move on to next
      # slot.
      if not _sanitizerSmCompleted( sanitizerStatus,
                                    slotNumToRequestTimestamp[ slotNum ] ):
         mode.addError( timeoutErrorMsg.format( slotNum=slotNum ) )
         continue
      cmisFwCliSanitizerResult = sanitizerStatus.cmisFwCliSanitizerResult

      # Print out an error message if sanitizerSm did not pass
      if cmisFwCliSanitizerResult in sanitizerResponseErrorMsgs:
         mode.addError( sanitizerResponseErrorMsgs[ cmisFwCliSanitizerResult ]
                        .format( slotNum=slotNum ) )

# --------------------------------------------------------------------------------
#
# transceiver firmware update slot <SLOTID> ( image <FILE> [ inactive ] | abort  )
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------
def transceiverFirmwareUpdateHandler( mode, args ):
   slotRange = args.get( 'SLOTS' )
   fwBinaryFilePath = args.get( 'FILE' )

   # Check that if a fwBinaryFile is given, it is a valid path
   if fwBinaryFilePath is not None:
      if fwBinaryFilePath.isdir():
         mode.addErrorAndStop( errorIsDir.format(
            fileUrl=fwBinaryFilePath.url,
            filePath=fwBinaryFilePath.pathname ) )
      if not fwBinaryFilePath.exists():
         mode.addErrorAndStop( errorFileNotPresent.format(
            fileUrl=fwBinaryFilePath.url,
            filePath=fwBinaryFilePath.pathname ) )

   if 'abort' in args:
      cmisFwRequestType = CmisFwRequestType.cmisFwAbortRequest
   elif 'inactive' in args:
      cmisFwRequestType = CmisFwRequestType.cmisFwDownloadRequest
   else:
      cmisFwRequestType = CmisFwRequestType.cmisFwUpdateRequest

   fwBinaryRealFilename = ( fwBinaryFilePath.realFilename_ if fwBinaryFilePath
                                                           else None )
   makeCmisFwRequests( mode, slotRange, cmisFwRequestType, fwBinaryRealFilename )

# --------------------------------------------------------------------------------
#
# transceiver firmware activate-standby slot <SLOTID>
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------
def transceiverFirmwareActivateStandbyHandler( mode, args ):
   slotRange = args.get( 'SLOTS' )
   makeCmisFwRequests( mode, slotRange,
                       CmisFwRequestType.cmisFwActivateStandbyRequest, None )

# --------------------------------------------------------------------------------
#
# show transceiver status slot <SLOTID> firmware version
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------
def _createFwVersionDetail( cmisFirmwareVersion, imageLabel, activeFw, inactiveFw ):
   '''
   Populates TransceiverFirmwareVersionDetail using firmwareVersion from fwStatus.
   This function will also append TransceiverFirmwareVersionDetail to either activeFw
   if it is running or into inactiveFw if it is not.

   Parameters
   ----------
   cmisFirmwareVersion : Xcvr::CmisFirmwareVersion
   imageLabel : str
   activeFw : dict ( str -> TransceiverFirmwareVersionDetail )
   inactiveFw : dict ( str -> TransceiverFirmwareVersionDetail )
   '''
   # Ignore firmware if it is invalid
   if not cmisFirmwareVersion or cmisFirmwareVersion.invalid:
      return

   # Check imageBankPresent to make sure that this image exists
   if not cmisFirmwareVersion.imageBankPresent:
      return

   fwVersionDetail = XCFM.TransceiverFirmwareVersionDetail()

   fwVersionDetail.majorVer = cmisFirmwareVersion.majorVersion
   fwVersionDetail.minorVer = cmisFirmwareVersion.minorVersion
   fwVersionDetail.buildNum = cmisFirmwareVersion.buildNum
   fwVersionDetail.committed = cmisFirmwareVersion.committed

   if cmisFirmwareVersion.running:
      activeFw[ imageLabel ] = fwVersionDetail
   else:
      inactiveFw[ imageLabel ] = fwVersionDetail

def _populateFwImages( model, slotId, fwStatus ):
   '''
   Populates TransceiverFirmwareVersionDetailDict model by looking at the three
   images in CmisFirmwareStatus for a single slot. Will then add this submodel into
   the parent model.

   Parameters
   ----------
   model : XcvrCmisFwModel.TransceiverFirmwareVersionSlotCollection
      Adds a new slot submodel into this model
   slotId : str
      Ex. "5" for fixed systems or "1/5" for modular systems
   fwStatus : Xcvr::CmisFwStatus
   '''
   fwVersionStatus = fwStatus.firmwareVersion

   # Check that fwVersion exists and that it is valid (has been read at least once)
   if fwVersionStatus is None or not fwVersionStatus.valid:
      return

   activeFw = {}
   inactiveFw = {}

   # Populate fwImageA
   fwImageA = fwVersionStatus.fwImageA
   _createFwVersionDetail( fwImageA, "A", activeFw, inactiveFw )
   # Populate fwImageB
   fwImageB = fwVersionStatus.fwImageB
   _createFwVersionDetail( fwImageB, "B", activeFw, inactiveFw )
   # Populate fwImageFactory
   fwImageFactory = fwVersionStatus.fwImageFactory
   _createFwVersionDetail( fwImageFactory, "factory default", activeFw, inactiveFw )

   # Set the model
   fwVersionType = XCFM.TransceiverFirmwareVersionDetailDict()

   # Only populate active/ inactive firmwares if there are at least one
   fwVersionType.active = activeFw
   fwVersionType.inactive = inactiveFw

   slotName = slotId

   firmwareCollection = XCFM.TransceiverFirmwareCollection()
   firmwareCollection.firmware = fwVersionType
   model.ports[ slotName ] = firmwareCollection

def handleShowFirmwareVersion( mode, args ):
   # Range of xcvrNames to loop over. It is either by looking at the slotRange or
   # just looping over all xcvrNames ( which happens when no range is specified )
   slotNumToXcvrName = _getSlotNumToXcvrNameMapping()
   xcvrNameRange = [ slotNumToXcvrName[ getSlotId( slot ) ]
                     for slot in args[ "SLOTS" ] ] if "SLOTS" in args else\
                                                  gv.xcvrStatusDir.xcvrStatus
   # Create Model
   model = XCFM.TransceiverFirmwareVersionSlotCollection()

   for xcvrName in xcvrNameRange:
      slotNum = getSlotId( xcvrName )
      xcvrStatus = gv.xcvrStatusDir.xcvrStatus[ xcvrName ]
      cmisFwCliHelper = CmisFwCliHelper( mode, xcvrStatus, slotNum )
      fwStatus = cmisFwCliHelper.getFwStatus()
      if fwStatus is None:
         continue
      _populateFwImages( model, slotNum, fwStatus )
   model.renderModel()
   return model

# --------------------------------------------------------------------------------
#
# show transceiver status slot <SLOTID> firmware update
#
# Note that "slot <SLOTID>" uses a single matcher that matches two tokens.
# --------------------------------------------------------------------------------
_cmisFwUpdateStateValue = {
   CmisFwUpdateState.fwUpdateIdle: "idle",
   CmisFwUpdateState.fwUpdateRunning: "running",
   CmisFwUpdateState.fwUpdateFailed: "failed",
   CmisFwUpdateState.fwUpdateFailedTerminated: "terminated",
   CmisFwUpdateState.fwUpdateSuccess: "success",
   CmisFwUpdateState.fwUpdateTimeout: "timeout",
}

_cdbCommandTypeValue = {
   CmisCdbFwCommandType.cdbStart: "start",
   CmisCdbFwCommandType.cdbComplete: "complete",
   CmisCdbFwCommandType.cdbDownload: "download",
   CmisCdbFwCommandType.cdbRun: "run",
   CmisCdbFwCommandType.cdbCommit: "commit",
   CmisCdbFwCommandType.cdbAbort: "abort",
}

_cdbStatusValue = {
   CmisCdbStatus.cdbUnknown: "unknown",
   CmisCdbStatus.cdbSuccess: "success",
   CmisCdbStatus.cdbFail: "fail",
   CmisCdbStatus.cdbInProgress: "in progress",
}

_decodedCdbSuccessResultMessage = {
   CmisCdbSuccessResult.cdbSuccessReserved: "Reserved",
   CmisCdbSuccessResult.cdbSuccessCustom: "Custom",
   CmisCdbSuccessResult.cdbSuccessGeneric: "Command completed successfully without "
                                            "specific message",
   CmisCdbSuccessResult.cdbSuccessAbort: "Previous command was aborted",
}

_decodedCdbFailResultMessage = {
   CmisCdbFailResult.cdbFailReserved: "Reserved",
   CmisCdbFailResult.cdbFailCustom: "Custom",
   CmisCdbFailResult.cdbFailCmdUnknown: "Unknown command",
   CmisCdbFailResult.cdbFailParameter: "Parameter not supported",
   CmisCdbFailResult.cdbFailCmdNotAborted: "Previous command not properly aborted",
   CmisCdbFailResult.cdbFailCheckingTimeOut: "Command checking time out",
   CmisCdbFailResult.cdbFailChkCodeError: "CDB check code error",
   CmisCdbFailResult.cdbFailPasswordError: "Password error",
   CmisCdbFailResult.cdbFailCmdNotCompatible: "Command not compatible with "
   "operating status",
   CmisCdbFailResult.cdbFailReservedStsChecking: "Reserved for STS command "
   "checking error",
   CmisCdbFailResult.cdbFailStsError: "Reserved for individual STS command or task "
                                       "error"
}

_decodedCdbInProgressResultMessage = {
   CmisCdbInProgressResult.cdbInProgressReserved: "Reserved",
   CmisCdbInProgressResult.cdbInProgressCustom: "Custom",
   CmisCdbInProgressResult.cdbInProgressNotProcessed: "Command captured but not "
   "processed",
   CmisCdbInProgressResult.cdbInProgressChecking: "Command checking in progress",
   CmisCdbInProgressResult.cdbInProgressExecution: "Command execution in progress",

}

def _decodedCdbResultMessage( cdbCommandStatus ):
   '''
   Determines the Cdb result message given the result status and reading the
   appropriate attribute for the status.

   Returns None if a resultMessage was not found. This should never be the case
   as long as the parsed results were populated correctly

   Parameters
   ----------
   cdbCommandStatus : Xcvr::CmisCdbCommandStatus
   '''
   cdbStatus = cdbCommandStatus.status
   if cdbStatus == CmisCdbStatus.cdbSuccess:
      resultMessage = _decodedCdbSuccessResultMessage.get(
         cdbCommandStatus.parsedSuccessResult )
   elif cdbStatus == CmisCdbStatus.cdbFail:
      resultMessage = _decodedCdbFailResultMessage.get(
         cdbCommandStatus.parsedFailResult )
   elif cdbStatus == CmisCdbStatus.cdbInProgress:
      resultMessage = _decodedCdbInProgressResultMessage.get(
         cdbCommandStatus.parsedInProgressResult )
   else:
      resultMessage = None

   return resultMessage

def _createFirmwareUpdateStatus( fwStatus ):
   '''
   Populates TransceiverFirmwareUpdateStatus model with values from Sysdb. This model
   is optional so if there is no data to show, this function returns "None"

   Parameters
   ----------
   fwStatus : Xcvr::CmisFwStatus
   '''
   # Check if firmwareUpdateReport has been created. There is nothing to show if it
   # hasn't
   fwUpdateReport = fwStatus.firmwareUpdateReport
   if fwUpdateReport is None:
      return None

   fwUpdateStatus = fwUpdateReport.fwUpdateStatus
   if fwUpdateStatus is None:
      return None

   fwUpdateProgress = fwUpdateReport.fwUpdateProgress
   fwResetProgress = fwUpdateReport.fwResetProgress

   submodel = XCFM.TransceiverFirmwareUpdateStatus()
   submodel.status = _cmisFwUpdateStateValue[ fwUpdateStatus.state ]
   submodel.statusLastChange = _toUtcTimestamp( fwUpdateStatus.lastChange )
   submodel.downloadPercentage = fwUpdateProgress.count
   submodel.downloadPercentageLastChange = _toUtcTimestamp( fwUpdateProgress.last )
   submodel.restartPercentage = fwResetProgress.count
   submodel.restartPercentageLastChange = _toUtcTimestamp( fwResetProgress.last )

   return submodel

def _createTransceiverFirmwareCdbTimeout( fwUpdateReport ):
   '''
   Creates/Populates TransceiverFirmwareCdbTimeout model

   Parameters
   ----------
   fwUpdateReport : Xcvr::CmisFwUpdateReport
   '''
   submodel = XCFM.TransceiverFirmwareCdbTimeout()
   submodel.timeoutCount = fwUpdateReport.cdbCompleteTimeout.count
   submodel.lastChange = _toUtcTimestamp( fwUpdateReport.cdbCompleteTimeout.last )

   return submodel

def _createHardwareStatusDetail( cdbCommandStatus ):
   '''
   Creates/Populates TransceiverFirmwareHardwareStatusDetail model

   Parameters
   ----------
   cdbCommandStatus : Xcvr::CmisCdbCommandStatus
   '''
   submodel = XCFM.TransceiverFirmwareHardwareStatusDetail()
   submodel.decoded = _decodedCdbResultMessage( cdbCommandStatus )
   submodel.lastChange = _toUtcTimestamp( cdbCommandStatus.resultLastChange )
   submodel.rawValue = cdbCommandStatus.rawResultCode

   return submodel

def _createHardwareDependentStatus( fwUpdateReport ):
   '''
   Creates/Populates TransceiverFirmwareHardwareStatus model. This submodel is
   optional in the JSON model. If there is no data to show, this function returns
   None to leave it out of the final model.

   Parameters
   ----------
   fwUpdateReport : Xcvr::CmisFwUpdateReport
   '''
   if fwUpdateReport is None:
      return None

   cdbFwCommand = fwUpdateReport.cdbFwCommand
   if cdbFwCommand is None or cdbFwCommand.command == CmisCdbFwCommandType.cdbNone:
      return None

   cdbCommandStatus = fwUpdateReport.cdbCommandStatus
   if cdbCommandStatus is None:
      return None

   submodel = XCFM.TransceiverFirmwareHardwareStatus()
   submodel.commandType = _cdbCommandTypeValue[ cdbFwCommand.command ]
   submodel.commandTypeLastChange = _toUtcTimestamp( cdbFwCommand.lastChange )
   submodel.status = _cdbStatusValue[ cdbCommandStatus.status ]
   submodel.statusLastChange = _toUtcTimestamp( cdbCommandStatus.statusLastChange )
   submodel.statusDetail = _createHardwareStatusDetail( cdbCommandStatus )
   submodel.cdbCommandAckTimeout =\
      _createTransceiverFirmwareCdbTimeout( fwUpdateReport )

   return submodel

def _populateFwUpdate( model, slotNum, fwStatus, xcvrStatus ):
   '''
   Creates/Populates TransceiverFirmwareUpdateSlotDetail model

   Parameters
   ----------
   slotNum : str
      ex. "3" for fixed systems or "1/3" for modular systems
   fwStatus : Xcvr::CmisFwStatus
   xcvrStatus : Xcvr::XcvrNewStatus
   '''
   slotDetail = XCFM.TransceiverFirmwareUpdateSlotDetail()
   slotDetail.serialNumber = xcvrStatus.vendorInfo.vendorSn
   slotDetail.serialNumberLastChange = _toUtcTimestamp(
      xcvrStatus.lastPresenceChange )
   slotDetail.firmwareUpdate = _createFirmwareUpdateStatus( fwStatus )
   slotDetail.hardwareDependentCommand =\
      _createHardwareDependentStatus( fwStatus.firmwareUpdateReport )

   model.ports[ slotNum ] = slotDetail

def handleShowFirmwareUpdate( mode, args ):
   # Range of xcvrNames to loop over. It is either by looking at the slotRange or
   # just looping over all xcvrNames ( which happens when no range is specified )
   slotNumToXcvrName = _getSlotNumToXcvrNameMapping()
   slots = args.get( 'SLOTS' )
   if slots:
      xcvrNameRange = [ slotNumToXcvrName[ getSlotId( slot ) ] for slot in slots ]
   else:
      xcvrNameRange = gv.xcvrStatusDir.xcvrStatus

   # Create the model
   model = XCFM.TransceiverFirmwareUpdateSlotCollection()

   for xcvrName in xcvrNameRange:
      slotNum = getSlotId( xcvrName )
      xcvrStatus = gv.xcvrStatusDir.xcvrStatus[ xcvrName ]
      cmisFwCliHelper = CmisFwCliHelper( mode, xcvrStatus, slotNum )
      fwStatus = cmisFwCliHelper.getFwStatus()
      if fwStatus is None:
         continue
      _populateFwUpdate( model, slotNum, fwStatus, xcvrStatus )

   return model

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   allCdbConfigCliSlices = [ "FixedSystem", "Linecard1", "Linecard2", "Linecard3",
                             "Linecard4", "Linecard5", "Linecard6", "Linecard7",
                             "Linecard8", "Linecard9", "Linecard10", "Linecard11",
                             "Linecard12", "Linecard13", "Linecard14", "Linecard15",
                             "Linecard16", "Linecard17", "Linecard18" ]

   gv.xcvrCmisCdbConfigCliSlices = {}
   for sliceName in allCdbConfigCliSlices:
      gv.xcvrCmisCdbConfigCliSlices[ sliceName ] = ConfigMount.mount( em,
         # pylint: disable-next=consider-using-f-string
         "hardware/xcvr/cli/cmisCdbConfig/slice/%s" % sliceName,
         "Xcvr::XcvrCmisCdbConfigCliDir", "wi" )

   gv.cmisFwStatusSliceDir = LazyMount.mount( em,
      "hardware/xcvr/status/cmis/cdbFirmware/slice", "Tac::Dir", "ri" )

   gv.cmisCdbCapabilitiesSliceDir = LazyMount.mount( em,
      "hardware/xcvr/status/cmis/cdbCapabilities/slice", "Tac::Dir", "ri" )
