# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Arnet
import TableOutput
import Ark
import Tac
from time import ctime
from CliModel import Model, Dict, Int, Bool, Str, Float, Submodel, Enum

# ------------------------------------------------------------------------------
#
# Model for "show transceiver status slot <slotId> firmware version"
#
# ------------------------------------------------------------------------------
class TransceiverFirmwareVersionDetail( Model ):
   majorVer = Int( help="Firmware major version" )
   minorVer = Int( help="Firmware minor version" )
   buildNum = Int( help="Firmware build number" )
   committed = Bool( help="This firmware image will become active upon transceiver "
                          "reinsertion" )

   def renderVersion( self ):
      '''
      Returns a string of the firmware's full version
      '''
      # pylint: disable-next=consider-using-f-string
      return "{major}.{minor}.{build}".format( major=self.majorVer,
                                               minor=self.minorVer,
                                               build=self.buildNum )

   def renderLabel( self, imageLabel ):
      '''
      Returns a string of the imageLabel and whether it is committed. The image label
      will be either "A", "B", or "factory default".

      Possible outputs may look like:
         (factory default)
         (image A)
         (image A, committed)
      '''
      renderedImageLabel = imageLabel
      if renderedImageLabel in [ "A", "B" ]:
         renderedImageLabel = "image " + imageLabel

      if self.committed:
         return f"({renderedImageLabel}, committed)"
      return f"({renderedImageLabel})"

class TransceiverFirmwareVersionDetailDict( Model ):
   active = Dict( keyType=str,
                  valueType=TransceiverFirmwareVersionDetail,
                  help="Mapping from an active image to the image details" )
   inactive = Dict( keyType=str,
                    valueType=TransceiverFirmwareVersionDetail,
                    help="Mapping from an inactive image to the image details" )

   def renderDict( self, fwDict ):
      '''
      Parameters
      ----------
      fwDict : Model.Dict
         can be either self.active or self.inactive. This function
         renders the firware versions within this dictionary.

      Returns a string. Outputs may look like:
         n/a
         2.16.0 (image A)
         2.16.0 (image A), 2.14.0 (image B )
         2.16.0 (image A, committed), 2.14.0 (image B )
      '''
      labelList = []
      for fwImage in fwDict:
         fwDetail = fwDict[ fwImage ]
         # Skip rendering "factory default" image if it is inactive
         if ( fwDict == self.inactive and fwImage == "factory default" ):
            continue
         # pylint: disable-next=consider-using-f-string
         labelList.append( "{version} {label}".format(
            version=fwDetail.renderVersion(),
            label=fwDetail.renderLabel( fwImage ) ) )

      # Return 'n/a' if there are no firmware images in the rendered output. This
      # is due to the "factory default" image that is present in the model but is
      # not rendered in the output
      return ', '.join( labelList ) or 'n/a'

class TransceiverFirmwareCollection( Model ):
   firmware = Submodel( valueType=TransceiverFirmwareVersionDetailDict,
                        help="images present on the transceiver" )

class TransceiverFirmwareVersionSlotCollection( Model ):
   ports = Dict( keyType=str, valueType=TransceiverFirmwareCollection,
                 optional=True,
                 help="Mapping between the slot name and the firmware version "
                      "information of the inserted transceiver" )

   def printRows( self, rowsToPrint ):
      '''
      Prints out all rows in rowsToPrint

      Parameters
      ----------
      rowsToPrint : list
         List of tuples of size three corresponding to the columns to print
      activeTextWidth : int
         Width of the "Active Firmware" column
      '''

      # Don't print anything if there are no ports to print
      if not rowsToPrint:
         return

      # Create the table
      headers = ( "Port", "Active Firmware", "Inactive Firmware" )
      table = TableOutput.createTable( headers )

      formatPortColumn = TableOutput.Format( justify="left", minWidth=5 )
      formatFwColumn = TableOutput.Format( justify="left", minWidth=30 )

      # Remove additional column paddings
      formatPortColumn.padLimitIs( True )
      formatFwColumn.padLimitIs( True )
      table.formatColumns( formatPortColumn, formatFwColumn, formatFwColumn )

      for slotName, activeText, inactiveText in rowsToPrint:
         table.newRow( slotName, activeText, inactiveText )

      print( table.output() )

   def renderModel( self ):
      '''
      Renders the full "show firmware version" command.
      '''
      if self.ports is None:
         return
      rowsToPrint = []
      # Sort slots in order to print the slots in increasing order
      slotRange = Arnet.sortIntf( self.ports )
      for slot in slotRange:
         slotDetail = self.ports[ slot ].firmware
         rowsToPrint.append(
            ( slot,
              slotDetail.renderDict( slotDetail.active ),
              slotDetail.renderDict( slotDetail.inactive ) ) )
      self.printRows( rowsToPrint )

# ------------------------------------------------------------------------------
#
# Model for "show transceiver status slot <slotId> firmware update"
#
# ------------------------------------------------------------------------------
firmwareUpdateStatusValues = ( "idle", "running", "failed", "terminated", "success",
                               "timeout" )
hardwareDependentCommandTypeValues = ( "start", "download", "complete", "run",
                                       "commit", "abort" )
hardwareDependentCommandStatusValues = ( "unknown", "success", "fail",
                                         "in progress" )

class TransceiverFirmwareUpdateStatus( Model ):
   status = Enum( values=firmwareUpdateStatusValues, help="Firmware update status" )
   statusLastChange = Float( help="UTC timestamp of the last change" )
   downloadPercentage = Int( help="Firmware update progress" )
   downloadPercentageLastChange = Float( help="UTC timestamp of the last change" )
   restartPercentage = Int( help="Firmware restart progress" )
   restartPercentageLastChange = Float( help="UTC timestamp of the last change" )

class TransceiverFirmwareHardwareStatusDetail( Model ):
   decoded = Str( help="Result of the last CDB command", optional=True )
   rawValue = Int( help="Raw status code of the last CDB commands", optional=True )
   lastChange = Float( help="UTC timestamp of the last change" )

class TransceiverFirmwareCdbTimeout( Model ):
   timeoutCount = Int( help="Number of CDB command timeouts" )
   lastChange = Float( help="UTC timestamp of the last change" )

class TransceiverFirmwareHardwareStatus( Model ):
   commandType = Enum( values=hardwareDependentCommandTypeValues,
                       help="Last CDB command issued" )
   commandTypeLastChange = Float( help="UTC timestamp of the last change" )
   status = Enum( values=hardwareDependentCommandStatusValues,
                  help="Last CDB command status" )
   statusLastChange = Float( help="UTC timestamp of the last change" )
   statusDetail = Submodel( valueType=TransceiverFirmwareHardwareStatusDetail,
                            optional=True,
                            help="Details for the last CDB result" )
   cdbCommandAckTimeout = Submodel( valueType=TransceiverFirmwareCdbTimeout,
                                    help="Number of CDB command timeouts" )

class TransceiverFirmwareUpdateSlotDetail( Model ):
   serialNumber = Str( help="Transceiver serial number" )
   serialNumberLastChange = Float( help="UTC timestamp of the last change" )
   firmwareUpdate = Submodel( valueType=TransceiverFirmwareUpdateStatus,
                              help="Firmware update status", optional=True )
   hardwareDependentCommand = Submodel(
      valueType=TransceiverFirmwareHardwareStatus,
      help="Hardware-level firmware update command status",
      optional=True )

   def formatLastChange( self, lastChange ):
      '''
      Returns a str of the formatted last change.

      For example, this function might return something like:
         "01:00:00 ago"
      if lastChange is an hour before Tac.utcNow()

      Parameters
      ----------
      lastChange : float
         UTC timestamp of the last change
      '''
      return Ark.timestampToStr( lastChange, now=Tac.utcNow() )

   def updateTableWithSlotDetail( self, slotName, table ):
      '''
      Appends rows into the table with the firware update status for a particular
      slot.

      Parameters
      ----------
      slotName : str
         ex. "1" or "1/1"
      table : TableOutput.TableFormatter
         This is the table created by TableOutput.createTable()
      '''
      fwUpdate = self.firmwareUpdate
      hwStatus = self.hardwareDependentCommand
      if fwUpdate is None and hwStatus is not None:
         # If the fwUpdate is None, it might be possible that it has a CDB status.
         # So we still display the output and show fwUpdate as "idle"
         fwUpdate = TransceiverFirmwareUpdateStatus()
         fwUpdate.status = "idle"
         fwUpdate.downloadPercentage = 0
         fwUpdate.restartPercentage = 0
      # If the above condition is not met and either statuses are None, then there is
      # nothing to show.
      elif fwUpdate is None or hwStatus is None:
         return
      table.newRow( f"Port {slotName}", "", "" )
      table.newRow( "  Transceiver SN", self.serialNumber,
                    self.formatLastChange( self.serialNumberLastChange ) )

      table.newRow( "  Firmware update status", fwUpdate.status,
                    self.formatLastChange( fwUpdate.statusLastChange ) )
      table.newRow(
         "  Firmware update progress",
         f"{fwUpdate.downloadPercentage}%",
         self.formatLastChange( fwUpdate.downloadPercentageLastChange ) )
      if hwStatus.commandType != "run" and fwUpdate.restartPercentage == 0:
         formattedRestartPercentage = "n/a"
      else:
         formattedRestartPercentage = f"{fwUpdate.restartPercentage}%"
      table.newRow(
         "  Firmware restart progress",
         formattedRestartPercentage,
         self.formatLastChange( fwUpdate.restartPercentageLastChange ) )
      table.newRow( "  Last CDB update command", hwStatus.commandType,
                    self.formatLastChange( hwStatus.commandTypeLastChange ) )
      table.newRow( "  Last CDB update command status", hwStatus.status,
                    self.formatLastChange( hwStatus.statusLastChange ) )
      # Render "n/a" if the decoded message is "None"
      formattedDecodedMsg = "n/a"
      if hwStatus.statusDetail.decoded is not None:
         formattedDecodedMsg = hwStatus.statusDetail.decoded
      if hwStatus.statusDetail.rawValue is not None:
         # pylint: disable-next=consider-using-f-string
         formattedDecodedMsg += " ({})".format( hex(
            hwStatus.statusDetail.rawValue ) )
      table.newRow( "  Last CDB update command result",
                    formattedDecodedMsg,
                    self.formatLastChange( hwStatus.statusDetail.lastChange ) )
      table.newRow( "  CDB block complete timeout count",
                    str( hwStatus.cdbCommandAckTimeout.timeoutCount ),
                    self.formatLastChange(
                       hwStatus.cdbCommandAckTimeout.lastChange ) )

class TransceiverFirmwareUpdateSlotCollection( Model ):
   ports = Dict( keyType=str, valueType=TransceiverFirmwareUpdateSlotDetail,
                 help="Mapping between the slot name and the firmware update "
                      "information of the inserted transceiver" )

   def render( self ):
      # If there are no ports to show, don't print anything
      if self.ports is None:
         return

      # Create the table
      table = TableOutput.TableFormatter( 0, tableWidth=130 )

      formatPortAndStateColumn = TableOutput.Format( justify="left", minWidth=30 )
      formatLcColumn = TableOutput.Format( justify="left", minWidth=15, wrap=True )

      # Remove additional column paddings
      formatPortAndStateColumn.padLimitIs( True )
      table.formatColumns( formatPortAndStateColumn, formatPortAndStateColumn,
                           formatLcColumn )

      table.newRow( "", "Current State", "Last Change" )
      table.newRow( "", "-------------", "-----------" )

      slotRange = Arnet.sortIntf( self.ports )
      headerLen = len( table.entries_ )
      for slot in slotRange:
         slotDetail = self.ports[ slot ]
         slotDetail.updateTableWithSlotDetail( slot, table )

      # If nothing was added to the table, don't print anything
      if len( table.entries_ ) == headerLen:
         return

      print( 'Current System Time:', ctime( Tac.utcNow() ) )
      print( table.output() )
