#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections
import itertools
import CliModel
import IntfModels
import TableOutput

from ArPyUtils import naturalsorted
from typing import Iterable
from CliPlugin.XcvrCliLib import subPort
from MultiRangeRule import multiRangeToCanonicalString

loopbackSourceVals = ( "system", "network" )
loopbackClassVals = ( *loopbackSourceVals, "perLane" )
loopbackSideVals = ( "host", "media" )
loopbackConfVals = ( "none", *loopbackSourceVals )
loopbackOpStateVals = ( *loopbackConfVals, "notSupported", "notPresent" )
notSupportedStr = "not supported"
notPresentStr = "not present"
supportedStr = "supported"

modelToDisplayMap = { "notSupported": notSupportedStr, "notPresent": notPresentStr }

def displayStrLoopbackOpState( val: str ):
   return modelToDisplayMap.get( val, val )

def displayStrBoolean( val: bool ):
   return supportedStr if val else notSupportedStr

class Capability( CliModel.Model ):
   kind = CliModel.Enum(
      values=loopbackClassVals,
      help=( "The type of loopback capability, either "
             "indicates traffic source or the per-lane capability" ) )
   side = CliModel.Enum(
      values=loopbackSideVals,
      help="The side of the transceiver on which the loopback is considered on" )
   supported = CliModel.Bool( help="Loopback capability is supported",
                              default=False )

class Capabilities( CliModel.Model ):
   capabilities = CliModel.List( valueType=Capability,
                                 help="Loopback capability set" )
   simultaneousHostMedia = CliModel.Bool(
      help=( "Module supports simultaneous "
             "loopback configuration on both the host and the media side" ),
      default=False )

class IntfCapabilities( CliModel.Model ):
   loopbackCaps = CliModel.Dict(
      keyType=IntfModels.Interface,
      valueType=Capabilities,
      help=( "Loopback capabilities of each transceiver that "
             "supports CMIS based loopback" ) )

   def _rowVals( self, caps: Capabilities ):
      return ( *itertools.chain(
         self._capsToStr( caps, t ) for t in ( "system", "network", "perLane" ) ),
               displayStrBoolean( caps.simultaneousHostMedia ) )

   def _capsToStr( self, caps, key ):
      res = ','.join(
         sorted( c.side for c in caps.capabilities
                 if c.kind == key and c.supported ) )
      return res or notSupportedStr

   def render( self ):
      table = TableOutput.createTable(
         ( ( "Slot", "r" ), ( "Traffic Source", "c", ( ( "System", "l" ),
                                                       ( "Network", "l" ) ) ),
           ( "Per-lane", "l", ( ( "Loopback", "l" ), ) ),
           ( "Simultaneous", "l", ( ( "Host & Media", "l" ), ) ) ) )

      for intf in naturalsorted( self.loopbackCaps.keys() ):
         caps = self.loopbackCaps[ intf ]
         table.newRow( subPort( intf, '' ), *self._rowVals( caps ) )
      print( table.output() )

def createCapabilitiesModel():
   model = Capabilities()
   model.capabilities = [
      Capability( kind=kind, side=side, supported=False )
      for kind, side in itertools.product( loopbackClassVals, loopbackSideVals )
   ]
   return model

class LaneInfo( CliModel.Model ):
   configured = CliModel.Enum( values=loopbackConfVals,
                               help="The loopback source configuration" )
   operational = CliModel.Enum(
      values=loopbackOpStateVals,
      help="The operational state of the loopback source configuration" )

   def _addItem( self, table: TableOutput.TableFormatter, keys ):
      table.newRow( *keys, self.configured,
                    displayStrLoopbackOpState( self.operational ) )

   def _getItems( self ):
      return ( self.configured, displayStrLoopbackOpState( self.operational ) )

class SlotInfo( CliModel.Model ):
   hostLanes = CliModel.Dict(
      keyType=int,
      valueType=LaneInfo,
      help="Host side lane configuration and status information, keyed by lane ID" )

   _hostLanesAllCfg = CliModel.Bool(
      help="Per-lane host configuration is not present", default=False )

   mediaLanes = CliModel.Dict(
      keyType=int,
      valueType=LaneInfo,
      help="Media side lane configuration and status information, keyed by lane ID" )

   _mediaLanesAllCfg = CliModel.Bool(
      help="Per-lane media configuration is not present", default=False )

   def _getLaneItems( self, kind, key ):
      # pylint: disable=protected-access
      attrMap = { "hostLanes": "host", "mediaLanes": "media" }
      return list(
         StatusRow( key, attrMap[ kind ], lane, *info._getItems(),
                    getattr( self, f"_{kind}AllCfg" ) )
         for lane, info in getattr( self, kind ).items() )
      # pylint: enable=protected-access

   def _getItems( self, key ):
      return list(
         itertools.chain( *( self._getLaneItems( t, key )
                             for t in ( "hostLanes", "mediaLanes" ) ) ) )

statusAttrs = ( 'slot', 'side', 'lane', 'conf', 'op', 'all' )
StatusRow = collections.namedtuple( 'StatusRow', statusAttrs )
StatusKey = collections.namedtuple( 'StatusKey',
                                    ( *statusAttrs[ 0 : 2 ], *statusAttrs[ 3 : ] ) )

class TrafficLoopback( CliModel.Model ):
   transceiverLoopback = CliModel.Dict(
      keyType=IntfModels.Interface,
      valueType=SlotInfo,
      help=( "Loopback configuration and status of each transceiver that "
             "supports CMIS based loopback" ) )

   def render( self ):
      # pylint: disable=protected-access
      t = TableOutput.createTable(
         ( ( "Slot", "r" ), ( "Side", "l" ), ( "Lane", "r" ), ( "Configured", "l" ),
           ( "Operational", "l" ) ) )
      items = list(
         itertools.chain( *( value._getItems( subPort( key, '' ) )
                             for key, value in self.transceiverLoopback.items() ) ) )
      for row in self._compactEntries( items ):
         t.newRow( *row )
      print( t.output() )
      # pylint: enable=protected-access

   def _groupKey( self, elem: StatusRow ) -> StatusKey:
      return StatusKey( *elem[ 0 : 2 ], *elem[ 3 : ] )

   def _compactEntries( self, items: Iterable[ StatusRow ] ):
      rows = []
      for key, group in itertools.groupby( sorted( items, key=self._groupKey ),
                                           self._groupKey ):
         if not key.all:
            sortedLanes = sorted( [ elem.lane for elem in group ] )
            lanes = multiRangeToCanonicalString( sortedLanes )
         else:
            lanes = "all"
         rows.append( ( key.slot, key.side, lanes, key.conf, key.op ) )
      return sorted( rows,
                     key=lambda e:
                     ( *( int( v ) for v in e[ 0 ].split( '/' ) ), e[ 1 ], e[ 2 ] ) )
