# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import EthIntfLib
import LazyMount
import CliCommand
from CliPlugin.XcvrConfigCli import ( intfCliReject,
                                      getXcvrConfigCliDir,
                                      getXcvrConfigCliForConfigCommand,
                                      xcvrConfigCliDelete )
from CliPlugin.XcvrCliLib import ( getAllIntfsWrapper, isModular )
from XcvrLib import getXcvrSlotName
from CliDynamicSymbol import CliDynamicPlugin
from TypeFuture import TacLazyType
import CliGlobal

XFDM = CliDynamicPlugin( 'XcvrCoherentFecDegradeModel' )

# aid/5819 - set up global variables that are initialized once in plugin load time
gv = CliGlobal.CliGlobal( cmisCoherentStatusSliceDir=None )

AlarmStatus = TacLazyType( "Xcvr::XcvrAlarm::AlarmStatus" )
MgmtIntfId = TacLazyType( "Arnet::MgmtIntfId" )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver error-correction coherent degrade <detected|excessive>
# raise RAISE clear CLEAR
# -------------------------------------------------------------- ------------------
def coherentFecDegradeHandler( mode, args ):
   intfName = mode.intf.name
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   # Pass in 'not noOrDefault' for 'create', meaning we will create a
   # xcvrConfigCli object only when the command is NOT the no/default variant.
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                     not noOrDefault )
   if xcvrConfigCli is None:
      assert noOrDefault
      return

   assert xcvrConfigCli
   if noOrDefault:
      cmisFecDegradeParameter = Tac.Value( 'Xcvr::CoherentCmis::FecDegradeParameter',
                                           False, 0, 0 )
   else:
      raiseValue = args.get( 'RAISE' )
      clearValue = args.get( 'CLEAR' )
      cmisFecDegradeParameter = Tac.Value( 'Xcvr::CoherentCmis::FecDegradeParameter',
                                           True, raiseValue, clearValue )

   cmisFecDegradeConfig = Tac.nonConst( xcvrConfigCli.cmisFecDegradeConfig )
   if 'excessive' in args:
      cmisFecDegradeConfig.excessiveDegrade = cmisFecDegradeParameter
   else:
      assert 'detected' in args
      cmisFecDegradeConfig.detectedDegrade = cmisFecDegradeParameter
   xcvrConfigCli.cmisFecDegradeConfig = cmisFecDegradeConfig

   if noOrDefault:
      if xcvrConfigCliDelete( xcvrConfigCli ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

# --------------------------------------------------------------------------------
# show transceiver error-correction coherent degrade [ interface INTFS ]
# --------------------------------------------------------------------------------
def getCmisCoherentStatusDir( intfName ):
   sliceName = "FixedSystem"
   if isModular() and not MgmtIntfId.isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
   return gv.cmisCoherentStatusSliceDir.get( sliceName )

def populateThresholds( thresholdsStatus ):
   thresholds = XFDM.TransceiverCoherentFecDegradeThresholds()
   thresholds.tRaise = thresholdsStatus.raiseThreshold
   thresholds.tClear = thresholdsStatus.clearThreshold
   return thresholds

def populateParameter( degradeConfig, degradeStatus, degradeType ):
   parameter = None
   assert degradeType in ( "detected", "excessive" )
   # degradeConfig.excessiveDegrade or degradeConfig.detectedDegrade
   degradeAttr = degradeType + "Degrade"
   # degradeStatus.excessiveOperationalStatus or
   # degradeStatus.detectedOperationalStatus
   operationalStatusAttr = degradeType + "OperationalStatus"
   # degradeStatus.excessiveAlarmStatus or degradeStatus.detectedAlarmStatus
   alarmStatusAttr = degradeType + "AlarmStatus"

   config = getattr( degradeConfig, degradeAttr )
   if config.enabled:
      parameter = XFDM.TransceiverCoherentFecDegradeParameter()
      parameter.configured = populateThresholds( config )
      if degradeStatus is not None:
         alarmStatus = getattr( degradeStatus, alarmStatusAttr, AlarmStatus.unknown )
         parameter.status = alarmStatus
         operationalStatus = getattr( degradeStatus, operationalStatusAttr )
         if operationalStatus.enabled:
            parameter.operational = (
               populateThresholds( operationalStatus ) )
      else:
         parameter.status = AlarmStatus.unknown
   return parameter

def populateDetectedDegradeModel( degradeConfig, degradeStatus ):
   # populate detected degrade info
   return populateParameter( degradeConfig, degradeStatus, "detected" )

def populateExcessiveDegradeModel( degradeConfig, degradeStatus ):
   # populate excessive degrade info
   return populateParameter( degradeConfig, degradeStatus, "excessive" )

def populateCoherentFecDegradeModel( intfName ):
   """
   This function parses coherent FEC Degrade status and alarm state
   for a given interface

   Parameters
   ----------
   intfName: Arnet::IntfId
      The interface whose status is parsed
   """
   model = None

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   if xcvrConfigCliDir is None:
      return model

   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   if xcvrConfigCli is None:
      return model

   degradeConfig = xcvrConfigCli.cmisFecDegradeConfig
   if ( not degradeConfig.excessiveDegrade.enabled and
        not degradeConfig.detectedDegrade.enabled ):
      return model

   cmisCoherentStatusDir = getCmisCoherentStatusDir( intfName )
   if cmisCoherentStatusDir is None:
      return model
   fecDegradeStatusDir = cmisCoherentStatusDir.fecDegradeStatusDir
   if fecDegradeStatusDir is None:
      return model

   model = XFDM.TransceiverCoherentFecDegradeSlot()
   degradeStatus = fecDegradeStatusDir.degradeStatus.get( intfName )
   model.detected = populateDetectedDegradeModel( degradeConfig,
                                                  degradeStatus )
   model.excessive = populateExcessiveDegradeModel( degradeConfig,
                                                    degradeStatus )
   return model

def showXcvrCoherentFecDegrade( mode, args ):
   intf = args.get( 'INTFS' )
   mod = args.get( 'MOD' )
   baseModel = XFDM.TransceiverCoherentFecDegradeCollection()
   ( intfs, _ ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return baseModel

   for intf in intfs:
      intfName = intf.name
      # Don't want secondary interfaces showing up in the output
      if intfCliReject( intf ):
         continue

      model = populateCoherentFecDegradeModel( intfName )
      slotName = getXcvrSlotName( intfName )
      if model is not None:
         baseModel.slots[ slotName ] = model
   return baseModel

def Plugin( em ):
   gv.cmisCoherentStatusSliceDir = LazyMount.mount( em,
                  "hardware/xcvr/status/cmis/coherent/slice", "Tac::Dir", "ri" )
